%{
/* ----------------------------------------------------------------------------
 * Copyright (C) 1995-1999, 2000 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * --------------------------------------------------------------------------*/

#include "globals.h"
#include "symtab.h"
#include "math.h"
#include <math.h>

extern void	printError(char *);
extern boolean	parseError;
extern complex	result;

%}

%union {
	complex		cval;
	struct symtab	*symp;
}

%token <symp> NAME
%token <cval> NUMBER
%token SRE SIM RE IM ABS PHI
%token SGN INT FRAC ROUND
%token EXP LN LOG SQRT
%token SIN COS TAN COT SINH COSH TANH COTH
%token ARCSIN ARCCOS ARCTAN ARCCOT ARSINH ARCOSH ARTANH ARCOTH
%token ERF ERFC FACT

%left '+' '-'
%left '*' '/'
%left ';'
%nonassoc UMINUS
%right '^'

%type <cval> expr
%type <cval> brexpr
%%

statement:
	|	expr			{ result.re = $1.re;
					  result.im = $1.im; }
	|	def			{ }
	|	def ';' expr		{ result.re = $3.re;
					  result.im = $3.im; }
	;

def:		def ';' def
	|	NAME '=' expr		{ if ($1) {
					     $1->value.re = $3.re;
					     $1->value.im = $3.im;
					     $1->def = TRUE;
					  }
					}
	;

expr:		expr '+' expr		{ $$ = cadd($1,$3); }
	|	expr '-' expr		{ $$ = csub($1,$3); }
	|	expr '*' expr		{ $$ = cmult($1,$3); }
	|	expr '^' expr		{ $$ = cpower($1,$3); }
	|	expr '/' expr		{ $$ = cdiv($1,$3); }
	|	EXP brexpr		{ $$ = cexp($2); }
	|	LN brexpr		{ $$ = clog($2); }
	|	LOG brexpr		{ $$ = clog10($2); }
	|	SQRT brexpr		{ $$ = csqrt($2); }
	|	SIN brexpr		{ $$ = csin($2); }
	|	COS brexpr		{ $$ = ccos($2); }
	|	TAN brexpr		{ $$ = ctan($2); }
	|	COT brexpr		{ $$ = ccot($2); }
	|	SINH brexpr		{ $$ = csinh($2); }
	|	COSH brexpr		{ $$ = ccosh($2); }
	|	TANH brexpr		{ $$ = ctanh($2); }
	|	COTH brexpr		{ $$ = ccoth($2); }
	|	ARCSIN brexpr		{ $$ = carcsin($2); }
	|	ARCCOS brexpr		{ $$ = carccos($2); }
	|	ARCTAN brexpr		{ $$ = carctan($2); }
	|	ARCCOT brexpr		{ $$ = carccot($2); }
	|	ARSINH brexpr		{ $$ = carsinh($2); }
	|	ARCOSH brexpr		{ $$ = carcosh($2); }
	|	ARTANH brexpr		{ $$ = cartanh($2); }
	|	ARCOTH brexpr		{ $$ = carcoth($2); }
	|	ERF brexpr		{ $$ = cerf($2); }
	|	ERFC brexpr		{ $$ = cerfc($2); }
	|	FACT brexpr		{ $$ = cfact($2); }
	|	SGN brexpr		{ $$ = csgn($2); }
	|	INT brexpr		{ $$ = cint($2); }
	|	FRAC brexpr		{ $$ = cfrac($2); }
	|	ROUND brexpr		{ $$ = cround($2); }
	|	SRE brexpr		{ $$ = sre($2); }
	|	SIM brexpr		{ $$ = sim($2); }
	|	ABS brexpr		{ $$.re = cabs($2);
					  $$.im = 0; }
	|	PHI brexpr		{ $$.re = cphi($2);
					  $$.im = 0; }
	|	RE brexpr		{ $$.re = $2.re;
					  $$.im = 0; }
	|	IM brexpr		{ $$.re = $2.im;
					  $$.im = 0; }
	|	'|' expr '|'		{ $$.re = cabs($2);
					  $$.im = 0; }
	|	'-' expr %prec UMINUS	{ $$ = cneg($2); }
	|	brexpr			{ $$ = $1; }
	|	NUMBER			{ $$ = $1; }
	|	NAME			{ if ($1) {
					     $$.re = $1->value.re;
					     $$.im = $1->value.im;
					     if (!$1->def)
					        yyerror("undefined symbol");
					  }
					}
	;

brexpr:		'(' expr ')'		{ $$.re = $2.re;
					  $$.im = $2.im; }
	|	'[' expr ']'		{ $$.re = $2.re;
					  $$.im = $2.im; }
	|	'{' expr '}'		{ $$.re = $2.re;
					  $$.im = $2.im; }
	|	'<' expr '>'		{ $$.re = $2.re;
					  $$.im = $2.im; }
	;

%%





/* ----------------------------------------------------------------------------
 * yyerror - print parsing error messages
 * ------------------------------------------------------------------------- */

void yyerror(char *message)
{
	parseError = TRUE;
	printError(message);
}





/* ----------------------------------------------------------------------------
 * lookSymtab - look up or add a symbol table entry
 * ------------------------------------------------------------------------- */

struct symtab *lookSymtab(char *s)
{
	char		*p;
	struct symtab	*sp;

	for (sp = symtab; sp < &symtab[NSYMS]; sp++) {

	    /* ---------------------------------- look up symbol table entry */
	
	    if (sp->name != NULL && strcmp(sp->name, s) == 0) {
	       return sp;
	    }

	    /* -------------------------------------- add symbol tabel entry */

	    if (sp->name == NULL) {
	       sp->name = strdup(s);
	       return sp;
	    }
	}

	yyerror("symbol overflow");
	return NULL;
}





/* ----------------------------------------------------------------------------
 * freeSymtab - free symbol table
 * ------------------------------------------------------------------------- */

void freeSymtab(int status)
{
	struct symtab	*sp;

	for (sp = symtab; sp < &symtab[NSYMS]; sp++) {
	    if (sp->name != NULL) {
	       free(sp->name);
	       sp->name = NULL;
	    }

	    sp->value.re = 0;
	    sp->value.im = 0;
	    sp->def = status;
	}
}
