/* ----------------------------------------------------------------------------
 * Copyright (C) 1995-1997, 2000 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * ------------------------------------------------------------------------- */

/* ----------------------------------------------------------------------------
 * extract - data reduction of two-column ASCII tables
 *
 * command options:
 *
 *         -i  input file
 *         -o  output file (default stdout)
 *         -e  reduction error
 *         -m  maintain minima and maxima
 *         -v  print copyright notice
 * ------------------------------------------------------------------------- */

#include "str_utils.h"
#include "license.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <unistd.h>

#define  MAXCHARS 256
#define  TRUE     1
#define  FALSE    0





char   release[]     = "Extract V.1.2";
char   copyright[]   = "Copyright (C) 1995-1997, 2000 by Karim Kaschani";





/* ----------------------------------------------------------------------------
 * usage
 * ------------------------------------------------------------------------- */

void usage()
{
    printf("usage: extract [-e <error>] [-x <error>] [-y <error>]\\\n");
    printf("               [-m] [-v] -i <infile> [-o <outfile>]\n");
}





/* ----------------------------------------------------------------------------
 * main
 * ------------------------------------------------------------------------- */

int main(int argc, char *argv[])
{
    int		i, index, num_source, num_target, num_skipped;
    int		INFILE, OUTFILE, IRRELEVANT, MINMAX, CHECKX, CHECKY;
    double	x[3], y[3], xmin, xmax, ymin, ymax, xold, yold, percentage;
    double	dx[2], dy[2], dxi, dyi, dxy, dxmax, dymax, dxe, dye;
    char	*target, *source, *word1, *word2, *date, *command;
    char	buffer[MAXCHARS];
    FILE	*infile, *outfile;
    time_t	seconds, *tloc=NULL;
    size_t	maxdate=30;
    extern char	*optarg;
    extern int	opterr, optind;


    target  = malloc(sizeof(char) * MAXCHARS);
    source  = malloc(sizeof(char) * MAXCHARS);
    word1   = malloc(sizeof(char) * MAXCHARS);
    word2   = malloc(sizeof(char) * MAXCHARS);
    date    = malloc(sizeof(char) * MAXCHARS);
    command = malloc(sizeof(char) * MAXCHARS);

    /* ----------------------------------------------------- set up defaults */

    seconds = time(tloc);
    (void) strftime(date,maxdate,"%a %d %b %Y (%X)",localtime(&seconds));
 
    MINMAX  = FALSE;
    OUTFILE = FALSE;
    INFILE  = FALSE;
    CHECKX  = FALSE;
    CHECKY  = FALSE;

    (void) strcpy(target,"stdout");
    outfile = stdout;
    
    dxy   = -1;
    dxe   = -1;
    dye   = -1;
    index = 0;
    xold = yold = 0;
    xmin = xmax = 0;
    ymin = ymax = 0;
    num_source = num_target = 0;

    /* ------------------------------------------ parse command line options */

    (void) strcpy(command, "extract");

    opterr = 0;

    while ((i = getopt(argc, argv, "e:x:y:mvi:o:")) != EOF) {

          if (i == -1) break;

          switch (i) {
                 case 'v': fprintf(stdout, "%s\n\n%s\n\n%s\n",
                                   release, copyright, License);
                           exit(0);
                           break;
                 case 'm': (void) strcat(command," -m");
                           MINMAX = TRUE;
                           break;
                 case 'e': sscanf(optarg,"%lf",&dxy);
                           (void) strcat(command," -e ");
                           (void) strcat(command,optarg);
                           break;
                 case 'x': sscanf(optarg,"%lf",&dxe);
                           (void) strcat(command," -x ");
                           (void) strcat(command,optarg);
                           CHECKX = TRUE;
                           break;
                 case 'y': sscanf(optarg,"%lf",&dye);
                           (void) strcat(command," -y ");
                           (void) strcat(command,optarg);
                           CHECKY = TRUE;
                           break;
                 case 'o': sscanf(optarg,"%s",target);
                           (void) strcat(command," -o ");
                           (void) strcat(command,optarg);
                           OUTFILE = TRUE;
                           break;
                 case 'i': sscanf(optarg,"%s",source);
                           (void) strcat(command," -i ");
                           (void) strcat(command,optarg);
                           INFILE = TRUE;
                           break;
                 default:  usage();
                           exit(1);
          }
    }

    if (optind < argc || !INFILE || (dxy < 0 && dxe < 0 && dye < 0)) {
       usage();
       exit(1);
    }

    /* --------------------------------------------------------- open infile */

    if (! (infile = fopen(source,"r"))) {
       fprintf(stderr,"extract: cannot open %s\n",source);
       exit(2);
    }

    /* ------------------------------------- scan infile for dxmax and dymax */

    while (fgets(buffer, MAXCHARS, infile)) {
          (void) sscanf(buffer,"%s %s",word1, word2);
          if (isnumber(word1[0]) && isnumber(word2[0])) {
             sscanf(buffer,"%lf %lf",&x[0],&y[0]);
             if (x[0] < xmin)
                xmin = x[0];
             if (x[0] > xmax)
                xmax = x[0];
             if (y[0] < ymin)
                ymin = y[0];
             if (y[0] > ymax)
                ymax = y[0];
          }
    }

    dxmax = xmax - xmin;
    dymax = ymax - ymin;

    (void) fclose(infile);

    /* -------------------------------------------------------- extract data */

    if (! (infile = fopen(source,"r"))) {
       fprintf(stderr,"extract: cannot open %s\n",source);
       exit(2);
    }

    if (OUTFILE) {
       if (! (outfile = fopen(target,"w"))) {
          fprintf(stderr,"extract: cannot open %s\n",target);
          (void) fclose(infile);
          exit(2);
       }
    }

    fprintf(outfile,"# %s --> %s\n",date,command);

    while (fgets(buffer, MAXCHARS, infile)) {
          (void) sscanf(buffer,"%s %s",word1, word2);

          if (isnumber(word1[0]) && isnumber(word2[0])) {
             sscanf(buffer,"%lf %lf",&x[index],&y[index]);
             index++;
             num_source++;
          } else {
             for (i=0; i<index; i++) {
                 fprintf(outfile,"  %1.8E  %1.8E\n",x[i],y[i]);
                 num_target++;
             }
                 fprintf(outfile,"%s",buffer);
                 index = 0;
          }

          (void) strcpy(word1,"\0");
          (void) strcpy(word2,"\0");

          /* search for irrelevant data */

          if (index >= 3) {
             dx[0] = x[1]-x[0];
             dx[1] = x[2]-x[1];
             dxi = dx[0]/dxmax*dx[0]/dxmax;

             dy[0] = y[1]-y[0];
             dy[1] = y[2]-y[1];
             dyi = dy[0]/dymax*dy[0]/dymax;

             IRRELEVANT = FALSE;

             /* check for absolute x error */

             if (CHECKX) {
                if ((fabs(x[1]-xold) >= dxe) && (fabs(x[2]-x[1]) >= dxe)) {
                   IRRELEVANT = TRUE;
                   xold = x[1];
                } else {
                   xold = x[0];
                }
             }

             /* check for absolute y error */

             if (CHECKY) {
                if ((fabs(y[1]-yold) >= dye) && (fabs(y[2]-y[1]) >= dye)) {
                   IRRELEVANT = TRUE;
                   yold = y[1];
                } else {
                   yold = y[0];
                }
             }

             /* check for relative error */

             if (sqrt(dxi+dyi) <= dxy) {

                /* irrelevant data ? */
             
                if ((dx[0]*dx[1] > 0) && (dy[0]*dy[1] > 0)) {

                   /* steady incline or decline */
                
                   IRRELEVANT = TRUE;
                } else {

                   /* minimum or maximum */
                
                   if (! MINMAX) {
                      IRRELEVANT = TRUE;
                   }
                }
             }

             if (IRRELEVANT) {

                /* x[1] and y[1] are irrelevant */
                 
                x[1] = x[2];
                y[1] = y[2];
             } else {

                /* x[1] and y[1] are not irrelevant */

                fprintf(outfile,"  %1.8E  %1.8E\n",x[0],y[0]);
                num_target++;

                for (i=1; i<index; i++) {
                    x[i-1] = x[i];
                    y[i-1] = y[i];
                }
             }

             index--;

          } else {
             xold = x[0];
             yold = y[0];
          }
    }

    /* print remaining data */

    for (i=0; i<index; i++) {
        fprintf(outfile,"  %1.8E  %1.8E\n",x[i],y[i]);
        num_target++;
    }

    if (OUTFILE) {
       (void) fclose(outfile);
    }

    /* print final message */

    num_skipped = num_source - num_target;
    percentage  = (double) num_target / (double) num_source * 100.0;

    fprintf(stderr,"%s: %d points extracted, %d points skipped, %2.1f%% data reduction.\n",
            target,num_target, num_skipped, percentage);
    
    exit(0);
}
