/* ----------------------------------------------------------------------------
 * Copyright (C) 1995-2000 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * --------------------------------------------------------------------------*/


#define KEYC

#include "globals.h"
#include <stdarg.h>
#include <unistd.h>
#include <curses.h>
#include "curs_utils.h"
#include "misc_utils.h"
#include "str_utils.h"
#include <math.h>

#include <dirent.h>
#include <sys/stat.h>
#include <signal.h>
#include <sys/termios.h>
#include <sys/ioctl.h>

#ifdef LINUX
#  include <ioctl-types.h>
#endif

static char *More = {
"Forward (f/F) / Backward (b/B) / BOF (H) / EOF (G) / Quit (q): --More--"
};

static char *MathExpressions[] = {
"Mathematical Expressions",

"\n\
 j       pi      e       re!     im!     re      im      abs     phi     + -   \n\
 * /     ^       sqrt    ln      log     exp     sin     cos     tan     cot   \n\
 sinh    cosh    tanh    coth    arcsin  arccos  arctan  arccot  arsinh  arcosh\n\
 artanh  arcoth  erf     erfc    fact    sgn     int     frac    round"
};

static char *EditCommands[] = {
"Editor Commands",

"\n\
 BSP   (^H)   DEL   (^X)   ESC   (^D)   enter (CR)  left  (^L)   right (^R)    \n\
 up    (^P)   down  (^N)   start (^A)   end   (^E)  undo  (^U)   next (TAB)    "
};





/* ----------------------------------------------------------------------------
 * initCurses - initialize curses
 * ------------------------------------------------------------------------- */

void initCurses()
{
  initscr();
  cbreak();
  noecho();
  keypad (stdscr, TRUE);
}





/* ----------------------------------------------------------------------------
 * fc_exit - nice exit on CTRL-C
 * ------------------------------------------------------------------------- */

void fc_exit()
{
  signal (SIGINT, SIG_IGN);

  if (!BATCH) {
     erase();
     move(0,0);
     refresh();
     endwin();
  }

  exit(EXIT_SUCCESS);
}





/* ----------------------------------------------------------------------------
 * updateWinSize - update the current window size
 *
 *                 In order to fool curses and resize the current window
 *                 although curses cannot perform such an operation, we:
 *
 *                 1. terminate curses clearing all windows
 *                 2. update the environment variables LINES and COLUMNS
 *                    to their new values
 *                 3. restart curses rewriting all windows
 * ------------------------------------------------------------------------- */

void updateWinsize()
{
  struct winsize	w;
  char			*tmptxt;

  /* ------------------------------------------------------ terminate curses */

  erase();
  move(0,0);
  refresh();
  endwin();

  /* ---------------------------------------------- update LINES and COLUMNS */

  tmptxt = (char *) malloc(sizeof(char) * 12);

  ioctl(fileno(stdout), TIOCGWINSZ, &w);

  sprintf(tmptxt, "LINES=%d", w.ws_row);
  putenv(tmptxt);
  sprintf(tmptxt, "COLUMNS=%d", w.ws_col);
  putenv(tmptxt);

  free(tmptxt);

  /* -------------------------------------------------------- restart curses */

  initCurses();
}





/* ----------------------------------------------------------------------------
 * GetKey - get key code
 * ------------------------------------------------------------------------- */

short GetKey()
{
  short  key;
  int    code;

  code = getch();

/* Debugging */
/*
   move(0,0);
   clrtoeol();
   mvprintw(0,0,"%d",code);
   refresh();
*/

  switch (code) {

            case KEY_UP: key = K_UP;
                         break;
          case KEY_DOWN: key = K_DOWN;
                         break;
          case KEY_LEFT: key = K_LEFT;
                         break;
         case KEY_RIGHT: key = K_RIGHT;
                         break;
             case K_C_D: key = K_ESC;
                         break;
            case KEY_DC:
             case K_C_X: key = K_DEL;
                         break;
     case KEY_BACKSPACE: key = K_BSP;
                         break;
         case KEY_CLEAR: key = K_ESC;
                         break;
                default: key = code;
                         break;
  }

  return key;
}





/* ----------------------------------------------------------------------------
 * reportError - report formula syntax error
 *               *parent  = parent curses window
 *               *label   = formula label
 *               *formula = miswritten formula
 *               pos      = error position in formula
 * ------------------------------------------------------------------------- */

void reportError(WINDOW *parent, char *label, char *formula, int pos)
{
	int	i, j, Lines, Cols;
	char	tmp[MAXCHARS];
	WINDOW	*errwin;
	extern	char ErrorMessage[];
	extern	boolean	SigWinch;
	extern	void Menu2();

	signal (SIGWINCH, delaySigWinch);

	Lines = 6;
	Cols  = 60;

	fprintf(stderr,"\007");

	errwin = newwin(Lines, Cols, 15, (COLS-Cols)/2);
	scrollok(errwin, FALSE);

	/* ---------------------------------------------- draw Window border */

	wstandout(errwin);

	for (i=0; i<Cols; i++) {
	    wmove(errwin, 0, i);
	    waddstr(errwin, " ");
	    wmove(errwin, Lines-1, i);
	    waddstr(errwin, " ");
	}

	for (i=0; i<Lines; i++) {
	    wmove(errwin, i, 0);
	    waddstr(errwin, " ");
	    wmove(errwin, i, Cols-1);
	    waddstr(errwin, " ");
	}

	/* ----------------------------------------------- draw Window title */

	wmove(errwin, 0, (Cols-Strlen(ErrorMessage))/2);
	waddstr(errwin, ErrorMessage);
	wstandend(errwin);

	/* -------------------------------------------- draw Window contents */

	wmove(errwin, 2, 2);
	waddstr(errwin, label);

	if (pos >= 0) {
	   j = pos-(Cols-Strlen(label)-4)/2;
	   j = j < 1 ? 1 : j;
	   pos = pos-j+1;
           Strncpy(tmp, &formula[j-1], (Cols-Strlen(label)-4));

           wmove(errwin, 2, 9);
           waddstr(errwin, tmp);
        }

	wstandout(errwin);
	if (pos >= 0) {
           wmove(errwin, 3, pos+8);
           waddstr(errwin, "^");
        } else {
           wmove(errwin, 2, Strlen(label)+2);
           waddstr(errwin, " ");
        }
	wmove(errwin, Lines-1, Cols-10);
	waddstr(errwin, "[  OK  ]");
	wstandend(errwin);
	if (pos >= 0) {
           wmove(errwin, 3, pos+8);
        } else {
           wmove(errwin, 2, Strlen(label)+2);
        }
        wrefresh(errwin);

	do {
	   i = GetKey();
	} while (i != K_CR);

	wclear(errwin);
        wrefresh(errwin);
	delwin(errwin);

	/* -------------------------------- restore underlying parent Window */

	touchwin(parent);
	wrefresh(parent);

	if (SigWinch) {
	   SigWinch = FALSE;
	   Menu2();
	}

	signal (SIGWINCH, Menu2);
}





/* ----------------------------------------------------------------------------
 * resizeNotice - display window size notice
 * ------------------------------------------------------------------------- */

void resizeNotice()
{
  move(0,0);
  printw("Window size %d/%d too small.\n",LINES, COLS);
  printw("Please re-size the window\n");
  printw("to be at least 24/80.\n");
}





/* ----------------------------------------------------------------------------
 * writeXY - write string 'txt' at screen position 'x,y'
 *           mode = -1: delete line 'y' until position 'x'
 *                   0: delete line 'y' completely
 *                   1: delete line 'y' beginning from position 'x'
 * ------------------------------------------------------------------------- */

void writeXY(short x, short y, char *txt, short mode)
{
  move(x, y);

  if (mode == 1)
    clrtoeol();
  else
    deleteln();

  addstr(txt);
  refresh();
}





/* ----------------------------------------------------------------------------
 * edit - edit string according to specified rules
 *        x,y        = screen position of string to be edited
 *        mode       = rule specification <kind><length>
 *                     t : all characters allowed
 *                     i : only integers allowed
 *                     r : all real numbers allowed
 *                     f : file names allowed
 *                     z.B. mode:='r10'
 *        editText   = string to be edited
 *        exitStatus = exit status
 *                     -1 : MOVE UP
 *                      0 : RETURN
 *                      1 : MOVE DOWN
 * ------------------------------------------------------------------------- */

void edit(short x, short y, char *mode, char *editText,
		 short *exitStatus)
{
  extern boolean	BLANKSCREEN;

  char			txt[256], oldtxt[256], tmptxt[256];
  short			i, l, len, pos, c_pos, offset, key, changes, inc;
  double		r;

  /* ----------------------------------------------------------------- setup */

  changes = TRUE;
  *exitStatus = 2;
  i = 2;
  r = atof(++mode);
  mode--;
  l = (short) r;
  strcpy(txt, editText);
  strcpy(oldtxt, txt);
  len = (short) Strlen(txt);
  pos = c_pos = offset = 0;

  /* ----------------------------------------------------------- edit string */

  do {
    if (changes && ! BLANKSCREEN) {
       if (c_pos < 0) {
          c_pos  = 0;
          offset--;
       }
       if (c_pos > COLS-y-1) {
          c_pos  = COLS-y-1;
          offset++;
       }
       Strncpy(tmptxt, &txt[offset], COLS-y-1);
       writeXY(x, y, tmptxt, 1);
    }

    if (! BLANKSCREEN) move(x, y+c_pos);

    refresh();
    key = GetKey();

    if (key < 0 || BLANKSCREEN) continue;

    if (key == K_BSP) {                                /* --- BACKSPACE --- */
       if (pos > 0) {
          for (i = pos; i <= len; i++)
              txt[i-1] = txt[i];
          len--;
          if (pos == c_pos)
             c_pos--;
          else
             offset--;
          pos--;
          changes = TRUE;
       }
    } else if (key == K_DEL) {                           /* --- DELETE --- */
       if (pos < len) {
          for (i = pos + 1; i <= len; i++)
              txt[i-1] = txt[i];
          len--;
          changes = TRUE;
       }
    } else if (key == K_LEFT) {                       /* --- MOVE LEFT --- */
       if (pos > 0) {
          pos--;
          c_pos--;
          changes = c_pos < 0 ? TRUE : FALSE;
       }
    } else if (key == K_RIGHT) {                     /* --- MOVE RIGHT --- */
       if (pos < len) {
          pos++;
          c_pos++;
          changes = c_pos > COLS-y-1 ? TRUE : FALSE;
       }
    } else if (key == K_START) {    /* --- MOVE RIGHT AT THE BEGINNING --- */
       pos    = 0;
       c_pos  = 0;
       offset = 0;
       changes = TRUE;
    } else if (key == K_END) {            /* --- MOVE RIGHT AT THE END --- */
       pos = len;
       if (len > COLS-y-1) {
          c_pos  = COLS-y-1;
          offset = len-(COLS-y-1);
       } else {
          c_pos  = len;
          offset = 0;
       }
       changes = TRUE;
    } else if (key == K_TAB) {                        /* --- NEXT ITEM --- */
       if (pos == len) {
          *exitStatus = 1;
          Strncpy(tmptxt, &txt[0], COLS-y-1);
          writeXY(x, y, tmptxt, 1);
          strcpy(editText, txt);
       } else {
          inc = strcspn(&txt[pos],";");
          inc += strspn(&txt[pos+inc],";");
          inc += strspn(&txt[pos+inc]," ");
          pos += inc;
          if (len <= COLS-y-1 || len - offset == COLS-y-1) {
             c_pos += inc;
          } else {
             offset += inc;
             if (len - offset < COLS-y-1) {
                offset = len - (COLS-y-1);
                c_pos = COLS-y-1 - (len-pos);
             }
          }
          changes = TRUE;
       }
    } else if (key == K_UP) {                           /* --- MOVE UP --- */
      *exitStatus = -1;
      Strncpy(tmptxt, &txt[0], COLS-y-1);
      writeXY(x, y, tmptxt, 1);
      strcpy(editText, txt);
    } else if (key == K_DOWN) {                       /* --- MOVE DOWN --- */
      *exitStatus = 1;
      Strncpy(tmptxt, &txt[0], COLS-y-1);
      writeXY(x, y, tmptxt, 1);
      strcpy(editText, txt);
    } else if (key == K_UNDO) {                            /* --- UNDO --- */
      strcpy(tmptxt, txt);
      strcpy(txt, oldtxt);
      strcpy(oldtxt, tmptxt);
      len = (short) Strlen(txt);
      pos = len;
      if (len > COLS-y-1) {
         c_pos  = COLS-y-1;
         offset = len-(COLS-y-1);
      } else {
         c_pos  = len;
         offset = 0;
      }
      changes = TRUE;
    } else if (key == K_CR) {                            /* --- RETURN --- */
      Strncpy(tmptxt, &txt[0], COLS-y-1);
      writeXY(x, y, tmptxt, 1);
      strcpy(editText, txt);
      *exitStatus = 0;
    } else if (key == K_ESC) {                           /* --- ESCAPE --- */
      strcpy(txt, "\0");
      len    = 0;
      pos    = 0;
      c_pos  = 0;
      offset = 0;
      changes = TRUE;
    } else {                                          /* --- CHARACTER --- */
      if (mode[0] == 't') {
	if (len < l) {
          for (i = len; i >= pos; i--)
              txt[i+1] = txt[i];
          txt[pos] = key;
	  len++;
	  pos++;
	  c_pos++;
	} else
	  printf("\007");
      } else if (mode[0] == 'i') {
	if (strrchr("0123456789",key) != NULL && len < l) {
          for (i = len; i >= pos; i--)
              txt[i+1] = txt[i];
          txt[pos] = key;
	  len++;
	  pos++;
	  c_pos++;
	} else
	  printf("\007");
      } else if (mode[0] == 'r') {
	if (strrchr("0123456789+-.eE",key) != NULL && len < l) {
          for (i = len; i >= pos; i--)
              txt[i+1] = txt[i];
          txt[pos] = key;
	  len++;
	  pos++;
	  c_pos++;
	} else
	  printf("\007");
      } else if (mode[0] == 'f') {
	if (strrchr("/",key) == NULL && len < l) {
          for (i = len; i >= pos; i--)
              txt[i+1] = txt[i];
          txt[pos] = key;
	  len++;
	  pos++;
	  c_pos++;
	} else
	  printf("\007");
      } else
	printf("\007");
      changes = TRUE;
    }
  } while (*exitStatus == 2);

}





/* ----------------------------------------------------------------------------
 * setupResourcePath - verifiy and set-up the resource path
 *                     *RCHOME = environment variable with optional resource
 *                               path
 *                     *suffix = default resource path suffix
 *                     *rcPath = resulting resource path
 * ------------------------------------------------------------------------- */

void setupResourcePath(const char *RCHOME, const char *suffix, char *rcPath)
{
    char	*home;
    mode_t	mode;

     if ((home = getenv(RCHOME)) != NULL) {
        strcpy(rcPath, home);
     } else {
        if ((home = getenv("HOME")) != NULL) {
           strcpy(rcPath, home);
           strcat(rcPath, suffix);
        } else {
           clear();
           refresh();
           endwin();
           fprintf(stderr, "ERROR: sorry - HOME directory not found\n");
           exit(2);
        }
     }

     mode = S_IRWXU | S_IRWXG | S_IRWXO;
     mkdirhier(rcPath, mode);
}





/* ----------------------------------------------------------------------------
 * helpMenu - display help menu in window "window"
 * ------------------------------------------------------------------------- */

void helpMenu(WINDOW *window)
{
	int	i;

	wmove(window, 0, 1);
	wstandout(window);
	for (i=COLS-2; i>0; i--)
            waddstr(window, " ");
	wmove(window, 0, (COLS-Strlen(MathExpressions[0]))/2);
	waddstr(window, MathExpressions[0]);
	wstandend(window);
	wmove(window, 1, 0);
	waddstr(window, MathExpressions[1]);

	wmove(window, 7, 1);
	wstandout(window);
	for (i=COLS-2; i>0; i--)
            waddstr(window, " ");
	wmove(window, 7, (COLS-Strlen(EditCommands[0]))/2);
	waddstr(window, EditCommands[0]);
	wstandend(window);
	wmove(window, 8, 0);
	waddstr(window, EditCommands[1]);
	wrefresh(window);
}





/* ----------------------------------------------------------------------------
 * showTitle - display title notes
 *             *release   = current release of program
 *             *copyright = copyright notice
 * ------------------------------------------------------------------------- */

void showTitle(char *release, char *copyright)
{
  short	i;

  move(1,1);
  standout();
  addstr(release);
  for (i=COLS-2-Strlen(release)-Strlen(copyright); i>0; i--)
      addstr(" ");
  addstr(copyright);
  standend();
  refresh();
}





/* ----------------------------------------------------------------------------
 * showCalculationNote - display subtitle notes
 *                       *prefix  = introduction
 *                       *dataset = current dataset
 *                       *suffix  = termination note
 * ------------------------------------------------------------------------- */

void showCalculationNote(char *prefix, char *dataset, char *suffix)
{
  short	i;
  char	*tmp;

  tmp = (char *) malloc(sizeof(char) * MAXCHARS);

  strcpy(tmp, prefix);
  strcat(tmp, " \"");
  strcat(tmp, dataset);
  strcat(tmp, "\" ");
  strcat(tmp, suffix);

  move(12, 1);
  standout();
  for (i=COLS-2; i>0; i--)
      addstr(" ");
  move(12, (COLS-Strlen(tmp))/2);
  addstr(tmp);
  standend();
  refresh();
  free(tmp);
}





/* ----------------------------------------------------------------------------
 * showViewerStatus - display view percentage
 *                    *calcwin = current curses window
 *                    ni       = current bottom line
 *                    nn       = total lines
 * ------------------------------------------------------------------------- */

void showViewerStatus(WINDOW *calcwin, int ni, int nn)
{
  double	P;

  P = MIN((ni-1)*100/nn,100);
  move(LINES-2,0);
  clrtoeol();
  standout();
  mvprintw(LINES-2,1,"%s(%d%%)",More, (short) P);
  standend();
  refresh();
  move(LINES-2, 5 + Strlen(More) + (short) log10(P));

  touchwin(calcwin);
  wrefresh(calcwin);
}





/* ----------------------------------------------------------------------------
 * showDataset - show dataset parameters
 *               position = 1st line to print 1st parameter of dataset
 *               *pattern = binary pattern, which holds information about the
 *                          following positions for printing of the following
 *                          dataset parameters. E.g. for position=3 "1101"
 *                          states:
 *                          "1" print 1st dataset parameter in line 3
 *                          "0" do not print anything in line 4
 *                          "1" print 2nd dataset parameter in line 5
 *                          "1" print 3rd dataset parameter in line 6
 *               ...      = the numbers of the following arguments of type
 *                          "char *" have to match the numbers of "1"
 *                          contained with "*pattern" exactly!
 * ------------------------------------------------------------------------- */

void showDataset(int position, char *pattern, ...)
{
  va_list	args;
  int		i;
  long		p, mask = 1;
  char		*label, *value;
  char		*tmptxt;

  tmptxt = (char *) malloc(sizeof(char) * MAXCHARS);

  p = strtol(pattern, NULL, 2);
  va_start(args, pattern);

  for (i=0; i<16; i++) {
      if (mask & p) {
         label = va_arg(args, char *);
         value = va_arg(args, char *);

         writeXY(position+i, 1, label, 1);
         Strncpy(tmptxt, value, COLS-Strlen(label)-2);
         addstr(tmptxt);
      }
      mask = mask << 1;
  }

  va_end(args);
  refresh();
  free(tmptxt);
}





/* ----------------------------------------------------------------------------
 * moreViewer - more like viewer
 *              **calcwin = curses window to display the lines to be browsed
 *              *xy0      = first line
 *              **xy      = current line
 *              **xyn     = last line
 *              nn        = total number of lines
 *              *ni       = last visible line in curses window
 * ------------------------------------------------------------------------- */

void moreViewer(WINDOW **calcwin, line *xy0, line **xy, line **xyn,
                int nn, int *ni)
{
  char			*tmptxt;
  int			d, j, k;
  extern boolean	BLANKSCREEN;

  tmptxt   = (char *) malloc(sizeof(char) * MAXCHARS);

  j = K_HOME;

  do {

     /* -------------------------- suspend execution until window size is ok */
      
     if (BLANKSCREEN) {
        sleep(1);
        continue;
     }

     if (j == K_HOME || j == 'H') {                            /* BOF, HOME */
        wclear(*calcwin);
        wmove(*calcwin, 0, 0);

        *xy = *xyn = xy0;
        *ni = 1;
        for (k=1; k<=LINES-17; k++) {
            if (*ni <= nn) {
               Strncpy(tmptxt, (char *) (*xyn), COLS-3);
               wprintw(*calcwin,"\n  %s",tmptxt);
               (*xyn)++;
            } else {
               wprintw(*calcwin,"\n  ~");
            }
            (*ni)++;
        }
     }

     if (j == 'G' && *ni <= nn) {                                      /* EOF */
        wclear(*calcwin);
        wmove(*calcwin, 0, 0);

        while (*ni <= nn - (LINES-17)) {
              (*ni)++;
              (*xy)++;
              (*xyn)++;
        }

        while (*ni - 1 > nn - (LINES-17) && *ni > 0) {
              (*ni)--;
              (*xy)--;
              (*xyn)--;
        }

        for (k=0; k<LINES-17; k++) {
            (*ni)++;
            (*xy)++;
            Strncpy(tmptxt, (char *) (*xyn), COLS-3);
            wprintw(*calcwin,"\n  %s",tmptxt);
            (*xyn)++;
        }
     }

     if ((j == 'F' || j == 32) && *ni <= nn) {                   /* Page down */
        wclear(*calcwin);
        wmove(*calcwin, 0, 0);

        d = MIN(nn+1 - *ni, LINES-17);

        if (d < LINES-17) {
           for (k=d; k<LINES-17; k++) {
               (*ni)--;
               (*xy)--;
               (*xyn)--;
           }
        }

        for (k=0; k<LINES-17; k++) {
            (*ni)++;
            (*xy)++;
            Strncpy(tmptxt, (char *) (*xyn), COLS-3);
            wprintw(*calcwin,"\n  %s",tmptxt);
            (*xyn)++;
        }
     }

     if (j == 'B' && *ni-1 > LINES-17) {                           /* Page up */
        wclear(*calcwin);
        wmove(*calcwin, 0, 0);

        d = MIN(*ni-1, (LINES-17)*2);

        for (k=0; k<d; k++) {
            (*ni)--;
            (*xy)--;
            if (*ni <= nn) (*xyn)--;
        }

        for (k=0; k<LINES-17; k++) {
            (*ni)++;
            (*xy)++;
            Strncpy(tmptxt, (char *) (*xyn), COLS-3);
            wprintw(*calcwin,"\n  %s",tmptxt);
            (*xyn)++;
        }
     }

     if ((j == K_DOWN || j == K_CR || j == 'f')                 /* Line down */
        && *ni <= nn) {
        (*ni)++;
        wscrl(*calcwin, 1);
        Strncpy(tmptxt, (char *) (*xyn), COLS-3);
        mvwprintw(*calcwin, LINES-18, 0,"  %s",tmptxt);
        (*xyn)++;
        (*xy)++;
     }

     if ((j == K_UP || j == 'b') && *ni-1 > LINES-17) {            /* Line up */
        (*ni)--;
        wscrl(*calcwin, -1);
        Strncpy(tmptxt, (char *) (--(*xy)), COLS-3);
        mvwprintw(*calcwin, 0, 0,"  %s",tmptxt);
        if (*ni <= nn) (*xyn)--;
     }

     showViewerStatus(*calcwin, (int) *ni, (int) nn);

     j = GetKey();

  } while (j != 'q');                                                /* Quit */

  free(tmptxt);
}





/* ----------------------------------------------------------------------------
 * saveData - save results
 *            *id      = file id (either "stdout" or "NULL")
 *            *outfile = name of output file if not "stdout"
 *            nn       = total number of lines
 *            *xy      = first line
 *            *Menu    = command line menu
 * ------------------------------------------------------------------------- */

void saveData(FILE *id, char *outfile, int nn, line *xy, char *Menu)
{
  int	i;

  if (id == NULL) id = fopen(outfile, "w");

  if (id != NULL) {
     for (i=1; i<=nn; i++)
         fprintf(id, "%s\n", (char *) xy++);

     if (id != stdout) {
        fclose(id);
        id = NULL;
     }

     if (!BATCH) {
        standout();
        writeXY(LINES-2, Strlen(Menu)+1, "- file saved ", 1);
        standend();
        addstr("! ");
        move(LINES-2, Strlen(Menu)+14);
     }
  }
}





/* ----------------------------------------------------------------------------
 * saveDataset - save resourcefile data
 *               *calcwin      = parent curses window
 *               *resourcefile = name of resourcefile to save dataset
 *               n             = number of dataset parameters to save
 *               ...           = the numbers of the following arguments
 *                               of type "char *" have to match the number
 *                               contained with "n" exactly!
 * ------------------------------------------------------------------------- */

void saveDataset(WINDOW *calcwin, char *resourcefile, int n, ...)
{
  FILE		*id;
  extern	char ErrorMessage[];
  va_list	parm;
  int		i;
  char		*rc;

  id = fopen(resourcefile, "w");

  if (id != NULL) {
     va_start(parm, n);

     for (i=0; i<n; i++) {
         rc = va_arg(parm, char *);
         fprintf(id, "%s\n", rc);
     }

     fclose(id);
     va_end(parm);
  } else {
     strcpy(ErrorMessage,  "               i/o error               ");
     reportError(calcwin,
                 " Saving calculation parameters to resource file failed.",
                 "", -1);
  }
}
