/* ----------------------------------------------------------------------------
 * Copyright (C) 2000 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * --------------------------------------------------------------------------*/

#include "globals.h"
#include "math.h"
#include "misc_utils.h"
#include "str_utils.h"
#include "cc_version.h"
#include "license.h"
#include "symtab.h"
#include <unistd.h>
#include <curses.h>
#include "curs_utils.h"
#include <dirent.h>
#include <sys/stat.h>
#include <signal.h>
#include <time.h>
#include <locale.h>
#include <math.h>
#include <values.h>

#define	ALLOCINC	100





extern char yytext[];
extern int  yyparse();

void Menu1();
void Menu2();
void Menu3();
void Menu4();
void Menu5();





/* ----------------------------------------------------------------- globals */

double		*phi;
int		nphi, iphi;
boolean		runtimeError, parseError, visible, BLANKSCREEN;
boolean		SigWinch = FALSE;
int		yyPos;
complex		fArg;
complex		result;
char		*formula, *formula0;
char		ErrorMessage[] = "             SYNTAX ERROR              ";

static char     resourcepath[FILENAME_MAX];
static char     resourcefile[FILENAME_MAX], outfile[FILENAME_MAX];
static char	dataset[FILENAME_MAX], infile[FILENAME_MAX];
static char     H[MAXCHARS], S[MAXCHARS];
static char     s_start[MAXCHARS], s_end[MAXCHARS];
static char	buf[MAXCHARS], word1[MAXCHARS], word2[MAXCHARS];
static char	*errString;
static WINDOW	*helpwin = NULL;
static WINDOW	*calcwin = NULL;
static short	errPos;
static line     *xy, *xyn, *files;
static int	n, nn, ni;
static int	ki;

static char	rc_path[]      = "/.FuncCalc/cc";
static char	*Prompt  = "Dataset: ";
static char	*Prompt1 = "h(s)   = ";
static char	*Prompt7 = "Symbls : ";
static char	*Prompt3 = "s(1)   = ";
static char	*Prompt4 = "s(n)   = ";
static char	*Prompt6 = "File   : ";

static char *Menu = {
"View (v) / Plot (p) / Save (s) / Continue (c/C) / Quit (q/Q) "
};

static char *Calculation = "Calculating ";
static char *Reading = "Reading input data ...";
static char *Convolution = "Convolution according to";





/* ----------------------------------------------------------------------------
 * printError - print runtime errors
 * ------------------------------------------------------------------------- */

void printError(char *error)
{
	strncpy(ErrorMessage, error, 40);

	if (visible && !runtimeError) {
	   if (!BATCH) {
	      wprintw(calcwin, "%s", error);
	   } else {
	      fprintf(stderr, "convcalc: %s\n", error);
	   }
	}
}





/* ----------------------------------------------------------------------------
 * checkFormula - check a given formula for syntax errors
 * ------------------------------------------------------------------------- */

int checkFormula(char f[])
{
	int	position;

	visible = FALSE;	/* suppress any error messages  */
	parseError = FALSE;	/* initialize parser error flag */
	fArg.re = fArg.im = 0;	/* initialize formula argument  */
	yyPos = 0;		/* initialize position counter  */

	formula = formula0;	/* initialize formula address   */
	strcpy(formula, f);
	yyparse();

	if (parseError == TRUE) {
	   position = yyPos;
	} else
	   position = -1;

	return(position);
}





/* ----------------------------------------------------------------------------
 * evalFormula - evaluate a given formula
 * ------------------------------------------------------------------------- */

complex evalFormula(char f[], double x)
{
	visible = TRUE;		/* report any error */

	fArg.re = x;		/* initialize formula argument  */
	fArg.im = 0;
	formula = formula0;	/* initialize formula address   */
	strcpy(formula, f);
	yyparse();

	return(result);
}





/* ----------------------------------------------------------------------------
 * SubMenu6 - display subtitle notes
 * ------------------------------------------------------------------------- */

void SubMenu6()
{
  short	i;
  char	tmp[MAXCHARS];

  strcpy(tmp, Reading);

  move(12, 1);
  standout();
  for (i=COLS-3; i>0; i--)
      addstr(" ");
  move(12, (COLS-Strlen(tmp))/2);
  addstr(tmp);
  standend();
  move(12,COLS-2);
  refresh();
}





/* ----------------------------------------------------------------------------
 * SubMenu7 - display subtitle notes
 * ------------------------------------------------------------------------- */

void SubMenu7()
{
  short	i;
  char	tmp[MAXCHARS];

  strcpy(tmp, Convolution);

  move(12, 0);
  clrtoeol();
  move(12, 1);
  standout();
  for (i=COLS-3; i>0; i--)
      addstr(" ");
  move(12, (COLS-Strlen(tmp))/2);
  addstr(tmp);
  standend();
  move(12,COLS-2);
  refresh();
}





/* ----------------------------------------------------------------------------
 * Menu1 - display file selection window
 * ------------------------------------------------------------------------- */

void Menu1()
{
  char		*tmptxt;
  short		i;
  line		*filesn;

  updateWinsize();

  clear();

  if (LINES < 24 || COLS < 80) {
     BLANKSCREEN = TRUE;
     resizeNotice();
  } else {
     BLANKSCREEN = FALSE;

     showTitle(release, copyright);

     /* ---------------------------------------------------- display content */
 
     tmptxt = (char *) malloc(sizeof(char) * MAXCHARS);

     move(5,1);
     standout();
     for (i=COLS-2; i>0; i--)
         addstr(" ");
     move(5,(COLS-Strlen("Datasets"))/2);
     addstr("Datasets");
     standend();
   
     writeXY(3, 1, Prompt, 1);
     writeXY(3, Strlen(Prompt)+1, dataset, 1);
   
     move(LINES-10,1);
     standout();
     for (i=COLS-2; i>0; i--)
         addstr(" ");
     move(LINES-10,(COLS-Strlen("Parameters"))/2);
     addstr("Parameters");
     standend();

     /* --------------------------------------- set-up file selection window */

     if (calcwin != NULL) delwin(calcwin);
     calcwin = newwin(LINES-18, COLS, 7, 0);
     scrollok(calcwin, TRUE);
     wsetscrreg(calcwin, 0, LINES-18);
     wclear(calcwin);

     ki = i = 1;

     /* ------------------------ scan resource directory for current dataset */

     if (Strlen(dataset) != 0) {
        while (strcmp(dataset, (char *) (files)) != 0) {
           Strncpy(tmptxt, (char *) (files), COLS-3);
           mvwprintw(calcwin, ki-1, 2,"%s", tmptxt);

           if (ni < n) {
	      ni++;
              files++;

              if (ki < LINES-18) {
                 ki++;
              } else {
	         wscrl(calcwin, 1);
              }
           } else {
              break;
           }
        }
        i = ki;
     }

     /* ------------------------------------------ list (remaining) datasets */

     filesn = files;
     while (i < LINES-17 && ni + i - 2 < n) {
        Strncpy(tmptxt, (char *) (filesn), COLS-3);
        mvwprintw(calcwin, i-1, 2,"%s", tmptxt);
        filesn++;
        i++;
     }

     wmove(calcwin, ki-1, 2);
     wstandout(calcwin);
     wprintw(calcwin,"%s",dataset);
     wstandend(calcwin);

     touchwin(calcwin);
     wrefresh(calcwin);

     showDataset(LINES-8, "01011101",
                 Prompt1, H, Prompt7, S, Prompt3, s_start, Prompt4, s_end,
                 Prompt6, outfile);
   
     free(tmptxt);
  }

  refresh();
  signal (SIGWINCH, Menu1);
}





/* ----------------------------------------------------------------------------
 * Menu2 - display formular entry menu
 * ------------------------------------------------------------------------- */

void Menu2()
{
  updateWinsize();

  clear();

  if (LINES < 24 || COLS < 80) {
     BLANKSCREEN = TRUE;
     resizeNotice();
  } else {
     BLANKSCREEN = FALSE;

     /* ------------------------------------------------------- display menu */

     showTitle(release, copyright);
     showDataset(3, "10111001",
                 Prompt1, H, Prompt7, S, Prompt3, s_start, Prompt4, s_end,
                 Prompt6, outfile);
 
     if (helpwin != NULL) delwin(helpwin);
     helpwin = newwin(11, COLS, 12, 0);
     scrollok(helpwin, FALSE);

     helpMenu(helpwin);
  }

  refresh();
  signal (SIGWINCH, Menu2);
}





/* ----------------------------------------------------------------------------
 * Menu3 - display calculation menu
 * ------------------------------------------------------------------------- */

void Menu3()
{
  int		i;
  char		tmp[MAXCHARS];

  updateWinsize();

  clear();

  if (LINES < 24 || COLS < 80) {
     BLANKSCREEN = TRUE;
     resizeNotice();
  } else {
     BLANKSCREEN = FALSE;

     /* ------------------------------------------------------- display menu */

     showTitle(release, copyright);
     showDataset(3, "10111001",
                 Prompt1, H, Prompt7, S, Prompt3, s_start, Prompt4, s_end,
                 Prompt6, outfile);

     strcpy(tmp, Calculation);
     strcat(tmp, "\"");
     strcat(tmp, dataset);
     strcat(tmp, "\" ...");

     standout();
     move(12, 1);
     for (i=COLS-2; i>0; i--)
         addstr(" ");
     move(12, (COLS-Strlen(tmp))/2);
     addstr(tmp);
     standend();

     if (calcwin != NULL) delwin(calcwin);
     calcwin = newwin(LINES-17, COLS, 14, 0);
     scrollok(calcwin, TRUE);
     wsetscrreg(calcwin, 0, LINES-17);
  }

  refresh();
  signal (SIGWINCH, Menu3);
}





/* ----------------------------------------------------------------------------
 * Menu4 - display result menu
 * ------------------------------------------------------------------------- */

void Menu4()
{
  updateWinsize();

  clear();

  if (LINES < 24 || COLS < 80) {
     BLANKSCREEN = TRUE;
     resizeNotice();
  } else {
     BLANKSCREEN = FALSE;

     /* ------------------------------------------------------- display menu */

     showTitle(release, copyright);
     showDataset(3, "10111001",
                 Prompt1, H, Prompt7, S, Prompt3, s_start, Prompt4, s_end,
                 Prompt6, outfile);
     showCalculationNote(Convolution, dataset, "... finished.");

     touchwin(calcwin);
     wrefresh(calcwin);

     standout();
     writeXY(LINES-2, 1, Menu, 0);
     standend();
     refresh();
     addstr("?");
     move(LINES-2, Strlen(Menu)+1);
  }

  refresh();
  signal (SIGWINCH, Menu4);
}





/* ----------------------------------------------------------------------------
 * Menu5 - display result menu
 * ------------------------------------------------------------------------- */

void Menu5()
{
  int		i;
  char		*tmptxt;

  ni -= (LINES-17);

  updateWinsize();

  clear();

  if (LINES < 24 || COLS < 80) {
     BLANKSCREEN = TRUE;
     ni += (LINES-17);
     resizeNotice();
  } else {
     tmptxt = (char *) malloc(sizeof(char) * MAXCHARS);

     BLANKSCREEN = FALSE;

     /* ------------------------------------------------------- display menu */

     showTitle(release, copyright);
     showDataset(3, "10111001",
                 Prompt1, H, Prompt7, S, Prompt3, s_start, Prompt4, s_end,
                 Prompt6, outfile);
     showCalculationNote(Convolution, dataset, "... finished.");

     delwin(calcwin);
     calcwin = newwin(LINES-17, COLS, 14, 0);
     scrollok(calcwin, TRUE);
     wsetscrreg(calcwin, 0, LINES-17);
     wclear(calcwin);
     wmove(calcwin, 0, 0);

     xyn = xy;
     for (i=1; i<=LINES-17; i++) {
         if (ni <= nn) {
            Strncpy(tmptxt, (char *) (xyn), COLS-3);
            wprintw(calcwin,"\n  %s",tmptxt);
            xyn++;
         } else {
            wprintw(calcwin,"\n  ~");
         }
         ni++;
     }

     showViewerStatus(calcwin, (int) ni, (int) nn);

     free(tmptxt);
  }

  refresh();
  signal (SIGWINCH, Menu5);
}





/* ----------------------------------------------------------------------------
 * nextData - read next data from input file if any
 * ------------------------------------------------------------------------- */

char *nextData(FILE *id, double *x, double *y)
{
    char	*status;

    while ((status = fgets(buf, MAXCHARS, id)) != NULL) {
          (void) sscanf(buf, "%s %s", word1, word2);

          if (isnumber(word1[0]) && isnumber(word2[0])) {
             sscanf(buf,"%lf %lf", x, y);
             break;
          }
    
    }

    return status;
}




 
/* ----------------------------------------------------------------------------
 * h - calculate h(x) by interpolation
 * ------------------------------------------------------------------------- */

double h(double x, double *xH, double *yH, int lH)
{
    double	y;
    int		i1, i2;

    if (x == xH[0]) {
       y = yH[0];
    } else if ( x == xH[lH-1]) {
       y = yH[lH-1];
    } else if (xH[0] < x && x < xH[lH-1]) {
       i1 = (int) ((x-xH[0])/(xH[lH-1]-xH[0]) * lH);
       i2 = i1+1;
       y = (yH[i2]-yH[i1])/(xH[i2]-xH[i1]) * (x-xH[i1]) + yH[i1];
    } else {
       y = 0;
    }

    return y;
}





/* ----------------------------------------------------------------------------
 * convolute - convolute infile data and h(s)
 *
 *             The area represented by the convolution integral is calculated
 *             by linear interpolation of the area between two adjoining data
 *             values.
 * ------------------------------------------------------------------------- */

double convolute(double xG, double *xIn, double *yIn, double *xH, double *yH,
                 int lIn, int lH)
{
    double	yG, yhold, yh, ds;
    int		T;
    boolean	FIRSTVALUE;

    yG = 0;
    yhold = 0;
    FIRSTVALUE = TRUE; /* the first product y*h != 0 does not contribute
                          to the convolution integral since it marks the
                          lower limit of the integration range (ds = 0) */

    for (T=0; T<lIn; T++) {

        /* we only sum up those incremental areas, which lie between the
           lower and the upper limit of y*h != 0 (all the others are 0) */
    
        if (xH[0] <= xG-xIn[T] && xG-xIn[T] <= xH[lH-1]) {
           if (FIRSTVALUE) {
              ds = 0;
              FIRSTVALUE = FALSE;
           } else {
              ds = xIn[T]-xIn[T-1];
           }
            
           yh = yIn[T] * h(xG-xIn[T], xH, yH, lH);
           yG += 0.5 * (yh+yhold) * ds;
           yhold = yh; /* in order to interpolate the area between two
                          succeeding products y*h != 0, we have to store
                          the value of the last product y*h != 0 */
        }
    }

    return yG;
}


 
/* ----------------------------------------------------------------------------
 * main - main program
 * ------------------------------------------------------------------------- */

int main(int argc, char **argv)
{
  char		*date;
  char		*tmptxt;
  short		j, ii, Errors, FehlerPos, Status;
  time_t	seconds, *tloc=NULL;
  size_t	maxdate=30;
  line		*xy0;
  double	xData, yData, *xIn, *yIn;
  double	*xH, *yH, xG, yG;
  double	s, s1, sn, ds;
  complex	fh, fs;
  FILE		*outID, *inID;
  extern char	*optarg;
  extern int	opterr, optind;
  int		i, c, t, lIn, lH;
  boolean	DATASET, CLIP, LIST;

  /* ------------------------------------------------------------ get option */

  opterr = 0;
  BATCH = FALSE;
  DATASET = TRUE;
  CLIP = FALSE;
  LIST = FALSE;

  strcpy(dataset, "\0");
  strcpy(H, "\0");
  strcpy(S, "\0");
  strcpy(s_start, "\0");
  strcpy(s_end, "\0");
  strcpy(outfile, "\0");
  strcpy(infile, "\0");

  outID = stdout;
  inID = stdin;
  i = 0;

  while ((c = getopt(argc, argv, "vlcd:i:o:h:m:M:")) != EOF) {
        switch (c) {
               case 'v': fprintf(stderr, "%s\n\n%s\n\n%s\n",
                                 release, copyright, License);
                         exit(0);
                         break;
               case 'i': strcpy(infile, optarg);
                         inID = NULL;
                         i += 1;
                         break;
               case 'd': strcpy(dataset, optarg);
                         BATCH = TRUE;
                         i += 2;
                         break;
               case 'o': strcpy(outfile, optarg);
                         outID = NULL;
                         BATCH = TRUE;
                         break;
               case 'h': strcpy(H, optarg);
                         BATCH = TRUE;
                         DATASET = FALSE;
                         i += 4;
                         break;
               case 'm': strcpy(s_start, optarg);
                         BATCH = TRUE;
                         DATASET = FALSE;
                         i += 8;
                         break;
               case 'M': strcpy(s_end, optarg);
                         BATCH = TRUE;
                         DATASET = FALSE;
                         i += 16;
                         break;
               case 'l': i += 32;
                         BATCH = TRUE;
                         LIST = TRUE;
                         break;
               case 'c': CLIP = TRUE;
                         break;
        }
  }

  if (optind < argc ||
      !(i == 1 || i == 2 || i == 3 || i == 28 || i == 29 || i == 32)) {
     fprintf(stderr, "usage: convcalc -v\n");
     fprintf(stderr, "       convcalc -l\n");
     fprintf(stderr, "       convcalc -i <file> [-c]\n");
     fprintf(stderr, "       convcalc -d <dataset> [-c] [-i <file>] [-o <file>]\n");
     fprintf(stderr, "       convcalc -h <formula> -m <smin> -M <smax> [-c] [-i <file>] [-o <file>]\n");
     exit(2);
  }

  if (BATCH == FALSE) outID = NULL;

  if (inID == NULL) {
     if (access(infile, R_OK) < 0) {
        fprintf(stderr, "ERROR: read access: %s\n", infile);
        exit(2);
     }
  }

  /* -------------------------------------------------------- initialization */

  phi  = (double *) malloc(sizeof(double) * PHIBUF);
  nphi = PHIBUF;

  date     = (char *) malloc(sizeof(char) * MAXCHARS);
  formula0 = (char *) malloc(sizeof(char) * (MAXCHARS+1));
  tmptxt   = (char *) malloc(sizeof(char) * MAXCHARS);

  signal (SIGINT, SIG_IGN);

  (void) setlocale(LC_NUMERIC, "C");

  if (!BATCH) initCurses();

  signal (SIGINT, fc_exit);

  /* ----------------------------------------------------- check window size */

  if (!BATCH && (LINES < 24 || COLS < 80)) {
     printw("ERROR: sorry - minimum required window size 24/80\n");
     refresh();
     endwin();
     exit(2);
  }

  /* ------------------------------------------------------------ check date */

  seconds = time(tloc);
  (void) strftime(date,maxdate,"%a %d %b %Y (%X)",localtime(&seconds));
 
  /* --------------------------------------------------- check resource path */

  if (DATASET) {
     setupResourcePath("CCHOME", rc_path, resourcepath);

     /* -------------------------------------------- list available datasets */

     if (LIST) {
        files = scanRcDir(resourcepath, &n);
        listDatasets(files);
        exit(0);
     }
  }

  /* ----------------------------------------------------------------- intro */

  do {

    if (DATASET && !BATCH) {

       files = scanRcDir(resourcepath, &n);

       /* ------------------------------------------ resource file selection */

       ki = ni = 1;
       Menu1();

       sprintf(tmptxt, "f%d", FILENAME_MAX);
       Status = 0;
       Strncpy(dataset, (char *) (files), COLS-3);

       do {

          /* --------------------- suspend execution until window size is ok */
      
          if (BLANKSCREEN) {
             sleep(1);
             continue;
          }

          wmove(calcwin, ki-1, 2);
          wprintw(calcwin,"%s",dataset);

          if (Status == 1) {                                    /* Line down */
             if (ni < n) {
	        ni++;
                files++;
	        Strncpy(dataset, (char *) (files), COLS-3);

                if (ki < LINES-18) {
                   ki++;
                } else {
	           wscrl(calcwin, 1);
                }
             }
          }

          if (Status == -1) {                                     /* Line up */
             if (ni > 1) {
	        ni--;
                files--;
	        Strncpy(dataset, (char *) (files), COLS-3);

                if (ki > 1) {
                   ki--;
                } else {
	           wscrl(calcwin, -1);
                }
             }
          }

         wmove(calcwin, ki-1, 2);
         wstandout(calcwin);
         wprintw(calcwin,"%s",dataset);
         wstandend(calcwin);

         readDataset(resourcefile, resourcepath, dataset, 5,
                     H, S, s_start, s_end, outfile);

         showDataset(LINES-8, "01011101",
                     Prompt1, H, Prompt7, S, Prompt3, s_start, Prompt4, s_end,
                     Prompt6, outfile);

         touchwin(calcwin);
         wrefresh(calcwin);
         refresh();

         edit(3, Strlen(Prompt)+1, tmptxt, dataset, &Status);
       } while (Status != 0);

       scrollok(calcwin, FALSE);
       delwin(calcwin);
       calcwin = NULL;
    }

    /* -------------------------------------------------- read resource file */

    if (DATASET) readDataset(resourcefile, resourcepath, dataset, 5,
                             H, S, s_start, s_end, outfile);

    /* -------------------------------------------------------- display menu */

    if (!BATCH) Menu2();

    freeSymtab(TRUE);

    if (!BATCH) {

       /* ----------------------------------------------------------- edit data */

       j = 1;

       do {
         switch (j) {

         /* -------------------------------------------------------- h formular */

         case 1:
	   edit(3, Strlen(Prompt1)+1, "t254", H, &Status);
           if ((FehlerPos = checkFormula(H)) >= 0) {
              errString = (char *) &H;
              errPos    = FehlerPos;
	      reportError(helpwin, Prompt1, errString, errPos);
           } else {
	      if (Status == 1)
	        j = 7;
	      else if (Status == -1)
	        j = 6;
	      else
	        j = 0;
	   }

	   break;

         /* -------------------------------------------------------- S formular */

         case 7:
	   edit(6, Strlen(Prompt7)+1, "t254", S, &Status);
           if ((FehlerPos = checkFormula(S)) >= 0) {
              errString = (char *) &S;
              errPos    = FehlerPos;
	      reportError(helpwin, Prompt7, errString, errPos);
           } else {
	      if (Status == 1)
	        j = 3;
	      else if (Status == -1)
	        j = 1;
	      else
	        j = 0;
	   }

	   break;

	 /* ---------------------------------------------------- start value */

	 case 3:
	   edit(7, Strlen(Prompt3)+1, "t254", s_start, &Status);
           if ((FehlerPos = checkFormula(s_start)) >= 0) {
              errString = (char *) &s_start;
              errPos    = FehlerPos;
	      reportError(helpwin, Prompt3, errString, errPos);
           } else {
	      if (Status == 1)
	        j = 4;
	      else if (Status == -1)
	        j = 7;
	      else
	        j = 0;
	   }
	
	   break;

	 /* ------------------------------------------------------ end value */

	 case 4:
	   edit(8, Strlen(Prompt4)+1, "t254", s_end, &Status);
           if ((FehlerPos = checkFormula(s_end)) >= 0) {
              errString = (char *) &s_end;
              errPos    = FehlerPos;
	      reportError(helpwin, Prompt4, errString, errPos);
	   } else {
	      if (Status == 1)
	        j = 6;
	      else if (Status == -1)
	        j = 3;
	      else
	        j = 0;
	   }

	   break;

         /* -------------------------------------------------------------- file */

         case 6:
	   edit(10, Strlen(Prompt6)+1, tmptxt, outfile, &Status);

	   if (Status == 1)
	     j = 1;
	   else if (Status == -1)
	     j = 4;
	   else
	     j = 0;

	   break;

         default:
	   break;
         }
       } while (j != 0);

       wclear(helpwin);
       wrefresh(helpwin);
       delwin(helpwin);
       helpwin = NULL;
    }

    /* ------------------------------------------------ read input file data */

    if (!BATCH) SubMenu6();

    if (inID == NULL) inID = fopen(infile, "r");

    i = ALLOCINC;
    xIn = (double *) malloc(sizeof(double) * (int) i);
    yIn = (double *) malloc(sizeof(double) * (int) i);

    lIn = 0;
    ds = MAXDOUBLE;

    while (nextData(inID, &xData, &yData) != NULL) {
      if (lIn >= i) {
         i += ALLOCINC;
         xIn = (double *) realloc(xIn, sizeof(double) * (int) i);
         yIn = (double *) realloc(yIn, sizeof(double) * (int) i);
      }

      xIn[lIn] = xData;
      yIn[lIn] = yData;
      if (lIn >= 1 && xIn[lIn]-xIn[lIn-1] < ds)
         ds = xIn[lIn] - xIn[lIn-1];

      lIn++;
    };

    if (inID != stdin) {
       fclose(inID);
       inID = NULL;
    }

    /* ----------------------------------- initialize calculation parameters */

    for (iphi=0; iphi<nphi; iphi++)
        phi[iphi] = 0;

    s1 = re(evalFormula(s_start, 0));
    sn = re(evalFormula(s_end, 0));
    
    lH = (int) ceil((sn - s1) / ds + 1);
    xH = (double *) malloc(sizeof(double) * lH);
    yH = (double *) malloc(sizeof(double) * lH);

    s = s1;
    n = 0;
    runtimeError = parseError = FALSE;
    Errors = 0;

    /* ------------------------------------------------------- evaluate H(s) */

    if (!BATCH) {
       Menu3();
       wmove(calcwin, 0, 0);
    }

    j = 0;
    freeSymtab(FALSE);

    for (n=0; n<lH; n++) {

      xH[n] = s;

      /* ------------------------- suspend execution until window size is ok */
      
      if (!BATCH && BLANKSCREEN) {
         sleep(1);
         continue;
      }

      /* ----------------------------------------------- reset phase pointer */

      iphi = 1;

      /* ------------------------------------------------ continue execution */

      if (!BATCH) {
         if (n > 0)
	    wprintw(calcwin,"\n");

         wprintw(calcwin,"  n = %4d    s = % .4E :   ", n, s);

         wstandout(calcwin);
      }

      fs = evalFormula(S, s);
      if (!BATCH) wstandend(calcwin);

      if (runtimeError || parseError) {
	 Errors++;
	 runtimeError = parseError = FALSE;
         if (!BATCH) wprintw(calcwin," ");
      } else {

         if (!BATCH) wstandout(calcwin);
         fh = evalFormula(H, s);
         if (!BATCH) wstandend(calcwin);

         if (runtimeError || parseError) {
	    Errors++;
	    runtimeError = parseError = FALSE;
            if (!BATCH) wprintw(calcwin," ");
         } else
            if (!BATCH) wprintw(calcwin," h = % .4E     ", fh.re);
      }

      if (Errors == 0) {
        yH[n] = fh.re;
      } else {
        yH[n] = 0;
	Errors = 0;
      }

      sn = s;
      s += ds;

      if (!BATCH) {
         wrefresh(calcwin);
         flushinp();     /* avoid hangups due to pending keystrokes */
      }
    }

    /* -------------------------------- enlarge input file data if necessary */

    if (xH[lH-1]-xH[0] > xIn[lIn-1]-xIn[0]) {
       i = lIn + (int) ceil(((xH[lH-1]-xH[0]) - (xIn[lIn-1]-xIn[0])) / ds + 1);
       xIn = (double *) realloc(xIn, sizeof(double) * (int) i);
       yIn = (double *) realloc(yIn, sizeof(double) * (int) i);

       while (lIn < i) {
             xIn[lIn] = xIn[lIn-1] + ds;
             yIn[lIn] = 0;
             lIn++;
       }
    }

    /* ---------------------------------------------- initialize output data */

    if (!BATCH) SubMenu7();

    i = lIn + 10 + lH;
    xy0 = (line *) malloc(sizeof(line) * (int) i);
    xy  = xy0;
    nn = 0;

    sprintf((char *) xy++, "# From   : %s", release); nn++;
    sprintf((char *) xy++, "# Date   : %s", date); nn++;
    sprintf((char *) xy++, "# %s%s", Prompt1, H); nn++;
    sprintf((char *) xy++, "# %s%s", Prompt7, S); nn++;
    sprintf((char *) xy++, "# %s%s", Prompt3, s_start); nn++;
    sprintf((char *) xy++, "# %s%s", Prompt4, s_end); nn++;
    strcpy((char *) xy++, "\0"); nn++;

    /* ----------------------------------------------- convolute 1st section */

    t = 0;
    while (xIn[t]-(xH[lH-1]-xH[0]) < xIn[0]) {
        if (!CLIP) {
           xG = xIn[t]+xH[0];
           yG = convolute(xG, xIn, yIn, xH, yH, lIn, lH);
	   sprintf((char *) xy++, "   % .8E   % .8E", xG, yG); nn++;
	}
	t++;
    }

    /* ---------------------------------------------- convolute main section */

    while (t < lIn-1) {
        xG = xIn[t]+xH[0];
        yG = convolute(xG, xIn, yIn, xH, yH, lIn, lH);
	sprintf((char *) xy++, "   % .8E   % .8E", xG, yG); nn++;
	t++;
    }

    /* ---------------------------------------------- convolute last section */

    t = lIn-1;
    while (xIn[t] > xIn[lIn-1]-(xH[lH-1]-xH[0])) {
          t--;
    }

    do {
        t++;
        if (!CLIP) {
           xG = xIn[t]+xH[lH-1];
           yG = convolute(xG, xIn, yIn, xH, yH, lIn, lH);
	   sprintf((char *) xy++, "   % .8E   % .8E", xG, yG); nn++;
	}
    } while (t < lIn-2);

    /* ------------------ view, plot, save data, continue or quit evaluation */

    if (!BATCH) {
       Menu4();
       flushinp();     /* avoid hangups due to pending keystrokes */
    }

    do {
      if (!BATCH) {
         ii = GetKey();
      } else {
         ii = 's';
      }

      /* --------------------------------------------------------- view data */

      if (ii == 'v') {
         xy = xy0;
         ni = (LINES-17) + 1;
         Menu5();

         moreViewer(&calcwin, xy0, &xy, &xyn, nn, &ni);

         Menu4();

	 standout();
	 writeXY(LINES-2, 1, Menu, 0);
	 standend();
	 addstr("? ");
	 move(LINES-2, Strlen(Menu)+1);
      }

      /* --------------------------------------------------------- plot data */

      if (ii == 'p') {
         plot2Data(xIn, yIn, lIn, xy0, nn);
         wrefresh(curscr);
      }

      /* --------------------------------------------------------- save data */

      if (ii == 's') {
         xy = xy0;
         if (BATCH) ii = 'Q';

         saveData(outID, outfile, nn, xy, Menu);
      }
    } while (ii != 'c' && ii != 'C' && ii != 'q' && ii != 'Q');

    free(xy0);

    /* ------------------------------------------ save evaluation parameters */

    if (ii == 'c' || ii == 'q') {
       saveDataset(calcwin, resourcefile, 5, H, S, s_start, s_end, outfile);
    }

    if (!BATCH) {
       scrollok(calcwin, FALSE);
       delwin(calcwin);
       calcwin = NULL;
    }

  } while (ii != 'q' && ii != 'Q');

  if (!BATCH) writeXY(LINES-2, 59, "", 1);
  fc_exit();
}
