#include <stdio.h>
#include <math.h>
#include <float.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>

#include "calcfloat.h"

extern long double lgammal(long double);

extern int strictmode;
extern disp_mode dispmode;

CalcFloat::CalcFloat(void)
{
  value = 0.0;
  check_value();
}


CalcFloat::CalcFloat(CalcFloat const &from)
{
  value = from.value;
  check_value();
}


CalcFloat::CalcFloat(double const &from)
{
  value = from;
  check_value();
}


CalcFloat::CalcFloat(int const &from)
{
  value = from;
  check_value();
}


/* Given a number of digits after the decimal to display, the mode of
 * the display, a buffer to store the printable value, the size of
 * that buffer, and pointers to storage which will hold the number of
 * rows and columns required for the output, produce an appropriate
 * printable value for the number stored. */
void CalcFloat::printvalue(int ndigits, disp_mode dmode, char *buffer,
			   int bsize, int *rows, int *cols) const 
{
  static double uplim = 0.0;
  double aval = fabs(value);
  int decimals;
  int wholepartdigs;
  intstring anint;


// First, if the value is unprintable, produce the error message and return
  if (!value_ok()) {
    strncpy(buffer, FLTERRORSTR, bsize - 1);
    if (rows != NULL) *rows = 1;
    if (cols != NULL) *cols = strlen(buffer);
    return;
  }

// we figure out the largest number which can be represented in fixed point
  if (uplim == 0.0)
    uplim = pow(10.0, (double) MAXDIGS);

  decimals = ndigits;
  switch (dmode) {
  case HEX:
    if (aval <= ULONG_MAX) {  /* Fits in a long */
      snprintf(buffer, bsize, "%s%s%lx", value < 0 ? "-" : "",  HEXPREFIX, 
	       (unsigned long) aval);
      break;
    }
    /* Fall through to FIX */
  case FIX:            // try to print in fixed point
    if (!(aval >= uplim ||   // should not be bigger than this
	  (value != 0.0 &&   // should not be so small as to appear as zero
	   aval < 0.5 * pow(10.0, (double) -ndigits)))) {

/* No matter how many digits we are asked for, never give more than
 * the maximum display resolution, MAXDIGS */
      if (aval > 1) {
	wholepartdigs = (int) floor(log10(aval));
	if (decimals + wholepartdigs > MAXDIGS - 1)
	  decimals = MAXDIGS - 1 - wholepartdigs;
      }

      if (snprintf(buffer, bsize,"%.*f", decimals, value) == bsize) {
	fprintf(stderr, "Number too large for output buffer\n");
	abort();
      }

/* sprintf doesn't append a decimal point if the precision is
 * zero. For consistency in automatic parsing of the output, we stick
 * the decimal point back on */
      if (strchr(buffer, '.') == NULL)  // No decimal point
	append_char(buffer, '.');

      break;
    }     // else it falls through to SCI
  case SCI:
    if (snprintf(buffer, bsize, "%.*e", decimals, value) == bsize) {
      fprintf(stderr, "Number too large for output buffer\n");
      abort();
    }

    {
      char *dptr = strchr(buffer, '.');  // the decimal point
      char *eptr = strchr(buffer, 'e');  // the 'e' marking the exponent
      int expn = atoi(eptr + 1);         // the value of the exponent
      intstring anint;
      
      sprintf(anint, "%+04d", expn);     // recast it with sign and leading 0s

      if (dptr == NULL) {   // no decimal point, put one in.
	*eptr = '.';   // where the 'e' used to be
	*(eptr + 1) = 'e';  // so, construct 'e' after it
	*(eptr + 2) = 0;
      } else {
	*(eptr + 1) = 0;
      }
      strcat(buffer, anint);  // tack the exponent back on the end
    }
    break;
  case ENG:  // like SCI, but the exponent is a multiple of 3
    if (snprintf(buffer, bsize, "%.*e", decimals, value) == bsize) {
      fprintf(stderr, "Number too large for output buffer\n");
      abort();
    }
    
    char *eptr = strchr(buffer, 'e');  // Same 'e' as in the snprintf
    int expn = atoi(eptr + 1);
    *(eptr + 1) = 0;
    while (expn % 3) {
      char *dptr = strchr(buffer, '.');
      if (dptr == NULL) {  // no decimal point present, put one in
	*(eptr + 2) = 0;
	*(eptr + 1) = 'e';
	*eptr = '.';
	dptr = eptr;
	eptr++;
      }

      if (dptr + 1 == eptr)  {    // decimal is immediately before 'e'
	*(eptr + 2) = 0;    // move the 'e' over
	*(eptr + 1) = 'e';
	*eptr = '.';        // put a '0' before the decimal point
	*dptr = '0';
	eptr++;
	expn--;
      } else {   // swap the decimal and the number to the right of it
	char temp = *(dptr + 1);
	*(dptr + 1) = *dptr;
	*dptr = temp;
	expn--;
      }
    }
    sprintf(anint, "%+04d", expn);
    strcat(buffer, anint);
    
// strip multiple leading zeroes
    while (buffer[0] == '0' &&
	   buffer[1] == '0') {
      char *cptr = buffer;
      
      while (*cptr != 0) {
	*cptr = *(cptr + 1);
	cptr++;
      }
	
    }
    break;
  }

  if (rows != NULL) *rows = 1;
  if (cols != NULL) *cols = strlen(buffer);
}



/* A very simple test. A floating point number must pass through *
 * strtod leaving after itself only whitespace and comments. If in hex
 * mode, check to see if the number is a valid hex string. */
parse_codes CalcFloat::parsevalue(char const *number, char **remainder) 
{
  int negnum = 0;

  if (dispmode == HEX) {
    if (number[0] == '-') negnum = 1;

    value = strtoul(number + negnum, remainder, 0);

  } else {
    value = strtod(number, remainder);
  }

  check_value();
  if (*remainder == number)
    return PARSE_BAD;  /* Got nothing from this */
  
  if (**remainder != 0) {
      
    *remainder += strspn(*remainder, WHITESPACE);
    if (**remainder == 0 || **remainder == COMMENT)
      return PARSE_OK;
      
    return PARSE_BAD;
  }
  return PARSE_OK;
}



CalcFloat &CalcFloat::operator= (double from)
{
  this->value = from;
  check_value();
  return *this;
}


CalcFloat &CalcFloat::operator= (int from)
{
  this->value = from;
  check_value();
  return *this;
}

CalcFloat &CalcFloat::operator= (Arithdatatype const &from)
{
  if (&from == NULL) {
    this->value = 0.0;
  } else {
    this->value = ((CalcFloat &)from).value;
  }
  check_value();
  return *this;    
}

CalcFloat &CalcFloat::operator= (CalcFloat const &from)
{
  if (&from == NULL) {
    this->value = 0.0;
  } else {
    this->value = from.value;
  }
  check_value();
  return *this;    
}


CalcFloat &CalcFloat::sin(void)
{
  this->value = ::sin(this->value);
  return *this;
}


CalcFloat &CalcFloat::cos(void)
{
  this->value = ::cos(this->value);
  return *this;
}


CalcFloat &CalcFloat::tan(void)
{
  this->value = ::tan(this->value);
  return *this;
}


CalcFloat &CalcFloat::arcsin(void)
{
  this->value = asin(this->value);
  return *this;
}


CalcFloat &CalcFloat::arccos(void)
{
  this->value = acos(this->value);
  return *this;
}


CalcFloat &CalcFloat::arctan(void)
{
  this->value = atan(this->value);
  return *this;
}


CalcFloat &CalcFloat::sqrt(void)
{
  this->value = ::sqrt(this->value);
  return *this;
}


CalcFloat &CalcFloat::square(void)
{
  this->value = this->value * this->value;
  return *this;
}


CalcFloat &CalcFloat::log_e(void)
{
  this->value = log(this->value);
  return *this;
}


CalcFloat &CalcFloat::exp_e(void)
{
  this->value = exp(this->value);
  return *this;
}


CalcFloat &CalcFloat::log_10(void)
{
  this->value = log10(this->value);
  return *this;
}


CalcFloat &CalcFloat::exp_10(void)
{
  this->value = pow(10.0, this->value);
  return *this;
}


CalcFloat &CalcFloat::recip(void)
{
  this->value = 1.0 / this->value;
  return *this;
}


CalcFloat &CalcFloat::power(Arithdatatype const &exponent)
{
  CalcFloat &ff = (CalcFloat &) exponent;
  this->value = pow(this->value, ff.value);
  return *this;
}


CalcFloat &CalcFloat::abs(void)
{
  this->value = ::fabs(this->value);
  return *this;
}


double factorial(double);

CalcFloat &CalcFloat::factorial(void)
{
  this->value = ::factorial(this->value);
  return *this;
}



double factorial(double x)
{
  double wholepart;
  double fracpart;
  double fval;

  fracpart = modf(x, &wholepart);

#ifdef HAVE_LONG_DOUBLE_LIBM
  fval = expl(lgammal(x + 1));
#else
#warning Without a libm supporting long doubles your factorial
#warning results may be uncertain in the 14th or 15th decimal place
  fval = exp(lgamma(x + 1));
#endif

  if (fracpart == 0) /* If the number was an integer, make sure that
		      * the factorial is an integer, too. */
    fracpart = modf(fval + 0.5, &fval);

  return fval;
}


void CalcFloat::rec2polar(Arithdatatype *x, Arithdatatype *y)
{
  CalcFloat *px = (CalcFloat *)x;
  CalcFloat *py = (CalcFloat *)y;
  CalcFloat ang, rad;

  ang = atan2(py->value, px->value);
  rad = ::sqrt(px->value * px->value + py->value * py->value);
  *px = rad;
  *py = ang;  
}



void CalcFloat::polar2rec(Arithdatatype *r, Arithdatatype *theta)
{
  CalcFloat *pr = (CalcFloat *)r;
  CalcFloat *ptheta = (CalcFloat *)theta;
  CalcFloat x, y;

  x = pr->value * ::cos(ptheta->value);
  y = pr->value * ::sin(ptheta->value);
  *pr = x;
  *ptheta = y;
}

