/****************************************************************************
*                                                                           *
*  The functions in this file are used by PBCTRL to get antenna pointing    *
*  data from PREDICT.  They are modelled directly on the functions of the   *
*  demonstration program 'demo.c' that is supplied as part of the PREDICT   *
*  package by John Magliacane, KD2BD.                                       *
*                                                                           *
*  The functions defaults to "localhost" as the name of machine on which    *
*  PREDICT is running in server mode.                                       *
*                                                                           *
*****************************************************************************
*                                                                           *
* This program was originally written by Ivan Galysh, KD4HBO on 24-Jan-2000 *
*    It was last modified by John A. Magliacane, KD2BD on 17-Oct-2001.      *
*    Bent Bagger, OZ6BL modified it on 18-Jul-2002 for use by PBCTRL        *
*                                                                           *
*****************************************************************************
*                                                                           *
* This program is free software; you can redistribute it and/or modify it   *
* under the terms of the GNU General Public License as published by the     *
* Free Software Foundation; either version 2 of the License or any later    *
* version.                                                                  *
*                                                                           *
* This program is distributed in the hope that it will useful, but WITHOUT  *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or     *
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License     *
* for more details.                                                         *
*                                                                           *
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <signal.h>
#include <errno.h>
#include <arpa/inet.h>

#include "predict-client.h"
#include "connect-client.h"

//define DEBUG 0

char	phostname[100];          /* Host Name of predict server */
char	pportname[] = "predict"; /* Port Name of predict service */

char	pstring[625];            /* Global buffer for transferring data to/from predict */
int	psock = -1;              /* Global socket for talking to predict. Make sure it starts up with an invalid value */

int get_predict_server (void) {
	char s1[100], s2[100];
	FILE *paramf;

	phostname[0] = '\0'; // Give it a known, empty value
	/* Open and read the parameter file pb.conf */
	if ((paramf = fopen ("/etc/pbpg/pb.conf", "r")) == NULL) {
		perror ("pb - open pb.conf");
		exit (1);
	}
	while (! feof (paramf)) {
		fscanf (paramf, "%100s %100s\n", s1, s2);
		if (!strcmp (s1, "PREDICT_SERVER")) { strncpy (phostname, s2, sizeof(phostname)); break; }
	}
	if (strlen(phostname) == 0) // Not found. Use 'localhost'
		strncpy(phostname, "localhost",sizeof(phostname));
	return 0;
}

int connectsock ()
{
	/* This function is used to connect to the predict server. 
	   Host name and port name are stored in phostname / pportname.
	   The protocol is always UDP */

	struct hostent *phe;
	struct servent *pse;
	struct protoent *ppe;
	struct sockaddr_in sin;
	
	int s, type;

	/* First of all check if psock have a value.  If so just reuse the existing socket and ignore the request */
	if (psock > -1) return psock;

	
	bzero((char *)&sin, sizeof(struct sockaddr_in));
	sin.sin_family=AF_INET;
	
	if ((pse=getservbyname(pportname,"udp")))
		sin.sin_port=pse->s_port;

	else if ((sin.sin_port=htons((unsigned short)atoi(pportname))) == 0) {
		printf("Can't get services\n");
		return -1;
	}

	if ((phe=gethostbyname(phostname)))
		bcopy(phe->h_addr,(char *)&sin.sin_addr,phe->h_length);

	else if ((sin.sin_addr.s_addr=inet_addr(phostname)) == INADDR_NONE) {
		printf("Can't get host: \"%s\".\n",phostname);
		return -1;
	}
	
	if ((ppe=getprotobyname("udp")) == 0)
		return -1;

	type=SOCK_DGRAM;
	
	s=socket(PF_INET,type,ppe->p_proto);

	if (s < 0) {
		printf("Can't create socket.\n");
		return -1;
	}

	if (connect(s,(struct sockaddr *)&sin,sizeof(sin)) < 0) {
		printf("Can't connect to socket.\n");
		return -1;
	}

	psock = s;
	return s;
}


char *send_command (char *command)
{
	int	retry = 5;		/* Retry counter */
	int	n;
	fd_set	readfds, writefds, exceptfds;
	struct	timeval	timeout;

	/* This function sends "command" to PREDICT running on
	   machine "host", and returns the result of the command
	   as a pointer to a character string. */

	while (retry-- > 0) {
		/* Make sure the network socket is open */
		//if (connectsock() < 0) exit (-1);	/* Not good if this fails */
		psock=connect_client(phostname,pportname,AF_UNSPEC, SOCK_DGRAM);
		
		if (psock == -1)
			exit(-1);
			
		/* Build a command buffer */
		snprintf(pstring, sizeof(pstring),"%s\n",command);
#ifdef DEBUG
		printf ("Sending %s", pstring);
#endif
		/* Send the command to the server */
   		write(psock,pstring,strlen(pstring));

		/* clear pstring[] so it can be re-used for the response */
		pstring[0]=0;

		/* Wait for the response (but only for a reasonable time */
		FD_ZERO (&readfds);
		FD_ZERO (&writefds);
		FD_ZERO (&exceptfds);
		FD_SET  (psock, &readfds);
		timeout.tv_sec  = 0;
		timeout.tv_usec = 500000;
		n = select (32, &readfds, &writefds, &exceptfds, &timeout);
		if (n >= 0 && FD_ISSET (psock, &readfds)) {
			/* Get the answer from the socket */
			n = read (psock, &pstring, 625);
			pstring [n] = 0;		/* Terminate the string properly */
#ifdef DEBUG
			printf ("Received %s\n", pstring);
#endif
			return pstring;
		}
		/* Apparently no timely reply received.  Clear the socket and try again */
#ifdef DEBUG
		printf ("Receiver timeout on retry %d\n", retry);
#endif
		close (psock);
		psock = -1;
	}

	/* If the program gets here it ran out of retry attempts.  Really annoying...!!!! */
	pstring [0] = 0;
	return pstring;
}


int initialize_predict_client (char *SatName) {

	char *reply;

	/* Try to get the list of satellite names from PREDICT */
	get_predict_server();
	reply = send_command("GET_LIST");
	if (strlen(reply) == 0) {
		//printf("Predict not running\n");
		return 0 ;
	}

	/* See if our satellite is amongst the ones known by PREDICT */
	if (strstr(pstring, SatName) == NULL) {
		//printf("Satellite %s not served by Predict\n", SatName);
		return 0;
	}
	return 1 ;
}

int get_antenna_data(char *SatName, float *az, float *el, float *dop) {
	char command [SIZEOFBUF];
	int x, y;
	char *reply;

	sprintf(command,"GET_SAT %s", SatName);
	if ((strlen(reply = send_command(command))) == 0) return 0 ;

	/* Parse the response from GET_SAT */

	/* The first element of the response is the satellite name.
	   It is terminated by a '\n' character and may contain spaces. */

	for (x=0; pstring[x]!='\n'; x++);
	x++;

	/* The rest of the data from GET_SAT is numerical, and
	   can be parsed using the sscanf() function.  First, the
	   satellite name is removed from "pstring", and then "pstring"
	   is parsed for numerical data using a sscanf(). */

	for (y=0; pstring[x+y]!=0; y++)
		pstring[y]=pstring[x+y];
	pstring[y]=0;
	sscanf(pstring,"%*f %*f %f %f %*ld %*f %*f %*f %*f %*ld %*c", az, el);

	sprintf(command,"GET_DOPPLER %s", SatName);
	if ((strlen(reply = send_command(command))) == 0) return 0 ;
	sscanf(pstring,"%f", dop);
	return 1;
}


