/*
 * twcwCB.c:  A gui application that sends Morse Code 
 * Copyright (C) 1997 Ted Williams WA0EIR 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
 * Version: 1.4 JAN 2008
 */

#include "common.h"
#include "twcw.h"
#include <sys/shm.h>

#define XMIT "1"
#define STANDBY "0"
#define TEXT  0
#define INSERT 1

extern AppRes app_res;

/*
 * txTextCB Callback
 * Modify/Verify Callback for txText 
 */
static int j;                           /* position in current word */
static char savech;
long int old_end = -1;

void txTextCB (Widget w, XtPointer client_data, XtPointer cdata)
{
   int  i;
   static char word[MAX_WORD_LENGTH];
   XmTextVerifyCallbackStruct *cbs = (XmTextVerifyCallbackStruct *) cdata;

   /* force cursor to end of txText */
   if (cbs->reason == XmCR_VALUE_CHANGED)
   {
      XmTextSetInsertionPosition (w, XmTextGetLastPosition (w));
      return;
   }
   /* Don't allow Big Delete */
   if (cbs->startPos <= old_end)
   {
      cbs->doit = False;  
      XmTextSetInsertionPosition (w, XmTextGetLastPosition (w));
      return;
   }
   /* But allow Back Space */
   if (cbs->endPos - cbs->startPos == 1)
   {
      j--;
      if (word[j] > 0 && word[j] < 7)  /* if this is a prosign */
      {
         XmTextReplace (w,
            XmTextGetLastPosition (w) - 2, 
            XmTextGetLastPosition (w),
            "\0");
      }
      return;
   }
   /* Now all to upper case and check for complete works */
   for (i = 0; i < cbs->text->length; i++)
   {
      cbs->text->ptr[i] = toupper (cbs->text->ptr[i]);
      word[j] = cbs->text->ptr[i];

      /* and look for a complete words */
      if (word[j] == ' ' || word[j] == '\n')
      {
         savech = word[j];
         word[j+1] = '\0';
         send_msg (WORD_MSG, word);     /* Send the word to the queue */
         word[j] = savech;
         j = 0;
         old_end = cbs->endPos;
      }
      else
      /* not a complete word, so look for zero and prosigns */
      {
         if (cbs->text->ptr[i] == '0')
         {
            cbs->text->ptr[i] = '\330'; /* make zeros have a slash */
            word[j] = '\330';
         }
         i = DoProSigns(cbs, i);        /* check for pro signs */
         j++;
      }
   }
}

/*
 * rxTextCB Callback - If its a value changed CB, set the cursor to
 * the last position.  Otherwise, see if we gota Pro Sign and add it
 * to the end of the text.  This callback is not registered when in
 * CWIRC mode
 */
void rxTextCB (Widget w,XtPointer client_data, XtPointer call_data)
{
   int i;
   XmTextVerifyCallbackStruct *cbs = (XmTextVerifyCallbackStruct *) call_data;

   /* Keep the cursor at the end */
   if (cbs->reason == XmCR_VALUE_CHANGED)
   {
      XmTextSetInsertionPosition (w, XmTextGetLastPosition (w));
      return;
   }

   for (i = 0; i < cbs->text->length; i++)
   {
      i = DoProSigns(cbs, i);           /* check for pro signs */
   }
}


/*
 * DoProSign Function
 * Check for a pro signs character in the text string and
 * if found, insert pro sign char strings in the text widgets.
 * Used by txText and rxText modify/verify callback.
 */
int DoProSigns(XmTextVerifyCallbackStruct *cbs, int i)
{
  switch (cbs->text->ptr[i])
  {
      case 1:                          /* Alt-Ctrl-F1 is printed as "er" */
         MkRoom (cbs,i);
         cbs->text->ptr[i++] = 'e';
         cbs->text->ptr[i] = 'r';
         break;

      case 2:                          /* Alt-Ctrl-F2 is printed as "ar" */
         MkRoom (cbs, i);
         cbs->text->ptr[i++] = 'a';
         cbs->text->ptr[i] = 'r';
         break;

      case 3:                          /* Alt-Ctrl-F3 is printed as "sk" */
         MkRoom (cbs, i);
         cbs->text->ptr[i++] = 's';
         cbs->text->ptr[i] = 'k';
         break;

      case 4:                          /* Alt-Ctrl-F4 is printed as "kn" */
         MkRoom (cbs, i);
         cbs->text->ptr[i++] = 'k';
         cbs->text->ptr[i] = 'n';
         break;

      case 5:                          /* Alt-Ctrl-F5 is printed as "as" */
         MkRoom (cbs, i);
         cbs->text->ptr[i++] = 'a';
         cbs->text->ptr[i] = 's';
         break;

      case 6:                          /* Alt-Ctrl-F6 is printed as "bk" */
         MkRoom (cbs, i);
         cbs->text->ptr[i++] = 'b';
         cbs->text->ptr[i] = 'k';
         break;

      default:
         break;
   }
   return i;
}


/*
 * MkRoom FUNCTION
 * Called by DoProSigns if a pro sign character is found in a text string.
 * This will make room for the pro sign's two character string in the text
 * widgit.
 */
void MkRoom (XmTextVerifyCallbackStruct *cbs, int i)
{
   int k;

   cbs->text->ptr = XtRealloc (cbs->text->ptr, cbs->text->length + 2);
   cbs->text->length = cbs->text->length + 1;
   for (k=cbs->text->length; k>i+1; k--)
   {
      cbs->text->ptr[k] = cbs->text->ptr[k-1];
   }
   return;
}


/*
 * fileCB FUNCTION
 * Activate Callback for File Button
 */
void fileCB (Widget w, XtPointer client_data, XtPointer call_data)
{
   int btn = (int) client_data;
   AppRes *resptr;
   char *filename;
   char *text;
   /*
    *    Get the dir path and filename, malloc space, and cat them   
    */
   XtVaGetValues (w,
      XmNuserData, (AppRes *) &resptr,
      NULL);

   filename = (char *) XtMalloc (strlen (dirpath)
                               + strlen (resptr->buttonNames[btn]) + 2);
   strcpy (filename, dirpath);
   strcat (filename, "/");            /* could have // but doesn't hurt */
   strcat (filename, resptr->buttonNames[btn]);

   /* 
    * Read the file contents and call the work proc to process it.
    * If getFile returns NULL, do nothing because getFile has caught 
    * the error and called errorDiag.
    */
   if ((text = getFile (filename)) != NULL)                          
   {
      XtAppAddWorkProc (ac, procData, (XtPointer) text);  
      XtFree (filename);
   }
} 


/*
 * getFile Function
 * Passed a pathname, this malloc's space for the file
 * text and reads the file.
 */
char *getFile (char *filename)
{
   FILE *fp;
   struct stat info;
   char *pt;

   if (stat (filename, &info) < 0)
   {
       errorDiag (shell,
                  "Can't find the file you requested.\n"
                  "This could be a configuration error.\n"
                  "Do the button names match your file names?",
                  NO_CANCEL);
       return (NULL);
   }

   if ((pt = (char *)XtMalloc ((int)info.st_size + 2)) == NULL)
   {
       errorDiag (shell,
                  "getFile: XtMalloc failed\n"
                  "twcw must exit",
                  NO_CANCEL);
       return (NULL);
   }

   if ((fp = fopen (filename, "r")) == NULL)   
   {
       errorDiag (shell,
                  "getFile: fopen failed\n"
                  "twcw must exit",
                  NO_CANCEL);
       return (NULL);
   }

   if (fread (pt, 1, info.st_size, fp) != (int)(info.st_size))
   {
       errorDiag (shell,
                  "getFile: fread failed\n"
                  "twcw must exit",
                  NO_CANCEL);
       return (NULL);
   }  
   pt[info.st_size] = '\0';
   return (pt);
} 

/*
 * procData FUNCTION - Work procedure for sending files.
 * Parse the data for strings of words, and passes then to the text
 * widget. Anytime a "*word" is found, use the word as a prompt in a
 * dialog box to get the text to insert.
 */
Boolean procData ( XtPointer data )
{   
   static char *start, *ptr, *begin;
   static Boolean first = True;
   static int ToDo = TEXT;
   XmTextPosition pos = 0;   
   int i = 0;

   if (first == True)
   {
      start = (char *) data;
      begin = (char *) data;           /* keep this to free later */ 
      first = False;
   }

   switch (ToDo)
   {
      case TEXT:                       /* send file text and check for *words */
         while (start[i] != '\0')
         {
            if (start[i] != '*')
            {
               i++;                    /* any character but '*' */
               continue;
            }
            else
            {
               start[i] = '\0';        /* found a *word, * is now '\0' */
               pos = XmTextGetLastPosition (txText);
               XmTextInsert (txText, pos, start);
               i++;                    /* all text so far is now in txText */
               start = start + i;
               ToDo = INSERT;          /* make us go to the other case next */              
               return (False);         /* reschedule the work proc */
            }
         }
         pos = XmTextGetLastPosition (txText);
         XmTextInsert (txText, pos, start);
         first = True;
         XtFree (begin);               /* Free space used by file */
         return (True);                /* All done so don't resked work proc */

      case INSERT:                     /* Inserts text for *word */
         ptr = start;
         while ( *start != ' ' && *start != '\n')
            start++;
         *start = '\0';                /* "*word " now is "word\0" */
         inputDiag (ptr);              /* popup the prompt dialog */
         start++;
         ToDo = TEXT;
         return (True);                /* Don't reschedule the work proc, */
                                       /* The diag will reschedule it */

      default:
         printf ("We never get here :-( \n");
   }
   return (False);                     /* We never get here either */
}  


/*
 * trCB FUNCTION
 * Value Changed Callback for Recv amd Xmit TB
 * Checks Recv state and send the appropriate STOP_MSG
 */
void trCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   Pixel bgcolor;
   XmToggleButtonCallbackStruct *ptr =
      (XmToggleButtonCallbackStruct *) cbs;

   if (ptr->set)
   {
      send_msg (STOP_MSG, XMIT);           /* Send a XMIT msg if set */
      // TJW 2.0 init_audio();

      XtVaGetValues (scTB,
         XmNselectColor, &bgcolor,
         NULL);

      XtVaSetValues (w,
         XmNbackground, bgcolor,
         NULL);
   }
   else
   {
      send_msg (STOP_MSG, STANDBY);        /* Send a STANDBY msg if not set */
      // TWJ 2.0 close_sound();

      XtVaGetValues (scTB,
         XmNbackground, &bgcolor,
         NULL);

      XtVaSetValues (w,
         XmNbackground, bgcolor,
         NULL);
   }
   XmProcessTraversal (txText, XmTRAVERSE_CURRENT);
}


/*
 * keyTBCB FUNCTION
 * Value Changed Callback for ToggleButton 2 (SC and Rig)
 */
void keyTBCB (Widget w, XtPointer client_data, XtPointer call_data)
{
   char val[2];
   int stateSC, statePC;

   stateSC = XmToggleButtonGetState (scTB) * 2;
   statePC = XmToggleButtonGetState (rigTB);
   sprintf (val, "%d", stateSC + statePC);

   send_msg (KEY_DEVICES_MSG, val);      /* Send a KEY msg with "0" */
   XmProcessTraversal (txText, XmTRAVERSE_CURRENT);
}


/*
 * toneCB for tone
 */
void toneCB (Widget w, XtPointer client_data, XtPointer call_data)
{
   int tone;
   char val[6];
   XmScaleCallbackStruct *pt = (XmScaleCallbackStruct *) call_data;

   XmScaleGetValue (w, &tone);
   tone = pt->value - pt->value % 10;   
   XmScaleSetValue (w, tone);
   sprintf (val, "%d", tone);
   send_msg (FREQ_MSG, val);              
   XmProcessTraversal (txText, XmTRAVERSE_CURRENT);
}


/*
 * practiceCB FUNCTION
 * Activate Callback for Practice Button
 */
void practiceTBCB (Widget w, XtPointer client_data, XtPointer call_data)
{
   Boolean state;
   char val[2];

   pcnt = 6;            /* set pcnt to 6 regardless */
   state = XmToggleButtonGetState (w);
   sprintf (val, "%d\n", state);

   /* set txText sensitive to 0 if practice or 1 if !practice */
   XtVaSetValues (txText,
      XmNsensitive, !state,
      NULL);

   if (state == True)   /* if we are setting practice */
   {
      /* then send a WORD_MSG to strart things */
      send_msg (WORD_MSG, "\n");
   }
}


/*
 * input_event Function
 * Catches Button 3 presses and pops up the popup menu. Also
 * catches enter window events on textf2 and txText and forces
 * focus to that window. 
 */
void input_event (Widget w, XtPointer client_data, XEvent *event, Boolean *cont)
{
   Widget popup = (Widget) client_data;

   if (event->type == ButtonPress && event->xbutton.button == Button3)
   {
      XmMenuPosition(popup, &(event->xbutton));
      XtManageChild(popup);
   }
   else  /* keep focus in the txText window */
   {
      XmProcessTraversal(txText, XmTRAVERSE_CURRENT);
   }
}


/*
 * popupCB Callback
 * Popup menu callback
 */
void popupCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   int reason = (int) client_data;
   int rtn;
   struct tag1 message;

   switch (reason)
   {
      case CLR_SEND:
         XtRemoveCallback (txText, XmNmodifyVerifyCallback, txTextCB, NULL);
         XtVaSetValues(txText,
            XmNvalue, "",
            XmNcursorPosition, 0,
            NULL);
         old_end = -1;
         XtAddCallback (txText, XmNmodifyVerifyCallback, txTextCB, NULL);
         break;

      case CLR_RCVD:
         XmTextSetString(rxText, "");
         break;

      case CLR_ALL:       /* Flush the Msg Queue and clear both text fields */
         while (1)
         {
            rtn = msgrcv (qid, (struct msgbuf *) &message,
                          MAX_WORD_LENGTH, 0, IPC_NOWAIT);
            if (rtn == -1)
            {
               if (errno == ENOMSG)
               {
                  break;
               }
               else
               {
                  perror ("popupCB flushing buffer:");
               }
            }
         }

         XtRemoveCallback (txText, XmNmodifyVerifyCallback, txTextCB, NULL);
         XtVaSetValues(txText,
            XmNvalue, "",
            XmNcursorPosition, 0,
            NULL);
         old_end = -1;
         XtAddCallback (txText, XmNmodifyVerifyCallback, txTextCB, NULL);

         /* if TWCW mode, then clear rxText */
         if (cwArgs.mode == TWCW)
         {
            XmTextSetString(rxText, "");         
         }
         break;

      case QRT:
         quitCB (w, client_data, cbs);
         break;
         
      default:
         printf ("Do nothing\n");
   }
}


/*
 * popupHelp Callback 
 */

void helpCB (Widget w, XtPointer client_data, XtPointer call_data)
{
int i;
Arg args[15];
static Widget helpDiag, helpTxt;
XmString xs;
char *textpt;

   if (helpDiag == NULL)
   {
      if ((textpt = getFile (helppath)) == NULL)      /* Read the help file */
      {
         /* we should never get here, because we already found the helpfile */
         /* but if we do, getFile has detected an error and errorDiag */
         /* has been called so just return and wait for our death */
         return;
      }

      helpDiag = XmCreateInformationDialog(shell, "helpDiag", NULL, 0);
      XtVaSetValues (XtParent(helpDiag),
         XmNtitle, "TWCW HELP FILE",
         NULL);
      XtUnmanageChild (XmMessageBoxGetChild (helpDiag, XmDIALOG_CANCEL_BUTTON));
      XtUnmanageChild (XmMessageBoxGetChild (helpDiag, XmDIALOG_HELP_BUTTON));

      xs = XmStringCreateLocalized ("TWCW HELP FILE");
      XtVaSetValues (helpDiag,
         XmNdialogStyle, XmDIALOG_MODELESS,
         XmNmessageString, xs,
         NULL);
      XmStringFree(xs);

      i=0;
      XtSetArg (args[i], XmNcolumns, 60); i++;
      XtSetArg (args[i], XmNmargin, 10); i++;
      XtSetArg (args[i], XmNrows, 10); i++;
      XtSetArg (args[i], XmNvalue, textpt); i++;
      XtSetArg (args[i], XmNeditMode, XmMULTI_LINE_EDIT); i++;
      XtSetArg (args[i], XmNscrollingPolicy, XmAUTOMATIC); i++;
      XtSetArg (args[i], XmNeditable, False); i++;
      XtSetArg (args[i], XmNresizeWidth, True); i++;
      XtSetArg (args[i], XmNautoShowCursorPosition, False); i++;
      XtSetArg (args[i], XmNcursorPositionVisible, False); i++;

      helpTxt = XmCreateScrolledText (helpDiag, "helpTxt", args, i);
      XtManageChild (helpTxt);
   }
   XtManageChild (helpDiag);
}


/*
 * Macro Actions
 */

/*
 * sendPro - Shift-Alt F1 -> F6 come here and insert 0 -> 6 into 
 * the txText widget, where the modify/verify callback promptly
 * changes the widgets text to the appropriate text.
 */
void sendPro (Widget w, XEvent *e, String args[], Cardinal *nargs)
{
   Position pos;
   int val;

   val = atoi(args[0]);
   pos = XmTextGetLastPosition (txText);
   XmTextSetInsertionPosition (txText, pos);
   XmTextInsert (txText, pos, (char *) &val);
}


/*
 * sendOver Action - gets the other call from shared memory,
 * gets your call, builds the "over" string, and sends it.
 * If there is no call - just send de mycall
 */
void sendOver (Widget w, XEvent *e, String args[], Cardinal *nargs)
{
   char *ch;

   if ((ch = getHisCall()) != (char *)0)        /* get his call */
   {                                            /* got it */
      procMacroText(ch);                        /* send his call, */
      shmdt ((void *)ch);                       /* detach shm   */
   }
   procMacroText ("de");                        /* send de */
   procMacroText (app_res.call);                /* and send my call */
}


/*
 * sendHisCall Action - send the call from twlog
 */
void sendHisCall (Widget w, XEvent *e, String args[], Cardinal *nargs)
{
   char *ch;

   if ((ch = getHisCall()) != (char *)0)        /* get his call */
   {                                            /* got it */
      procMacroText(ch);                        /* send his call, */
      shmdt ((void *)ch);                       /* detach shm   */
   }
}


/*
 * sendMacro Action - send all the action arguments
 */
void sendMacro (Widget w, XEvent *e, String args[], Cardinal *nargs)
{
   int i;

   for (i=0; i<(int)*nargs; i++)
   {
      procMacroText (args[i]);
   }
}


/*
 * getHisCall - attach the shared menory if running twlog and returns
 * a char * to his call
 * Used by sendOver and sendCall
 */
#define KEY 6146     /* must match Twlog's value */
#define SHMSIZE 40

char *getHisCall ()
{
   int shmid;
   void *pt;

   if ((shmid = shmget ((key_t) KEY, SHMSIZE, 0600)) < 0)
   {
      fprintf (stderr, "No twlog!\n");
      return ((char*) 0);
   }

   if ((pt = shmat (shmid, NULL, 0)) ==  (void *) -1)
   {
      perror ("twpsk - shmat failed");
      return ((char *) 0);
   }
   return ((char *)pt);
}


/*
 * procMacroText - sends a string to appendTXtext
 */
void procMacroText (char *ch)
{
   XmTextInsert (txText,
      XmTextGetLastPosition(txText), ch);
   XmTextInsert (txText,
      XmTextGetLastPosition(txText), " ");
}

#if 0   // TJW start of 2.0 CB stuff
/*
 * scale2CB - sets word gap scale factor
 */
void scale2CB (Widget w, XtPointer client_data, XtPointer cbs)
{
   XmScaleCallbackStruct *pt = (XmScaleCallbackStruct *) cbs;

   // TJW 2.0 set_sens (pt->value);
}


/*
 * scale3CB - sets word gap scale factor
 */
void scale3CB (Widget w, XtPointer client_data, XtPointer cbs)
{
   XmScaleCallbackStruct *pt = (XmScaleCallbackStruct *) cbs;

   // TJW 2.0 set_wordsf (pt->value);
}


/*
 * scale4CB - sets character gap scale factor
 */
void scale4CB (Widget w, XtPointer client_data, XtPointer cbs)
{
   XmScaleCallbackStruct *pt = (XmScaleCallbackStruct *) cbs;

   // TJW 2.0 set_charsf (pt->value);
}
#endif     //end of 2.0 CB stuff


/*
 * QRT Callback
 */
void quitCB (Widget w, XtPointer client_data, XtPointer cbs)
{
      cleanup();
}  


/*
 * controlsBtnsCB Callback - callbacks for the Recv and Xmit main menu
 * buttons.  This allows for the togglebuttons to work with Control keys.
 * Set the togglebutton, and let the valueChange callback do the rest
 */
void controlBtnsCB (Widget w, XtPointer cdata, XtPointer cbs)
{
   int state = (int) cdata;

   XmToggleButtonSetState (trTB, state, True);
}


/*
 * wpmCB Callback
 * Value changed and Drag Callback for wpm (WPM) and Farnsworth scales
 */
void wpmCB (Widget w, XtPointer client_data, XtPointer call_data)
{
   int type = (int) client_data;
   XmString xs;
   char val[4];
   int wpm, fwpm;
   Cursor watch;
   Boolean validFarns; //, farnsOff;

   XmScaleGetValue (farnsScale, &fwpm);
   XmScaleGetValue (wpmScale, &wpm);
   validFarns = (wpm > fwpm) && (fwpm > FARNS_OFF);

   XtVaSetValues (farnsScale,
      XmNshowValue, (fwpm == FARNS_OFF) ? False : True,
      NULL);

   xs = XmStringCreateLocalized (
                validFarns ? "Farnsworth-On ":"Farnsworth-Off");

   #ifdef SCALE_BUG                        /* See twcw.h */
   XtVaSetValues (farnsLabel,              /* set the label string */
      XmNlabelString, xs,
      NULL);
   #else
   XtVaSetValues (farnsScale,               /* set the Scale gadget label */
      XmNtitleString, xs,
      NULL);
   #endif
   XmStringFree(xs);

   sprintf (val, "%d", fwpm);               /* set farns wpm */
   send_msg (FARNS_MSG, val);

   sprintf (val, "%d", wpm);                /* set wpm */
   send_msg (WPM_MSG, val);

   if (type == WPM_MSG)
   {
      /* 
       * Change cursor to watch while wpm is being changed.
       * WPM message from sendCW will reset it 
       */
      watch = XCreateFontCursor (XtDisplay(shell), XC_watch);
      XDefineCursor (XtDisplay (shell), XtWindow (shell), watch);
   }

   XmProcessTraversal (txText, XmTRAVERSE_CURRENT);
}


/*
 * form1 focus Callback
 * Check if $HOME/.twcwDir has been installed.  If not,
 * give them the option to create and populate it.
 */
void initCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   struct stat buf;
   Widget initDiag;
   XmString xs;
   char mess[] = "Twcw needs to create .twcwDir in your home directory\n"
                 "and populate it.  You will want to modify the files\n\n"
                 "Click OK to create it or Cancel to exit.\n";

   /* remove this callback so it is only called once at start up */
   XtRemoveCallback (w, XmNfocusCallback, initCB, NULL);

   /* now check for $HOME/.twcwDir */
   strcpy (dirpath, getenv ("HOME"));
   strcat (dirpath, TWCWDIR);          /* $HOME/.twcwDir */

   if (stat (dirpath, &buf) < 0)       /* if $HOME/.twcwDir does not exist */
   {
      /* create and popup the dialog */
      initDiag = XmCreateQuestionDialog (shell, "initDiag", NULL, 0);

      XtVaSetValues (XtParent (initDiag),
         XmNtitle, "CREATE TWCW FILES",
         NULL);

      xs = XmStringCreateLtoR (mess, XmSTRING_DEFAULT_CHARSET);
      XtVaSetValues (initDiag,
         XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL,
         XmNmessageString, xs,
         NULL);
      XmStringFree (xs);
      XtUnmanageChild (XmMessageBoxGetChild (initDiag, XmDIALOG_HELP_BUTTON));
      XtAddCallback (initDiag, XmNokCallback, initDiagCB, (XtPointer) NULL);
      XtAddCallback (initDiag, XmNcancelCallback, initDiagCB, (XtPointer) NULL);
      XtManageChild (initDiag);
   }
   return;
}


/*
 * initDiagCB
 * Ok and Cancel callbacks for initDiag
 */
void initDiagCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   XmAnyCallbackStruct *ptr = (XmAnyCallbackStruct *) cbs;

   switch (ptr->reason)
   {
      case XmCR_CANCEL:
         cleanup();
         exit (-1);
         break;

      case XmCR_OK:
         getQSOfiles (dirpath);
         break;

      default:
         printf ("Invalid BTN\n");
   }
}


/*
 * Copy files from PKG_DATA_DIR/twcw/twcwDir to destdir ($HOME/.twcwDIR)
 * Just needed the first time you run twcw
 */
int getQSOfiles (char *destdir)
{
   int n;
   DIR *dirptr;
   struct dirent *dirstruct;
   int fdin, fdout;
   char dname[100];
   char ifile[100], ofile[100];
   char buff[BUFSIZ];

   /* create .twcwDir in $HOME */
   if (mkdir (dirpath, 0755) < 0)
   {
      errorDiag (shell,
                 "getQSOfiles: can't create $HOME/.twcwDir\n"
                 "twcw must exit",
                  NO_CANCEL);
      return (-1);
   }

   /* open the data dir at PKG_DATA_DIR/twcwDir */
   strcpy (dname, PKG_DATA_DIR);
   strcat (dname, "/twcwDir");

   if ((dirptr = opendir (dname)) == NULL)
   {
      errorDiag (shell,
                 "getQSOfiles: Can't open PKG_DATA_DIR/twcwDir\n"
                 "Installation error?\n"
                 "twcw must exit",
                  NO_CANCEL);
      return (-1);
   }

   while ((dirstruct = readdir (dirptr)) != NULL)
   {  /* skip the . and .. entries */
      if ((strcmp (dirstruct->d_name, ".") == 0) |
         (strcmp (dirstruct->d_name, "..") == 0))
      {
         continue;
      }
      strcpy (ifile, dname);
      strcat (ifile, "/");
      strcat (ifile, dirstruct->d_name);

      strcpy (ofile, destdir);
      strcat (ofile, "/");
      strcat (ofile, dirstruct->d_name);

      fdin = open (ifile, O_RDONLY, 0655);
      fdout = open (ofile, O_RDWR | O_CREAT, 0655);

      while ((n = read (fdin, buff, BUFSIZ)) > 0)
      {
         write (fdout, buff, n);
      }
      close (fdin);
      close (fdout);
   }
   closedir (dirptr);
   return 1;
}
