/*
 * twcw:  A gui application that sends Morse Code 
 * Copyright (C) 1997 Ted Williams WA0EIR 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
 * Version 1.4: JAN 2008
 */

/*
 * setup Function
 * Setup traps and the message queue
 */
#include "common.h"
#include "twcw.h"

int setup()
{
   int  wpm, tone;
   char speed[4], val[6];
   Boolean mode;
   Cursor watch;

   /* Trap the INT, TERM, and QUIT signals to do housekeeping */
   signal (SIGINT,  cleanup);
   signal (SIGTERM, cleanup);
   signal (SIGQUIT, cleanup);
   signal (SIGSEGV, cleanup);

   /* Open the public message queue */
   if ((qid = msgget (TWCW_KEY, IPC_CREAT | IPC_EXCL | 0666)) == -1)
   {
      /* Maybe it exists already, so just try to get the id */
      if ((qid = msgget (TWCW_KEY, 0666)) == -1)
      {
         perror ("twcw msgget");
         return (0);      /* Open Queue Failed */
      }
   }
   /*
    * Queue is Open, so now sync the child process with the current
    * Sidetone On/Off, Sidetone Frequency, and WPM.
    * First tell the child if the sidetone is on or off
    */
   XtVaGetValues (scTB,
      XmNset, &mode,
      NULL);
   if (mode)
      send_msg (KEY_DEVICES_MSG, "2");
   else
      send_msg (KEY_DEVICES_MSG, "1");

   /* Now sync the tone frequency */
   XmScaleGetValue (toneScale, &tone);
   if ((tone < MIN_FREQ) || (tone > MAX_FREQ))
   {
      /* You goofed in Twcw, I'll set the tone to 800 Hx */
      tone = 800;
      XmScaleSetValue (toneScale, tone);
      fprintf (stderr, "Warning: problem with Tone in Twcw\n");
   }
   sprintf (val, "%d", tone);
   send_msg (FREQ_MSG, val);


   /* 
    * Next, we tell the child the selected WPM in the resource file 
    * and change the cursor to the XC_watch watch while WPM is being set
    */    
   XmScaleGetValue (wpmScale, &wpm);
   if ((wpm < MIN_WPM) || (wpm > MAX_WPM))
   {
      /* You goofed in Twcw, so I'll set it to 15 wpm */
      wpm = 15;
      XmScaleSetValue (wpmScale, wpm);
   }
   sprintf (speed, "%d", wpm);

   watch = XCreateFontCursor (XtDisplay(shell), XC_watch);
   XDefineCursor (XtDisplay (shell), XtWindow (shell), watch);
   send_msg (WPM_MSG, speed);

   /* The last thing is to send a STOP MESSAGE to the child */
   send_msg (STOP_MSG, "0");
   return 1;
}


/*
 * send_msg Function
 * Sends a WORD, WPM, STOP, KEY_DEVICES, FARNSWORTH or FREQ message to
 * the send_cw process
 */
int  send_msg (long type, char *data)
{
   int  val;
   struct tag1 buff;
   Cursor watch = 0;

   buff.type = type;
   switch (type)
   {
      case WORD_MSG:
      case STOP_MSG:
         strcpy (buff.data.word, data);
         break;

      case WPM_MSG:
         val = atoi (data);
         if ((val < MIN_WPM) || (val > MAX_WPM))
         {
            XtVaSetValues (wpmScale, XmNvalue, data, NULL);
            return (False);
         }
         else
         {
            buff.data.val = val;
            /* Change the cursor to the watch while WPM is being set */
            watch = XCreateFontCursor (XtDisplay(shell), XC_watch);
            XDefineCursor (XtDisplay(shell), XtWindow (shell), watch);
            break;
         }

      case FREQ_MSG:
      case FARNS_MSG:
         val = atoi (data);
         buff.data.val = val;
         break;

      case KEY_DEVICES_MSG:
         val = atoi (data);
         buff.data.val = val;
         break;

      default:
         fprintf (stderr, "send_msg - Invalid Message Number\n");
         return (False);
   }

   if (msgsnd (qid, (struct msgbuf *) &buff, sizeof (buff.data), IPC_NOWAIT) == -1)
   {
      perror ("send_msg");
      cleanup();
      exit (1);
   }
   return (True);
}


/*
 * cleanup Function
 * Does some housekeeping for traped signals
 */
extern int qid;			/* qid to be removed by cleanup */

void cleanup (void)
{
   int status;
   int rtn;

   signal (SIGINT,  SIG_IGN);
   signal (SIGTERM, SIG_IGN);
   signal (SIGQUIT, SIG_IGN);
   signal (SIGSEGV, SIG_IGN);

   fprintf (stderr, "Sending TERM signal to sendCW\n");
   kill (pid, SIGTERM);
   rtn = wait(&status);
   if (rtn == pid)
      fprintf(stderr, "sendCW terminated - status = %d OM\n",
                       WEXITSTATUS(status));

   fprintf (stderr, "Removing message qid %d\n", qid);

   if (msgctl (qid, IPC_RMID, 0) == -1)
   {
      perror ("Queue didn't close");
   }
   // TJW 2.0 destroy_plan();
   // TJW 2.0 fprintf (stderr, "Plan destroyed\n");
   fprintf (stderr, "twcw done.\n");
   exit (0);
}
