/* Copyright (C) 1999, 2000, 2001, 2002 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

/*

The classes in this file provide the dialogs which the program uses.
Most are modal (that is, they do not start their own event loop and do
not allow other interaction with the program while they are
displayed).

A few are modeless.  These are self-owning objects, which (via user
input) control their own destiny.  When closing, they call 'delete
this'.  Such modeless dialogs are one of the few cases where this is
entirely safe.

*/

#ifndef DIALOGS_H
#define DIALOGS_H

#ifndef CONFIG_H
#define CONFIG_H
#include "config.h"
#endif

#include <qdialog.h>
#include <qwidget.h>
#include <qlineedit.h>
#include <qmultilinedit.h>
#include <qlabel.h>
#include <qevent.h>
#include <qslider.h>
#include <qlcdnumber.h>
#include <qcheckbox.h>
#include <qstring.h>
#include <fstream>
#include <cstring>

#include <qmultilinedit.h>

#include "prog_defs.h"

extern Prog_func prog_func;
class Tnc;

class ConnectDialog;

class ConnectEditor: public QLineEdit {
    
    Q_OBJECT

    Tnc* tnc_p;
    ConnectEditor(QWidget*, const char*, Tnc*);   // private class
                                                  //only ConnectDialog can create an object
protected:
    virtual void keyPressEvent(QKeyEvent*);
signals:
    void esc_pressed(void);
public:
    friend class ConnectDialog;
};

class ConnectDialog: public QDialog {

    Q_OBJECT

    ConnectEditor* editbox_p;
    Tnc* tnc_p;
public:
    QString get_call(void) const {return editbox_p->text();}
    ConnectDialog(Tnc*, int, QWidget* qwidget_p = 0);
    ~ConnectDialog(void) {}
};

class CallsignDialog;

class CallsignEditor: public QLineEdit {
    
    Q_OBJECT

    Tnc* tnc_p;
    CallsignEditor(QWidget* widget_p, Tnc* tnc): QLineEdit(widget_p),
                                                 tnc_p(tnc) {} // private class
                                                               // only CallsignDialog can create an object
protected:
    virtual void keyPressEvent(QKeyEvent*);
signals:
    void esc_pressed(void);
public:
    friend class CallsignDialog;
};

class CallsignDialog: public QDialog {

    Q_OBJECT

    CallsignEditor* editbox_p;
public:
    QString get_call(void) const {return editbox_p->text();}
    CallsignDialog(Tnc*, int, QWidget* qwidget_p = 0);
    ~CallsignDialog(void) {}
};

class CommandDialog;

class CommandEditor: public QLineEdit {
    
    Q_OBJECT

    int count;
    CommandEditor(QWidget* widget_p): QLineEdit(widget_p), count(0) {} // private class
                                           // only CommandDialog can create an object
protected:
    virtual void keyPressEvent(QKeyEvent*);
signals:
    void key_pressed(int);
public:
    friend class CommandDialog;
};

class CommandDialog: public QDialog {

    Q_OBJECT

public:
    enum {rejected, accepted, script};
private:
    CommandEditor* editbox_p;
private slots:
    void command_accept(void) {done(accepted);}
    void command_reject(void) {done(rejected);}
    void command_script(void) {done(script);}
public:
    QString get_command(void) const {return editbox_p->text();}
    CommandDialog(int, QWidget* qwidget_p = 0);
    ~CommandDialog(void) {}
};

class RstDialog;

class RstEditor: public QLineEdit {
    
    Q_OBJECT

    int count;
    RstEditor(QWidget* widget_p): QLineEdit(widget_p), count(strlen(prog_func.rst)) // private class
                                                         {setText(prog_func.rst);}  // only RstDialog can create an object
                                     
protected:
    virtual void keyPressEvent(QKeyEvent*);
signals:
    void esc_pressed(void);
public:
    friend class RstDialog;
};

class RstDialog: public QDialog {

    Q_OBJECT

private:
    RstEditor* editbox_p;
public:
    QString get_rst(void) const {return editbox_p->text();}
    RstDialog(int, QWidget* qwidget_p = 0);
    ~RstDialog(void) {}
};

class Autocq_modeDialog: public QDialog {

    Q_OBJECT

public slots:
    void mode_button_pressed(int);
public:
    Autocq_modeDialog(int standard_size, QWidget* qwidget_p = 0);
};

class GplDialog;

class GplEditor: public QMultiLineEdit {

    Q_OBJECT

private:
    GplEditor(QWidget* widget_p): QMultiLineEdit(widget_p) {} // private class
                                                              // only GplDialog can create an object
protected:
    virtual void keyPressEvent(QKeyEvent*);
signals:
    void esc_pressed(void);
public:
    friend class GplDialog;
};

class GplDialog: public QDialog {

    Q_OBJECT

    GplEditor* editbox_p;
    int standard_size;
    QSize button_size;
    QLabel* label_p;
    QPushButton* ok_button_p;
    QPushButton* cancel_button_p;
protected:
    virtual void resizeEvent(QResizeEvent*);
public:
    GplDialog(int size, QWidget* qwidget_p = 0);
    ~GplDialog(void) {}
};

class PromptDialog: public QDialog {

    Q_OBJECT

public:
    PromptDialog(const char* text, const char* caption, int standard_size, QWidget* qwidget_p = 0);
    ~PromptDialog(void) {}
};

class HelpDialog;

class HelpEditor: public QMultiLineEdit {

    Q_OBJECT

private:
    HelpEditor(QWidget* widget_p): QMultiLineEdit(widget_p) {} // private class
                                                               // only HelpDialog can create an object
protected:
    virtual void keyPressEvent(QKeyEvent*);
signals:
    void esc_pressed(void);
public:
    friend class HelpDialog;
};

// this dialog is modeless
class HelpDialog: public QDialog {

    Q_OBJECT

    HelpEditor* editbox_p;
    int standard_size;
    QSize button_size;
    QPushButton* cancel_button_p;
    bool open_flag;
protected:
    virtual void resizeEvent(QResizeEvent*);
    virtual void closeEvent(QCloseEvent*) {kill_dialog();}
public slots:
    void kill_dialog(void) {hide(); delete this;}
public:
    int is_opened(void) const {return open_flag;}
    HelpDialog(int size);
    ~HelpDialog(void) {}
};

class CwSpeedSlider: public QSlider {
protected:
    virtual void keyPressEvent(QKeyEvent*);
public:
    CwSpeedSlider(int initial_speed, QWidget* qwidget_p = 0, const char* name = 0);
};

class CwSpeedDialog: public QDialog {

    Q_OBJECT

    CwSpeedSlider* cw_speed_slider_p;
    QLCDNumber* qlcdnumber_p;
private slots:
    void display(int num) {qlcdnumber_p->display(((num + 2)/5) * 5);}
    void set_sliderpos(void) {cw_speed_slider_p->setValue(qlcdnumber_p->intValue());}
public:
    int get_speed(void) const {return qlcdnumber_p->intValue();}
    CwSpeedDialog(int, int, QWidget* qwidget_p = 0);
    ~CwSpeedDialog(void) {}
};

class RttySpeedDialog: public QDialog {

    Q_OBJECT

public:
    enum Rtty_speed {b45, b50, b57, b75} rtty_speed;
private:
    Rtty_speed speed;
public slots:
    void speed_button_pressed(int);
public:
    RttySpeedDialog(Rtty_speed speed_, int standard_size, QWidget* qwidget_p = 0);
    Rtty_speed get_speed(void) const {return speed;}
};

class AsciiSpeedDialog: public QDialog {

    Q_OBJECT

public:
    enum Ascii_speed {b50, b100, b110, b200} ascii_speed;
private:
    Ascii_speed speed;
public slots:
    void speed_button_pressed(int);
public:
    AsciiSpeedDialog(Ascii_speed speed_, int standard_size, QWidget* qwidget_p = 0);
    Ascii_speed get_speed(void) const {return speed;}
};

class MainScreen;
class ReceiveWin;

// this dialog is modeless
class DownloadDialog: public QDialog {

    Q_OBJECT

    Tnc* tnc_p;
    MainScreen* mainscreen_p;
    ReceiveWin* receivewin_p;
    int stream;
    int port;
    QLabel* bytes_label_p;
private slots:
    void end_download(void);
protected:
    virtual void closeEvent(QCloseEvent* e) {e->ignore();}
public:
    void show_bytes(long); 
    DownloadDialog(const char*, Tnc*, MainScreen*, ReceiveWin*, int);
    ~DownloadDialog(void) {}
};

class BufferList;

// this dialog is modeless
class UploadDialog: public QDialog {

    Q_OBJECT

    Tnc* tnc_p;
    BufferList& buffer_list;
    MainScreen* mainscreen_p;
    ReceiveWin* receivewin_p;
    int stream;
    int port;
    QLabel* bytes_label_p;
private slots:
    void end_upload(void);
protected:
    virtual void closeEvent(QCloseEvent* e) {e->ignore();}
public:
    void show_bytes(long); 
    UploadDialog(const char*, long, Tnc*, BufferList&, MainScreen*, ReceiveWin*, int);
    ~UploadDialog(void) {}
};

class PrintMarkDialog: public QDialog {

    Q_OBJECT

public:
    enum {accepted, rejected, cancel_job};
private slots:
    void command_accept(void) {done(accepted);}
    void command_reject(void) {done(rejected);}
    void command_cancel_job(void) {done(cancel_job);}
protected:
    virtual void keyPressEvent(QKeyEvent*);
public:
    PrintMarkDialog(int, QWidget* qwidget_p = 0);
    ~PrintMarkDialog(void) {}
};

// this dialog is modeless
class QsoCountDialog: public QDialog {

    Q_OBJECT

signals:
    void inc(void);
    void dec(void);
    void quit_button_pressed(void);
private:
    QLabel* qso_label_p;
private slots:    
    void show_qso_count(int);
    void destroy(void);
protected:
    virtual void closeEvent(QCloseEvent* e) {e->ignore();}
    virtual void keyPressEvent(QKeyEvent*);
public: 
    QsoCountDialog(int);
    ~QsoCountDialog(void);
};

#endif
