/* Copyright (C) 1999, 2000, 2001, 2002 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

/*

The classes in this file are responsible for the main program display
- including the send window and the receive window, the menu bar,
information display bar and status bar.

The class Sendwin object also controls the keyboard, as it represents
the window on which text entered on the program for transmission is
displayed.  It communicates with other class objects mainly by means
of the signal/slot mechanism.

class Receivewin is derived from class QWidgetStack.  The QWidgetStack
object contains the window widgets for each stream.

Dialog classes are generally found in the file dialogs.h/dialogs.cpp.

Except where the slots are local (for example, they relate to the
printing out or saving of selected text in the receive window), most
of the slot methods are found in file eventslots.h/eventslots.cpp.

*/

#ifndef WIDGETS_H
#define WIDGETS_H

#ifndef CONFIG_H
#define CONFIG_H
#include "config.h"
#endif

#include <qapplication.h>
#include <qmainwindow.h>
#include <qpopupmenu.h>
#include <qwidgetstack.h>
#include <qevent.h>
#include <qlabel.h>
#include <qstatusbar.h>
#include <qtoolbutton.h>
#include <qmessagebox.h>
#include <qclipboard.h>
#include <qcursor.h>
#include <qcolor.h>
#include <qhbox.h>

#include <qmultilinedit.h>

#include "dialogs.h"
#include "prog_defs.h"
#include "tnc.h"

#define SCROLL_STORE_SIZE 10000
#define END_OF_WINDOW 1000000      // this needs to be an arbitrary value greater than MAXLINES representing
                                   // the place the cursor is to be placed to reach the end of the window

class MainScreen;

class StatusLine: public QStatusBar {
    int with_tor_fleximinsize;
    int no_tor_fleximinsize;
    Tnc* tnc_p;
    QLabel* connected_status_label_p;
    QLabel* lock_label_p;
    QLabel* tx_label_p;
    QLabel* idle_label_p;
    QLabel* err_label_p;
    QLabel* irs_label_p;
    QLabel* speed_label_p;
    QLabel* huff_label_p;
    QLabel* freebytes_label_p;
    QLabel* flexi_label_p;
    int torline_flag;
public:
    void write_connected_status(const char* text) {connected_status_label_p->setText(text);}
    void write_lock_status(const char* text) {lock_label_p->setText(text);}
    void write_tx_status(const char* text) {tx_label_p->setText(text);}
    void write_idle_status(const char* text) {idle_label_p->setText(text);}
    void write_err_status(const char* text) {err_label_p->setText(text);}
    void write_irs_status(const char* text) {irs_label_p->setText(text);}
    void write_speed_status(const char* text) {speed_label_p->setText(text);}
    void write_huff_status(const char* text) {huff_label_p->setText(text);}
    void write_freebytes(const char* text) {freebytes_label_p->setText(text);}
    void make_torline(void);
    StatusLine(QWidget*, Tnc*, int, int);
};

class SendWin;
class ReceiveWin;
class DisplayLine;
class BufferList;
class EventSlots;

class MainScreen: public QMainWindow {

    Q_OBJECT
private:
    bool letters_to_send_flag;
    bool flush_chars;
    int count;
    const int standard_size;
    unsigned char info_byte;
    unsigned char tx_byte;
    int buffer_count;
    int beep_count;
    int interval_count;
    int upload_item;
    int download_item;
    int upload_7plus_item;
    int upload_binary_item;
    int menu_copy_item;
    int menu_print_selection_item;
    int menu_save_selection_item;
    int mouse_copy_item;
    int mouse_print_selection_item;
    int mouse_save_selection_item;
    int menu_set_print_mark_item;
    int menu_print_from_mark_item;
    int mouse_set_print_mark_item;
    int mouse_print_from_mark_item;
    int menu_qso_count_item;
    int mouse_qso_count_item;
    DisplayLine* display_line_p;
    StatusLine* status_line_p;
    QPopupMenu* filemenu_p;
    QPopupMenu* upload_p;
    QPopupMenu* streammenu_p;
    QPopupMenu* hfpacket_p;
    QPopupMenu* hftor_p;
    QPopupMenu* mouse_popup_p;
    QPopupMenu* mouse_streammenu_p;
    QPopupMenu* mouse_hfpacket_p;
    QPopupMenu* mouse_hftor_p;
    QToolButton* connect_button_p;
    QToolButton* disconnect_button_p;
    QToolButton* call_lock_button_p;
    QToolButton* capture_button_p;
    QToolButton* auto_cq_button_p;
    QToolButton* speed_lock_button_p;
    QToolButton* ident_button_p;
    QToolButton* sync_button_p;
    QToolButton* abort_button_p;
    QToolButton* rx_button_p;
    QToolButton* tx_button_p;
    QToolButton* change_speed_button_p;
    QToolButton* dx_cluster_button_p;
    EventSlots* event_slots_p;
    Tnc* tnc_p;
    Tnc_base* tnc_base_p;
    Pipe_fifo& receive_pipe;
    Transmit_buffer& tr_buffer;
    BufferList& buffer_list;
    SendWin* sendwin_p;
    ReceiveWin* receivewin_p;
    enum {rx, tx} tx_status;
private slots:
    void process_receive_pipe(int);
    void mouse_popup(void) {mouse_popup_p->exec(QCursor::pos());}
    void activate_textselected_items(void);
    void disactivate_textselected_items(void);
    void toolbar_moved(void);
    void about_kamplus(void);
    void about_qt(void) {QMessageBox::aboutQt(this, "About Qt");}
protected:
    virtual void timerEvent(QTimerEvent*);
public:
    // the following method is to enable objects of other classes and other functions
    // to use the public methods of ReceiveWin whilst only holding a MainScreen pointer
    //(in particular, to enable writing to the window)
    ReceiveWin* get_receivewin(void) const {return receivewin_p;}
    void update_file_load_items(void);
    void update_print_mark_items(void);
    void update_qso_counter_items(void);
    void show_packetmenu_streams(void) {streammenu_p->removeItem(1);
                                        mouse_streammenu_p->removeItem(1);
                                        streammenu_p->insertItem("&Hf", hfpacket_p, 1);
                                        mouse_streammenu_p->insertItem("Hf", mouse_hfpacket_p, 1);}
    void show_tormenu_streams(void) {streammenu_p->removeItem(1);
                                     mouse_streammenu_p->removeItem(1);
                                     streammenu_p->insertItem("&Hf", hftor_p, 1);
                                     mouse_streammenu_p->insertItem("Hf", mouse_hftor_p, 1);}
    void display_capture_status(void);
    void display_current_stream(void);
    void display_connected_status(void);
    void display_mode(void);
    void display_send_mode_status(void);
    void display_callsign(void);
    void display_freebytes(void);
    void set_connect_button(void);
    void set_disconnect_button(void);
    void set_capture_button(void) {if (tnc_p->tnc_func.capturefile_flag) capture_button_p->setOn(true);
                                     else capture_button_p->setOn(false);}
    void set_call_lock_button(void);
    void set_auto_cq_button(void);
    void set_speed_lock_button(void);
    void set_ident_button(void);
    void set_sync_button(void);
    void set_abort_button(void);
    void set_rx_button(void) {if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) rx_button_p->setEnabled(true);
                              else rx_button_p->setEnabled(false);}
    void set_tx_button(void) {if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) tx_button_p->setEnabled(true);
                             else tx_button_p->setEnabled(false);}
    void set_change_speed_button(void) {if (tnc_p->tnc_func.active_port
					    && (tnc_p->tnc_func.hfmode == Tnc_func::cw
						|| tnc_p->tnc_func.hfmode == Tnc_func::rtty
						|| tnc_p->tnc_func.hfmode == Tnc_func::ascii))
                                            change_speed_button_p->setEnabled(true);
					else change_speed_button_p->setEnabled(false);}
    void set_dx_cluster_button(void);
    void make_torline(void) {status_line_p->make_torline();}
    void update_lockinfo(void);
    void update_torinfo(unsigned char info = 255); // if info = 255, the tor info line will be updated from info_byte
    void update_txinfo(unsigned char tx = 255); // if tx = 255, the tx indication will be updated from tx_byte
    void beep(int count) {beep_count = count;}
    MainScreen(Tnc*, Tnc_base*, Pipe_fifo&, Transmit_buffer&, BufferList&, int win_fontsize = 12);
    ~MainScreen(void) {}
};

class SendWin: public QMultiLineEdit {

    Q_OBJECT

signals:
    void CtrlA_pressed(void);
    void CtrlD_pressed(void);
    void CtrlT_pressed(void);
    void CtrlZ_pressed(void);
    void f1_pressed(void);
    void f2_pressed(void);
    void f3_pressed(int);
    void f4_pressed(void);
    void f5_pressed(void);
    void f7_pressed(void);
    void f8_pressed(void);
    void f9_pressed(void);
    void f10_pressed(void);
    void f11_pressed(void);
    void f12_pressed(void);
    void altB_pressed(void);
    void altI_pressed(void);
    void altL_pressed(void);
    void altN_pressed(void);
    void altP_pressed(void);
    void altR_pressed(void);
    void altS_pressed(void);
    void altX_pressed(void);
    void alt_plus_pressed(void);
    void alt_minus_pressed(void);
    void pageup_pressed(void);
    void pagedown_pressed(void);
    void sig_auto_cq(int, int);
    void sig_change_hfmode(int, int);
    void sig_upload(int);
    void sig_download(int);
    void sig_send_message(int, int);

    void keyup(void);
    void keydown(void);
    void keyright(int);

    void letter_to_send(uchar);

    void mouse_right_clicked(void);
protected:
    virtual void keyPressEvent(QKeyEvent*);
    virtual void mousePressEvent(QMouseEvent* event_p) {if (event_p->button() == RightButton) emit mouse_right_clicked();}
    virtual void mouseMoveEvent(QMouseEvent*) {}         // ignore other mouse events in this window
    virtual void mouseReleaseEvent(QMouseEvent*) {}      // ignore other mouse events in this window
    virtual void mouseDoubleClickEvent(QMouseEvent*) {}  // ignore other mouse events in this window
public:
    void write(const char*);
    void new_line(void) {newLine();}
    void backspace(void) {QMultiLineEdit::backspace();}
    void del_line(void) {QMultiLineEdit::killLine();}
    void font_change(int);
    SendWin(QWidget*, int fontsize = 12);
    ~SendWin(void) {}
};

// we inherit from QMulitLineEdit, and we can make Receivewin a friend and
// access some of the protected methods
class ReceiveWinNode: public QMultiLineEdit {
    Q_OBJECT

signals:
    void mouse_right_clicked(void);
    void mouse_released(void);
protected:
    virtual void mousePressEvent(QMouseEvent*);
    virtual void mouseReleaseEvent(QMouseEvent*);
public:
    friend class ReceiveWin;
    ReceiveWinNode(QWidget*, int);
    ~ReceiveWinNode(void) {}
};

struct ScrollStoreNode {
    char letter;
    enum Colour {black, red, blue, green, purple, end_marker} colour;
};

class ScrollStore {
    ScrollStoreNode store[SCROLL_STORE_SIZE + 1];
    int index;
public:
    void add_letter(ScrollStoreNode letter) {store[index++] = letter;}
    void remove_letter(void) {if (index) index--;}
    const ScrollStoreNode* extract_store(void) {store[index].colour = ScrollStoreNode::end_marker;
                                                      return store;}
    void clear_store(void) {store->colour = ScrollStoreNode::end_marker; index = 0;}
    int is_free(void) const {if (index < SCROLL_STORE_SIZE) return SCROLL_STORE_SIZE - index;
                             return false;}
    bool is_empty(void) const {return !index;}
    ScrollStoreNode::Colour last_colour(void) const {if (index) return store[index - 1].colour;
                                                     return ScrollStoreNode::black;}
    ScrollStore(void): index(0) {store->colour = ScrollStoreNode::end_marker;}
};

class ReceiveWin: public QWidgetStack {

    Q_OBJECT

signals:
    void mouse_right_clicked(void);
    void activate_textselected_items(void);
    void disactivate_textselected_items(void);
private:
    ReceiveWinNode* node_ptr[MAXUSERS][2];  // have a pointer for each stream/port;
    Tnc* tnc_p;
    int standard_size;
    bool scroll_flag;
    QColor black_colour;
    QColor red_colour;
    QColor blue_colour;
    QColor green_colour;
    QColor purple_colour;
    ScrollStore scroll_store;
    void write_store(void);
public slots:
    void scrollup(void);
    void scrolldown(void);
    void scrollout(int silent);
    void copy(void);
    void print_selection(void);
    void print_scroll_buffer(void);
    void set_textselected_items(void);
    void save(void);
public:
    bool is_scrolling(void) const {return scroll_flag;};
    void check_scroll_condition(void);

    // these colour write methods don't show colours with Qt-1.4 or 2.*
    // they are there to provide a consistent interface for Tnc objects
    void write(const char* text) {write(text, tnc_p->tnc_func.active_stream(),
                 tnc_p->tnc_func.active_port);}
    void write_red(const char* text) {write_red(text, tnc_p->tnc_func.active_stream(),
                 tnc_p->tnc_func.active_port);}
    void write_blue(const char* text) {write(text, tnc_p->tnc_func.active_stream(),
                 tnc_p->tnc_func.active_port);}
    void write_green(const char* text) {write(text, tnc_p->tnc_func.active_stream(),
                 tnc_p->tnc_func.active_port);}
    void write_purple(const char* text) {write(text, tnc_p->tnc_func.active_stream(),
                 tnc_p->tnc_func.active_port);}

    void write(const char* text, int stream, int port, const QColor* colour = 0);
    void write_red(const char* text, int stream, int port)
                                      {write(text, stream, port, &red_colour);}
    void write_blue(const char* text, int stream, int port)
                                      {write(text, stream, port, &blue_colour);}
    void write_green(const char* text, int stream, int port)
                                      {write(text, stream, port, &green_colour);}
    void write_purple(const char* text, int stream, int port)
                                      {write(text, stream, port, &purple_colour);}
    void newline(void) {newline(tnc_p->tnc_func.active_stream(), tnc_p->tnc_func.active_port);}
    void newline(int stream, int port);
    void del_letter(void) {del_letter(tnc_p->tnc_func.active_stream(), tnc_p->tnc_func.active_port);}
    void del_letter(int stream, int port);
    void font_change(int);

    ReceiveWin(QWidget*, Tnc*, int, int fontsize = 12);
    ~ReceiveWin(void);
};


class DisplayLine: public QHBox {
    QLabel* hiscall_label_p;
    QLabel* hfmode_label_p;
    QLabel* capture_stream_label_p;
    QLabel* hfsend_label_p;
    QLabel* current_stream_label_p;
public:
    void write_call(const char* text);
    void write_hfmode(const char* text) {hfmode_label_p->setText(text);}
    void write_capture(const char* text) {capture_stream_label_p->setText(text);}
    void write_hfsend(const char* text) {hfsend_label_p->setText(text);}
    void write_stream(const char* text) {current_stream_label_p->setText(text);}
    int get_minsize(void) const {return (sizeHint().width() < 640) ? sizeHint().width() : 640;}
    DisplayLine(QWidget*, int);
};

#endif
