/* Copyright (C) 1999, 2000 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <qpushbutton.h>
#include <qkeycode.h>
#include <qapplication.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qvbuttongroup.h>
#include <qmessagebox.h>
#include <qlayout.h>
#include <qhbox.h>

#include <cstring>
#include <cctype>
#include <iostream>
#include <strstream>
#include <string>
#include <iomanip>
#include <cstdlib>

#if QT_VERSION >= 300
#include "dialogs-qt3.h"
#include "widgets-qt3.h"
#else
#include "dialogs.h"
#include "widgets.h"
#endif

#include "gpl.h"
#include "tnc.h"
#include "buffers.h"
#include "filesend.h"
#include "download.h"

#define COMMAND_SIZE 100

ConnectDialog::ConnectDialog(Tnc* tnc_p_, int standard_size, QWidget* qwidget_p):
                             QDialog(qwidget_p, "Connect", true), tnc_p(tnc_p_) {
    string connect_name;
    if (tnc_p->tnc_func.active_port && (tnc_p->tnc_func.hfmode == Tnc_func::amtor
					|| tnc_p->tnc_func.hfmode == Tnc_func::lamtor
					|| tnc_p->tnc_func.hfmode == Tnc_func::fec)) {
        connect_name = tnc_p->tnc_func.selCall;
    }
    else {
        connect_name = tnc_p->tnc_func.hisCall[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port];
    }

    QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
    if (!top_layout_p) {
        cerr << "Memory allocation error in ConnectDialog::ConnectDialog()" << endl;
	exit(MEM_ERROR);
    }

    QLabel* label_p = new QLabel("Station to connect with?", this);
    if (!label_p) {
        cerr << "Memory allocation error in ConnectDialog::ConnectDialog()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setAlignment(AlignCenter);
    top_layout_p->addWidget(label_p);

    editbox_p = new ConnectEditor(this, connect_name.c_str(), tnc_p);
    if (!editbox_p) {
        cerr << "Memory allocation error in ConnectDialog::ConnectDialog()" << endl;
	exit(MEM_ERROR);
    }
    int box_width = standard_size * 4;
    editbox_p->setMinimumHeight(standard_size);
    editbox_p->setMinimumWidth(box_width);

    QBoxLayout* editbox_layout_p = new QHBoxLayout(top_layout_p, 0);
    if (!editbox_layout_p) {
        cerr << "Memory allocation error in ConnectDialog::ConnectDialog()" << endl;
	exit(MEM_ERROR);
    }
    editbox_layout_p->addSpacing((standard_size * 3)/2);
    editbox_layout_p->addWidget(editbox_p, 1);
    editbox_layout_p->addSpacing((standard_size * 3)/2);

    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in ConnectDialog::ConnectDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in ConnectDialog::ConnectDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setFixedSize(button_size);
    cancel_button_p->setFixedSize(button_size);

    QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, standard_size);
    if (!button_layout_p) {
        cerr << "Memory allocation error in ConnectDialog::ConnectDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);
    button_layout_p->addWidget(ok_button_p);
    button_layout_p->addStretch(2);
    button_layout_p->addWidget(cancel_button_p);
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);

    setCaption("Connect");

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    editbox_p->setFocus();

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));
    QObject::connect(editbox_p, SIGNAL(returnPressed()), this, SLOT(accept()));
    QObject::connect(editbox_p, SIGNAL(esc_pressed()), this, SLOT(reject()));

    top_layout_p->activate();
    top_layout_p->freeze();
}

ConnectEditor::ConnectEditor(QWidget* widget_p, const char* text, Tnc* tnc): QLineEdit(widget_p), tnc_p(tnc) {
    setText(text);
}

void ConnectEditor::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();
    int letter = event->ascii();
    bool valid_flag = false;
    bool marked_text = false;
    bool first_letter_marked = false;

    if (hasMarkedText()) {
        marked_text = true;
	if (!strncmp(text(), markedText(), strlen(markedText()))) first_letter_marked = true;
    }

    if (keycode ==  Key_Return) {
        if (!text().isEmpty() && !marked_text) emit returnPressed();
	else QApplication::beep();
    }
    else if (keycode == Key_Escape) emit esc_pressed();
    else if (keycode == Key_Backspace
	     && (cursorPosition() != 0 || marked_text)) {
	valid_flag = true;
    }
    else if (keycode == Key_Delete
	     && (cursorPosition() < static_cast<int>(text().length()) || marked_text)) {
	valid_flag = true;
    }

    else if (keycode == Key_Right || keycode == Key_Left
	       || keycode == Key_Home || keycode == Key_End) {
        valid_flag = true;
    }

    else if (isalnum(static_cast<unsigned char> (letter))) {
        letter = toupper(static_cast<unsigned char> (letter));
	valid_flag = true;
    }
    else if ((letter == '/' && cursorPosition() && !first_letter_marked)
	     || (letter == '!'
		 && (cursorPosition() == 0 || first_letter_marked)
		 && tnc_p->tnc_func.active_port
		 && (tnc_p->tnc_func.hfmode == Tnc_func:: pactor
		     || tnc_p->tnc_func.hfmode == Tnc_func:: gtor))
	     || ((letter == ',' || letter == ' ' || letter == '-')
		 && !first_letter_marked
		 && cursorPosition()
		 && (!tnc_p->tnc_func.active_port
		     || tnc_p->tnc_func.hfmode == Tnc_func::packet))) {
        valid_flag = true;
    }
    
    if (valid_flag) {
        QKeyEvent modified_event(QEvent::KeyPress, keycode,
				    letter, event->state(),
				    event->text().upper(),
				    event->isAutoRepeat(), event->count());
	QLineEdit::keyPressEvent(&modified_event);
    }
}

CallsignDialog::CallsignDialog(Tnc* tnc_p, int standard_size, QWidget* qwidget_p):
                               QDialog(qwidget_p, "Connect", true) {

    QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
    if (!top_layout_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }

    QLabel* label_p = new QLabel("Callsign?", this);
    if (!label_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setAlignment(AlignCenter);
    top_layout_p->addWidget(label_p);

    editbox_p = new CallsignEditor(this, tnc_p);
    if (!editbox_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }
    int box_width = standard_size * 4;
    editbox_p->setMinimumHeight(standard_size);
    editbox_p->setMinimumWidth(box_width);

    QBoxLayout* editbox_layout_p = new QHBoxLayout(top_layout_p, 0);
    if (!editbox_layout_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }
    editbox_layout_p->addSpacing((standard_size * 3)/2);
    editbox_layout_p->addWidget(editbox_p, 1);
    editbox_layout_p->addSpacing((standard_size * 3)/2);

    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setFixedSize(button_size);
    cancel_button_p->setFixedSize(button_size);

    QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, standard_size);
    if (!button_layout_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);
    button_layout_p->addWidget(ok_button_p);
    button_layout_p->addStretch(2);
    button_layout_p->addWidget(cancel_button_p);
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);

    setCaption("Callsign");

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    editbox_p->setFocus();

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));
    QObject::connect(editbox_p, SIGNAL(returnPressed()), this, SLOT(accept()));
    QObject::connect(editbox_p, SIGNAL(esc_pressed()), this, SLOT(reject()));

    top_layout_p->activate();
    top_layout_p->freeze();
}

void CallsignEditor::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();
    int letter = event->ascii();
    bool valid_flag = false;
    bool marked_text = false;
    bool first_letter_marked = false;

    if (hasMarkedText()) {
        marked_text = true;
	if (!strncmp(text(), markedText(), strlen(markedText()))) first_letter_marked = true;
    }

    if (keycode ==  Key_Return) {
        if (!text().isEmpty() && !marked_text) emit returnPressed();
	else QApplication::beep();
    }
    else if (keycode == Key_Escape) emit esc_pressed();
    else if (keycode == Key_Backspace
	     && (cursorPosition() != 0 || marked_text)) {
	valid_flag = true;
    }
    else if (keycode == Key_Delete
	     && (cursorPosition() < static_cast<int>(text().length()) || marked_text)) {
	valid_flag = true;
    }

    else if (keycode == Key_Right || keycode == Key_Left
	       || keycode == Key_Home || keycode == Key_End) {
        valid_flag = true;
    }

    else if (isalnum(static_cast<unsigned char> (letter))) {
        letter = toupper(static_cast<unsigned char> (letter));
	valid_flag = true;
    }
    else if ((letter == '/' && cursorPosition() && !first_letter_marked)
	     || (letter == '-'
		 && !first_letter_marked
		 && cursorPosition()
		 && (!tnc_p->tnc_func.active_port
		     || tnc_p->tnc_func.hfmode == Tnc_func::packet))) {
        valid_flag = true;
    }
       
    if (valid_flag) {
        QKeyEvent modified_event(QEvent::KeyPress, keycode,
				    letter, event->state(),
				    event->text().upper(),
				    event->isAutoRepeat(), event->count());
	QLineEdit::keyPressEvent(&modified_event);
    }
}

CommandDialog::CommandDialog(int standard_size, QWidget* qwidget_p):
                               QDialog(qwidget_p, "Command", true) {

    QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
    if (!top_layout_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }

    QLabel* label_p = new QLabel("Kam command to send?\n"
				 "(Press F5 or script button to send\n"
				 "configuration script)", this);
    if (!label_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setAlignment(AlignCenter);
    top_layout_p->addWidget(label_p);

    editbox_p = new CommandEditor(this);
    if (!editbox_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }
    int box_width = standard_size * 5;
    editbox_p->setMinimumHeight(standard_size);
    editbox_p->setMinimumWidth(box_width);

    QBoxLayout* editbox_layout_p = new QHBoxLayout(top_layout_p, 0);
    if (!editbox_layout_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }
    editbox_layout_p->addSpacing(standard_size * 3);
    editbox_layout_p->addWidget(editbox_p, 1);
    editbox_layout_p->addSpacing(standard_size * 3);

    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* script_button_p = new QPushButton("Script", this);
    if (!script_button_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setFixedSize(button_size);
    script_button_p->setFixedSize(button_size);
    cancel_button_p->setFixedSize(button_size);

    QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, standard_size/4);
    if (!button_layout_p) {
        cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);
    button_layout_p->addWidget(ok_button_p);
    button_layout_p->addStretch(2);
    button_layout_p->addWidget(script_button_p);
    button_layout_p->addStretch(2);
    button_layout_p->addWidget(cancel_button_p);
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);

    setCaption("Command");

    ok_button_p->setAutoDefault(true);
    script_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    editbox_p->setFocus();

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(command_accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(command_reject()));
    QObject::connect(script_button_p, SIGNAL(clicked()), this, SLOT(command_script()));
    QObject::connect(editbox_p, SIGNAL(key_pressed(int)), this, SLOT(done(int)));

    top_layout_p->activate();
    top_layout_p->freeze();
}

void CommandEditor::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();
    int letter = event->ascii();
    bool valid_flag = false;
    bool marked_text = false;

    if (hasMarkedText()) marked_text = true;

    if (keycode == Key_Escape) emit key_pressed(CommandDialog::rejected);
    else if (keycode == Key_F5) emit key_pressed(CommandDialog::script);
    else if (keycode == Key_Return && count && !marked_text) emit key_pressed(CommandDialog::accepted);
    else if (keycode == Key_Backspace
	     && (cursorPosition() != 0 || marked_text)) {
        if (count) count--;
	valid_flag = true;
    }
    else if (keycode == Key_Delete
	     && (cursorPosition() < count || marked_text)) {
        if (count) count--;
	valid_flag = true;
    }

    else if (keycode == Key_Right || keycode == Key_Left
	       || keycode == Key_Home || keycode == Key_End) {
        valid_flag = true;
    }

    else if (letter >= 32 && letter <= 127){
        if (count < COMMAND_SIZE || marked_text) {
	    count++;
	    letter = toupper(static_cast<unsigned char> (letter));
	    valid_flag = true;
	}
	else QApplication::beep();
    }
       
    if (valid_flag) {
        QKeyEvent modified_event(QEvent::KeyPress, keycode,
				    letter, event->state(),
				    event->text().upper(),
				    event->isAutoRepeat(), event->count());
	QLineEdit::keyPressEvent(&modified_event);
	if (marked_text) count = strlen(text());  // if we deleted some marked text reset count
    }
}

RstDialog::RstDialog(int standard_size, QWidget* qwidget_p):
                               QDialog(qwidget_p, "RST", true) {

    QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
    if (!top_layout_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }

    QLabel* label_p = new QLabel("New RST Report?", this);
    if (!label_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setAlignment(AlignCenter);
    top_layout_p->addWidget(label_p);

    editbox_p = new RstEditor(this);
    if (!editbox_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }
    int box_width = (standard_size * 4)/3;
    editbox_p->setMinimumHeight(standard_size);
    editbox_p->setFixedWidth(box_width);

    QBoxLayout* editbox_layout_p = new QHBoxLayout(top_layout_p, 0);
    if (!editbox_layout_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }
    editbox_layout_p->addStretch(1);
    editbox_layout_p->addWidget(editbox_p, 0);
    editbox_layout_p->addStretch(1);

    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setFixedSize(button_size);
    cancel_button_p->setFixedSize(button_size);

    QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, standard_size);
    if (!button_layout_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);
    button_layout_p->addWidget(ok_button_p);
    button_layout_p->addStretch(2);
    button_layout_p->addWidget(cancel_button_p);
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);

    setCaption("RST Report");

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));

    editbox_p->setFocus();

    QObject::connect(editbox_p, SIGNAL(returnPressed()), this, SLOT(accept()));
    QObject::connect(editbox_p, SIGNAL(esc_pressed()), this, SLOT(reject()));

    top_layout_p->activate();
    top_layout_p->freeze();
}

void RstEditor::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();
    int letter = event->ascii();
    bool valid_flag = false;
    bool marked_text = false;

    if (hasMarkedText()) marked_text = true;

    if (keycode ==  Key_Return) {
        if (count == 3 && !marked_text) emit returnPressed();
	else QApplication::beep();
    }
    else if (keycode == Key_Escape) emit esc_pressed();
    else if (keycode == Key_Backspace
	     && (cursorPosition() != 0 || marked_text)) {
        if (count) count--;
	valid_flag = true;
    }
    else if (keycode == Key_Delete
	     && (cursorPosition() < count || marked_text)) {
        if (count) count--;
	valid_flag = true;
    }

    else if (keycode == Key_Right || keycode == Key_Left
	     || keycode == Key_Home || keycode == Key_End) {
        valid_flag = true;
    }

    else if (isdigit(letter)
	     && (count < 3 || marked_text)) {
        valid_flag = true;
	count++;
    }

    if (valid_flag) {
        QLineEdit::keyPressEvent(event);
	if (marked_text) count = strlen(text());  // if we deleted some marked text reset count
    }
}

PromptDialog::PromptDialog(const char* text, const char* caption, int standard_size, QWidget* widget_p): QDialog(widget_p, caption, true) {

    QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
    if (!top_layout_p) {
        cerr << "Memory allocation error in PromptDialog::PromptDialog()" << endl;
	exit(MEM_ERROR);
    }

    QLabel* label_p = new QLabel(text, this);
    if (!label_p) {
        cerr << "Memory allocation error in PromptDialog::PromptDialog()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setAlignment(AlignCenter);

    top_layout_p->addWidget(label_p);

    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in PromptDialog::PromptDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in PromptDialog::PromptDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setFixedSize(button_size);
    cancel_button_p->setFixedSize(button_size);

    QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, standard_size);
    if (!button_layout_p) {
        cerr << "Memory allocation error in PromptDialog::PromptDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);
    button_layout_p->addWidget(ok_button_p);
    button_layout_p->addStretch(2);
    button_layout_p->addWidget(cancel_button_p);
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    ok_button_p->setFocus();

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));

    setCaption(caption);

    top_layout_p->activate();
    top_layout_p->freeze();
}


Autocq_modeDialog::Autocq_modeDialog(int standard_size, QWidget* qwidget_p): QDialog(qwidget_p, 0, true) {

    QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
    if (!top_layout_p) {
        cerr << "Memory allocation error in Autocq_modeDialog::Autocq_modeDialog()" << endl;
	exit(MEM_ERROR);
    }

    QButtonGroup* buttongroup_p = new QVButtonGroup(this);
    if (!buttongroup_p) {
        cerr << "Memory allocation error in Autocq_modeDialog::Autocq_modeDialog()" << endl;
	exit(MEM_ERROR);
    }
    buttongroup_p->setTitle("Auto-CQ Mode?");

    QRadioButton* radio_button_p;
    radio_button_p = new QRadioButton("Amtor FEC", buttongroup_p);
    if (!radio_button_p) {
        cerr << "Memory allocation error in Autocq_modeDialog::Autocq_modeDialog()" << endl;
	exit(MEM_ERROR);
    }

    radio_button_p = new QRadioButton("Pactor FEC", buttongroup_p);
    if (!radio_button_p) {
        cerr << "Memory allocation error in Autocq_modeDialog::Autocq_modeDialog()" << endl;
	exit(MEM_ERROR);
    }
    buttongroup_p->setButton(0);

    top_layout_p->addWidget(buttongroup_p, 0, Qt::AlignCenter);

    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setFixedSize(button_size);
    cancel_button_p->setFixedSize(button_size);

    QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, standard_size);
    if (!button_layout_p) {
        cerr << "Memory allocation error in Autocq_modeDialog::Autocq_modeDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);
    button_layout_p->addWidget(ok_button_p);
    button_layout_p->addStretch(2);
    button_layout_p->addWidget(cancel_button_p);
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    ok_button_p->setFocus();

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));
    QObject::connect(buttongroup_p, SIGNAL(clicked(int)), this, SLOT(mode_button_pressed(int)));

    setCaption("Auto-CQ Mode");

    top_layout_p->activate();
    top_layout_p->freeze();
}

void Autocq_modeDialog::mode_button_pressed(int mode) {
    if (mode) prog_func.tor_autocq_mode = Prog_func::pactor;
    else prog_func.tor_autocq_mode = Prog_func::amtor;
}

GplDialog::GplDialog(int size, QWidget* qwidget_p): QDialog(qwidget_p, "GPL", true), standard_size(size) {

    editbox_p = new GplEditor(this);
    if (!editbox_p) {
        cerr << "Memory allocation error in GplDialog::GplDialog()" << endl;
	exit(MEM_ERROR);
    }
    editbox_p->setFont(QFont("Courier", 12, QFont::Normal));
    setCaption("kamplus-qt: Conditions, Notices and Disclaimers");

    label_p = new QLabel("Do you accept the Conditions, Notices and Disclaimers shown above?", this);
    if (!label_p) {
        cerr << "Memory allocation error in CallsignDialog::CallsignDialog()" << endl;
	exit(MEM_ERROR);
    }

    label_p->setAlignment(AlignCenter);

    ok_button_p = new QPushButton("Accept", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in GplDialog::GplDialog()" << endl;
	exit(MEM_ERROR);
    }

    cancel_button_p = new QPushButton("Reject", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in GplDialog::GplDialog()" << endl;
	exit(MEM_ERROR);
    }

    button_size = cancel_button_p->sizeHint();
    
    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));

    QFont font("courier", 12, QFont::Normal);
    editbox_p->setFont(font);
    editbox_p->setFocus();
    editbox_p->setReadOnly(true);
    QObject::connect(editbox_p, SIGNAL(esc_pressed()), this, SLOT(reject()));

    setMinimumSize(standard_size * 20, standard_size * 14);
#if QT_VERSION >= 300
    editbox_p->setTextFormat(PlainText);
#endif
    editbox_p->setText(copyright_msg);

    ok_button_p->setAutoDefault(false);
    cancel_button_p->setAutoDefault(false);
}

void GplDialog::resizeEvent(QResizeEvent* event) {
    int widget_height = event->size().height();
    int widget_width = event->size().width();
    
    editbox_p->setGeometry(2, standard_size/2, widget_width - 4, widget_height - (standard_size * 7)/2);
    label_p->setGeometry(0, widget_height - (standard_size * 5)/2, widget_width, standard_size/2);
    ok_button_p->setGeometry(standard_size, widget_height - (standard_size * 3)/2,
			     button_size.width(), button_size.height());
    cancel_button_p->setGeometry(widget_width - (standard_size + button_size.width()),
				 widget_height - (standard_size * 3)/2, button_size.width(),
				 button_size.height());
}

void GplEditor::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();

    if (keycode == Key_Escape) emit esc_pressed();

    else if (keycode == Key_Right || keycode == Key_Left
	     || keycode == Key_Home || keycode == Key_End
	     || keycode == Key_Up || keycode == Key_Down
	     || keycode == Key_PageUp || keycode == Key_PageDown) {
#if QT_VERSION >= 300
        QTextEdit::keyPressEvent(event);
#else
        QMultiLineEdit::keyPressEvent(event);
#endif
    }
}


HelpDialog::HelpDialog(int size): QDialog(0, "Help", false), standard_size(size),
                                                      open_flag(false) {

    string filename(prog_func.filedir);
    filename += "/helpfile-qt";
#ifdef HAVE_IOS_NOCREATE
    ifstream helpfile(filename.c_str(), ios::in | ios::nocreate);
#else
    // we must have Std C++ so we probably don't need a ios::nocreate
    // flag on a read open to ensure uniqueness
    ifstream helpfile(filename.c_str(), ios::in);
#endif

    if (helpfile) {
        helpfile.seekg(0, ios::end);
	streampos size = helpfile.tellg();
	if (size) {
	    helpfile.seekg(0); // go to beginning of file
	    char letter;
	    char* help_msg = new char[(int)size];
	    if (!help_msg) {
	        cerr << "Memory allocation error in HelpDialog::HelpDialog()" << endl;
		exit(MEM_ERROR);
	    }

	    char* char_p;
	    for (char_p = help_msg; (helpfile.get(letter)); char_p++) {
	        *char_p = letter;
	    }
	    char_p--;
	    *char_p = 0;    // null terminate help_msg
	    open_flag = true;
	    helpfile.close();

	    editbox_p = new HelpEditor(this);
	    if (!editbox_p) {
	        cerr << "Memory allocation error in HelpDialog::HelpDialog()" << endl;
		exit(MEM_ERROR);
	    }
	    editbox_p->setFont(QFont("courier", 12, QFont::Normal));
	    setCaption("kam-qt: Helpfile");

	    cancel_button_p = new QPushButton("Cancel", this);
	    if (!cancel_button_p) {
	        cerr << "Memory allocation error in HelpDialog::HelpDialog()" << endl;
		exit(MEM_ERROR);
	    }

	    button_size = cancel_button_p->sizeHint();
    
	    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(kill_dialog()));

	    QFont font("Courier", 12, QFont::Normal);
	    editbox_p->setFont(font);
	    editbox_p->setFocus();
	    editbox_p->setReadOnly(true);
	    QObject::connect(editbox_p, SIGNAL(esc_pressed()), this, SLOT(kill_dialog()));
	    
	    setMinimumSize(standard_size * 20, standard_size * 14);
#if QT_VERSION >= 300
	    editbox_p->setTextFormat(PlainText);
#endif
	    editbox_p->setText(help_msg);
	    delete[] help_msg;

	    cancel_button_p->setAutoDefault(false);
	}
    }
}

void HelpDialog::resizeEvent(QResizeEvent* event) {
    int widget_height = event->size().height();
    int widget_width = event->size().width();
    
    editbox_p->setGeometry(2, standard_size/2, widget_width - 4, widget_height - (standard_size * 5)/2);
    cancel_button_p->setGeometry((widget_width - button_size.width())/2,
				 widget_height - (standard_size * 3)/2, button_size.width(),
				 button_size.height());
}

void HelpEditor::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();

    if (keycode == Key_Escape) emit esc_pressed();

    else if (keycode == Key_Right || keycode == Key_Left
	     || keycode == Key_Home || keycode == Key_End
	     || keycode == Key_Up || keycode == Key_Down
	     || keycode == Key_PageUp || keycode == Key_PageDown) {
#if QT_VERSION >= 300
        QTextEdit::keyPressEvent(event);
#else
        QMultiLineEdit::keyPressEvent(event);
#endif
    }
}


CwSpeedSlider::CwSpeedSlider(int initial_speed, QWidget* qwidget_p, const char* name): 
                             QSlider(5, 50, 5, initial_speed, QSlider::Horizontal, qwidget_p, name) {
    setTickmarks(QSlider::Below);
    setTickInterval(5);
    setTracking(true);
}

void CwSpeedSlider::keyPressEvent(QKeyEvent* event) {
    if (event->key() != Key_Return) QSlider::keyPressEvent(event); // we just want to intercept the return key
}

CwSpeedDialog::CwSpeedDialog(int standard_size, int cw_speed, QWidget* qwidget_p):
                               QDialog(qwidget_p, "CwSpeed", true) {

    QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
    if (!top_layout_p) {
        cerr << "Memory allocation error in CwSpeedDialog::CwSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }

    QLabel* label_p = new QLabel("New Speed (5 - 50 WPM)?", this);
    if (!label_p) {
        cerr << "Memory allocation error in CwSpeedDialog::CwSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setAlignment(AlignCenter);

    qlcdnumber_p = new QLCDNumber(2, this);
    if (!qlcdnumber_p) {
        cerr << "Memory allocation error in CwSpeedDialog::CwSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    qlcdnumber_p->setFixedSize(standard_size, standard_size);
    qlcdnumber_p->display(cw_speed);

    QBoxLayout* speed_display_layout_p = new QHBoxLayout(top_layout_p, 0);
    if (!speed_display_layout_p) {
        cerr << "Memory allocation error in CwSpeedDialog::CwSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    speed_display_layout_p->addWidget(label_p, 1);
    speed_display_layout_p->addWidget(qlcdnumber_p, 0);

    cw_speed_slider_p = new CwSpeedSlider(cw_speed, this, "Cw Speed Slider");
    if (!cw_speed_slider_p) {
        cerr << "Memory allocation error in CwSpeedDialog::CwSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }

    QBoxLayout* slider_layout_p = new QHBoxLayout(top_layout_p, 0);
    if (!slider_layout_p) {
        cerr << "Memory allocation error in CwSpeedDialog::CwSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    slider_layout_p->addSpacing(standard_size/2);
    slider_layout_p->addWidget(cw_speed_slider_p);
    slider_layout_p->addSpacing(standard_size/2);

    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in RstDialog::RstDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in CwSpeedDialog::CwSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setFixedSize(button_size);
    cancel_button_p->setFixedSize(button_size);

    QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, standard_size);
    if (!button_layout_p) {
        cerr << "Memory allocation error in CwSpeedDialog::CwSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);
    button_layout_p->addWidget(ok_button_p);
    button_layout_p->addStretch(2);
    button_layout_p->addWidget(cancel_button_p);
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);

    setCaption("CW Speed");

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    ok_button_p->setFocus();

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));
    QObject::connect(cw_speed_slider_p, SIGNAL(valueChanged(int)), this, SLOT(display(int)));
    QObject::connect(cw_speed_slider_p, SIGNAL(sliderReleased(void)), this, SLOT(set_sliderpos()));

    top_layout_p->activate();
    top_layout_p->freeze();
}

RttySpeedDialog::RttySpeedDialog(Rtty_speed speed_, int standard_size, QWidget* qwidget_p):
                                 QDialog(qwidget_p, 0, true), speed(speed_) {

    QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
    if (!top_layout_p) {
        cerr << "Memory allocation error in RttySpeedDialog::RttySpeedDialog()" << endl;
	exit(MEM_ERROR);
    }

    QButtonGroup* buttongroup_p = new QButtonGroup(this);
    if (!buttongroup_p) {
        cerr << "Memory allocation error in RttySpeedDialog::RttySpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    buttongroup_p->hide();

    QGridLayout* button_grid_p = new QGridLayout(top_layout_p, 4, 3, 0);
    if (!button_grid_p) {
        cerr << "Memory allocation error in RttySpeedDialog::RttySpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_grid_p->addColSpacing(0, standard_size);
    button_grid_p->setColStretch(2, 1);

    QRadioButton* radio_button_p;
    radio_button_p = new QRadioButton("45 baud", this);
    if (!radio_button_p) {
        cerr << "Memory allocation error in RttySpeedDialog::RttySpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    buttongroup_p->insert(radio_button_p);
    button_grid_p->addWidget(radio_button_p, 0, 1);

    radio_button_p = new QRadioButton("50 baud", this);
    if (!radio_button_p) {
        cerr << "Memory allocation error in RttySpeedDialog::RttySpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    buttongroup_p->insert(radio_button_p);
    button_grid_p->addWidget(radio_button_p, 1, 1);

    radio_button_p = new QRadioButton("57 baud", this);
    if (!radio_button_p) {
        cerr << "Memory allocation error in RttySpeedDialog::RttySpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    buttongroup_p->insert(radio_button_p);
    button_grid_p->addWidget(radio_button_p, 2, 1);

    radio_button_p = new QRadioButton("75 baud", this);
    if (!radio_button_p) {
        cerr << "Memory allocation error in RttySpeedDialog::RttySpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    buttongroup_p->insert(radio_button_p);
    button_grid_p->addWidget(radio_button_p, 3, 1);

    if (speed == b45) buttongroup_p->setButton(0);
    else if (speed == b50) buttongroup_p->setButton(1);
    else if (speed == b57) buttongroup_p->setButton(2);
    else if (speed == b75) buttongroup_p->setButton(3);
    
    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in RttySpeedDialog::RttySpeedDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in RttySpeedDialog::RttySpeedDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setFixedSize(button_size);
    cancel_button_p->setFixedSize(button_size);

    QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, standard_size);
    if (!button_layout_p) {
        cerr << "Memory allocation error in RttySpeedDialog::RttySpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);
    button_layout_p->addWidget(ok_button_p);
    button_layout_p->addStretch(2);
    button_layout_p->addWidget(cancel_button_p);
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    ok_button_p->setFocus();

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));
    QObject::connect(buttongroup_p, SIGNAL(clicked(int)), this, SLOT(speed_button_pressed(int)));

    setCaption("RTTY Speed");

    top_layout_p->activate();
    top_layout_p->freeze();
}

void RttySpeedDialog::speed_button_pressed(int speed_button) {
    if (!speed_button) speed = b45;
    else if (speed_button == 1) speed = b50;
    else if (speed_button == 2) speed = b57;
    else if (speed_button == 3) speed = b75;
}

AsciiSpeedDialog::AsciiSpeedDialog(Ascii_speed speed_, int standard_size, QWidget* qwidget_p):
                                 QDialog(qwidget_p, 0, true), speed(speed_) {

    QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
    if (!top_layout_p) {
        cerr << "Memory allocation error in AsciiSpeedDialog::AsciiSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }

    QButtonGroup* buttongroup_p = new QButtonGroup(this);
    if (!buttongroup_p) {
        cerr << "Memory allocation error in AsciiSpeedDialog::AsciiSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    buttongroup_p->hide();

    QGridLayout* button_grid_p = new QGridLayout(top_layout_p, 4, 3, 0);
    if (!button_grid_p) {
        cerr << "Memory allocation error in AsciiSpeedDialog::AsciiSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_grid_p->addColSpacing(0, standard_size);
    button_grid_p->setColStretch(2, 1);

    QRadioButton* radio_button_p;
    radio_button_p = new QRadioButton("50 baud", this);
    if (!radio_button_p) {
        cerr << "Memory allocation error in AsciiSpeedDialog::AsciiSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    buttongroup_p->insert(radio_button_p);
    button_grid_p->addWidget(radio_button_p, 0, 1);

    radio_button_p = new QRadioButton("100 baud", this);
    if (!radio_button_p) {
        cerr << "Memory allocation error in AsciiSpeedDialog::AsciiSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    buttongroup_p->insert(radio_button_p);
    button_grid_p->addWidget(radio_button_p, 1, 1);

    radio_button_p = new QRadioButton("110 baud", this);
    if (!radio_button_p) {
        cerr << "Memory allocation error in AsciiSpeedDialog::AsciiSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    buttongroup_p->insert(radio_button_p);
    button_grid_p->addWidget(radio_button_p, 2, 1);

    radio_button_p = new QRadioButton("200 baud", this);
    if (!radio_button_p) {
        cerr << "Memory allocation error in AsciiSpeedDialog::AsciiSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    buttongroup_p->insert(radio_button_p);
    button_grid_p->addWidget(radio_button_p, 3, 1);

    if (speed == b50) buttongroup_p->setButton(0);
    else if (speed == b100) buttongroup_p->setButton(1);
    else if (speed == b110) buttongroup_p->setButton(2);
    else if (speed == b200) buttongroup_p->setButton(3);
    
    QPushButton* ok_button_p = new QPushButton("OK", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in AsciiSpeedDialog::AsciiSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in AsciiSpeedDialog::AsciiSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setFixedSize(button_size);
    cancel_button_p->setFixedSize(button_size);

    QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, standard_size);
    if (!button_layout_p) {
        cerr << "Memory allocation error in AsciiSpeedDialog::AsciiSpeedDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);
    button_layout_p->addWidget(ok_button_p);
    button_layout_p->addStretch(2);
    button_layout_p->addWidget(cancel_button_p);
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);

    ok_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    ok_button_p->setFocus();

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(accept()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));
    QObject::connect(buttongroup_p, SIGNAL(clicked(int)), this, SLOT(speed_button_pressed(int)));

    setCaption("ASCII Speed");

    top_layout_p->activate();
    top_layout_p->freeze();
}

void AsciiSpeedDialog::speed_button_pressed(int speed_button) {
    if (!speed_button) speed = b50;
    else if (speed_button == 1) speed = b100;
    else if (speed_button == 2) speed = b110;
    else if (speed_button == 3) speed = b200;
}

DownloadDialog::DownloadDialog(const char* filename, Tnc* tnc_p_, MainScreen* ms_p, ReceiveWin* rw_p,
			        int standard_size):
                                  QDialog(0, "Download", false), tnc_p(tnc_p_),
                                  mainscreen_p(ms_p), receivewin_p(rw_p) {

    QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
    if (!top_layout_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }

    stream = tnc_p->tnc_func.active_stream();
    port = tnc_p->tnc_func.active_port;
    ostrstream strm;
    strm << "\nDownloading ";
    if (!port) {
        strm << "Vhf port stream " << (char)(stream + 0x41) << '\n';
    } 
    else {
        strm << "Hf port ";
	if (tnc_p->tnc_func.hfmode != Tnc_func::packet) strm << "Tor stream\n";
	else strm << "stream " <<  (char)(stream + 0x41) << '\n';
    }
    strm << "to file: " << filename << '\n' << ends;
    char* text = strm.str();

    QLabel* label_p = new QLabel(text, this);
    if (!label_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }
    delete[] text;
    label_p->setAlignment(AlignCenter);

    top_layout_p->addWidget(label_p, 1);

    QPushButton* cancel_button_p = new QPushButton("End", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }
    cancel_button_p->setFixedSize(cancel_button_p->sizeHint());
    
    QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, 0);
    if (!button_layout_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_layout_p->addStretch(1);
    button_layout_p->addWidget(cancel_button_p);
    button_layout_p->addStretch(1);

    QHBox* hbox_p = new QHBox(this);
    if (!hbox_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }
    hbox_p->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    hbox_p->setMinimumSize(standard_size * 8, standard_size);
    hbox_p->setLineWidth(1);

    QLabel* intro_label_p = new QLabel(" Bytes downloaded: ", hbox_p);
    if (!intro_label_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }
    intro_label_p->setAlignment(AlignLeft | AlignVCenter);
    hbox_p->setStretchFactor(intro_label_p, 0);

    bytes_label_p = new QLabel(hbox_p);
    if (!bytes_label_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }
    bytes_label_p->setAlignment(AlignLeft | AlignVCenter);
    hbox_p->setStretchFactor(bytes_label_p, 1);

    top_layout_p->addWidget(hbox_p);
    show_bytes(0);

    setCaption("kam-qt: Download");
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(end_download()));

    cancel_button_p->setAutoDefault(false);

    top_layout_p->activate();
    top_layout_p->freeze();
}	

void DownloadDialog::end_download(void) {
    DownloadList* download_list_p = tnc_p->tnc_func.download_list_ptr;
    // we need to find the download file object receiving the
    // file on the dialog's stream and port
    download_list_p->reset(DList_enum::bottom_end);
    DownloadFile* dload_file_p;
    while ((dload_file_p = (DownloadFile*)download_list_p->inspect(DList_enum::up)) != 0
	     && !(dload_file_p->get_stream() == stream
		    && dload_file_p->get_port() == port)) {}
    if (dload_file_p) {
        download_list_p->extract();
	mainscreen_p->display_connected_status();
	dload_file_p->dialog_take_ownership(); // we are now responsible for deleting this object
	delete dload_file_p;  // the DownloadFile object used to have ownership of the relevant dialog
	                      // object of this class so it is now safe to do 'delete this'
	delete this;          // this is safe - see above
    }
    else {
        receivewin_p->write("\nOops - can't find the download file object to delete\n"
			    "Please report bug\n");
	QApplication::beep();
    }
}

void DownloadDialog::show_bytes(long bytes) {
    ostrstream strm;
    strm << bytes << ends;
    char* text = strm.str();
    bytes_label_p->setText(text);
    delete[] text;
}

UploadDialog::UploadDialog(const char* filename, long filesize, Tnc* tnc_p_, BufferList& bl,
			   MainScreen* ms_p, ReceiveWin* rw_p, int standard_size):
                                  QDialog(0, "Upload", false), tnc_p(tnc_p_),
                                  buffer_list(bl), mainscreen_p(ms_p), receivewin_p(rw_p) {

    QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
    if (!top_layout_p) {
        cerr << "Memory allocation error in UploadDialog::UploadDialog()" << endl;
	exit(MEM_ERROR);
    }

    stream = tnc_p->tnc_func.active_stream();
    port = tnc_p->tnc_func.active_port;
    ostrstream strm;
    strm << "Uploading file: " << filename << "\nto ";
    if (!port) {
        strm << "Vhf port stream " << (char)(stream + 0x41) << '\n';
    } 
    else {
        strm << "Hf port ";
	if (tnc_p->tnc_func.hfmode != Tnc_func::packet) strm << "Tor stream\n";
	else strm << "stream " <<  (char)(stream + 0x41) << '\n';
    }
    strm << "(File size: ";
    if (filesize > 99999) strm << filesize/1024 << " Kb)" << ends;
    else strm << filesize << " bytes)" << ends;
    char* text = strm.str();

    QLabel* label_p = new QLabel(text, this);
    if (!label_p) {
        cerr << "Memory allocation error in UploadDialog::UploadDialog()" << endl;
	exit(MEM_ERROR);
    }
    delete[] text;
    label_p->setAlignment(AlignCenter);

    top_layout_p->addWidget(label_p, 1);

    QPushButton* cancel_button_p = new QPushButton("Cancel", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in UploadDialog::UploadDialog()" << endl;
	exit(MEM_ERROR);
    }
    cancel_button_p->setFixedSize(cancel_button_p->sizeHint());
    
    QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, 0);
    if (!button_layout_p) {
        cerr << "Memory allocation error in UploadDialog::UploadDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_layout_p->addStretch(1);
    button_layout_p->addWidget(cancel_button_p);
    button_layout_p->addStretch(1);

    QHBox* hbox_p = new QHBox(this);
    if (!hbox_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }
    hbox_p->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    hbox_p->setMinimumSize(standard_size * 8, standard_size);
    hbox_p->setLineWidth(1);

    QLabel* intro_label_p = new QLabel(" Bytes sent to Kam: ", hbox_p);
    if (!intro_label_p) {
        cerr << "Memory allocation error in UploadDialog::UploadDialog()" << endl;
	exit(MEM_ERROR);
    }
    intro_label_p->setAlignment(AlignLeft | AlignVCenter);
    hbox_p->setStretchFactor(intro_label_p, 0);

    bytes_label_p = new QLabel(hbox_p);
    if (!bytes_label_p) {
        cerr << "Memory allocation error in UploadDialog::UploadDialog()" << endl;
	exit(MEM_ERROR);
    }
    bytes_label_p->setAlignment(AlignLeft | AlignVCenter);
    hbox_p->setStretchFactor(bytes_label_p, 1);

    top_layout_p->addWidget(hbox_p);
    show_bytes(0);

    setCaption("kam-qt: Upload");
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(end_upload()));

    cancel_button_p->setAutoDefault(false);

    top_layout_p->activate();
    top_layout_p->freeze();
}	

void UploadDialog::end_upload(void) {
    if (buffer_list.get_upload_status(stream, port) == BufferList::file) { // check for sanity
        // we now need to find the file buffer sending the
        // message on the dialog's stream and port
        buffer_list.reset(DList_enum::bottom_end);
	FileBuffer* file_buffer_p;
	while ((file_buffer_p = (FileBuffer*)buffer_list.inspect(DList_enum::up)) != 0
	         && !(file_buffer_p->get_stream() == stream
		        && file_buffer_p->get_port() == port)) {}
	if (file_buffer_p) file_buffer_p->end_loading();
	else {
	    receivewin_p->write("\nOops - can't find the file buffer to remove\n"
	                        "Please report bug\n");
	    QApplication::beep();
	}
    }
    // This method simply marks the FileBuffer object as available for removal
    // from the BufferList object, and does not remove the object itself
    // UploadDialog objects are modeless and are deleted by the FilesendBuffer object
    // when that object is deleted in MainScreen::timerEvent 
}

void UploadDialog::show_bytes(long bytes) {
    ostrstream strm;
    strm << bytes << ends;
    char* text = strm.str();
    bytes_label_p->setText(text);
    delete[] text;
}

PrintMarkDialog::PrintMarkDialog(int standard_size, QWidget* qwidget_p):
                                 QDialog(qwidget_p, "PrintMark", true) {

    QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
    if (!top_layout_p) {
        cerr << "Memory allocation error in PrintMarkDialog::PrintMarkDialog()" << endl;
	exit(MEM_ERROR);
    }

    QLabel* label_p = new QLabel("Print from the mark?\n"
				 "(Press Cancel button to cancel the print job)", this);
    if (!label_p) {
        cerr << "Memory allocation error in PrintMarkDialog::PrintMarkDialog()" << endl;
	exit(MEM_ERROR);
    }
    label_p->setAlignment(AlignCenter);
    top_layout_p->addWidget(label_p);

    QPushButton* ok_button_p = new QPushButton("Ok", this);
    if (!ok_button_p) {
        cerr << "Memory allocation error in PrintMarkDialog::PrintMarkDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* no_button_p = new QPushButton("No", this);
    if (!no_button_p) {
        cerr << "Memory allocation error in PrintMarkDialog::PrintMarkDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* cancel_button_p = new QPushButton("Cancel job", this);
    if (!cancel_button_p) {
        cerr << "Memory allocation error in PrintMarkDialog::PrintMarkDialog()" << endl;
	exit(MEM_ERROR);
    }

    QSize button_size = cancel_button_p->sizeHint();
    ok_button_p->setFixedSize(button_size);
    no_button_p->setFixedSize(button_size);
    cancel_button_p->setFixedSize(button_size);

    QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, standard_size/4);
    if (!button_layout_p) {
        cerr << "Memory allocation error in PrintMarkDialog::PrintMarkDialog()" << endl;
	exit(MEM_ERROR);
    }
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);
    button_layout_p->addWidget(ok_button_p);
    button_layout_p->addStretch(2);
    button_layout_p->addWidget(no_button_p);
    button_layout_p->addStretch(2);
    button_layout_p->addWidget(cancel_button_p);
    button_layout_p->addStretch(1);
    button_layout_p->addSpacing(standard_size/2);

    ok_button_p->setAutoDefault(true);
    no_button_p->setAutoDefault(true);
    cancel_button_p->setAutoDefault(true);

    ok_button_p->setFocus();

    setCaption("Print from mark");

    QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(command_accept()));
    QObject::connect(no_button_p, SIGNAL(clicked()), this, SLOT(command_reject()));
    QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(command_cancel_job()));

    top_layout_p->activate();
    top_layout_p->freeze();
}

void PrintMarkDialog::keyPressEvent(QKeyEvent* event) {
    if (event->key() == Key_Escape) done(rejected);
    else QDialog::keyPressEvent(event);
}

QsoCountDialog::QsoCountDialog(int standard_size): QDialog(0, "QsoCount", false) {

    QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
    if (!top_layout_p) {
        cerr << "Memory allocation error in QsoCountDialog::QsoCountDialog()" << endl;
	exit(MEM_ERROR);
    }

    QHBox* hbox_p = new QHBox(this);
    if (!hbox_p) {
        cerr << "Memory allocation error in QsoCountDialog::QsoCountDialog()" << endl;
	exit(MEM_ERROR);
    }
    hbox_p->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    hbox_p->setMinimumSize(standard_size * 6, standard_size);
    hbox_p->setLineWidth(1);

    QLabel* intro_label_p = new QLabel(" QSO number: ", hbox_p);
    if (!intro_label_p) {
        cerr << "Memory allocation error in QsoCountDialog::QsoCountDialog()" << endl;
	exit(MEM_ERROR);
    }
    intro_label_p->setAlignment(AlignRight | AlignVCenter);
    hbox_p->setStretchFactor(intro_label_p, 0);
 
    qso_label_p = new QLabel(hbox_p);
    if (!qso_label_p) {
        cerr << "Memory allocation error in DownloadDialog::DownloadDialog()" << endl;
	exit(MEM_ERROR);
    }
    qso_label_p->setAlignment(AlignLeft | AlignVCenter);
    hbox_p->setStretchFactor(qso_label_p, 1);

    top_layout_p->addWidget(hbox_p);

    QPushButton* dec_button_p = new QPushButton("-", this);
    if (!dec_button_p) {
        cerr << "Memory allocation error in QsoCountDialog::QsoCountDialog()" << endl;
	exit(MEM_ERROR);
    }

    QPushButton* inc_button_p = new QPushButton("+", this);
    if (!inc_button_p) {
        cerr << "Memory allocation error in QsoCountDialog::QsoCountDialog()" << endl;
	exit(MEM_ERROR);
    }

    const int inc_dec_button_width = (standard_size * 3)/2;
    dec_button_p->setFixedSize(inc_dec_button_width, inc_dec_button_width);
    inc_button_p->setFixedSize(inc_dec_button_width, inc_dec_button_width);
    dec_button_p->setFont(QFont("Helvetica", 18, QFont::Normal));
    inc_button_p->setFont(QFont("Helvetica", 18, QFont::Normal));

    top_layout_p->addStretch(1);
    QBoxLayout* inc_dec_button_layout_p = new QHBoxLayout(top_layout_p, standard_size);
    if (!inc_dec_button_layout_p) {
        cerr << "Memory allocation error in QsoCountDialog::QsoCountDialog()" << endl;
	exit(MEM_ERROR);
    }
    inc_dec_button_layout_p->addStretch(1);
    inc_dec_button_layout_p->addSpacing(standard_size/2);
    inc_dec_button_layout_p->addWidget(dec_button_p);
    inc_dec_button_layout_p->addStretch(2);
    inc_dec_button_layout_p->addWidget(inc_button_p);
    inc_dec_button_layout_p->addStretch(1);
    inc_dec_button_layout_p->addSpacing(standard_size/2);

    QPushButton* quit_button_p = new QPushButton("Quit", this);
    if (!quit_button_p) {
        cerr << "Memory allocation error in QsoCountDialog::QsoCountDialog()" << endl;
	exit(MEM_ERROR);
    }

    quit_button_p->setFixedSize(quit_button_p->sizeHint());
    
    QBoxLayout* quit_button_layout_p = new QHBoxLayout(top_layout_p, 0);
    if (!quit_button_layout_p) {
        cerr << "Memory allocation error in QsoCountDialog::QsoCountDialog()" << endl;
	exit(MEM_ERROR);
    }
    quit_button_layout_p->addStretch(1);
    quit_button_layout_p->addWidget(quit_button_p);
    quit_button_layout_p->addStretch(1);

    setCaption("QSO Counter");

    QObject::connect(inc_button_p, SIGNAL(clicked()), this, SIGNAL(inc()));
    QObject::connect(dec_button_p, SIGNAL(clicked()), this, SIGNAL(dec()));
    QObject::connect(quit_button_p, SIGNAL(clicked()), this, SLOT(destroy()));

    top_layout_p->activate();
    top_layout_p->freeze();

    dec_button_p->setAutoDefault(false);
    inc_button_p->setAutoDefault(false);
    quit_button_p->setAutoDefault(false);
}

void QsoCountDialog::destroy(void) {
    delete this; // this is entirely safe - this object is self-owning and cannot be used again
                 // after it destroys itself - it has no publicly usable methods
}

QsoCountDialog::~QsoCountDialog(void) {
    emit quit_button_pressed();
}

void QsoCountDialog::show_qso_count(int qso_count) {
    ostrstream strm;
    strm << setfill('0') << setw(3);
    strm << qso_count << ends;
    char* text = strm.str();
    qso_label_p->setText(text);
    delete[] text;
}

void QsoCountDialog::keyPressEvent(QKeyEvent* event) {

    int keycode = event->key();

    if (keycode == Key_Plus || keycode == Key_Equal || keycode == Key_Up) emit inc();
    else if (keycode == Key_Minus || keycode == Key_Underscore || keycode == Key_Down) emit dec();
}
