/* Copyright (C) 2001, 2002 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#ifndef SETTINGS_H
#define SETTINGS_H

#ifndef CONFIG_H
#define CONFIG_H
#include "config.h"
#endif

#include <unistd.h>

#include <string>

#include <qwidget.h>
#include <qdialog.h>
#include <qlineedit.h>
#include <qradiobutton.h>
#include <qcheckbox.h>
#include <qlayout.h>

#include "settings_help.h"
#include "prog_defs.h"

class IdentityTable: public QWidget {

  Q_OBJECT

  IdentityMessages help_messages;
  QLineEdit* mycall_entry_p;
  QLineEdit* mypacketcall_entry_p;
  QLineEdit* myselcall_entry_p;
private slots:
  void show_mycall_help(void);
  void show_mypacketcall_help(void);
  void show_myselcall_help(void);
signals:
  void show_help_sig(const std::string&, const std::string&);
public:
  std::string get_mycall(void) const {return static_cast<const char*>(mycall_entry_p->text());}
  std::string get_mypacketcall(void) const {return static_cast<const char*>(mypacketcall_entry_p->text());}
  std::string get_myselcall(void) const {return static_cast<const char*>(myselcall_entry_p->text());}
  void set_mycall(const std::string& mycall) {mycall_entry_p->setText(mycall.c_str());}
  void set_mypacketcall(const std::string& mypacketcall) {mypacketcall_entry_p->setText(mypacketcall.c_str());}
  void set_myselcall(const std::string& myselcall) {myselcall_entry_p->setText(myselcall.c_str());}
  void clear(void);
  IdentityTable(const int, QWidget* widget_p = 0);
};

class ComTable: public QWidget {

  Q_OBJECT

  ComMessages help_messages;
  QLineEdit* port_entry_p;
  QRadioButton* b1200_button_p;
  QRadioButton* b2400_button_p;
  QRadioButton* b4800_button_p;
  QRadioButton* b9600_button_p;
private slots:
  void show_port_help(void);
  void show_speed_help(void);
signals:
  void show_help_sig(const std::string&, const std::string&);
public:
  std::string get_port(void) const {return static_cast<const char*>(port_entry_p->text());}
  std::string get_speed(void) const;
  void set_port(const std::string& port) {port_entry_p->setText(port.c_str());}
  void set_speed(const int);
  void clear(void);
  ComTable(const int, QWidget* widget_p = 0);
};

class PacketTable: public QWidget {

  Q_OBJECT

  PacketMessages help_messages;
  QLineEdit* vhf_packet_size_entry_p;
  QLineEdit* hf_packet_size_entry_p;
  QLineEdit* keepalive_mins_entry_p;
  QCheckBox* check_box_p;
private slots:
  void show_vhf_packet_size_help(void);
  void show_hf_packet_size_help(void);
  void show_keepalive_mins_help(void);
signals:
  void show_help_sig(const std::string&, const std::string&);
public:
  std::string get_vhf_packet_size(void) const {return static_cast<const char*>(vhf_packet_size_entry_p->text());}
  std::string get_hf_packet_size(void) const {return static_cast<const char*>(hf_packet_size_entry_p->text());}
  std::string get_keepalive_mins(void) const {return static_cast<const char*>(keepalive_mins_entry_p->text());}
  bool get_keepalive(void) {return check_box_p->isChecked();}
  void set_vhf_packet_size(const std::string& vhf_packet_size) {vhf_packet_size_entry_p->setText(vhf_packet_size.c_str());}
  void set_hf_packet_size(const std::string& hf_packet_size) {hf_packet_size_entry_p->setText(hf_packet_size.c_str());}
  void set_keepalive_mins(const std::string& keepalive_mins) {keepalive_mins_entry_p->setText(keepalive_mins.c_str());}
  void set_keepalive(bool active) {check_box_p->setChecked(active);}
  void clear(void);
  PacketTable(const int, QWidget* widget_p = 0);
};

class InterfaceTable: public QWidget {

  Q_OBJECT

  InterfaceMessages help_messages;
  QCheckBox* receive_cw_box_p;
  QCheckBox* sound_bell_box_p;
  QCheckBox* no_prompt_box_p;
private slots:
  void show_receive_cw_help(void);
  void show_sound_bell_help(void);
  void show_no_prompt_help(void);
signals:
  void show_help_sig(const std::string&, const std::string&);
public:
  bool get_receive_cw(void) {return receive_cw_box_p->isChecked();}
  bool get_sound_bell(void) {return sound_bell_box_p->isChecked();}
  bool get_no_prompt(void) {return no_prompt_box_p->isChecked();}
  void set_receive_cw(bool active) {receive_cw_box_p->setChecked(active);}
  void set_sound_bell(bool active) {sound_bell_box_p->setChecked(active);}
  void set_no_prompt(bool active) {no_prompt_box_p->setChecked(active);}
  void clear(void);
  InterfaceTable(const int, QWidget* widget_p = 0);
};

class CharsetTable: public QWidget {

  Q_OBJECT

  CharsetMessages help_messages;
  QRadioButton* cp437_button_p;
  QRadioButton* latin_1_button_p;
private slots:
  void show_charset_help(void);
signals:
  void show_help_sig(const std::string&, const std::string&);
public:
  std::string get_charset(void) const;
  void set_charset(const int charset);
  void clear(void);
  CharsetTable(const int, QWidget* widget_p = 0);
};

class FontsizeTable: public QWidget {

  Q_OBJECT

  FontsizeMessages help_messages;
  QRadioButton* small_button_p;
  QRadioButton* medium_button_p;
  QRadioButton* large_button_p;
private slots:
  void show_fontsize_help(void);
signals:
  void show_help_sig(const std::string&, const std::string&);
public:
  std::string get_fontsize(void) const;
  void set_fontsize(const int fontsize);
  void clear(void);
  FontsizeTable(const int, QWidget* widget_p = 0);
};

class MiscTable: public QWidget {

  Q_OBJECT

  MiscMessages help_messages;
  QLineEdit* filedir_entry_p;
  QLineEdit* print_cmd_entry_p;
  QLineEdit* autocq_entry_p;
private slots:
  void show_filedir_help(void);
  void show_print_cmd_help(void);
  void show_autocq_help(void);
signals:
  void show_help_sig(const std::string&, const std::string&);
public:
  std::string get_filedir(void) const {return static_cast<const char*>(filedir_entry_p->text());}
  std::string get_print_cmd(void) const {return static_cast<const char*>(print_cmd_entry_p->text());}
  std::string get_autocq(void) const {return static_cast<const char*>(autocq_entry_p->text());}
  void set_filedir(const std::string& filedir) {filedir_entry_p->setText(filedir.c_str());}
  void set_print_cmd(const std::string& print_cmd) {print_cmd_entry_p->setText(print_cmd.c_str());}
  void set_autocq(const std::string& autocq) {autocq_entry_p->setText(autocq.c_str());}
  void clear(void);
  MiscTable(const int, QWidget* widget_p = 0);
};

class SettingsDialog: public QDialog {

  Q_OBJECT

  struct {
    std::string port, speed;
  } old_settings;

  const int standard_size;
  bool is_home_config;
  std::string rcfile;

  IdentityTable* identity_table_p;
  ComTable* com_table_p;
  PacketTable* packet_table_p;
  InterfaceTable* interface_table_p;
  CharsetTable* charset_table_p;
  FontsizeTable* fontsize_table_p;
  MiscTable* misc_table_p;

  void read_config(bool search_localfile = true);
  bool get_prog_parm(const char*, std::string&, std::string&);
  bool find_prog_parm(const char*, const string&);
  bool get_rcfile_path(bool search_localfile = true);
  void get_reset_settings(void);
  void strip(std::string&);
private slots:
  void write_config(void);
  void get_reset_settings_prompt(void);
  void show_help(const std::string& message, const std::string& caption);
public:
  SettingsDialog(const int standard_size, QWidget* widget_p = 0);
};

class InitSettingsDialog: public QDialog {

  Q_OBJECT

  QBoxLayout* top_layout_p;
private slots:
  virtual void write_config(void) = 0;
protected:
  const int standard_size;
  void attach_settings_widget(QWidget*);
  bool find_prog_parm(const char*, const string&);
  void strip(std::string&);
  std::string get_rcfile_path(void);
protected slots:
  void show_help(const std::string& message, const std::string& caption);
public:
  InitSettingsDialog(const int standard_size, QWidget* widget_p = 0);
  virtual ~InitSettingsDialog(void) {}
};

class IdentitySettingsDialog: public InitSettingsDialog {

  Q_OBJECT

  IdentityTable* identity_table_p;
private slots:
  void write_config(void);
public:
  IdentitySettingsDialog(const int standard_size, QWidget* widget_p = 0);
};

class ComSettingsDialog: public InitSettingsDialog {

  Q_OBJECT

  ComTable*  com_table_p;
private slots:
  void write_config(void);
public:
  ComSettingsDialog(const int standard_size, const int speed, const std::string& port, QWidget* widget_p = 0);
};

class FiledirTable: public QWidget {

  Q_OBJECT

  MiscMessages help_messages;
  QLineEdit* filedir_entry_p;
private slots:
  void show_filedir_help(void);
signals:
  void show_help_sig(const std::string&, const std::string&);
public:
  std::string get_filedir(void) const {return static_cast<const char*>(filedir_entry_p->text());}
  void set_filedir(const std::string& filedir) {filedir_entry_p->setText(filedir.c_str());}
  void clear(void) {filedir_entry_p->setText("");}
  FiledirTable(const int, QWidget* widget_p = 0);
};

class FiledirSettingsDialog: public InitSettingsDialog {

  Q_OBJECT

  FiledirTable* filedir_table_p;
private slots:
  void write_config(void);
public:
  FiledirSettingsDialog(const int standard_size, QWidget* widget_p = 0);
};

#endif
