/* Copyright (C) 2001, 2002 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <unistd.h>

#include <iostream>
#include <fstream>
#include <strstream>
#include <vector>
#include <algorithm>
#include <iterator>
#include <cstdlib>
#include <cctype>

#include <qtabwidget.h>
#include <qpushbutton.h>
#include <qtoolbutton.h>
#include <qlabel.h>
#include <qpixmap.h>
#include <qbuttongroup.h>
#include <qvbuttongroup.h>
#include <qmessagebox.h>

#include "settings.h"
#include "settings_icons.h"
#include "dialogs.h"

IdentityTable::IdentityTable(const int standard_size, QWidget* widget_p): QWidget(widget_p) {

  QGridLayout* grid_p = new QGridLayout(this, 5, 3, standard_size/2);
  if (!grid_p) {
    cerr << "Memory allocation error in IdentityTable::IdentityTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->setRowStretch(0, 1);
  grid_p->setRowStretch(4, 1);
  grid_p->setColStretch(1, 1);
  
  QLabel* label_p = new QLabel("Callsign: ", this);
  if (!label_p) {
    cerr << "Memory allocation error in IdentityTable::IdentityTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(label_p, 1, 0, Qt::AlignRight | Qt::AlignVCenter);

  mycall_entry_p = new QLineEdit(this);
  if (!mycall_entry_p) {
    cerr << "Memory allocation error in IdentityTable::IdentityTable()" << endl;
    exit(MEM_ERROR);
  }
  mycall_entry_p->setMinimumSize(standard_size * 6, (standard_size * 5)/6);
  grid_p->addWidget(mycall_entry_p, 1, 1);

  QPixmap helpIcon(help_xpm);
  QToolButton* button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in IdentityTable::IdentityTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(IdentityMessages::mycall).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_mycall_help()));
  grid_p->addWidget(button_p, 1, 2);

  label_p = new QLabel("Packet callsign: ", this);
  if (!label_p) {
    cerr << "Memory allocation error in IdentityTable::IdentityTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(label_p, 2, 0, Qt::AlignRight | Qt::AlignVCenter);

  mypacketcall_entry_p = new QLineEdit(this);
  if (!mypacketcall_entry_p) {
    cerr << "Memory allocation error in IdentityTable::IdentityTable()" << endl;
    exit(MEM_ERROR);
  }
  mypacketcall_entry_p->setMinimumSize(standard_size * 6, (standard_size * 5)/6);
  grid_p->addWidget(mypacketcall_entry_p, 2, 1);

  button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in IdentityTable::IdentityTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(IdentityMessages::mypacketcall).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_mypacketcall_help()));
  grid_p->addWidget(button_p, 2, 2);

  label_p = new QLabel("Selcall: ", this);
  if (!label_p) {
    cerr << "Memory allocation error in IdentityTable::IdentityTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(label_p, 3, 0, Qt::AlignRight | Qt::AlignVCenter);

  myselcall_entry_p = new QLineEdit(this);
  if (!myselcall_entry_p) {
    cerr << "Memory allocation error in IdentityTable::IdentityTable()" << endl;
    exit(MEM_ERROR);
  }
  myselcall_entry_p->setMinimumSize(standard_size * 6, (standard_size * 5)/6);
  grid_p->addWidget(myselcall_entry_p, 3, 1);

  button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in IdentityTable::IdentityTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(IdentityMessages::myselcall).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_myselcall_help()));
  grid_p->addWidget(button_p, 3, 2);

  grid_p->activate();
}

void IdentityTable::show_mycall_help(void) {
  emit show_help_sig(help_messages.get_message(IdentityMessages::mycall),
		     help_messages.get_caption(IdentityMessages::mycall));
}

void IdentityTable::show_mypacketcall_help(void) {
  emit show_help_sig(help_messages.get_message(IdentityMessages::mypacketcall),
		     help_messages.get_caption(IdentityMessages::mypacketcall));
}

void IdentityTable::show_myselcall_help(void) {
  emit show_help_sig(help_messages.get_message(IdentityMessages::myselcall),
		     help_messages.get_caption(IdentityMessages::myselcall));
}

void IdentityTable::clear(void) {
  mycall_entry_p->setText("");
  mypacketcall_entry_p->setText("");
  myselcall_entry_p->setText("");
}

ComTable::ComTable(const int standard_size, QWidget* widget_p): QWidget(widget_p) {

  QGridLayout* grid_p = new QGridLayout(this, 6, 3, standard_size/2);
  if (!grid_p) {
    cerr << "Memory allocation error in ComTable::ComTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->setRowStretch(0, 1);
  grid_p->setRowStretch(5, 1);
  grid_p->setColStretch(1, 1);
  
  QLabel* label_p = new QLabel("Port device: ", this);
  if (!label_p) {
    cerr << "Memory allocation error in ComTable::ComTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(label_p, 1, 0, Qt::AlignRight | Qt::AlignVCenter);

  port_entry_p = new QLineEdit(this);
  if (!port_entry_p) {
    cerr << "Memory allocation error in ComTable::ComTable()" << endl;
    exit(MEM_ERROR);
  }
  port_entry_p->setMinimumSize(standard_size * 6, (standard_size * 5)/6);
  grid_p->addWidget(port_entry_p, 1, 1);

  QPixmap helpIcon(help_xpm);
  QToolButton* button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in ComTable::ComTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(ComMessages::port).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_port_help()));
  grid_p->addWidget(button_p, 1, 2);

  label_p = new QLabel("(Don't include '/dev/' part of device name)", this);
  if (!label_p) {
    cerr << "Memory allocation error in ComTable::ComTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(label_p, 2, 1, Qt::AlignLeft | Qt::AlignTop);

  grid_p->addRowSpacing(3, standard_size/2);

  QButtonGroup* button_group_p = new QButtonGroup(2, Qt::Horizontal, "Port speed", this);
  if (!button_group_p) {
    cerr << "Memory allocation error in ComTable::ComTable()" << endl;
    exit(MEM_ERROR);
  }

  b1200_button_p = new QRadioButton("1200 baud", button_group_p);
  if (!b1200_button_p) {
    cerr << "Memory allocation error in ComTable::ComTable()" << endl;
    exit(MEM_ERROR);
  }

  b2400_button_p = new QRadioButton("2400 baud", button_group_p);
  if (!b2400_button_p) {
    cerr << "Memory allocation error in ComTable::ComTable()" << endl;
    exit(MEM_ERROR);
  }

  b4800_button_p = new QRadioButton("4800 baud", button_group_p);
  if (!b4800_button_p) {
    cerr << "Memory allocation error in ComTable::ComTable()" << endl;
    exit(MEM_ERROR);
  }

  b9600_button_p = new QRadioButton("9600 baud", button_group_p);
  if (!b9600_button_p) {
    cerr << "Memory allocation error in ComTable::ComTable()" << endl;
    exit(MEM_ERROR);
  }
  b9600_button_p->setChecked(true);
  grid_p->addMultiCellWidget(button_group_p, 4, 4, 0, 1, Qt::AlignCenter);

  button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in ComTable::ComTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(ComMessages::speed).c_str(), true);
  button_p->setMinimumHeight((standard_size * 5)/6);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_speed_help()));
  grid_p->addWidget(button_p, 4, 2, Qt::AlignCenter);

  grid_p->activate();
}

void ComTable::show_port_help(void) {
  emit show_help_sig(help_messages.get_message(ComMessages::port),
		     help_messages.get_caption(ComMessages::port));
}

void ComTable::show_speed_help(void) {
  emit show_help_sig(help_messages.get_message(ComMessages::speed),
		     help_messages.get_caption(ComMessages::speed));
}

string ComTable::get_speed(void) const {
  string return_val;
  if (b1200_button_p->isChecked()) return_val = "1200";
  else if (b2400_button_p->isChecked()) return_val = "2400";
  else if (b4800_button_p->isChecked()) return_val = "4800";
  else if (b9600_button_p->isChecked()) return_val = "9600";
  return return_val;
}

void ComTable::set_speed(const int speed) {
  switch (speed) {
  case 1200:
    b1200_button_p->setChecked(true);
    break;
  case 2400:
    b2400_button_p->setChecked(true);
    break;
  case 4800:
    b4800_button_p->setChecked(true);
    break;
  case 9600:
    b9600_button_p->setChecked(true);
    break;
  }
}

void ComTable::clear(void) {
  port_entry_p->setText("");
  b9600_button_p->setChecked(true);
}

PacketTable::PacketTable(const int standard_size, QWidget* widget_p): QWidget(widget_p) {

  QGridLayout* grid_p = new QGridLayout(this, 6, 3, standard_size/2);
  if (!grid_p) {
    cerr << "Memory allocation error in PacketTable::PacketTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->setRowStretch(0, 1);
  grid_p->setRowStretch(5, 1);
  grid_p->setColStretch(1, 1);
  
  QLabel* label_p = new QLabel("Vhf packet size: ", this);
  if (!label_p) {
    cerr << "Memory allocation error in PacketTable::PacketTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(label_p, 1, 0, Qt::AlignRight | Qt::AlignVCenter);

  vhf_packet_size_entry_p = new QLineEdit(this);
  if (!vhf_packet_size_entry_p) {
    cerr << "Memory allocation error in PacketTable::PacketTable()" << endl;
    exit(MEM_ERROR);
  }
  vhf_packet_size_entry_p->setMinimumSize(standard_size * 6, (standard_size * 5)/6);
  grid_p->addWidget(vhf_packet_size_entry_p, 1, 1);

  QPixmap helpIcon(help_xpm);
  QToolButton* button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in PacketTable::PacketTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(PacketMessages::vhf_packet_size).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_vhf_packet_size_help()));
  grid_p->addWidget(button_p, 1, 2);

  label_p = new QLabel("Hf packet size: ", this);
  if (!label_p) {
    cerr << "Memory allocation error in PacketTable::PacketTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(label_p, 2, 0, Qt::AlignRight | Qt::AlignVCenter);

  hf_packet_size_entry_p = new QLineEdit(this);
  if (!hf_packet_size_entry_p) {
    cerr << "Memory allocation error in PacketTable::PacketTable()" << endl;
    exit(MEM_ERROR);
  }
  hf_packet_size_entry_p->setMinimumSize(standard_size * 6, (standard_size * 5)/6);
  grid_p->addWidget(hf_packet_size_entry_p, 2, 1);

  button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in PacketTable::PacketTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(PacketMessages::hf_packet_size).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_hf_packet_size_help()));
  grid_p->addWidget(button_p, 2, 2);

  label_p = new QLabel("Keep-alive interval: ", this);
  if (!label_p) {
    cerr << "Memory allocation error in PacketTable::PacketTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(label_p, 3, 0, Qt::AlignRight | Qt::AlignVCenter);

  keepalive_mins_entry_p = new QLineEdit(this);
  if (!keepalive_mins_entry_p) {
    cerr << "Memory allocation error in PacketTable::PacketTable()" << endl;
    exit(MEM_ERROR);
  }
  keepalive_mins_entry_p->setMinimumSize(standard_size * 6, (standard_size * 5)/6);
  grid_p->addWidget(keepalive_mins_entry_p, 3, 1);

  button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in PacketTable::PacketTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(PacketMessages::keepalive_mins).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_keepalive_mins_help()));
  grid_p->addWidget(button_p, 3, 2);

  check_box_p = new QCheckBox("Send keep-alive packets", this);
  if (!check_box_p) {
    cerr << "Memory allocation error in PacketTable::PacketTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(check_box_p, 4, 1, Qt::AlignLeft | Qt::AlignVCenter);

  grid_p->activate();
}

void PacketTable::show_vhf_packet_size_help(void) {
  emit show_help_sig(help_messages.get_message(PacketMessages::vhf_packet_size),
		     help_messages.get_caption(PacketMessages::vhf_packet_size));
}

void PacketTable::show_hf_packet_size_help(void) {
  emit show_help_sig(help_messages.get_message(PacketMessages::hf_packet_size),
		     help_messages.get_caption(PacketMessages::hf_packet_size));
}

void PacketTable::show_keepalive_mins_help(void) {
  emit show_help_sig(help_messages.get_message(PacketMessages::keepalive_mins),
		     help_messages.get_caption(PacketMessages::keepalive_mins));
}

void PacketTable::clear(void) {
  vhf_packet_size_entry_p->setText("");
  hf_packet_size_entry_p->setText("");
  keepalive_mins_entry_p->setText("");
  check_box_p->setChecked(false);
}

InterfaceTable::InterfaceTable(const int standard_size, QWidget* widget_p): QWidget(widget_p) {

  QGridLayout* grid_p = new QGridLayout(this, 6, 2, standard_size/2);
  if (!grid_p) {
    cerr << "Memory allocation error in InterfaceTable::InterfaceTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->setRowStretch(0, 1);
  grid_p->setRowStretch(5, 1);
  grid_p->setColStretch(0, 1);
  
  receive_cw_box_p = new QCheckBox("Display received CW", this);
  if (!receive_cw_box_p) {
    cerr << "Memory allocation error in InterfaceTable::InterfaceTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(receive_cw_box_p, 1, 0, Qt::AlignLeft | Qt::AlignVCenter);

  QPixmap helpIcon(help_xpm);
  QToolButton* button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in PacketTable::PacketTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(InterfaceMessages::receive_cw).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_receive_cw_help()));
  grid_p->addWidget(button_p, 1, 1);

  sound_bell_box_p = new QCheckBox("Sound bell character", this);
  if (!sound_bell_box_p) {
    cerr << "Memory allocation error in InterfaceTable::InterfaceTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(sound_bell_box_p, 2, 0, Qt::AlignLeft | Qt::AlignVCenter);

  button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in InterfaceTable::InterfaceTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(InterfaceMessages::sound_bell).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_sound_bell_help()));
  grid_p->addWidget(button_p, 2, 1);

  no_prompt_box_p = new QCheckBox("No prompt when sending messages", this);
  if (!no_prompt_box_p) {
    cerr << "Memory allocation error in InterfaceTable::InterfaceTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(no_prompt_box_p, 3, 0, Qt::AlignLeft | Qt::AlignVCenter);

  button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in InterfaceTable::InterfaceTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(InterfaceMessages::no_prompt).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_no_prompt_help()));
  grid_p->addWidget(button_p, 3, 1);

  receive_cw_box_p->setChecked(true);
  sound_bell_box_p->setChecked(true);

  grid_p->activate();
}

void InterfaceTable::show_receive_cw_help(void) {
  emit show_help_sig(help_messages.get_message(InterfaceMessages::receive_cw),
		     help_messages.get_caption(InterfaceMessages::receive_cw));
}

void InterfaceTable::show_sound_bell_help(void) {
  emit show_help_sig(help_messages.get_message(InterfaceMessages::sound_bell),
		     help_messages.get_caption(InterfaceMessages::sound_bell));
}

void InterfaceTable::show_no_prompt_help(void) {
  emit show_help_sig(help_messages.get_message(InterfaceMessages::no_prompt),
		     help_messages.get_caption(InterfaceMessages::no_prompt));
}

void InterfaceTable::clear(void) {
  receive_cw_box_p->setChecked(true);
  sound_bell_box_p->setChecked(true);
  no_prompt_box_p->setChecked(false);
}

CharsetTable::CharsetTable(const int standard_size, QWidget* widget_p): QWidget(widget_p) {

  QGridLayout* grid_p = new QGridLayout(this, 3, 2, standard_size/2);
  if (!grid_p) {
    cerr << "Memory allocation error in CharsetTable::CharsetTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->setRowStretch(0, 1);
  grid_p->setRowStretch(2, 1);
  grid_p->setColStretch(0, 1);
  
  QButtonGroup* button_group_p = new QVButtonGroup("Tx/Rx character set", this);
  if (!button_group_p) {
    cerr << "Memory allocation error in CharsetTable::CharsetTable()" << endl;
    exit(MEM_ERROR);
  }

  cp437_button_p = new QRadioButton("CP437", button_group_p);
  if (!cp437_button_p) {
    cerr << "Memory allocation error in CharsetTable::CharsetTable()" << endl;
    exit(MEM_ERROR);
  }

  latin_1_button_p = new QRadioButton("Latin 1", button_group_p);
  if (!latin_1_button_p) {
    cerr << "Memory allocation error in CharsetTable::CharsetTable()" << endl;
    exit(MEM_ERROR);
  }
  cp437_button_p->setChecked(true);
  grid_p->addWidget(button_group_p, 1, 0, Qt::AlignCenter);

  QPixmap helpIcon(help_xpm);
  QToolButton* button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in CharsetTable::CharsetTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(CharsetMessages::message).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_charset_help()));
  grid_p->addWidget(button_p, 1, 1, Qt::AlignCenter);

  grid_p->activate();
}

string CharsetTable::get_charset(void) const {
  string return_val;
  if (cp437_button_p->isChecked()) return_val = "CP437";
  else if (latin_1_button_p->isChecked()) return_val = "LATIN-1";
  return return_val;
}

void CharsetTable::set_charset(const int charset) {

  switch (charset) {
  case Prog_func::cp437:
    cp437_button_p->setChecked(true);
    break;
  case Prog_func::latin_1:
    latin_1_button_p->setChecked(true);
    break;
  }
}
void CharsetTable::show_charset_help(void) {
  emit show_help_sig(help_messages.get_message(CharsetMessages::message),
		     help_messages.get_caption(CharsetMessages::message));
}

void CharsetTable::clear(void) {
  cp437_button_p->setChecked(true);
}

FontsizeTable::FontsizeTable(const int standard_size, QWidget* widget_p): QWidget(widget_p) {

  QGridLayout* grid_p = new QGridLayout(this, 3, 2, standard_size/2);
  if (!grid_p) {
    cerr << "Memory allocation error in FontsizeTable::FontsizeTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->setRowStretch(0, 1);
  grid_p->setRowStretch(2, 1);
  grid_p->setColStretch(0, 1);
  
  QButtonGroup* button_group_p = new QVButtonGroup("Font size", this);
  if (!button_group_p) {
    cerr << "Memory allocation error in FontsizeTable::FontsizeTable()" << endl;
    exit(MEM_ERROR);
  }

  small_button_p = new QRadioButton("Small", button_group_p);
  if (!small_button_p) {
    cerr << "Memory allocation error in FontsizeTable::FontsizeTable()" << endl;
    exit(MEM_ERROR);
  }

  medium_button_p = new QRadioButton("Medium", button_group_p);
  if (!medium_button_p) {
    cerr << "Memory allocation error in FontsizeTable::FontsizeTable()" << endl;
    exit(MEM_ERROR);
  }

  large_button_p = new QRadioButton("Large", button_group_p);
  if (!large_button_p) {
    cerr << "Memory allocation error in FontsizeTable::FontsizeTable()" << endl;
    exit(MEM_ERROR);
  }
  medium_button_p->setChecked(true);
  grid_p->addWidget(button_group_p, 1, 0, Qt::AlignCenter);

  QPixmap helpIcon(help_xpm);
  QToolButton* button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in FontsizeTable::FontsizeTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(FontsizeMessages::message).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_fontsize_help()));
  grid_p->addWidget(button_p, 1, 1, Qt::AlignCenter);

  grid_p->activate();
}

string FontsizeTable::get_fontsize(void) const {
  string return_val;
  if (small_button_p->isChecked()) return_val = "small";
  else if (medium_button_p->isChecked()) return_val = "medium";
  else if (large_button_p->isChecked()) return_val = "large";
  return return_val;
}

void FontsizeTable::set_fontsize(const int fontsize) {

  switch (fontsize) {
  case 10:
    small_button_p->setChecked(true);
    break;
  case 14:
    large_button_p->setChecked(true);
    break;
  default:
    medium_button_p->setChecked(true);
  }
}

void FontsizeTable::show_fontsize_help(void) {
  emit show_help_sig(help_messages.get_message(FontsizeMessages::message),
		     help_messages.get_caption(FontsizeMessages::message));
}

void FontsizeTable::clear(void) {
  medium_button_p->setChecked(true);
}

MiscTable::MiscTable(const int standard_size, QWidget* widget_p): QWidget(widget_p) {

  QGridLayout* grid_p = new QGridLayout(this, 5, 3, standard_size/2);
  if (!grid_p) {
    cerr << "Memory allocation error in MiscTable::MiscTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->setRowStretch(0, 1);
  grid_p->setRowStretch(4, 1);
  grid_p->setColStretch(1, 1);
  
  QLabel* label_p = new QLabel("File directory: ", this);
  if (!label_p) {
    cerr << "Memory allocation error in MiscTable::MiscTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(label_p, 1, 0, Qt::AlignRight | Qt::AlignVCenter);

  filedir_entry_p = new QLineEdit(this);
  if (!filedir_entry_p) {
    cerr << "Memory allocation error in MiscTable::MiscTable()" << endl;
    exit(MEM_ERROR);
  }
  filedir_entry_p->setMinimumSize(standard_size * 6, (standard_size * 5)/6);
  grid_p->addWidget(filedir_entry_p, 1, 1);

  QPixmap helpIcon(help_xpm);
  QToolButton* button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in MiscTable::MiscTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(MiscMessages::filedir).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_filedir_help()));
  grid_p->addWidget(button_p, 1, 2);

  label_p = new QLabel("Print command: ", this);
  if (!label_p) {
    cerr << "Memory allocation error in MiscTable::MiscTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(label_p, 2, 0, Qt::AlignRight | Qt::AlignVCenter);

  print_cmd_entry_p = new QLineEdit(this);
  if (!print_cmd_entry_p) {
    cerr << "Memory allocation error in MiscTable::MiscTable()" << endl;
    exit(MEM_ERROR);
  }
  print_cmd_entry_p->setMinimumSize(standard_size * 6, (standard_size * 5)/6);
  grid_p->addWidget(print_cmd_entry_p, 2, 1);

  button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in MiscTable::MiscTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(MiscMessages::print_cmd).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_print_cmd_help()));
  grid_p->addWidget(button_p, 2, 2);

  label_p = new QLabel("Autocq interval: ", this);
  if (!label_p) {
    cerr << "Memory allocation error in MiscTable::MiscTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(label_p, 3, 0, Qt::AlignRight | Qt::AlignVCenter);

  autocq_entry_p = new QLineEdit(this);
  if (!autocq_entry_p) {
    cerr << "Memory allocation error in MiscTable::MiscTable()" << endl;
    exit(MEM_ERROR);
  }
  autocq_entry_p->setMinimumSize(standard_size * 6, (standard_size * 5)/6);
  grid_p->addWidget(autocq_entry_p, 3, 1);

  button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in MiscTable::MiscTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(MiscMessages::autocq).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_autocq_help()));
  grid_p->addWidget(button_p, 3, 2);

  grid_p->activate();
}

void MiscTable::show_filedir_help(void) {
  emit show_help_sig(help_messages.get_message(MiscMessages::filedir),
		     help_messages.get_caption(MiscMessages::filedir));
}

void MiscTable::show_print_cmd_help(void) {
  emit show_help_sig(help_messages.get_message(MiscMessages::print_cmd),
		     help_messages.get_caption(MiscMessages::print_cmd));
}

void MiscTable::show_autocq_help(void) {
  emit show_help_sig(help_messages.get_message(MiscMessages::autocq),
		     help_messages.get_caption(MiscMessages::autocq));
}

void MiscTable::clear(void) {
  filedir_entry_p->setText("");
  print_cmd_entry_p->setText("");
  autocq_entry_p->setText("");
}

SettingsDialog::SettingsDialog(const int size, QWidget* widget_p): QDialog(widget_p, "Settings", true),
								   standard_size(size), is_home_config(false) {

  QBoxLayout* top_layout_p = new QVBoxLayout(this, standard_size/2);
  if (!top_layout_p) {
    cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
    
  QTabWidget* tab_widget_p = new QTabWidget(this, "Tab Widget");
  if (!tab_widget_p) {
    cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
  top_layout_p->addWidget(tab_widget_p, 1);

  // now make up the notebook pages

  identity_table_p = new IdentityTable(standard_size, tab_widget_p);
  if (!identity_table_p) {
    cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
  tab_widget_p->addTab(identity_table_p, "Identity");

  com_table_p = new ComTable(standard_size, tab_widget_p);
  if (!com_table_p) {
    cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
  tab_widget_p->addTab(com_table_p, "Com");

  packet_table_p = new PacketTable(standard_size, tab_widget_p);
  if (!packet_table_p) {
    cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
  tab_widget_p->addTab(packet_table_p, "Packet");

  interface_table_p = new InterfaceTable(standard_size, tab_widget_p);
  if (!interface_table_p) {
    cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
  tab_widget_p->addTab(interface_table_p, "Interface");

  charset_table_p = new CharsetTable(standard_size, tab_widget_p);
  if (!charset_table_p) {
    cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
  tab_widget_p->addTab(charset_table_p, "Charset");

  fontsize_table_p = new FontsizeTable(standard_size, tab_widget_p);
  if (!fontsize_table_p) {
    cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
  tab_widget_p->addTab(fontsize_table_p, "Font");

  misc_table_p = new MiscTable(standard_size, tab_widget_p);
  if (!misc_table_p) {
    cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
  tab_widget_p->addTab(misc_table_p, "Misc");

  QObject::connect(identity_table_p, SIGNAL(show_help_sig(const std::string&, const std::string&)),
		   this, SLOT(show_help(const std::string&, const std::string&)));
  QObject::connect(com_table_p, SIGNAL(show_help_sig(const std::string&, const std::string&)),
		   this, SLOT(show_help(const std::string&, const std::string&)));
  QObject::connect(packet_table_p, SIGNAL(show_help_sig(const std::string&, const std::string&)),
		   this, SLOT(show_help(const std::string&, const std::string&)));
  QObject::connect(interface_table_p, SIGNAL(show_help_sig(const std::string&, const std::string&)),
		   this, SLOT(show_help(const std::string&, const std::string&)));
  QObject::connect(charset_table_p, SIGNAL(show_help_sig(const std::string&, const std::string&)),
		   this, SLOT(show_help(const std::string&, const std::string&)));
  QObject::connect(fontsize_table_p, SIGNAL(show_help_sig(const std::string&, const std::string&)),
		   this, SLOT(show_help(const std::string&, const std::string&)));
  QObject::connect(misc_table_p, SIGNAL(show_help_sig(const std::string&, const std::string&)),
		   this, SLOT(show_help(const std::string&, const std::string&)));

  // we must call read_config() before setting up the dialog buttons, because the reset button/warning label
  // depends on the state of is_home_config, which is set by get_rcfile_path(), which is in turn called
  // by read_config()
  read_config();

  QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, standard_size/4);
  if (!button_layout_p) {
    cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
  button_layout_p->addStretch(1);
  button_layout_p->addSpacing(standard_size/2);

  QPushButton* ok_button_p = new QPushButton("OK", this);
  if (!ok_button_p) {
    cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
  
  QPushButton* cancel_button_p = new QPushButton("Cancel", this);
  if (!cancel_button_p) {
    cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
    exit(MEM_ERROR);
  }

  QSize button_size = cancel_button_p->sizeHint();
  ok_button_p->setFixedSize(button_size);
  cancel_button_p->setFixedSize(button_size);

  if (!is_home_config) {

    QLabel* message_label_p = new QLabel("Note: pressing the OK button will\n"
					 "save the settings in file ~/." RC_FILE, this);
    if (!message_label_p) {
      cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
      exit(MEM_ERROR);
    }
    message_label_p->setAlignment(AlignCenter);
    button_layout_p->addWidget(message_label_p);
  }
  else {
    QPushButton* reset_button_p = new QPushButton("Reset", this);
    if (!reset_button_p) {
      cerr << "Memory allocation error in SettingsDialog::SettingsDialog()" << endl;
      exit(MEM_ERROR);
    }
    QObject::connect(reset_button_p, SIGNAL(clicked()), this, SLOT(get_reset_settings_prompt()));
    reset_button_p->setFixedSize(button_size);
    button_layout_p->addWidget(reset_button_p);
  }

  button_layout_p->addStretch(2);
  button_layout_p->addWidget(ok_button_p);
  button_layout_p->addStretch(2);
  button_layout_p->addWidget(cancel_button_p);
  button_layout_p->addStretch(1);
  button_layout_p->addSpacing(standard_size/2);

  setCaption("kamplus: settings");
    
  QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(write_config()));
  QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));

  tab_widget_p->setMinimumSize(standard_size * 14, standard_size * 9);

  top_layout_p->activate();
}

void SettingsDialog::write_config(void) {

  bool good_write = true;
  ifstream filein;

  if (!rcfile.empty()) {
    filein.open(rcfile.c_str(), ios::in);
    if (!filein) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      QMessageBox::critical(this, "Config Error", message.c_str(),
			    QMessageBox::Ok | QMessageBox::Default, 0);
    }
  }

  vector<string> file_list;
  string file_read;
  
  string temp;

  string mycall_line("MY_CALL: ");
  temp = identity_table_p->get_mycall();
  strip(temp);
  if (temp.empty()) {
    good_write = false;
    QMessageBox::critical(this, "Config Error", "No home callsign specified",
			  QMessageBox::Ok | QMessageBox::Default, 0);
  }
  else {
    transform(temp.begin(), temp.end(),
	      temp.begin(), std::toupper);
    mycall_line += temp;
  }
  
  string mypacketcall_line;
  temp =  identity_table_p->get_mypacketcall();
  strip(temp);
  if (temp.empty()) mypacketcall_line = "#PKT_CALL: ";
  else {
    transform(temp.begin(), temp.end(),
	      temp.begin(), std::toupper);
    mypacketcall_line = "PKT_CALL: ";
    mypacketcall_line += temp;
  }
  
  string myselcall_line;
  temp =  identity_table_p->get_myselcall();
  strip(temp);
  if (temp.empty()) myselcall_line = "#SELCALL: ";
  else {
    transform(temp.begin(), temp.end(),
	      temp.begin(), std::toupper);
    myselcall_line = "SELCALL: ";
    myselcall_line += temp;
  }
   
  string port_line("SERIALPORT: ");
  temp = com_table_p->get_port();
  strip(temp);
  if (temp.empty()) {
    good_write = false;
    QMessageBox::critical(this, "Config Error", "No serial port specified",
			  QMessageBox::Ok | QMessageBox::Default, 0);
  }
  else {
    port_line += temp;
    if (temp.compare(old_settings.port)) {
      QMessageBox::warning(this, "Config", "Note: the change of port setting will not\n"
			                   "take effect until the program is restarted",
			   QMessageBox::Ok | QMessageBox::Default, 0);
    }
  }

  string speed_line("SPEED: ");
  speed_line += com_table_p->get_speed();
  if (com_table_p->get_speed().compare(old_settings.speed)) {
    QMessageBox::warning(this, "Config", "Note: the change of speed setting will not\n"
			 "take effect until the program is restarted",
			 QMessageBox::Ok | QMessageBox::Default, 0);
  }

  string vpaclen_line;
  temp =  packet_table_p->get_vhf_packet_size();
  strip(temp);
  if (temp.empty()) vpaclen_line = "#V_PACLEN: ";
  else if (atoi(temp.c_str()) < 32 || atoi(temp.c_str()) > MAX_FRAMESIZE) {
    ostrstream paclen_strm;
    ostrstream message_strm;
    paclen_strm << "V_PACLEN: " << MAX_FRAMESIZE << ends;
    message_strm << "Invalid vhf packet length specified.\n"
                    "Defaulting to "  << MAX_FRAMESIZE << ends;

    char* text = paclen_strm.str();
    vpaclen_line = text;
    delete[] text;

    text = message_strm.str();
    QMessageBox::warning(this, "Config", text,
			 QMessageBox::Ok | QMessageBox::Default, 0);
    delete[] text;
  }
  else {
    vpaclen_line = "V_PACLEN: ";
    vpaclen_line += temp;
  }
  
  string hpaclen_line;
  temp =  packet_table_p->get_hf_packet_size();
  strip(temp);
  if (temp.empty()) hpaclen_line = "#H_PACLEN: ";
  else if (atoi(temp.c_str()) < 32 || atoi(temp.c_str()) > MAX_FRAMESIZE) {
    ostrstream paclen_strm;
    ostrstream message_strm;
    paclen_strm << "H_PACLEN: " << MAX_FRAMESIZE << ends;
    message_strm << "Invalid hf packet length specified.\n"
                    "Defaulting to "  << MAX_FRAMESIZE << ends;

    char* text = paclen_strm.str();
    hpaclen_line = text;
    delete[] text;

    text = message_strm.str();
    QMessageBox::warning(this, "Config", text,
			 QMessageBox::Ok | QMessageBox::Default, 0);
    delete[] text;
  }
  else {
    hpaclen_line = "H_PACLEN: ";
    hpaclen_line += temp;
  }

  string keepalive_line("KEEPALIVE: ");
  if (packet_table_p->get_keepalive()) keepalive_line += "true";
  else keepalive_line += "false";

  string keepalive_mins_line;
  temp =  packet_table_p->get_keepalive_mins();
  strip(temp);
  if (temp.empty()) keepalive_mins_line = "#KEEPALIVE_MINS: ";
  else if (atoi(temp.c_str()) < 1) {
    keepalive_mins_line = "KEEPALIVE_MINS: 10";
    QMessageBox::warning(this, "Config",
			 "Invalid keep-alive interval specified\n"
			 "Defaulting to 10 minutes",
			 QMessageBox::Ok | QMessageBox::Default, 0);
  }
  else {
    keepalive_mins_line = "KEEPALIVE_MINS: ";
    keepalive_mins_line += temp;
  }

  string receive_cw_line("NO_CW_RX: ");
  if (interface_table_p->get_receive_cw()) receive_cw_line += "false";
  else receive_cw_line += "true";

  string sound_bell_line("BELL: ");
  if (interface_table_p->get_sound_bell()) sound_bell_line += "true";
  else sound_bell_line += "false";

  string no_prompt_line("NOPROMPT: ");
  if (interface_table_p->get_no_prompt()) no_prompt_line += "true";
  else no_prompt_line += "false";

  string charset_line;
  temp =  charset_table_p->get_charset();
  strip(temp);
  if (temp.empty()) charset_line = "#CHAR_SET: ";
  else {
    charset_line = "CHAR_SET: ";
    charset_line += temp;
  }
  
  string fontsize_line;
  temp =  fontsize_table_p->get_fontsize();
  strip(temp);
  if (temp.empty()) fontsize_line = "#WIN_FONTSIZE: ";
  else {
    fontsize_line = "WIN_FONTSIZE: ";
    fontsize_line += temp;
  }
  
  string filedir_line("FILEDIR: ");
  temp = misc_table_p->get_filedir();
  strip(temp);
  if (temp.empty()) {
      good_write = false;
      QMessageBox::critical(this, "Config Error", "No file directory specified",
			    QMessageBox::Ok | QMessageBox::Default, 0);
  }
  else filedir_line += temp;

  string print_cmd_line;
  temp = misc_table_p->get_print_cmd();
  strip(temp);
  if (temp.empty()) print_cmd_line = "#PRINT_CMD: ";
  else {
    print_cmd_line = "PRINT_CMD: ";
    print_cmd_line += temp;
  }

  string autocq_line;
  temp = misc_table_p->get_autocq();
  strip(temp);
  if (temp.empty()) autocq_line = "#AUTOCQ: ";
  else if (atoi(temp.c_str()) < 1) {
    autocq_line = "AUTOCQ: 90";
    QMessageBox::warning(this, "Config",
			 "Invalid autocq interval specified\n"
			 "Defaulting to 90 seconds",
			 QMessageBox::Ok | QMessageBox::Default, 0);
  }
  else {
    autocq_line = "AUTOCQ: ";
    autocq_line += temp;
  }
  
  if (good_write) { // no errors -- write out the configuration file
    
    bool found_mycall = false;
    bool found_mypacketcall = false;
    bool found_myselcall = false;
    bool found_port = false;
    bool found_speed = false;
    bool found_vpaclen = false;
    bool found_hpaclen = false;
    bool found_keepalive = false;
    bool found_keepalive_mins = false;
    bool found_receive_cw = false;
    bool found_sound_bell = false;
    bool found_no_prompt = false;
    bool found_charset = false;
    bool found_fontsize = false;
    bool found_filedir = false;
    bool found_print_cmd = false;
    bool found_autocq = false;
    
    const char terminating_line[] = "## end of " RC_FILE " ##";
    
    while (getline(filein, file_read)) {
      
      // look for "MY_CALL:"
      if (find_prog_parm("MY_CALL:", file_read) || find_prog_parm("#MY_CALL:", file_read)) {
	if (!found_mycall) {
	  found_mycall = true;
	  file_list.push_back(mycall_line);
	}
      }
      
      // look for "PKT_CALL:"
      else if (find_prog_parm("PKT_CALL:", file_read) || find_prog_parm("#PKT_CALL:", file_read)) {
	if (!found_mypacketcall) {
	  found_mypacketcall = true;
	  file_list.push_back(mypacketcall_line);
	}
      }
      
      // look for "SELCALL:"
      else if (find_prog_parm("SELCALL:", file_read) || find_prog_parm("#SELCALL:", file_read)) {
	if (!found_myselcall) {
	  found_myselcall = true;
	  file_list.push_back(myselcall_line);
	}
      }
      
      // look for "SERIALPORT:"
      else if (find_prog_parm("SERIALPORT:", file_read)  || find_prog_parm("#SERIALPORT:", file_read)) {
	if (!found_port) {
	  found_port = true;
	  file_list.push_back(port_line);
	}
      }
      
      // look for "SPEED:"
      else if (find_prog_parm("SPEED:", file_read) || find_prog_parm("#SPEED:", file_read))  {
	if (!found_speed) {
	  found_speed = true;
	  file_list.push_back(speed_line);
	}
      }
      
      // look for "V_PACLEN:"
      else if (find_prog_parm("V_PACLEN:", file_read) || find_prog_parm("#V_PACLEN:", file_read))  {
	if (!found_vpaclen) {
	  found_vpaclen = true;
	  file_list.push_back(vpaclen_line);
	}
      }
      
      // look for "H_PACLEN:"
      else if (find_prog_parm("H_PACLEN:", file_read) || find_prog_parm("#H_PACLEN", file_read))  {
	if (!found_hpaclen) {
	  found_hpaclen = true;
	  file_list.push_back(hpaclen_line);
	}
      }
      
      // look for "KEEPALIVE:"
      else if (find_prog_parm("KEEPALIVE:", file_read) || find_prog_parm("#KEEPALIVE:", file_read)) {
	if (!found_keepalive) {
	  found_keepalive = true;
	  file_list.push_back(keepalive_line);
	}
      }
      
      // look for "KEEPALIVE_MINS:"
      else if (find_prog_parm("KEEPALIVE_MINS:", file_read) || find_prog_parm("#KEEPALIVE_MINS:", file_read)) {
	if (!found_keepalive_mins) {
	  found_keepalive_mins = true;
	  file_list.push_back(keepalive_mins_line);
	}
      }
      
      // look for "NO_CW_RX:"
      else if (find_prog_parm("NO_CW_RX:", file_read) || find_prog_parm("#NO_CW_RX:", file_read)) {
	if (!found_receive_cw) {
	  found_receive_cw = true;
	  file_list.push_back(receive_cw_line);
	}
      }
      
      // look for "BELL:"
      else if (find_prog_parm("BELL:", file_read) || find_prog_parm("#BELL:", file_read)) {
	if (!found_sound_bell) {
	  found_sound_bell = true;
	  file_list.push_back(sound_bell_line);
	}
      }
      
      // look for "NOPROMPT:"
      else if (find_prog_parm("NOPROMPT:", file_read) || find_prog_parm("#NOPROMPT:", file_read)) {
	if (!found_no_prompt) {
	  found_no_prompt = true;
	  file_list.push_back(no_prompt_line);
	}
      }
      
      // look for "CHAR_SET:"
      else if (find_prog_parm("CHAR_SET:", file_read) || find_prog_parm("#CHAR_SET:", file_read)) {
	if (!found_charset) {
	  found_charset = true;
	  file_list.push_back(charset_line);
	}
      }
      
      // look for "WIN_FONTSIZE:"
      else if (find_prog_parm("WIN_FONTSIZE:", file_read) || find_prog_parm("#WIN_FONTSIZE:", file_read)) {
	if (!found_fontsize) {
	  found_fontsize = true;
	  file_list.push_back(fontsize_line);
	}
      }
      
      // look for "FILEDIR:"
      else if (find_prog_parm("FILEDIR:", file_read) || find_prog_parm("#FILEDIR:", file_read)) {
	if (!found_filedir) {
	  found_filedir = true;
	  file_list.push_back(filedir_line);
	}
      }
      
      // look for "PRINT_CMD:"
      else if (find_prog_parm("PRINT_CMD:", file_read) || find_prog_parm("#PRINT_CMD:", file_read)) {
	if (!found_print_cmd) {
	  found_print_cmd = true;
	  file_list.push_back(print_cmd_line);
	}
      }
      
      // look for "AUTOCQ:"
      else if (find_prog_parm("AUTOCQ:", file_read) || find_prog_parm("#AUTOCQ:", file_read)) {
	if (!found_autocq) {
	  found_autocq = true;
	  file_list.push_back(autocq_line);
	}
      }
      
      // add any residual lines to the list, except the terminating line
      else if (!find_prog_parm(terminating_line, file_read)) file_list.push_back(file_read);
    }
    
    if (found_mycall == false) file_list.push_back(mycall_line);
    if (found_mypacketcall == false) file_list.push_back(mypacketcall_line);
    if (found_myselcall == false) file_list.push_back(myselcall_line);
    if (found_port == false) file_list.push_back(port_line);
    if (found_speed == false) file_list.push_back(speed_line);
    if (found_vpaclen == false) file_list.push_back(vpaclen_line);
    if (found_hpaclen == false) file_list.push_back(hpaclen_line);
    if (found_keepalive == false) file_list.push_back(keepalive_line);
    if (found_keepalive_mins = false) file_list.push_back(keepalive_mins_line);
    if (found_receive_cw == false) file_list.push_back(receive_cw_line);
    if (found_sound_bell == false) file_list.push_back(sound_bell_line);
    if (found_no_prompt == false) file_list.push_back(no_prompt_line);
    if (found_charset == false) file_list.push_back(charset_line);
    if (found_fontsize == false) file_list.push_back(fontsize_line);
    if (found_filedir == false) file_list.push_back(filedir_line);
    if (found_print_cmd == false) file_list.push_back(print_cmd_line);
    if (found_autocq == false) file_list.push_back(autocq_line);
    
    // add the terminating line
    file_list.push_back(terminating_line);

    // now write out the new config file
    
    rcfile = prog_func.homedir + "/.";
    rcfile += RC_FILE;
    
    ofstream fileout(rcfile.c_str(), ios::out);
    if (!fileout) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      QMessageBox::critical(this, "Config Error", message.c_str(),
			    QMessageBox::Ok | QMessageBox::Default, 0);
      good_write = false;
    }
    else {
      copy(file_list.begin(), file_list.end(),
	   ostream_iterator<string>(fileout, "\n"));
    }
  }
  // if we were successful, exit the settings dialog and return Qt::Accepted
  if (good_write) emit accept();
}

void SettingsDialog::read_config(bool search_localfile) {

// search_localfile has a default value of true
// get rc file
  if (!get_rcfile_path(search_localfile)) {
    string message;
    if (search_localfile) {
      message = "Can't find or open file /etc/" RC_FILE ",\n"
	        "/usr/local/etc/" RC_FILE " or ";
      message += prog_func.homedir + "/." RC_FILE "\n";
    }
    else {
      message = "Can't find or open file /etc/" RC_FILE "\n"
	        "or /usr/local/etc/" RC_FILE;
      message += prog_func.homedir + "/." RC_FILE;
    }
    cerr << message << endl;
    QMessageBox::critical(this, "Config Error", message.c_str(),
			  QMessageBox::Ok | QMessageBox::Default, 0);
  }

  else {
// now extract settings from file

    ifstream filein(rcfile.c_str(), ios::in);
    if (!filein) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      QMessageBox::critical(this, "Config Error", message.c_str(),
			    QMessageBox::Ok | QMessageBox::Default, 0);
    }

    else {

      old_settings.port = "";
      old_settings.speed = "9600";

      string file_read;
      string result;
      while (getline(filein, file_read)) {

	if (!file_read.empty() && file_read[0] != '#') { // valid line to check
	  // now check for other comment markers
	  string::size_type pos = file_read.find_first_of('#');
	  if (pos != string::npos) file_read.resize(pos); // truncate
	
	  // look for "MY_CALL:"
	  if (get_prog_parm("MY_CALL:", file_read, result)) {
	    transform(result.begin(), result.end(),
		      result.begin(), std::toupper);
	    identity_table_p->set_mycall(result);
	  }
	
	  // look for "PKT_CALL:"
	  else if (get_prog_parm("PKT_CALL:", file_read, result)) {
	    transform(result.begin(), result.end(),
		      result.begin(), std::toupper);
	    identity_table_p->set_mypacketcall(result);
	  }
	
	  // look for "SELCALL:"
	  else if (get_prog_parm("SELCALL:", file_read, result)) {
	    transform(result.begin(), result.end(),
		      result.begin(), std::toupper);
	    identity_table_p->set_myselcall(result);
	  }
	
	  // look for "SERIALPORT:"
	  else if (get_prog_parm("SERIALPORT:", file_read, result)) {
	    com_table_p->set_port(result);
	    old_settings.port = result;
	  }

	  // look for "SPEED:"
	  else if (get_prog_parm("SPEED:", file_read, result)) {
	    int speed = atoi(result.c_str());
	    if (speed == 1200 || speed == 2400 || speed == 4800 || speed == 9600) {
	      com_table_p->set_speed(speed);
	      old_settings.speed = result;
	    }
	    else {
	      cerr << "Invalid speed value specified\nDefaulting to 9600 baud" << endl;
	      com_table_p->set_speed(9600);
	    }
	  }

	  // look for "V_PACLEN:"
	  else if (get_prog_parm("V_PACLEN:", file_read, result)) {
	    packet_table_p->set_vhf_packet_size(result);
	  }

	  // look for "H_PACLEN:"
	  else if (get_prog_parm("H_PACLEN:", file_read, result)) {
	    packet_table_p->set_hf_packet_size(result);
	  }
	
	  // look for "KEEPALIVE:"
	  else if (get_prog_parm("KEEPALIVE:", file_read, result)) {
	    if (!result.compare("TRUE") || !result.compare("true")) packet_table_p->set_keepalive(true);
	    else packet_table_p->set_keepalive(false);
	  }
	
	  // look for "KEEPALIVE_MINS:"
	  else if (get_prog_parm("KEEPALIVE_MINS:", file_read, result)) {
	    packet_table_p->set_keepalive_mins(result);
	  }
	
	  // look for "NO_CW_RX:"
	  else if (get_prog_parm("NO_CW_RX:", file_read, result)) {
	    if (!result.compare("TRUE") || !result.compare("true")) interface_table_p->set_receive_cw(false);
	    else interface_table_p->set_receive_cw(true);
	  }
	
	  // look for "BELL:"
	  else if (get_prog_parm("BELL:", file_read, result)) {
	    if (!result.compare("TRUE") || !result.compare("true")) interface_table_p->set_sound_bell(true);
	    else interface_table_p->set_sound_bell(false);
	  }
	
	  // look for "NOPROMPT:"
	  else if (get_prog_parm("NOPROMPT:", file_read, result)) {
	    if (!result.compare("TRUE") || !result.compare("true")) interface_table_p->set_no_prompt(true);
	    else interface_table_p->set_no_prompt(false);
	  }
	
	  // look for "CHAR_SET:"
	  else if (get_prog_parm("CHAR_SET:", file_read, result)) {
	    if (!result.compare("latin-1") || !result.compare("LATIN-1")) {
	      charset_table_p->set_charset(Prog_func::latin_1);
	    }
	    else charset_table_p->set_charset(Prog_func::cp437);
	  }

	  // look for "WIN_FONTSIZE:"
	  else if (get_prog_parm("WIN_FONTSIZE:", file_read, result)) {
	    if (!result.compare("SMALL") || !result.compare("small")) fontsize_table_p->set_fontsize(10);
	    else if (!result.compare("MEDIUM") || !result.compare("medium")) fontsize_table_p->set_fontsize(12);
	    else if (!result.compare("LARGE") || !result.compare("large")
	             || !result.compare("LARGER") || !result.compare("larger")) fontsize_table_p->set_fontsize(14);
	    else {
	      cerr << "Invalid fontsize specified\nDefaulting to medium size" << endl;
	      fontsize_table_p->set_fontsize(12);
	    }
	  }

	  // look for "FILEDIR:"
	  else if (get_prog_parm("FILEDIR:", file_read, result)) {
	    misc_table_p->set_filedir(result);
	  }

	  // look for "PRINT_CMD:"
	  else if (get_prog_parm("PRINT_CMD:", file_read, result)) {
	    misc_table_p->set_print_cmd(result);
	  }

	  // look for "AUTOCQ:"
	  else if (get_prog_parm("AUTOCQ:", file_read, result)) {
	    misc_table_p->set_autocq(result);
	  }
	}
      }
    }
  }
}

void SettingsDialog::get_reset_settings_prompt(void) {
  PromptDialog* dialog_p = new PromptDialog("Enter settings from /etc/" RC_FILE " or\n"
					    "/usr/local/etc/" RC_FILE "?",
					    "Reset settings", standard_size, this);
  if (!dialog_p) {
    cerr << "Memory allocation error in SettingsDialog::get_reset_settings_prompt()" << endl;
    exit(MEM_ERROR);
  }
  int result = dialog_p->exec();
  if (result == QDialog::Accepted) get_reset_settings();
}

void SettingsDialog::get_reset_settings(void) {

  // clear all the existing settings
  identity_table_p->clear();
  com_table_p->clear();
  packet_table_p->clear();
  interface_table_p->clear();
  charset_table_p->clear();
  fontsize_table_p->clear();
  misc_table_p->clear();

  read_config(false); // read settings without searching for config file in home directory
}

bool SettingsDialog::get_prog_parm(const char* name, string& line, string& result) {
// this function looks for a setting named `name' in the string `line'
// and returns the values stated after it in string `result'
// it returns `true' if the setting was found 
// if there are trailing spaces or tabs, string `line' will be modified
// string `result' is only modified if the `name' setting is found

  const string::size_type length = strlen(name);
  // we have to use std::string::substr() because libstdc++-2
  // doesn't support the Std-C++ std::string::compare() functions
  if (!line.substr(0, length).compare(name)) {
    // erase any trailing space or tab
    while (line.find_last_of(" \t") == line.size() - 1) line.resize(line.size() - 1);
    if (line.size() > length) {
      // ignore any preceding space or tab from the setting value given
      string::size_type pos = line.find_first_not_of(" \t", length); // pos now is set to beginning of setting value
      if (pos != string::npos) result.assign(line, pos, line.size() - pos);
    }
    return true;
  }
  return false;
}

bool SettingsDialog::find_prog_parm(const char* name, const string& line) {
// this function looks for a setting named `name' in the string `line'
// it returns `true' if the setting was found or false otherwise

  const string::size_type length = strlen(name);
  // we have to use std::string::substr() because libstdc++-2
  // doesn't support the Std-C++ std::string::compare() functions
  if (!line.substr(0, length).compare(name)) return true;
  return false;
}

bool SettingsDialog::get_rcfile_path(bool search_localfile) {
// search_localfile has a default value of true

  bool found_rcfile = false;

  if (search_localfile) {
    rcfile = prog_func.homedir;
    rcfile += "/." RC_FILE;

    if (!access(rcfile.c_str(), F_OK)) {
      found_rcfile = true;
      is_home_config = true;
    }
  }

  if (!found_rcfile) {
    rcfile = "/usr/local/etc/" RC_FILE;
    if (!access(rcfile.c_str(), F_OK)) found_rcfile = true;
  }

  if (!found_rcfile) {
    rcfile = "/etc/" RC_FILE;
    if (!access(rcfile.c_str(), F_OK)) found_rcfile = true;
  }
  if (!found_rcfile) rcfile = "";
  return found_rcfile;
}

void SettingsDialog::show_help(const string& message, const string& caption) {
  SettingsHelpDialog* dialog_p = new SettingsHelpDialog(standard_size,
							message.c_str(), caption.c_str(), this);
  if (!dialog_p) {
    cerr << "Memory allocation error in SettingsDialog::show_help()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->exec();
  delete dialog_p;
}

void SettingsDialog::strip(string& setting) {

  // erase any trailing space or tab
  while (!setting.empty() && setting.find_last_of(" \t") == setting.size() - 1) {
    setting.resize(setting.size() - 1);
  }
  // erase any leading space or tab
  while (!setting.empty() && (setting[0] == ' ' || setting[0] == '\t')) {
    setting.erase(0, 1);
  }
}

InitSettingsDialog::InitSettingsDialog(const int size, QWidget* widget_p):
                                                    QDialog(widget_p, "Serial Port Settings", true),
						    standard_size(size) {

  top_layout_p = new QVBoxLayout(this, standard_size/2);
  if (!top_layout_p) {
    cerr << "Memory allocation error in  InitSettingsDialog:: InitSettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
    
  QBoxLayout* button_layout_p = new QHBoxLayout(top_layout_p, standard_size/4);
  if (!button_layout_p) {
    cerr << "Memory allocation error in  InitSettingsDialog:: InitSettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
  button_layout_p->addStretch(1);
  button_layout_p->addSpacing(standard_size/2);

  QLabel* message_label_p = new QLabel("Note: pressing the OK button will\n"
				       "save the settings in file ~/." RC_FILE, this);
  if (!message_label_p) {
    cerr << "Memory allocation error in  InitSettingsDialog:: InitSettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
  message_label_p->setAlignment(AlignCenter);
  button_layout_p->addWidget(message_label_p);

  QPushButton* ok_button_p = new QPushButton("OK", this);
  if (!ok_button_p) {
    cerr << "Memory allocation error in  InitSettingsDialog:: InitSettingsDialog()" << endl;
    exit(MEM_ERROR);
  }
  
  QPushButton* cancel_button_p = new QPushButton("Cancel", this);
  if (!cancel_button_p) {
    cerr << "Memory allocation error in  InitSettingsDialog:: InitSettingsDialog()" << endl;
    exit(MEM_ERROR);
  }

  QSize button_size = cancel_button_p->sizeHint();
  ok_button_p->setFixedSize(button_size);
  cancel_button_p->setFixedSize(button_size);

  button_layout_p->addStretch(2);
  button_layout_p->addWidget(ok_button_p);
  button_layout_p->addStretch(2);
  button_layout_p->addWidget(cancel_button_p);
  button_layout_p->addStretch(1);
  button_layout_p->addSpacing(standard_size/2);

  QObject::connect(ok_button_p, SIGNAL(clicked()), this, SLOT(write_config()));
  QObject::connect(cancel_button_p, SIGNAL(clicked()), this, SLOT(reject()));
}

void InitSettingsDialog::attach_settings_widget(QWidget* settings_widget_p) {
  top_layout_p->insertWidget(0, settings_widget_p);
  top_layout_p->activate();
  top_layout_p->freeze();
}

bool InitSettingsDialog::find_prog_parm(const char* name, const string& line) {
// this function looks for a setting named `name' in the string `line'
// it returns `true' if the setting was found or false otherwise

  const string::size_type length = strlen(name);
  // we have to use std::string::substr() because libstdc++-2
  // doesn't support the Std-C++ std::string::compare() functions
  if (!line.substr(0, length).compare(name)) return true;
  return false;
}

void InitSettingsDialog::strip(string& setting) {

  // erase any trailing space or tab
  while (!setting.empty() && setting.find_last_of(" \t") == setting.size() - 1) {
    setting.resize(setting.size() - 1);
  }
  // erase any leading space or tab
  while (!setting.empty() && (setting[0] == ' ' || setting[0] == '\t')) {
    setting.erase(0, 1);
  }
}

string InitSettingsDialog::get_rcfile_path(void) {
// search_localfile has a default value of true

  string rcfile;
  bool found_rcfile = false;

  rcfile = prog_func.homedir;
  rcfile += "/." RC_FILE;
  
  if (!access(rcfile.c_str(), F_OK)) found_rcfile = true;

  if (!found_rcfile) {
    rcfile = "/usr/local/etc/" RC_FILE;
    if (!access(rcfile.c_str(), F_OK)) found_rcfile = true;
  }

  if (!found_rcfile) {
    rcfile = "/etc/" RC_FILE;
    if (!access(rcfile.c_str(), F_OK)) found_rcfile = true;
  }

  if (!found_rcfile) rcfile = "";
  return rcfile;
}

void InitSettingsDialog::show_help(const string& message, const string& caption) {
  SettingsHelpDialog* dialog_p = new SettingsHelpDialog(standard_size,
							message.c_str(), caption.c_str(), this);
  if (!dialog_p) {
    cerr << "Memory allocation error in InitSettingsDialog::show_help()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->exec();
  delete dialog_p;
}

IdentitySettingsDialog::IdentitySettingsDialog(const int size, QWidget* widget_p):
                                                               InitSettingsDialog(size, widget_p) {

  identity_table_p = new IdentityTable(standard_size, this);
  if (!identity_table_p) {
    cerr << "Memory allocation error in  IdentitySettingsDialog:: IdentitySettingsDialog()" << endl;
    exit(MEM_ERROR);
  }

  attach_settings_widget(identity_table_p);

  QObject::connect(identity_table_p, SIGNAL(show_help_sig(const std::string&, const std::string&)),
		   this, SLOT(show_help(const std::string&, const std::string&)));

  setCaption("kamplus: callsign settings");
    
  if (!prog_func.myCall.empty()) identity_table_p->set_mycall(prog_func.myCall);
  if (!prog_func.myPktCall.empty()) identity_table_p->set_mypacketcall(prog_func.myPktCall);
  if (!prog_func.mySelCall.empty()) identity_table_p->set_myselcall(prog_func.mySelCall);
}

void IdentitySettingsDialog::write_config(void) {
  string rcfile(get_rcfile_path());

  bool good_write = true;
  ifstream filein;

  if (!rcfile.empty()) {
    filein.open(rcfile.c_str(), ios::in);
    if (!filein) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      QMessageBox::critical(this, "Config Error", message.c_str(),
			    QMessageBox::Ok | QMessageBox::Default, 0);
      exit(FILEOPEN_ERROR);
    }
  }
  else {
    string message("Can't open file ");
    message += rcfile;
    cerr << message << endl;
    QMessageBox::critical(this, "Config Error", message.c_str(),
			  QMessageBox::Ok | QMessageBox::Default, 0);
    exit(FILEOPEN_ERROR);
  }

  vector<string> file_list;
  string file_read;
  string temp;

  string mycall_line("MY_CALL: ");
  temp = identity_table_p->get_mycall();
  strip(temp);
  if (temp.empty()) {
    good_write = false;
    QMessageBox::critical(this, "Config Error", "No home callsign specified",
			  QMessageBox::Ok | QMessageBox::Default, 0);
  }
  else {
    transform(temp.begin(), temp.end(),
	      temp.begin(), std::toupper);
    mycall_line += temp;
  }
  
  string mypacketcall_line;
  temp =  identity_table_p->get_mypacketcall();
  strip(temp);
  if (temp.empty()) mypacketcall_line = "#PKT_CALL: ";
  else {
    transform(temp.begin(), temp.end(),
	      temp.begin(), std::toupper);
    mypacketcall_line = "PKT_CALL: ";
    mypacketcall_line += temp;
  }
  
  string myselcall_line;
  temp =  identity_table_p->get_myselcall();
  strip(temp);
  if (temp.empty()) myselcall_line = "#SELCALL: ";
  else {
    transform(temp.begin(), temp.end(),
	      temp.begin(), std::toupper);
    myselcall_line = "SELCALL: ";
    myselcall_line += temp;
  }
   
  if (good_write) { // no errors -- write out the configuration file

    bool found_mycall = false;
    bool found_mypacketcall = false;
    bool found_myselcall = false;

    const char terminating_line[] = "## end of " RC_FILE " ##";

    while (getline(filein, file_read)) {
      
      // look for "MY_CALL:"
      if (find_prog_parm("MY_CALL:", file_read) || find_prog_parm("#MY_CALL:", file_read)) {
	if (!found_mycall) {
	  found_mycall = true;
	  file_list.push_back(mycall_line);
	}
      }
      
      // look for "PKT_CALL:"
      else if (find_prog_parm("PKT_CALL:", file_read) || find_prog_parm("#PKT_CALL:", file_read)) {
	if (!found_mypacketcall) {
	  found_mypacketcall = true;
	  file_list.push_back(mypacketcall_line);
	}
      }
      
      // look for "SELCALL:"
      else if (find_prog_parm("SELCALL:", file_read) || find_prog_parm("#SELCALL:", file_read)) {
	if (!found_myselcall) {
	  found_myselcall = true;
	  file_list.push_back(myselcall_line);
	}
      }
      
      // add any residual lines to the list, except the terminating line
      else if (!find_prog_parm(terminating_line, file_read)) file_list.push_back(file_read);
    }
    
    // complete file_list if no existing callsign placeholders in the configuration file
    if (found_mycall == false) file_list.push_back(mycall_line);
    if (found_mypacketcall == false) file_list.push_back(mypacketcall_line);
    if (found_myselcall == false) file_list.push_back(myselcall_line);
      
    // add the terminating line
    file_list.push_back(terminating_line);

    // now write out the new config file
    
    rcfile = prog_func.homedir;
    rcfile += "/." RC_FILE;
    
    ofstream fileout(rcfile.c_str(), ios::out);
    if (!fileout) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      QMessageBox::critical(this, "Config Error", message.c_str(),
			    QMessageBox::Ok | QMessageBox::Default, 0);
      exit(FILEOPEN_ERROR);
    }
    else {
      copy(file_list.begin(), file_list.end(),
	   ostream_iterator<string>(fileout, "\n"));
    }
  }
  // if we were successful, exit the settings dialog and return Qt::Accepted
  if (good_write) emit accept();
}

ComSettingsDialog::ComSettingsDialog(const int size, const int speed, const string& port, QWidget* widget_p):
                                                                            InitSettingsDialog(size, widget_p) {

  com_table_p = new ComTable(standard_size, this);
  if (!com_table_p) {
    cerr << "Memory allocation error in  ComSettingsDialog:: ComSettingsDialog()" << endl;
    exit(MEM_ERROR);
  }

  attach_settings_widget(com_table_p);

  QObject::connect(com_table_p, SIGNAL(show_help_sig(const std::string&, const std::string&)),
		   this, SLOT(show_help(const std::string&, const std::string&)));

  setCaption("kamplus: serial port settings");
    
  if (!port.empty()) com_table_p->set_port(port);
  if (speed) com_table_p->set_speed(speed);
}

void ComSettingsDialog::write_config(void) {
  string rcfile(get_rcfile_path());

  bool good_write = true;
  ifstream filein;

  if (!rcfile.empty()) {
    filein.open(rcfile.c_str(), ios::in);
    if (!filein) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      QMessageBox::critical(this, "Config Error", message.c_str(),
			    QMessageBox::Ok | QMessageBox::Default, 0);
      exit(FILEOPEN_ERROR);
    }
  }
  else {
    string message("Can't open file ");
    message += rcfile;
    cerr << message << endl;
    QMessageBox::critical(this, "Config Error", message.c_str(),
			  QMessageBox::Ok | QMessageBox::Default, 0);
    exit(FILEOPEN_ERROR);
  }

  vector<string> file_list;
  string file_read;
  string temp;

  string port_line("SERIALPORT: ");
  temp = com_table_p->get_port();
  strip(temp);
  if (temp.empty()) {
    good_write = false;
    QMessageBox::critical(this, "Config Error", "No serial port specified",
			  QMessageBox::Ok | QMessageBox::Default, 0);
  }
  else port_line += temp;

  string speed_line("SPEED: ");
  speed_line += com_table_p->get_speed();

  if (good_write) { // no errors -- write out the configuration file
    bool found_port = false;
    bool found_speed = false;

    const char terminating_line[] = "## end of " RC_FILE " ##";

    while (getline(filein, file_read)) {
      
      // look for "SERIALPORT:"
      if (find_prog_parm("SERIALPORT:", file_read)  || find_prog_parm("#SERIALPORT:", file_read)) {
	if (!found_port) {
	  found_port = true;
	  file_list.push_back(port_line);
	}
      }
      
      // look for "SPEED:"
      else if (find_prog_parm("SPEED:", file_read) || find_prog_parm("#SPEED:", file_read))  {
	if (!found_speed) {
	  found_speed = true;
	  file_list.push_back(speed_line);
	}
      }
      // add any residual lines to the list, except the terminating line
      else if (!find_prog_parm(terminating_line, file_read)) file_list.push_back(file_read);
    }
    
    // complete file_list if no existing port or speed placeholder in the configuration file
    if (found_port == false) file_list.push_back(port_line);
    if (found_speed == false) file_list.push_back(speed_line);
      
    // add the terminating line
    file_list.push_back(terminating_line);

    // now write out the new config file
    
    rcfile = prog_func.homedir;
    rcfile += "/." RC_FILE;
    
    ofstream fileout(rcfile.c_str(), ios::out);
    if (!fileout) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      QMessageBox::critical(this, "Config Error", message.c_str(),
			    QMessageBox::Ok | QMessageBox::Default, 0);
      exit(FILEOPEN_ERROR);
    }
    else {
      copy(file_list.begin(), file_list.end(),
	   ostream_iterator<string>(fileout, "\n"));
    }
  }
  // if we were successful, exit the settings dialog and return Qt::Accepted
  if (good_write) emit accept();
}

FiledirTable::FiledirTable(const int standard_size, QWidget* widget_p): QWidget(widget_p) {

  QGridLayout* grid_p = new QGridLayout(this, 3, 3, standard_size/2);
  if (!grid_p) {
    cerr << "Memory allocation error in FiledirTable::FiledirTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->setRowStretch(0, 1);
  grid_p->setRowStretch(2, 1);
  grid_p->setColStretch(1, 1);

  QLabel* label_p = new QLabel("File directory: ", this);
  if (!label_p) {
    cerr << "Memory allocation error in FiledirTable::FiledirTable()" << endl;
    exit(MEM_ERROR);
  }
  grid_p->addWidget(label_p, 1, 0, Qt::AlignRight | Qt::AlignVCenter);

  filedir_entry_p = new QLineEdit(this);
  if (!filedir_entry_p) {
    cerr << "Memory allocation error in FiledirTable::FiledirTable()" << endl;
    exit(MEM_ERROR);
  }
  filedir_entry_p->setMinimumSize(standard_size * 6, (standard_size * 5)/6);
  grid_p->addWidget(filedir_entry_p, 1, 1);

  QPixmap helpIcon(help_xpm);
  QToolButton* button_p = new QToolButton(this);
  if (!button_p) {
    cerr << "Memory allocation error in FiledirTable::FiledirTable()" << endl;
    exit(MEM_ERROR);
  }
  button_p->setPixmap(helpIcon);
  button_p->setTextLabel(help_messages.get_message(MiscMessages::filedir).c_str(), true);
  QObject::connect(button_p, SIGNAL(clicked()), this, SLOT(show_filedir_help()));
  grid_p->addWidget(button_p, 1, 2);

  grid_p->activate();
}

void FiledirTable::show_filedir_help(void) {
  emit show_help_sig(help_messages.get_message(MiscMessages::filedir),
		     help_messages.get_caption(MiscMessages::filedir));
}

FiledirSettingsDialog::FiledirSettingsDialog(const int size, QWidget* widget_p):
                                                             InitSettingsDialog(size, widget_p) {

  filedir_table_p = new FiledirTable(standard_size, this);
  if (!filedir_table_p) {
    cerr << "Memory allocation error in FiledirSettingsDialog:: FiledirSettingsDialog()" << endl;
    exit(MEM_ERROR);
  }

  attach_settings_widget(filedir_table_p);

  QObject::connect(filedir_table_p, SIGNAL(show_help_sig(const std::string&, const std::string&)),
		   this, SLOT(show_help(const std::string&, const std::string&)));

  setCaption("kamplus: file directory setting");
}

void FiledirSettingsDialog::write_config(void) {
  string rcfile(get_rcfile_path());

  bool good_write = true;
  ifstream filein;

  if (!rcfile.empty()) {
    filein.open(rcfile.c_str(), ios::in);
    if (!filein) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      QMessageBox::critical(this, "Config Error", message.c_str(),
			    QMessageBox::Ok | QMessageBox::Default, 0);
      exit(FILEOPEN_ERROR);
    }
  }
  else {
    string message("Can't open file ");
    message += rcfile;
    cerr << message << endl;
    QMessageBox::critical(this, "Config Error", message.c_str(),
			  QMessageBox::Ok | QMessageBox::Default, 0);
    exit(FILEOPEN_ERROR);
  }

  vector<string> file_list;
  string file_read;
  string temp;

  string filedir_line("FILEDIR: ");
  temp = filedir_table_p->get_filedir();
  strip(temp);
  if (temp.empty()) {
      good_write = false;
      QMessageBox::critical(this, "Config Error", "No file directory specified",
			    QMessageBox::Ok | QMessageBox::Default, 0);
  }
  else filedir_line += temp;
  
  if (good_write) { // no errors -- write out the configuration file

    bool found_filedir = false;

    const char terminating_line[] = "## end of " RC_FILE " ##";

    while (getline(filein, file_read)) {
      
      // look for "FILEDIR:"
      if (find_prog_parm("FILEDIR:", file_read) || find_prog_parm("#FILEDIR:", file_read)) {
	if (!found_filedir) {
	  found_filedir = true;
	  file_list.push_back(filedir_line);
	}
      }
      
      // add any residual lines to the list, except the terminating line
      else if (!find_prog_parm(terminating_line, file_read)) file_list.push_back(file_read);
    }
    
    // complete file_list if no existing filedir placeholder in the configuration file
    if (found_filedir == false) file_list.push_back(filedir_line);
      
    // add the terminating line
    file_list.push_back(terminating_line);

    // now write out the new config file
    
    rcfile = prog_func.homedir;
    rcfile += "/." RC_FILE;
    
    ofstream fileout(rcfile.c_str(), ios::out);
    if (!fileout) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      QMessageBox::critical(this, "Config Error", message.c_str(),
			    QMessageBox::Ok | QMessageBox::Default, 0);
      exit(FILEOPEN_ERROR);
    }
    else {
      copy(file_list.begin(), file_list.end(),
	   ostream_iterator<string>(fileout, "\n"));
    }
  }
  // if we were successful, exit the settings dialog and return Qt::Accepted
  if (good_write) emit accept();
}
