// fdstream.cpp

#include "fdstream.h"
// for memmove():
#include <cstring>

/* constructor of fdoutbuf
 * - initialize file descriptor or set to -1 to use attach() later
 * - initialize data buffer
 * - one character less to let the bufSizeth character
 *   cause a call to overflow()
 */
fdoutbuf::fdoutbuf(int _fd, bool _manage) : fd(_fd), manage(_manage) {
  setp (buffer,                // beginning of buffer
	buffer + bufSize - 1); // position to cause an overflow()
}

fdoutbuf::~fdoutbuf() {
  if (fd >= 0) {
    sync();
    if (manage) ::close(fd);
  }
}

int fdoutbuf::sync() {
  if (flushBuffer() == EOF) {
    // ERROR
    return -1;
  }
  return 0;
}

void fdoutbuf::attach_fd(int _fd, bool _manage) {
  if (fd >= 0) {
    sync();
    if (manage) ::close(fd);
  }    
  fd = _fd;
  manage = _manage;
}

void fdoutbuf::close_fd() {
  if (fd >= 0) {
    sync(); 
    ::close(fd); 
    fd = -1;
  }
}

fdoutbuf::int_type fdoutbuf::overflow(fdoutbuf::int_type c) {
  // the stream must be full: empty buffer and write c
  if (c != EOF) {
    *pptr() = c;
    pbump(1);
  }

  // flush buffer
  if (flushBuffer() == EOF) {
    // ERROR
    return EOF;
  }
  return c;
}

std::streamsize fdoutbuf::xsputn(const char* s, std::streamsize num) {
  // this method writes multiple characters
  // if num is less than blockSize use sputc() on each character
  // this is most easily done by calling the default xsputn()
  if (num < blockSize) return std::streambuf::xsputn(s, num);

  // if there is a block transfer, first flush anything in the buffer
  //  and then use Unix ::write()
  std::streamsize return_val = num;
  if (flushBuffer() == EOF || write(fd, s, num) != num) return_val = EOF;
  return return_val;
}
    

/* constructor of fdinbuf
 * - initialize file descriptor or set to -1 to use attach() later
 * - initialize empty data buffer
 * - reset for no putback area and forced underflow
 */
fdinbuf::fdinbuf(int _fd, bool _manage = true): fd(_fd), manage(_manage) {
  reset();
}

fdinbuf::~fdinbuf() {
  if (manage && fd >= 0) ::close(fd);
}

void fdinbuf::attach_fd(int _fd, bool _manage) {
  reset();
  if (manage && fd >= 0) ::close(fd);
  fd = _fd;
  manage = _manage;
}

void fdinbuf::close_fd() {
  if (fd >= 0) {
    reset();
    ::close(fd);
    fd = -1;}
}

fdinbuf::int_type fdinbuf::underflow() {
  // insert new characters into the buffer when it is empty

  // is read position before end of buffer?
  if (gptr() < egptr()) {
    return traits_type::to_int_type(*gptr());
  }

  /* process size of putback area
   * - use number of characters read
   * - but at most size of putback area
   */
  int numPutback;
  numPutback = gptr() - eback();
  if (numPutback > pbSize) {
    numPutback = pbSize;
  }

  /* copy up to pbSize characters previously read into the putback
   * area -- use memmove() because if the size of file modulus bufSize
   * is less than pbSize, the source and destination ranges can
   * overlap on the last call to underflow() (ie the one which leads
   * to EOF being returned)
   */
  std::memmove(buffer+(pbSize-numPutback), gptr()-numPutback,
	       numPutback);

  // read at most bufSize new characters
  int num;
  num = read(fd, buffer+pbSize, bufSize);
  if (num <= 0) {
    // ERROR or EOF
    // we might as well make the put back area readable so reset
    // the buffer pointers to enable this
      setg(buffer+(pbSize-numPutback),   // beginning of putback area
	   buffer+pbSize,                // start of read area of buffer
	   buffer+pbSize);               // end of buffer - it's now empty
    return EOF;
  }
  
  // reset buffer pointers
  setg(buffer+(pbSize-numPutback),   // beginning of putback area
       buffer+pbSize,                // read position
       buffer+pbSize+num);           // end of buffer

  // return next character
  return traits_type::to_int_type(*gptr());
}
