/* Copyright (C) 1999, 2000, 2001, 2002 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

// main.cpp

#ifndef CONFIG_H
#define CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include <cstdlib>
#include <iostream>
#include <strstream>
#include <string>
#include <algorithm>
#include <csignal>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <limits.h>
#include <fstream>
#include <cstring>
#include <cctype>
#include <qapplication.h>
#include <qmessagebox.h>

#if QT_VERSION<200
#error Wrong Qt version - either Qt-2.* or Qt-3.* required
#endif

#include "iqueue.cpp"   // contains template definition
#include "buffers.h"
#include "filesend.h"
#include "com.h"
#include "prog_defs.h"
#include "pipes.h"
#include "tnc.h"
#include "settings.h"

#if QT_VERSION >= 300
#include "dialogs-qt3.h"
#include "widgets-qt3.h"
#else
#include "dialogs.h"
#include "widgets.h"
#endif

///////////////////////// function declarations //////////////////////

void configure_program(int&, string&, int&, int&, bool&, bool&, int&, Tnc_func::Keep_alive&);
void parent_process(int, char**, pid_t, Pipe_fifo&, Pipe_fifo&, int, int, bool, bool, int, Tnc_func::Keep_alive);
void child_process(int, const string&, Pipe_fifo&, Pipe_fifo&);
bool get_prog_parm(const char*, string&, string&);
void com_error_dialog(const char*);
extern "C" void parent_exit_handler(int);
extern "C" void child_exit_handler(int);
void init_prog(int&, string&);
void read_init(int&, string&);

///////////////////////// global/static data /////////////////////////

Prog_func prog_func = {Prog_func::word, Prog_func::normal, false, "", "", "", "", "", "",
		         false, false, 0, false, {0}, 0, 
		         Prog_func::amtor, Prog_func::cp437};

static struct {int argc; char** argv;} main_parms; // needed for com_error_dialog() and get_port()
                                                   // (com_error_dialog() is intended to be passed as
                                                   // a function pointer to the serial port object - it
                                                   // is too cumbersome, and bad for code modularity,
                                                   // to pass the parameters of main() to the
                                                   // serial port object, so we have put them in
                                                   // a static object of file scope so com_error_dialog()
                                                   // can see them)

//////////////////////////////////////////////////////////////////////

static volatile bool child_exit_flag = false;

int main(int argc, char* argv[]) {

    main_parms.argc = argc;
    main_parms.argv = argv;

    string port;
    int speed;
    int vhf_paclen;
    int hf_paclen;
    bool no_rx_cw_flag;
    bool rx_bell_flag;
    int win_fontsize;
    struct Tnc_func::Keep_alive keep_alive;

    int count;
    for (count = 1; count < argc; count++) {
        if (!strcmp(argv[count], "setup") || !strcmp(argv[count], "-setup")
	    || !strcmp(argv[count], "--setup")) {
	    prog_func.prog_mode = Prog_func::setup;
	}
    }

    configure_program(speed, port, vhf_paclen, hf_paclen, no_rx_cw_flag, rx_bell_flag,
		      win_fontsize, keep_alive);

    if (!prog_func.GPL_flag
	|| !speed || port.empty()
	|| prog_func.myCall.empty()
	|| prog_func.filedir.empty()) {
      init_prog(speed, port);
    }

    if (Com::check_port(port.c_str(), com_error_dialog) == -1) exit(COM_ERROR);  // check that port not in use

    Pipe_fifo receive_pipe(Pipe_fifo::non_block);
    Pipe_fifo send_pipe(Pipe_fifo::non_block);
    
    pid_t child_pid = fork();
    if (child_pid == -1) {
        cerr << "Fork error - exiting" << endl;
	exit(FORK_ERROR);
    }
    else if (child_pid) {
        parent_process(argc, argv, child_pid, receive_pipe, send_pipe, vhf_paclen, hf_paclen,
		       no_rx_cw_flag, rx_bell_flag, win_fontsize, keep_alive);
    }
    else child_process(speed, port, receive_pipe, send_pipe);
    return 0;
}

//////////////////////////// function definitions //////////////////////////


/////////////////// parent process ////////////////////

void parent_process(int argc, char** argv, pid_t child_pid, Pipe_fifo& receive_pipe, Pipe_fifo& send_pipe,
		    int vhf_paclen, int hf_paclen, bool no_rx_cw_flag, bool rx_bell_flag, int win_fontsize,
		    Tnc_func::Keep_alive keep_alive) {

// catch SIGQUIT, SIGTERM SIGINT SIGHUP and SIGPIPE
    signal(SIGQUIT, parent_exit_handler);
    signal(SIGTERM, parent_exit_handler);
    signal(SIGINT, parent_exit_handler);
    signal(SIGHUP, parent_exit_handler);
    signal(SIGPIPE, parent_exit_handler);

////////////////////// make the pipes unidirectional

    receive_pipe.make_readonly();
    send_pipe.make_writeonly();

////////////////////// release privileges

    // if we are running suid/sgid root to enable easy access to the
    // serial port, we can now give up root privileges in this
    // process, as only the child process (which manages the com port)
    // will need root access

#ifdef RELEASE_PRIVILEGES
    setgid(getgid());
    setuid(getuid());
#endif

////////////////////// create the necessary buffers

    Transmit_buffer tr_buffer;
    BufferList buffer_list;

////////////////////// create the tnc objects

    Tnc* tnc_p = 0;
    Tnc_base* tnc_base_p;
    if (prog_func.prog_mode == Prog_func::normal) {
        tnc_p = new Tnc(tr_buffer, receive_pipe, send_pipe,
		vhf_paclen, hf_paclen, no_rx_cw_flag, rx_bell_flag,
		keep_alive);

	if (!tnc_p) {
	    cerr << "Memory allocation error in parent_process()" << endl;
	    exit(MEM_ERROR);
	}
	tnc_base_p = tnc_p;
    }
    else {
        tnc_base_p = new Tnc_setup(tr_buffer, receive_pipe, send_pipe);
	if (!tnc_base_p) {
	    cerr << "Memory allocation error in parent_process()" << endl;
	    exit(MEM_ERROR);
	}
    }

////////////////////// now do the Qt stuff

    QApplication app(argc, argv);

    int global_fontsize = 0;
    int count;
    char** parm_list = new char*[app.argc()];
    char** list_p = parm_list;
    bool invalid_param = false;

    for (count = 1; count < app.argc(); count++) {
        int temp = 0;
        if ((!strcmp(app.argv()[count], "fontsize") || !strcmp(app.argv()[count], "-fontsize")
	       || !strcmp(app.argv()[count], "--fontsize"))
	    && ((temp = atoi(app.argv()[count + 1])) == 10
		|| temp == 11
		|| temp == 12)) {
	    global_fontsize = temp;
	    count++;
	}
        else if (!(!strcmp(app.argv()[count], "setup") || !strcmp(app.argv()[count], "-setup")
	    || !strcmp(app.argv()[count], "--setup"))) {
	    *list_p = app.argv()[count];
	    list_p++;
	    invalid_param = true;
	}
    }
    *list_p = 0;

    if (invalid_param) {
        cerr << "Unrecognised parameters: ";
	for (list_p = parm_list; *list_p != 0; list_p++) {
	    cerr << *list_p << ' ';
	}
	cerr << endl;
    }

    delete[] parm_list;
        
    if (global_fontsize) app.setFont(QFont("Helvetica", global_fontsize, QFont::Normal));  // set the default font

    MainScreen* mainscreen_p = new MainScreen(tnc_p, tnc_base_p, receive_pipe, tr_buffer,
						  buffer_list, win_fontsize);
    if (!mainscreen_p) {
        cerr << "Memory allocation error in parent_process()" << endl;
	exit(MEM_ERROR);
    }
    
    app.setMainWidget(mainscreen_p);
    mainscreen_p->show();

// we enter the main Qt program loop below when everything is set up


////////////////// now we can pass information about the Qt windows to the tnc objects

    tnc_base_p->set_wins(mainscreen_p, mainscreen_p->get_receivewin());
    
////////////////// make sure relevant Kam parameters are correct

// set up our selcall if configure_program() has not found it
    if (prog_func.mySelCall.empty() && tnc_p) {
        if (tnc_p->is_validcall(prog_func.myCall)) tnc_p->make_selcall(prog_func.myCall, prog_func.mySelCall);
	else {
	    mainscreen_p->get_receivewin()->write
	      ("You have a non-standard callsign, which means I cannot work out your "
	       "Amtor Selcall\nPlease enter it via the Settings dialog\n");
	}
    }

// now make sure the callsign and selcall in the Kam and the MAXUSERS parameter match the ones in this program
    if (tnc_p) {
        string commandmessage("MYCALL ");
	commandmessage += prog_func.myPktCall;
	tnc_p->send_kamcommand(commandmessage.c_str());
	usleep(200000);

#ifndef NO_PACTOR
  #ifndef NO_GTOR
	commandmessage = "MYGTCALL ";
	commandmessage += prog_func.myCall;
	tnc_p->send_kamcommand(commandmessage.c_str());
	usleep(200000);
  #endif
	commandmessage = "MYPTCALL ";
	commandmessage += prog_func.myCall;
	tnc_p->send_kamcommand(commandmessage.c_str());
	usleep(200000);
#endif
	commandmessage = "MYSELCAL ";
	commandmessage += prog_func.mySelCall;
	tnc_p->send_kamcommand(commandmessage.c_str());
      
	usleep(200000);
	ostrstream s1;
	s1 << "MAXUSERS " << MAXUSERS << ends;
	commandmessage = s1.str();
	tnc_p->send_kamcommand(commandmessage.c_str());
	usleep(100000);
    }

// everything is set up
// now enter the main program loop

    app.exec();

//  app has terminated - clean up and then quit
// kill child process
    kill(child_pid, SIGQUIT);

// make sure child process is really dead
    usleep(500000);
    kill(child_pid, SIGKILL);
}

/////////////////////// child process ////////////////////////////

void child_process(int speed, const string& port, Pipe_fifo& receive_pipe, Pipe_fifo& send_pipe) {

// set up the signal that will tell this process to terminate
    signal(SIGQUIT, child_exit_handler);
// also catch SIGTERM SIGINT SIGHUP and SIGPIPE
    signal(SIGTERM, child_exit_handler);
    signal(SIGINT, child_exit_handler);
    signal(SIGHUP, child_exit_handler);
    signal(SIGPIPE, child_exit_handler);

// set up variables and objects for this process
    char combuffer[PIPE_BUF];
    Com com_obj(speed, port.c_str(), com_error_dialog);

// make the pipes unidirectional

    receive_pipe.make_writeonly();
    send_pipe.make_readonly();

    if (prog_func.prog_mode == Prog_func::normal) {
// make sure the Kam is in host mode

        com_obj.send(3);           // make sure we are in command mode if not already in host mode
	usleep(100000);
	com_obj.send('\r');        // clear out any debris in the file buffer
	usleep(100000);
	com_obj.send((unsigned char*)"INTFACE HOST\r");        // put the Kam in host mode
	sleep(1);
	com_obj.send((unsigned char*)"RESET\r");
	usleep(100000);
    }
    else {  // we must be trying to set up the Kam
// make sure we are not in host mode
	usleep(100000);
	const unsigned char message[4] = {FEND, 'Q', FEND, 0};
        com_obj.send(message);
    }

    int result1;
    int result2;
    while (!child_exit_flag) {
        // receive input and insert into pipe

        while ((result1 = com_obj.receive((uchar*)combuffer, PIPE_BUF)) > 0) {
	    do {
	        result2 = receive_pipe.write(combuffer, result1);
	    } while (!result2);  // keep going until pipe will accept input
	}
	// now send anything waiting in pipe to be sent

	if ((result1 = send_pipe.read(combuffer, PIPE_BUF)) > 0) {
	    do {
	        result2 = com_obj.send((uchar*)combuffer, result1);
	    } while (!result2);  // keep going until com port will accept output
	}
    }
}

///////////////////////////////////////////////////////////


void configure_program(int& speed, string& port, int& vhf_paclen, int& hf_paclen, 
		       bool& no_rx_cw_flag, bool& rx_bell_flag, int& win_fontsize,
		       Tnc_func::Keep_alive& keep_alive) {
    char* home = getenv("HOME");
    if (!home) cerr << "Your HOME environmental variable is not defined!" << endl;
    else prog_func.homedir = home;

    bool found_rcfile = false;
    ifstream filein;
    string rcfile;

    if (!prog_func.homedir.empty()) {
        rcfile = prog_func.homedir;
	rcfile += "/." RC_FILE;

#ifdef HAVE_IOS_NOCREATE
	filein.open(rcfile.c_str(), ios::in | ios::nocreate);
#else
	// we must have Std C++ so we probably don't need a ios::nocreate
	// flag on a read open to ensure uniqueness
	filein.open(rcfile.c_str(), ios::in);
#endif

	if (filein) found_rcfile = true;
	else filein.clear();
    }

    if (!found_rcfile) {

        rcfile = "/usr/local/etc/" RC_FILE;

#ifdef HAVE_IOS_NOCREATE
	filein.open(rcfile.c_str(), ios::in | ios::nocreate);
#else
	// we must have Std C++ so we probably don't need a ios::nocreate
	// flag on a read open to ensure uniqueness
	filein.open(rcfile.c_str(), ios::in);
#endif

	if (filein) found_rcfile = true;
	else filein.clear();
    }

    if (!found_rcfile) {

        rcfile = "/etc/" RC_FILE;

#ifdef HAVE_IOS_NOCREATE
	filein.open(rcfile.c_str(), ios::in | ios::nocreate);
#else
	// we must have Std C++ so we probably don't need a ios::nocreate
	// flag on a read open to ensure uniqueness
	filein.open(rcfile.c_str(), ios::in);
#endif

	if (filein) found_rcfile = true;
	else filein.clear();
    }

    if (!found_rcfile) {
        cerr << "Can't find or open file /etc/" RC_FILE << endl;
	cerr << "/usr/local/etc" RC_FILE " or file $HOME/." RC_FILE << endl;
        exit(FILEOPEN_ERROR);
    }

// check to see if GPL terms have been accepted

    string gpl_file(prog_func.homedir + "/.kamgpl-qt");
    int result = access(gpl_file.c_str(), F_OK);
  
    if (!result) prog_func.GPL_flag = true;

// set default values

    vhf_paclen = MAX_FRAMESIZE;
    hf_paclen = MAX_FRAMESIZE;
    no_rx_cw_flag = false;
    rx_bell_flag = false;
    win_fontsize = 12;
    prog_func.charset = Prog_func::cp437;
    keep_alive.keep_alive_flag = false;
    keep_alive.keep_alive_minutes = 10;
    speed = 0;
    
// now extract settings from file

    string file_read;
    string temp;

    while (getline(filein, file_read)) {

        if (!file_read.empty() && file_read[0] != '#') { // valid line to check
	    // now check for other comment markers
	    string::size_type pos = file_read.find_first_of('#');
	    if (pos != string::npos) file_read.resize(pos); // truncate
	
	    // look for "MY_CALL:"
	    if (get_prog_parm("MY_CALL:", file_read, prog_func.myCall)) {
	        transform(prog_func.myCall.begin(), prog_func.myCall.end(),
			  prog_func.myCall.begin(), std::toupper);
	        if (prog_func.myPktCall.empty()) { // only overwrite myPktCall if no PKT_CALL: yet specified
		    prog_func.myPktCall = prog_func.myCall;
		}
	    }
	    
	    // look for "PKT_CALL:"
	    else if (get_prog_parm("PKT_CALL:", file_read, prog_func.myPktCall)) {
	        transform(prog_func.myPktCall.begin(), prog_func.myPktCall.end(),
			  prog_func.myPktCall.begin(), std::toupper);
	    }
	    
	    // look for "SELCALL:"
	    else if (get_prog_parm("SELCALL:", file_read, prog_func.mySelCall)) {
	        transform(prog_func.mySelCall.begin(), prog_func.mySelCall.end(),
			  prog_func.mySelCall.begin(), std::toupper);
	    }

	    // look for "SERIALPORT:"
	    else if (get_prog_parm("SERIALPORT:", file_read, port));

	    // look for "FILEDIR:"
	    else if (get_prog_parm("FILEDIR:", file_read, prog_func.filedir));

	    // look for "PRINT_CMD:"
	    else if (get_prog_parm("PRINT_CMD:", file_read, prog_func.print_cmd));

	    // look for "SPEED:"
	    else if (get_prog_parm("SPEED:", file_read, temp)) {
	        int speed_val = atoi(temp.c_str());
		if (speed_val == 1200 || speed_val == 2400 || speed_val == 4800 || speed_val == 9600) {
		    speed = speed_val;
		}
	    }

	    // look for "NOPROMPT:"
	    else if (get_prog_parm("NOPROMPT:", file_read, temp)) {
	        if (!temp.compare("true") || !temp.compare("TRUE")) {
		    prog_func.noprompt_flag = true;
		}
	    }

	    // look for "AUTOCQ:"
	    else if (get_prog_parm("AUTOCQ:", file_read, temp)) {
		prog_func.autocq_delay = atoi(temp.c_str());
		if (prog_func.autocq_delay < 0) prog_func.autocq_delay = 0;
	    }

	    // look for "V_PACLEN:"
	    else if (get_prog_parm("V_PACLEN:", file_read, temp)) {
		vhf_paclen = atoi(temp.c_str());
		if (vhf_paclen < 32) vhf_paclen = 32;
		if (vhf_paclen > MAX_FRAMESIZE) vhf_paclen = MAX_FRAMESIZE;
	    }

	    // look for "H_PACLEN:"
	    else if (get_prog_parm("H_PACLEN:", file_read, temp)) {
		hf_paclen = atoi(temp.c_str());
		if (hf_paclen < 32) hf_paclen = 32;
		if (hf_paclen > MAX_FRAMESIZE) hf_paclen = MAX_FRAMESIZE;
	    }

	    // look for "NO_CW_RX:"
	    else if (get_prog_parm("NO_CW_RX:", file_read, temp)) {
	        if (!temp.compare("true") || !temp.compare("TRUE")) {
		    no_rx_cw_flag = true;
		}
	    }

	    // look for "BELL:"
	    else if (get_prog_parm("BELL:", file_read, temp)) {
	        if (!temp.compare("true") || !temp.compare("TRUE")) {
		    rx_bell_flag = true;
		}
	    }

	    // look for "CHAR_SET:"
	    else if (get_prog_parm("CHAR_SET:", file_read, temp)) {
	        if (!temp.compare("latin-1") || !temp.compare("LATIN-1")) {
		    prog_func.charset = Prog_func::latin_1;
		}
	    }

	    // look for "KEEPALIVE:"
	    else if (get_prog_parm("KEEPALIVE:", file_read, temp)) {
	        if (!temp.compare("true") || !temp.compare("TRUE")) {
		    keep_alive.keep_alive_flag = true;
		}
	    }

	    // look for "KEEPALIVE_MINS:"
	    else if (get_prog_parm("KEEPALIVE_MINS:", file_read, temp)) {
		keep_alive.keep_alive_minutes = atoi(temp.c_str());
		if (keep_alive.keep_alive_minutes < 1) keep_alive.keep_alive_minutes = 10;
	    }

	    // look for "WIN_FONTSIZE:"
	    else if (get_prog_parm("WIN_FONTSIZE:", file_read, temp)) {
	        if (!temp.compare("large") || !temp.compare("LARGE")
		    || !temp.compare("larger") || !temp.compare("LARGER")) {
		    win_fontsize = 14;
		}
	        else if (!temp.compare("small") || !temp.compare("SMALL")) {
		    win_fontsize = 10;
		}
	    }
	}
    }

    if (prog_func.print_cmd.empty()) {
        cerr << "No print command specified in kamrc,\n"
	        "defaulting to lpr" << endl;
	prog_func.print_cmd = "lpr";
    }
    filein.close();
}

bool is_baudot(char letter) {
// this returns true if the character falls within the permitted Baudot character set
// and otherwise it returns false.
// Both upper and lower case letters are passed as true.
// Note that the US code substitutes ; for =, and " for +, as used by
// CCIT No. 2 code.  All of these are passed true by is_baudot(), but may not be
// transmitted by the Kam, depending on the CODE RTTY and CODE AMTOR settings

    bool return_value = false;
    if (letter == '\n'
	|| (letter > 31 && letter < 64
	    && letter != 37 
	    && letter != 42
	    && letter != 60
	    && letter != 62)
	|| (letter > 64 && letter < 91)
	|| (letter > 96 && letter < 123)) {
        return_value = true;
    }
    return return_value;
}

bool is_morse(char letter) {
// this returns true if the character falls within the permitted Morse character set
// or one of the Kam CW special letters, and otherwise it returns false.
// Both upper and lower case letters are passed as true.

    bool return_value = false;
    if (letter == '\n'
	|| (letter > 31 && letter < 34)
	|| letter == 35
	|| (letter > 36 && letter < 60)
	|| letter == 61
	|| letter == 63
	|| (letter > 64 && letter < 91)
	|| letter == 92
	|| (letter > 96 && letter < 123)) {
        return_value = true;
    }
    return return_value;
}

int get_month(char (&month)[4], int mon) {
// month is of type reference to array of length 4 chars
// no array of a different length can be assigned to it, so
// buffer overruns are impossible

// this converts the month number in the tm struct into a 3 letter string
// the character array passed to month must have sufficient capacity to
// take a 3 letter string and null terminating character
// if the month number is not valid, the function returns -1
// otherwise it returns 0
    int result = 0;
    switch(mon) {
    case 0:
        strcpy(month, "JAN");
	break;
    case 1:
        strcpy(month, "FEB");
	break;
    case 2:
        strcpy(month, "MAR");
	break;
    case 3:
        strcpy(month, "APR");
	break;
    case 4:
        strcpy(month, "MAY");
	break;
    case 5:
        strcpy(month, "JUN");
	break;
    case 6:
        strcpy(month, "JUL");
	break;
    case 7:
        strcpy(month, "AUG");
	break;
    case 8:
        strcpy(month, "SEP");
	break;
    case 9:
        strcpy(month, "OCT");
	break;
    case 10:
        strcpy(month, "NOV");
	break;
    case 11:
        strcpy(month, "DEC");
	break;
    default:
        result = -1;
    }
    return result;
}

bool get_prog_parm(const char* name, string& line, string& result) {
// this function looks for a setting named `name' in the string `line'
// and returns the values stated after it in string `result'
// it returns `true' if the setting was found 
// if there are trailing spaces or tabs, string `line' will be modified
// string `result' is only modified if the `name' setting is found

  const string::size_type length = strlen(name);
  // we have to use std::string::substr() because libstdc++-2
  // doesn't support the Std-C++ std::string::compare() functions
  if (!line.substr(0, length).compare(name)) {
    // erase any trailing space or tab
    while (line.find_last_of(" \t") == line.size() - 1) line.resize(line.size() - 1);
    if (line.size() > length) {
      // ignore any preceding space or tab from the setting value given
      string::size_type pos = line.find_first_not_of(" \t", length); // pos now is set to beginning of setting value
      if (pos != string::npos) result.assign(line, pos, line.size() - pos);
    }
    return true;
  }
  return false;
}

void com_error_dialog(const char* message) {
    // if we are running suid/sgid root to enable easy access to the
    // serial port, we can now give up root privileges in this
    // process

#ifdef RELEASE_PRIVILEGES
    setgid(getgid());
    setuid(getuid());
#endif

    QApplication app(main_parms.argc, main_parms.argv);   // create the singleton QApplication object
                                                          // here - we will never enter the main program loop
    QMessageBox::critical(0, "Kamplus-qt: Serial port error", message,
			  QMessageBox::Ok | QMessageBox::Default, 0);
}

void parent_exit_handler(int) {
    prog_func.exitflag = true;
}

void child_exit_handler(int) {
    child_exit_flag = true;
}

void init_prog(int& speed, string& port) {
    pid_t child_pid = fork();
    if (child_pid == -1) {
        cerr << "Fork error - exiting" << endl;
	exit(FORK_ERROR);
    }
    else if (!child_pid) {
#ifdef RELEASE_PRIVILEGES
        setgid(getgid());
	setuid(getuid());
#endif
	int exit_code = 0;
	{ // put the QApplication object in its own scope, to ensure orderly destruction before we hit _exit()

	    QApplication app(main_parms.argc, main_parms.argv); // create a QApplication object - this is in a new process
	                                                        // and will not affect the QApplication object in the parent process
	    int global_fontsize = 0;
	    int count;
	    char** parm_list = new char*[app.argc()];
	    char** list_p = parm_list;

	    for (count = 1; count < app.argc(); count++) {
	        int temp = 0;
		if ((!strcmp(app.argv()[count], "fontsize") || !strcmp(app.argv()[count], "-fontsize")
		     || !strcmp(app.argv()[count], "--fontsize"))
		    && ((temp = atoi(app.argv()[count + 1])) == 10
			|| temp == 11
			|| temp == 12)) {
		    global_fontsize = temp;
		    count++;
		}
	    }
	    *list_p = 0;
	    delete[] parm_list;
        
	    if (global_fontsize) app.setFont(QFont("Helvetica", global_fontsize, QFont::Normal));  // set the default font

	    if (!prog_func.GPL_flag) {
	        GplDialog* gpl_dialog_p = new GplDialog(30);
		if (!gpl_dialog_p) {
		    cerr << "Memory allocation error in init_prog()" << endl;
		    exit_code = MEM_ERROR;
		}
		else {
		    int result = gpl_dialog_p->exec();
		    delete gpl_dialog_p;
		    if (result == QDialog::Accepted) {
		        string gpl_file(prog_func.homedir);
			gpl_file += "/.kamgpl-qt";
			ofstream gplfile(gpl_file.c_str(), ios::out);
		    }
		    else {
		        QApplication::beep();
			exit_code = CONFIG_ERROR;
		    }
		}
	    }
	    if (!exit_code && prog_func.myCall.empty()) {
	        IdentitySettingsDialog* dialog_p = new IdentitySettingsDialog(30);
		if (!dialog_p) {
		    cerr << "Memory allocation error in init_prog()" << endl;
		    exit_code = MEM_ERROR;
		}
		else {
		    int result = dialog_p->exec();
		    delete dialog_p;
		    if (result == QDialog::Rejected) {
		        QApplication::beep();
			exit_code = CONFIG_ERROR;
		    }
		}
	    }
	    if (!exit_code && (!speed || port.empty())) {
	        ComSettingsDialog* dialog_p = new ComSettingsDialog(30, speed, port);
		if (!dialog_p) {
		    cerr << "Memory allocation error in init_prog()" << endl;
		    exit_code = MEM_ERROR;
		}
		else {
		    int result = dialog_p->exec();
		    delete dialog_p;
		    if (result == QDialog::Rejected) {
		        QApplication::beep();
			exit_code = CONFIG_ERROR;
		    }
		}
	    }
	    if (!exit_code && prog_func.filedir.empty()) {
	        FiledirSettingsDialog* dialog_p = new FiledirSettingsDialog(30);
		if (!dialog_p) {
		    cerr << "Memory allocation error in init_prog()" << endl;
		    exit_code = MEM_ERROR;
		}
		else {
		    int result = dialog_p->exec();
		    delete dialog_p;
		    if (result == QDialog::Rejected) {
		        QApplication::beep();
			exit_code = CONFIG_ERROR;
		    }
		}
	    }
	}
	_exit(exit_code); // child process ends here: use _exit() not exit()
    }
    int exit_code = 0;
    wait(&exit_code); // make the main program process wait until we have got the port and written it to the configuration file
    if (exit_code) exit(exit_code);
    read_init(speed, port);
}

void read_init(int& speed, string& port) {
  string rcfile(prog_func.homedir);
  rcfile += "/." RC_FILE;
    
  ifstream filein(rcfile.c_str(), ios::in);
  if (!filein) {
    string message("Can't open file ");
    message += rcfile;
    cerr << message << endl;
    QApplication app(main_parms.argc, main_parms.argv);   // create the singleton QApplication object
                                                          // here - we will never enter the main program loop
    QMessageBox::critical(0, "Config Error", message.c_str(),
			  QMessageBox::Ok | QMessageBox::Default, 0);
    exit(FILEOPEN_ERROR);
  }

  string file_read;
  string temp;

  prog_func.myPktCall = "";
  prog_func.mySelCall = "";

  while (getline(filein, file_read)) {

    if (!file_read.empty() && file_read[0] != '#') { // valid line to check
      // now check for other comment markers
      string::size_type pos = file_read.find_first_of('#');
      if (pos != string::npos) file_read.resize(pos); // truncate
	
      // look for "MY_CALL:"
      if (get_prog_parm("MY_CALL:", file_read, prog_func.myCall)) {
	transform(prog_func.myCall.begin(), prog_func.myCall.end(),
		  prog_func.myCall.begin(), std::toupper);
	if (prog_func.myPktCall.empty()) { // only overwrite myPktCall if no PKT_CALL: yet specified
	  prog_func.myPktCall = prog_func.myCall;
	}
      }
	    
      // look for "PKT_CALL:"
      else if (get_prog_parm("PKT_CALL:", file_read, prog_func.myPktCall)) {
	transform(prog_func.myPktCall.begin(), prog_func.myPktCall.end(),
		  prog_func.myPktCall.begin(), std::toupper);
      }
	    
      // look for "SELCALL:"
      else if (get_prog_parm("SELCALL:", file_read, prog_func.mySelCall)) {
	transform(prog_func.mySelCall.begin(), prog_func.mySelCall.end(),
		  prog_func.mySelCall.begin(), std::toupper);
      }

      // look for "SERIALPORT:"
      else if (get_prog_parm("SERIALPORT:", file_read, port));
      
      // look for "FILEDIR:"
      else if (get_prog_parm("FILEDIR:", file_read, prog_func.filedir));

      // look for "SPEED:"
      else if (get_prog_parm("SPEED:", file_read, temp)) {
	int speed_val = atoi(temp.c_str());
	if (speed_val == 1200 || speed_val == 2400 || speed_val == 4800 || speed_val == 9600) {
	  speed = speed_val;
	}
      }
    }
  }
}

#ifndef HAVE_ATTACH
#include "misc/fdstream.cpp" // we need to compile this with this file
#endif
