/* Copyright (C) 1999 - 2001 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <strstream>
#include <iostream>
#include <string>
#include <cstring>
#include <unistd.h>
#include <cctype>
#include <cstdlib>
#include <iomanip>
#include <algorithm>

#include "event_slots.h"
#include "download.h"
#include "buffers.h"
#include "filesend.h"
#include "modeless_dialogs.h"
#include "char_funcs.h"

extern Prog_func prog_func;

// PRINT_MARK is the mark which will be printed to screen when a print mark is set
#define PRINT_MARK 167

/*
The run() method is not called on the modal dialogs objects
constructed in this file.  Instead the dialogs are created on the heap
and will delete themselves automatically when they are close (there is
no memory leak).

I have chosen to do it this way, because when the event loop is
appropriated with run() it does not take the timer event servicing
MainScreen::timer_event_handler() with it.
*/

void EventSlots::about_kamplus(void) {
  const char message[] = "kamplus-gtk-1.1.2b\n"
                         "Copyright (C) 1999 - 2002 Chris Vine, G3XXF\n"
                         "chris@cvine.freeserve.co.uk\n"
                         "G3XXF@GB7MSW.#33.GBR.EU\n\n"
                         "This program is released under the\n"
                         "GNU General Public License, version 2\n";
  InfoDialog* dialog_p = new InfoDialog(message, "About kamplus-gtk", standard_size, InfoDialog::information, *mainscreen_p);
  if (!dialog_p) {
    cerr << "Memory allocation error in EventSlots::about_kamplus()" << endl;
    exit(MEM_ERROR);
  }
}


void EventSlots::make_connection_prompt(void) {
  //check preconditions
  if (connect_script_flag == running
      || tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port]
      == Tnc_func::connected
      || (tnc_p->tnc_func.active_port
	  && (tnc_p->tnc_func.hfmode == Tnc_func::rtty
	      || tnc_p->tnc_func.hfmode == Tnc_func::tor
	      || tnc_p->tnc_func.hfmode == Tnc_func::ascii
	      || tnc_p->tnc_func.hfmode == Tnc_func::cw))) {
    beep();
    return;
  }
  
// proceed
  ConnectDialog* connect_dialog_p = new ConnectDialog(tnc_p, standard_size, *mainscreen_p);
  if (!connect_dialog_p) {
    cerr << "Memory allocation error in EventSlots::make_connection_prompt()" << endl;
    exit(MEM_ERROR);
  }
  connect_dialog_p->accepted.connect(SigC::slot(this, &EventSlots::make_connection));
}


void EventSlots::make_connection(string connect_call) {
  //check preconditions
  if (connect_script_flag == running
      || tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port]
      == Tnc_func::connected
      || (tnc_p->tnc_func.active_port
	  && (tnc_p->tnc_func.hfmode == Tnc_func::rtty
	      || tnc_p->tnc_func.hfmode == Tnc_func::tor
	      || tnc_p->tnc_func.hfmode == Tnc_func::ascii
	      || tnc_p->tnc_func.hfmode == Tnc_func::cw))) {
    beep();
    return;
  }
  
// proceed
  if (connect_call.size()) {
    if (!tnc_p->tnc_func.active_port
	|| tnc_p->tnc_func.hfmode == Tnc_func::packet) { // check to see if we need to run a connect script

      string filename(prog_func.filedir);
      filename += "/";
      filename += connect_call;

      if (!access(filename.c_str(), F_OK)) run_connect_script(filename.c_str());
    }
    
    if (connect_script_flag == not_running) {
      if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) {
	tnc_p->send_specialcommand(packetCMD);
	usleep(200000);
      }
      
      string commandmessage;
      
      if (!tnc_p->tnc_func.active_port || tnc_p->tnc_func.hfmode == Tnc_func::packet) {
	commandmessage = packet_connectCMD;
      }
      
      else if (tnc_p->tnc_func.hfmode == Tnc_func::amtor ||
	       tnc_p->tnc_func.hfmode == Tnc_func::lamtor || 
	       tnc_p->tnc_func.hfmode == Tnc_func::fec) {
	commandmessage = amtor_connectCMD;
	tnc_p->tnc_func.hfmode = Tnc_func::amtor;
	mainscreen_p->display_mode();
	mainscreen_p->set_abort_button();
      }
      
      else if (tnc_p->tnc_func.hfmode == Tnc_func::gtor ||
	       tnc_p->tnc_func.hfmode == Tnc_func::gmon) {
	commandmessage = gtor_connectCMD;
	tnc_p->tnc_func.hfmode = Tnc_func::gtor;
	mainscreen_p->display_mode();
      }
      
      else if (tnc_p->tnc_func.hfmode == Tnc_func::pactor) {
	commandmessage = pactor_connectCMD;
      }
      
      commandmessage += connect_call;
      tnc_p->send_kamcommand(commandmessage.c_str());
      
      if (!tnc_p->tnc_func.active_port || (tnc_p->tnc_func.hfmode != Tnc_func::amtor
					   && tnc_p->tnc_func.hfmode != Tnc_func::lamtor
					   && tnc_p->tnc_func.hfmode != Tnc_func::fec)) {

	if (connect_call[0] == '!') connect_call.erase(0, 1);
	string::size_type end_pos = connect_call.find(' ');
	if (end_pos != string::npos) connect_call.erase(end_pos);

	tnc_p->tnc_func.hisCall[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port] = connect_call;
	if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) {
	  tnc_p->tnc_func.hisCall_lock = Tnc_func::on;
	  if (tnc_p->is_validcall(connect_call)) tnc_p->make_selcall(connect_call, tnc_p->tnc_func.selCall);
	  mainscreen_p->set_call_lock_button();
	}
	mainscreen_p->display_callsign();
      }
      else if (tnc_p->tnc_func.active_port) tnc_p->tnc_func.selCall = connect_call;
    }
  }
}

void EventSlots::vhf_menu(int stream) {
  // check preconditions
  if (connect_script_flag == running 
      || send_parms_flag == running
      || tnc_p->get_active_freebytes() <= tr_buffer.letters_used() + 3) {
    beep();
    return;
  }

  if (tnc_p->tnc_func.active_port || tnc_p->tnc_func.active_stream() != stream) {
    if(tnc_p->get_active_freebytes() > tr_buffer.letters_used() + 3) {
      receivewin_p->scrollout(true);
      tnc_p->tnc_func.active_port = 0;
      tnc_p->tnc_func.active_vhf_stream = stream;
      receivewin_p->raise(stream, 0);
      receivewin_p->set_textselected_items();
      receivewin_p->scrollout(true);

      mainscreen_p->display_current_stream();
      mainscreen_p->display_callsign();
      mainscreen_p->display_connected_status();
      mainscreen_p->make_torline();
      mainscreen_p->display_freebytes();
      mainscreen_p->set_call_lock_button();
      mainscreen_p->set_connect_button();
      mainscreen_p->set_disconnect_button();
      mainscreen_p->set_auto_cq_button();
      mainscreen_p->set_speed_lock_button();
      mainscreen_p->set_ident_button();
      mainscreen_p->set_sync_button();
      mainscreen_p->set_abort_button();
      mainscreen_p->set_rx_button();
      mainscreen_p->set_tx_button();
      mainscreen_p->set_change_speed_button();
      mainscreen_p->update_file_load_items();
      mainscreen_p->update_print_mark_items();
      
      // notify the Kam LEDs of the change
      tnc_p->send_kamcommand("");
    }
    else beep();
  }
}

void EventSlots::hf_menu(int stream) {
  // check preconditions
  if (connect_script_flag == running 
      || send_parms_flag == running
      || tnc_p->get_active_freebytes() <= tr_buffer.letters_used() + 3) {
    beep();
    return;
  }
  
  if (!tnc_p->tnc_func.active_port
      || (tnc_p->tnc_func.active_stream() != stream && tnc_p->tnc_func.hfmode == Tnc_func::packet)) {
    if(tnc_p->get_active_freebytes() > tr_buffer.letters_used() + 3) {
      receivewin_p->scrollout(true);
      tnc_p->tnc_func.active_port = 1;
      if (tnc_p->tnc_func.hfmode == Tnc_func::packet) tnc_p->tnc_func.active_hf_stream = stream;
      else tnc_p->tnc_func.active_hf_stream = 0;
      receivewin_p->raise(stream, 1);
      receivewin_p->set_textselected_items();
      receivewin_p->scrollout(true);

      mainscreen_p->display_current_stream();
      mainscreen_p->display_callsign();
      mainscreen_p->display_connected_status();
      mainscreen_p->make_torline();
      mainscreen_p->display_freebytes();
      mainscreen_p->update_lockinfo();
      mainscreen_p->update_torinfo();
      mainscreen_p->update_txinfo();
      mainscreen_p->set_call_lock_button();
      mainscreen_p->set_connect_button();
      mainscreen_p->set_disconnect_button();
      mainscreen_p->set_auto_cq_button();
      mainscreen_p->set_speed_lock_button();
      mainscreen_p->set_ident_button();
      mainscreen_p->set_sync_button();
      mainscreen_p->set_abort_button();
      mainscreen_p->set_rx_button();
      mainscreen_p->set_tx_button();
      mainscreen_p->set_change_speed_button();
      mainscreen_p->update_file_load_items();
      mainscreen_p->update_print_mark_items();
      
      // notify the Kam LEDs of the change
      tnc_p->send_kamcommand("");
    }
    else beep();
  }
}

void EventSlots::run_connect_script(const char* script) {
// script can have a value of 0 (null pointer) if no file needs to be opened
// because run_connect_script_flag == running
// script has a default value of 0
  
  if (script && connect_script_flag == not_running) {
    tnc_p->tnc_func.got_disconnect_flag = false;
#ifdef HAVE_IOS_NOCREATE
    script_file.open(script, ios::in | ios::nocreate);
#else
    // we must have Std C++ so we probably don't need a ios::nocreate
    // flag on a read open to ensure uniqueness
    script_file.open(script, ios::in);
#endif
    if (!script_file) {
      ostrstream strm;
      strm << script << " cannot be opened" << ends;
      char* message = strm.str();
      InfoDialog* dialog_p = new InfoDialog(message, "Connect Script", standard_size, InfoDialog::warning, *mainscreen_p);
      if (!dialog_p) {
	cerr << "Memory allocation error in EventSlots::run_connect_script()" << endl;
	exit(MEM_ERROR);
      }
      delete[] message;
      script_file.clear();
    }
    else {
      char buffer[257];
      bool firstline_flag = false;
      int count;
      while (!firstline_flag && (script_file.getline(buffer, 256), !script_file.eof() && !script_file.bad())) {
	if ((count = script_file.gcount()) > 1 && *buffer != '#') {  // # at start of line is a comment
	  buffer[--count] = 0;  // null terminate the string
	  firstline_flag = true;
	}
	if (!script_file) script_file.clear(); // clear failbit - the line was longer than 256
      }
      if (!firstline_flag) {
	ostrstream strm;
	strm << script << " is empty - check the script" << ends;
	char* message = strm.str();
	InfoDialog* dialog_p = new InfoDialog(message, "Connect Script", standard_size, InfoDialog::warning, *mainscreen_p);
	if (!dialog_p) {
	  cerr << "Memory allocation error in EventSlots::run_connect_script()" << endl;
	  exit(MEM_ERROR);
	}
	delete[] message;
	script_file.clear();
	script_file.close();
      }
      
      else { 
	char* commandmessage = new char[strlen(packet_connectCMD) + strlen(buffer) + 1];
	if (!commandmessage) {
	  cerr << "Memory allocation error in Keyboard_controller::run_connect_script()" << endl;
	  exit(MEM_ERROR);
	}
	strcpy(commandmessage, packet_connectCMD);
	strcat(commandmessage, buffer);
	tnc_p->send_kamcommand(commandmessage);
	delete[] commandmessage;
	connect_script_flag = running;
	connect_script_timeout_base = time(0);
      }
    }
  }
  
  // if we never made the initial connect 
  // cancel the attempt to run the script
  else if (connect_script_flag == running
	   && tnc_p->tnc_func.got_disconnect_flag
	   && tnc_p->tnc_func.disconnect_stream.port == tnc_p->tnc_func.active_port
	   && tnc_p->tnc_func.disconnect_stream.stream == tnc_p->tnc_func.active_stream()) {
    connect_script_flag = not_running;
    script_file.clear();
    script_file.close();
  }
  
  else if (connect_script_flag == running
	   && tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()]
	   [tnc_p->tnc_func.active_port] ==  Tnc_func::connected) {
    char buffer[256];
    char* buffer_ptr;
    int count;
    if ((tnc_p->get_packet_freebytes() > 256)
	&& (script_file.getline(buffer, 256), !script_file.eof() && !script_file.bad())) {
      if ((count = script_file.gcount()) > 1 && *buffer != '#') {  // # at start of line is a comment
	buffer_ptr = buffer;
	int index;
	count--;
	for (index = 0; index < count; index++, buffer_ptr++) {
	  tr_buffer.add_letter(*buffer_ptr);
	}
	buffer[index] = 0;  // null terminate string
	sendwin_p->write(buffer);
	tr_buffer.add_letter('\n');
	sendwin_p->new_line();
	usleep(100000);
      }
      if (!script_file) script_file.clear(); // clear failbit - the line was longer than 256
    }
    if (!script_file) {              // finished sending script
      script_file.clear();
      script_file.close();
      connect_script_flag = not_running;
    }
  }
  
  // if we never made the initial connect and 60 seconds have elapsed
  // cancel the attempt to run the script
  else if (connect_script_flag == running
	   && time(0) - connect_script_timeout_base > 60) {
    connect_script_flag = not_running;
    script_file.clear();
    script_file.close();
    tnc_p->send_kamcommand(packet_disconnectCMD);
    receivewin_p->write("Connect script timed out after 60 seconds");
    receivewin_p->newline();
  }
}

void EventSlots::send_parms_prompt(const char* file) {
//check preconditions
  if (connect_script_flag == running) {
    beep();
    return;
  }

  PromptDialog* dialog_p = new PromptDialog("Send Kam set-up parameters?", "Send Parms", standard_size, *mainscreen_p);
  if (!dialog_p) {
    cerr << "Memory allocation error in EventSlots::send_parms_prompt()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->accepted.connect(SigC::bind<const char*>(SigC::slot(this, &EventSlots::send_parms), file));
}

void EventSlots::send_parms(const char*  file) {
// file can have a value of 0 (null pointer) if no file needs to be opened
// because send_parms_flag == running

//check preconditions
  if (connect_script_flag == running) {
    beep();
    return;
  }

  // proceed        
  if (send_parms_flag == not_running) {
    if (file) {
      // test to see whether file contains path information
      char* result = strchr(file, '/');
      
      ostrstream s1;
      if (!result) s1 << prog_func.filedir << "/parms/";
      s1 << file << ends;
      char* filename = s1.str();
      
#ifdef HAVE_IOS_NOCREATE
      parm_file.open(filename, ios::in | ios::nocreate);
#else
      // we must have Std C++ so we probably don't need a ios::nocreate
      // flag on a read open to ensure uniqueness
      parm_file.open(filename, ios::in);
#endif
      if (!parm_file) {
	ostrstream s2;
	s2 << "File " << filename << "\ncannot be found/opened"
	   << "\nCheck that FILEDIR: in kamrc is correct" << ends;
	char* message = s2.str();
	InfoDialog* dialog_p = new InfoDialog(message, "Send Parameters", standard_size, InfoDialog::warning, *mainscreen_p);
	if (!dialog_p) {
	  cerr << "Memory allocation error in EventSlots::send_parms()" << endl;
	  exit(MEM_ERROR);
	}
	delete[] message;
	parm_file.clear();
      }
      else {
	send_parms_flag = running;
	if (!strcmp(file, "tnc.parms")) { // if we are sending tnc.parms, set up myCall and mySelCall also
          string commandmessage("MYCALL ");
	  commandmessage += prog_func.myCall;
	  tnc_p->send_kamcommand(commandmessage.c_str());
	  usleep(200000);
	  commandmessage = "MYSELCAL ";
	  commandmessage += prog_func.mySelCall;
	  tnc_p->send_kamcommand(commandmessage.c_str());
	}
	delete[] filename;
      }
    }
    else {
      receivewin_p->write("\nError in  EventSlots::send_parms()");
      receivewin_p->write("\nNo file open or file parameter passed()");
      receivewin_p->write("\nPlease report to address in BUGS file\n");
    }
  }
  if (send_parms_flag == running) {
    char buffer[258];
    int count = 0;
    
// now keep going in a while loop until we get a line which isn't empty
// and isn't a comment(# at start of line is a comment)
    while((parm_file.getline(buffer, 256), !parm_file.eof() && !parm_file.bad())
	  && ((count = parm_file.gcount()) < 2  || *buffer == '#')) {}
    if (!parm_file.eof() && !parm_file.bad()) { // we have a command
      buffer[--count] = 0;   // null terminate the string
      char* buffer_ptr = buffer; // get rid of any trailing comment
      while(*buffer_ptr != 0 && *buffer_ptr != '#') buffer_ptr++;
      if (*buffer_ptr == '#') {
	*buffer_ptr = 0;
      }
      tnc_p->send_kamcommand(buffer);
      usleep(200000);
      if (!parm_file) parm_file.clear(); // clear failbit - the line was longer than 256
    }
    else {  // we have sent all the commands in tnc.parms
      parm_file.clear();
      parm_file.close();
      send_parms_flag = not_running;
    }
  }
}

void EventSlots::send_message_preprocess(int message_number, bool prompt_flag) {
  //check preconditions
  if (connect_script_flag == running) {
    beep();
    return;
  }
// now proceed

  string filename(prog_func.filedir);
  filename += "/messages";

  bool message_found = false;
  
#ifdef HAVE_IOS_NOCREATE
  message_file.open(filename.c_str(), ios::in | ios::nocreate);
#else
  // we must have Std C++ so we probably don't need a ios::nocreate
  // flag on a read open to ensure uniqueness
  message_file.open(filename.c_str(), ios::in);
#endif
  if(!message_file) {
    ostrstream strm;
    strm << filename << " cannot be found/opened" << ends;
    char* message = strm.str();
    InfoDialog* dialog_p = new InfoDialog(message, "Send Message", standard_size, InfoDialog::warning, *mainscreen_p);
    if (!dialog_p) {
      cerr << "Memory allocation error in EventSlots::send_message_preprocess()" << endl;
      exit(MEM_ERROR);
    }
    delete[] message;
    message_file.clear();
  }
  else {
    char buffer[MESSAGE_SIZE + 1];
    while(!message_found
	  && (message_file.getline(buffer, MESSAGE_SIZE), !message_file.eof() && !message_file.bad())) {
      if (*buffer == '#' && *(buffer + 1) == (char)message_number) {
	message_found = true;
	if (prompt_flag && !prog_func.noprompt_flag) {
	  ostrstream strm;
	  strm << "Send message";
	  // get any title
	  if (message_file.gcount() > 4) {
	    char* temp_p;
	    for (temp_p = buffer + message_file.gcount() - 1; *temp_p == ' '; temp_p--); // eliminate trailing white space
	    *(temp_p + 1) = 0; // null terminate the string
	    for (temp_p = buffer + 2; *temp_p == ' '; temp_p++); // eliminate any leading white space
	    if (strlen(temp_p) > 8) *(temp_p + 8) = 0; // make sure the title is no more than 8 letters long
	    strm << ": " << temp_p;
	  }
	  strm << '?' << ends;
	  char* message = strm.str();
	  PromptDialog* dialog_p = new PromptDialog(message, "Send Message", standard_size, *mainscreen_p);
	  if (!dialog_p) {
	    cerr << "Memory allocation error in EventSlots::send_message()" << endl;
	    exit(MEM_ERROR);
	  }
	  dialog_p->accepted.connect(SigC::bind(SigC::slot(this, &EventSlots::send_message), message_number, true));
	  dialog_p->rejected.connect(SigC::bind(SigC::slot(this, &EventSlots::send_message), message_number, false));

	  delete[] message;
	}
	else send_message(message_number, true);
      }
      if (!message_file) message_file.clear(); // clear failbit - the line was longer than MESSAGE_SIZE
    }
  }
  if (!message_found) {
    message_file.clear();
    message_file.close();
    beep();
  }
}

void EventSlots::send_message(int message_number, bool send) {
  //check preconditions
  if (connect_script_flag == running) {
    beep();
  }
  
  else if (send) {
    bool got_line = false;
    char buffer[MESSAGE_SIZE + 1];
    int extracted = 0;
    // message_file has been opened and read to the message header line: now get next line
    if ((message_file.getline(buffer, MESSAGE_SIZE), !message_file.eof() && !message_file.bad())
	&& (extracted = message_file.gcount()) > 0)  {
      buffer[extracted - 1] = 0;  // null terminate line extracted
      got_line = true;
    }

    if (got_line) {
      if ((char)message_number == '0' || (char)message_number > '5') {  // send a file
	FilesendBuffer* filesend_ptr = new FilesendBuffer(FileBuffer::text,
				tnc_p->tnc_func.active_stream(), tnc_p->tnc_func.active_port, tnc_p,
				mainscreen_p, sendwin_p, receivewin_p, standard_size);
	if (!filesend_ptr) {
	  cerr << "Memory allocation error in Keyboard_controller::send_message()" << endl;
	  exit(MEM_ERROR);
	}
	if (filesend_ptr->open_file(buffer, true)) {
	  wordwrap_buffer_add('\n');
	  flush_wordwrap_buffer(); // follow the lf by a flush in case not enough space for the lf
	  // so it didn't flush by itself
	  buffer_list.add(filesend_ptr);
	}
	else delete filesend_ptr;
      }
      else {  // send the message in ./messages
	int count = 0;
	int index;
	char letter;
	int result = 0;

	do {
	  if (!message_file) message_file.clear(); // clear failbit - the line was longer than MESSAGE_SIZE
	  if (count) {  // terminate any former line with '\n'
	    wordwrap_buffer_add('\n');
	  }
	  buffer[--extracted] = 0;
	  for (index = 0; (letter = buffer[index]) != 0
		 && count < MESSAGE_SIZE && result != -1; index++, count++) {
	    
	    if (tnc_p->tnc_func.active_port && (uchar)letter > 31) {
	      if (tnc_p->tnc_func.hfmode == Tnc_func::rtty
		  || tnc_p->tnc_func.hfmode == Tnc_func::fec
		  || tnc_p->tnc_func.hfmode == Tnc_func::amtor
		  || (tnc_p->tnc_func.hfmode == Tnc_func::gtor
		      && tnc_p->tnc_func.stream_status[0][1] == Tnc_func::disconnected)
		  || (tnc_p->tnc_func.hfmode == Tnc_func::tor
		      && (tnc_p->tnc_func.stream_status[0][1] == Tnc_func::disconnected
			  || tnc_p->tnc_func.tor_connected_mode == Tnc_func::not_pactor_gtor))) {
		if (!is_baudot(letter)) letter = 0;
		else if (tnc_p->tnc_func.hfmode == Tnc_func::rtty) letter = toupper(static_cast<unsigned char> (letter));
	      }
	      else if (tnc_p->tnc_func.hfmode == Tnc_func::cw) {
		if (!is_morse(letter) && letter != '$') letter = 0; // allow the $ token through
		else letter = toupper(static_cast<unsigned char> (letter));
	      }
	    }

#if CHAR_SET==LATIN_1
	    if ((uchar)letter > 31 || letter == '\n') {
#elif CHAR_SET==CP437
	    if (((uchar) letter < 256 && (uchar)letter > 31) || letter == '\n') {
#else
	    if ((letter < 127 && letter > 31) || letter == '\n') {
#endif
	      if (letter == '$') {
		int second_letter = buffer[index + 1];
		if (second_letter == 'a') {
		  for (const char* letter_ptr = prog_func.myCall.c_str();
		       result != -1 && *letter_ptr; letter_ptr++) {
		    result = wordwrap_buffer_add(*letter_ptr);
		  }
		  index++; // get rid of the 'a' code
		}
		else if (second_letter == 'b') {
		  for (const char* letter_ptr = prog_func.mySelCall.c_str();
		       result != -1 && *letter_ptr; letter_ptr++) {
		    result = wordwrap_buffer_add(*letter_ptr);
		  }
		  index++; // get rid of the 'b' code
		}
		else if (second_letter == 'c') {
		  for (const char* letter_ptr = tnc_p->tnc_func.hisCall[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port].c_str();
		       result != -1 && *letter_ptr; letter_ptr++) {
		    result = wordwrap_buffer_add(*letter_ptr);
		  }
		  index++; // get rid of the 'c' code
		}
		else if (second_letter == 'd') {
		  time_t time_val;
		  time(&time_val);
		  tm* time_ptr = gmtime(&time_val);
		  ostrstream s1;
		  s1 << setfill('0') << setw(2) << time_ptr->tm_hour
		     << ":" << setw(2) << time_ptr->tm_min << ends;
		  char* time_string = s1.str();
		  for (char* letter_ptr = time_string;
		       result != -1 && *letter_ptr; letter_ptr++) {
		    result = wordwrap_buffer_add(*letter_ptr);
		  }
		  delete[] time_string;
		  index++; // get rid of the 'd' code
		}
		else if (second_letter == 'e') {
		  time_t time_val;
		  time(&time_val);
		  tm* time_ptr = gmtime(&time_val);
		  ostrstream s1;
		  char month[4];
		  get_month(month, time_ptr->tm_mon);
		  s1 << time_ptr->tm_mday << "-" << month  << "-"
		     << setfill('0') << setw(2) << (time_ptr->tm_year)%100 << ends;
		  char* date_string = s1.str();
		  for (char* letter_ptr = date_string;
		       result != -1 && *letter_ptr; letter_ptr++) {
		    result = wordwrap_buffer_add(*letter_ptr);
		  }
		  delete[] date_string;
		  index++; // get rid of the 'e' code
		}
		else if (second_letter == 'f') {
		  for (char* letter_ptr = prog_func.rst;
		       result != -1 && *letter_ptr; letter_ptr++) {
		    result = wordwrap_buffer_add(*letter_ptr);
		  }
		  index++; // get rid of the 'f' code
		}
		else if (second_letter == 'g') {
		  ostrstream s1;
		  s1 << setfill('0') << setw(3) << prog_func.qso_count << ends;
		  char* qso_count_string = s1.str();
		  for (char* letter_ptr = qso_count_string;
		       result != -1 && *letter_ptr; letter_ptr++) {
		    result = wordwrap_buffer_add(*letter_ptr);
		  }
		  delete[] qso_count_string;
		  index++; // get rid of the 'g' code
		}
		else if (second_letter == 0) {  // last letter of line is a '$'
		  result = wordwrap_buffer_add('$');
		}
	      }
	      else {
		result = wordwrap_buffer_add(letter);
	      }
	    }
	  }
        } while (result != -1 && count < MESSAGE_SIZE
		 && (message_file.getline(buffer, MESSAGE_SIZE), !message_file.eof() && !message_file.bad())
		 && *buffer != '#' && *buffer != 0 && (extracted = message_file.gcount()) > 0);
      }
    }
  }
  message_file.clear();
  message_file.close();
}

void EventSlots::change_hfmode_prompt(int mode) {
  // check preconditions
  if (connect_script_flag == running 
      || prog_func.sending_autocq
      || tnc_p->tnc_func.hfmode == mode) {
    if (mode == Tnc_func::cw          // if in cw, rtty or ascii modes, we cannot be running
	|| mode == Tnc_func::rtty     // a connect script or sending an autocq, so we must be
	|| mode == Tnc_func::ascii) { // in the mode concerned already - so change speed
      change_speed();
    }
    return;
  }

// now proceed

  ostrstream message_strm;

  message_strm << "Change to ";

  switch(mode) {
  case(Tnc_func::packet):
    message_strm << "Packet";
    break;
  case(Tnc_func::pactor):
    message_strm << "Pactor";
    break;
  case(Tnc_func::rtty):
    message_strm << "RTTY";
    break;
  case(Tnc_func::ascii):
    message_strm << "ASCII";
    break;
  case(Tnc_func::amtor):
    message_strm << "Amtor";
    break;
  case(Tnc_func::lamtor):
    message_strm << "Lamtor";
    break;
  case(Tnc_func::fec):
    message_strm << "Fec";
    break;
  case(Tnc_func::gtor):
    message_strm << "Gtor";
    break;
  case(Tnc_func::gmon):
    message_strm << "Gmon";
    break;
  case(Tnc_func::tor):
    message_strm << "Tor";
    break;
  case(Tnc_func::cw):
    message_strm << "CW";
    break;
  }

  message_strm << "?" << ends;
  char* message = message_strm.str();
  
  PromptDialog* dialog_p = new PromptDialog(message, "Change Mode", standard_size, *mainscreen_p);
  if (!dialog_p) {
    cerr << "Memory allocation error in EventSlots::change_hfmode_prompt()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->accepted.connect(SigC::bind(SigC::slot(this, &EventSlots::change_hfmode), mode));
  delete[] message;
}

void EventSlots::change_hfmode(int mode) {
  // check preconditions
  if (connect_script_flag == running 
      || prog_func.sending_autocq
      || tnc_p->tnc_func.hfmode == mode) {
    if (mode == Tnc_func::cw          // if in cw, rtty or ascii modes, we cannot be running
	|| mode == Tnc_func::rtty     // a connect script or sending an autocq, so we must be
	|| mode == Tnc_func::ascii) { // in the mode concerned already - so change speed
      change_speed();
    }
    return;
  }

// now proceed

  ostrstream command_strm;

  switch(mode) {
  case(Tnc_func::packet):
    command_strm << ends;
    break;
  case(Tnc_func::pactor):
    command_strm << pactorCMD << ends;
    break;
  case(Tnc_func::rtty):
    command_strm << rttyCMD << ends;
    break;
  case(Tnc_func::ascii):
    command_strm << asciiCMD << ends;
    break;
  case(Tnc_func::amtor):
    command_strm << amtorCMD << ends;
    break;
  case(Tnc_func::lamtor):
    command_strm << lamtorCMD << ends;
    break;
  case(Tnc_func::fec):
    command_strm << fecCMD << ends;
    break;
  case(Tnc_func::gtor):
    command_strm << gtorCMD << ends;
    break;
  case(Tnc_func::gmon):
    command_strm << gmonCMD << ends;
    break;
  case(Tnc_func::tor):
    command_strm << torCMD << ends;
    break;
  case(Tnc_func::cw):
    command_strm << cwCMD << ends;
    break;
  }

  char* command = command_strm.str();
  
  tr_buffer.reset();
  if (!tnc_p->tnc_func.active_port || tnc_p->tnc_func.active_hf_stream != 0) {
    tnc_p->tnc_func.active_port = 1;
    tnc_p->tnc_func.active_hf_stream = 0;
    receivewin_p->raise(0, 1);
  }
  receivewin_p->scrollout(true);
  
  tnc_p->send_specialcommand(packetCMD);
  
  if (tnc_p->tnc_func.capture_stream.port   // if file capture was set for Packet and changing to Tor,
      && tnc_p->tnc_func.capturefile_flag   // or set for Tor and changing to Packet, cancel file capture
      && (tnc_p->tnc_func.hfmode == Tnc_func::packet // we must be changing from Packet to Tor
	  || mode == Tnc_func::packet)) {     // we must be changing from Tor to packet
    tnc_p->tnc_func.capturefile_flag = false;
    tnc_p->tnc_func.capturefile.clear();
    tnc_p->tnc_func.capturefile.close();
    mainscreen_p->display_capture_status();
  }

  if (mode != Tnc_func::packet) {
    usleep(200000);
    tnc_p->send_kamcommand(command);
  }
  
  tnc_p->tnc_func.hfmode = (Tnc_func::Hfmode)mode;
  mainscreen_p->display_mode();
  
  tnc_p->tnc_func.speed_lock = Tnc_func::off;
  tnc_p->tnc_func.have_sent_lock = Tnc_func::no;
  tnc_p->tnc_func.sending_cw_flag = false;
  mainscreen_p->display_current_stream();
  mainscreen_p->display_callsign();
  mainscreen_p->display_connected_status();
  mainscreen_p->make_torline();
  mainscreen_p->display_freebytes();
  mainscreen_p->set_call_lock_button();
  mainscreen_p->set_connect_button();
  mainscreen_p->set_disconnect_button();
  mainscreen_p->set_auto_cq_button();
  mainscreen_p->set_speed_lock_button();
  mainscreen_p->set_ident_button();
  mainscreen_p->set_sync_button();
  mainscreen_p->set_abort_button();
  mainscreen_p->set_rx_button();
  mainscreen_p->set_tx_button();
  mainscreen_p->set_change_speed_button();
  mainscreen_p->set_capture_button();
  mainscreen_p->update_file_load_items();
  
  if (mode != Tnc_func::packet) {
    mainscreen_p->update_lockinfo();
    mainscreen_p->update_torinfo();
    mainscreen_p->update_txinfo();
    mainscreen_p->show_tormenu_streams();
    reset_speeds(); // it doesn't really matter if we do an unecessary reset in Pactor/Gtor/Amtor modes
  }
  else mainscreen_p->show_packetmenu_streams();
  // notify the Kam LEDs of the change
  tnc_p->send_kamcommand("");

  delete[] command;
}

void EventSlots::capture(void) {
// check pre-conditions
  if (mainscreen_p->setting_capture_button()) return;

// proceed
  if (tnc_p->tnc_func.capturefile_flag) {
    tnc_p->tnc_func.capturefile_flag = false;
    tnc_p->tnc_func.capturefile.clear();
    tnc_p->tnc_func.capturefile.close();
  }

  else {
    tnc_p->tnc_func.capturefile_flag = true;
    tnc_p->tnc_func.capture_stream.port = tnc_p->tnc_func.active_port;
    tnc_p->tnc_func.capture_stream.stream = tnc_p->tnc_func.active_stream();

    string filename(prog_func.filedir);
    filename += "/capture.txt";
    
    tnc_p->tnc_func.capturefile.open(filename.c_str(), ios::out | ios::app);
    if (!tnc_p->tnc_func.capturefile) {
      InfoDialog* dialog_p = new InfoDialog("File capture.txt could not be opened", "Capture",
					    standard_size, InfoDialog::warning, *mainscreen_p);
      if (!dialog_p) {
	cerr << "Memory allocation error in EventSlots::capture()" << endl;
	exit(MEM_ERROR);
      }
      tnc_p->tnc_func.capturefile_flag = false;
    }
  }
  mainscreen_p->display_capture_status();
  mainscreen_p->set_capture_button();
}

void EventSlots::lock_call(void) {
// check preconditions
  if (mainscreen_p->setting_call_lock_button()
      || !tnc_p->tnc_func.active_port
      || tnc_p->tnc_func.hfmode == Tnc_func::packet) {
    return;
  }

// proceed
  
  if (tnc_p->tnc_func.hisCall_lock == Tnc_func::off) {
    tnc_p->tnc_func.hisCall_lock = Tnc_func::on;
  }
  else tnc_p->tnc_func.hisCall_lock = Tnc_func::off;
  mainscreen_p->display_callsign();
  mainscreen_p->set_call_lock_button();
}

void EventSlots::disconnect_prompt(void) {

// check preconditions
  if (tnc_p->tnc_func.active_port
      && tnc_p->tnc_func.hfmode != Tnc_func::packet
	&& tnc_p->tnc_func.hfmode != Tnc_func::amtor
      && tnc_p->tnc_func.hfmode != Tnc_func::gtor
      && tnc_p->tnc_func.hfmode != Tnc_func::pactor
      && tnc_p->tnc_func.hfmode != Tnc_func::tor) {
    return;
  }
  PromptDialog* dialog_p = new PromptDialog("Disconnect?", "Disconnect", standard_size, *mainscreen_p);
  if (!dialog_p) {
    cerr << "Memory allocation error in EventSlots::disconnect_prompt()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->accepted.connect(SigC::slot(this, &EventSlots::disconnect));
}


void EventSlots::disconnect(void) {

// check preconditions
  if (tnc_p->tnc_func.active_port
      && tnc_p->tnc_func.hfmode != Tnc_func::packet
      && tnc_p->tnc_func.hfmode != Tnc_func::amtor
      && tnc_p->tnc_func.hfmode != Tnc_func::gtor
      && tnc_p->tnc_func.hfmode != Tnc_func::pactor
      && tnc_p->tnc_func.hfmode != Tnc_func::tor) {
    return;
  }
  
// proceed
  tr_buffer.reset();
  if (!tnc_p->tnc_func.active_port || tnc_p->tnc_func.hfmode == Tnc_func::packet) {
    tnc_p->send_kamcommand(packet_disconnectCMD);
  }
  else tnc_p->send_specialcommand(tor_disconnectCMD);
}

void EventSlots::ident(void) {
// check preconditions
  if (!tnc_p->tnc_func.active_port
      || tnc_p->tnc_func.hfmode == Tnc_func::packet
      || connect_script_flag == running) {
    
    beep();
    return;
  }

// proceed
  string message(tnc_p->tnc_func.hisCall[0][1]);
  if (tnc_p->tnc_func.hfmode == Tnc_func::rtty
      || tnc_p->tnc_func.hfmode == Tnc_func::cw) {
    message += " DE ";
  }
  else message += " de ";
  message += prog_func.myCall;
  message += " ";

  string::size_type index;
  int result;
  for (index = 0, result = 0; result != -1 && index < message.size(); index++) {
    result = wordwrap_buffer_add(message[index]);
  }
}

void EventSlots::sync(void) {
//check preconditions
  if (!tnc_p->tnc_func.active_port
      || (tnc_p->tnc_func.hfmode != Tnc_func::cw
	  && tnc_p->tnc_func.hfmode != Tnc_func::lamtor
	  && tnc_p->tnc_func.hfmode != Tnc_func::fec
	  && !(tnc_p->tnc_func.hfmode == Tnc_func::amtor
	       && tnc_p->tnc_func.stream_status[0][1] == Tnc_func::disconnected))) {
    return;
  }

// proceed
  if (tnc_p->tnc_func.hfmode == Tnc_func::cw) {
    tnc_p->send_specialcommand('L');
    usleep(200000);
    tnc_p->send_specialcommand('U');
  }
  else {
    tnc_p->send_specialcommand(immediate_rxCMD);
    usleep(100000);
  }
}

void EventSlots::abort_prompt(void) {
// check preconditions
  if (!tnc_p->tnc_func.active_port
      || (tnc_p->tnc_func.hfmode != Tnc_func::amtor
	  && tnc_p->tnc_func.hfmode != Tnc_func::pactor
	  && tnc_p->tnc_func.hfmode != Tnc_func::gtor
	  && tnc_p->tnc_func.hfmode != Tnc_func::tor)) {
    return;
  }
  
// proceed
  PromptDialog* dialog_p = new PromptDialog("Abort?", "Abort", standard_size, *mainscreen_p);
  if (!dialog_p) {
    cerr << "Memory allocation error in EventSlots::abort_prompt()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->accepted.connect(SigC::slot(this, &EventSlots::abort));
}

void EventSlots::abort(void) {
// check preconditions
  if (!tnc_p->tnc_func.active_port
      || (tnc_p->tnc_func.hfmode != Tnc_func::amtor
	  && tnc_p->tnc_func.hfmode != Tnc_func::pactor
	  && tnc_p->tnc_func.hfmode != Tnc_func::gtor
	  && tnc_p->tnc_func.hfmode != Tnc_func::tor)) {
    return;
  }
  
  tr_buffer.reset();
  tnc_p->send_specialcommand(aborttxCMD);
}

void EventSlots::enter_call_prompt(void) {
  CallsignDialog* dialog_p = new CallsignDialog(tnc_p, standard_size, *mainscreen_p);
  if (!dialog_p) {
    cerr << "Memory allocation error in EventSlots::enter_call_prompt()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->accepted.connect(SigC::slot(this, &EventSlots::enter_call));
}

void EventSlots::enter_call(string callsign) {
  if (!callsign.empty()) {
    tnc_p->tnc_func.hisCall[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port] = callsign;
    if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) {
      tnc_p->tnc_func.hisCall_lock = Tnc_func::on;
      if (tnc_p->is_validcall(callsign)) tnc_p->make_selcall(callsign, tnc_p->tnc_func.selCall);
      mainscreen_p->set_call_lock_button();
    }
    mainscreen_p->display_callsign();
  }
}

void EventSlots::command_prompt(void) {
  CommandDialog* dialog_p = new CommandDialog(standard_size, *mainscreen_p);
  if (!dialog_p) {
    cerr << "Memory allocation error in EventSlots::command_prompt()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->send_command.connect(SigC::slot(this, &EventSlots::command));
  dialog_p->send_script.connect(SigC::slot(this, &EventSlots::send_parms));
}

void EventSlots::command(string command) {
  if (command.size()) tnc_p->send_kamcommand(command.c_str());
}

void EventSlots::changeover_rx(void) {
// check preconditions
  if (!tnc_p->tnc_func.active_port || tnc_p->tnc_func.hfmode == Tnc_func::packet) {
    return;
  }
  
// proceed
  if (tr_buffer.letters_free() > 5) {
    if (tnc_p->tnc_func.stream_status[0][1] == Tnc_func::connected) {
      if (tnc_p->tnc_func.tor_connected_mode == Tnc_func::pactor_gtor) {
	wordwrap_buffer_add('=');
	wordwrap_buffer_add('>');
      }
      else if (tnc_p->tnc_func.hfmode == Tnc_func::amtor
	       || (tnc_p->tnc_func.hfmode == Tnc_func::tor
		   && tnc_p->tnc_func.tor_connected_mode == Tnc_func::not_pactor_gtor)) {
	sendwin_p->write("+?");
      }
    }
    flush_wordwrap_buffer();
    if (tnc_p->tnc_func.hfmode == Tnc_func::cw) {
      tnc_p->send_specialcommand(immediate_rxCMD);
      tnc_p->tnc_func.sending_cw_flag = false;
      usleep(100000);
    }
    else {
      tr_buffer.add_letter(CMDinbuffer);
#ifdef BUFFERED_CO
      tr_buffer.add_letter(rxCMD);
#else
      tr_buffer.add_letter(immediate_rxCMD);
#endif
      tr_buffer.add_letter(CMDinbuffer);
    }
  }
  else beep();
}


void EventSlots::changeover_tx(void) {
// check preconditions
  if (!tnc_p->tnc_func.active_port || tnc_p->tnc_func.hfmode == Tnc_func::packet) {
    return;
  }
  
// proceed
  tnc_p->send_specialcommand(txCMD);
  if (tnc_p->tnc_func.hfmode == Tnc_func::cw) tnc_p->tnc_func.sending_cw_flag = true;
  usleep(100000);
}

int EventSlots::wordwrap_buffer_add(uchar letter) {
  int return_val = letter;
  char text[2] = {0};
  if (letter == ' ') {
    if (tr_buffer.letters_free() > wordwrap_buffer_count + 2) { // leave space for ' ' and possible '\n'
      flush_wordwrap_buffer(); // this will cause space_flag to be set false
      if (line_lettercount < WORDWRAP) {
	*text = ' ';
	sendwin_p->write(text);
	line_lettercount++;
	space_flag = true; // reset space_flag
      }
      else {
	tr_buffer.add_letter('\n');
	sendwin_p->new_line();
	line_lettercount = 0;
      }
    }
    else {
      beep();
      return_val = -1;
    }
  }
  
  else if (letter == '\n') {
    if (tr_buffer.letters_free() > wordwrap_buffer_count + 2) { // leave space for '\n' and possible ' '
      flush_wordwrap_buffer();
      tr_buffer.add_letter('\n');
      sendwin_p->new_line();
      line_lettercount = 0;
    }
    else {
      beep();
      return_val = -1;
    }
  }
  
  else if (letter == 8) {
    if (line_lettercount) {
      if (wordwrap_buffer_count) {
	wordwrap_buffer_count--;
	line_lettercount--;
	sendwin_p->backspace();        // now erase letter from screen
      }
      else if (space_flag) {
	space_flag = false;            // we cancel the space reserved to go before the word in wordwrap_buffer
	line_lettercount--;
	sendwin_p->backspace();        // now erase space from screen
      }
      else if (tr_buffer.view_letter() != 8) {  // we don't want to delete a delete!
	int result = tr_buffer.erase_letter();
	if (!result) {
	  beep();
	}
	else {
	  if (result == -1) {  // if last letter already extracted, and we are in pactor or gtor,
	    // we need to send value 8 to the Kam
	    tr_buffer.add_letter(8);
	  }
	  line_lettercount--;
	  sendwin_p->backspace();        // now erase letter from screen
	}
      }
      else { // add another delete to a delete
	tr_buffer.add_letter(8);
	line_lettercount--;
	sendwin_p->backspace();        // now erase letter from screen
      }
    }
    else {
      beep();
      return_val = -1;
    }
  }
  
  else if (line_lettercount > WORDWRAP) {
    if (tr_buffer.letters_free() > wordwrap_buffer_count + 2) { // leave space for a '\n'
      if (line_lettercount > wordwrap_buffer_count) { // check to see if the whole
	// line is one word!
	sendwin_p->backspace(wordwrap_buffer_count);
	tr_buffer.add_letter('\n');
	sendwin_p->new_line();
	line_lettercount = wordwrap_buffer_count;
	space_flag = false;
	wordwrap_buffer[wordwrap_buffer_count] = 0; // null terminate wordwrap_buffer
	
	sendwin_p->write(wordwrap_buffer);
      }
      else {           // the whole line was one word - break it
	flush_wordwrap_buffer();
	tr_buffer.add_letter('\n');
	sendwin_p->new_line();
	line_lettercount = 0;
      }
      wordwrap_buffer[wordwrap_buffer_count++] = letter;
      *text = (unsigned char)letter;
      sendwin_p->write(text);
      line_lettercount++;
    }
    else {
      beep();
      return_val = -1;
    }
  }
  
  else {
    if (tr_buffer.letters_free() > wordwrap_buffer_count + 2) { // leave space for a possible ' '
      wordwrap_buffer[wordwrap_buffer_count++] = letter;
      *text = (unsigned char)letter;
      sendwin_p->write(text);
      line_lettercount++;
    }
    else {
      beep();
      return_val = -1;
    }
  }
  return return_val;
}

void EventSlots::flush_wordwrap_buffer(void) {
  if (space_flag) tr_buffer.add_letter(' ');
  int index;
  for (index = 0; index < wordwrap_buffer_count; index++) {
    tr_buffer.add_letter(wordwrap_buffer[index]);
  }
  wordwrap_buffer_count = 0;
  space_flag = false;
}

void EventSlots::send_letter(uchar letter) {
// check preconditions
  if (tnc_p
      && (connect_script_flag == EventSlots::running
	  ||  buffer_list.get_upload_status(tnc_p->tnc_func.active_stream(),
					    tnc_p->tnc_func.active_port) == BufferList::file)) {
    return;
  }
  if (tnc_p
      && connect_script_flag == EventSlots::running) {
    return;
  }
  
// proceed
  if (tnc_p && tnc_p->tnc_func.active_port) {
    if (tnc_p->tnc_func.hfmode == Tnc_func::rtty
	|| tnc_p->tnc_func.hfmode == Tnc_func::fec
	|| tnc_p->tnc_func.hfmode == Tnc_func::amtor
	|| (tnc_p->tnc_func.hfmode == Tnc_func::gtor
	    && tnc_p->tnc_func.stream_status[0][1] == Tnc_func::disconnected)
	|| (tnc_p->tnc_func.hfmode == Tnc_func::tor
	    && (tnc_p->tnc_func.stream_status[0][1] == Tnc_func::disconnected
		|| tnc_p->tnc_func.tor_connected_mode == Tnc_func::not_pactor_gtor))) {
      if (!is_baudot((char)letter) && letter != 8) letter = 0;
      else if (tnc_p->tnc_func.hfmode == Tnc_func::rtty) letter = toupper(static_cast<unsigned char> (letter));
    }
    
    else if (tnc_p->tnc_func.hfmode == Tnc_func::cw) {
      if (!is_morse((char)letter) && letter != 8) letter = 0;
      else letter = toupper(static_cast<unsigned char> (letter));
    }
  }
  
#if CHAR_SET==LATIN_1
  if (letter > 31 || letter == '\n' || letter == 8) {
#elif CHAR_SET==CP437
  if ((letter < 256 && letter > 31) || letter == '\n' || letter == 8) {
#else
  if ((letter < 127 && letter > 31) || letter == '\n' || letter == 8) {
#endif
    wordwrap_buffer_add(letter);
  }
}

void EventSlots::lock_speed(void) {
//check preconditions
  if (mainscreen_p->setting_speed_lock_button()
      || !tnc_p->tnc_func.active_port
      || (tnc_p->tnc_func.hfmode != Tnc_func::pactor
	  && tnc_p->tnc_func.hfmode != Tnc_func::gtor
	  && tnc_p->tnc_func.hfmode != Tnc_func::tor)) {
    
    return;
  }

// proceed
   if (tnc_p->tnc_func.speed_lock == Tnc_func::off) {
    tnc_p->tnc_func.speed_lock = Tnc_func::on;
    tnc_p->tnc_func.have_sent_lock = Tnc_func::yes;
    tnc_p->send_specialcommand('1');
  }

  else {
    tnc_p->tnc_func.speed_lock = Tnc_func::off;
    tnc_p->tnc_func.have_sent_lock = Tnc_func::no;
    tnc_p->send_specialcommand('0');
  }
  mainscreen_p->update_lockinfo();
  mainscreen_p->set_speed_lock_button();
}

void EventSlots::rst_prompt(void) {
  RstDialog* rst_dialog_p = new RstDialog(standard_size, *mainscreen_p);
  if (!rst_dialog_p) {
    cerr << "Memory allocation error in EventSlots::rst_prompt()" << endl;
    exit(MEM_ERROR);
  }
  rst_dialog_p->accepted.connect(SigC::slot(this, &EventSlots::rst));
}

void EventSlots::rst(string report) {
  if (report.size()) strncpy(prog_func.rst, report.c_str(), 3);
}

void EventSlots::upload_prompt(int mode) {
  // check preconditions
  if (buffer_list.get_upload_status(tnc_p->tnc_func.active_stream(),
				    tnc_p->tnc_func.active_port) == BufferList::file
      || (mode != FileBuffer::text
	  && (tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()]
	      [tnc_p->tnc_func.active_port] != Tnc_func::connected
	      || (tnc_p->tnc_func.active_port
		  && (tnc_p->tnc_func.hfmode == Tnc_func::amtor
		      || (tnc_p->tnc_func.hfmode == Tnc_func::tor
			  && tnc_p->tnc_func.tor_connected_mode != Tnc_func::pactor_gtor)))))) {
    return;
  }

// proceed
  FileReadSelectDialog* dialog_p = new FileReadSelectDialog(standard_size, *mainscreen_p, prog_func.filedir.c_str());
  if (!dialog_p) {
    cerr << "Memory allocation error in EventSlots::upload_prompt()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->file_to_read.connect(SigC::bind(SigC::slot(this, &EventSlots::upload), mode));
}

void EventSlots::upload(string filename, streampos size, int mode) {
// check preconditions
  if (buffer_list.get_upload_status(tnc_p->tnc_func.active_stream(),
				    tnc_p->tnc_func.active_port) == BufferList::file
      || (mode != FileBuffer::text
	  && (tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()]
	      [tnc_p->tnc_func.active_port] != Tnc_func::connected
	      || (tnc_p->tnc_func.active_port
		  && (tnc_p->tnc_func.hfmode == Tnc_func::amtor
		      || (tnc_p->tnc_func.hfmode == Tnc_func::tor
			  && tnc_p->tnc_func.tor_connected_mode != Tnc_func::pactor_gtor)))))) {
    return;
  }
  
// proceed
  if (size) {
    UploadDialog* dialog_p = new UploadDialog(filename.c_str(), size, tnc_p, buffer_list,
					      receivewin_p, standard_size);
                         // UploadDialog objects are modeless and are deleted when removed
                         // from MainScreen::upload_dialogs in method MainScreen::timerEvent()
    if (!dialog_p) {
      cerr << "Memory allocation error in EventSlots::upload()" << endl;
      exit(MEM_ERROR);
    }
    // the FilesendBuffer object will now take ownership of the UploadDialog object
    FilesendBuffer* filesend_p = new FilesendBuffer((FileBuffer::Buffer_mode)mode,
				    tnc_p->tnc_func.active_stream(), tnc_p->tnc_func.active_port, tnc_p,
				    mainscreen_p, sendwin_p, receivewin_p, standard_size, dialog_p);
    if (!filesend_p) {
      cerr << "Memory allocation error in EventSlots::upload()" << endl;
      exit(MEM_ERROR);
    }
    if (filesend_p->open_file(filename.c_str(), false)) {
      wordwrap_buffer_add('\n');
      flush_wordwrap_buffer(); // follow the lf by a flush in case not enough space for the lf
                               // so it didn't flush by itself
      buffer_list.add(filesend_p);
    }
    else delete filesend_p;  // this will also delete the UploadDialog object
  }
}

void EventSlots::download_prompt(int mode) {

  DownloadList* download_list_p = tnc_p->tnc_func.download_list_ptr;
// check preconditions
  if (download_list_p->get_download_status(tnc_p->tnc_func.active_stream(),
					   tnc_p->tnc_func.active_port) == DownloadList::on
      || tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()]
      [tnc_p->tnc_func.active_port] != Tnc_func::connected
      || (tnc_p->tnc_func.active_port
	  && (tnc_p->tnc_func.hfmode == Tnc_func::amtor
	      || (tnc_p->tnc_func.hfmode == Tnc_func::tor
		  && tnc_p->tnc_func.tor_connected_mode != Tnc_func::pactor_gtor)))) {
    return;
  }

// proceed
  FileSaveSelectDialog* dialog_p = new FileSaveSelectDialog(standard_size, *mainscreen_p, prog_func.filedir.c_str());
  if (!dialog_p) {
    cerr << "Memory allocation error in EventSlots::download_prompt()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->file_to_save.connect(SigC::bind(SigC::slot(this, &EventSlots::download), mode));
}


void EventSlots::download(string filename, int mode) {

  DownloadList* download_list_p = tnc_p->tnc_func.download_list_ptr;
// check preconditions
  if (download_list_p->get_download_status(tnc_p->tnc_func.active_stream(),
					   tnc_p->tnc_func.active_port) == DownloadList::on
      || tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()]
      [tnc_p->tnc_func.active_port] != Tnc_func::connected
      || (tnc_p->tnc_func.active_port
	  && (tnc_p->tnc_func.hfmode == Tnc_func::amtor
	      || (tnc_p->tnc_func.hfmode == Tnc_func::tor
		  && tnc_p->tnc_func.tor_connected_mode != Tnc_func::pactor_gtor)))) {
    return;
  }
  
// proceed
  DownloadDialog* dialog_p = new DownloadDialog(filename.c_str(), tnc_p, mainscreen_p, receivewin_p,
					        standard_size);
  if (!dialog_p) {
    cerr << "Memory allocation error in EventSlots::download()" << endl;
    exit(MEM_ERROR);
  }
  // the DownloadFile object will now take ownership of the DownloadDialog object
  DownloadFile* download_file_p = new DownloadFile((DownloadFile::Download_mode)mode,
				   tnc_p->tnc_func.active_stream(), tnc_p->tnc_func.active_port, dialog_p);
                       // DownloadDialog objects are modeless and are deleted when the DownloadFile
	               // object is deleted in the method DownloadDialog::end_download() or
	               // in Tnc::read_state_message()
  if (!download_file_p) {
    cerr << "Memory allocation error in EventSlots::download()" << endl;
    exit(MEM_ERROR);
  }
  if (download_file_p->open_file(filename.c_str())) {
    download_list_p->add(download_file_p);
    mainscreen_p->display_connected_status();
  }
  else {
    delete download_file_p;  // this will also delete the DownloadDialog object
    ostrstream strm;
    strm << filename.c_str() << " cannot be opened for writing" << ends;
    char* message = strm.str();
    InfoDialog* dialog_p = new InfoDialog(message, "Download File",
					  standard_size, InfoDialog::warning, *mainscreen_p);
    if (!dialog_p) {
      cerr << "Memory allocation error in EventSlots::download()" << endl;
      exit(MEM_ERROR);
    }
    delete[] message;
  }
}

void EventSlots::auto_cq_preprocess(int mode, bool prompt) {
// check preconditions

  if (mainscreen_p->setting_auto_cq_button()
      || !tnc_p->tnc_func.active_port
      || (tnc_p->tnc_func.hfmode != Tnc_func::pactor
	  && tnc_p->tnc_func.hfmode != Tnc_func::amtor
	  && tnc_p->tnc_func.hfmode != Tnc_func::gtor
	  && tnc_p->tnc_func.hfmode != Tnc_func::tor)
      || tnc_p->tnc_func.stream_status[0][1] == Tnc_func::connected
      || (mode == EventSlots::pactor && tnc_p->tnc_func.hfmode != Tnc_func::tor)
      || (!prog_func.sending_autocq
	  && buffer_list.get_upload_status(0, 1) == BufferList::file)) {
    return;
  }

// proceed
  if (prog_func.sending_autocq) {
    if (prompt) {
      PromptDialog* dialog_p = new PromptDialog("Stop sending Auto-CQ?", "Stop Auto-CQ", standard_size, *mainscreen_p);
      if (!dialog_p) {
	cerr << "Memory allocation error in EventSlots::auto_cq_preprocess()" << endl;
	exit(MEM_ERROR);
      }
      dialog_p->accepted.connect(SigC::slot(this, &EventSlots::end_auto_cq));
    }
    else end_auto_cq();
  }
  else {
    if (mode == EventSlots::pactor) prog_func.tor_autocq_mode = Prog_func::pactor;
    else prog_func.tor_autocq_mode = Prog_func::amtor;

    if (mode == EventSlots::choose && tnc_p->tnc_func.hfmode == Tnc_func::tor) {
      Autocq_modeDialog* autocq_dialog_p = new Autocq_modeDialog(standard_size, *mainscreen_p);
      if (!autocq_dialog_p) {
	cerr << "Memory allocation error in EventSlots::auto_cq_preprocess()" << endl;
	exit(MEM_ERROR);
      }
      autocq_dialog_p->accepted.connect(SigC::bind(SigC::slot(this, &EventSlots::start_auto_cq), true));
      autocq_dialog_p->rejected.connect(SigC::bind(SigC::slot(this, &EventSlots::start_auto_cq), false));
    }
    else if (prompt) {
      PromptDialog* dialog_p = new PromptDialog("Send Auto-CQ?", "Auto-CQ", standard_size, *mainscreen_p);
      if (!dialog_p) {
	cerr << "Memory allocation error in EventSlots::auto_cq_preprocess()" << endl;
	exit(MEM_ERROR);
      }
      dialog_p->accepted.connect(SigC::bind(SigC::slot(this, &EventSlots::start_auto_cq), true));
      dialog_p->rejected.connect(SigC::bind(SigC::slot(this, &EventSlots::start_auto_cq), false));
      
    }
    else start_auto_cq(true);
  }
}

void EventSlots::start_auto_cq(bool result) {
  if (result) {
    CqsendBuffer* cqsend_p = new CqsendBuffer(tnc_p, mainscreen_p, sendwin_p,
					      receivewin_p, standard_size);
    if (!cqsend_p) {
      cerr << "Memory allocation error in EventSlots::start_auto_cq()" << endl;
      exit(MEM_ERROR);
    }
    // EventSlots::display_connected_status() and
    // EventSlots::set_auto_cq_button() will be called in CqsendBuffer::load_buffer()
    if (cqsend_p->load_buffer()) buffer_list.add(cqsend_p);
    else delete cqsend_p;
  }
  else mainscreen_p->set_auto_cq_button(); // if the prompt was not accepted, unset button
}

void EventSlots::end_auto_cq(void) {
  buffer_list.reset(DList_enum::bottom_end);
  FileBuffer* file_buffer_p;
  while ((file_buffer_p = (FileBuffer*)buffer_list.inspect(DList_enum::up)) != 0
	 && !(file_buffer_p->get_stream() == 0
	      && file_buffer_p->get_port() == 1)) {}
  if (file_buffer_p) file_buffer_p->end_loading();
  // EventSlots::display_connected_status() will be called in CqsendBuffer::load_buffer(),
  // and MainScreen::set_auto_cq_button() will be called in MainScreen::timer_event_handler()
  // once the buffer is extracted - calling them now will give a false display
  
  else {
    receivewin_p->write("\nOops - can't find the cq file buffer to finish loading\n");
    beep();
    prog_func.sending_autocq = false;
    mainscreen_p->display_connected_status();
    mainscreen_p->set_auto_cq_button();
  }
}


void EventSlots::helpfile(void) {
  HelpDialog* dialog_p = new HelpDialog(standard_size);
  if (!dialog_p) {
    cerr << "Memory allocation error in EventSlots::helpfile()" << endl;
    exit(MEM_ERROR);
  }
}

void EventSlots::ctrl_a(void) {
  if (tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()]
      [tnc_p->tnc_func.active_port] == Tnc_func::connected
      && (!tnc_p->tnc_func.active_port
	  || tnc_p->tnc_func.hfmode == Tnc_func::packet
	  || tnc_p->tnc_func.tor_connected_mode == Tnc_func::pactor_gtor)) {
    flush_wordwrap_buffer();
    tr_buffer.add_letter(1);
    beep();
  }
  else change_hfmode_prompt(Tnc_func::amtor);
}

void EventSlots::ctrl_t(void) {
  if (tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()]
      [tnc_p->tnc_func.active_port] == Tnc_func::connected
      && (!tnc_p->tnc_func.active_port
	  || tnc_p->tnc_func.hfmode == Tnc_func::packet
	  || tnc_p->tnc_func.tor_connected_mode == Tnc_func::pactor_gtor)) {
    flush_wordwrap_buffer();
    tr_buffer.add_letter(20);
  }
#ifndef NO_PACTOR
  else change_hfmode_prompt(Tnc_func::tor);
#endif
}

void EventSlots::ctrl_z(void) {
  if (tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()]
      [tnc_p->tnc_func.active_port] == Tnc_func::connected
      && (!tnc_p->tnc_func.active_port
	  || tnc_p->tnc_func.hfmode == Tnc_func::packet
	  || tnc_p->tnc_func.tor_connected_mode == Tnc_func::pactor_gtor)) {
    flush_wordwrap_buffer();
    tr_buffer.add_letter(26);
  }
}

void EventSlots::change_speed(void) {

  Tnc_func::Hfmode mode = tnc_p->tnc_func.hfmode;
  if (mode == Tnc_func::cw) set_cw_speed_prompt();
  else if (mode == Tnc_func::rtty) set_rtty_speed_prompt();
  else if (mode == Tnc_func::ascii) set_ascii_speed_prompt();
  else return; // we are not in a mode providing for change of speed
}

void EventSlots::set_cw_speed_prompt(void) {

  CwSpeedDialog* cw_speed_dialog_p = new CwSpeedDialog(cw_speed_val, standard_size, *mainscreen_p);
  if (!cw_speed_dialog_p) {
    cerr << "Memory allocation error in EventSlots::cw_speed_prompt()" << endl;
    exit(MEM_ERROR);
  }
  cw_speed_dialog_p->accepted.connect(SigC::slot(this, &EventSlots::set_cw_speed));
}

void EventSlots::set_cw_speed(int speed) {

  cw_speed_val = speed;
  int tnc_val = cw_speed_val/5;
  if (tnc_val && tnc_val < 11) {
    if (tnc_val == 10) tnc_p->send_specialcommand('0');
    else tnc_p->send_specialcommand(tnc_val | 0x30);
    usleep(100000);
  }
  else {
    InfoDialog* dialog_p = new InfoDialog("Invalid speed entered", "Cw Speed",
					  standard_size, InfoDialog::warning, *mainscreen_p);
    if (!dialog_p) {
      cerr << "Memory allocation error in EventSlots::set_cw_speed()" << endl;
      exit(MEM_ERROR);
    }
  }
}

void EventSlots::set_rtty_speed_prompt(void) {
  
  RttySpeedDialog* rtty_speed_dialog_p = new RttySpeedDialog(rtty_speed, standard_size, *mainscreen_p);
  if (!rtty_speed_dialog_p) {
    cerr << "Memory allocation error in EventSlots::rtty_speed_prompt()" << endl;
    exit(MEM_ERROR);
  }
  rtty_speed_dialog_p->accepted.connect(SigC::slot(this, &EventSlots::set_rtty_speed));
}

void EventSlots::set_rtty_speed(RttySpeedDialog::Rtty_speed speed) {
  
  switch(speed) {
  case RttySpeedDialog::b45:
    tnc_p->send_specialcommand('1');
    break;
  case RttySpeedDialog::b50:
    tnc_p->send_specialcommand('2');
    break;
  case RttySpeedDialog::b57:
    tnc_p->send_specialcommand('3');
    break;
  case RttySpeedDialog::b75:
    tnc_p->send_specialcommand('4');
    break;
  }
  rtty_speed = speed;
  usleep(100000);
}

void EventSlots::set_ascii_speed_prompt(void) {
  
  AsciiSpeedDialog* ascii_speed_dialog_p = new AsciiSpeedDialog(ascii_speed, standard_size, *mainscreen_p);
  if (!ascii_speed_dialog_p) {
    cerr << "Memory allocation error in EventSlots::ascii_speed_prompt()" << endl;
    exit(MEM_ERROR);
  }
  ascii_speed_dialog_p->accepted.connect(SigC::slot(this, &EventSlots::set_ascii_speed));
}

void EventSlots::set_ascii_speed(AsciiSpeedDialog::Ascii_speed speed) {
  
  switch(speed) {
  case AsciiSpeedDialog::b50:
    tnc_p->send_specialcommand('2');
    break;
  case AsciiSpeedDialog::b100:
    tnc_p->send_specialcommand('5');
    break;
  case AsciiSpeedDialog::b110:
    tnc_p->send_specialcommand('6');
    break;
  case AsciiSpeedDialog::b200:
    tnc_p->send_specialcommand('8');
    break;
  }
  ascii_speed = speed;
  usleep(100000);
}


void EventSlots::settings_prompt(void) {
  SettingsDialog* settings_dialog_p = new SettingsDialog(standard_size, *mainscreen_p);
  if (!settings_dialog_p) {
    cerr << "Memory allocation error in EventSlots::settings_prompt()" << endl;
    exit(MEM_ERROR);
  }
  settings_dialog_p->accepted.connect(SigC::slot(this, &EventSlots::read_settings));
}

void EventSlots::read_settings(void) {
  string rcfile = prog_func.homedir + "/.";
  rcfile += RC_FILE;

  ifstream filein;
#ifdef HAVE_IOS_NOCREATE
  filein.open(rcfile.c_str(), ios::in | ios::nocreate);
#else
  // we must have Std C++ so we probably don't need a ios::nocreate
  // flag on a read open to ensure uniqueness
  filein.open(rcfile.c_str(), ios::in);
#endif

  if (!filein) {
    string message("Can't open file ");
    message += rcfile;
    cerr << message << endl;
    InfoDialog dialog(message.c_str(), "Config Error",
		      standard_size, InfoDialog::critical, *mainscreen_p);
    dialog.run();
  }
  else {

    bool is_connected = false;
    int port;
    int stream;
    for (port = 0; port < 2 && !is_connected; port++) {
      for (stream = 0; stream < MAXUSERS && !is_connected; stream++) {
	if (tnc_p->tnc_func.stream_status[stream][port] == Tnc_func::connected) is_connected = true;
      }
    }

    string file_read;
    string temp;

    string new_call;
    string new_pktcall;
    string new_selcall;

    bool found_vpaclen = false;
    bool found_hpaclen = false;
    bool found_keepalive_mins = false;
    bool found_print_cmd = false;
    bool found_autocq = false;

    while (getline(filein, file_read)) {
      
      if (!file_read.empty() && file_read[0] != '#') { // valid line to check
	// now check for other comment markers
	string::size_type pos = file_read.find_first_of('#');
	if (pos != string::npos) file_read.resize(pos); // truncate
	
	// look for "MY_CALL:"
	if (get_prog_parm("MY_CALL:", file_read, new_call)) {
	  transform(new_call.begin(), new_call.end(),
		    new_call.begin(), to_upper);
	  if (new_pktcall.empty()) { // only overwrite new_pktcall if no PKT_CALL: yet specified
	    new_pktcall = new_call;
	  }
	}
	    
	// look for "PKT_CALL:"
	else if (get_prog_parm("PKT_CALL:", file_read, new_pktcall)) {
	  transform(new_pktcall.begin(), new_pktcall.end(),
		    new_pktcall.begin(), to_upper);
	}
	    
	// look for "SELCALL:"
	else if (get_prog_parm("SELCALL:", file_read, new_selcall)) {
	  transform(new_selcall.begin(), new_selcall.end(),
		    new_selcall.begin(), to_upper);
	}
	
	// look for "FILEDIR:"
	else if (get_prog_parm("FILEDIR:", file_read, prog_func.filedir));

	// look for "PRINT_CMD:"
	else if (get_prog_parm("PRINT_CMD:", file_read, prog_func.print_cmd)) {
	  found_print_cmd = true;
	}

	// look for "NOPROMPT:"
	else if (get_prog_parm("NOPROMPT:", file_read, temp)) {
	  if (!temp.compare("true") || !temp.compare("TRUE")) {
	    prog_func.noprompt_flag = true;
	  }
	  else prog_func.noprompt_flag = false;
	}

	// look for "AUTOCQ:"
	else if (get_prog_parm("AUTOCQ:", file_read, temp)) {
	  prog_func.autocq_delay = atoi(temp.c_str());
	  if (prog_func.autocq_delay < 0) prog_func.autocq_delay = 0;
	  found_autocq = true;
	}

	// look for "V_PACLEN:"
	else if (get_prog_parm("V_PACLEN:", file_read, temp)) {
	  tnc_p->tnc_func.paclen.vhf_paclen = atoi(temp.c_str());
	  if (tnc_p->tnc_func.paclen.vhf_paclen < 32) tnc_p->tnc_func.paclen.vhf_paclen = 32;
	  if (tnc_p->tnc_func.paclen.vhf_paclen > MAX_FRAMESIZE) tnc_p->tnc_func.paclen.vhf_paclen = MAX_FRAMESIZE;
	  found_vpaclen = true;
	}
	
	// look for "H_PACLEN:"
	else if (get_prog_parm("H_PACLEN:", file_read, temp)) {
	  tnc_p->tnc_func.paclen.hf_paclen = atoi(temp.c_str());
	  if (tnc_p->tnc_func.paclen.hf_paclen < 32) tnc_p->tnc_func.paclen.hf_paclen = 32;
	  if (tnc_p->tnc_func.paclen.hf_paclen > MAX_FRAMESIZE) tnc_p->tnc_func.paclen.hf_paclen = MAX_FRAMESIZE;
	  found_hpaclen = true;
	}

	// look for "NO_CW_RX:"
	else if (get_prog_parm("NO_CW_RX:", file_read, temp)) {
	  if (!temp.compare("true") || !temp.compare("TRUE")) {
	    tnc_p->tnc_func.no_rx_cw_flag = true;
	  }
	  else tnc_p->tnc_func.no_rx_cw_flag = false;
	}

	// look for "BELL:"
	else if (get_prog_parm("BELL:", file_read, temp)) {
	  if (!temp.compare("true") || !temp.compare("TRUE")) {
	    tnc_p->tnc_func.rx_bell_flag = true;
	  }
	  else tnc_p->tnc_func.rx_bell_flag = false;
	}

	// look for "CHAR_SET:"
	else if (get_prog_parm("CHAR_SET:", file_read, temp)) {
	  if (!temp.compare("latin-1") || !temp.compare("LATIN-1")) {
	    prog_func.charset = Prog_func::latin_1;
	  }
	  else prog_func.charset = Prog_func::cp437;
	}

	// look for "KEEPALIVE:"
	else if (get_prog_parm("KEEPALIVE:", file_read, temp)) {
	  if (!temp.compare("true") || !temp.compare("TRUE")) {
	    tnc_p->tnc_func.keep_alive.keep_alive_flag = true;
	  }
	  else tnc_p->tnc_func.keep_alive.keep_alive_flag = false;
	}

	// look for "KEEPALIVE_MINS:"
	else if (get_prog_parm("KEEPALIVE_MINS:", file_read, temp)) {
	  tnc_p->tnc_func.keep_alive.keep_alive_minutes = atoi(temp.c_str());
	  if (tnc_p->tnc_func.keep_alive.keep_alive_minutes < 1) tnc_p->tnc_func.keep_alive.keep_alive_minutes = 10;
	  found_keepalive_mins = true;
	}

	// look for "WIN_FONTSIZE:"
	else if (get_prog_parm("WIN_FONTSIZE:", file_read, temp)) {
	  int win_fontsize = Fontsize::medium;
	  if (!temp.compare("larger") || !temp.compare("LARGER")) {
	    win_fontsize = Fontsize::larger;
	  }
	  else if (!temp.compare("large") || !temp.compare("LARGE")) {
	    win_fontsize = Fontsize::large;
	  }
	  else if (!temp.compare("small") || !temp.compare("SMALL")) {
	    win_fontsize = Fontsize::small;
	  }
	  font_change(win_fontsize);
	}
      }
    }

    if (!found_vpaclen) tnc_p->tnc_func.paclen.vhf_paclen = MAX_FRAMESIZE;
    if (!found_hpaclen) tnc_p->tnc_func.paclen.hf_paclen = MAX_FRAMESIZE;
    if (!found_keepalive_mins) tnc_p->tnc_func.keep_alive.keep_alive_minutes = 10;
    if (!found_print_cmd) prog_func.print_cmd = "lpr";
    if (!found_autocq) prog_func.autocq_delay = 0;

    if (new_selcall.empty()) {
      if (tnc_p->is_validcall(new_call)) tnc_p->make_selcall(new_call, new_selcall);
      else {
	InfoDialog dialog("You have a non-standard callsign, which means your\n"
			  "Amtor Selcall cannot be automatically generated\n"
			  "Please enter it via the Settings dialog",
			  "Config Error", standard_size, InfoDialog::warning, *mainscreen_p);
	dialog.run();
      }
    }
    
    if (new_pktcall.compare(prog_func.myPktCall)
	|| new_call.compare(prog_func.myCall)
	|| (!new_selcall.empty() && new_selcall.compare(prog_func.mySelCall))) {
      
      // we need to update the callsigns/selcall
      if (!is_connected) {
	string commandmessage;
	if (new_pktcall.compare(prog_func.myPktCall)) {
	  prog_func.myPktCall = new_pktcall;
	  commandmessage = "MYCALL ";
	  commandmessage += new_pktcall;
	  tnc_p->send_kamcommand(commandmessage.c_str());
	  usleep(200000);
	}

	if (new_call.compare(prog_func.myCall)) {
	  prog_func.myCall = new_call;
#ifndef NO_PACTOR
  #ifndef NO_GTOR
	  commandmessage = "MYGTCALL ";
	  commandmessage += new_call;
	  tnc_p->send_kamcommand(commandmessage.c_str());
	  usleep(200000);
  #endif
	  commandmessage = "MYPTCALL ";
	  commandmessage += new_call;
	  tnc_p->send_kamcommand(commandmessage.c_str());
	  usleep(200000);
#endif
	}

	if (!new_selcall.empty() && new_selcall.compare(prog_func.mySelCall)) {
	  prog_func.mySelCall = new_selcall;
	  commandmessage = "MYSELCAL ";
	  commandmessage += new_selcall;
	  tnc_p->send_kamcommand(commandmessage.c_str());
	  
	  usleep(100000);
	}
      }
      else {
	InfoDialog dialog("Cannot change callsign or selcall when connected.\n"
			  "Please end all connections, and then press the OK\n"
			  "button in the Settings dialog again",
			  "Config Error", standard_size, InfoDialog::warning, *mainscreen_p);
	dialog.run();
      }
    }
  }
}

bool EventSlots::get_prog_parm(const char* name, string& line, string& result) {
// this function looks for a setting named `name' in the string `line'
// and returns the values stated after it in string `result'
// it returns `true' if the setting was found 
// if there are trailing spaces or tabs, string `line' will be modified
// string `result' is only modified if the `name' setting is found

  const string::size_type length = strlen(name);
  // we have to use std::string::substr() because libstdc++-2
  // doesn't support the Std-C++ std::string::compare() functions
  if (!line.substr(0, length).compare(name)) {
    // erase any trailing space or tab
    while (line.find_last_of(" \t") == line.size() - 1) line.resize(line.size() - 1);
    if (line.size() > length) {
      // ignore any preceding space or tab from the setting value given
      string::size_type pos = line.find_first_not_of(" \t", length); // pos now is set to beginning of setting value
      if (pos != string::npos) result.assign(line, pos, line.size() - pos);
    }
    return true;
  }
  return false;
}

void EventSlots::set_print_mark_prompt(void) {

  PrintList* print_list_p = tnc_p->tnc_func.print_list_ptr;
// check pre-conditions
  if (print_list_p->get_print_status(tnc_p->tnc_func.active_stream(),
				     tnc_p->tnc_func.active_port) ==  PrintList::on) {
    return;
  }

// proceed
  PromptDialog* dialog_p = new PromptDialog("Set print mark?", "Set Print Mark", standard_size, *mainscreen_p);
  if (!dialog_p) {
    cerr << "Memory allocation error in EventSlots::set_print_mark_prompt()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->accepted.connect(SigC::slot(this, &EventSlots::set_print_mark));
}

void EventSlots::set_print_mark(void) {

  PrintList* print_list_p = tnc_p->tnc_func.print_list_ptr;
// check pre-conditions
  if (print_list_p->get_print_status(tnc_p->tnc_func.active_stream(),
				     tnc_p->tnc_func.active_port) ==  PrintList::on) {
    return;
  }

// proceed
  PrintFile* print_file_p = new PrintFile(tnc_p->tnc_func.active_stream(), tnc_p->tnc_func.active_port);
  if (!print_file_p) {
    cerr << "Memory allocation error in EventSlots::set_print_mark()" << endl;
    exit(MEM_ERROR);
  }
  if (print_file_p->start_store()) {
    print_list_p->add(print_file_p);
    unsigned char mark_string[2] = {PRINT_MARK, 0};
    receivewin_p->write((char*)mark_string);
    mainscreen_p->update_print_mark_items();
  }
  else {
    delete print_file_p;
    const char message[] = "Temporary print file cannot be opened for writing\n"
                           "Please check permissions in /tmp directory";
    InfoDialog* dialog_p = new InfoDialog(message, "Print File",
					  standard_size, InfoDialog::warning, *mainscreen_p);
    if (!dialog_p) {
      cerr << "Memory allocation error in EventSlots::set_print_mark()" << endl;
      exit(MEM_ERROR);
    }
  }
}

void EventSlots::print_from_mark_prompt(void) {

  PrintList* print_list_p = tnc_p->tnc_func.print_list_ptr;
// check pre-conditions
  if (print_list_p->get_print_status(tnc_p->tnc_func.active_stream(),
				     tnc_p->tnc_func.active_port) ==  PrintList::off) {
    return;
  }

// proceed
  PrintMarkDialog* dialog_p = new PrintMarkDialog(standard_size, *mainscreen_p);
  if (!dialog_p) {
    cerr << "Memory allocation error in EventSlots::print_from_mark_prompt()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->out_selection.connect(SigC::slot(this, &EventSlots::print_from_mark));
}

void EventSlots::print_from_mark(PrintMarkDialog::Result result) {

  PrintList* print_list_p = tnc_p->tnc_func.print_list_ptr;
// check pre-conditions
  if (print_list_p->get_print_status(tnc_p->tnc_func.active_stream(),
				     tnc_p->tnc_func.active_port) ==  PrintList::off) {
    return;
  }

// proceed
  print_list_p->reset(DList_enum::bottom_end);
  PrintFile* print_file_p;
  while ((print_file_p = (PrintFile*)print_list_p->inspect(DList_enum::up)) != 0
	 && !(print_file_p->get_stream() == tnc_p->tnc_func.active_stream()
	      && print_file_p->get_port() == tnc_p->tnc_func.active_port)) {}
  if (print_file_p) {
    print_list_p->extract();
    if (result == PrintMarkDialog::accepted) print_file_p->print();
    else print_file_p->cancel();
    delete print_file_p;
    mainscreen_p->update_print_mark_items();
  }
  else {
    receivewin_p->write("\nOops - can't find the print file object to delete\n"
			"Please report bug\n");
    beep();
  }
}

void EventSlots::context_print_mark(void) {
  if (tnc_p->tnc_func.print_list_ptr->get_print_status(tnc_p->tnc_func.active_stream(),
						       tnc_p->tnc_func.active_port) ==  PrintList::off) {
    set_print_mark_prompt();
  }
  else print_from_mark_prompt();
}


void EventSlots::increment_qso_count(void) {
  prog_func.qso_count++;
  if (prog_func.qso_count == 1) {
    QsoCountDialog* dialog_p = new QsoCountDialog(standard_size);
    if (!dialog_p) {
      cerr << "Memory allocation error in EventSlots::increment_qso_count()" << endl;
      exit(MEM_ERROR);
    }
    show_qso_count.connect(SigC::slot(dialog_p, &QsoCountDialog::show_qso_count));
    dialog_p->inc.connect(SigC::slot(this, &EventSlots::increment_qso_count));
    dialog_p->dec.connect(SigC::slot(this, &EventSlots::decrement_qso_count));
    dialog_p->quit_button_pressed.connect(SigC::slot(this, &EventSlots::reset_qso_count));
    show_qso_count(1);
    mainscreen_p->update_qso_counter_items();
  }
  else if (prog_func.qso_count > 0) show_qso_count(prog_func.qso_count);
  else {
    prog_func.qso_count--;
    beep();
  }
}

void EventSlots::decrement_qso_count(void) {
  if (prog_func.qso_count > 1) {
    prog_func.qso_count--;
    show_qso_count(prog_func.qso_count);
  }
  else beep();
}

void EventSlots::reset_qso_count(void) {
  prog_func.qso_count = 0;
  mainscreen_p->update_qso_counter_items();
}


void EventSlots::read_cluster(void) {
//check preconditions
  if (mainscreen_p->setting_dx_cluster_button()) return;

// proceed

  if (!tnc_p->tnc_func.read_cluster_flag) {
    if (tnc_p->tnc_func.stream_status[MAXUSERS - 1][0] == Tnc_func::connected) {
      ostrstream strm;
      strm << "Connection already on Vhf stream " << char(MAXUSERS - 1 + 'A') << ends;
      char* message = strm.str();
      InfoDialog* dialog_p = new InfoDialog(message, "Read Dx Cluster",
					    standard_size, InfoDialog::warning, *mainscreen_p);
      if (!dialog_p) {
	cerr << "Memory allocation error in EventSlots::read_cluster()" << endl;
	exit(MEM_ERROR);
      }
      delete[] message;
      mainscreen_p->set_dx_cluster_button();
    }
    else {
      int success_flag = true;
      if (tnc_p->tnc_func.active_port || tnc_p->tnc_func.active_vhf_stream != MAXUSERS - 1) {
	if (tnc_p->get_active_freebytes() > tr_buffer.letters_used() + 3) {
	  receivewin_p->scrollout(true);
	  tnc_p->tnc_func.active_port = 0;
	  tnc_p->tnc_func.active_vhf_stream = MAXUSERS - 1;
	  receivewin_p->raise(MAXUSERS - 1, 0);
	  receivewin_p->set_textselected_items();
	  receivewin_p->scrollout(true);
	  
	  mainscreen_p->display_current_stream();
	  mainscreen_p->display_callsign();
	  mainscreen_p->make_torline();
	  mainscreen_p->display_freebytes();
	  mainscreen_p->set_call_lock_button();
	  mainscreen_p->set_connect_button();
	  mainscreen_p->set_disconnect_button();
	  mainscreen_p->set_auto_cq_button();
	  mainscreen_p->set_speed_lock_button();
	  mainscreen_p->set_ident_button();
	  mainscreen_p->set_sync_button();
	  mainscreen_p->set_abort_button();
	  mainscreen_p->set_rx_button();
	  mainscreen_p->set_tx_button();
	  mainscreen_p->set_change_speed_button();
	  mainscreen_p->update_file_load_items();
	  mainscreen_p->update_print_mark_items();
	  
	  // notify the Kam LEDs of the change
	  tnc_p->send_kamcommand("");
	}
	else {
	  beep();
	  success_flag = false;
	}
      }
      if (success_flag) {
	tnc_p->tnc_func.read_cluster_flag = true;
	mainscreen_p->display_connected_status();
	mainscreen_p->set_dx_cluster_button();
      }
    }
  }
  else {
    tnc_p->reset_dx_spot_list();
    tnc_p->tnc_func.read_cluster_flag = false;
    mainscreen_p->display_connected_status();
    mainscreen_p->set_dx_cluster_button();
  }
}

