/* Copyright (C) 2001, 2002 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <unistd.h>

#include <iostream>
#include <fstream>
#include <strstream>
#include <vector>
#include <algorithm>
#include <iterator>
#include <cstdlib>
#include <cctype>

#include <gtk--/main.h>
#include <gtk--/style.h>
#include <gtk--/pixmap.h>

#include "settings.h"
#include "settings_icons.h"
#include "modal_dialogs.h"
#include "char_funcs.h"

IdentityTable::IdentityTable(const int standard_size): Gtk::Table(5, 3, false), mycall_label("Callsign: "),
				           mypacketcall_label("Packet Callsign: "), myselcall_label("Selcall: ") {

  mycall_label.set_justify(GTK_JUSTIFY_RIGHT);
  mypacketcall_label.set_justify(GTK_JUSTIFY_RIGHT);
  myselcall_label.set_justify(GTK_JUSTIFY_RIGHT);
  mycall_entry.set_usize(standard_size * 6, (standard_size * 5)/6);
  mypacketcall_entry.set_usize(standard_size * 6, (standard_size * 5)/6);
  myselcall_entry.set_usize(standard_size * 6, (standard_size * 5)/6);

  Gtk::Pixmap* mycall_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  mycall_help_button.add(*mycall_helpIcon_p);
  Gtk::Pixmap* mypacketcall_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  mypacketcall_help_button.add(*mypacketcall_helpIcon_p);
  Gtk::Pixmap* myselcall_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  myselcall_help_button.add(*myselcall_helpIcon_p);

  attach(filler1, 0, 2, 0, 1, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);
  attach(mycall_label, 0, 1, 1, 2, 0,
         0, standard_size/3, standard_size/2);
  attach(mycall_entry, 1, 2, 1, 2, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/2);
  attach(mycall_help_button, 2, 3, 1, 2, 0,
         0, standard_size/3, standard_size/2);
  attach(mypacketcall_label, 0, 1, 2, 3, 0,
         0, standard_size/3, standard_size/2);
  attach(mypacketcall_entry, 1, 2, 2, 3, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/2);
  attach(mypacketcall_help_button, 2, 3, 2, 3, 0,
         0, standard_size/3, standard_size/2);
  attach(myselcall_label, 0, 1, 3, 4, 0,
         0, standard_size/3, standard_size/2);
  attach(myselcall_entry, 1, 2, 3, 4, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/2);
  attach(myselcall_help_button, 2, 3, 3, 4, 0,
         0, standard_size/3, standard_size/2);
  attach(filler2, 0, 2, 4, 5, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);

  mycall_help_button.clicked.connect(SigC::bind(SigC::slot(this, &IdentityTable::show_help),
					       IdentityMessages::mycall));
  mypacketcall_help_button.clicked.connect(SigC::bind(SigC::slot(this, &IdentityTable::show_help),
					       IdentityMessages::mypacketcall));
  myselcall_help_button.clicked.connect(SigC::bind(SigC::slot(this, &IdentityTable::show_help),
					       IdentityMessages::myselcall));

  tooltips.set_tip(mycall_help_button, help_messages.get_message(IdentityMessages::mycall), 0);
  tooltips.set_tip(mypacketcall_help_button, help_messages.get_message(IdentityMessages::mypacketcall), 0);
  tooltips.set_tip(myselcall_help_button, help_messages.get_message(IdentityMessages::myselcall), 0);

  tooltips.set_delay(50);

  set_border_width(standard_size/3);
}

void IdentityTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

void IdentityTable::clear(void) {
  mycall_entry.set_text("");
  mypacketcall_entry.set_text("");
  myselcall_entry.set_text("");
}

ComTable::ComTable(const int standard_size): Gtk::Table(5, 3, false), port_label("Port device: "),
						  device_message("(Don't include '/dev/' part of device name)"),
						  b1200_button("1200 baud"), b2400_button("2400 baud"),
						  b4800_button("4800 baud"), b9600_button("9600 baud"),
						  speed_table(2, 2, true), speed_frame("Port speed") {

  port_label.set_justify(GTK_JUSTIFY_RIGHT);
  port_entry.set_usize(standard_size * 6, (standard_size * 5)/6);

  Gtk::Pixmap* port_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  port_help_button.add(*port_helpIcon_p);
  Gtk::Pixmap* speed_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  speed_help_button.add(*speed_helpIcon_p);

  b2400_button.set_group(b1200_button.group());
  b4800_button.set_group(b1200_button.group());
  b9600_button.set_group(b1200_button.group());
  b9600_button.set_active(true);

  speed_table.attach(b1200_button, 0, 1, 0, 1, 0, 0, standard_size/4, 0);
  speed_table.attach(b2400_button, 1, 2, 0, 1, 0, 0, standard_size/4, 0);
  speed_table.attach(b4800_button, 0, 1, 1, 2, 0, 0, standard_size/4, 0);
  speed_table.attach(b9600_button, 1, 2, 1, 2, 0, 0, standard_size/4, 0);
  speed_table.set_border_width(standard_size/2);

  speed_frame.add(speed_table);

  attach(filler1, 0, 2, 0, 1, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);
  attach(port_label, 0, 1, 1, 2, 0,
         0, standard_size/3, 0);
  attach(port_entry, 1, 2, 1, 2, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, 0);
  attach(port_help_button, 2, 3, 1, 2, 0,
         0, standard_size/3, 0);
  attach(device_message, 1, 2, 2, 3, 0,
         0, standard_size/3, standard_size/4);
  attach(speed_frame, 0, 2, 3, 4, 0,
         0, standard_size/3, standard_size/2);
  attach(speed_help_button, 2, 3, 3, 4, 0,
         0, standard_size/3, standard_size/2);
  attach(filler2, 0, 2, 4, 5, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);

  port_help_button.clicked.connect(SigC::bind(SigC::slot(this, &ComTable::show_help),
					       ComMessages::port));
  speed_help_button.clicked.connect(SigC::bind(SigC::slot(this, &ComTable::show_help),
					       ComMessages::speed));

  tooltips.set_tip(port_help_button, help_messages.get_message(ComMessages::port), 0);
  tooltips.set_tip(speed_help_button, help_messages.get_message(ComMessages::speed), 0);

  tooltips.set_delay(50);

  set_border_width(standard_size/3);
}

void ComTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

string ComTable::get_speed(void) const {
  string return_val;
  if (b1200_button.get_active()) return_val = "1200";
  else if (b2400_button.get_active()) return_val = "2400";
  else if (b4800_button.get_active()) return_val = "4800";
  else if (b9600_button.get_active()) return_val = "9600";
  return return_val;
}

void ComTable::set_speed(const int speed) {

  switch (speed) {
  case 1200:
    b1200_button.set_active(true);
    break;
  case 2400:
    b2400_button.set_active(true);
    break;
  case 4800:
    b4800_button.set_active(true);
    break;
  case 9600:
    b9600_button.set_active(true);
    break;
  }
}

void ComTable::clear(void) {
  port_entry.set_text("");
  b9600_button.set_active(true);
}

PacketTable::PacketTable(const int standard_size): Gtk::Table(6, 3, false), vhf_packet_size_label("Vhf packet size: "),
			  hf_packet_size_label("Hf packet size: "), keepalive_mins_label("Keep-alive interval: "),
			  check_button("Send keep-alive packets"){

  vhf_packet_size_label.set_justify(GTK_JUSTIFY_RIGHT);
  hf_packet_size_label.set_justify(GTK_JUSTIFY_RIGHT);
  keepalive_mins_label.set_justify(GTK_JUSTIFY_RIGHT);
  vhf_packet_size_entry.set_usize(standard_size * 6, (standard_size * 5)/6);
  hf_packet_size_entry.set_usize(standard_size * 6, (standard_size * 5)/6);
  keepalive_mins_entry.set_usize(standard_size * 6, (standard_size * 5)/6);

  Gtk::Pixmap* vhf_packet_size_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  vhf_packet_size_help_button.add(*vhf_packet_size_helpIcon_p);
  Gtk::Pixmap* hf_packet_size_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  hf_packet_size_help_button.add(*hf_packet_size_helpIcon_p);
  Gtk::Pixmap* keepalive_mins_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  keepalive_mins_help_button.add(*keepalive_mins_helpIcon_p);

  check_box.pack_start(check_button, false, false, 0);
  check_box.pack_start(filler3, true, true, 0);

  attach(filler1, 0, 2, 0, 1, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);
  attach(vhf_packet_size_label, 0, 1, 1, 2, 0,
         0, standard_size/3, standard_size/3);
  attach(vhf_packet_size_entry, 1, 2, 1, 2, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/3);
  attach(vhf_packet_size_help_button, 2, 3, 1, 2, 0,
         0, standard_size/3, standard_size/3);
  attach(hf_packet_size_label, 0, 1, 2, 3, 0,
         0, standard_size/3, standard_size/3);
  attach(hf_packet_size_entry, 1, 2, 2, 3, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/3);
  attach(hf_packet_size_help_button, 2, 3, 2, 3, 0,
         0, standard_size/3, standard_size/3);
  attach(keepalive_mins_label, 0, 1, 3, 4, 0,
         0, standard_size/3, standard_size/3);
  attach(keepalive_mins_entry, 1, 2, 3, 4, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/3);
  attach(keepalive_mins_help_button, 2, 3, 3, 4, 0,
         0, standard_size/3, standard_size/3);
  attach(check_box, 1, 2, 4, 5, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, 0);
  attach(filler2, 0, 2, 5, 6, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);

  vhf_packet_size_help_button.clicked.connect(SigC::bind(SigC::slot(this, &PacketTable::show_help),
					       PacketMessages::vhf_packet_size));
  hf_packet_size_help_button.clicked.connect(SigC::bind(SigC::slot(this, &PacketTable::show_help),
					       PacketMessages::hf_packet_size));
  keepalive_mins_help_button.clicked.connect(SigC::bind(SigC::slot(this, &PacketTable::show_help),
					       PacketMessages::keepalive_mins));

  tooltips.set_tip(vhf_packet_size_help_button, help_messages.get_message(PacketMessages::vhf_packet_size), 0);
  tooltips.set_tip(hf_packet_size_help_button, help_messages.get_message(PacketMessages::hf_packet_size), 0);
  tooltips.set_tip(keepalive_mins_help_button, help_messages.get_message(PacketMessages::keepalive_mins), 0);

  tooltips.set_delay(50);

  set_border_width(standard_size/3);
}

void PacketTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

void PacketTable::clear(void) {
  vhf_packet_size_entry.set_text("");
  hf_packet_size_entry.set_text("");
  keepalive_mins_entry.set_text("");
  check_button.set_active(false);
}

InterfaceTable::InterfaceTable(const int standard_size): Gtk::Table(5, 2, false), receive_cw_label("Display received CW"),
			    sound_bell_label("Sound bell character"), no_prompt_label("No prompt when sending messages") {

  receive_cw_label.set_justify(GTK_JUSTIFY_LEFT);
  sound_bell_label.set_justify(GTK_JUSTIFY_LEFT);
  no_prompt_label.set_justify(GTK_JUSTIFY_LEFT);

  receive_cw_box.pack_start(receive_cw_button, false, false, 0);
  receive_cw_box.pack_start(receive_cw_label, false, false, 0);
  sound_bell_box.pack_start(sound_bell_button, false, false, 0);
  sound_bell_box.pack_start(sound_bell_label, false, false, 0);
  no_prompt_box.pack_start(no_prompt_button, false, false, 0);
  no_prompt_box.pack_start(no_prompt_label, false, false, 0);

  Gtk::Pixmap* receive_cw_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  receive_cw_help_button.add(*receive_cw_helpIcon_p);
  Gtk::Pixmap* sound_bell_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  sound_bell_help_button.add(*sound_bell_helpIcon_p);
  Gtk::Pixmap* no_prompt_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  no_prompt_help_button.add(*no_prompt_helpIcon_p);

  attach(filler1, 0, 2, 0, 1, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);
  attach(receive_cw_box, 0, 1, 1, 2, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/3);
  attach(receive_cw_help_button, 1, 2, 1, 2, 0,
         0, standard_size/3, standard_size/3);
  attach(sound_bell_box, 0, 1, 2, 3, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/3);
  attach(sound_bell_help_button, 1, 2, 2, 3, 0,
         0, standard_size/3, standard_size/3);
  attach(no_prompt_box, 0, 1, 3, 4, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/3);
  attach(no_prompt_help_button, 1, 2, 3, 4, 0,
         0, standard_size/3, standard_size/3);
  attach(filler2, 0, 2, 4, 5, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);

  receive_cw_help_button.clicked.connect(SigC::bind(SigC::slot(this, &InterfaceTable::show_help),
					       InterfaceMessages::receive_cw));
  sound_bell_help_button.clicked.connect(SigC::bind(SigC::slot(this, &InterfaceTable::show_help),
					       InterfaceMessages::sound_bell));
  no_prompt_help_button.clicked.connect(SigC::bind(SigC::slot(this, &InterfaceTable::show_help),
					       InterfaceMessages::no_prompt));

  tooltips.set_tip(receive_cw_help_button, help_messages.get_message(InterfaceMessages::receive_cw), 0);
  tooltips.set_tip(sound_bell_help_button, help_messages.get_message(InterfaceMessages::sound_bell), 0);
  tooltips.set_tip(no_prompt_help_button, help_messages.get_message(InterfaceMessages::no_prompt), 0);

  tooltips.set_delay(50);

  set_border_width(standard_size/3);
}

void InterfaceTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

void InterfaceTable::clear(void) {
  receive_cw_button.set_active(true);
  sound_bell_button.set_active(true);
  no_prompt_button.set_active(false);
}

CharsetTable::CharsetTable(const int standard_size): Gtk::Table(3, 2, false), cp437_label("CP437"),
						        latin_1_label("Latin 1"), frame("Tx/Rx character set") {

  cp437_label.set_justify(GTK_JUSTIFY_LEFT);
  latin_1_label.set_justify(GTK_JUSTIFY_LEFT);

  cp437_box.pack_start(cp437_button, false, false, 0);
  cp437_box.pack_start(cp437_label, false, false, 0);
  latin_1_box.pack_start(latin_1_button, false, false, 0);
  latin_1_box.pack_start(latin_1_label, false, false, 0);

  cp437_button.set_group(latin_1_button.group());
  cp437_button.set_active(true);

  charset_box.pack_start(cp437_box, false, false, standard_size/6);
  charset_box.pack_start(latin_1_box, false, false, standard_size/6);
  charset_box.set_border_width(standard_size/2);

  frame.set_usize(standard_size * 6, 0);
  frame.add(charset_box);

  Gtk::Pixmap* helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  help_button.add(*helpIcon_p);

  attach(filler1, 0, 2, 0, 1, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);
  attach(frame, 0, 1, 1, 2, GTK_EXPAND,
         0, 0, 0);
  attach(help_button, 1, 2, 1, 2, 0,
         0, standard_size/3, standard_size/3);
  attach(filler2, 0, 2, 2, 3, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);

  help_button.clicked.connect(SigC::bind(SigC::slot(this, &CharsetTable::show_help),
					       CharsetMessages::message));
  tooltips.set_tip(help_button, help_messages.get_message(CharsetMessages::message), 0);
  tooltips.set_delay(50);

  set_border_width(standard_size/3);
}

string CharsetTable::get_charset(void) const {
  string return_val;
  if (cp437_button.get_active()) return_val = "CP437";
  else if (latin_1_button.get_active()) return_val = "LATIN-1";
  return return_val;
}

void CharsetTable::set_charset(const int charset) {

  switch (charset) {
  case Prog_func::cp437:
    cp437_button.set_active(true);
    break;
  case Prog_func::latin_1:
    latin_1_button.set_active(true);
    break;
  }
}
void CharsetTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

void CharsetTable::clear(void) {
  cp437_button.set_active(true);
}

FontsizeTable::FontsizeTable(const int standard_size): Gtk::Table(3, 2, false), small_label("Small"),
						       medium_label("Medium"), large_label("Large"),
						       larger_label("Larger"), frame("Font size") {

  small_label.set_justify(GTK_JUSTIFY_LEFT);
  medium_label.set_justify(GTK_JUSTIFY_LEFT);
  large_label.set_justify(GTK_JUSTIFY_LEFT);
  larger_label.set_justify(GTK_JUSTIFY_LEFT);

  small_box.pack_start(small_button, false, false, 0);
  small_box.pack_start(small_label, false, false, 0);
  medium_box.pack_start(medium_button, false, false, 0);
  medium_box.pack_start(medium_label, false, false, 0);
  large_box.pack_start(large_button, false, false, 0);
  large_box.pack_start(large_label, false, false, 0);
  larger_box.pack_start(larger_button, false, false, 0);
  larger_box.pack_start(larger_label, false, false, 0);

  medium_button.set_group(small_button.group());
  large_button.set_group(small_button.group());
  larger_button.set_group(small_button.group());
  medium_button.set_active(true);

  fontsize_box.pack_start(small_box, false, false, standard_size/6);
  fontsize_box.pack_start(medium_box, false, false, standard_size/6);
  fontsize_box.pack_start(large_box, false, false, standard_size/6);
  fontsize_box.pack_start(larger_box, false, false, standard_size/6);
  fontsize_box.set_border_width(standard_size/2);

  frame.set_usize(standard_size * 6, 0);
  frame.add(fontsize_box);

  Gtk::Pixmap* helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  help_button.add(*helpIcon_p);

  attach(filler1, 0, 2, 0, 1, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);
  attach(frame, 0, 1, 1, 2, GTK_EXPAND,
         0, 0, 0);
  attach(help_button, 1, 2, 1, 2, 0,
         0, standard_size/3, standard_size/3);
  attach(filler2, 0, 2, 2, 3, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);

  help_button.clicked.connect(SigC::bind(SigC::slot(this, &FontsizeTable::show_help),
					       FontsizeMessages::message));
  tooltips.set_tip(help_button, help_messages.get_message(FontsizeMessages::message), 0);
  tooltips.set_delay(50);

  set_border_width(standard_size/3);
}

string FontsizeTable::get_fontsize(void) const {
  string return_val;
  if (small_button.get_active()) return_val = "small";
  else if (medium_button.get_active()) return_val = "medium";
  else if (large_button.get_active()) return_val = "large";
  else if (larger_button.get_active()) return_val = "larger";
  return return_val;
}

void FontsizeTable::set_fontsize(const int fontsize) {

  switch (fontsize) {
  case Fontsize::small:
    small_button.set_active(true);
    break;
  case Fontsize::medium:
    medium_button.set_active(true);
    break;
  case Fontsize::large:
    large_button.set_active(true);
    break;
  case Fontsize::larger:
    larger_button.set_active(true);
    break;
  }
}
void FontsizeTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

void FontsizeTable::clear(void) {
  medium_button.set_active(true);
}

MiscTable::MiscTable(const int standard_size): Gtk::Table(5, 3, false), filedir_label("File directory: "),
				           print_cmd_label("Print command: "), autocq_label("Autocq interval: ") {

  filedir_label.set_justify(GTK_JUSTIFY_RIGHT);
  print_cmd_label.set_justify(GTK_JUSTIFY_RIGHT);
  autocq_label.set_justify(GTK_JUSTIFY_RIGHT);
  filedir_entry.set_usize(standard_size * 6, (standard_size * 5)/6);
  print_cmd_entry.set_usize(standard_size * 6, (standard_size * 5)/6);
  autocq_entry.set_usize(standard_size * 6, (standard_size * 5)/6);

  Gtk::Pixmap* filedir_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  filedir_help_button.add(*filedir_helpIcon_p);
  Gtk::Pixmap* print_cmd_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  print_cmd_help_button.add(*print_cmd_helpIcon_p);
  Gtk::Pixmap* autocq_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  autocq_help_button.add(*autocq_helpIcon_p);

  attach(filler1, 1, 2, 0, 1, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);
  attach(filedir_label, 0, 1, 1, 2, 0,
         0, standard_size/3, standard_size/2);
  attach(filedir_entry, 1, 2, 1, 2, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/2);
  attach(filedir_help_button, 2, 3, 1, 2, 0,
         0, standard_size/3, standard_size/2);
  attach(print_cmd_label, 0, 1, 2, 3, 0,
         0, standard_size/3, standard_size/2);
  attach(print_cmd_entry, 1, 2, 2, 3, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/2);
  attach(print_cmd_help_button, 2, 3, 2, 3, 0,
         0, standard_size/3, standard_size/2);
  attach(autocq_label, 0, 1, 3, 4, 0,
         0, standard_size/3, standard_size/2);
  attach(autocq_entry, 1, 2, 3, 4, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/2);
  attach(autocq_help_button, 2, 3, 3, 4, 0,
         0, standard_size/3, standard_size/2);
  attach(filler2, 1, 2, 4, 5, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);

  filedir_help_button.clicked.connect(SigC::bind(SigC::slot(this, &MiscTable::show_help),
					       MiscMessages::filedir));
  print_cmd_help_button.clicked.connect(SigC::bind(SigC::slot(this, &MiscTable::show_help),
					       MiscMessages::print_cmd));
  autocq_help_button.clicked.connect(SigC::bind(SigC::slot(this, &MiscTable::show_help),
					       MiscMessages::autocq));

  tooltips.set_tip(filedir_help_button, help_messages.get_message(MiscMessages::filedir), 0);
  tooltips.set_tip(print_cmd_help_button, help_messages.get_message(MiscMessages::print_cmd), 0);
  tooltips.set_tip(autocq_help_button, help_messages.get_message(MiscMessages::autocq), 0);

  tooltips.set_delay(50);

  set_border_width(standard_size/3);
}

void MiscTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

void MiscTable::clear(void) {
  filedir_entry.set_text("");
  print_cmd_entry.set_text("");
  autocq_entry.set_text("");
}

SettingsDialog::SettingsDialog(const int size, Gtk::Window& window):
                                            Gtk::Window(GTK_WINDOW_DIALOG),
					    standard_size(size), in_run_loop(false),
			                    is_home_config(false), ok_button("OK"),
					    cancel_button("Cancel"), window_table(2, 3, false),
					    parent(window), identity_table(standard_size),
					    com_table(standard_size), packet_table(standard_size),
					    interface_table(standard_size), charset_table(standard_size),
					    fontsize_table(standard_size), misc_table(standard_size) {

  read_config();

  window_table.attach(notebook, 0, 3, 0, 1, GTK_FILL | GTK_EXPAND,
		      GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);
  window_table.attach(ok_button, 1, 2, 1, 2, GTK_EXPAND,
		      0, standard_size/2, standard_size/4);
  window_table.attach(cancel_button, 2, 3, 1, 2, GTK_EXPAND,
			0, standard_size/2, standard_size/4);

  if (!is_home_config) {

    Gtk::Label* message_label_p = manage(new Gtk::Label);
    message_label_p->set_line_wrap(true);
    message_label_p->set_text("Note: pressing the OK button will save the "
			      "settings in file ~/." RC_FILE);
    window_table.attach(*message_label_p, 0, 1, 1, 2, GTK_EXPAND,
			0, standard_size/2, standard_size/4);
  }

  else {
    Gtk::Button* reset_button_p = manage(new Gtk::Button("Reset"));
    reset_button_p->clicked.connect(SigC::slot(this, &SettingsDialog::get_reset_settings_prompt));
    reset_button_p->set_usize((standard_size * 5)/2, (standard_size * 4)/5);
    window_table.attach(*reset_button_p, 0, 1, 1, 2, GTK_EXPAND,
			0, standard_size/2, standard_size/4);
  }
    
  notebook.set_tab_pos(GTK_POS_TOP);
  notebook.set_homogeneous_tabs(true);
  notebook.set_scrollable(true);
  // set up the notebook pages
  {
    using namespace Gtk::Notebook_Helpers;
    PageList& page_list = notebook.pages();
    page_list.push_back(TabElem(identity_table, "Identity"));
    page_list.push_back(TabElem(com_table, "Com"));
    page_list.push_back(TabElem(packet_table, "Packet"));
    page_list.push_back(TabElem(interface_table, "Interface"));
    page_list.push_back(TabElem(charset_table, "Charset"));
    page_list.push_back(TabElem(fontsize_table, "Font"));
    page_list.push_back(TabElem(misc_table, "Misc"));
  }

  ok_button.clicked.connect(SigC::bind(SigC::slot(this, &SettingsDialog::selected), true));
  cancel_button.clicked.connect(SigC::bind(SigC::slot(this, &SettingsDialog::selected), false));

  identity_table.show_help_sig.connect(SigC::slot(this, &SettingsDialog::show_help));
  com_table.show_help_sig.connect(SigC::slot(this, &SettingsDialog::show_help));
  packet_table.show_help_sig.connect(SigC::slot(this, &SettingsDialog::show_help));
  interface_table.show_help_sig.connect(SigC::slot(this, &SettingsDialog::show_help));
  charset_table.show_help_sig.connect(SigC::slot(this, &SettingsDialog::show_help));
  fontsize_table.show_help_sig.connect(SigC::slot(this, &SettingsDialog::show_help));
  misc_table.show_help_sig.connect(SigC::slot(this, &SettingsDialog::show_help));

  add(window_table);
  
  set_title("kamplus: settings");
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  notebook.set_usize(standard_size * 14, standard_size * 9);
  ok_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  cancel_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  set_border_width(standard_size/3);
  cancel_button.grab_focus();
  set_position(GTK_WIN_POS_CENTER);
  //set_policy(false, false, false);

  show_all();
}

void SettingsDialog::selected(bool accept) {
  bool finish = false;

  if (accept) {
    if (write_config()) {
      parent.set_sensitive(true); // do this before we emit accepted()
      hide_all();
      finish = true;
      accepted();
    }
  }
  else {
    parent.set_sensitive(true); // do this before we emit accepted()
    hide_all();
    finish = true;
  }

  if (finish) {
    if (in_run_loop) Gtk::Main::quit();
    // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
    else delete this;
  }
}

gint SettingsDialog::delete_event_impl(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

void SettingsDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

bool SettingsDialog::write_config(void) {
  // returns false if any of the entered parameters is invalid, otherwise it returns true

  bool return_val = true;
  ifstream filein;

  if (!rcfile.empty()) {
    filein.open(rcfile.c_str(), ios::in);
    if (!filein) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      InfoDialog dialog(message.c_str(), "Config Error",
			standard_size, InfoDialog::critical, *this);
      dialog.run();
    }
  }

  vector<string> file_list;
  string file_read;
  
  string temp;

  string mycall_line("MY_CALL: ");
  temp = identity_table.get_mycall();
  strip(temp);
  if (temp.empty()) {
    return_val = false;
    InfoDialog dialog("No home callsign specified", "Config Error",
		      standard_size, InfoDialog::critical, *this);
    dialog.run();
  }
  else {
    transform(temp.begin(), temp.end(),
	      temp.begin(), to_upper);
    mycall_line += temp;
  }

  string mypacketcall_line;
  temp =  identity_table.get_mypacketcall();
  strip(temp);
  if (temp.empty()) mypacketcall_line = "#PKT_CALL: ";
  else {
    transform(temp.begin(), temp.end(),
	      temp.begin(), to_upper);
    mypacketcall_line = "PKT_CALL: ";
    mypacketcall_line += temp;
  }
  
  string myselcall_line;
  temp =  identity_table.get_myselcall();
  strip(temp);
  if (temp.empty()) myselcall_line = "#SELCALL: ";
  else {
    transform(temp.begin(), temp.end(),
	      temp.begin(), to_upper);
    myselcall_line = "SELCALL: ";
    myselcall_line += temp;
  }
  
  string port_line("SERIALPORT: ");
  temp = com_table.get_port();
  strip(temp);
  if (temp.empty()) {
    return_val = false;
    InfoDialog dialog("No serial port specified", "Config Error",
		      standard_size, InfoDialog::critical, *this);
    dialog.run();
  }
  else {
    port_line += temp;
    if (temp.compare(old_settings.port)) {
      InfoDialog dialog("Note: the change of port setting will not\n"
			"take effect until the program is restarted",
			"Config", standard_size, InfoDialog::warning, *this);
      dialog.run();
    }
  }

  string speed_line("SPEED: ");
  speed_line += com_table.get_speed();
  if (com_table.get_speed().compare(old_settings.speed)) {
    InfoDialog dialog("Note: the change of speed setting will not\n"
		      "take effect until the program is restarted",
		      "Config", standard_size, InfoDialog::warning, *this);
    dialog.run();
  }

  string vpaclen_line;
  temp =  packet_table.get_vhf_packet_size();
  strip(temp);
  if (temp.empty()) vpaclen_line = "#V_PACLEN: ";
  else if (atoi(temp.c_str()) < 32 || atoi(temp.c_str()) > MAX_FRAMESIZE) {
    ostrstream paclen_strm;
    ostrstream message_strm;
    paclen_strm << "V_PACLEN: " << MAX_FRAMESIZE << ends;
    message_strm << "Invalid vhf packet length specified.\n"
                    "Defaulting to "  << MAX_FRAMESIZE << ends;

    char* text = paclen_strm.str();
    vpaclen_line = text;
    delete[] text;

    text = message_strm.str();
    InfoDialog dialog(text, "Config", standard_size, InfoDialog::warning, *this);
    dialog.run();
    delete[] text;
  }
  else {
    vpaclen_line = "V_PACLEN: ";
    vpaclen_line += temp;
  }
  
  string hpaclen_line;
  temp =  packet_table.get_hf_packet_size();
  strip(temp);
  if (temp.empty()) hpaclen_line = "#H_PACLEN: ";
  else if (atoi(temp.c_str()) < 32 || atoi(temp.c_str()) > MAX_FRAMESIZE) {
    ostrstream paclen_strm;
    ostrstream message_strm;
    paclen_strm << "H_PACLEN: " << MAX_FRAMESIZE << ends;
    message_strm << "Invalid hf packet length specified.\n"
                    "Defaulting to "  << MAX_FRAMESIZE << ends;

    char* text = paclen_strm.str();
    hpaclen_line = text;
    delete[] text;

    text = message_strm.str();
    InfoDialog dialog(text, "Config", standard_size, InfoDialog::warning, *this);
    dialog.run();
    delete[] text;
  }
  else {
    hpaclen_line = "H_PACLEN: ";
    hpaclen_line += temp;
  }

  string keepalive_line("KEEPALIVE: ");
  if (packet_table.get_keepalive()) keepalive_line += "true";
  else keepalive_line += "false";

  string keepalive_mins_line;
  temp =  packet_table.get_keepalive_mins();
  strip(temp);
  if (temp.empty()) keepalive_mins_line = "#KEEPALIVE_MINS: ";
  else if (atoi(temp.c_str()) < 1) {
    keepalive_mins_line = "KEEPALIVE_MINS: 10";
    InfoDialog dialog("Invalid keep-alive interval specified\n"
		      "Defaulting to 10 minutes",
		      "Config", standard_size, InfoDialog::warning, *this);
    dialog.run();
  }
  else {
    keepalive_mins_line = "KEEPALIVE_MINS: ";
    keepalive_mins_line += temp;
  }

  string receive_cw_line("NO_CW_RX: ");
  if (interface_table.get_receive_cw()) receive_cw_line += "false";
  else receive_cw_line += "true";

  string sound_bell_line("BELL: ");
  if (interface_table.get_sound_bell()) sound_bell_line += "true";
  else sound_bell_line += "false";

  string no_prompt_line("NOPROMPT: ");
  if (interface_table.get_no_prompt()) no_prompt_line += "true";
  else no_prompt_line += "false";

  string charset_line;
  temp =  charset_table.get_charset();
  strip(temp);
  if (temp.empty()) charset_line = "#CHAR_SET: ";
  else {
    charset_line = "CHAR_SET: ";
    charset_line += temp;
  }
  
  string fontsize_line;
  temp =  fontsize_table.get_fontsize();
  strip(temp);
  if (temp.empty()) fontsize_line = "#WIN_FONTSIZE: ";
  else {
    fontsize_line = "WIN_FONTSIZE: ";
    fontsize_line += temp;
  }
  
  string filedir_line("FILEDIR: ");
  temp = misc_table.get_filedir();
  strip(temp);
  if (temp.empty()) {
      return_val = false;
      InfoDialog dialog("No file directory specified", "Config Error",
			standard_size, InfoDialog::critical, *this);
      dialog.run();
  }
  else filedir_line += temp;

  string print_cmd_line;
  temp = misc_table.get_print_cmd();
  strip(temp);
  if (temp.empty()) print_cmd_line = "#PRINT_CMD: ";
  else {
    print_cmd_line = "PRINT_CMD: ";
    print_cmd_line += temp;
  }

  string autocq_line;
  temp = misc_table.get_autocq();
  strip(temp);
  if (temp.empty()) autocq_line = "#AUTOCQ: ";
  else if (atoi(temp.c_str()) < 1) {
    autocq_line = "AUTOCQ: 90";
    InfoDialog dialog("Invalid autocq interval specified\n"
		      "Defaulting to 90 seconds",
		      "Config", standard_size, InfoDialog::warning, *this);
    dialog.run();
  }
  else {
    autocq_line = "AUTOCQ: ";
    autocq_line += temp;
  }
  
  if (return_val) { // no errors -- write out the configuration file
    
    bool found_mycall = false;
    bool found_mypacketcall = false;
    bool found_myselcall = false;
    bool found_port = false;
    bool found_speed = false;
    bool found_vpaclen = false;
    bool found_hpaclen = false;
    bool found_keepalive = false;
    bool found_keepalive_mins = false;
    bool found_receive_cw = false;
    bool found_sound_bell = false;
    bool found_no_prompt = false;
    bool found_charset = false;
    bool found_fontsize = false;
    bool found_filedir = false;
    bool found_print_cmd = false;
    bool found_autocq = false;
    
    const char terminating_line[] = "## end of " RC_FILE " ##";
    
    while (getline(filein, file_read)) {
      
      // look for "MY_CALL:"
      if (find_prog_parm("MY_CALL:", file_read) || find_prog_parm("#MY_CALL:", file_read)) {
	if (!found_mycall) {
	  found_mycall = true;
	  file_list.push_back(mycall_line);
	}
      }
      
      // look for "PKT_CALL:"
      else if (find_prog_parm("PKT_CALL:", file_read) || find_prog_parm("#PKT_CALL:", file_read)) {
	if (!found_mypacketcall) {
	  found_mypacketcall = true;
	  file_list.push_back(mypacketcall_line);
	}
      }
      
      // look for "SELCALL:"
      else if (find_prog_parm("SELCALL:", file_read) || find_prog_parm("#SELCALL:", file_read)) {
	if (!found_myselcall) {
	  found_myselcall = true;
	  file_list.push_back(myselcall_line);
	}
      }
      
      // look for "SERIALPORT:"
      else if (find_prog_parm("SERIALPORT:", file_read)  || find_prog_parm("#SERIALPORT:", file_read)) {
	if (!found_port) {
	  found_port = true;
	  file_list.push_back(port_line);
	}
      }
      
      // look for "SPEED:"
      else if (find_prog_parm("SPEED:", file_read) || find_prog_parm("#SPEED:", file_read))  {
	if (!found_speed) {
	  found_speed = true;
	  file_list.push_back(speed_line);
	}
      }
      
      // look for "V_PACLEN:"
      else if (find_prog_parm("V_PACLEN:", file_read) || find_prog_parm("#V_PACLEN:", file_read))  {
	if (!found_vpaclen) {
	  found_vpaclen = true;
	  file_list.push_back(vpaclen_line);
	}
      }
      
      // look for "H_PACLEN:"
      else if (find_prog_parm("H_PACLEN:", file_read) || find_prog_parm("#H_PACLEN", file_read))  {
	if (!found_hpaclen) {
	  found_hpaclen = true;
	  file_list.push_back(hpaclen_line);
	}
      }
      
      // look for "KEEPALIVE:"
      else if (find_prog_parm("KEEPALIVE:", file_read) || find_prog_parm("#KEEPALIVE:", file_read)) {
	if (!found_keepalive) {
	  found_keepalive = true;
	  file_list.push_back(keepalive_line);
	}
      }
      
      // look for "KEEPALIVE_MINS:"
      else if (find_prog_parm("KEEPALIVE_MINS:", file_read) || find_prog_parm("#KEEPALIVE_MINS:", file_read)) {
	if (!found_keepalive_mins) {
	  found_keepalive_mins = true;
	  file_list.push_back(keepalive_mins_line);
	}
      }
      
      // look for "NO_CW_RX:"
      else if (find_prog_parm("NO_CW_RX:", file_read) || find_prog_parm("#NO_CW_RX:", file_read)) {
	if (!found_receive_cw) {
	  found_receive_cw = true;
	  file_list.push_back(receive_cw_line);
	}
      }
      
      // look for "BELL:"
      else if (find_prog_parm("BELL:", file_read) || find_prog_parm("#BELL:", file_read)) {
	if (!found_sound_bell) {
	  found_sound_bell = true;
	  file_list.push_back(sound_bell_line);
	}
      }
      
      // look for "NOPROMPT:"
      else if (find_prog_parm("NOPROMPT:", file_read) || find_prog_parm("#NOPROMPT:", file_read)) {
	if (!found_no_prompt) {
	  found_no_prompt = true;
	  file_list.push_back(no_prompt_line);
	}
      }
      
      // look for "CHAR_SET:"
      else if (find_prog_parm("CHAR_SET:", file_read) || find_prog_parm("#CHAR_SET:", file_read)) {
	if (!found_charset) {
	  found_charset = true;
	  file_list.push_back(charset_line);
	}
      }
      
      // look for "WIN_FONTSIZE:"
      else if (find_prog_parm("WIN_FONTSIZE:", file_read) || find_prog_parm("#WIN_FONTSIZE:", file_read)) {
	if (!found_fontsize) {
	  found_fontsize = true;
	  file_list.push_back(fontsize_line);
	}
      }
      
      // look for "FILEDIR:"
      else if (find_prog_parm("FILEDIR:", file_read) || find_prog_parm("#FILEDIR:", file_read)) {
	if (!found_filedir) {
	  found_filedir = true;
	  file_list.push_back(filedir_line);
	}
      }
      
      // look for "PRINT_CMD:"
      else if (find_prog_parm("PRINT_CMD:", file_read) || find_prog_parm("#PRINT_CMD:", file_read)) {
	if (!found_print_cmd) {
	  found_print_cmd = true;
	  file_list.push_back(print_cmd_line);
	}
      }
      
      // look for "AUTOCQ:"
      else if (find_prog_parm("AUTOCQ:", file_read) || find_prog_parm("#AUTOCQ:", file_read)) {
	if (!found_autocq) {
	  found_autocq = true;
	  file_list.push_back(autocq_line);
	}
      }
      
      // add any residual lines to the list, except the terminating line
      else if (!find_prog_parm(terminating_line, file_read)) file_list.push_back(file_read);
    }
    
    if (found_mycall == false) file_list.push_back(mycall_line);
    if (found_mypacketcall == false) file_list.push_back(mypacketcall_line);
    if (found_myselcall == false) file_list.push_back(myselcall_line);
    if (found_port == false) file_list.push_back(port_line);
    if (found_speed == false) file_list.push_back(speed_line);
    if (found_vpaclen == false) file_list.push_back(vpaclen_line);
    if (found_hpaclen == false) file_list.push_back(hpaclen_line);
    if (found_keepalive == false) file_list.push_back(keepalive_line);
    if (found_keepalive_mins = false) file_list.push_back(keepalive_mins_line);
    if (found_receive_cw == false) file_list.push_back(receive_cw_line);
    if (found_sound_bell == false) file_list.push_back(sound_bell_line);
    if (found_no_prompt == false) file_list.push_back(no_prompt_line);
    if (found_charset == false) file_list.push_back(charset_line);
    if (found_fontsize == false) file_list.push_back(fontsize_line);
    if (found_filedir == false) file_list.push_back(filedir_line);
    if (found_print_cmd == false) file_list.push_back(print_cmd_line);
    if (found_autocq == false) file_list.push_back(autocq_line);
    
    // add the terminating line
    file_list.push_back(terminating_line);

    // now write out the new config file
    
    rcfile = prog_func.homedir + "/.";
    rcfile += RC_FILE;
    
    ofstream fileout(rcfile.c_str(), ios::out);
    if (!fileout) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      InfoDialog dialog(message.c_str(), "Config Error",
			standard_size, InfoDialog::critical, *this);
      dialog.run();
      return_val = false;
    }
    else {
      copy(file_list.begin(), file_list.end(),
	   ostream_iterator<string>(fileout, "\n"));
    }
  }
  return return_val;
}

void SettingsDialog::read_config(bool search_localfile) {

// search_localfile has a default value of true
// get rc file
  if (!get_rcfile_path(search_localfile)) {
    string message;
    if (search_localfile) {
      message = "Can't find or open file /etc/" RC_FILE ",\n"
	        "/usr/local/etc/" RC_FILE " or ";
      message += prog_func.homedir + "/." RC_FILE "\n";
    }
    else {
      message = "Can't find or open file /etc/" RC_FILE "\n"
	        "or /usr/local/etc/" RC_FILE;
      message += prog_func.homedir + "/." RC_FILE;
    }
    cerr << message << endl;
    InfoDialog dialog(message.c_str(), "Config Error",
		      standard_size, InfoDialog::critical, *this);
    dialog.run();
  }

  else {
// now extract settings from file

    ifstream filein(rcfile.c_str(), ios::in);
    if (!filein) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      InfoDialog dialog(message.c_str(), "Config Error",
			standard_size, InfoDialog::critical, *this);
      dialog.run();
    }

    else {

      old_settings.port = "";
      old_settings.speed = "9600";

      string file_read;
      string result;
      while (getline(filein, file_read)) {

	if (!file_read.empty() && file_read[0] != '#') { // valid line to check
	  // now check for other comment markers
	  string::size_type pos = file_read.find_first_of('#');
	  if (pos != string::npos) file_read.resize(pos); // truncate
	
	  // look for "MY_CALL:"
	  if (get_prog_parm("MY_CALL:", file_read, result)) {
	    transform(result.begin(), result.end(),
		      result.begin(), to_upper);
	    identity_table.set_mycall(result);
	  }
	
	  // look for "PKT_CALL:"
	  else if (get_prog_parm("PKT_CALL:", file_read, result)) {
	    transform(result.begin(), result.end(),
		      result.begin(), to_upper);
	    identity_table.set_mypacketcall(result);
	  }
	
	  // look for "SELCALL:"
	  else if (get_prog_parm("SELCALL:", file_read, result)) {
	    transform(result.begin(), result.end(),
		      result.begin(), to_upper);
	    identity_table.set_myselcall(result);
	  }
	
	  // look for "SERIALPORT:"
	  else if (get_prog_parm("SERIALPORT:", file_read, result)) {
	    com_table.set_port(result);
	    old_settings.port = result;
	  }

	  // look for "SPEED:"
	  else if (get_prog_parm("SPEED:", file_read, result)) {
	    int speed = atoi(result.c_str());
	    if (speed == 1200 || speed == 2400 || speed == 4800 || speed == 9600) {
	      com_table.set_speed(speed);
	      old_settings.speed = result;
	    }
	    else {
	      cerr << "Invalid speed value specified\nDefaulting to 9600 baud" << endl;
	      com_table.set_speed(9600);
	    }
	  }

	  // look for "V_PACLEN:"
	  else if (get_prog_parm("V_PACLEN:", file_read, result)) {
	    packet_table.set_vhf_packet_size(result);
	  }

	  // look for "H_PACLEN:"
	  else if (get_prog_parm("H_PACLEN:", file_read, result)) {
	    packet_table.set_hf_packet_size(result);
	  }
	
	  // look for "KEEPALIVE:"
	  else if (get_prog_parm("KEEPALIVE:", file_read, result)) {
	    if (!result.compare("TRUE") || !result.compare("true")) packet_table.set_keepalive(true);
	    else packet_table.set_keepalive(false);
	  }
	
	  // look for "KEEPALIVE_MINS:"
	  else if (get_prog_parm("KEEPALIVE_MINS:", file_read, result)) {
	    packet_table.set_keepalive_mins(result);
	  }
	
	  // look for "NO_CW_RX:"
	  else if (get_prog_parm("NO_CW_RX:", file_read, result)) {
	    if (!result.compare("TRUE") || !result.compare("true")) interface_table.set_receive_cw(false);
	    else interface_table.set_receive_cw(true);
	  }
	
	  // look for "BELL:"
	  else if (get_prog_parm("BELL:", file_read, result)) {
	    if (!result.compare("TRUE") || !result.compare("true")) interface_table.set_sound_bell(true);
	    else interface_table.set_sound_bell(false);
	  }
	
	  // look for "NOPROMPT:"
	  else if (get_prog_parm("NOPROMPT:", file_read, result)) {
	    if (!result.compare("TRUE") || !result.compare("true")) interface_table.set_no_prompt(true);
	    else interface_table.set_no_prompt(false);
	  }
	
	  // look for "CHAR_SET:"
	  else if (get_prog_parm("CHAR_SET:", file_read, result)) {
	    if (!result.compare("latin-1") || !result.compare("LATIN-1")) {
	      charset_table.set_charset(Prog_func::latin_1);
	    }
	    else charset_table.set_charset(Prog_func::cp437);
	  }

	  // look for "WIN_FONTSIZE:"
	  else if (get_prog_parm("WIN_FONTSIZE:", file_read, result)) {
	    if (!result.compare("SMALL") || !result.compare("small")) fontsize_table.set_fontsize(Fontsize::small);
	    else if (!result.compare("MEDIUM") || !result.compare("medium")) fontsize_table.set_fontsize(Fontsize::medium);
	    else if (!result.compare("LARGE") || !result.compare("large")) fontsize_table.set_fontsize(Fontsize::large);
	    else if (!result.compare("LARGER") || !result.compare("larger")) fontsize_table.set_fontsize(Fontsize::larger);
	    else {
	      cerr << "Invalid fontsize specified\nDefaulting to medium size" << endl;
	      fontsize_table.set_fontsize(Fontsize::medium);
	    }
	  }

	  // look for "FILEDIR:"
	  else if (get_prog_parm("FILEDIR:", file_read, result)) {
	    misc_table.set_filedir(result);
	  }

	  // look for "PRINT_CMD:"
	  else if (get_prog_parm("PRINT_CMD:", file_read, result)) {
	    misc_table.set_print_cmd(result);
	  }

	  // look for "AUTOCQ:"
	  else if (get_prog_parm("AUTOCQ:", file_read, result)) {
	    misc_table.set_autocq(result);
	  }
	}
      }
    }
  }
}

void SettingsDialog::get_reset_settings_prompt(void) {
  PromptDialog* dialog_p = new PromptDialog("Enter settings from /etc/" RC_FILE " or\n"
					    "/usr/local/etc/" RC_FILE "?",
					    "Reset settings", standard_size, *this);
  if (!dialog_p) {
    cerr << "Memory allocation error in SettingsDialog::get_reset_settings_prompt()" << endl;
    exit(MEM_ERROR);
  }
  dialog_p->accepted.connect(SigC::slot(this, &SettingsDialog::get_reset_settings));
  // there is no memory leak -- the memory will be deleted when PromptDialog closes
}

void SettingsDialog::get_reset_settings(void) {

  // clear all the existing settings
  identity_table.clear();
  com_table.clear();
  packet_table.clear();
  interface_table.clear();
  charset_table.clear();
  fontsize_table.clear();
  misc_table.clear();

  read_config(false); // read settings without searching for config file in home directory
}

bool SettingsDialog::get_prog_parm(const char* name, string& line, string& result) {
// this function looks for a setting named `name' in the string `line'
// and returns the values stated after it in string `result'
// it returns `true' if the setting was found 
// if there are trailing spaces or tabs, string `line' will be modified
// string `result' is only modified if the `name' setting is found

  const string::size_type length = strlen(name);
  // we have to use std::string::substr() because libstdc++-2
  // doesn't support the Std-C++ std::string::compare() functions
  if (!line.substr(0, length).compare(name)) {
    // erase any trailing space or tab
    while (line.find_last_of(" \t") == line.size() - 1) line.resize(line.size() - 1);
    if (line.size() > length) {
      // ignore any preceding space or tab from the setting value given
      string::size_type pos = line.find_first_not_of(" \t", length); // pos now is set to beginning of setting value
      if (pos != string::npos) result.assign(line, pos, line.size() - pos);
    }
    return true;
  }
  return false;
}

bool SettingsDialog::find_prog_parm(const char* name, const string& line) {
// this function looks for a setting named `name' in the string `line'
// it returns `true' if the setting was found or false otherwise

  const string::size_type length = strlen(name);
  // we have to use std::string::substr() because libstdc++-2
  // doesn't support the Std-C++ std::string::compare() functions
  if (!line.substr(0, length).compare(name)) return true;
  return false;
}

bool SettingsDialog::get_rcfile_path(bool search_localfile) {
// search_localfile has a default value of true

  bool found_rcfile = false;

  if (search_localfile) {
    rcfile = prog_func.homedir;
    rcfile += "/." RC_FILE;

    if (!access(rcfile.c_str(), F_OK)) {
      found_rcfile = true;
      is_home_config = true;
    }
  }

  if (!found_rcfile) {
    rcfile = "/usr/local/etc/" RC_FILE;
    if (!access(rcfile.c_str(), F_OK)) found_rcfile = true;
  }

  if (!found_rcfile) {
    rcfile = "/etc/" RC_FILE;
    if (!access(rcfile.c_str(), F_OK)) found_rcfile = true;
  }
  if (!found_rcfile) rcfile = "";
  return found_rcfile;
}

void SettingsDialog::show_help(const string& message, const string& caption) {
  SettingsHelpDialog* dialog_p = new SettingsHelpDialog(standard_size,
							message, caption, *this);
  if (!dialog_p) {
    cerr << "Memory allocation error in SettingsDialog::show_help()" << endl;
    exit(MEM_ERROR);
  }
  // there is no memory leak
  // the SettingsHelpDialog object will delete its own memory when it is closed
}

void SettingsDialog::strip(string& setting) {

  // erase any trailing space or tab
  while (!setting.empty() && setting.find_last_of(" \t") == setting.size() - 1) {
    setting.resize(setting.size() - 1);
  }
  // erase any leading space or tab
  while (!setting.empty() && (setting[0] == ' ' || setting[0] == '\t')) {
    setting.erase(0, 1);
  }
}

InitSettingsDialog::InitSettingsDialog(const int size, const int fontsize):
                                            Gtk::Window(GTK_WINDOW_DIALOG),
			                    message_label("Note: pressing the OK button will save the "
			                                  "settings in file ~/." RC_FILE),
                                            ok_button("OK"), cancel_button("Cancel"), 
					    window_table(2, 3, false), standard_size(size) {

  if (fontsize >=10 && fontsize <= 12) {
    Gtk::Style* new_style_p = get_style(); // this will grab the global style
                                           // (we did not use get_style()->copy())
                                           // accordingly there is no need to call set_style(*new_style_p)
                                           // after we have called set_font()
    ostrstream strm;
    strm << "-adobe-helvetica-medium-r-normal--" << fontsize << "-*-*-*-*-*-iso8859-1" << ends;
    const char* font_name = strm.str();
    new_style_p->set_font(Gdk_Font(font_name));
    delete[] font_name;
  }

  message_label.set_line_wrap(true);
  window_table.attach(message_label, 0, 1, 1, 2, GTK_EXPAND,
		      0, standard_size/2, standard_size/4);
  window_table.attach(ok_button, 1, 2, 1, 2, GTK_EXPAND,
		      0, standard_size/2, standard_size/4);
  window_table.attach(cancel_button, 2, 3, 1, 2, GTK_EXPAND,
		      0, standard_size/2, standard_size/4);

  ok_button.clicked.connect(SigC::bind(SigC::slot(this, &InitSettingsDialog::selected), true));
  cancel_button.clicked.connect(SigC::bind(SigC::slot(this, &InitSettingsDialog::selected), false));

  add(window_table);
  
  set_modal(true);

  ok_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  cancel_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  set_border_width(standard_size/3);
  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);
}

void InitSettingsDialog::attach_settings_widget(Gtk::Widget& settings_widget) {
  window_table.attach(settings_widget, 0, 3, 0, 1, GTK_FILL | GTK_EXPAND,
		      GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);
}

void InitSettingsDialog::selected(bool accept) {

  if (accept) {
    if (write_config()) {
      hide_all();
      result = accepted;
      Gtk::Main::quit();
    }
  }
  else {
    hide_all();
    result = rejected;
    Gtk::Main::quit();
  }
}

bool InitSettingsDialog::find_prog_parm(const char* name, const string& line) {
// this function looks for a setting named `name' in the string `line'
// it returns `true' if the setting was found or false otherwise

  const string::size_type length = strlen(name);
  // we have to use std::string::substr() because libstdc++-2
  // doesn't support the Std-C++ std::string::compare() functions
  if (!line.substr(0, length).compare(name)) return true;
  return false;
}

gint InitSettingsDialog::delete_event_impl(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

InitSettingsDialog::Result InitSettingsDialog::run(void) {
  Gtk::Main::run();
  return result;
}

void InitSettingsDialog::strip(string& setting) {

  // erase any trailing space or tab
  while (!setting.empty() && setting.find_last_of(" \t") == setting.size() - 1) {
    setting.resize(setting.size() - 1);
  }
  // erase any leading space or tab
  while (!setting.empty() && (setting[0] == ' ' || setting[0] == '\t')) {
    setting.erase(0, 1);
  }
}

string InitSettingsDialog::get_rcfile_path(void) {
// search_localfile has a default value of true

  string rcfile;
  bool found_rcfile = false;

  rcfile = prog_func.homedir;
  rcfile += "/." RC_FILE;
  
  if (!access(rcfile.c_str(), F_OK)) found_rcfile = true;

  if (!found_rcfile) {
    rcfile = "/usr/local/etc/" RC_FILE;
    if (!access(rcfile.c_str(), F_OK)) found_rcfile = true;
  }

  if (!found_rcfile) {
    rcfile = "/etc/" RC_FILE;
    if (!access(rcfile.c_str(), F_OK)) found_rcfile = true;
  }

  if (!found_rcfile) rcfile = "";
  return rcfile;
}

void InitSettingsDialog::show_help(const string& message, const string& caption) {
  SettingsHelpDialog* dialog_p = new SettingsHelpDialog(standard_size,
							message, caption, *this);
  if (!dialog_p) {
    cerr << "Memory allocation error in InitSettingsDialog::show_help()" << endl;
    exit(MEM_ERROR);
  }
  // there is no memory leak
  // the SettingsHelpDialog object will delete its own memory when it is closed
}

IdentitySettingsDialog::IdentitySettingsDialog(const int size, const int fontsize):
                                      InitSettingsDialog(size, fontsize), identity_table(size) {


  attach_settings_widget(identity_table);

  identity_table.show_help_sig.connect(SigC::slot(this, &InitSettingsDialog::show_help));

  set_title("kamplus: callsign settings");
  set_modal(true);

  if (!prog_func.myCall.empty()) identity_table.set_mycall(prog_func.myCall);
  if (!prog_func.myPktCall.empty()) identity_table.set_mypacketcall(prog_func.myPktCall);
  if (!prog_func.mySelCall.empty()) identity_table.set_myselcall(prog_func.mySelCall);

  show_all();
}

bool IdentitySettingsDialog::write_config(void) {
  // returns false if any of the entered parameters is invalid, otherwise it returns true

  string rcfile(get_rcfile_path());

  bool return_val = true;
  ifstream filein;

  if (!rcfile.empty()) {
    filein.open(rcfile.c_str(), ios::in);
    if (!filein) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      InfoDialog dialog(message.c_str(), "Config Error",
			standard_size, InfoDialog::critical, *this);
      dialog.run();
      exit(FILEOPEN_ERROR);
    }
  }
  else {
    string message("Can't open file ");
    message += rcfile;
    cerr << message << endl;
    InfoDialog dialog(message.c_str(), "Config Error",
		      standard_size, InfoDialog::critical, *this);
    dialog.run();
    exit(FILEOPEN_ERROR);
  }

  vector<string> file_list;
  string file_read;
  string temp;

  string mycall_line("MY_CALL: ");
  temp = identity_table.get_mycall();
  strip(temp);
  if (temp.empty()) {
    return_val = false;
    InfoDialog dialog("No home callsign specified", "Config Error",
		      standard_size, InfoDialog::critical, *this);
    dialog.run();
  }
  else {
    transform(temp.begin(), temp.end(),
	      temp.begin(), to_upper);
    mycall_line += temp;
  }

  string mypacketcall_line;
  temp =  identity_table.get_mypacketcall();
  strip(temp);
  if (temp.empty()) mypacketcall_line = "#PKT_CALL: ";
  else {
    transform(temp.begin(), temp.end(),
	      temp.begin(), to_upper);
    mypacketcall_line = "PKT_CALL: ";
    mypacketcall_line += temp;
  }
  
  string myselcall_line;
  temp =  identity_table.get_myselcall();
  strip(temp);
  if (temp.empty()) myselcall_line = "#SELCALL: ";
  else {
    transform(temp.begin(), temp.end(),
	      temp.begin(), to_upper);
    myselcall_line = "SELCALL: ";
    myselcall_line += temp;
  }
  
  if (return_val) { // no errors -- write out the configuration file

    bool found_mycall = false;
    bool found_mypacketcall = false;
    bool found_myselcall = false;

    const char terminating_line[] = "## end of " RC_FILE " ##";

    while (getline(filein, file_read)) {
      
      // look for "MY_CALL:"
      if (find_prog_parm("MY_CALL:", file_read) || find_prog_parm("#MY_CALL:", file_read)) {
	if (!found_mycall) {
	  found_mycall = true;
	  file_list.push_back(mycall_line);
	}
      }
      
      // look for "PKT_CALL:"
      else if (find_prog_parm("PKT_CALL:", file_read) || find_prog_parm("#PKT_CALL:", file_read)) {
	if (!found_mypacketcall) {
	  found_mypacketcall = true;
	  file_list.push_back(mypacketcall_line);
	}
      }
      
      // look for "SELCALL:"
      else if (find_prog_parm("SELCALL:", file_read) || find_prog_parm("#SELCALL:", file_read)) {
	if (!found_myselcall) {
	  found_myselcall = true;
	  file_list.push_back(myselcall_line);
	}
      }
      
      // add any residual lines to the list, except the terminating line
      else if (!find_prog_parm(terminating_line, file_read)) file_list.push_back(file_read);
    }
    
    // complete file_list if no existing callsign placeholders in the configuration file
    if (found_mycall == false) file_list.push_back(mycall_line);
    if (found_mypacketcall == false) file_list.push_back(mypacketcall_line);
    if (found_myselcall == false) file_list.push_back(myselcall_line);
      
    // add the terminating line
    file_list.push_back(terminating_line);

    // now write out the new config file
    
    rcfile = prog_func.homedir;
    rcfile += "/." RC_FILE;
    
    ofstream fileout(rcfile.c_str(), ios::out);
    if (!fileout) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      InfoDialog dialog(message.c_str(), "Config Error",
			standard_size, InfoDialog::critical, *this);
      dialog.run();
      exit(FILEOPEN_ERROR);
    }
    else {
      copy(file_list.begin(), file_list.end(),
	   ostream_iterator<string>(fileout, "\n"));
    }
  }
  return return_val;
}

ComSettingsDialog::ComSettingsDialog(const int size, const int speed, const string& port, const int fontsize):
                                      InitSettingsDialog(size, fontsize), com_table(size) {


  attach_settings_widget(com_table);

  com_table.show_help_sig.connect(SigC::slot(this, &InitSettingsDialog::show_help));

  set_title("kamplus: serial port settings");
  set_modal(true);

  if (!port.empty()) com_table.set_port(port);
  if (speed) com_table.set_speed(speed);

  show_all();
}

bool ComSettingsDialog::write_config(void) {
  // returns false if any of the entered parameters is invalid, otherwise it returns true

  string rcfile(get_rcfile_path());

  bool return_val = true;
  ifstream filein;

  if (!rcfile.empty()) {
    filein.open(rcfile.c_str(), ios::in);
    if (!filein) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      InfoDialog dialog(message.c_str(), "Config Error",
			standard_size, InfoDialog::critical, *this);
      dialog.run();
      exit(FILEOPEN_ERROR);
    }
  }
  else {
    string message("Can't open file ");
    message += rcfile;
    cerr << message << endl;
    InfoDialog dialog(message.c_str(), "Config Error",
		      standard_size, InfoDialog::critical, *this);
    dialog.run();
    exit(FILEOPEN_ERROR);
  }

  vector<string> file_list;
  string file_read;
  string temp;

  string port_line("SERIALPORT: ");
  temp = com_table.get_port();
  strip(temp);
  if (temp.empty()) {
    return_val = false;
    InfoDialog dialog("No serial port specified", "Config Error",
		      standard_size, InfoDialog::critical, *this);
    dialog.run();
  }
  else port_line += temp;

  string speed_line("SPEED: ");
  speed_line += com_table.get_speed();

  if (return_val) { // no errors -- write out the configuration file
    bool found_port = false;
    bool found_speed = false;

    const char terminating_line[] = "## end of " RC_FILE " ##";

    while (getline(filein, file_read)) {
      
      // look for "SERIALPORT:"
      if (find_prog_parm("SERIALPORT:", file_read)  || find_prog_parm("#SERIALPORT:", file_read)) {
	if (!found_port) {
	  found_port = true;
	  file_list.push_back(port_line);
	}
      }
      
      // look for "SPEED:"
      else if (find_prog_parm("SPEED:", file_read) || find_prog_parm("#SPEED:", file_read))  {
	if (!found_speed) {
	  found_speed = true;
	  file_list.push_back(speed_line);
	}
      }
      // add any residual lines to the list, except the terminating line
      else if (!find_prog_parm(terminating_line, file_read)) file_list.push_back(file_read);
    }
    
    // complete file_list if no existing port or speed placeholder in the configuration file
    if (found_port == false) file_list.push_back(port_line);
    if (found_speed == false) file_list.push_back(speed_line);
      
    // add the terminating line
    file_list.push_back(terminating_line);

    // now write out the new config file
    
    rcfile = prog_func.homedir;
    rcfile += "/." RC_FILE;
    
    ofstream fileout(rcfile.c_str(), ios::out);
    if (!fileout) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      InfoDialog dialog(message.c_str(), "Config Error",
			standard_size, InfoDialog::critical, *this);
      dialog.run();
      exit(FILEOPEN_ERROR);
    }
    else {
      copy(file_list.begin(), file_list.end(),
	   ostream_iterator<string>(fileout, "\n"));
    }
  }
  return return_val;
}

FiledirTable::FiledirTable(const int standard_size): Gtk::Table(3, 3, false), filedir_label("File directory: ") {

  filedir_label.set_justify(GTK_JUSTIFY_RIGHT);
  filedir_entry.set_usize(standard_size * 6, (standard_size * 5)/6);
  Gtk::Pixmap* filedir_helpIcon_p = manage(new Gtk::Pixmap(help_xpm));
  filedir_help_button.add(*filedir_helpIcon_p);

  attach(filler1, 1, 2, 0, 1, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);
  attach(filedir_label, 0, 1, 1, 2, 0,
         0, standard_size/3, standard_size/2);
  attach(filedir_entry, 1, 2, 1, 2, GTK_EXPAND | GTK_FILL,
         0, standard_size/3, standard_size/2);
  attach(filedir_help_button, 2, 3, 1, 2, 0,
         0, standard_size/3, standard_size/2);
  attach(filler2, 1, 2, 2, 3, GTK_EXPAND | GTK_FILL,
	 GTK_EXPAND | GTK_FILL, 0, 0);

  filedir_help_button.clicked.connect(SigC::bind(SigC::slot(this, &FiledirTable::show_help),
					       MiscMessages::filedir));
  tooltips.set_tip(filedir_help_button, help_messages.get_message(MiscMessages::filedir), 0);
  tooltips.set_delay(50);

  set_border_width(standard_size/3);
}

void FiledirTable::show_help(int message_index) {
  show_help_sig(help_messages.get_message(message_index),
		help_messages.get_caption(message_index));
}

FiledirSettingsDialog::FiledirSettingsDialog(const int size, const int fontsize):
                                      InitSettingsDialog(size, fontsize), filedir_table(size) {

  attach_settings_widget(filedir_table);

  filedir_table.show_help_sig.connect(SigC::slot(this, &InitSettingsDialog::show_help));

  set_title("kamplus: file directory setting");
  set_modal(true);

  show_all();
}

bool FiledirSettingsDialog::write_config(void) {
  // returns false if any of the entered parameters is invalid, otherwise it returns true

  string rcfile(get_rcfile_path());

  bool return_val = true;
  ifstream filein;

  if (!rcfile.empty()) {
    filein.open(rcfile.c_str(), ios::in);
    if (!filein) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      InfoDialog dialog(message.c_str(), "Config Error",
			standard_size, InfoDialog::critical, *this);
      dialog.run();
      exit(FILEOPEN_ERROR);
    }
  }
  else {
    string message("Can't open file ");
    message += rcfile;
    cerr << message << endl;
    InfoDialog dialog(message.c_str(), "Config Error",
		      standard_size, InfoDialog::critical, *this);
    dialog.run();
    exit(FILEOPEN_ERROR);
  }

  vector<string> file_list;
  string file_read;
  string temp;

  string filedir_line("FILEDIR: ");
  temp = filedir_table.get_filedir();
  strip(temp);
  if (temp.empty()) {
      return_val = false;
      InfoDialog dialog("No file directory specified", "Config Error",
			standard_size, InfoDialog::critical, *this);
      dialog.run();
  }
  else filedir_line += temp;

  if (return_val) { // no errors -- write out the configuration file

    bool found_filedir = false;

    const char terminating_line[] = "## end of " RC_FILE " ##";

    while (getline(filein, file_read)) {
      
      // look for "FILEDIR:"
      if (find_prog_parm("FILEDIR:", file_read) || find_prog_parm("#FILEDIR:", file_read)) {
	if (!found_filedir) {
	  found_filedir = true;
	  file_list.push_back(filedir_line);
	}
      }
      
      // add any residual lines to the list, except the terminating line
      else if (!find_prog_parm(terminating_line, file_read)) file_list.push_back(file_read);
    }
    
    // complete file_list if no existing callsign placeholders in the configuration file
    if (found_filedir == false) file_list.push_back(filedir_line);
      
    // add the terminating line
    file_list.push_back(terminating_line);

    // now write out the new config file
    
    rcfile = prog_func.homedir;
    rcfile += "/." RC_FILE;
    
    ofstream fileout(rcfile.c_str(), ios::out);
    if (!fileout) {
      string message("Can't open file ");
      message += rcfile;
      cerr << message << endl;
      InfoDialog dialog(message.c_str(), "Config Error",
			standard_size, InfoDialog::critical, *this);
      dialog.run();
      exit(FILEOPEN_ERROR);
    }
    else {
      copy(file_list.begin(), file_list.end(),
	   ostream_iterator<string>(fileout, "\n"));
    }
  }
  return return_val;
}
