/* Copyright (C) 1999, 2000 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/


#include <iostream>
#include <fstream>
#include <strstream>
#include <cstdlib>
#include <cstdio>
#include <gtk--/scrollbar.h>
#include <gtk--/adjustment.h>
//#include <gdk--/font.h>
//#include <gdk--/color.h>

#include "receivewin.h"
#include "modal_dialogs.h"

#define MAX_CHARS 15000
#define DEL_BLOCK_SIZE 500

gint ReceiveWinText::button_press_event_impl(GdkEventButton* event_p) {

  if (event_p->button == 3) mouse_right_clicked(event_p);
  else if (event_p->button == 1) {
    Gtk::Text::button_press_event_impl(event_p);
    return false;
  }
  return true;
}

gint ReceiveWinText::button_release_event_impl(GdkEventButton* event_p) {

  if (event_p->button != 2) {
    Gtk::Text::button_release_event_impl(event_p);
    if (event_p->button == 1) mouse_released();
    return false;
  }
  return true;
}

ReceiveWinNode::ReceiveWinNode(int fontsize): Gtk::Table(1, 2, false), is_frozen_flag(false) {
  
  attach(text, 0, 1, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, 0);
  
  // put a VScrollbar in the upper right
  Gtk::Scrollbar* scrollbar_p = manage(new Gtk::VScrollbar(*(text.get_vadjustment())));
  attach (*scrollbar_p, 1, 2, 0, 1, GTK_FILL, GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0);

  text.set_word_wrap(true);
  text.set_line_wrap(true);

  if (!font_list[fontsize].empty()) text.set_context(Gdk_Font(font_list[fontsize]));
  else if (!font_list[Fontsize::medium].empty()) text.set_context(Gdk_Font(font_list[Fontsize::medium]));

  text.set_editable(false);
  text.set_usize(MIN_WIDTH, (MIN_HEIGHT * 2)/3);
  text.unset_flags(GTK_CAN_FOCUS);
}

void ReceiveWinNode::freeze(void) {
  // we use the is_frozen flag so that we don't make unnecessary calls to thaw() (when are not frozen),
  // which causes flicker
  if (!is_frozen()) {
    is_frozen_flag = true;
    text.freeze();
  }
}

void ReceiveWinNode::thaw(void) {
  // we use the is_frozen flag so that we don't make unnecessary calls to thaw() (when are not frozen),
  // which causes flicker
  if (is_frozen()) {
    is_frozen_flag = false;
    text.thaw();
  }
}

void ReceiveWinNode::scrollup(void) {
  Gtk::Adjustment* adj_p = text.get_vadjustment();
  if (adj_p->get_value() == adj_p->get_lower()) beep();
  else adj_p->set_value(adj_p->get_value() - adj_p->get_page_size()/2);
  // in case we were frozen, now thaw (thawing will not cause a scroll to the end
  // of the window even if we have added text since last frozen)
  // this will update the scrollbars
  thaw();
}

void ReceiveWinNode::scrolldown(void) {
  // in case we were frozen, now thaw (thawing will not cause a scroll to the end
  // of the window even if we have added text since last frozen)
  // this will update the scrollbars
  thaw();
  Gtk::Adjustment* adj_p = text.get_vadjustment();
  if (adj_p->get_value() == adj_p->get_upper() - adj_p->get_page_size()) beep();
  else adj_p->set_value(adj_p->get_value() + adj_p->get_page_size()/2);

  if (adj_p->get_value() == adj_p->get_upper() - adj_p->get_page_size()) { // check to see if we are still scrolling
    while (text.get_length() > MAX_CHARS + DEL_BLOCK_SIZE) del_block();
  }
}

void ReceiveWinNode::scrollout(bool silent) {
  // in case we were frozen, now thaw
  thaw();
  Gtk::Adjustment* adj_p = text.get_vadjustment();
  if (adj_p->get_value() == adj_p->get_upper() - adj_p->get_page_size()) {
    if (!silent) beep();
  }
  else {
    adj_p->set_value(adj_p->get_upper());
  }
  // and now delete some text at the beginning if the text stored is getting too big
  // (we only do this while not scrolling in order not to confuse the user)
  while (text.get_length() > MAX_CHARS + DEL_BLOCK_SIZE) del_block();
}

void ReceiveWinNode::del_block(void) {
  text.set_editable(true);
  text.freeze();
  text.set_point(0);
  text.forward_delete(DEL_BLOCK_SIZE);
  text.thaw();
  text.set_point(text.get_length()); // we must do this after we have thawed to get the correct result
  text.set_editable(false);
}

ReceiveWin::ReceiveWin(Gtk::VPaned& a, Tnc* b, int c, Gtk::Window& d, int fontsize): parent(a), tnc_p(b),
					      standard_size(c), top_level_window(d) {
  int port;
  int stream;
  
  for (port = 0; port < 2; port++) {
    for (stream = 0; stream < MAXUSERS; stream++) {
      if (!(node_ptr[stream][port] = new ReceiveWinNode(fontsize))) {
	cerr << "Memory allocation error in ReceiveWin::ReceiveWin()" << endl;
	exit(MEM_ERROR);
      }
      // we now relay any mouse_right_clicked signals to slot mouse_popup() in class Mainscreen
      node_ptr[stream][port]->text.mouse_right_clicked.connect(mouse_right_clicked.slot());
      // and now cause any mouse release event to set the text selected items in menus
      node_ptr[stream][port]->text.mouse_released.connect(SigC::slot(this, &ReceiveWin::set_textselected_items));
    }
  }
  current_p = node_ptr[0][0];

  if (!font_list[fontsize].empty()) {
    red_context.set_font(Gdk_Font(font_list[fontsize]));
    blue_context.set_font(Gdk_Font(font_list[fontsize]));
    green_context.set_font(Gdk_Font(font_list[fontsize]));
    purple_context.set_font(Gdk_Font(font_list[fontsize]));
  }

  else if (!font_list[Fontsize::medium].empty()) {
    red_context.set_font(Gdk_Font(font_list[Fontsize::medium]));
    blue_context.set_font(Gdk_Font(font_list[Fontsize::medium]));
    green_context.set_font(Gdk_Font(font_list[Fontsize::medium]));
    purple_context.set_font(Gdk_Font(font_list[Fontsize::medium]));
  }

  Gdk_Color red;
  red.set_rgb_p(0.8, 0, 0);
  Gdk_Color blue;
  blue.set_rgb_p(0, 0, 0.8);
  Gdk_Color green;
  green.set_rgb_p(0, 0.6, 0);
  Gdk_Color purple;
  purple.set_rgb_p(0.5, 0, 0.5);
  red_context.set_foreground(red);
  blue_context.set_foreground(blue);
  green_context.set_foreground(green);
  purple_context.set_foreground(purple);
}

ReceiveWin::~ReceiveWin(void) {
  int port;
  int stream;
  
  for (port = 0; port < 2; port++) {
    for (stream = 0; stream < MAXUSERS; stream++) delete node_ptr[stream][port];
  }
}

void ReceiveWin::set_textselected_items(void) {
  if (current_p->text.has_selection()) {
    activate_textselected_items();
  }
  else disactivate_textselected_items();
}

void ReceiveWin::copy(void) {
  if (current_p->text.has_selection()) current_p->text.copy_clipboard();
}

void ReceiveWin::print_selection_prompt(void) {
  if (current_p->text.has_selection()) {
    guint start_pos = current_p->text.get_selection_start_pos();
    guint end_pos = current_p->text.get_selection_end_pos();
    if (start_pos > end_pos) {
      guint temp = start_pos;
      start_pos = end_pos;
      end_pos = temp;
    }
    string text(current_p->text.get_chars(start_pos, end_pos));
    PromptDialog* dialog_p = new PromptDialog("Print selected text?", "Print Text", standard_size,
					      top_level_window);
    if (!dialog_p) {
      cerr << "Memory allocation error in ReceiveWin::print_selection_prompt()" << endl;
      exit(MEM_ERROR);
    }
    dialog_p->accepted.connect(SigC::bind(SigC::slot(this, &ReceiveWin::print_text), text));
  }
}

void ReceiveWin::print_scroll_buffer_prompt(void) {
  if (current_p->text.get_length()) {
    string text(current_p->text.get_chars(0, current_p->text.get_length()));
    PromptDialog* dialog_p = new PromptDialog("Print the scroll buffer?", "Print Buffer", standard_size,
					      top_level_window);
    if (!dialog_p) {
      cerr << "Memory allocation error in ReceiveWin::print_scroll_buffer_prompt()" << endl;
      exit(MEM_ERROR);
    }
    dialog_p->accepted.connect(SigC::bind(SigC::slot(this, &ReceiveWin::print_text), text));
  }
}

void ReceiveWin::print_text(string text) {
    
  FILE* pipe_fp = popen(prog_func.print_cmd.c_str(), "w");
  fwrite(text.c_str(), sizeof(char), text.size(), pipe_fp);
  fclose(pipe_fp);
}


void ReceiveWin::save_prompt(void) {
  if (current_p->text.has_selection()) {
    guint start_pos = current_p->text.get_selection_start_pos();
    guint end_pos = current_p->text.get_selection_end_pos();
    if (start_pos > end_pos) {
      guint temp = start_pos;
      start_pos = end_pos;
      end_pos = temp;
    }
    string text(current_p->text.get_chars(start_pos, end_pos));
    string filedir(prog_func.filedir);
    filedir += '/';
    FileSaveSelectDialog* dialog_p = new FileSaveSelectDialog(standard_size, top_level_window, filedir.c_str());
    if (!dialog_p) {
      cerr << "Memory allocation error in ReceiveWin::save_prompt()" << endl;
      exit(MEM_ERROR);
    }
    dialog_p->file_to_save.connect(SigC::bind(SigC::slot(this, &ReceiveWin::save), text));
  }
}

void ReceiveWin::save(string filename, string text) {

  if (!text.empty()) {
    ofstream file(filename.c_str(), ios::out);
    if (!file) {
      ostrstream strm;
      strm << filename << " cannot be opened" << ends;
      char* message = strm.str();
      new InfoDialog(message, "Save",
		     standard_size, InfoDialog::warning, top_level_window);
      delete[] message;
    }
    else {
      file << text;
      file.clear();
      file.close();
    }
  }
}

void ReceiveWin::write(const char* text, int stream, int port, Gtk::Text_Helpers::Context* context_p) {
  // context_p has a default value of 0

  ReceiveWinNode* node_p = node_ptr[stream][port];
  bool scrolling = node_p->is_scrolling();
  node_p->text.set_point(node_p->text.get_length());
  // we get a smoother display if we don't freeze() before inserting text
  // but we want to freeze() if we are scrolling to prevent the text window
  // scrolling out automatically when text is entered
  // however, we won't thaw() here, to avoid unnecessary screen flicker
  // we will thaw() in ReceiveWin::check_scroll_condition() only when we were
  // frozen because we were scrolling, but are now no longer scrolling
  // check_scroll_condition() is called by MainScreen::timer_event_handler()
  // (we will also thaw() whenever we scrollout())
  if (scrolling && current_p == node_p) node_p->freeze();
  if (context_p) node_p->text.insert(*context_p, text);
  else node_p->text.insert(text);

  // and now delete some text at the beginning if the text stored is getting too big
  // (we will only do this while not scrolling in order not to confuse the user)
  // Note:  the text doesn't scroll out when added to a window not being displayed,
  // so we also need to call scrollout() in EventSlots::vhf_menu(),
  // EventSlots::hf_menu() and EventSlots::read_cluster when changing streams
  // There is a further problem with Gtk+ 1.2.8, which can cause a seg fault if we try
  // to delete a block in a window which is not showing, so I have commented out the
  // next line and substituted something which only deletes the stream currently displayed
  //if (current_p != node_p || !scrolling) {
  if (current_p == node_p && !scrolling) {
    node_p->thaw(); // just check we are not frozen, in case we have just unscrolled
                    // but have got here before MainScreen::timer_event_handler() has thawed us
                    // thawing will ensure that node_p->text.get_length() will give the correct
                    // result
    while (node_p->text.get_length() > MAX_CHARS + DEL_BLOCK_SIZE) node_p->del_block();
  }
}

void ReceiveWin::del_letter(int stream, int port) {
  ReceiveWinNode* node_p = node_ptr[stream][port];
  if (node_p->text.has_selection()) {
    node_p->text.select_region(node_p->text.get_selection_start_pos(), node_p->text.get_selection_start_pos());
    disactivate_textselected_items();
  }
  node_p->text.set_point(node_p->text.get_length());
  node_p->text.backward_delete(1);
}

void ReceiveWin::raise(int stream, int port) {
  current_p->hide_all();
  parent.remove(*current_p);
  current_p = node_ptr[stream][port];
  parent.add2(*current_p);
  current_p->show_all();
}

void ReceiveWin::font_change(int size) {
  if (size >= 0 && size <= Fontsize::giant
      && !font_list[size].empty()) {
    
    int port;
    int stream;
	
    for (port = 0; port < 2; port++) {
      for (stream = 0; stream < MAXUSERS; stream++) {
	node_ptr[stream][port]->text.set_context(Gdk_Font(font_list[size]));
      }
    }
    red_context.set_font(Gdk_Font(font_list[size]));
    blue_context.set_font(Gdk_Font(font_list[size]));
    green_context.set_font(Gdk_Font(font_list[size]));
    purple_context.set_font(Gdk_Font(font_list[size]));
    if (tnc_p) scrollout(true);
  }
  else beep();
}
