/* Copyright (C) 1999, 2000, 2001, 2002 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

/*

The classes in this file provide the modal dialogs which the program
uses.  Modal dialogs do not allow other interaction with the program
while they are displayed.

The modal dialogs have a run() method which will block the event
loop and take over ownership of the dialog.  A dialog constructed on
the heap for which the run() method is called must be deleted
explictly by the user.

Where the run() method is not called, the dialog remains self-owning
and will delete itself when it closes, by calling `delete this'. This
is useful, as it enables dialogs to call dialogs (if run() is called
in a dialog created by a dialog which had also called run(), we would
get errors when unrecursing).  Because the dialog is self-owning, it
is also completely safe.  You should not store pointers to such
self-owning objects.  It is unnecessary and dangerous.  If run() is
not called, then the dialog must be created on the heap.  If run() is
called, then it can be a local (automatic) object.

*/

#ifndef MODAL_DIALOGS_H
#define MODAL_DIALOGS_H

#ifndef CONFIG_H
#define CONFIG_H
#include "config.h"
#endif

#include <string>
#include <gtk--/window.h>
#include <gtk--/button.h>
#include <gtk--/label.h>
#include <gtk--/table.h>
#include <gtk--/box.h>
#include <gtk--/frame.h>
#include <gtk--/pixmap.h>
#include <gtk--/entry.h>
#include <gtk--/text.h>
#include <gtk--/fileselection.h>
#include <gtk--/radiobutton.h>
#include <gtk--/checkbutton.h>
#include <gtk--/adjustment.h>
#include <gtk--/scale.h>
#include <sigc++/object.h>
#include <sigc++/signal_system.h>

#include "prog_defs.h"

class Tnc;

class PromptDialog: public Gtk::Window {
private:
  bool in_run_loop;
  Gtk::Button accept_button;
  Gtk::Button reject_button;
  Gtk::Label label;
  Gtk::Table table;
  Gtk::Window& parent;
  void selected(bool);
protected:
  virtual gint delete_event_impl(GdkEventAny*);
  virtual gint key_press_event_impl(GdkEventKey*);
public:
  SigC::Signal0<void> accepted;
  SigC::Signal0<void> rejected;
  void run(void);
  PromptDialog(const char* text, const char* caption, int standard_size, Gtk::Window& window, const char* accept_name = "OK", const char* reject_name = "Cancel");
};

class InfoDialog: public Gtk::Window {
private:
  bool in_run_loop;
  Gtk::Button ok_button;
  Gtk::Label label;
  Gtk::Table table;
  Gtk::Pixmap* pixmap_p;
  Gtk::Window& parent;
  void selected(void);
protected:
  virtual gint delete_event_impl(GdkEventAny*);
  virtual gint key_press_event_impl(GdkEventKey*);
public:
  enum InfoType {information, warning, critical};
  void run(void);
  InfoDialog(const char* text, const char* caption, int standard_size, InfoType info_type, Gtk::Window&);
  ~InfoDialog(void);
};

class ConnectDialog;

class ConnectEditor: public Gtk::Entry {
  Tnc* tnc_p;
  ConnectEditor(const char*, Tnc*); // private class
                                         //only ConnectDialog can create an object
protected:
  virtual gint key_press_event_impl(GdkEventKey*);
public:
  SigC::Signal1<void, bool> key_accept_pressed;
  friend class ConnectDialog;
};

class ConnectDialog: public Gtk::Window  {
  bool in_run_loop;
  Gtk::Button ok_button;
  Gtk::Button esc_button;
  Gtk::Label label;
  Gtk::Table table;
  Gtk::Window& parent;
  ConnectEditor* editbox_p;
  Tnc* tnc_p;
protected:
  virtual gint delete_event_impl(GdkEventAny*);
public:
  void selected(bool);
  SigC::Signal1<void, string> accepted;
  void run(void);
  ConnectDialog(Tnc*, int, Gtk::Window&);
  ~ConnectDialog(void);
};

class FileReadSelectDialog: public Gtk::FileSelection {
  bool in_run_loop;
  int standard_size;
  Gtk::Window& parent;
  void selected(bool);
protected:
  virtual gint delete_event_impl(GdkEventAny*);
public:
  SigC::Signal2<void, string, streampos> file_to_read; 
  void retry(void);
  void finish(void);
  void run(void);
  FileReadSelectDialog(int standard_size, Gtk::Window& window, const char* file_dir = 0);

};

class FileSaveSelectDialog: public Gtk::FileSelection {
  bool in_run_loop;
  int standard_size;
  Gtk::Window& parent;
  void selected(bool);
  void cancel(void);
protected:
  virtual gint delete_event_impl(GdkEventAny*);
public:
  SigC::Signal1<void, string> file_to_save; 
  void retry(void);
  void got_filename(string);
  void run(void);
  FileSaveSelectDialog(int standard_size, Gtk::Window& window, const char* file_dir = 0);
};

class CommandEditor;

class CommandDialog: public Gtk::Window {
public:
  enum action {rejected, accepted, script};
private:
  bool in_run_loop;
  int standard_size;
  Gtk::Button ok_button;
  Gtk::Button script_button;
  Gtk::Button esc_button;
  Gtk::Label label;
  Gtk::Table table;
  Gtk::Window& parent;
  CommandEditor* editbox_p;
protected:
  virtual gint delete_event_impl(GdkEventAny*);
public:
  void selected(action);
  void pass_file(string filename, streampos) {send_script(filename.c_str());}
  SigC::Signal1<void, string> send_command;
  SigC::Signal1<void, const char*> send_script;
  void run(void);
  CommandDialog(int,  Gtk::Window&);
};

class CommandEditor: public Gtk::Entry {
  int count;
  CommandEditor(void): count(0) {} // private class
                                   //only CommandDialog can create an object
protected:
  virtual gint key_press_event_impl(GdkEventKey*);
public:
  SigC::Signal1<void, CommandDialog::action> key_pressed;
  friend class CommandDialog;
};

class CallsignDialog;

class CallsignEditor: public Gtk::Entry {
    
  Tnc* tnc_p;
  CallsignEditor(Tnc* tnc): tnc_p(tnc) {} // private class
                                          // only CallsignDialog can create an object
protected:
  virtual gint key_press_event_impl(GdkEventKey*);
public:
  SigC::Signal1<void, bool> key_accept_pressed;
  friend class CallsignDialog;
};

class CallsignDialog: public Gtk::Window {
  bool in_run_loop;
  Gtk::Button ok_button;
  Gtk::Button esc_button;
  Gtk::Label label;
  Gtk::Table table;
  Gtk::Window& parent;
  CallsignEditor editbox;
protected:
  virtual gint delete_event_impl(GdkEventAny*);
public:
  void selected(bool);
  SigC::Signal1<void, string> accepted;
  void run(void);
  CallsignDialog(Tnc*, int, Gtk::Window&);
  ~CallsignDialog(void) {}
};

class RttySpeedDialog: public Gtk::Window {
public:
  enum Rtty_speed {b45, b50, b57, b75};
private:
  bool in_run_loop;
  Rtty_speed speed;
  Gtk::Button ok_button;
  Gtk::Button esc_button;
  Gtk::RadioButton baud_45;
  Gtk::RadioButton baud_50;
  Gtk::RadioButton baud_57;
  Gtk::RadioButton baud_75;
  Gtk::VBox vbox;
  Gtk::Table table;
  Gtk::Window& parent;
  void speed_button_pressed(Rtty_speed);
  void selected(bool);
protected:
  virtual gint delete_event_impl(GdkEventAny*);
  virtual gint key_press_event_impl(GdkEventKey*);
public:
  SigC::Signal1<void, Rtty_speed> accepted;
  void run(void);
  RttySpeedDialog(Rtty_speed speed_, int standard_size, Gtk::Window& window);
};

class AsciiSpeedDialog: public Gtk::Window {
public:
  enum Ascii_speed {b50, b100, b110, b200};
private:
  bool in_run_loop;
  Ascii_speed speed;
  Gtk::Button ok_button;
  Gtk::Button esc_button;
  Gtk::RadioButton baud_50;
  Gtk::RadioButton baud_100;
  Gtk::RadioButton baud_110;
  Gtk::RadioButton baud_200;
  Gtk::VBox vbox;
  Gtk::Table table;
  Gtk::Window& parent;
  void speed_button_pressed(Ascii_speed);
  void selected(bool);
protected:
  virtual gint delete_event_impl(GdkEventAny*);
  virtual gint key_press_event_impl(GdkEventKey*);
public:
  SigC::Signal1<void, Ascii_speed> accepted;
  void run(void);
  AsciiSpeedDialog(Ascii_speed speed_, int standard_size, Gtk::Window& window);
};

class CwSpeedDialog: public Gtk::Window {
  bool in_run_loop;
  int speed;
  Gtk::Button ok_button;
  Gtk::Button esc_button;
  Gtk::Label label;
  Gtk::Label speed_label;
  Gtk::Table table;
  Gtk::Adjustment adjustment;
  Gtk::HScale slider;
  Gtk::HBox hbox;
  Gtk::Window& parent;
  void update_speed(void);
  void selected(bool);
protected:
  virtual gint delete_event_impl(GdkEventAny*);
  virtual gint key_press_event_impl(GdkEventKey*);
public:
  SigC::Signal1<void, int> accepted;
  void run(void);
  CwSpeedDialog(int speed_, int standard_size, Gtk::Window& window);
};

class PrintMarkDialog: public Gtk::Window {
public:
  enum Result {accepted, rejected, cancel_job};
private:
  bool in_run_loop;
  Gtk::Button ok_button;
  Gtk::Button no_button;
  Gtk::Button cancel_button;
  Gtk::Label label;
  Gtk::Table table;
  Gtk::Window& parent;
  void selected(Result);
protected:
  virtual gint delete_event_impl(GdkEventAny*);
  virtual gint key_press_event_impl(GdkEventKey*);
public:
  SigC::Signal1<void, Result> out_selection;
  void run(void);
  PrintMarkDialog(int standard_size, Gtk::Window& window);
};

class GplDialog;

class GplEditor: public Gtk::Text {
  GplEditor(void) {} // private class
                     // only GplDialog can create an object
  friend class GplDialog;
};


// the run() method must be called on this dialog so that the result can be obtained
// (the result is in the return value of run())
class GplDialog: public Gtk::Window {
public:
  enum Result {rejected, accepted};
private:
  Result result;
  GplEditor editbox;
  Gtk::Button accept_button;
  Gtk::Button reject_button;
  Gtk::Label label;
  Gtk::Table table;
  void selected(Result);
protected:
  virtual gint delete_event_impl(GdkEventAny*);
  virtual gint key_press_event_impl(GdkEventKey*);
public:
  int run(void);
  GplDialog(int standard_size, int fontsize);
};

class Autocq_modeDialog: public Gtk::Window {
  bool in_run_loop;
  Gtk::Button ok_button;
  Gtk::Button esc_button;
  Gtk::RadioButton amtor_button;
  Gtk::RadioButton pactor_button;
  Gtk::Label amtor_label;
  Gtk::Label pactor_label;
  Gtk::HBox amtor_box;
  Gtk::HBox pactor_box;
  Gtk::Table table;
  Gtk::VBox frame_vbox;
  Gtk::Frame frame;
  Gtk::Window& parent;
  void button_pressed(void);
  void selected(bool);
protected:
  virtual gint delete_event_impl(GdkEventAny*);
  virtual gint key_press_event_impl(GdkEventKey*);
public:
  SigC::Signal0<void> accepted;
  SigC::Signal0<void> rejected;
  void run(void);
  Autocq_modeDialog(int standard_size, Gtk::Window& window);
};

class RstDialog;

class RstEditor: public Gtk::Entry {
  int count;
  RstEditor(const char*); // private class
                          // only RstDialog can create an object
protected:
  virtual gint key_press_event_impl(GdkEventKey*);
public:
  SigC::Signal1<void, bool> key_accept_pressed;
  friend class RstDialog;
};

class RstDialog: public Gtk::Window  {
  bool in_run_loop;
  Gtk::Button ok_button;
  Gtk::Button esc_button;
  Gtk::Label label;
  Gtk::Table table;
  Gtk::Window& parent;
  RstEditor editbox;
protected:
  virtual gint delete_event_impl(GdkEventAny*);
public:
  void selected(bool);
  SigC::Signal1<void, string> accepted;
  void run(void);
  RstDialog(int, Gtk::Window&);
};

// the run() method must be called on this dialog so that the program does not exit until dialog closed
class ComErrorDialog: public Gtk::Window {
private:
  Gtk::Button ok_button;
  Gtk::Label label;
  Gtk::Table table;
  Gtk::Pixmap* pixmap_p;
  void selected(void);
protected:
  virtual gint delete_event_impl(GdkEventAny*);
  virtual gint key_press_event_impl(GdkEventKey*);
public:
  void run(void);
  ComErrorDialog(const char* text, int standard_size);
  ~ComErrorDialog(void);
};

#endif
