/* Copyright (C) 1999, 2000, 2001 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <unistd.h>
#include <iostream>
#include <fstream>
#include <strstream>
#include <cstring>
#include <cstdlib>
#include <gtk--/main.h>
#include <gtk--/scrollbar.h>
#include <gtk--/style.h>
#include <gdk/gdkkeysyms.h> // the key codes are here
#include <X11/X.h>          // the state masks are here



#include "modal_dialogs.h"
#include "tnc.h"
#include "gpl.h"
#include "info_icons.h"

#define COMMAND_SIZE 100

PromptDialog::PromptDialog(const char* text, const char* caption, int standard_size, Gtk::Window& window,
			   const char* accept_name, const char* reject_name):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
			     accept_button(accept_name), reject_button(reject_name),
			     label(text), table(2, 2, false), parent(window) {
  table.attach(label, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(accept_button, 0, 1, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(reject_button, 1, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  accept_button.clicked.connect(SigC::bind(SigC::slot(this, &PromptDialog::selected), true));
  reject_button.clicked.connect(SigC::bind(SigC::slot(this, &PromptDialog::selected), false));

  add(table);
  
  set_title(caption);
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  accept_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  reject_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  set_border_width(standard_size/2);
  accept_button.grab_focus();
  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void PromptDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void PromptDialog::selected(bool accept) {
  parent.set_sensitive(true); // do this before we emit accepted()
  hide_all();
  if (accept) accepted();
  else rejected();
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint PromptDialog::delete_event_impl(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

gint PromptDialog::key_press_event_impl(GdkEventKey* event_p) {
  if (event_p->keyval == GDK_Escape) selected(false);
  if (event_p->keyval == GDK_Return || event_p->keyval == GDK_Tab) Gtk::Window::key_press_event_impl(event_p);
  return false;
}

InfoDialog::InfoDialog(const char* text, const char* caption, int standard_size, InfoType info_type,
		       Gtk::Window& window):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
			     ok_button("OK"), label(text),
			     table(2, 2, false), parent(window) {
  if (info_type == information) pixmap_p = new Gtk::Pixmap(information_xpm);
  else if (info_type == warning) pixmap_p = new Gtk::Pixmap(warning_xpm);
  else pixmap_p = new Gtk::Pixmap(critical_xpm);
  if (!pixmap_p) {
    cerr << "Memory allocation error in InfoDialog::InfoDialog()" << endl;
    exit(MEM_ERROR);
  }

  table.attach(*pixmap_p, 0, 1, 0, 1, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(label, 1, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(ok_button, 0, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  ok_button.clicked.connect(SigC::slot(this, &InfoDialog::selected));

  add(table);
  
  set_title(caption);
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  ok_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  set_border_width(standard_size/2);

  ok_button.grab_focus();

  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void InfoDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void InfoDialog::selected(void) {
  parent.set_sensitive(true);
  hide_all();
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint InfoDialog::delete_event_impl(GdkEventAny*) {
  selected();
  return true; // returning true prevents destroy sig being emitted
}

gint InfoDialog::key_press_event_impl(GdkEventKey* event_p) {
  if (event_p->keyval == GDK_Return || event_p->keyval == GDK_Escape) selected();
  return false;
}

InfoDialog::~InfoDialog(void) {
  delete pixmap_p;
}

ConnectDialog::ConnectDialog(Tnc* tnc_p_, int standard_size, Gtk::Window& window):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
			     ok_button("OK"), esc_button("Cancel"),
			     label("Station to connect with?"), table(3, 2, false),
                             parent(window), tnc_p(tnc_p_) {
    string connect_name;
    if (tnc_p->tnc_func.active_port && (tnc_p->tnc_func.hfmode == Tnc_func::amtor
					|| tnc_p->tnc_func.hfmode == Tnc_func::lamtor
					|| tnc_p->tnc_func.hfmode == Tnc_func::fec)) {
        connect_name = tnc_p->tnc_func.selCall;
    }
    else {
        connect_name = tnc_p->tnc_func.hisCall[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port];
    }

  editbox_p = new ConnectEditor(connect_name.c_str(), tnc_p);
  if (!editbox_p) {
    cerr << "Memory allocation error in ConnectDialog::ConnectDialog()" << endl;
    exit(MEM_ERROR);
  }

  table.attach(label, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(*editbox_p, 0, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(ok_button, 0, 1, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(esc_button, 1, 2, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  ok_button.clicked.connect(SigC::bind(SigC::slot(this, &ConnectDialog::selected), true));
  esc_button.clicked.connect(SigC::bind(SigC::slot(this, &ConnectDialog::selected), false));
  editbox_p->key_accept_pressed.connect(SigC::slot(this, &ConnectDialog::selected));

  add(table);
  
  set_title("Connect");
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  ok_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  esc_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  editbox_p->set_usize(standard_size * 4, standard_size);
  set_border_width(standard_size/2);

  editbox_p->grab_focus();

  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void ConnectDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void ConnectDialog::selected(bool accept) {
  parent.set_sensitive(true); // do this before we emit accepted()
  hide_all();
  if (accept) accepted(editbox_p->get_text());
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint ConnectDialog::delete_event_impl(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

ConnectDialog::~ConnectDialog(void) {
  delete editbox_p;
}

ConnectEditor::ConnectEditor(const char* text, Tnc* tnc): tnc_p(tnc) {
  set_text(text);
}

gint ConnectEditor::key_press_event_impl(GdkEventKey* event_p) {

  int keycode = event_p->keyval;
  char* letter = event_p->string;
  bool marked_text = false;
  bool first_letter_marked = false;
  bool valid_flag = false;

  if (has_selection()) {
    marked_text = true;
    if (!get_selection_start_pos() || !get_selection_end_pos()) first_letter_marked = true;
  }
  
  if (keycode == GDK_Return) {
    if (!get_text().empty() && !marked_text) key_accept_pressed(true);
    else beep();
  }
  else if (keycode == GDK_Escape) key_accept_pressed(false);
  else if (keycode == GDK_BackSpace
	   && (get_position() != 0 || marked_text)) {
    valid_flag = true;
  }
  else if (keycode == GDK_Delete
	   && (get_position() < static_cast<gint>(get_text().size()) || marked_text)) {
    valid_flag = true;
  }
  
  else if (keycode == GDK_Home || keycode == GDK_End) {
    valid_flag = true;
  }

  else if (keycode == GDK_Right && get_position() < static_cast<gint>(get_text().size())) {
    set_position(get_position() + 1);
  }

  else if (keycode == GDK_Left && get_position()) {
    set_position(get_position() - 1);
  }
  
  else if (keycode == GDK_Tab) {
    valid_flag = true;
  }

  else if (keycode < 256 && isalnum(static_cast<unsigned char> (keycode))) {
    *letter = toupper(static_cast<unsigned char> (*letter));
    keycode = toupper(static_cast<unsigned char> (keycode));
    valid_flag = true;
  }

  else if ((*letter == '/' && get_position() && !first_letter_marked)
	   || (*letter == '!'
	       && (get_position() == 0 || first_letter_marked)
	       && tnc_p->tnc_func.active_port
	       && (tnc_p->tnc_func.hfmode == Tnc_func:: pactor
		   || tnc_p->tnc_func.hfmode == Tnc_func:: gtor))
	   || ((*letter == ',' || *letter == ' ' || *letter == '-')
	       && !first_letter_marked
	       && get_position()
	       && (!tnc_p->tnc_func.active_port
		   || tnc_p->tnc_func.hfmode == Tnc_func::packet))) {
    valid_flag = true;
  }
    
  if (valid_flag) {
    event_p->keyval = keycode;
    Gtk::Entry::key_press_event_impl(event_p);
  }
  if (valid_flag) return false;
  return true;
}

FileReadSelectDialog::FileReadSelectDialog(int size, Gtk::Window& window, const char* file_dir):
                                           Gtk::FileSelection("Open file"), in_run_loop(false),
					   standard_size(size), parent(window) {
  get_ok_button()->clicked.connect(SigC::bind(SigC::slot(this, &FileReadSelectDialog::selected), true));
  get_cancel_button()->clicked.connect(SigC::bind(SigC::slot(this, &FileReadSelectDialog::selected), false));

  hide_fileop_buttons();

  string filename;
  if (file_dir) filename = file_dir;
  else {
    const char* home = getenv("HOME");
    if (home) filename = home;
  }
  filename += '/';
  set_filename(filename);
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);
  set_position(GTK_WIN_POS_CENTER);

  show_all();
}

void FileReadSelectDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void FileReadSelectDialog::selected(bool accept) {
  if (accept) {
    string filename = get_filename();
    if (filename.size()) {
      ifstream file;
#ifdef HAVE_IOS_NOCREATE
      file.open(filename.c_str(), ios::in | ios::nocreate); // we just want to find the file size
#else
      // we must have Std C++ so we probably don't need a ios::nocreate
      // flag on a read open to ensure uniqueness
      file.open(filename.c_str(), ios::in); // we just want to find the file size
#endif
      if (!file) {
	file.clear();
	ostrstream strm;
	strm << filename.c_str() << " cannot be found/opened" << ends;
	char* message = strm.str();
	PromptDialog* dialog_p = new PromptDialog(message, "Upload", standard_size, *this, "Retry");
	if (!dialog_p) {
	  cerr << "Memory allocation error in FileReadSelectDialog::selected()" << endl;
	  exit(MEM_ERROR);
	}
	dialog_p->accepted.connect(SigC::slot(this, &FileReadSelectDialog::retry));
	dialog_p->rejected.connect(SigC::slot(this, &FileReadSelectDialog::finish));
	delete[] message;
	// we will not set up a separate event loop for the PromptDialog dialog -- if we do
	// the dialogs will not end their loops in the correct order
	// instead, the dialog will be self-owning and delete itself
      }
      else {
        file.seekg(0, ios::end);
	streampos size = file.tellg();
	file.close(); // got the file size - now close it
	file_to_read(filename, size);
	finish();
      }
    }
    else {
      beep();
      retry();
    }
  }
  else finish();
}

void FileReadSelectDialog::retry(void) {
  string filename(prog_func.filedir);
  filename += '/';
  set_filename(filename);
}

void FileReadSelectDialog::finish(void) {
  parent.set_sensitive(true);
  hide_all();
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint FileReadSelectDialog::delete_event_impl(GdkEventAny*) {
  finish();
  return true; // returning true prevents destroy sig being emitted
}

FileSaveSelectDialog::FileSaveSelectDialog(int size, Gtk::Window& window, const char* file_dir):
                                           Gtk::FileSelection("Save file"), in_run_loop(false),
					   standard_size(size), parent(window) {
  get_ok_button()->clicked.connect(SigC::bind(SigC::slot(this, &FileSaveSelectDialog::selected), true));
  get_cancel_button()->clicked.connect(SigC::bind(SigC::slot(this, &FileSaveSelectDialog::selected), false));

  hide_fileop_buttons();

  string filename;
  if (file_dir) filename = file_dir;
  else {
    const char* home = getenv("HOME");
    if (home) filename = home;
  }
  filename += '/';
  set_filename(filename);
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);
  set_position(GTK_WIN_POS_CENTER);

  show_all();
}

void FileSaveSelectDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void FileSaveSelectDialog::selected(bool accept) {
  if (accept) {
    string filename = get_filename();
    if (filename.size()) {
      if (!access(filename.c_str(), F_OK)) { // file exists, check whether to overwrite
	ostrstream strm;
	strm << "File " << filename.c_str() << " already exists\nOverwrite it?" << ends;
	char* message = strm.str();
	PromptDialog* dialog_p = new PromptDialog(message, "Download", standard_size, *this,
						  "OK", "Retry");
	if (!dialog_p) {
	  cerr << "Memory allocation error in FileSaveSelectDialog::selected()" << endl;
	  exit(MEM_ERROR);
	}
	dialog_p->accepted.connect(SigC::bind(SigC::slot(this, &FileSaveSelectDialog::got_filename), filename));
	dialog_p->rejected.connect(SigC::slot(this, &FileSaveSelectDialog::retry));
	delete[] message;
	// we will not set up a separate event loop for the PromptDialog dialog -- if we do
	// the dialogs will not end their loops in the correct order
	// instead, the dialog will be self-owning and delete itself
      }
      else got_filename(filename);
    }
    else {
      beep();
      retry();
    }
  }
  else cancel();
}

void FileSaveSelectDialog::retry(void) {
  string filename(prog_func.filedir);
  filename += '/';
  set_filename(filename);
}

void FileSaveSelectDialog::got_filename(string filename) {
  parent.set_sensitive(true);
  hide_all();
  file_to_save(filename);
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

void FileSaveSelectDialog::cancel(void) {
  parent.set_sensitive(true);
  hide_all();
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint FileSaveSelectDialog::delete_event_impl(GdkEventAny*) {
  cancel();
  return true; // returning true prevents destroy sig being emitted
}

CommandDialog::CommandDialog(int size, Gtk::Window& window):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false), standard_size(size), 
			     ok_button("OK"), script_button("Script"), esc_button("Cancel"),
			     label("Kam command to send?\n"
				   "(Press F5 or script button to send\nconfiguration script)"),
			     table(3, 3, false), parent(window) {

  editbox_p = new CommandEditor;
  if (!editbox_p) {
    cerr << "Memory allocation error in CommandDialog::CommandDialog()" << endl;
    exit(MEM_ERROR);
  }

  table.attach(label, 0, 3, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(*editbox_p, 0, 3, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(ok_button, 0, 1, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(script_button, 1, 2, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(esc_button, 2, 3, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  ok_button.clicked.connect(SigC::bind(SigC::slot(this, &CommandDialog::selected), accepted));
  script_button.clicked.connect(SigC::bind(SigC::slot(this, &CommandDialog::selected), script));
  esc_button.clicked.connect(SigC::bind(SigC::slot(this, &CommandDialog::selected), rejected));
  editbox_p->key_pressed.connect(SigC::slot(this, &CommandDialog::selected));

  add(table);
  
  set_title("Command");
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  ok_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  script_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  esc_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  editbox_p->set_usize(standard_size * 5, standard_size);
  set_border_width(standard_size/2);

  editbox_p->grab_focus();

  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void CommandDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void CommandDialog::selected(action choice) {
  if (choice == script) {
    string parmsdir(prog_func.filedir);
    parmsdir += "/parms";
    FileReadSelectDialog* dialog_p = new FileReadSelectDialog(standard_size, *this, parmsdir.c_str());
    if (!dialog_p) {
      cerr << "Memory allocation error in CommandDialog::selected()" << endl;
      exit(MEM_ERROR);
    }
    dialog_p->file_to_read.connect(SigC::slot(this, &CommandDialog::pass_file));
    dialog_p->destroy.connect(SigC::bind(SigC::slot(this, &CommandDialog::selected), rejected));
    dialog_p->run();
    delete dialog_p;
  }
  else {
    parent.set_sensitive(true);
    hide_all();
    if (choice == accepted) send_command(editbox_p->get_text());
    if (in_run_loop) Gtk::Main::quit();
    // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
    else delete this;
  }
}

gint CommandDialog::delete_event_impl(GdkEventAny*) {
  selected(rejected);
  return true; // returning true prevents destroy sig being emitted
}

gint CommandEditor::key_press_event_impl(GdkEventKey* event_p) {;

  int keycode = event_p->keyval;
  char* letter = event_p->string;
  bool marked_text = false;
  bool valid_flag = false;

  if (has_selection()) marked_text = true;
  
  if (keycode == GDK_Return && count && !marked_text) key_pressed(CommandDialog::accepted);
  else if (keycode == GDK_Escape) key_pressed(CommandDialog::rejected);
  else if (keycode == GDK_F5) key_pressed(CommandDialog::script);
  else if (keycode == GDK_BackSpace
	   && (get_position() != 0 || marked_text)) {
    if (count) count--;
    valid_flag = true;
  }
  else if (keycode == GDK_Delete
	   && (get_position() < count || marked_text)) {
    if (count) count--;
    valid_flag = true;
  }
  
  else if (keycode == GDK_Home || keycode == GDK_End) {
    valid_flag = true;
  }

  else if (keycode == GDK_Right && get_position() < count) {
    set_position(get_position() + 1);
  }

  else if (keycode == GDK_Left && get_position()) {
    set_position(get_position() - 1);
  }
  
  else if (keycode == GDK_Tab) {
    valid_flag = true;
  }

  else if (keycode >= 32 && keycode <= 127){
    if (count < COMMAND_SIZE || marked_text) {
      count++;
      *letter = toupper(static_cast<unsigned char> (*letter));
      keycode = toupper(static_cast<unsigned char> (keycode));
      valid_flag = true;
    }
    else beep();
  }

  if (valid_flag) {
    event_p->keyval = keycode;
    Gtk::Entry::key_press_event_impl(event_p);
    if (marked_text) count = get_text().size();  // if we deleted some marked text reset count
  }
  if (valid_flag) return false;
  return true;
}

CallsignDialog::CallsignDialog(Tnc* tnc_p, int standard_size, Gtk::Window& window):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false), ok_button("OK"),
			     esc_button("Cancel"), label("Callsign?"), table(3, 2, false),
			     parent(window), editbox(tnc_p) {

  table.attach(label, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(editbox, 0, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(ok_button, 0, 1, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(esc_button, 1, 2, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  ok_button.clicked.connect(SigC::bind(SigC::slot(this, &CallsignDialog::selected), true));
  esc_button.clicked.connect(SigC::bind(SigC::slot(this, &CallsignDialog::selected), false));
  editbox.key_accept_pressed.connect(SigC::slot(this, &CallsignDialog::selected));

  add(table);
  
  set_title("Callsign");
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  ok_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  esc_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  editbox.set_usize(standard_size * 4, standard_size);
  set_border_width(standard_size/2);

  editbox.grab_focus();

  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void CallsignDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void CallsignDialog::selected(bool accept) {
  parent.set_sensitive(true); // do this before we emit accepted()
  hide_all();
  if (accept) accepted(editbox.get_text());
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint CallsignDialog::delete_event_impl(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

gint CallsignEditor::key_press_event_impl(GdkEventKey* event_p) {

  int keycode = event_p->keyval;
  char* letter = event_p->string;
  bool marked_text = false;
  bool first_letter_marked = false;
  bool valid_flag = false;

  if (has_selection()) {
    marked_text = true;
    if (!get_selection_start_pos() || !get_selection_end_pos()) first_letter_marked = true;
  }
  
  if (keycode == GDK_Return) {
    if (!get_text().empty() && !marked_text) key_accept_pressed(true);
    else beep();
  }
  else if (keycode == GDK_Escape) key_accept_pressed(false);
  else if (keycode == GDK_BackSpace
	   && (get_position() != 0 || marked_text)) {
    valid_flag = true;
  }
  else if (keycode == GDK_Delete
	   && (get_position() < static_cast<gint>(get_text().size()) || marked_text)) {
    valid_flag = true;
  }
  
  else if (keycode == GDK_Home || keycode == GDK_End) {
    valid_flag = true;
  }

  else if (keycode == GDK_Right && get_position() < static_cast<gint>(get_text().size())) {
    set_position(get_position() + 1);
  }

  else if (keycode == GDK_Left && get_position()) {
    set_position(get_position() - 1);
  }
  
  else if (keycode == GDK_Tab) {
    valid_flag = true;
  }

  else if (keycode < 256 && isalnum(static_cast<unsigned char> (keycode))) {
    *letter = toupper(static_cast<unsigned char> (*letter));
    keycode = toupper(static_cast<unsigned char> (keycode));
    valid_flag = true;
  }

  else if ((*letter == '/' && get_position() && !first_letter_marked)
	   || (*letter == '-'
	       && !first_letter_marked
	       && get_position()
	       && (!tnc_p->tnc_func.active_port
		   || tnc_p->tnc_func.hfmode == Tnc_func::packet))) {
    valid_flag = true;
  }
    
  if (valid_flag) {
    event_p->keyval = keycode;
    Gtk::Entry::key_press_event_impl(event_p);
  }
  if (valid_flag) return false;
  return true;
}

RttySpeedDialog::RttySpeedDialog(Rtty_speed speed_, int standard_size, Gtk::Window& window):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
                             speed(speed_), ok_button("OK"), esc_button("Cancel"),
                             baud_45("45 baud"), baud_50("50 baud"),
			     baud_57("57 baud"), baud_75("75 baud"),
                             table(2, 2, false), parent(window) {

  baud_50.set_group(baud_45.group());
  baud_57.set_group(baud_45.group());
  baud_75.set_group(baud_45.group());

  vbox.pack_start(baud_45, false, false, 0);
  vbox.pack_start(baud_50, false, false, 0);
  vbox.pack_start(baud_57, false, false, 0);
  vbox.pack_start(baud_75, false, false, 0);

  if (speed == b45) baud_45.set_active(true);
  else if (speed == b50) baud_50.set_active(true);
  else if (speed == b57) baud_57.set_active(true);
  else if (speed == b75) baud_75.set_active(true);
    
  table.attach(vbox, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(ok_button, 0, 1, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(esc_button, 1, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  baud_45.clicked.connect(SigC::bind(SigC::slot(this, &RttySpeedDialog::speed_button_pressed), b45));
  baud_50.clicked.connect(SigC::bind(SigC::slot(this, &RttySpeedDialog::speed_button_pressed), b50));
  baud_57.clicked.connect(SigC::bind(SigC::slot(this, &RttySpeedDialog::speed_button_pressed), b57));
  baud_75.clicked.connect(SigC::bind(SigC::slot(this, &RttySpeedDialog::speed_button_pressed), b75));

  ok_button.clicked.connect(SigC::bind(SigC::slot(this, &RttySpeedDialog::selected), true));
  esc_button.clicked.connect(SigC::bind(SigC::slot(this, &RttySpeedDialog::selected), false));

  add(table);
  
  set_title("RTTY Speed");
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  ok_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  esc_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  set_border_width(standard_size/2);
  ok_button.grab_focus();
  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void RttySpeedDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void RttySpeedDialog::speed_button_pressed(Rtty_speed speed_button) {
  speed = speed_button;
  ok_button.grab_focus();
}

void RttySpeedDialog::selected(bool accept) {
  parent.set_sensitive(true); // do this before we emit accepted()
  hide_all();
  if (accept) accepted(speed);
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint RttySpeedDialog::delete_event_impl(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

gint RttySpeedDialog::key_press_event_impl(GdkEventKey* event_p) {
  if (event_p->keyval == GDK_Escape) selected(false);
  if (event_p->keyval == GDK_Return || event_p->keyval == GDK_Tab) Gtk::Window::key_press_event_impl(event_p);
  return false;
}

AsciiSpeedDialog::AsciiSpeedDialog(Ascii_speed speed_, int standard_size, Gtk::Window& window):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
                             speed(speed_),  ok_button("OK"), esc_button("Cancel"),
                             baud_50("50 baud"), baud_100("100 baud"),
			     baud_110("110 baud"), baud_200("200 baud"),
                             table(2, 2, false), parent(window) {

  baud_100.set_group(baud_50.group());
  baud_110.set_group(baud_50.group());
  baud_200.set_group(baud_50.group());

  vbox.pack_start(baud_50, false, false, 0);
  vbox.pack_start(baud_100, false, false, 0);
  vbox.pack_start(baud_110, false, false, 0);
  vbox.pack_start(baud_200, false, false, 0);

  if (speed == b50) baud_50.set_active(true);
  else if (speed == b100) baud_100.set_active(true);
  else if (speed == b110) baud_110.set_active(true);
  else if (speed == b200) baud_200.set_active(true);
    
  table.attach(vbox, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(ok_button, 0, 1, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(esc_button, 1, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  baud_50.clicked.connect(SigC::bind(SigC::slot(this, &AsciiSpeedDialog::speed_button_pressed), b50));
  baud_100.clicked.connect(SigC::bind(SigC::slot(this, &AsciiSpeedDialog::speed_button_pressed), b100));
  baud_110.clicked.connect(SigC::bind(SigC::slot(this, &AsciiSpeedDialog::speed_button_pressed), b110));
  baud_200.clicked.connect(SigC::bind(SigC::slot(this, &AsciiSpeedDialog::speed_button_pressed), b200));

  ok_button.clicked.connect(SigC::bind(SigC::slot(this, &AsciiSpeedDialog::selected), true));
  esc_button.clicked.connect(SigC::bind(SigC::slot(this, &AsciiSpeedDialog::selected), false));

  add(table);
  
  set_title("ASCII Speed");
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  ok_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  esc_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  set_border_width(standard_size/2);
  ok_button.grab_focus();
  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void AsciiSpeedDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void AsciiSpeedDialog::speed_button_pressed(Ascii_speed speed_button) {
  speed = speed_button;
  ok_button.grab_focus();
}

void AsciiSpeedDialog::selected(bool accept) {
  parent.set_sensitive(true); // do this before we emit accepted()
  hide_all();
  if (accept) accepted(speed);
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint AsciiSpeedDialog::delete_event_impl(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

gint AsciiSpeedDialog::key_press_event_impl(GdkEventKey* event_p) {
  if (event_p->keyval == GDK_Escape) selected(false);
  if (event_p->keyval == GDK_Return || event_p->keyval == GDK_Tab) Gtk::Window::key_press_event_impl(event_p);
  return false;
}

CwSpeedDialog::CwSpeedDialog(int speed_, int standard_size, Gtk::Window& window):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
			     speed(speed_), ok_button("OK"), esc_button("Cancel"),
			     label("New Speed (5 - 50 WPM)?"), table(3, 2, false),
			     adjustment(speed_, 5, 50, 5, 5),
			     parent(window) {

  slider.set_adjustment(&adjustment);
  slider.set_usize(standard_size * 5, standard_size);
  slider.set_update_policy(GTK_UPDATE_CONTINUOUS);
  slider.set_draw_value(false);

  speed_label.set_usize(standard_size, standard_size);
  ostrstream strm;
  strm << speed_ << ends;
  const char* text = strm.str();
  speed_label.set_text(text);
  delete[] text;

  hbox.pack_start(label, true, false, 0);
  hbox.pack_end(speed_label, true, false, 10);
  
  table.attach(hbox, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(slider, 0, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(ok_button, 0, 1, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(esc_button, 1, 2, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);


  ok_button.clicked.connect(SigC::bind(SigC::slot(this, &CwSpeedDialog::selected), true));
  esc_button.clicked.connect(SigC::bind(SigC::slot(this, &CwSpeedDialog::selected), false));
  adjustment.value_changed.connect(SigC::slot(this, &CwSpeedDialog::update_speed));

  add(table);
  
  set_title("CW Speed");
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  ok_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  esc_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  set_border_width(standard_size/2);

  ok_button.grab_focus();

  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void CwSpeedDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void CwSpeedDialog::selected(bool accept) {
  parent.set_sensitive(true); // do this before we emit accepted()
  hide_all();
  if (accept) accepted(speed);
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

void CwSpeedDialog::update_speed(void) {
  int input_value = static_cast<int>(adjustment.get_value());
  int old_speed = speed;
  speed = ((input_value + 2)/5) * 5;
 
  if (old_speed != speed) {
    ostrstream strm;
    strm << speed << ends;
    const char* text = strm.str();
    speed_label.set_text(text);
    delete[] text;
  }
}

gint CwSpeedDialog::delete_event_impl(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

gint CwSpeedDialog::key_press_event_impl(GdkEventKey* event_p) {
  if (event_p->keyval == GDK_Escape) selected(false);
  if (event_p->keyval == GDK_Return || event_p->keyval == GDK_Tab) Gtk::Window::key_press_event_impl(event_p);
  return false;
}

PrintMarkDialog::PrintMarkDialog(int standard_size, Gtk::Window& window):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
			     ok_button("Ok"), no_button("No"), cancel_button("Cancel job"),
			     label("Print from the mark?\n"
				 "(Press Cancel button to cancel the print job)"),
			     table(2, 3, false), parent(window) {
  table.attach(label, 0, 3, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(ok_button, 0, 1, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(no_button, 1, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(cancel_button, 2, 3, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  ok_button.clicked.connect(SigC::bind(SigC::slot(this, &PrintMarkDialog::selected), accepted));
  no_button.clicked.connect(SigC::bind(SigC::slot(this, &PrintMarkDialog::selected), rejected));
  cancel_button.clicked.connect(SigC::bind(SigC::slot(this, &PrintMarkDialog::selected), cancel_job));

  add(table);
  
  set_title("Print from mark");
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  ok_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  no_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  cancel_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);

  set_border_width(standard_size/2);
  ok_button.grab_focus();
  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void PrintMarkDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void PrintMarkDialog::selected(Result result) {
  parent.set_sensitive(true); // do this before we emit accepted()
  hide_all();
  if (result != rejected) out_selection(result);
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint PrintMarkDialog::delete_event_impl(GdkEventAny*) {
  selected(rejected);
  return true; // returning true prevents destroy sig being emitted
}

gint PrintMarkDialog::key_press_event_impl(GdkEventKey* event_p) {
  if (event_p->keyval == GDK_Escape) selected(rejected);
  if (event_p->keyval == GDK_Return || event_p->keyval == GDK_Tab) Gtk::Window::key_press_event_impl(event_p);
  return false;
}

GplDialog::GplDialog(int standard_size, int fontsize):
                             Gtk::Window(GTK_WINDOW_DIALOG), result(rejected),
			     accept_button("Accept"), reject_button("Reject"),
		             label("Do you accept the Conditions, Notices and Disclaimers shown above?"),
			     table(3, 3, false) {

  if (fontsize >=10 && fontsize <= 12) {
    Gtk::Style* new_style_p = get_style(); // this will grab the global style
                                           // (we did not use get_style()->copy())
                                           // accordingly there is no need to call set_style(*new_style_p)
                                           // after we have called set_font()
    ostrstream strm;
    strm << "-adobe-helvetica-medium-r-normal--" << fontsize << "-*-*-*-*-*-iso8859-1" << ends;
    const char* font_name = strm.str();
    new_style_p->set_font(Gdk_Font(font_name));
    delete[] font_name;
  }

  editbox.set_editable(false);
  editbox.set_context(Gdk_Font(font_list[Fontsize::medium]));
  editbox.freeze();
  editbox.insert(copyright_msg);
  editbox.thaw();
  Gtk::Scrollbar* scrollbar_p = manage(new Gtk::VScrollbar(*(editbox.get_vadjustment())));

  table.attach(editbox, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	       GTK_FILL | GTK_EXPAND, 0, 0);
  table.attach(*scrollbar_p, 2, 3, 0, 1, GTK_FILL, GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0);
  table.attach(label, 0, 2, 1, 2, GTK_FILL | GTK_EXPAND,
	       0, 0, standard_size/3);
  table.attach(accept_button, 0, 1, 2, 3,
	       0, 0, 0, standard_size/3);
  table.attach(reject_button, 1, 2, 2, 3,
	       0, 0, 0, standard_size/3);

  accept_button.clicked.connect(SigC::bind(SigC::slot(this, &GplDialog::selected), accepted));
  reject_button.clicked.connect(SigC::bind(SigC::slot(this, &GplDialog::selected), rejected));
  
  add(table);
  
  set_title("kamplus-gtk: Conditions, Notices and Disclaimers");
  set_modal(true);

  accept_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  reject_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  set_usize(standard_size * 20, standard_size * 14);
  
  set_border_width(standard_size/4);
  set_position(GTK_WIN_POS_CENTER);
  
  grab_focus();
  scrollbar_p->unset_flags(GTK_CAN_FOCUS);
  accept_button.unset_flags(GTK_CAN_FOCUS);
  reject_button.unset_flags(GTK_CAN_FOCUS);
  editbox.unset_flags(GTK_CAN_FOCUS);
  
  show_all();
}

int GplDialog::run(void) {
  Gtk::Main::run();
  return result;
}

void GplDialog::selected(Result selection) {
  hide_all();
  result = selection;
  Gtk::Main::quit(); // this will cause the run() method to return, with result as its return value
}

gint GplDialog::delete_event_impl(GdkEventAny*) {
  selected(rejected);
  return true; // returning true prevents destroy sig being emitted
}

gint GplDialog::key_press_event_impl(GdkEventKey* event_p) {

  int keycode = event_p->keyval;
  
  if (keycode == GDK_Escape) selected(rejected);
  
  else if (keycode == GDK_Home || keycode == GDK_End
	   || keycode == GDK_Up || keycode == GDK_Down
	   || keycode == GDK_Page_Up || keycode == GDK_Page_Down) {
    editbox.key_press_event_impl(event_p);
    return false;
  }
  return true;
}

Autocq_modeDialog::Autocq_modeDialog(int standard_size, Gtk::Window& window):
                             Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
                             ok_button("OK"), esc_button("Cancel"),
                             amtor_label("Amtor FEC"), pactor_label("Pactor FEC"),
			     table(2, 2, false),
                             frame("Auto-CQ Mode?"), parent(window) {

  // we have to do all this nonsense with HBoxes in order to get the toggle button labels to left justify
  amtor_label.set_justify(GTK_JUSTIFY_LEFT);
  pactor_label.set_justify(GTK_JUSTIFY_LEFT);

  amtor_button.set_active(true);

  amtor_box.pack_start(amtor_button, false, false, 0);
  amtor_box.pack_start(amtor_label, false, false, 0);
  pactor_box.pack_start(pactor_button, false, false, 0);
  pactor_box.pack_start(pactor_label, false, false, 0);

  pactor_button.set_group(amtor_button.group());
  frame_vbox.pack_start(amtor_box, false, false, 0);
  frame_vbox.pack_start(pactor_box, false, false, standard_size/4);
  frame_vbox.set_border_width(standard_size/2);

  frame.set_usize(standard_size * 5, standard_size * 3);
  frame.add(frame_vbox);

  table.attach(frame, 0, 2, 0, 1, GTK_EXPAND,
	 GTK_EXPAND, standard_size, standard_size/4);

  table.attach(ok_button, 0, 1, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(esc_button, 1, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  amtor_button.clicked.connect(SigC::slot(this, &Autocq_modeDialog::button_pressed));
  pactor_button.clicked.connect(SigC::slot(this, &Autocq_modeDialog::button_pressed));

  ok_button.clicked.connect(SigC::bind(SigC::slot(this, &Autocq_modeDialog::selected), true));
  esc_button.clicked.connect(SigC::bind(SigC::slot(this, &Autocq_modeDialog::selected), false));

  add(table);
  
  set_title("Auto-CQ Mode");
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  ok_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  esc_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  set_border_width(standard_size/2);
  ok_button.grab_focus();
  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}                                     

void Autocq_modeDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void Autocq_modeDialog::selected(bool accept) {
  parent.set_sensitive(true); // do this before we emit accepted()
  hide_all();
  if (accept) {
    if (amtor_button.get_active()) prog_func.tor_autocq_mode = Prog_func::amtor;
    else prog_func.tor_autocq_mode = Prog_func::pactor;
    accepted();
  }
  else rejected();
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

void Autocq_modeDialog::button_pressed(void) {
  ok_button.grab_focus();
}

gint Autocq_modeDialog::delete_event_impl(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

gint Autocq_modeDialog::key_press_event_impl(GdkEventKey* event_p) {
  if (event_p->keyval == GDK_Escape) selected(false);
  if (event_p->keyval == GDK_Return || event_p->keyval == GDK_Tab) Gtk::Window::key_press_event_impl(event_p);
  return false;
}

RstDialog::RstDialog(int standard_size, Gtk::Window& window):
                     Gtk::Window(GTK_WINDOW_DIALOG), in_run_loop(false),
		     ok_button("OK"), esc_button("Cancel"),
		     label("New RST Report?"), table(3, 2, false),
		     parent(window), editbox(prog_func.rst) {

  table.attach(label, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(editbox, 0, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(ok_button, 0, 1, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(esc_button, 1, 2, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  ok_button.clicked.connect(SigC::bind(SigC::slot(this, &RstDialog::selected), true));
  esc_button.clicked.connect(SigC::bind(SigC::slot(this, &RstDialog::selected), false));
  editbox.key_accept_pressed.connect(SigC::slot(this, &RstDialog::selected));

  add(table);
  
  set_title("RST Report");
  set_transient_for(parent);
  parent.set_sensitive(false);
  set_modal(true);

  ok_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  esc_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  editbox.set_usize((standard_size * 4)/3, standard_size);
  set_border_width(standard_size/2);

  editbox.grab_focus();

  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void RstDialog::run(void) {
  in_run_loop = true;
  Gtk::Main::run();
}

void RstDialog::selected(bool accept) {
  parent.set_sensitive(true); // do this before we emit accepted()
  hide_all();
  if (accept) accepted(editbox.get_text());
  if (in_run_loop) Gtk::Main::quit();
  // if we have not called run(), then this dialog is self-owning and it is safe to call `delete this'
  else delete this;
}

gint RstDialog::delete_event_impl(GdkEventAny*) {
  selected(false);
  return true; // returning true prevents destroy sig being emitted
}

RstEditor::RstEditor(const char* text) {
  set_text(text);
  count = strlen(text);
}

gint RstEditor::key_press_event_impl(GdkEventKey* event_p) {

  int keycode = event_p->keyval;
  bool marked_text = false;
  bool valid_flag = false;

  if (has_selection()) marked_text = true;

  if (keycode == GDK_Return) {
    if (count == 3 && !marked_text) key_accept_pressed(true);
    else beep();
  }
  else if (keycode == GDK_Escape) key_accept_pressed(false);
  else if (keycode == GDK_BackSpace
	   && (get_position() != 0 || marked_text)) {
    if (count) count--;
    valid_flag = true;
  }
  else if (keycode == GDK_Delete
	   && (get_position() < count || marked_text)) {
    if (count) count--;
    valid_flag = true;
  }
  
  else if (keycode == GDK_Home || keycode == GDK_End) {
    valid_flag = true;
  }

  else if (keycode == GDK_Right && get_position() < count) {
    set_position(get_position() + 1);
  }

  else if (keycode == GDK_Left && get_position()) {
    set_position(get_position() - 1);
  }
  
  else if (keycode == GDK_Tab) {
    valid_flag = true;
  }

  else if (isdigit((char)keycode)
	   && (count < 3 || marked_text)) {
    valid_flag = true;
    count++;
  }

  if (valid_flag) {
    Gtk::Entry::key_press_event_impl(event_p);
    if (marked_text) count = get_text().size();  // if we deleted some marked text reset count
  }
  if (valid_flag) return false;
  return true;
}

ComErrorDialog::ComErrorDialog(const char* text, int standard_size):
                             Gtk::Window(GTK_WINDOW_DIALOG),
			     ok_button("OK"), label(text),
			     table(2, 2, false) {
  pixmap_p = new Gtk::Pixmap(critical_xpm);
  if (!pixmap_p) {
    cerr << "Memory allocation error in ComErrorDialog::ComErrorDialog()" << endl;
    exit(MEM_ERROR);
  }

  table.attach(*pixmap_p, 0, 1, 0, 1, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(label, 1, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, standard_size/2, standard_size/4);

  table.attach(ok_button, 0, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, standard_size/2, standard_size/4);

  ok_button.clicked.connect(SigC::slot(this, &ComErrorDialog::selected));

  add(table);
  
  set_title("Kamplus-gtk: Serial port error");
  set_modal(true);

  ok_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
  set_border_width(standard_size/2);

  ok_button.grab_focus();

  set_position(GTK_WIN_POS_CENTER);
  set_policy(false, false, false);

  show_all();
}

void ComErrorDialog::run(void) {
  Gtk::Main::run();
}

void ComErrorDialog::selected(void) {
  hide_all();
  Gtk::Main::quit(); // this will cause the run() method to return
}

gint ComErrorDialog::delete_event_impl(GdkEventAny*) {
  selected();
  return true; // returning true prevents destroy sig being emitted
}

gint ComErrorDialog::key_press_event_impl(GdkEventKey* event_p) {

  if (event_p->keyval == GDK_Return || event_p->keyval == GDK_Escape) selected();
  return true;
}

ComErrorDialog::~ComErrorDialog(void) {
  delete pixmap_p;
}
