/* The following code declares classes to read from and write to
 * Unix file descriptors.
 *
 * It is based on code published by Nicolai M. Jossuttis, to which the
 * following copyright notice applies --
 *
 * (C) Copyright Nicolai M. Josuttis 2001.
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.

 * -----------------------------------------------------------------

 * attach(), close() and xsputn() methods added by Chris Vine, 2001.

 * Note that by default, like the fstream::fstream(int fd) and
 * fstream::attach(fd) extensions in libstdc++-v2, the destructors of
 * these classes close the file descriptors concerned, which helps
 * exception safety (the attach() method will also close any previous
 * file descriptor).  If this behaviour is not wanted, pass `false' as
 * the second parameter of fdostream/fdistream constructor or of the
 * attach() method .  This will enable the same file descriptor to be
 * used successively for, say, reading and writing, or to be shared
 * between fdistream and fdostream objects (but take great care if
 * doing the latter, since the file pointers can easily get out of
 * sync).

 * These modifications are added on the same terms as to copying, use,
 * modification, sale, distribution, suitability and no implied
 * warranties as those mentioned above.  In particular, this code is
 * made available "as is", and no warranty or assurance of any kind as
 * to fitness for purpose or as to its correct operation is given.

 * ----------------------------------------------------------------- */

#ifndef FDSTREAM_H
#define FDSTREAM_H

#include <istream>
#include <ostream>
#include <streambuf>
// for low-level read, write and close functions:
#include <unistd.h>
// for EOF:
#include <cstdio>

/************************************************************
 * fdostream
 * - a stream that writes on a file descriptor
 ************************************************************/


class fdoutbuf: public std::streambuf {
protected:
  int fd;    // file descriptor
  bool manage;

  static const int bufSize = 1024;   // size of the data read buffer
  static const int blockSize = 16;   // size at which a block write will be made by
                                     // xsputn() even if buffer not full
  char buffer[bufSize];
public:
  fdoutbuf(int _fd = -1, bool _manage = true); // a default _fd value of -1 means that we will use attach() later
  virtual ~fdoutbuf();

  void attach_fd(int _fd, bool _manage = true);
  void close_fd();
  int get_fd() const {return fd;}

protected:
  int flushBuffer();
  virtual int sync();
  virtual int_type overflow(int_type);
  virtual std::streamsize xsputn (const char*, std::streamsize);
};

inline int fdoutbuf::flushBuffer() {
  int num = pptr() - pbase();
  if (num) {
    if (write(fd, buffer, num) != num) return EOF;
    pbump(-num);
  }
  return num;
}

class fdostream: public std::ostream {
protected:
  fdoutbuf buf;
public:
  fdostream(int fd, bool manage = true): std::ostream(0), buf(fd, manage) { // pass the descriptor at construction
    rdbuf(&buf);
  }
  fdostream(): std::ostream(0) {                // attach the descriptor later
    rdbuf(&buf);
  }
  void attach(int fd, bool manage = true) {buf.attach_fd(fd, manage);}
  void close() {buf.close_fd();}
  int filedesc() const {return buf.get_fd();}
};


/************************************************************
 * fdistream
 * - a stream that reads on a file descriptor
 ************************************************************/

class fdinbuf : public std::streambuf {
protected:
  int fd;    // file descriptor
  bool manage;

  /* data buffer:
   * - at most, pbSize characters in putback area plus
   * - at most, bufSize characters in ordinary read buffer
   */
  static const int pbSize = 4;        // size of putback area
  static const int bufSize = 1024;    // size of the data buffer
  char buffer[bufSize+pbSize];        // data buffer
public:
  fdinbuf(int _fd = -1, bool _manage = true); // a default value of -1 means that we will use attach() later
  virtual ~fdinbuf();

  void attach_fd(int _fd, bool _manage = true);
  void close_fd();
  int get_fd() const {return fd;}
protected:
  void reset();
  virtual int_type underflow();
};

inline void fdinbuf::reset() {
  setg (buffer+pbSize,     // beginning of putback area
	buffer+pbSize,     // read position
	buffer+pbSize);    // end position
}

class fdistream : public std::istream {
protected:
  fdinbuf buf;
public:
  fdistream (int fd, bool manage = true) : std::istream(0), buf(fd, manage) { // pass the descriptor at construction
    rdbuf(&buf);
  }
  fdistream () : std::istream(0) {                // attach the descriptor later
    rdbuf(&buf);
  }
  void attach(int fd, bool manage = true) {buf.attach_fd(fd, manage);}
  void close() {buf.close_fd();}
  int filedesc() const {return buf.get_fd();}
};

#endif /*FDSTREAM_H*/
