/* Copyright (C) 1999 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include "com.h"

#ifndef HAVE_ATTACH
#include "misc/fdstream.h"
#endif

#ifndef NL0
#define NL0 0
#endif

#ifndef CR0
#define CR0 0
#endif

#ifndef TAB0
#define TAB0 0
#endif

#ifndef BS0
#define BS0 0
#endif

#ifndef VT0
#define VT0 0
#endif

#ifndef FF0
#define FF0 0
#endif

Com::Com(int speed, const char* port, void (*error_dialog)(const char*)) {

    ostrstream s1;
    ostrstream s2;

    s1 << "/dev/";
    s2 << LOCK_DIR << "/LCK..";
    if (!port) {
        s1 << "modem" << ends;
	s2 << "modem" << ends;
    }
    else {
	s1 << port << ends;
	s2 << port << ends;
    }
    const char* portdevice = s1.str();
    lockfile_name = s2.str();

    // check whether lockfile in existence

    int count = 0;

    int new_lock_fd = open(lockfile_name, O_WRONLY | O_CREAT | O_EXCL, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
    while (count < 2 && new_lock_fd == -1) {     // check to see why we could not open new_lock
        if (access(lockfile_name, F_OK) == -1) { // file doesn't exist, so there must be a problem with permissions
	    cerr << "Cannot create lockfile " << lockfile_name << endl;
	    cerr << "Please investigate permissions in " << LOCK_DIR << endl;
	    count = 2;                           // signal a failure requiring exit from the program
	    if (error_dialog) { // we have an external error dialog to display
	        ostrstream strm;
		strm << "Cannot create lockfile " << lockfile_name << '\n';
		strm << "Please investigate permissions in " << LOCK_DIR << ends;
		const char* message = strm.str();
		error_dialog(message);
		delete[] message;
	    }
	}
	else {
	    ifstream existing_lock(lockfile_name, ios::in);
	    if(!existing_lock) {                 // can't open lock file
	        cerr << "Not able to access existing lockfile " << lockfile_name << endl;
		cerr << "Please investigate why it exists and its permissions" << endl;
		if (error_dialog) { // we have an external error dialog to display
		    ostrstream strm;
		    strm << "Not able to access existing lockfile " << lockfile_name << '\n';
		    strm << "Please investigate why it exists and its permissions" << ends;
		    const char* message = strm.str();
		    error_dialog(message);
		    delete[] message;
		}
	    }
	    else {                               // lock file already exists and we can open it - check if stale
	        pid_t existing_lock_pid;
		existing_lock >> existing_lock_pid;
		if (existing_lock && (kill(existing_lock_pid, 0)) < 0) { // stale lock
		    unlink(lockfile_name);       // delete stale lock file
		    new_lock_fd = open(lockfile_name, O_WRONLY | O_CREAT | O_EXCL,
				       S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
		    count++;                     // signal we can only have one more try if this open fails
		}
		else {                           // active lock  - we have to bail out
		    if (existing_lock) {         // check if there was a read error
		        cerr << "Serial device " << portdevice << " in use by process " << existing_lock_pid << endl;
		    }
		    else cerr << "Serial device " << portdevice << " in use" << endl;
		    count = 2;                   // signal a failure requiring exit from the program
		    if (error_dialog) { // we have an external error dialog to display
		        ostrstream strm;
			if (existing_lock) {     // check if there was a read error
			    strm << "Serial device " << portdevice << " in use by process " << existing_lock_pid << ends;
			}
			else strm << "Serial device " << portdevice << " in use" << ends;
			const char* message = strm.str();
			error_dialog(message);
			delete[] message;
		    }
		}
		existing_lock.close();
	    }
	}
    }
	

    if (count < 2)  {  // we have successfully created a lock file - now write our pid to it
#ifdef HAVE_ATTACH
        ofstream new_lock;
#else
#include "misc/fdstream.h"
	fdostream new_lock;
#endif

	new_lock.attach(new_lock_fd);
        new_lock << setfill('0') << setw(10) << getpid() << endl;
	new_lock.close();

	if (!speed) speed = 9600;
        speed_t speedflag;
	if (speed == 300) speedflag = B300;
	else if (speed == 600) speedflag = B600;
	else if (speed == 1200) speedflag = B1200;
	else if (speed == 2400) speedflag = B2400;
	else if (speed == 4800) speedflag = B4800;
	else if (speed == 9600) speedflag = B9600;
	else {
	    cerr << "Invalid speed setting, defaulting to 9600 baud" << endl;
	    speedflag = B9600;
	    if (error_dialog) { // we have an external error dialog to display
		error_dialog("Invalid speed setting, defaulting to 9600 baud");
	    }
	}

	comfd = open(portdevice, O_RDWR | O_NDELAY | O_NOCTTY);
	                      // use O_NDELAY in case CLOCAL not set,
                              // otherwise the open will hang until DCD goes high
                              // O_NONBLOCK can be substituted for O_NDELAY on most systems
	if (comfd == -1) {
	    cerr << "Cannot open " << portdevice << endl;
	    if (error_dialog) { // we have an external error dialog to display
	        ostrstream strm;
		strm << "Cannot open " << portdevice << ends;
		const char* message = strm.str();
		error_dialog(message);
		delete[] message;
	    }
	    exit(COM_ERROR);
	}

	int fdflags = fcntl(comfd, F_GETFL);  // unset O_NDELAY so VMIN and VTIME will work
	fdflags &= ~O_NDELAY;
	fcntl(comfd, F_SETFL, fdflags);

	tcgetattr(comfd, &startsettings);
	termios termsettings  = startsettings;
	termsettings.c_iflag = IGNBRK | IGNPAR;
	termsettings.c_oflag = NL0 | CR0 | TAB0 | BS0 | VT0 | FF0;
	termsettings.c_cflag = CS8 | CLOCAL | CREAD | CRTSCTS;
	termsettings.c_lflag = 0;
// I have commented the next line out because libc6 doesn't define it in the usual places - is it necessary?
	//termsettings.c_line = N_TTY;
	termsettings.c_cc[VMIN] = 0;
	termsettings.c_cc[VTIME] = 1;
	cfsetospeed(&termsettings, speedflag);
	cfsetispeed(&termsettings, speedflag);
	tcsetattr(comfd, TCSANOW, &termsettings);
	delete[] portdevice;
    }
    else {         // we couldn't create a new lock file
        delete[] portdevice;
	exit(FILEOPEN_ERROR);
    }
}

Com::~Com(void) {
    tcflush(comfd, TCIOFLUSH);
    tcsetattr(comfd, TCSANOW, &startsettings);
    close(comfd);
    unlink(lockfile_name);
    delete[] lockfile_name;
}

int Com::check_port(const char* port, void (*error_dialog)(const char*)) {
    int return_value = 0;
    bool port_free = false;

    ostrstream s1;
    s1 << LOCK_DIR << "/LCK..";
    if (!port) s1 << "modem" << ends;
    else s1 << port << ends;
    const char* checkfile = s1.str();

    // check whether checkfile in existence

    if (access(checkfile, F_OK) == -1) port_free = true;
    else {
        ifstream existing_lock(checkfile, ios::in);
	if(!existing_lock) {                  // can't open lock file
	    cerr << "Not able to access existing lockfile " << checkfile << endl;
	    cerr << "Please investigate why it exists and its permissions" << endl;
	    if (error_dialog) { // we have an external error dialog to display
	        ostrstream strm;
		strm << "Not able to access existing lockfile " << checkfile << '\n';
		strm << "Please investigate why it exists and its permissions" << ends;
		const char* message = strm.str();
		error_dialog(message);
		delete[] message;
	    }
        }
	else {                                // existing lock file opened
	    pid_t existing_lock_pid;
	    existing_lock >> existing_lock_pid;
	    if (existing_lock && (kill(existing_lock_pid, 0)) < 0) { // stale lock
		unlink(checkfile);            // delete stale lockfile
		port_free = true;
	    }

	    else {                            // active lock  - we have to bail out
	        if (existing_lock) {          // check if there was a read error
		    cerr << "Serial port in use by process " << existing_lock_pid << endl;
		}
		else cerr << "Serial port in use" << endl;
		if (error_dialog) { // we have an external error dialog to display
		    ostrstream strm;
		    if (existing_lock) {      // check if there was a read error
		        strm << "Serial port in use by process " << existing_lock_pid << ends;
		    }
		    else strm << "Serial port in use" << ends;
		    const char* message = strm.str();
		    error_dialog(message);
		    delete[] message;
		}
	    }
	    existing_lock.close();
	}
    }

    if (!port_free) return_value = -1;
    
    else {      // now check that we will be able to create the lock file
        int new_lock_fd = open(checkfile, O_WRONLY | O_CREAT | O_EXCL,
			       S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
	if (new_lock_fd == -1) {
	    cerr << "Not able to create lockfile " << checkfile << endl;
	    cerr << "Please investigate permissions in " << LOCK_DIR << endl;
	    return_value = -1;
	    if (error_dialog) { // we have an external error dialog to display
	        ostrstream strm;
		strm << "Not able to create lockfile " << checkfile << '\n';
		strm << "Please investigate permissions in " << LOCK_DIR << ends;
		const char* message = strm.str();
		error_dialog(message);
		delete[] message;
	    }
	}
	else {
	    close(new_lock_fd);
	    unlink(checkfile);
	}
    }
    delete[] checkfile;
    return return_value;
}
