/* Copyright (C) 1999, 2000, 2001 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <string>
#include <iostream>
#include <iomanip>
#include <fstream>
#include <strstream>
#include <gtk--/scrollbar.h>
#include <gtk--/style.h>
#include <X11/X.h>          // the state masks are here
#include <cstdlib>

#include "modeless_dialogs.h"
#include "tnc.h"
#include "mainscreen.h"
#include "receivewin.h"
#include "download.h"
#include "buffers.h"
#include "filesend.h"
#include "modal_dialogs.h"

DownloadDialog::DownloadDialog(const char* filename, Tnc* tnc_p_, MainScreen* ms_p, ReceiveWin* rw_p,
			        int standard_size):
                                  Gtk::Window(GTK_WINDOW_DIALOG),
				  end_button("End"), bytes_intro_label(" Bytes downloaded: "),
				  table(3, 1, false), hbox(false, 0), tnc_p(tnc_p_),
				  mainscreen_p(ms_p), receivewin_p(rw_p) {
  stream = tnc_p->tnc_func.active_stream();
  port = tnc_p->tnc_func.active_port;
  ostrstream strm;
  strm << "\nDownloading ";
  if (!port) {
    strm << "Vhf port stream " << (char)(stream + 0x41) << '\n';
  } 
  else {
    strm << "Hf port ";
    if (tnc_p->tnc_func.hfmode != Tnc_func::packet) strm << "Tor stream\n";
    else strm << "stream " <<  (char)(stream + 0x41) << '\n';
  }
  strm << "to file: " << filename << '\n' << ends;
  char* text = strm.str();
  message_label.set_text(text);
  delete[] text;
  
  hbox.pack_start(bytes_intro_label, false, false, 0);
  hbox.pack_start(bytes_label, false, false, 0);
  
  bytes_frame.set_usize(standard_size * 7, standard_size);
  bytes_frame.set_shadow_type(GTK_SHADOW_IN);
  bytes_frame.add(hbox);

  end_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);

  table.attach(message_label, 0, 1, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, 0, 0);

  table.attach(end_button, 0, 1, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, 0, standard_size/2);

  table.attach(bytes_frame, 0, 1, 2, 3, GTK_FILL | GTK_EXPAND,
	 GTK_EXPAND, 0, 0);

  add(table);

  set_title("kam-gtk: Download");
  set_border_width(standard_size/2);

  show_bytes(0);

  end_button.clicked.connect(SigC::slot(this, &DownloadDialog::end_download));

  set_position(GTK_WIN_POS_NONE);
  set_policy(false, false, false);

  show_all();
}	

void DownloadDialog::end_download(void) {
  DownloadList* download_list_p = tnc_p->tnc_func.download_list_ptr;
  // we need to find the download file object receiving the
  // file on the dialog's stream and port
  download_list_p->reset(DList_enum::bottom_end);
  DownloadFile* dload_file_p;
  while ((dload_file_p = (DownloadFile*)download_list_p->inspect(DList_enum::up)) != 0
	 && !(dload_file_p->get_stream() == stream
	      && dload_file_p->get_port() == port)) {}
  if (dload_file_p) {
    download_list_p->extract();
    mainscreen_p->display_connected_status();
    dload_file_p->dialog_take_ownership(); // we are now responsible for deleting this object
    delete dload_file_p;  // the DownloadFile object used to have ownership of the relevant dialog
                          // object of this class so it is now safe to do 'delete this'
    hide_all();
    delete this;
  }
  else {
    receivewin_p->write("\nOops - can't find the download file object to delete\n"
			"Please report bug\n");
    beep();
  }
}

void DownloadDialog::show_bytes(long bytes) {
  ostrstream strm;
  strm << bytes << ends;
  char* text = strm.str();
  bytes_label.set_text(text);
  delete[] text;
}

UploadDialog::UploadDialog(const char* filename, long filesize, Tnc* tnc_p_, BufferList& bl,
			     ReceiveWin* rw_p, int standard_size):
                             Gtk::Window(GTK_WINDOW_DIALOG), end_button("Cancel"),
			     bytes_intro_label(" Bytes sent to Kam: "), table(3, 1, false),
			     hbox(false, 0), tnc_p(tnc_p_), buffer_list(bl), receivewin_p(rw_p) {
  stream = tnc_p->tnc_func.active_stream();
  port = tnc_p->tnc_func.active_port;
  ostrstream strm;
  strm << "Uploading file: " << filename << "\nto ";
  if (!port) {
    strm << "Vhf port stream " << (char)(stream + 0x41) << '\n';
    } 
  else {
    strm << "Hf port ";
    if (tnc_p->tnc_func.hfmode != Tnc_func::packet) strm << "Tor stream\n";
    else strm << "stream " <<  (char)(stream + 0x41) << '\n';
  }
  strm << "(File size: ";
  if (filesize > 99999) strm << filesize/1024 << " Kb)" << ends;
  else strm << filesize << " bytes)" << ends;
  char* text = strm.str();
  message_label.set_text(text);
  delete[] text;
  
  hbox.pack_start(bytes_intro_label, false, false, 0);
  hbox.pack_start(bytes_label, false, false, 0);
  
  bytes_frame.set_usize(standard_size * 7, standard_size);
  bytes_frame.set_shadow_type(GTK_SHADOW_IN);
  bytes_frame.add(hbox);

  end_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);

  table.attach(message_label, 0, 1, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_FILL | GTK_EXPAND, 0, 0);

  table.attach(end_button, 0, 1, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, 0, standard_size/2);

  table.attach(bytes_frame, 0, 1, 2, 3, GTK_FILL | GTK_EXPAND,
	 GTK_EXPAND, 0, 0);

  add(table);

  set_title("kam-gtk: Upload");
  set_border_width(standard_size/2);

  show_bytes(0);

  end_button.clicked.connect(SigC::slot(this, &UploadDialog::end_upload));

  set_position(GTK_WIN_POS_NONE);
  set_policy(false, false, false);

  show_all();
}	

void UploadDialog::end_upload(void) {
  if (buffer_list.get_upload_status(stream, port) == BufferList::file) { // check for sanity
    // we now need to find the file buffer sending the
    // message on the dialog's stream and port
    buffer_list.reset(DList_enum::bottom_end);
    FileBuffer* file_buffer_p;
    while ((file_buffer_p = (FileBuffer*)buffer_list.inspect(DList_enum::up)) != 0
	   && !(file_buffer_p->get_stream() == stream
		&& file_buffer_p->get_port() == port)) {}
    if (file_buffer_p) file_buffer_p->end_loading();
    else {
      receivewin_p->write("\nOops - can't find the file buffer to remove\n"
			  "Please report bug\n");
      beep();
    }
  }
  // This method simply marks the FileBuffer object as available for removal
  // from the BufferList object, and does not remove the object itself
  // UploadDialog objects are modeless and are deleted by the FilesendBuffer object
  // when that object is deleted in MainScreen::timer_event_handler()
}

void UploadDialog::show_bytes(long bytes) {
  ostrstream strm;
  strm << bytes << ends;
  char* text = strm.str();
  bytes_label.set_text(text);
  delete[] text;
}

HelpDialog::HelpDialog(int standard_size): Gtk::Window(GTK_WINDOW_DIALOG),
			     cancel_button("Cancel"),
			     table(2, 2, false) {

  string filename(prog_func.filedir);
  filename += "/helpfile-gtk";
#ifdef HAVE_IOS_NOCREATE
  ifstream helpfile(filename.c_str(), ios::in | ios::nocreate);
#else
  // we must have Std C++ so we probably don't need a ios::nocreate
  // flag on a read open to ensure uniqueness
  ifstream helpfile(filename.c_str(), ios::in);
#endif

  if (helpfile) {
    helpfile.seekg(0, ios::end);
    streampos size = helpfile.tellg();
    if (size) {
      helpfile.seekg(0); // go to beginning of file
      char letter;
      char* help_msg = new char[(int)size];
      if (!help_msg) {
	cerr << "Memory allocation error in HelpDialog::HelpDialog()" << endl;
	exit(MEM_ERROR);
      }
      
      char* char_p;
      for (char_p = help_msg; (helpfile.get(letter)); char_p++) {
	*char_p = letter;
      }
      char_p--;
      *char_p = 0;    // null terminate help_msg
      helpfile.close();
      
      editbox.set_editable(false);
      editbox.set_context(Gdk_Font(font_list[Fontsize::medium]));
      editbox.freeze();
      editbox.insert(help_msg);
      editbox.thaw();
      Gtk::Scrollbar* scrollbar_p = manage(new Gtk::VScrollbar(*(editbox.get_vadjustment())));
      
      table.attach(editbox, 0, 1, 0, 1, GTK_FILL | GTK_EXPAND,
		   GTK_FILL | GTK_EXPAND, 0, 0);
      table.attach(*scrollbar_p, 1, 2, 0, 1, GTK_FILL, GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0);
      table.attach(cancel_button, 0, 2, 1, 2,
		   0, 0, 0, standard_size/3);
      
      cancel_button.clicked.connect(SigC::slot(this, &HelpDialog::kill_dialog));
      
      add(table);
      
      set_title("kam-gtk: Helpfile");
      cancel_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);
      set_usize(standard_size * 20, standard_size * 14);
      
      set_border_width(standard_size/4);
      set_position(GTK_WIN_POS_NONE);
      
      grab_focus();
      scrollbar_p->unset_flags(GTK_CAN_FOCUS);
      cancel_button.unset_flags(GTK_CAN_FOCUS);
      editbox.unset_flags(GTK_CAN_FOCUS);

      show_all();
      delete[] help_msg;
    }
  }
  else {
    filename += " cannot be found/opened";
    InfoDialog dialog(filename.c_str(), "Helpfile", standard_size, InfoDialog::warning, *this);
    dialog.run();
    delete this;  // this a modeless dialog, and so safe
  }
}

void HelpDialog::kill_dialog(void) {
  hide_all();
  delete this;  // this a modeless dialog, and so safe
}

gint HelpDialog::delete_event_impl(GdkEventAny*) {
  kill_dialog();
  return true; // returning true prevents destroy sig being emitted
}

gint HelpDialog::key_press_event_impl(GdkEventKey* event_p) {

  int keycode = event_p->keyval;
  
  if (keycode == GDK_Escape) kill_dialog();
  
  else if (keycode == GDK_Home || keycode == GDK_End
	   || keycode == GDK_Up || keycode == GDK_Down
	   || keycode == GDK_Page_Up || keycode == GDK_Page_Down) {
    editbox.key_press_event_impl(event_p);
  }
  return 0;
}

QsoCountDialog::QsoCountDialog(int standard_size):
                                  Gtk::Window(GTK_WINDOW_DIALOG),
				  quit_button("Quit"), count_intro_label(" QSO Number: "),
				  table(3, 2, false), hbox(false, 0) {
  
  hbox.pack_start(count_intro_label, false, false, 0);
  hbox.pack_start(count_label, false, false, 0);
  
  count_frame.set_usize(standard_size * 6, standard_size);
  count_frame.set_shadow_type(GTK_SHADOW_IN);
  count_frame.add(hbox);

  dec_button.set_usize((standard_size * 3)/2, (standard_size * 3)/2);
  inc_button.set_usize((standard_size * 3)/2, (standard_size * 3)/2);
  quit_button.set_usize((standard_size * 5)/2, (standard_size * 4)/5);

  // the font setting code commented out doesn't work with gtkmm-1.2
  // it is there for the time (if/when) it does
  // so instead now insert some text
  dec_button.add_label("Dec", 0.5, 0.5);
  inc_button.add_label("Inc", 0.5, 0.5);

  /*
  // set the fonts for the inc and dec buttons
  // Note: we need to take a copy of the existing style, as the existing
  // style is shared by all the program's widgets.  If we operate
  // directly on the Gtk::Style object returned (by pointer) by
  // Gtk::Widget::get_style(), say with Gtk::Style::set_fg() or
  // Gtk::Style::set_font(), we will alter it for everything!
  if (!font_list[Fontsize::giant].empty()) {
    Gtk::Style* new_style_p = dec_button.get_style()->copy();
    new_style_p->set_font(Gdk_Font(font_list[Fontsize::giant]));
    dec_button.set_style(*new_style_p);
    inc_button.set_style(*new_style_p);
  }
  dec_button.add_label("-", 0.5, 0.5);
  inc_button.add_label("+", 0.5, 0.5);
  */

  table.attach(count_frame, 0, 2, 0, 1, GTK_FILL | GTK_EXPAND,
	 GTK_EXPAND, 0, 0);

  table.attach(dec_button, 0, 1, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, 0, standard_size/2);

  table.attach(inc_button, 1, 2, 1, 2, GTK_EXPAND,
	 GTK_EXPAND, 0, standard_size/2);

  table.attach(quit_button, 0, 2, 2, 3, GTK_EXPAND,
	 GTK_EXPAND, 0, 0);

  add(table);

  set_title("QSO Counter");
  set_border_width(standard_size/2);

  dec_button.clicked.connect(dec.slot());
  inc_button.clicked.connect(inc.slot());
  quit_button.clicked.connect(SigC::slot(this, &QsoCountDialog::destroy_dialog));

  set_position(GTK_WIN_POS_NONE);
  set_policy(false, false, false);

  show_all();
}	

QsoCountDialog::~QsoCountDialog(void) {
  quit_button_pressed();
}

void QsoCountDialog::destroy_dialog(void) {
  hide_all();
  delete this;  // this a modeless dialog, and so safe
}

void QsoCountDialog::show_qso_count(int qso_count) {
  ostrstream strm;
  strm << setfill('0') << setw(3);
  strm << qso_count << ends;
  char* text = strm.str();
  count_label.set_text(text);
  delete[] text;
}

gint QsoCountDialog::key_press_event_impl(GdkEventKey* event_p) {

  int keycode = event_p->keyval;

  if (keycode == GDK_plus || keycode == GDK_equal || keycode == GDK_Up) inc();
  else if (keycode == GDK_minus || keycode == GDK_underscore || keycode == GDK_Down) dec();
  return 0;
}
