/* Copyright (C) 1999, 2000, 2001 Chris Vine, G3XXF

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYRIGHT distributed with the source files.

*/

#include <iostream>
#include <strstream>
#include <string>
#include <cstdlib>
#include <cstring>
#include <gtk--/main.h>
#include <gtk--/paned.h>
#include <gtk--/accelgroup.h>
#include <gtk--/pixmap.h>
#include <gtk--/style.h>
#include <gdk/gdkkeysyms.h> // the key codes are here
#include <X11/X.h>          // the state masks are here

#include "mainscreen.h"
#include "receivewin.h"
#include "info_bars.h"
#include "event_slots.h"
#include "tnc.h"
#include "buffers.h"
#include "download.h"
#include "filesend.h"
#include "pipes.h"
#include "icons.h"

#define TIMER_INTERVAL 100                // number of milliseconds between timer events
#define INTERVAL_COUNT 200/TIMER_INTERVAL // beeps need to fire every 200 milliseconds

void MainScreen::destroy_slot(void) {
  // there are a number of ways to close the program
  // one is to call Gtk::Main::quit() and then to follow it with
  // an explicit emission of destroy()
  // or delete_event_impl() can call Gtk::Main::quit() and return false
  // (which in fact just emits destroy() and so has the same
  // effect as the first case)
  // another (which we do here) is to call Gtk::Main::quit() within a method
  // connected to the destroy object slot, and quit the program by emitting
  // the destroy signal
  Gtk::Main::quit();
}

gint MainScreen::key_press_event_impl(GdkEventKey* event_p) {

  int keycode = event_p->keyval;
  int state = event_p->state;
  bool letter_flag = false;
  int letter = keycode;

  if (tnc_p) {
    if (keycode == GDK_F1) event_slots_p->helpfile();
    else if (keycode == GDK_F2) event_slots_p->capture();
    else if (keycode == GDK_F3) {
      int mode;
      if (prog_func.send_mode == Prog_func::line) mode = Prog_func::word;
      else if (prog_func.send_mode == Prog_func::word) mode = Prog_func::guard;
      else mode = Prog_func::line;
      event_slots_p->hfsend_mode(mode);
    }
    else if (keycode == GDK_F4) event_slots_p->make_connection_prompt();
    else if (keycode == GDK_F5) event_slots_p->command_prompt();
    else if (keycode == GDK_F6) event_slots_p->upload_prompt(FileBuffer::text);
    else if (keycode == GDK_F7) event_slots_p->enter_call_prompt();
    else if (keycode == GDK_F8) event_slots_p->lock_call();
    else if (keycode == GDK_F9) event_slots_p->send_parms_prompt("tnc.parms");
    else if (keycode == GDK_F10) event_slots_p->change_port();
    else if (keycode == GDK_F11) event_slots_p->stream_down();
    else if (keycode == GDK_F12)  event_slots_p->stream_up();
  
    else if (!(state & Mod1Mask) && (state & ControlMask)) {
      if (keycode == GDK_A || keycode == GDK_a) event_slots_p->ctrl_a();
      else if (keycode == GDK_C || keycode == GDK_c) event_slots_p->change_hfmode_prompt(Tnc_func::cw);
      else if (keycode == GDK_D || keycode == GDK_d) event_slots_p->disconnect_prompt();
      else if (keycode == GDK_F || keycode == GDK_f) event_slots_p->change_hfmode_prompt(Tnc_func::fec);
      else if (keycode == GDK_L || keycode == GDK_l) event_slots_p->change_hfmode_prompt(Tnc_func::lamtor);
      else if (keycode == GDK_R || keycode == GDK_r) event_slots_p->change_hfmode_prompt(Tnc_func::rtty);
      else if (keycode == GDK_S || keycode == GDK_s) event_slots_p->change_hfmode_prompt(Tnc_func::ascii);
      else if (keycode == GDK_T || keycode == GDK_t) event_slots_p->ctrl_t();
      else if (keycode == GDK_X || keycode == GDK_x) event_slots_p->change_hfmode_prompt(Tnc_func::packet);
      else if (keycode == GDK_Z || keycode == GDK_z) event_slots_p->ctrl_z();
#ifndef NO_PACTOR
      else if (keycode == GDK_P || keycode == GDK_p) event_slots_p->change_hfmode_prompt(Tnc_func::pactor);
  #ifndef NO_GTOR
      else if (keycode == GDK_G || keycode == GDK_g) event_slots_p->change_hfmode_prompt(Tnc_func::gtor);
      else if (keycode == GDK_O || keycode == GDK_o) event_slots_p->change_hfmode_prompt(Tnc_func::gmon);
  #endif
#endif
    }
    else if ((state & Mod1Mask) && !(state & ControlMask)) {
      if (keycode == GDK_B || keycode == GDK_b) receivewin_p->print_scroll_buffer_prompt();
      else if (keycode == GDK_C || keycode == GDK_c) event_slots_p->auto_cq_preprocess(EventSlots::amtor, true);
      else if (keycode == GDK_D || keycode == GDK_d) event_slots_p->download_prompt(DownloadFile::s_plus);
      else if (keycode == GDK_I || keycode == GDK_i) event_slots_p->ident();
      else if (keycode == GDK_L || keycode == GDK_l) event_slots_p->lock_speed();
      else if (keycode == GDK_N || keycode == GDK_n) event_slots_p->read_cluster();
      else if (keycode == GDK_P || keycode == GDK_p) event_slots_p->context_print_mark();
      else if (keycode == GDK_R || keycode == GDK_r) event_slots_p->rst_prompt();
      else if (keycode == GDK_S || keycode == GDK_s) event_slots_p->sync();
      else if (keycode == GDK_T || keycode == GDK_t) event_slots_p->auto_cq_preprocess(EventSlots::pactor, true);
      else if (keycode == GDK_U || keycode == GDK_u) event_slots_p->upload_prompt(FileBuffer::s_plus);
      else if (keycode == GDK_X || keycode == GDK_x) event_slots_p->abort_prompt();
      else if (keycode == GDK_0) event_slots_p->send_message_preprocess('0', true);
      else if (keycode == GDK_1) event_slots_p->send_message_preprocess('1', true);
      else if (keycode == GDK_2) event_slots_p->send_message_preprocess('2', true);
      else if (keycode == GDK_3) event_slots_p->send_message_preprocess('3', true);
      else if (keycode == GDK_4) event_slots_p->send_message_preprocess('4', true);
      else if (keycode == GDK_5) event_slots_p->send_message_preprocess('5', true);
      else if (keycode == GDK_6) event_slots_p->send_message_preprocess('6', true);
      else if (keycode == GDK_7) event_slots_p->send_message_preprocess('7', true);
      else if (keycode == GDK_8) event_slots_p->send_message_preprocess('8', true);
      else if (keycode == GDK_9) event_slots_p->send_message_preprocess('9', true);
      else if (keycode == GDK_plus || keycode == GDK_equal) event_slots_p->increment_qso_count();
      else if (keycode == GDK_minus || keycode == GDK_underscore) event_slots_p->decrement_qso_count();
      // activate the meunbar keys
      else Gtk::Window::key_press_event_impl(event_p);
    }
    else if ((state & Mod1Mask) && (state & ControlMask)) {
      if (keycode == GDK_D || keycode == GDK_d) event_slots_p->download_prompt(DownloadFile::binary);
      else if (keycode == GDK_U || keycode == GDK_u) event_slots_p->upload_prompt(FileBuffer::binary);
    }

    else if (keycode == GDK_Page_Up) event_slots_p->changeover_tx();
    else if (keycode == GDK_Page_Down) event_slots_p->changeover_rx();
    
    else if (keycode == GDK_Up) receivewin_p->scrollup();
    else if (keycode == GDK_Down) receivewin_p->scrolldown();
    else if (keycode == GDK_Right) receivewin_p->scrollout(false);
  }
  
  if (!(state & Mod1Mask) && (!(state & ControlMask))) {
    if (keycode == GDK_Return) {
      letter = '\n';
      letter_flag = true;
    }
    else if (keycode == GDK_BackSpace) {
      letter = 8;
      letter_flag = true;
    }
    else if (keycode ==  GDK_Delete); // ignore delete key
    
    else if (keycode < 256) letter_flag = true;
    
    if (letter_flag
	&& (letter == '\n' || letter == 8 || letter > 31)) {
      event_slots_p->send_letter(letter);
    }
  }
  return true;
}

MainScreen::MainScreen(Tnc* a, Tnc_base* b, Pipe_fifo& c, Transmit_buffer& d,
		       BufferList& e, int global_fontsize, int win_fontsize):
                                        Gtk::Window(GTK_WINDOW_TOPLEVEL),
					letters_to_send_flag(false), flush_chars(false),
					count(20), buffer_count(0), beep_count(0), interval_count(0),
					left_tool_bar(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS), 
					right_tool_bar(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS),
					sendwin(win_fontsize), hf_stream_status(MainScreen::packet),
					setting_call_lock_button_flag(false), setting_capture_button_flag(false),
					setting_auto_cq_button_flag(false), setting_speed_lock_button_flag(false),
					setting_dx_cluster_button_flag(false),
					tnc_p(a), tnc_base_p(b), receive_pipe(c), tr_buffer(d), buffer_list(e) {

  // standard_size is used as a measure in making various widgets and dialogs
  const int standard_size = 30;

  // connect to the slot to close the application
  destroy.connect(SigC::slot(this, &MainScreen::destroy_slot));

  if (global_fontsize >=10 && global_fontsize <= 12) {
    Gtk::Style* new_style_p = get_style(); // this will grab the global style
                                           // (we did not use get_style()->copy())
                                           // accordingly there is no need to call set_style(*new_style_p)
                                           // after we have called set_font()
    ostrstream strm;
    strm << "-adobe-helvetica-medium-r-normal--" << global_fontsize << "-*-*-*-*-*-iso8859-1" << ends;
    const char* font_name = strm.str();
    new_style_p->set_font(Gdk_Font(font_name));
    delete[] font_name;
  }

  Gtk::VPaned* vpaned_p = manage(new Gtk::VPaned);
  receivewin_p = new ReceiveWin(*vpaned_p, tnc_p, standard_size, *this, win_fontsize);
  if (!receivewin_p) {
    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
    exit(MEM_ERROR);
  }

  // make the event_slots object
  event_slots_p = new EventSlots(this, receivewin_p, &sendwin, tnc_p, tr_buffer, buffer_list, standard_size);
  if (!event_slots_p) {
    cerr << "Memory allocation error in MainScreen::MainScreen()" << endl;
    exit(MEM_ERROR);
  }

  // set up the menu bar
  {
    using namespace Gtk::Menu_Helpers;

    // Create the file menu
    Gtk::Menu* file_menu_p = manage(new Gtk::Menu);
    // and the stream menu
    Gtk::Menu* stream_menu_p = manage(new Gtk::Menu);
    // Create the hf mode menu
    Gtk::Menu* mode_menu_p = manage(new Gtk::Menu);
    // Create the send menu
    Gtk::Menu* send_menu_p = manage(new Gtk::Menu);
    // Create the message menu
    Gtk::Menu* message_menu_p = manage(new Gtk::Menu);
    // Create the help menu
    Gtk::Menu* help_menu_p = manage(new Gtk::Menu());

    // Create the mouse stream menu
    Gtk::Menu* mouse_stream_menu_p = manage(new Gtk::Menu);
    // Create the mouse mode menu
    Gtk::Menu* mouse_mode_menu_p = manage(new Gtk::Menu);

    
    //menu_bar.set_shadow_type(GTK_SHADOW_NONE);
      
    MenuList& file_list = file_menu_p->items();

    if (tnc_p) {
      // Create the upload menu
      Gtk::Menu* upload_menu_p = manage(new Gtk::Menu);

      MenuList& upload_list = upload_menu_p->items();
      upload_list.push_back(MenuElem("_Text(CR->LF and CP437 conversion)",
			SigC::bind(SigC::slot(event_slots_p, &EventSlots::upload_prompt), FilesendBuffer::text)));
      upload_list.push_back(MenuElem("_7-plus(CR->LF conversion)",
			SigC::bind(SigC::slot(event_slots_p, &EventSlots::upload_prompt), FilesendBuffer::s_plus)));
      upload_7plus_item_p = upload_list.back();
      upload_list.push_back(MenuElem("_Binary",
			SigC::bind(SigC::slot(event_slots_p, &EventSlots::upload_prompt), FilesendBuffer::binary)));
      upload_binary_item_p = upload_list.back();

      // Create the download menu
      Gtk::Menu* download_menu_p = manage(new Gtk::Menu);

      MenuList& download_list = download_menu_p->items();
      download_list.push_back(MenuElem("_7-plus(CR->LF conversion)",
			SigC::bind(SigC::slot(event_slots_p, &EventSlots::download_prompt), DownloadFile::s_plus)));
      download_list.push_back(MenuElem("_Binary",
			SigC::bind(SigC::slot(event_slots_p, &EventSlots::download_prompt), DownloadFile::binary)));

      file_list.push_back(MenuElem("_Upload", *upload_menu_p));
      upload_item_p = file_list.back();
      file_list.push_back(MenuElem("_Download", *download_menu_p));
      download_item_p = file_list.back();

      file_list.push_back(SeparatorElem());

      file_list.push_back(MenuElem("_Copy selected text", SigC::slot(receivewin_p, &ReceiveWin::copy)));
      menu_copy_item_p = file_list.back();
      file_list.push_back(SeparatorElem());
      file_list.push_back(MenuElem("_Print selected text", SigC::slot(receivewin_p, &ReceiveWin::print_selection_prompt)));
      menu_print_selection_item_p = file_list.back();
      file_list.push_back(MenuElem("Print scroll _buffer", SigC::slot(receivewin_p, &ReceiveWin::print_scroll_buffer_prompt)));
      file_list.push_back(MenuElem("Set print _mark", SigC::slot(event_slots_p, &EventSlots::set_print_mark_prompt)));
      menu_set_print_mark_item_p = file_list.back();
      file_list.push_back(MenuElem("Print _from mark", SigC::slot(event_slots_p, &EventSlots::print_from_mark_prompt)));
      menu_print_from_mark_item_p = file_list.back();

      file_list.push_back(SeparatorElem());

      file_list.push_back(MenuElem("S_ave selected text", SigC::slot(receivewin_p, &ReceiveWin::save_prompt)));
      menu_save_selection_item_p = file_list.back();

      file_list.push_back(SeparatorElem());
      file_list.push_back(MenuElem("Qs_o counter", SigC::slot(event_slots_p, &EventSlots::increment_qso_count)));
      menu_qso_count_item_p = file_list.back();
      
      file_list.push_back(SeparatorElem());
      file_list.push_back(MenuElem("Set up _Kam", SigC::bind(SigC::slot(event_slots_p, &EventSlots::send_parms_prompt), "tnc.parms")));
      file_list.push_back(SeparatorElem());
      file_list.push_back(MenuElem("_Settings", SigC::slot(event_slots_p, &EventSlots::settings_prompt)));
      file_list.push_back(SeparatorElem());
    
      upload_item_p->set_sensitive(false);
      upload_7plus_item_p->set_sensitive(false);
      upload_binary_item_p->set_sensitive(false);
      download_item_p->set_sensitive(false);

      menu_copy_item_p->set_sensitive(false);
      menu_print_selection_item_p->set_sensitive(false);
      menu_print_from_mark_item_p->set_sensitive(false);
      menu_save_selection_item_p->set_sensitive(false);


      // Create the stream menu
      Gtk::Menu* vhf_menu_p = manage(new Gtk::Menu);
      MenuList& vhf_menu_list = vhf_menu_p->items();
      int stream;
      char vhfstreamlabel[] = "Vhf-_A";
      char* char_p = vhfstreamlabel + 5;
      for (stream = 0; stream < MAXUSERS; stream++, (*char_p)++) {
	vhf_menu_list.push_back(MenuElem(vhfstreamlabel,
				SigC::bind(SigC::slot(event_slots_p, &EventSlots::vhf_menu), stream)));
      }
      *char_p = 0;
      char_p--;
      *char_p = 'A';
      Gtk::Menu* mouse_vhf_menu_p = manage(new Gtk::Menu);
      MenuList& vhf_mouse_list = mouse_vhf_menu_p->items();
      for (stream = 0; stream < MAXUSERS; stream++, (*char_p)++) {
	vhf_mouse_list.push_back(MenuElem(vhfstreamlabel,
				   SigC::bind(SigC::slot(event_slots_p, &EventSlots::vhf_menu), stream)));
      }

      hf_menu_p = manage(new Gtk::Menu);
      MenuList& hf_list = hf_menu_p->items();
      char hfstreamlabel[] = "Hf-_A";
      char_p = hfstreamlabel + 4;
      for (stream = 0; stream < MAXUSERS; stream++, (*char_p)++) {
	hf_list.push_back(MenuElem(hfstreamlabel,
				   SigC::bind(SigC::slot(event_slots_p, &EventSlots::hf_menu), stream)));
      }

      *char_p = 0;
      char_p--;
      *char_p = 'A';
      mouse_hf_menu_p = manage(new Gtk::Menu);
      MenuList& hf_mouse_list = mouse_hf_menu_p->items();
      for (stream = 0; stream < MAXUSERS; stream++, (*char_p)++) {
	hf_mouse_list.push_back(MenuElem(hfstreamlabel,
				   SigC::bind(SigC::slot(event_slots_p, &EventSlots::hf_menu), stream)));
      }

      // now fill the stream menu
      MenuList& stream_list = stream_menu_p->items();
      stream_list.push_back(MenuElem("Vhf", *vhf_menu_p));
      stream_list.push_back(MenuElem("Hf", *hf_menu_p));
      
      // and the mouse stream menu
      MenuList& mouse_stream_list = mouse_stream_menu_p->items();
      mouse_stream_list.push_back(MenuElem("Vhf", *mouse_vhf_menu_p));
      mouse_stream_list.push_back(MenuElem("Hf", *mouse_hf_menu_p));
      
      // now fill the mode menu
      MenuList& mode_list = mode_menu_p->items();
      
      mode_list.push_back(MenuElem("Packet(_X)",
			 SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::packet)));
#ifndef NO_PACTOR
      mode_list.push_back(MenuElem("_Pactor",
			  SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::pactor)));
#endif
      mode_list.push_back(MenuElem("_Amtor",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::amtor)));
      mode_list.push_back(MenuElem("_Lamtor",
			 SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::lamtor)));
      mode_list.push_back(MenuElem("_Fec",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::fec)));
      mode_list.push_back(MenuElem("_Rtty",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::rtty)));
      mode_list.push_back(MenuElem("A_scii",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::ascii)));
#ifndef NO_PACTOR
  #ifndef NO_GTOR
      mode_list.push_back(MenuElem("_Gtor",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::gtor)));
      mode_list.push_back(MenuElem("G_mon",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::gmon)));
  #endif
      mode_list.push_back(MenuElem("_Tor",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::tor)));
#endif
      mode_list.push_back(MenuElem("_Cw",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::cw)));



      // now fill the mouse mode menu
      MenuList& mouse_mode_list = mouse_mode_menu_p->items();

      mouse_mode_list.push_back(MenuElem("Packet",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::packet)));
#ifndef NO_PACTOR
      mouse_mode_list.push_back(MenuElem("Pactor",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::pactor)));
#endif
      mouse_mode_list.push_back(MenuElem("Amtor",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::amtor)));
      mouse_mode_list.push_back(MenuElem("Lamtor",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::lamtor)));
      mouse_mode_list.push_back(MenuElem("Fec",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::fec)));
      mouse_mode_list.push_back(MenuElem("Rtty",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::rtty)));
      mouse_mode_list.push_back(MenuElem("Ascii",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::ascii)));
#ifndef NO_PACTOR
  #ifndef NO_GTOR
      mouse_mode_list.push_back(MenuElem("Gtor",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::gtor)));
      mouse_mode_list.push_back(MenuElem("Gmon",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::gmon)));
  #endif
      mouse_mode_list.push_back(MenuElem("Tor",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::tor)));
#endif
      mouse_mode_list.push_back(MenuElem("Cw",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::change_hfmode), Tnc_func::cw)));

      // now fill the send menu
      MenuList& send_list = send_menu_p->items();

      send_list.push_back(MenuElem("_Word",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::hfsend_mode), Prog_func::word)));
      send_list.push_back(MenuElem("_Guard",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::hfsend_mode), Prog_func::guard)));
      send_list.push_back(MenuElem("_Line",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::hfsend_mode), Prog_func::line)));
    
      // now fill the message menu
      MenuList& message_list = message_menu_p->items();

      message_list.push_back(MenuElem("Message 1",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::send_message_preprocess), '1', false)));
      message_list.push_back(MenuElem("Message 2",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::send_message_preprocess), '2', false)));
      message_list.push_back(MenuElem("Message 3",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::send_message_preprocess), '3', false)));
      message_list.push_back(MenuElem("Message 4",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::send_message_preprocess), '4', false)));
      message_list.push_back(MenuElem("Message 5",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::send_message_preprocess), '5', false)));
      message_list.push_back(MenuElem("File Msg 6",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::send_message_preprocess), '6', false)));
      message_list.push_back(MenuElem("File Msg 7",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::send_message_preprocess), '7', false)));
      message_list.push_back(MenuElem("File Msg 8",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::send_message_preprocess), '8', false)));
      message_list.push_back(MenuElem("File Msg 9",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::send_message_preprocess), '9', false)));
      message_list.push_back(MenuElem("File Msg 0",
		         SigC::bind(SigC::slot(event_slots_p, &EventSlots::send_message_preprocess), '0', false)));

    }

    // complete the file menu
    file_list.push_back(MenuElem("_Quit", destroy.slot()));

    // fill the help menu
    MenuList& help_list = help_menu_p->items();
    help_list.push_back(MenuElem("_About kamplus-gtk", SigC::slot(event_slots_p, &EventSlots::about_kamplus)));
    help_list.push_back(SeparatorElem());
    help_list.push_back(MenuElem("_Helpfile", SigC::slot(event_slots_p, &EventSlots::helpfile)));
    
    // now enter all the menu items in the menu bar
    MenuList& bar_list = menu_bar.items();
    bar_list.push_front(MenuElem("_Help", "<mod1>h", *help_menu_p));
    bar_list.front()->right_justify();
    
    if (tnc_p) {
      bar_list.push_front(MenuElem("Messa_ge","<mod1>g",*message_menu_p));
      bar_list.push_front(MenuElem("Hf S_end","<mod1>e",*send_menu_p));
      bar_list.push_front(MenuElem("Hf _Mode","<mod1>m",*mode_menu_p));
      bar_list.push_front(MenuElem("Stre_am","<mod1>a",*stream_menu_p));
    }
    bar_list.push_front(MenuElem("_File","<mod1>f",*file_menu_p));

    if (tnc_p) {
      // now set up the pop-up menu

      MenuList& mouse_popup_list = menu_popup.items();
      mouse_popup_list.push_back(MenuElem("Stream", *mouse_stream_menu_p));
      mouse_popup_list.push_back(MenuElem("Hf Mode", *mouse_mode_menu_p));
      mouse_popup_list.push_back(SeparatorElem());
      mouse_popup_list.push_back(MenuElem("Settings", SigC::slot(event_slots_p, &EventSlots::settings_prompt)));
      mouse_popup_list.push_back(SeparatorElem());
      mouse_popup_list.push_back(MenuElem("Copy selected text", SigC::slot(receivewin_p, &ReceiveWin::copy)));
      mouse_copy_item_p = mouse_popup_list.back();
      mouse_popup_list.push_back(SeparatorElem());
      mouse_popup_list.push_back(MenuElem("Print selected text", SigC::slot(receivewin_p, &ReceiveWin::print_selection_prompt)));
      mouse_print_selection_item_p = mouse_popup_list.back();
      mouse_popup_list.push_back(MenuElem("Print scroll buffer", SigC::slot(receivewin_p, &ReceiveWin::print_scroll_buffer_prompt)));
      mouse_popup_list.push_back(MenuElem("Set print mark", SigC::slot(event_slots_p, &EventSlots::set_print_mark_prompt)));
      mouse_set_print_mark_item_p = mouse_popup_list.back();
      mouse_popup_list.push_back(MenuElem("Print from mark", SigC::slot(event_slots_p, &EventSlots::print_from_mark_prompt)));
      mouse_print_from_mark_item_p = mouse_popup_list.back();
      mouse_popup_list.push_back(SeparatorElem());
      mouse_popup_list.push_back(MenuElem("Save selected text", SigC::slot(receivewin_p, &ReceiveWin::save_prompt)));
      mouse_save_selection_item_p = mouse_popup_list.back();

      mouse_popup_list.push_back(SeparatorElem());
      mouse_popup_list.push_back(MenuElem("Qso counter", SigC::slot(event_slots_p, &EventSlots::increment_qso_count)));
      mouse_qso_count_item_p = mouse_popup_list.back();

      mouse_copy_item_p->set_sensitive(false);
      mouse_print_selection_item_p->set_sensitive(false);
      mouse_print_from_mark_item_p->set_sensitive(false);
      mouse_save_selection_item_p->set_sensitive(false);
    }
  }
  // set up the tool bar
  if (tnc_p) {
    using namespace Gtk::Toolbar_Helpers;
    
    // first make the pixmaps
    Gtk::Pixmap* abortIcon_p = manage(new Gtk::Pixmap(abort_xpm));
    Gtk::Pixmap* autocqIcon_p = manage(new Gtk::Pixmap (auto_cq_xpm));
    Gtk::Pixmap* callLockIcon_p = manage(new Gtk::Pixmap(call_lock_xpm));
    Gtk::Pixmap* captureIcon_p = manage(new Gtk::Pixmap(capture_xpm));
    Gtk::Pixmap* commandIcon_p = manage(new Gtk::Pixmap(command_xpm));
    Gtk::Pixmap* connectIcon_p = manage(new Gtk::Pixmap(connect_xpm));
    Gtk::Pixmap* changeSpeedIcon_p = manage(new Gtk::Pixmap(change_speed_xpm));
    Gtk::Pixmap* disconnectIcon_p = manage(new Gtk::Pixmap(disconnect_xpm));
    Gtk::Pixmap* dxclusterIcon_p = manage(new Gtk::Pixmap(dx_cluster_xpm));
    Gtk::Pixmap* enterCallIcon_p = manage(new Gtk::Pixmap(enter_call_xpm));
    Gtk::Pixmap* identIcon_p = manage(new Gtk::Pixmap(ident_xpm));
    Gtk::Pixmap* lockSpeedIcon_p = manage(new Gtk::Pixmap(lock_speed_xpm));
    Gtk::Pixmap* rstIcon_p = manage(new Gtk::Pixmap(rst_xpm));
    Gtk::Pixmap* rxIcon_p = manage(new Gtk::Pixmap(rx_xpm));
    Gtk::Pixmap* syncIcon_p = manage(new Gtk::Pixmap(sync_xpm));
    Gtk::Pixmap* txIcon_p = manage(new Gtk::Pixmap(tx_xpm));
    
    left_tool_bar.set_tooltips(true);
    right_tool_bar.set_tooltips(true);
    left_tool_bar.set_button_relief(GTK_RELIEF_NORMAL);
    right_tool_bar.set_button_relief(GTK_RELIEF_NORMAL);
    
    ToolList& left_tool_list = left_tool_bar.tools();
    left_tool_list.push_back(ButtonElem(*connectIcon_p, SigC::slot(event_slots_p, &EventSlots::make_connection_prompt),
					"Connect"));
    connect_button_p = static_cast<Gtk::Button*>(left_tool_list.back()->get_widget());
    left_tool_list.push_back(ButtonElem(*disconnectIcon_p, SigC::slot(event_slots_p, &EventSlots::disconnect_prompt),
					"Disconnect"));
    disconnect_button_p = static_cast<Gtk::Button*>(left_tool_list.back()->get_widget());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(ButtonElem(*enterCallIcon_p, SigC::slot(event_slots_p, &EventSlots::enter_call_prompt),
					"Enter callsign"));
    left_tool_list.push_back(ToggleElem(*callLockIcon_p, SigC::slot(event_slots_p, &EventSlots::lock_call),
					"Toggle callsign lock"));
    call_lock_button_p = static_cast<Gtk::ToggleButton*>(left_tool_list.back()->get_widget());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(ButtonElem(*commandIcon_p, SigC::slot(event_slots_p, &EventSlots::command_prompt),
					"Command Kam"));
    left_tool_list.push_back(Space());
    left_tool_list.push_back(ToggleElem(*captureIcon_p, SigC::slot(event_slots_p, &EventSlots::capture),
					"Toggle capture stream on/off"));
    capture_button_p = static_cast<Gtk::ToggleButton*>(left_tool_list.back()->get_widget());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(ToggleElem(*autocqIcon_p, SigC::bind(SigC::slot(event_slots_p, &EventSlots::auto_cq_preprocess),
					  EventSlots::choose, false), "Send Auto-CQ"));
    auto_cq_button_p = static_cast<Gtk::ToggleButton*>(left_tool_list.back()->get_widget());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(ToggleElem(*lockSpeedIcon_p, SigC::slot(event_slots_p, &EventSlots::lock_speed),
					"Lock Speed to 100 baud"));
    speed_lock_button_p = static_cast<Gtk::ToggleButton*>(left_tool_list.back()->get_widget());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(ButtonElem(*identIcon_p, SigC::slot(event_slots_p, &EventSlots::ident),
					"Send Ident"));
    ident_button_p = static_cast<Gtk::Button*>(left_tool_list.back()->get_widget());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(ButtonElem(*rstIcon_p, SigC::slot(event_slots_p, &EventSlots::rst_prompt),
					"Enter RST"));
    left_tool_list.push_back(Space());
    left_tool_list.push_back(ButtonElem(*abortIcon_p, SigC::slot(event_slots_p, &EventSlots::abort_prompt),
					"Abort"));
    abort_button_p = static_cast<Gtk::Button*>(left_tool_list.back()->get_widget());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(ButtonElem(*syncIcon_p, SigC::slot(event_slots_p, &EventSlots::sync),
					"Sync Amtor/CW"));
    sync_button_p = static_cast<Gtk::Button*>(left_tool_list.back()->get_widget());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(ButtonElem(*changeSpeedIcon_p, SigC::slot(event_slots_p, &EventSlots::change_speed),
					"Set CW/RTTY/ASCII Speed"));
    change_speed_button_p = static_cast<Gtk::Button*>(left_tool_list.back()->get_widget());
    left_tool_list.push_back(Space());
    left_tool_list.push_back(ToggleElem(*dxclusterIcon_p, SigC::slot(event_slots_p, &EventSlots::read_cluster),
					"Monitor DX Cluster"));
    dx_cluster_button_p = static_cast<Gtk::ToggleButton*>(left_tool_list.back()->get_widget());
    
    ToolList& right_tool_list = right_tool_bar.tools();
    right_tool_list.push_back(ButtonElem(*rxIcon_p, SigC::slot(event_slots_p, &EventSlots::changeover_rx),
					 "Receive"));
    rx_button_p = static_cast<Gtk::Button*>(right_tool_list.back()->get_widget());
    right_tool_list.push_back(ButtonElem(*txIcon_p, SigC::slot(event_slots_p, &EventSlots::changeover_tx),
					 "Transmit"));
    tx_button_p = static_cast<Gtk::Button*>(right_tool_list.back()->get_widget());
    
    tool_bar.pack_start(left_tool_bar, false, false);
    tool_bar.pack_end(right_tool_bar, false, false);
  }

  Gtk::VBox* vbox_p = manage(new Gtk::VBox);
  add(*vbox_p);

  vbox_p->pack_start(menu_bar, false, false);
  if (tnc_p) vbox_p->pack_start(tool_bar, false, false);

  // make a display bar
  display_line_p = manage(new DisplayLine(standard_size));
  vbox_p->pack_start(*display_line_p, false, false);

  vbox_p->pack_start(*vpaned_p, true, true);
  vpaned_p->set_handle_size (8);
  vpaned_p->set_gutter_size (10);                       
   
  vpaned_p->add1 (sendwin);
  vpaned_p->add2 (*receivewin_p);

  status_line_p = manage(new StatusLine(tnc_p, standard_size));
  vbox_p->pack_end(*status_line_p, false, false);

  set_flags(GTK_HAS_FOCUS);

  Gtk::Main::input.connect(SigC::slot(this, &MainScreen::process_receive_pipe), receive_pipe.get_read_fd(), GDK_INPUT_READ);
  Gtk::Main::timeout.connect(SigC::slot(this, &MainScreen::timer_event_handler), TIMER_INTERVAL);

  if (tnc_p) {
    sendwin.text.mouse_right_clicked.connect(SigC::slot(this, &MainScreen::mouse_popup));
    receivewin_p->mouse_right_clicked.connect(SigC::slot(this, &MainScreen::mouse_popup));
    receivewin_p->activate_textselected_items.connect(SigC::slot(this, &MainScreen::activate_textselected_items));
    receivewin_p->disactivate_textselected_items.connect(SigC::slot(this, &MainScreen::disactivate_textselected_items));

    display_capture_status();
    display_send_mode_status();
    display_current_stream();
    display_connected_status();
    display_mode();
    set_call_lock_button();
    set_auto_cq_button();
    set_speed_lock_button();
    set_ident_button();
    set_sync_button();
    set_abort_button();
    set_rx_button();
    set_tx_button();
    set_change_speed_button();
    display_freebytes();
  }
  show_all();
}

MainScreen::~MainScreen(void) {
  delete receivewin_p;
}

void MainScreen::process_receive_pipe(gint fd, GdkInputCondition) {
  if (fd == receive_pipe.get_read_fd()) {
    usleep(10000); // as the com port only runs at 9600 baud, we can let a few characters
    // accumulate in receive_pipe and be even more kind to system resources
    tnc_base_p->process_received_bytes();
  }
}

gint MainScreen::timer_event_handler(void) {

// send any connect script being run, or any parameter file being sent

  if (event_slots_p->get_connect_script_flag() == EventSlots::running) event_slots_p->run_connect_script();
  if (event_slots_p->get_send_parms_flag() == EventSlots::running) {
    event_slots_p->send_parms();
    // Gtk+ needs some help in handling queued X events
    while(Gtk::Main::events_pending()) Gtk::Main::iteration();
  }

// check if anything needs to be sent out of tr_buffer to send_pipe
// (this includes a check whether there is a command in tr_buffer - the
// only command normally placed in the buffer is the receive command 'E':
// others are placed directly in send_pipe)

  int letter = tr_buffer.view_letter();
  if (letter == '\n' || letter == CMDinbuffer
      || letter == 1 || letter == 8
      || letter == 20 || letter == 26) {
    flush_chars = true;
  }

  if (tnc_p) {  // in normal mode
    if (flush_chars
	|| (tnc_p->tnc_func.active_port
	    && tnc_p->tnc_func.hfmode != Tnc_func::packet
	    && ((prog_func.send_mode == Prog_func::word
		 && tr_buffer.letters_used()) || 
		(prog_func.send_mode == Prog_func::guard 
		 && tr_buffer.letters_used() > GUARD_QUEUE)))) {
      letters_to_send_flag = true;
    }
  }
  else if (letter != -1) letters_to_send_flag = true; // in set-up mode
    
// send anything to be sent

  if (letters_to_send_flag) {
    tnc_base_p->send_to_tnc(flush_chars);
    if (!tnc_p || !tnc_p->get_unsent_frame_flag()) {
      letters_to_send_flag = false;
      flush_chars = false;
    }
  }

// check if anything to be sent out of filesend buffers
  if(!buffer_list.is_empty()) {
    if (!buffer_count) buffer_count = 1;
    buffer_list.reset(DList_enum::bottom_end);
    FileBuffer* file_buffer_ptr;
    int loop_count;
    for (loop_count = 0;
	 (file_buffer_ptr = (FileBuffer*)buffer_list.inspect(DList_enum::up)) != 0;
	 loop_count++) {
      if (!file_buffer_ptr->load_buffer()) {
	// if FileBuffer::load_buffer() does not return TRUE
	// then it indicates that the FileBuffer object is be extracted and deleted
	buffer_list.extract();
	delete file_buffer_ptr; // this will also delete the UploadDialog object
	loop_count--;
	if (buffer_count) buffer_count--;
	set_auto_cq_button();  // in case it is a CqsendBuffer object which has been extracted
      }
      else tnc_p->send_file(file_buffer_ptr, buffer_count, loop_count);
    }
    buffer_count = loop_count;
    // Gtk+ needs some help in handling queued X events
    while(Gtk::Main::events_pending()) Gtk::Main::iteration();
  }
  else buffer_count = 0; 

// get Kam information, check free bytes in Kam buffer and check the keep alive status
  if (tnc_p) {
    tnc_p->get_info();
    if (count == 20) {
      tnc_p->find_freebytes();
      if (tnc_p->tnc_func.keep_alive.keep_alive_flag) tnc_p->check_keep_alive();
      count = 0;
    }
    else count++;
  }

// check the scrolling condition
  receivewin_p->check_scroll_condition();

// check to see if we are beeping
  if (beep_count) {
    if (!interval_count) {
      ::beep();
      beep_count--;
      if (beep_count) interval_count = INTERVAL_COUNT - 1;
    }
    else interval_count--;
  }

  // check whether we have received SIGQUIT, SIGTERM SIGINT SIGHUP and SIGPIPE
  if (prog_func.exitflag) destroy();

  return true; // we want a multi-shot timer
}


void MainScreen::activate_textselected_items(void) {
  menu_copy_item_p->set_sensitive(true);
  menu_print_selection_item_p->set_sensitive(true);
  menu_save_selection_item_p->set_sensitive(true);
  mouse_copy_item_p->set_sensitive(true);
  mouse_print_selection_item_p->set_sensitive(true);
  mouse_save_selection_item_p->set_sensitive(true);
}

void MainScreen::disactivate_textselected_items(void) {
  menu_copy_item_p->set_sensitive(false);
  menu_print_selection_item_p->set_sensitive(false);
  menu_save_selection_item_p->set_sensitive(false);
  mouse_copy_item_p->set_sensitive(false);
  mouse_print_selection_item_p->set_sensitive(false);
  mouse_save_selection_item_p->set_sensitive(false);
}

void MainScreen::show_packetmenu_streams(void) {

  if (hf_stream_status == MainScreen::tor) {

    Gtk::Menu_Helpers::MenuList& hf_list = hf_menu_p->items();
    hf_list.pop_back();
    Gtk::Menu_Helpers::MenuList& hf_mouse_list = mouse_hf_menu_p->items();
    hf_mouse_list.pop_back();

    char hfstreamlabel[] = "Hf-_A";
    char* char_p = hfstreamlabel + 4;
    int stream;
    for (stream = 0; stream < MAXUSERS; stream++, (*char_p)++) {
      hf_list.push_back(Gtk::Menu_Helpers::MenuElem(hfstreamlabel,
				  SigC::bind(SigC::slot(event_slots_p, &EventSlots::hf_menu), stream)));
    }
    *char_p = 0;
    char_p--;
    *char_p = 'A';
    for (stream = 0; stream < MAXUSERS; stream++, (*char_p)++) {
      hf_mouse_list.push_back(Gtk::Menu_Helpers::MenuElem(hfstreamlabel,
				   SigC::bind(SigC::slot(event_slots_p, &EventSlots::hf_menu), stream)));
    }

    hf_stream_status = MainScreen::packet;
  }
}


void MainScreen::show_tormenu_streams(void) {
  
  if (hf_stream_status == MainScreen::packet) {

    Gtk::Menu_Helpers::MenuList& hf_list = hf_menu_p->items();
    while (!hf_list.empty()) hf_list.pop_back();
    Gtk::Menu_Helpers::MenuList& hf_mouse_list = mouse_hf_menu_p->items();
    while (!hf_mouse_list.empty()) hf_mouse_list.pop_back();

    hf_list.push_back(Gtk::Menu_Helpers::MenuElem("Hf-_Tor",
      SigC::bind(SigC::slot(event_slots_p, &EventSlots::hf_menu), 0)));
    hf_mouse_list.push_back(Gtk::Menu_Helpers::MenuElem("Hf-Tor",
      SigC::bind(SigC::slot(event_slots_p, &EventSlots::hf_menu), 0)));

    hf_stream_status = MainScreen::tor;
  }
}

void MainScreen::update_file_load_items(void) {
  if (tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()]
      [tnc_p->tnc_func.active_port] != Tnc_func::connected
      || (tnc_p->tnc_func.active_port
	  && (tnc_p->tnc_func.hfmode == Tnc_func::amtor
	      || (tnc_p->tnc_func.hfmode == Tnc_func::tor
		  && tnc_p->tnc_func.tor_connected_mode != Tnc_func::pactor_gtor)))) {
    download_item_p->set_sensitive(false);
    upload_7plus_item_p->set_sensitive(false);
    upload_binary_item_p->set_sensitive(false);

  }
  else {
    download_item_p->set_sensitive(true);
    upload_7plus_item_p->set_sensitive(true);
    upload_binary_item_p->set_sensitive(true);
  }
  if (tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()]
      [tnc_p->tnc_func.active_port] != Tnc_func::connected
      && (!tnc_p->tnc_func.active_port
	  || tnc_p->tnc_func.hfmode == Tnc_func::packet)) {
    upload_item_p->set_sensitive(false);
  }
  else upload_item_p->set_sensitive(true);
}

void MainScreen::update_print_mark_items(void) {
    if (tnc_p->tnc_func.print_list_ptr->get_print_status(tnc_p->tnc_func.active_stream(),
	  tnc_p->tnc_func.active_port) ==  PrintList::off) {
      menu_print_from_mark_item_p->set_sensitive(false);
      menu_set_print_mark_item_p->set_sensitive(true);
      mouse_print_from_mark_item_p->set_sensitive(false);
      mouse_set_print_mark_item_p->set_sensitive(true);
    }
    else {
      menu_print_from_mark_item_p->set_sensitive(true);
      menu_set_print_mark_item_p->set_sensitive(false);
      mouse_print_from_mark_item_p->set_sensitive(true);
      mouse_set_print_mark_item_p->set_sensitive(false);
    }
}

void MainScreen::update_qso_counter_items(void) {
  if (prog_func.qso_count) {
    menu_qso_count_item_p->set_sensitive(false);
    mouse_qso_count_item_p->set_sensitive(false);
  }
  else {
    menu_qso_count_item_p->set_sensitive(true);
    mouse_qso_count_item_p->set_sensitive(true);
  }
}

void MainScreen::set_connect_button(void) {
  if (tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port]
      != Tnc_func::connected
      && (!tnc_p->tnc_func.active_port ||
	  (tnc_p->tnc_func.hfmode != Tnc_func::rtty
	   && tnc_p->tnc_func.hfmode != Tnc_func::tor
	   && tnc_p->tnc_func.hfmode != Tnc_func::ascii
	   && tnc_p->tnc_func.hfmode != Tnc_func::cw))) {
    connect_button_p->set_relief(GTK_RELIEF_NORMAL);
    connect_button_p->set_sensitive(true);
  }
  else {
    connect_button_p->set_relief(GTK_RELIEF_NONE);
    connect_button_p->set_sensitive(false);
  }
}

void MainScreen::set_disconnect_button(void) {
  if (!tnc_p->tnc_func.active_port
      || tnc_p->tnc_func.hfmode == Tnc_func::packet
      || tnc_p->tnc_func.hfmode == Tnc_func::amtor
      || tnc_p->tnc_func.hfmode == Tnc_func::gtor
      || tnc_p->tnc_func.hfmode == Tnc_func::pactor
      || tnc_p->tnc_func.hfmode == Tnc_func::tor) {
    disconnect_button_p->set_relief(GTK_RELIEF_NORMAL);
    disconnect_button_p->set_sensitive(true);
  }
  else {
    disconnect_button_p->set_relief(GTK_RELIEF_NONE);
    disconnect_button_p->set_sensitive(false);
  }
}

void MainScreen::set_capture_button(void) {
  setting_capture_button_flag = true;
  if (tnc_p->tnc_func.capturefile_flag) capture_button_p->set_active(true);
  else capture_button_p->set_active(false);
  setting_capture_button_flag = false;
}

void MainScreen::set_call_lock_button(void) {
  setting_call_lock_button_flag = true;
  if (!tnc_p->tnc_func.active_port || tnc_p->tnc_func.hfmode == Tnc_func::packet) {
    call_lock_button_p->set_relief(GTK_RELIEF_NONE);
    call_lock_button_p->set_sensitive(false);
    call_lock_button_p->set_active(false);
  }
  else if (tnc_p->tnc_func.hisCall_lock == Tnc_func::on) {
    call_lock_button_p->set_relief(GTK_RELIEF_NORMAL);
    call_lock_button_p->set_sensitive(true);
    call_lock_button_p->set_active(true);
  }
  else {
    call_lock_button_p->set_relief(GTK_RELIEF_NORMAL);
    call_lock_button_p->set_sensitive(true);
    call_lock_button_p->set_active(true); // we need to do this because of a bug in Gtk+ 1.2.8
    call_lock_button_p->set_active(false);
  }
  setting_call_lock_button_flag = false;
}

void MainScreen::set_auto_cq_button(void) {
  setting_auto_cq_button_flag = true;
  if (!tnc_p->tnc_func.active_port
      || (tnc_p->tnc_func.hfmode != Tnc_func::pactor
	  && tnc_p->tnc_func.hfmode != Tnc_func::amtor
	  && tnc_p->tnc_func.hfmode != Tnc_func::gtor
	  && tnc_p->tnc_func.hfmode != Tnc_func::tor)
      || tnc_p->tnc_func.stream_status[0][1] == Tnc_func::connected
      || (!prog_func.sending_autocq
	  && buffer_list.get_upload_status(0, 1) == BufferList::file)) {
    auto_cq_button_p->set_relief(GTK_RELIEF_NONE);
    auto_cq_button_p->set_sensitive(false);
    auto_cq_button_p->set_active(false);
  }
  else if (prog_func.sending_autocq) {
    auto_cq_button_p->set_relief(GTK_RELIEF_NORMAL);
    auto_cq_button_p->set_sensitive(true);
    auto_cq_button_p->set_active(true);
  }
  else {
    auto_cq_button_p->set_relief(GTK_RELIEF_NORMAL);
    auto_cq_button_p->set_sensitive(true);
    auto_cq_button_p->set_active(true); // we need to do this because of a bug in Gtk+ 1.2.8
    auto_cq_button_p->set_active(false);
  }
  setting_auto_cq_button_flag = false;
}

void MainScreen::set_speed_lock_button(void) {
  setting_speed_lock_button_flag = true;
  if (!tnc_p->tnc_func.active_port
      || !(tnc_p->tnc_func.hfmode == Tnc_func::pactor
	   || tnc_p->tnc_func.hfmode == Tnc_func::gtor
	   || tnc_p->tnc_func.hfmode == Tnc_func::tor)) {
    speed_lock_button_p->set_relief(GTK_RELIEF_NONE);
    speed_lock_button_p->set_sensitive(false);
    speed_lock_button_p->set_active(false);
  }
  else if (tnc_p->tnc_func.speed_lock == Tnc_func::on) {
    speed_lock_button_p->set_relief(GTK_RELIEF_NORMAL);
    speed_lock_button_p->set_sensitive(true);
    speed_lock_button_p->set_active(true);
  }
  else {
    speed_lock_button_p->set_relief(GTK_RELIEF_NORMAL);
    speed_lock_button_p->set_sensitive(true);
    speed_lock_button_p->set_active(true); // we need to do this because of a bug in Gtk+ 1.2.8
    speed_lock_button_p->set_active(false);
  }
  setting_speed_lock_button_flag = false;
}

void MainScreen::set_ident_button(void) {
  if (tnc_p->tnc_func.active_port
      && tnc_p->tnc_func.hfmode != Tnc_func::packet) {
    ident_button_p->set_relief(GTK_RELIEF_NORMAL);
    ident_button_p->set_sensitive(true);
  }
  else {
    ident_button_p->set_relief(GTK_RELIEF_NONE);
    ident_button_p->set_sensitive(false);
  }
}

void MainScreen::set_sync_button(void) {
  if (tnc_p->tnc_func.active_port
      && (tnc_p->tnc_func.hfmode == Tnc_func::cw
	  || tnc_p->tnc_func.hfmode == Tnc_func::lamtor
	  || tnc_p->tnc_func.hfmode == Tnc_func::fec
	  || (tnc_p->tnc_func.hfmode == Tnc_func::amtor
	      && tnc_p->tnc_func.stream_status[0][1] == Tnc_func::disconnected))) {
    sync_button_p->set_relief(GTK_RELIEF_NORMAL);
    sync_button_p->set_sensitive(true);
  }
  else {
    sync_button_p->set_relief(GTK_RELIEF_NONE);
    sync_button_p->set_sensitive(false);
  }
}

void MainScreen::set_abort_button(void) {
  if (tnc_p->tnc_func.active_port
      && (tnc_p->tnc_func.hfmode == Tnc_func::amtor
	  || tnc_p->tnc_func.hfmode == Tnc_func::gtor
	  || tnc_p->tnc_func.hfmode == Tnc_func::pactor
	  || tnc_p->tnc_func.hfmode == Tnc_func::tor)) {
    abort_button_p->set_relief(GTK_RELIEF_NORMAL);
    abort_button_p->set_sensitive(true);
  }
  else {
    abort_button_p->set_relief(GTK_RELIEF_NONE);
    abort_button_p->set_sensitive(false);
  }
}

void MainScreen::set_rx_button(void) {
  if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) {
    rx_button_p->set_relief(GTK_RELIEF_NORMAL);
    rx_button_p->set_sensitive(true);
  }
  else {
    rx_button_p->set_relief(GTK_RELIEF_NONE);
    rx_button_p->set_sensitive(false);
  }
}

void MainScreen::set_tx_button(void) {
  if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) {
    tx_button_p->set_relief(GTK_RELIEF_NORMAL);
    tx_button_p->set_sensitive(true);
  }
  else {
    tx_button_p->set_relief(GTK_RELIEF_NONE);
    tx_button_p->set_sensitive(false);
  }
}

void MainScreen::set_change_speed_button(void) {
  if (tnc_p->tnc_func.active_port
      && (tnc_p->tnc_func.hfmode == Tnc_func::cw
	  || tnc_p->tnc_func.hfmode == Tnc_func::rtty
	  || tnc_p->tnc_func.hfmode == Tnc_func::ascii)) {
    change_speed_button_p->set_relief(GTK_RELIEF_NORMAL);
    change_speed_button_p->set_sensitive(true);
  }
  else {
    change_speed_button_p->set_relief(GTK_RELIEF_NONE);
    change_speed_button_p->set_sensitive(false);
  }
}

void MainScreen::set_dx_cluster_button(void) {
  setting_dx_cluster_button_flag = true;
  if (tnc_p->tnc_func.stream_status[0][MAXUSERS - 1] == Tnc_func::connected) {
    dx_cluster_button_p->set_relief(GTK_RELIEF_NONE);
    dx_cluster_button_p->set_sensitive(false);
    dx_cluster_button_p->set_active(false);
  }
  else if (tnc_p->tnc_func.read_cluster_flag) {
    dx_cluster_button_p->set_relief(GTK_RELIEF_NORMAL);
    dx_cluster_button_p->set_sensitive(true);
    dx_cluster_button_p->set_active(true);
  }
  else {
    dx_cluster_button_p->set_relief(GTK_RELIEF_NORMAL);
    dx_cluster_button_p->set_sensitive(true);
    dx_cluster_button_p->set_active(true); // we need to do this because of a bug in Gtk+ 1.2.8
    dx_cluster_button_p->set_active(false);
  }
  setting_dx_cluster_button_flag = false;
}

void MainScreen::display_capture_status(void) {
  if (!tnc_p->tnc_func.capturefile_flag) display_line_p->write_capture("Off");
  else {
    char text[6] = {0};
    if (tnc_p->tnc_func.capture_stream.port) strcpy(text, "Hf-");
    else strcpy(text, "Vhf-");
    if (!tnc_p->tnc_func.capture_stream.port || tnc_p->tnc_func.hfmode == Tnc_func::packet)  {
      char stream[2] = {0};
      *stream = tnc_p->tnc_func.capture_stream.stream + 0x41;
      strcat(text, stream);
    }
    else strcat(text, "Tor");
    display_line_p->write_capture(text);
  }
}

void MainScreen::display_current_stream(void) {
  char text[6] = {0};
  if (tnc_p->tnc_func.active_port) strcpy(text, "Hf-");
  else strcpy(text, "Vhf-");
  if (!tnc_p->tnc_func.active_port || tnc_p->tnc_func.hfmode == Tnc_func::packet)  {
    char stream[2] = {0};
    *stream = tnc_p->tnc_func.active_stream() + 0x41;
    strcat(text, stream);
  }
  else strcat(text, "Tor");
  display_line_p->write_stream(text);
}

void MainScreen::display_connected_status(void) {
  if (tnc_p->tnc_func.stream_status[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port]
      ==  Tnc_func::disconnected) {
    if (tnc_p->tnc_func.active_port && prog_func.sending_autocq) {
      if (prog_func.tor_autocq_mode == Prog_func::amtor) {
	status_line_p->write_connected_status("Auto-CQ");
      }
      else status_line_p->write_connected_status("Auto-CQ (Pt)");
    }
    else if (tnc_p->tnc_func.read_cluster_flag
	     && !tnc_p->tnc_func.active_port
	     && tnc_p->tnc_func.active_vhf_stream == MAXUSERS - 1) {
      status_line_p->write_connected_status("DX Cluster");
    }
    else status_line_p->write_connected_status("Disconnected");
  }
  else if (tnc_p->tnc_func.download_list_ptr->get_download_status(tnc_p->tnc_func.active_stream(),
						       tnc_p->tnc_func.active_port) == DownloadList::on) {
    status_line_p->write_connected_status("Download");
  }
  else status_line_p->write_connected_status("Connected");
}


void MainScreen::display_mode(void) {
  switch(tnc_p->tnc_func.hfmode) {
  case Tnc_func::packet:
    display_line_p->write_hfmode("Packet");
    break;
  case Tnc_func::pactor:
    display_line_p->write_hfmode("Pactor");
    break;
  case Tnc_func::rtty:
    display_line_p->write_hfmode("RTTY");
    break;
  case Tnc_func::ascii:
    display_line_p->write_hfmode("ASCII");
    break;
  case Tnc_func::amtor:
    display_line_p->write_hfmode("Amtor");
    break;
  case Tnc_func::lamtor:
    display_line_p->write_hfmode("Lamtor");
    break;
  case Tnc_func::fec:
    display_line_p->write_hfmode("FEC");
    break;
  case Tnc_func::gtor:
    display_line_p->write_hfmode("Gtor");
    break;
  case Tnc_func::gmon:
    display_line_p->write_hfmode("Gmon");
    break;
  case Tnc_func::tor:
    display_line_p->write_hfmode("Tor");
    break;
  case Tnc_func::cw:
    display_line_p->write_hfmode("CW");
    break;
  }
}

void MainScreen::display_send_mode_status(void) {
  if (prog_func.send_mode == Prog_func::word) display_line_p->write_hfsend("Word ");
  else if (prog_func.send_mode == Prog_func::guard) display_line_p->write_hfsend("Guard");
  else display_line_p->write_hfsend("Line ");
}

void MainScreen::display_callsign(void) {
  string text;
  if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet 
      && tnc_p->tnc_func.hisCall_lock == Tnc_func::on) text = '*';
  else text = ' ';
  text +=  tnc_p->tnc_func.hisCall[tnc_p->tnc_func.active_stream()][tnc_p->tnc_func.active_port];
  display_line_p->write_call(text.c_str());
}

void MainScreen::display_freebytes(void) {
  ostrstream strm;
  strm << tnc_p->get_active_freebytes() << ends;
  const char* text = strm.str();
  status_line_p->write_freebytes(text);
  delete[] text;
}

void MainScreen::update_lockinfo(void) {
  if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) {
    if (tnc_p->tnc_func.speed_lock == Tnc_func::on) status_line_p->write_lock_status("LOCK");
    else status_line_p->write_lock_status("");
  }
}

void MainScreen::update_torinfo(unsigned char info) {
// info has a default value of 255
  if (info != 255) info_byte = info; // if info = 255 we just use the value stored in info_byte
  if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) {
    if (info_byte & 1) status_line_p->write_idle_status("IDLE");
    else status_line_p->write_idle_status("");
    if (info_byte & 2) status_line_p->write_err_status("ERR");
    else if (info_byte & 4) status_line_p->write_err_status("CMB");
    else if (info_byte & 8) status_line_p->write_err_status("RQ ");
    else status_line_p->write_err_status("");
    if (info_byte & 16) status_line_p->write_huff_status("HUFF");
    else status_line_p->write_huff_status("");
    if (info_byte & 32) status_line_p->write_irs_status("ISS");
    else if (tnc_p->tnc_func.stream_status[0][1] == Tnc_func::connected) {
      status_line_p->write_irs_status("IRS");
    }
    else status_line_p->write_irs_status("");
    if (info_byte & 64) status_line_p->write_speed_status("200");
    else if (info_byte & 128) status_line_p->write_speed_status("300");
    else if (tnc_p->tnc_func.stream_status[0][1] == Tnc_func::connected) {
      status_line_p->write_speed_status("100");
    }
    else status_line_p->write_speed_status("");
  }
}

void MainScreen::update_txinfo(unsigned char tx) {
// tx has a default value of 255
  if (tx != 255) tx_byte = tx; // if tx = 255 we just use the value stored in tx_byte
  if (tnc_p->tnc_func.active_port && tnc_p->tnc_func.hfmode != Tnc_func::packet) {
    if ((tx_byte & 2) || tnc_p->tnc_func.sending_cw_flag) {
      status_line_p->write_tx_status("TX");
      if (tx_status == MainScreen::rx) tx_status = MainScreen::tx;
    }
    else {
      status_line_p->write_tx_status("RX");
      if (tx_status == MainScreen::tx) tx_status = MainScreen::rx;
    }
  }
}
