/* SCCS-info %W% %E% */

/*--------------------------------------------------------------------*/
/*								      */
/*		VCG : Visualization of Compiler Graphs		      */
/*		--------------------------------------		      */
/*								      */
/*   file:	   step3.c					      */
/*   version:	   1.00.00					      */
/*   creation:	   14.4.93					      */
/*   author:	   I. Lemke  (...-Version 0.99.99)		      */
/*		   G. Sander (Version 1.00.00-...)		      */  
/*		   Universitaet des Saarlandes, 66041 Saarbruecken    */
/*		   ESPRIT Project #5399 Compare 		      */
/*   description:  Layout phase 3: calculation of coordinates	      */
/*		   of nodes					      */
/*   status:	   in work					      */
/*								      */
/*--------------------------------------------------------------------*/

#ifndef lint
static char *id_string="$Id: step3.c,v 3.10 1994/08/09 10:44:03 sander Exp $";
#endif

/*
 *   Copyright (C) 1993, 1994 by Georg Sander, Iris Lemke, and
 *                               the Compare Consortium 
 *
 *  This program and documentation is free software; you can redistribute 
 *  it under the terms of the  GNU General Public License as published by
 *  the  Free Software Foundation;  either version 2  of the License,  or
 *  (at your option) any later version.
 *
 *  This  program  is  distributed  in  the hope that it will be useful,
 *  but  WITHOUT ANY WARRANTY;  without  even  the  implied  warranty of
 *  MERCHANTABILITY  or  FITNESS  FOR  A  PARTICULAR  PURPOSE.  See  the
 *  GNU General Public License for more details.
 *
 *  You  should  have  received a copy of the GNU General Public License
 *  along  with  this  program;  if  not,  write  to  the  Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  The software is available per anonymous ftp at ftp.cs.uni-sb.de.
 *  Contact  sander@cs.uni-sb.de  for additional information.
 */


/* 
 * $Log: step3.c,v $
 * Revision 3.10  1994/08/09  10:44:03  sander
 * macro touching for xraster corrected.
 *
 * Revision 3.9  1994/08/08  16:01:47  sander
 * Attributes xraster, xlraster, yraster added.
 *
 * Revision 3.8  1994/08/05  12:13:25  sander
 * Treelayout added. Attributes "treefactor" and "spreadlevel" added.
 * Scaling as abbreviation of "stretch/shrink" added.
 *
 * Revision 3.7  1994/08/02  15:36:12  sander
 * dumpmediumshifts corrected: regions can be combined if
 * their deflection pull them together.
 *
 * Revision 3.6  1994/06/07  14:09:59  sander
 * Splines implemented.
 * HP-UX, Linux, AIX, Sun-Os, IRIX compatibility tested.
 * The tool is now ready to be distributed.
 *
 * Revision 3.5  1994/05/16  08:56:03  sander
 * shape attribute (boxes, rhombs, ellipses, triangles) added.
 *
 * Revision 3.4  1994/05/05  08:20:30  sander
 * Bug with x_base and y_base solved.
 *
 * Revision 3.3  1994/04/27  16:05:19  sander
 * Some general changes for the PostScript driver.
 * Horizontal order added. Bug fixes of the folding phases:
 * Folding of nested graphs works now.
 *
 * Revision 3.2  1994/03/03  14:12:21  sander
 * Shift the unconnected parts together after the layout.
 *
 * Revision 3.1  1994/03/01  10:59:55  sander
 * Copyright and Gnu Licence message added.
 * Problem with "nearedges: no" and "selfloops" solved.
 *
 * Revision 2.3  1994/01/21  19:33:46  sander
 * VCG Version tested on Silicon Graphics IRIX, IBM R6000 AIX and Sun 3/60.
 * Option handling improved. Option -grabinputfocus installed.
 * X11 Font selection scheme implemented. The user can now select a font
 * during installation.
 * Sun K&R C (a nonansi compiler) tested. Some portabitility problems solved.
 *
 */ 


/************************************************************************
 * The situation here is the following:
 * -----------------------------------
 * We are still in the layout phase of the graph. The graph is in
 * adjacency list representation available, further it is partitioned
 * into layers that represent a proper hierarchy. All nodes have already
 * their relative position inside the layers.
 * We want to calculate now the co-ordinated of the nodes.
 * We have:
 *    1)  The array layer contains all visible nodes.
 *	  They are distributed at the layer[i] lists and connected by two
 *	  lists TPRED and TSUCC to allow to traverse the nodes of one
 *	  layer[i] backwards and forwards.
 *    2)  Note that the nodes reacheable via forward connections are now
 *	  in the TSUCC and TPRED lists, too.
 *	  TANZ(layer[i]) is the number of nodes in layer[i].
 *    3)  The hierarchy in layer is proper.
 *    4)  Further, all visible nodes are in nodelist, labellist and dummylist.
 *    5)  All pot. visible edges are in the lists edgelist or tmpedgelist,
 *	  Visible edges can be detected by the EINVISIBLE flag (==0) in these
 *	  lists. Note: invisible edges may also be in edgelist or tmpedgelist.
 *	  An edge is visible iff
 *		   a) it is used in the adjacency lists.
 *		or b) it is a direct neigbour edge in NCONNECT(v) for
 *		      some node v.
 *    6)  maxindeg and maxoutdeg are upper estimations of NINDEG and
 *	  NOUTDEG of nodes.
 *    7)  maxdepth+1 is the maximal layer !!! NOT maxdepth !!!
 *    8)  NTIEFE(node) is filled for all nodes. NINDEG and NOUTDEG are
 *	  filled. Forward connections are not counted.
 *	  NCONNECT(node) is filled for nodes that have direct neighbours
 *	  in the layout. The edges in NCONNECT are not anymore in the
 *	  adjacency lists, but still visible.
 *	  See point 2 !!!
 *    9)  Reverted edges are marked with EART(e)='R' and bidirectional
 *	  edges between adjacent levels are marked with EART(e)='D'.
 *	  Self loops don't anymore exist.
 *    10) NPOS(v) gives the horizontal position of a node inside the
 *	  layer. Adjacency edges are sorted according to these NPOS
 *	  values. The ordering inside a layer is such that the number
 *	  of cossings is small (but may be not optimal).
 *    11) NSUCCL(v) and NSUCCR(v) are the leftest and rightest successor
 *	  edge of v, and NPREDL(v) and NPREDR(v) the leftest and rightest 
 *	  predecessor edge. 
 *
 * To calculate the y-coordinates, we must simply look which level
 * the node belongs to. All nodes of one level have the same y-coordinates.
 * But for the x-position, we must shift the nodes inside the layer
 * to calculate the correct space between the nodes. This is done
 * by two steps: first we try to assign positions to the most left
 * and right upper and lower node. The we center all nodes with respect
 * to their parent and child nodes. We use a special weight G here,
 * which has to be minimized.
 *
 * After this, we have the following situation:
 *    1-11) same as before
 *    12)   All nodes have filled NX, NY, NWIDTH and NHEIGHT such that
 *	    they do not overlap. NX and NY are absolutely. NWIDTH and
 *	    NHEIGHT are stretched or shrinked according to the local
 *	    factors at the nodes.
 *
 * This file provides the following functions:
 * ------------------------------------------
 * step3_main		Main routine to calculate coordinates 
 *
 ************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "globals.h"
#include "alloc.h"
#include "main.h"
#include "folding.h"
#include "drawlib.h"
#include "steps.h"
#include "timing.h"

/* Prototypes
 * ----------
 */

#undef DRAWDEBUG
#ifdef DRAWDEBUG
extern void     debug_display_part	_PP((void));
extern void     display_part		_PP((void));
static void 	drawdebug_show_graph	_PP((char *m,int i,int j,int k));
#else
#define		drawdebug_show_graph(a,b,c,d)		
#endif

static void	init_xy_coordinates	_PP((void));
static void	center_layout		_PP((void));
static void	shift_left_layout	_PP((void));

static void 	shift_left_together_layout _PP((void));
static void 	shift_intermixed_part	   _PP((int t));
static void 	shift_left_part		   _PP((GNODE node, int i));
static void 	mark_and_calc_maxx	   _PP((GNODE node));

static void	iterate_dump_mediumshifts _PP((void));
static int	dumpmediumshift		_PP((int f));

static void	iterate_centershifts	_PP((void));
static int 	changed_nw_sum          _PP((void));

static int	nwsdump_mediumshift 	_PP((int i,int dir));
static int	nwpdump_mediumshift 	_PP((int i,int dir));
static int	nwdump_mediumshift 	_PP((int i,int dir));
static int 	summarize_dumpshift	_PP((int i,int dir));
static int	center_weight		_PP((void));
static int	center_layer		_PP((int i));
static int	do_leftshifts		_PP((int i));
static int	do_rightshifts		_PP((int i));

static int	nw		_PP((GNODE node));
static int	nwbend		_PP((GNODE node,GNODE lnode, GNODE rnode));
static int	ews		_PP((GEDGE edge));
static int	nws		_PP((GNODE node));
static int	ewp		_PP((GEDGE edge));
static int	nwp		_PP((GNODE node));

static void	save_plevel	_PP((int i));



/* Global variables
 * ----------------
 */

/* The following two arrays are used to store the constraints of the
 * nodes inside the levels.
 * levelshift gives the summary number of pixels a node should be shifted.
 * levelweight gives the number of constains.
 * Thus levelshift[i]/levelweight[i] is the number of real pixels
 * a node with position i should be shifted.
 */

static int	size_of_levelsw;	/* size of arrays levelshift */
					/* and levelweight	     */ 
static int	*levelshift =NULL;
static int	*levelweight=NULL;

/* array where we can temporary save one layer. It has always the
 * same size as the levelshift-array.
 * This is also used in step4.
 */

static GNODE    *slayer_array = NULL;


/* Two arrays that allow to find a node's list cell in the TPRED and TSUCC  
 * lists of a layer. This is used in step 4 later.
 * Here, these arrays are overloaded with slayer_array and levelweight.
 */

GNLIST	 *tpred_connection1 = NULL;
GNLIST	 *tpred_connection2 = NULL;


/* Macros
 * ------
 */

#define forward_connection1(c)  ((CEDGE(c))&& (EEND(CEDGE(c))==CTARGET(c)))
#define forward_connection2(c)  ((CEDGE2(c))&&(EEND(CEDGE2(c))==CTARGET2(c)))
#define backward_connection1(c) ((CEDGE(c))&& (EEND(CEDGE(c))!=CTARGET(c)))
#define backward_connection2(c) ((CEDGE2(c))&&(EEND(CEDGE2(c))!=CTARGET2(c)))


/*--------------------------------------------------------------------*/
/*  Calculation of co-ordinates 				      */
/*--------------------------------------------------------------------*/

#ifdef DRAWDEBUG
static int firstcall = 1;
#endif

void	step3_main()
{
	start_time();
	debugmessage("step3_main","");
	assert((layer));

#ifdef DRAWDEBUG
	if (firstcall) {
		firstcall = 0;
		PRINTF("Enter ` ` into the drawwindow\n");
		display_part();
		return;
	}
#endif
	
        /* First we calculate how many up-down ports a node has, and how
         * the edges are scheduled among these ports. See step 4.
         */
 
	calc_all_ports(0);

	/* Calculate NWIDTH and NHEIGHT of all nodes */

	calc_all_node_sizes();

	/* Start with a minimal-distance-leftbound layout */

	init_xy_coordinates();
	
	/* Center the layout at the y-axis */

	center_layout();

	alloc_levelshift();
	

	/* Shift nodes of an edge by a medium measure, to become a
	 * not too dense situation. We do until the leftest and
	 * rightest upper and lower nodes do not change anymore. 
	 */

	iterate_dump_mediumshifts();

/*	Not anymore needed:
 *      ------------------ 
 *	iterate_mediumshifts();
 *	iterate_pendling_mediumshifts();
 */

	/* Shift nodes such that they are centered between all edges
	 * pointing to or from the nodes. 
	 */

	iterate_centershifts();
	
	/* Shift the layout to the left such that the minimal x coordinate
	 * is G_xbase.
	 */

	shift_left_layout();
	shift_left_together_layout(); 

	/* Once again: check the ports, because they may have changed
	 * for some shapes.
	 */
	calc_all_ports(1);

	stop_time("step3_main");

}


/*--------------------------------------------------------------------*/
/*  Allocation of the arrays levelshift, etc.  */
/*--------------------------------------------------------------------*/
	

void alloc_levelshift()
{
	debugmessage("alloc_levelshift","");

	if (sizeof(GNLIST)<sizeof(GNODE))
		Fatal_error("Assertion 1 about structs is wrong.","");
	if (sizeof(GNLIST)<sizeof(int))
		Fatal_error("Assertion 2 about structs is wrong.","");

#ifdef TESTLAYOUT
	PRINTF("Maximal nodes per layer: %d\n", max_nodes_per_layer);
	PRINTF("Maximal layers: %d\n", maxdepth);
#endif
	if (max_nodes_per_layer+2 > size_of_levelsw) {
		if (levelshift)        free(levelshift);
		if (tpred_connection1)  free(tpred_connection1);
		if (tpred_connection2)  free(tpred_connection2);
		levelshift = (int *)malloc((max_nodes_per_layer+2)
					* sizeof(int));
		tpred_connection1 = (GNLIST *)malloc((max_nodes_per_layer+2)
					* sizeof(GNLIST));
		tpred_connection2 = (GNLIST *)malloc((max_nodes_per_layer+2)
					* sizeof(GNLIST));
		levelweight  = (int *)tpred_connection1;
		slayer_array = (GNODE *)tpred_connection2;
		if ((!levelshift)||(!levelweight)||(!slayer_array))
				Fatal_error("memory exhausted","");
		size_of_levelsw = max_nodes_per_layer+2;
#ifdef DEBUG
		PRINTF("Sizeof tables `levelshift',`levelweight',`slayer_array': %ld Bytes\n",
			(max_nodes_per_layer+2)*sizeof(int));
#endif
	}
}


/*--------------------------------------------------------------------*/
/*  Calculation of NWIDTH and NHEIGHT				      */
/*--------------------------------------------------------------------*/

/* Traverse all visible nodes and set NWIDTH and NHEIGHT of the nodes.
 * If NWIDTH and NHEIGHT are not already set, they are derived from
 * the size of the label. 
 */

void calc_all_node_sizes()
{
	GNODE v;
	int   h,hh;
	ADJEDGE a;

	debugmessage("calc_all_node_sizes","");
	v = nodelist;
	while (v) {
		calc_node_size(v);

		if (  (G_orientation==LEFT_TO_RIGHT) 
		    ||(G_orientation==RIGHT_TO_LEFT)) {
			h          = NWIDTH(v);
			NWIDTH(v)  = NHEIGHT(v);
			NHEIGHT(v) = h;
		}
		v = NNEXT(v);
	}

	/* For the labellist, width and height are already set.
	 * See folding.c: adapt_labelpos.
 	 * But if orientation is left to right or converse, we must 
	 * turn here.
	 */
	if (  (G_orientation==LEFT_TO_RIGHT) 
	    ||(G_orientation==RIGHT_TO_LEFT)) {
		v = labellist;
		while (v) {
			h          = NWIDTH(v);
			NWIDTH(v)  = NHEIGHT(v);
			NHEIGHT(v) = h;
			v = NNEXT(v);
		}
	}

	v = dummylist;
	while (v) {
		if (NANCHORNODE(v)) {
			a = NPRED(v);
			h = 0;
			while (a) { a=ANEXT(a); h++; }
			a = NSUCC(v);
			hh = 0;
			while (a) { a=ANEXT(a); hh++; }
			if (hh>h) h = hh;
			assert((NCONNECT(v)));
			NHEIGHT(v) = NHEIGHT(CTARGET(NCONNECT(v)));
			NWIDTH(v) =  (h+1)*G_xspace;
		}
		else {
			NWIDTH(v)   = 0;  
			NHEIGHT(v)  = 0;
		} 
		if (  (G_orientation==LEFT_TO_RIGHT) 
	   	    ||(G_orientation==RIGHT_TO_LEFT)) {
			h          = NWIDTH(v);
			NWIDTH(v)  = NHEIGHT(v);
			NHEIGHT(v) = h;
		}
		v = NNEXT(v);
	}
}


/*--------------------------------------------------------------------*/
/*  Initialize x and y coordinates				      */
/*--------------------------------------------------------------------*/

/*  Initialize with minimal x-y coordinates
 *  ---------------------------------------
 *  The initial layout is the leftbound. All nodes have minimal
 *  distance between, ignoring constraints because of edges. 
 *  Something like this (Example)
 *    A B C 
 *    D E F G H
 *    I J
 *    K L M 
 */

#define xralign(a)  ((((a)+G_xraster-1)/G_xraster)*G_xraster)
#define yralign(a)  ((((a)+G_yraster-1)/G_yraster)*G_yraster)
#define dxralign(a) ((((a)+G_dxraster-1)/G_dxraster)*G_dxraster)
#define dyralign(a) ((((a)+G_dyraster-1)/G_dyraster)*G_dyraster)

#define xllalign(a)  ((((a)-G_xraster+1)/G_xraster)*G_xraster)
#define xlalign(a)  ((((a))/G_xraster)*G_xraster)
#define ylalign(a)  ((((a))/G_yraster)*G_yraster)
#define dxlalign(a) ((((a))/G_dxraster)*G_dxraster)
#define dylalign(a) ((((a))/G_dyraster)*G_dyraster)


static void init_xy_coordinates()
{
	int	actxpos, actypos;
	int	maxboxheight;
	GNLIST	li;
	GNODE 	v;
	int	i;

	debugmessage("init_xy_coordinates","");
	if (G_yspace<5)  G_yspace=5;
	if (G_xspace<5)  G_xspace=5;
	if (G_dspace==0) {
		if (G_spline) G_dspace = 4*G_xspace/5;
		else	      G_dspace = G_xspace/2;
	}
	if (G_flat_factor<1)   G_flat_factor = 1;
	if (G_flat_factor>100) G_flat_factor = 100;

	actypos = G_ybase;
	actypos = yralign(actypos);

	for (i=0; i<=maxdepth+1; i++) {
		actxpos = G_xbase;
		maxboxheight = 0;
		li	= TSUCC(layer[i]);
		while (li) {
			v = GNNODE(li);

			if ((NWIDTH(v)==0)&&(NHEIGHT(v)==0))
			      NX(v) = dxralign(actxpos+NWIDTH(v)/2)-NWIDTH(v)/2;
			else  NX(v) =  xralign(actxpos+NWIDTH(v)/2)-NWIDTH(v)/2;
			NY(v) = actypos;

			actxpos = NX(v) + NWIDTH(v) + G_xspace;
			if (maxboxheight<NHEIGHT(v))  
				maxboxheight = NHEIGHT(v);
			li = GNNEXT(li);
		}

		if (G_yalign==AL_CENTER) {
			li	= TSUCC(layer[i]);
			while (li) {
				NY(GNNODE(li))   += (maxboxheight-
						      NHEIGHT(GNNODE(li)))/2;
				li = GNNEXT(li);
			}
		}
		else if (G_yalign==AL_BOTTOM) {
			li	= TSUCC(layer[i]);
			while (li) {
				NY(GNNODE(li))   += (maxboxheight-
						      NHEIGHT(GNNODE(li)));
				li = GNNEXT(li);
			}
		}
		actypos += (maxboxheight + G_yspace);
		actypos = yralign(actypos);
	}
}


/*--------------------------------------------------------------------*/
/*  Center the layout						      */
/*--------------------------------------------------------------------*/

/*  Center the layers
 *  -----------------
 *  This moves the layers independently, such that the layers are
 *  centered at the y-axis. The result is symmetrical.
 *  Something like this (Example)
 *	A B C 
 *    D E F G H    (the x==0 axis goes though B, F and L)
 *	 I J
 *	K L M 
 */


static void center_layout()
{
	GNLIST	li;
	int	i;
	int	shift_value;

	debugmessage("center_layout","");
	for (i=0; i<=maxdepth+1; i++) {
		li	= TPRED(layer[i]); /* rightest node */
		if (li) {
			assert((TSUCC(layer[i])));
			shift_value = (NX(GNNODE(li))+NWIDTH(GNNODE(li)))/2;  
			shift_value = xlalign(shift_value);
			while (li) {
				NX(GNNODE(li)) -= shift_value;
				li = GNNEXT(li);
			}
		}
	}
}

/*--------------------------------------------------------------------*/
/*  Shift the layout to the left x-position			      */
/*--------------------------------------------------------------------*/

/*  Shift the complete layout 
 *  -------------------------
 *  This moves all layers together, such that the minimal x value
 *  is G_xbase, and the minimal y value is G_ybase. With other words, 
 *  the layout is shifted to the left upper part of the coordinate
 *  system.
 */


static void shift_left_layout()
{
	GNLIST li;
	int	i;
	int	minx,miny;

	debugmessage("shift_left_layout","");

	/* first, calculate minx and miny */
	minx = miny = MAXINT;
	for (i=0; i<=maxdepth+1; i++) {
		li	= TSUCC(layer[i]);
		while (li) {
			if (NX(GNNODE(li))<minx) minx = NX(GNNODE(li));
			if (NY(GNNODE(li))<miny) miny = NY(GNNODE(li));
			li = GNNEXT(li);
		}
	}
	
	minx = minx - G_xbase;
	miny = miny - G_ybase;
	minx = xlalign(minx);
	miny = xlalign(miny);
	for (i=0; i<=maxdepth+1; i++) {
		li	= TSUCC(layer[i]);
		while (li) {
			NX(GNNODE(li)) -= minx;
			NY(GNNODE(li)) -= miny;
			li = GNNEXT(li);
		}
	}
	
}



/*--------------------------------------------------------------------*/
/*  Shift the layout together					      */
/*--------------------------------------------------------------------*/

/*  Shift unconnected parts together
 *  --------------------------------
 *  It may happen that the graph is splitted into several connected parts,
 *  and that only the leftest part is shifted to the left upper part of
 *  the coordinate system, because dump shifting has moved the other
 *  parts to much to the right.
 *  This is now corrected here. We move the layout together such that
 *  the parts have at most a minimal x - distance G_xspace.
 *  In y - direction, this is not necessary.  
 */

/*  graph_maxx is the maximal x-coordinate of the actual inspected
 *  graph.
 */

static int graph_maxx;

static void shift_left_together_layout()
{
	GNLIST li;
	int	i;
	int    minx, part_is_missing; 
	GNODE  node;

	debugmessage("shift_left_together_layout","");

	/* first, set the NMARK field to 0 */
	for (i=0; i<=maxdepth+1; i++) {
		li	= TSUCC(layer[i]);
		while (li) {
			NMARK(GNNODE(li)) = 0;
			li = GNNEXT(li);
		}
	}

	graph_maxx = G_xbase-G_xspace-5;

	part_is_missing = 1;
	while (part_is_missing) {

		part_is_missing =0;
		minx = MAXINT;
		node = (GNODE)0;
		/* look for an untouched connected component, i.e.
		 * for the node with minimal x-co-ordinate.
		 */
		for (i=0; i<=maxdepth+1; i++) {
			li	= TSUCC(layer[i]);
			while (li) {
				if (NMARK(GNNODE(li)) == 0) {
					if (minx > NX(GNNODE(li))) {
						node = GNNODE(li);
						minx = NX(node);
					}
					break;
				}
				li = GNNEXT(li);
			}
		}
		if (node) {
			assert((NMARK(node)==0));
			part_is_missing =1;
			if (minx>graph_maxx+G_xspace+5) {
				i = minx-graph_maxx-G_xspace-5;
				i = xlalign(i);
				shift_left_part(node,i);
				shift_intermixed_part(i);
			}
			mark_and_calc_maxx(node);
		}
#ifdef DRAWDEBUG
#ifdef NEVER
		PRINTF("After one shift left together\n");
		PRINTF("Enter CR into the text window\n");
		step4_main();
		debug_display_part();
		fgetc(stdin);
#endif
#endif
	}
}


/* Search and shift left a subgraph that is intermixed with another
 * ----------------------------------------------------------------
 * E.g. the following situation:
 *
 *              A             We should consider this as one part, i.e.
 *             / \            the subgraph A-C,E must be shifted by the
 *            /   \           same amount as B,D-F.
 *        B  C  D  E
 *         \   /
 *          \ /
 *           F
 */

static void shift_intermixed_part(t)
int t;
{
	int i, intermixed_part_found;
	GNLIST li;
	GNODE node;

	intermixed_part_found = 1;

	while (intermixed_part_found) {

		intermixed_part_found = 0;
		node = NULL;

		/* look for an untouched intermixed component, i.e.
		 * for the untouched node whose successor is touched.
		 */
		for (i=0; i<=maxdepth+1; i++) {
			li	= TSUCC(layer[i]);
			while (li) {
				if (NMARK(GNNODE(li)) == 0) {
					if (  (GNNEXT(li))
					    &&(NMARK(GNNODE(GNNEXT(li))))) {
						node = GNNODE(li);
						intermixed_part_found = 1;
						break;
					}
				}
				li = GNNEXT(li);
			}
			if (intermixed_part_found) break;
		}

		if (intermixed_part_found) {
			assert((NMARK(node)==0));
			shift_left_part(node,t);
			mark_and_calc_maxx(node);
		}
	}
}



/* Shift left a connected part by i pixels
 * ---------------------------------------
 * this is done for all nodes reachable from node.
 */

static void shift_left_part(node, i)
GNODE node;
int i;
{
	ADJEDGE e;

	debugmessage("shift_left_part","");

	if (NMARK(node)) return;
	NMARK(node) = 1;

	NX(node) -= i;

	if (NCONNECT(node)) {
		if (CTARGET(NCONNECT(node)))
			shift_left_part(CTARGET(NCONNECT(node)),i);
		if (CTARGET2(NCONNECT(node)))
			shift_left_part(CTARGET2(NCONNECT(node)),i);
	}
	e = NSUCC(node);
	while (e) {
		shift_left_part(TARGET(e),i);
		e = ANEXT(e);
	}
	e = NPRED(node);
	while (e) {
		shift_left_part(SOURCE(e),i);
		e = ANEXT(e);
	}
}


/* Mark and update graph_maxx
 * --------------------------
 * this is done for all nodes reachable from node.
 */

static void mark_and_calc_maxx(node)
GNODE node;
{
	ADJEDGE e;

	debugmessage("mark_and_calc_maxx","");

	if (NMARK(node)==2) return;
	NMARK(node) = 2;

	if (NX(node)+NWIDTH(node)>graph_maxx) 
		graph_maxx = NX(node)+NWIDTH(node);

	if (NCONNECT(node)) {
		if (CTARGET(NCONNECT(node)))
			mark_and_calc_maxx(CTARGET(NCONNECT(node)));
		if (CTARGET2(NCONNECT(node)))
			mark_and_calc_maxx(CTARGET2(NCONNECT(node)));
	}
	e = NSUCC(node);
	while (e) {
		mark_and_calc_maxx(TARGET(e));
		e = ANEXT(e);
	}
	e = NPRED(node);
	while (e) {
		mark_and_calc_maxx(SOURCE(e));
		e = ANEXT(e);
	}
}



/* Stretch the layout by medium shifts (dumping)
 * ===================================
 * This task is to stretch a dense layout into a reasonable nondense
 * layout. The most important effect is that the leftest and rightest
 * node of the most lower and most upper level are moved into useful
 * positions:
 * Example:
 *	      ABC		 A   B	C 
 *	    DEFGHIJK		 DEFGHIJK
 *	      LMNO	  =>	  L M N O
 *	     PQRSTU		  PQR STU
 *	       VW		   V   W
 *
 * We use the positions of the leftest and rightest node of the levels 
 * as stop criterium of the iteration. The x-position of the leftest nodes
 * should decrease, the x-position of the rightest nodes should increase.
 * Thus the iteration stops, if no decrease or increase is recognizable. 
 */

/* The leftest, rightest upper and lower nodes */

static int dumpfactor;
static int old_nw = MAXINT;

static void iterate_dump_mediumshifts()
{
	int count;
	int changed;
	int tryout;

	debugmessage("iterate_dump_mediumshifts","");

	/* initialize leftupper_node, rightupper_node, ... */

	old_nw = MAXINT;

	count = 0;
	tryout = 2;
	dumpfactor = 2;
	while (1) {
		if (count%5==0) gs_wait_message('m');
		count++;
		drawdebug_show_graph("dump mediumshift",count,0,0);
		shift_left_layout();
		changed = dumpmediumshift(0); 
		changed += dumpmediumshift(1);
		dumpfactor = 1;

		if (!changed) break;
		if (count>=max_mediumshifts) {
			gs_wait_message('t');
			break; 
		}
		if (!changed_nw_sum()) {
			tryout--;
			if (tryout==0) break;
		} 
		else tryout = 2;
	} 
}


/* Check whether some position has changed
 * ----------------------------------------
 * return 1 if there is a leftest x-position of a level that has decreased,
 * or a rightest x-position of a level that has increased.
 */

static int changed_nw_sum() 
{
	int i, nwval;
	int changed;
	GNLIST li;

	debugmessage("changed_nw_sum","");

	changed = nwval = 0;
	for (i=0; i<=maxdepth+1; i++) {
		li = TSUCC(layer[i]); 
		while (li) {
			nwval += nw(GNNODE(li));	
			li = GNNEXT(li);
		}
	}

	if (nwval<old_nw) changed = 1;
	old_nw = nwval;
	return(changed);
}



/* Apply a sequence of dumpmedium_shifts to all layers
 * ---------------------------------------------------
 * dumpnwpdump_mediumshift and nwsdump_mediumshift is applied to all layers.
 * Warning: Iterations of this routine need not to be convergent !!!
 * We can go into a endless loop, if we do
 *     while (changed) changed = dumpmediumshift();
 *
 * This is done initially, to stretch the layers such that they 
 * become nearly the width of the most dense layer.
 */

static int dumpmediumshift(first)
int first;
{
	int i;
	int layer_changed;
	int dir;

	debugmessage("dumpmediumshift","");

	layer_changed = 0;
	dir = 0;

	if (first) {
		for (i=1; i<=maxdepth+1; i++) {
			dir = 1-dir;
			layer_changed += nwpdump_mediumshift(i,dir);
			dir = 1-dir;
			layer_changed += nwpdump_mediumshift(i,dir);
		}
	}
	for (i=maxdepth; i>=0; i--) {
		dir = 1-dir;
		layer_changed += nwsdump_mediumshift(i,dir);
		dir = 1-dir;
		layer_changed += nwsdump_mediumshift(i,dir);
	}
	if (!first) {
		for (i=1; i<=maxdepth+1; i++) {
			dir = 1-dir;
			layer_changed += nwpdump_mediumshift(i,dir);
			dir = 1-dir;
			layer_changed += nwpdump_mediumshift(i,dir);
		}
	}

	if (nwdumping_phase) {
		for (i=maxdepth; i>=0; i--) {
			dir = 1-dir;
			layer_changed += nwdump_mediumshift(i,dir);
			dir = 1-dir;
			layer_changed += nwdump_mediumshift(i,dir);
		}
	}

	if (layer_changed) return(1);
	return(0);
}


/*--------------------------------------------------------------------*/
/*  Treatment of one layer: Dump pendling method	      	      */
/*--------------------------------------------------------------------*/

/*  Shift layer i by respecting its nws values:
 *  -------------------------------------------
 *  We try to shift the nodes by the nws-value itself.
 *  nws > 0 => shift to the right, nws < 0 => shift to the left
 *  The return value is 1, if something has changed.
 *
 *  Example:   nws=5			 nws=0
 *		  N			   N (shifted 5 pixels to the right)
 *		  |\	     ==>	   /\
 *	    ews=0 | \ ews=10	   ews=-5 /  \ ews=5
 *		  X  Y			 X    Y
 *
 *  The difference to nws_mediumshift is, that we respect neighbor constraints.
 *  If a neighbour block must be shifted by n pixels, then we shift the nodes
 *  even if their nws values do not need a shift.
 * 
 *  Example:  Y  X  Z   K           Y X Z  K   (K is shifted even if nws(K)=0)
 *             \_ \ |   |     ==>   | | |  |
 *               \_\|   |            \|/  /
 *                  N   M             N   M
 */

static int nwsdump_mediumshift(i,dir)
int i;
int dir;
{
	GNLIST	li;
	int	j;
	int 	sign;

	debugmessage("nwsdump_mediumshift","");

	assert((i<=maxdepth));
	
	save_plevel(i);

	/* calculate shift weights */

	li = TSUCC(layer[i]);
	j = 0; 
	sign = 1;
	while (li) {
		levelshift[j] = nws(GNNODE(li)); 
		if ((sign<0) && (levelshift[j]>=0)) levelweight[j]=1; 
		else levelweight[j]= MAXINT;
		if (levelshift[j]<0) sign = -1; else sign = 1; 
		j++;
		li = GNNEXT(li);
	}
	levelweight[0]= 1;

	return(summarize_dumpshift(i,dir));
}

/*  Shift layer i by respecting its nwp values:
 *  -------------------------------------------
 *  We try to shift the nodes by the nwp-value itself.
 *  nwp > 0 => shift to the right, nwp < 0 => shift to the left
 *  The return value is 1, if something has changed.
 *
 *  Example:	  X  Y		       X      Y
 *		  |  /			\    /
 *		  | /	     ==>   ewp=-5\  /ewp=5
 *	    ewp=0 |/ewp=10		  \/	   
 *		  N			  N   (shifted 5 pixels to the right)
 *		nwp=5			nwp=0
 *
 *  The difference to nws_mediumshift is, that we respect neighbor constraints.
 *  If a neighbour block must be shifted by n pixels, then we shift the nodes
 *  even if their nws values do not need a shift.
 *
 *  Example:    __N   M             N   M
 *            _/ /|   |            /|\  \
 *           /  / |   |     ==>   | | |  |
 *          Y  X  Z   K           Y X Z  K   (K is shifted even if nwp(K)=0)
 */

static int nwpdump_mediumshift(i,dir)
int i;
int dir;
{
	GNLIST	li;
	int	j;
	int	sign;

	debugmessage("nwpdump_mediumshift","");

	assert((i>0));
	
	save_plevel(i);

	/* calculate shift weights */

	li = TSUCC(layer[i]);
	j = 0; 
	sign = 1;
	while (li) {
		levelshift[j] = nwp(GNNODE(li)); 
		if ((sign<0) && (levelshift[j]>=0)) levelweight[j]=1; 
		else levelweight[j]= MAXINT;
		if (levelshift[j]<0) sign = -1; else sign = 1; 
		j++;
		li = GNNEXT(li);
	}
	levelweight[0]= 1;

	return(summarize_dumpshift(i,dir));
}


/*  Shift layer i by respecting its nw values:
 *  ------------------------------------------
 *  We try to shift the nodes by the nw-value itself.
 *  nw > 0 => shift to the right, nw < 0 => shift to the left
 *  The return value is 1, if something has changed.
 *
 *  nw is (nearly) the sum of nws and nwp, i.e. we respect now the
 *  predecessors and the successors in the same step.
 *
 */

static int nwdump_mediumshift(i,dir)
int i;
int dir;
{
	GNLIST	li, li1;
	int	j;
	int 	sign;
	GNODE lnode, node, rnode;

	debugmessage("nwdump_mediumshift","");

	assert((i<=maxdepth));
	
	save_plevel(i);

	/* calculate shift weights */

	li = TSUCC(layer[i]);
	j = 0; 
	sign = 1;
	lnode = NULL;
	while (li) {
		node = GNNODE(li);
		if (NWIDTH(node)==0) {
			li1 = GNNEXT(li);
			rnode = NULL;
			while (li1) {
				if (NWIDTH(GNNODE(li1))!=0) {
					rnode = GNNODE(li1);
					break;
				}
				li1 = GNNEXT(li1);
			}
			levelshift[j] = nwbend(node,lnode,rnode);
		}
		else { 	levelshift[j] = nw(node); 
			lnode = node;
		}
		if ((sign<0) && (levelshift[j]>=0)) levelweight[j]=1; 
		else levelweight[j]= MAXINT;
		if (levelshift[j]<0) sign = -1; else sign = 1; 
		j++;
		li = GNNEXT(li);
	}
	levelweight[0]= 1;

	return(summarize_dumpshift(i,dir));
}


/* Common part of all dump_mediumshifts
 * ------------------------------------
 */


#define touching(v,w)  (  (NX(v)+NWIDTH(v)+G_xspace >= NX(w)) \
      			||(NX(w)-NX(v)-NWIDTH(v)<=2*G_xraster)) 


static int summarize_dumpshift(i,dir)
int i;
int dir;
{
	GNLIST	li;
	int	j;
	int	changed;
	int 	oldpos,sum,nrnodes;
	GNODE   v,w;

	debugmessage("summarize_dumpshift","");

	/* First we look whether there is space between nodes on one
 	 * region. If yes, we must split the region at this point.
	 */

	li = TSUCC(layer[i]);
	j = 0; 
	while (li) {
		v = GNNODE(li);
		if (GNNEXT(li)) {
			w = GNNODE(GNNEXT(li));	
			if (!touching(v,w)) 
				levelweight[j+1]=1; /* space => new region */
		}		
		j++;
		li = GNNEXT(li);
	}

	/* Now, the level is partitioned into regions of directly neigboured
 	 * nodes. If we traverse levelweight, then levelweight[j]=1 indicates
	 * a new region.
 	 * We calculate now the shift values of the regions and store them
	 * at the region indicator levelweight[j]. Because the shift values
	 * should be < MAXINT, we can still use them as region indicators. 
	 */

	changed = 1;
	while (changed) {
		changed = 0;
		li = TSUCC(layer[i]);
		j = 0; 
		oldpos = -1;
		nrnodes = 1;  /* not needed, only for the compiler */
		while (li) {
			if (levelweight[j]!=MAXINT) {
				if (oldpos!=-1) levelweight[oldpos] = sum/nrnodes;
				sum = levelshift[j];
				nrnodes = 1;
				oldpos = j;	
			}
			else { sum += levelshift[j]; nrnodes++; }
			j++;
			li = GNNEXT(li);
		}
		if (oldpos!=-1) levelweight[oldpos] = sum/nrnodes;

		/* check whether two regions must be combined: This occurs
		 * if both regions are neighboured and have a tendence in
		 * the same direction. 
		 */

		li = TSUCC(layer[i]);
		j = 0; 
		sum = 0;
		while (li) {
			if (levelweight[j]!=MAXINT) sum = levelweight[j];
			if (GNNEXT(li)&&(levelweight[j+1]!=MAXINT)) {
				v = GNNODE(li);
				w = GNNODE(GNNEXT(li));	
				if (touching(v,w)) {
					if (  (sum>=0)&&(levelweight[j+1]>=0)
					    &&(sum>=levelweight[j+1])) {
						levelweight[j+1] = MAXINT;
						changed = 1;
					}
					if (  (sum<0)&&(levelweight[j+1]<0)
					    &&(sum>=levelweight[j+1])) {
						levelweight[j+1] = MAXINT;
						changed = 1;
					}
					if (  (sum>0)&&(levelweight[j+1]<0)) {
						levelweight[j+1] = MAXINT;
						changed = 1;
					}
				} 
			}
			j++;
			li = GNNEXT(li);
		}
	}

	/* Now, we have the final region partitioning, and levelweight
	 * contains appropriate shift values. These must be moved to 
	 * levelshift now.
	 */

	li = TSUCC(layer[i]);
	j = 0; 
	assert((levelweight[0]!=MAXINT));
	while (li) {
		if (levelweight[j]!=MAXINT) sum = dumpfactor * levelweight[j];
		levelshift[j] = sum;
		levelweight[j]= 1;
		j++;
		li = GNNEXT(li);
	}
	if (dir) {
		changed  = do_rightshifts(i);
		changed += do_leftshifts(i);
	}
	else {
		changed  = do_leftshifts(i);
		changed += do_rightshifts(i);
	}

	return(changed);
}

/*--------------------------------------------------------------------*/


/*--------------------------------------------------------------------*/
/*  Centering method						      */
/*--------------------------------------------------------------------*/

/*  If we have the leftest and rightest node of the upperst and
 *  the lowerst level in useful positions (not too dense, and pendled),
 *  we can try to minimize the constraints of a node.
 *  As constrains, we define:
 *	   G = sum( ewp(predecessors) )  + sum ( ews(successors) )
 *
 *  To minimize this value, we successively move a node by 
 *	   nw(node) = G(node) / number(edges to n)
 */

static void iterate_centershifts()
{
	int	i,count;
	int	weight,h;
	int	second_try;

	debugmessage("iterate_centershifts","");

	weight = center_weight();
	second_try = 2;
	count=0;
	while (1) {
		if (count%5==0) gs_wait_message('c');
		count++;
		if (count>=max_centershifts) { 
			gs_wait_message('t');
			break; 
		}
		for (i=0; i<=maxdepth+1; i++) center_layer(i);
		for (i=maxdepth+1; i>=0; i--) center_layer(i);

		h = center_weight();
		drawdebug_show_graph("centershift",count,h,weight);

		if (h<weight) {
			weight = h;
			second_try = 2;
		}
		else if (h==weight) {
			second_try--;
			if (second_try<=0) break; 
		}
		else break;  /* should never occur */
	}
}



/*  Calculate the summary weight of all layers 
 *  ------------------------------------------
 *  This is  sum( abs(G(all nodes)) )
 */

static int center_weight()
{
	int	i;
	GNLIST	li;
	ADJEDGE a;
	int	weight,h;

	debugmessage("center_weight","");

	weight = 0;
	for (i=0; i<=maxdepth+1; i++) {
		li = TSUCC(layer[i]);
		while (li) {
			h = 0;
			a = NSUCC(GNNODE(li));
			while (a) {
				h += ews(AKANTE(a));
				a = ANEXT(a);
			}
			a = NPRED(GNNODE(li));
			while (a) {
				h += ewp(AKANTE(a));
				a = ANEXT(a);
			}
			if (h<0) h = -h;	
			weight += h;
			li = GNNEXT(li);
		}
	}
	return(weight);
}


/*  Shift layer i by respecting its nw values:
 *  ------------------------------------------
 *  We try to shift the nodes by the nw-value itself.
 *  Example:
 *	       A  B		  A    B
 *		\ |		   \  /
 *		 \|		    \/
 *		  N	   =>	    N  (shifted to the left)
 *		 /|		    /\
 *		/ |		   /  \
 *	       C  D		  C    D
 */

static int center_layer(i)
int i;
{
	GNLIST	li, li1;
	int	j;
	int	changed;
	int	dir;
	GNODE   lnode, node, rnode;

	debugmessage("center_layer","");

	save_plevel(i);

	/* calculate shift weights */

	li = TSUCC(layer[i]);
	j = 0; 
	dir = 0;
	lnode = NULL;
	while (li) {
		node = GNNODE(li);
		if (NWIDTH(node)==0) {
			li1 = GNNEXT(li);
			rnode = NULL;
			while (li1) {
				if (NWIDTH(GNNODE(li1))!=0) {
					rnode = GNNODE(li1);
					break;
				} 
				li1 = GNNEXT(li1);
			}
			levelshift[j] = nwbend(node,lnode,rnode);
		}
		else { 	levelshift[j] = nw(node);
			lnode = node;
		}
		dir += levelshift[j];
		levelweight[j++]= 1;
		li = GNNEXT(li);
	}
	if (dir>=0) {
		changed  = do_rightshifts(i);
		changed += do_leftshifts(i);
	}
	else {
		changed  = do_leftshifts(i);
		changed += do_rightshifts(i);
	}
	return(changed);
}


/*--------------------------------------------------------------------*/
/*  Layer-Shifts						      */
/*--------------------------------------------------------------------*/

/*  Leftshifts of a layer
 *  ---------------------
 *  The nodes of a layer i are in slayer_array. The negative values
 *  levelshift[j]/levelweight[j] give the numbers of 
 *  pixels we want to shift. We start just at the left side
 *  of the level and shift to the left.
 *  The return value is 1, if something has changed.
 */

static int	do_leftshifts(i)
int i;
{
	int   j;
	int   diff;
	int   oldx;
	GNODE node,lnode;
	int   changed;

	assert((i>=0) && (i<=maxdepth+1));
	lnode = NULL;
	for (j=0; j<TANZ(layer[i]); j++) {
		if (levelweight[j]) diff = levelshift[j]/levelweight[j];
		else diff = 0;
		node  = slayer_array[j];
		if (diff<0) {
			oldx = NX(node);
			NX(node) = oldx + diff;
			if (lnode) { 
				if ((NWIDTH(node)==0)||(NWIDTH(lnode)==0)) {
			    		if (NX(node)<NX(lnode)
						+NWIDTH(lnode)+G_dspace)
						NX(node) = NX(lnode)
						     +NWIDTH(lnode)+G_dspace;
				}
				else {
			    		if (NX(node)<NX(lnode)
						+NWIDTH(lnode)+G_xspace)
						NX(node) = NX(lnode)
							+NWIDTH(lnode)+G_xspace;
				}
			}

			if ((NWIDTH(node)==0)&&(NHEIGHT(node)==0))
			      	NX(node) = dxralign(NX(node)+NWIDTH(node)/2)
						-NWIDTH(node)/2;
			else  	NX(node) =  xralign(NX(node)+NWIDTH(node)/2)
						-NWIDTH(node)/2;
			if (NX(node)<oldx) changed = 1;
		}
		lnode = node;
	}
	return(changed);
}

/*  Rightshifts of a layer
 *  ----------------------
 *  The nodes of a layer i are in slayer_array. The positive values
 *  levelshift[j]/levelweight[j] give the numbers of 
 *  pixels we want to shift. We start just at the right side
 *  of the level and shift to the right.
 *  The return value is 1, if something has changed.
 */

static int	do_rightshifts(i)
int i;
{
	int   j;
	int   diff;
	int   oldx;
	GNODE node,rnode;
	int   changed;

	assert((i>=0) && (i<=maxdepth+1));
	rnode = NULL;
	changed = 0;
	for (j=TANZ(layer[i])-1; j>=0; j--) {
		if (levelweight[j]) diff = levelshift[j]/levelweight[j];
		else diff = 0;
		node  = slayer_array[j];
		if (diff>0) {
			oldx = NX(node);
			NX(node) = oldx + diff;
			if (rnode) { 
				if ((NWIDTH(node)==0)||(NWIDTH(rnode)==0)) {
			    		if (NX(node)+NWIDTH(node)+G_dspace
						>NX(rnode))
						NX(node) = NX(rnode)
						      -NWIDTH(node)-G_dspace;
				}
				else {
			    		if (NX(node)+NWIDTH(node)+G_xspace
						>NX(rnode))
						NX(node) = NX(rnode)
							-NWIDTH(node)-G_xspace;
				}
			}
			if ((NWIDTH(node)==0)&&(NHEIGHT(node)==0))
			      	NX(node) = dxlalign(NX(node)+NWIDTH(node)/2)
						-NWIDTH(node)/2;
			else  	NX(node) =  xlalign(NX(node)+NWIDTH(node)/2)
						-NWIDTH(node)/2;
			if (NX(node)>oldx) changed = 1;
		}
		rnode = node;
	}
	return(changed);
}



/*--------------------------------------------------------------------*/
/*  Calculation of layout weights				      */
/*--------------------------------------------------------------------*/

/* Edge wights ews and ewp
 * -----------------------
 *    Examples:
 *	  /  here is ews<0     \  here is ews>0
 *	 /   and     ewp>0	\ and	  ewp<0
 *     |/_			_\|
 */

static int	ews(edge)
GEDGE	edge;
{
	GNODE	start, ende;
	int 	x1,x2;
	
	start = ESTART(edge);
	ende  = EEND(edge); 
	x1 = NX(ende) + (NWIDTH(ende)*EWEIGHTP(edge)/(NWEIGHTP(ende)+1));
	x2 = NX(start) + (NWIDTH(start)*EWEIGHTS(edge)/(NWEIGHTS(start)+1));
	return(x1-x2);
}


static int	ewp(edge)
GEDGE	edge;
{
	GNODE	start, ende;
	int 	x1,x2;

	ende  = EEND(edge);
	start = ESTART(edge);
	x1 = NX(ende) + (NWIDTH(ende)*EWEIGHTP(edge)/(NWEIGHTP(ende)+1));
	x2 = NX(start) + (NWIDTH(start)*EWEIGHTS(edge)/(NWEIGHTS(start)+1));
	return(x2-x1);
}


/* Node weights nws and nwp
 * ------------------------
 * nws(n) is the sum of ews(successor of n) / number successors
 * nwp(n) is the sum of ewp(predecessor of n) / number predecessors
 * 
 * nws(n) < 0  => shift n to the left
 * nws(n) > 0  => shift n to the right
 * nwp(n) < 0  => shift n to the left
 * nwp(n) > 0  => shift n to the right
 */

static int	nws(node)
GNODE	node;
{
	int     h;
	int	weight;
	int	nr_edges;
	ADJEDGE a;
	CONNECT c;

	a = NSUCC(node);
	weight = 0; 
	nr_edges = 0;
	while (a) {
		h = (EPRIO(AKANTE(a))*layout_downfactor) * ews(AKANTE(a));
		weight += h;
		nr_edges += (EPRIO(AKANTE(a))*layout_downfactor);
		a = ANEXT(a);
	}
	c = NCONNECT(node);
	if (c) {
		if (CTARGET(c)) {
			if (NX(CTARGET(c))<NX(node))
				weight -= (EPRIO(CEDGE(c))*layout_nearfactor *
					     (NX(node) - (NX(CTARGET(c))
						+NWIDTH(CTARGET(c))+G_xspace)));
			else	weight += (EPRIO(CEDGE(c))*layout_nearfactor *
					     (NX(CTARGET(c)) - (NX(node)
						+NWIDTH(node)+G_xspace))); 
			nr_edges += (EPRIO(CEDGE(c))*layout_nearfactor);
		}
		if (CTARGET2(c)) {
			if (NX(CTARGET2(c))<NX(node))
				weight -= (EPRIO(CEDGE2(c))*layout_nearfactor *
					    (NX(node) - (NX(CTARGET2(c))
					       +NWIDTH(CTARGET2(c))+G_xspace)));
			else	weight += (EPRIO(CEDGE2(c))*layout_nearfactor *
					    (NX(CTARGET2(c)) - (NX(node)
						+NWIDTH(node)+G_xspace))); 
			nr_edges += (EPRIO(CEDGE2(c))*layout_nearfactor);
		}
	}

	if (nr_edges) return(weight/nr_edges);
	else	      return(0);
}


static int	nwp(node)
GNODE	node;
{
	int     h;
	int	weight;
	int	nr_edges;
	ADJEDGE a;
	CONNECT c;

	a = NPRED(node);
	weight = 0; 
	nr_edges = 0;
	while (a) {
		h = (EPRIO(AKANTE(a))*layout_upfactor) * ewp(AKANTE(a));
		weight += h;
		nr_edges += (EPRIO(AKANTE(a))*layout_upfactor);
		a = ANEXT(a);
	}
	c = NCONNECT(node);
	if (c) {
		if (CTARGET(c)) {
			if (NX(CTARGET(c))<NX(node))
				weight -= (EPRIO(CEDGE(c))*layout_nearfactor *
					     (NX(node) - (NX(CTARGET(c))
						+NWIDTH(CTARGET(c))+G_xspace)));
			else	weight += (EPRIO(CEDGE(c))*layout_nearfactor *
					     (NX(CTARGET(c)) - (NX(node)
						+NWIDTH(node)+G_xspace))); 
			nr_edges += (EPRIO(CEDGE(c))*layout_nearfactor);
		}
		if (CTARGET2(c)) {
			if (NX(CTARGET2(c))<NX(node))
				weight -= (EPRIO(CEDGE2(c))*layout_nearfactor *
					    (NX(node) - (NX(CTARGET2(c))
					       +NWIDTH(CTARGET2(c))+G_xspace)));
			else	weight += (EPRIO(CEDGE2(c))*layout_nearfactor *
					    (NX(CTARGET2(c)) - (NX(node)
						+NWIDTH(node)+G_xspace))); 
			nr_edges += (EPRIO(CEDGE2(c))*layout_nearfactor);
		}
	}

	if (nr_edges) return(weight/nr_edges);
	else	      return(0);
}


/* Node weights nw 
 * ---------------
 * If we have multiple predecessors and successors:
 * nw(n) is the sum of ews(successor) + ewp(predecessors) / number edges 
 * If we have 1 predecessor and 1 successors:
 * nw(n) is (dx2 * dy1 - dx1 * dy2) / (dy1+dy2) in
 *
 *              dx1  dx2
 *           A -----|--|
 *            \     |  |
 *             \    |  |dy1
 *              B'--B--|        nw(B) is the distance to shift B to B'.
 *               \     |
 *                \    |
 *                 \   |dy2
 *                  \  |
 *                   \ |
 *                     C
 * 
 * nw(n) < 0  => shift n to the left
 * nw(n) > 0  => shift n to the right
 */

static int	nw(node)
GNODE	node;
{
	int	weight;
	int     dx1, dx2, dy1, dy2;
	int	nr_edges,p1,p2;
	GEDGE   edge;
	GNODE   v;
	ADJEDGE a;
	CONNECT c;

	weight = 0; 
	nr_edges = 0;
	if (   (layout_downfactor==1)&&(layout_upfactor==1)
	    && NSUCC(node) && (ANEXT(NSUCC(node))==0)
	    && NPRED(node) && (ANEXT(NPRED(node))==0)
	    && (EPRIO(AKANTE(NSUCC(node)))==EPRIO(AKANTE(NPRED(node))))
	    && NCONNECT(node)==NULL) {
		edge = AKANTE(NPRED(node));
		v    = ESTART(edge);
		p1 = NX(node)+(NWIDTH(node)*EWEIGHTP(edge)/(NWEIGHTP(node)+1));
		p2 = NX(v) + (NWIDTH(v)*EWEIGHTS(edge)/(NWEIGHTS(v)+1));
		dx1 = p1 - p2; 

		edge = AKANTE(NSUCC(node));
		v    = EEND(edge);
		p1 = NX(node)+(NWIDTH(node)*EWEIGHTS(edge)/(NWEIGHTS(node)+1));
		p2 = NX(v) + (NWIDTH(v)*EWEIGHTP(edge)/(NWEIGHTP(v)+1));
		dx2 = p2 - p1;
		p1  = NY(node)+NHEIGHT(node)/2;
		p2  = NY(SOURCE(NPRED(node)))+NHEIGHT(SOURCE(NPRED(node)))/2;
		dy1 = p1 - p2;
		p2  = NY(TARGET(NSUCC(node)))+NHEIGHT(TARGET(NSUCC(node)))/2;
		dy2 = p2 - p1;
		weight = (dx2*dy1-dx1*dy2)/(dy1+dy2);
		return(weight);
	}

	a = NSUCC(node);
	while (a) {
		weight += (ews(AKANTE(a))*EPRIO(AKANTE(a))
				*layout_downfactor);
		nr_edges += (EPRIO(AKANTE(a))*layout_downfactor);
		a = ANEXT(a);
	}
	a = NPRED(node);
	while (a) {
		weight += (ewp(AKANTE(a))*EPRIO(AKANTE(a))
				*layout_upfactor);
		nr_edges += (EPRIO(AKANTE(a))*layout_upfactor);
		a = ANEXT(a);
	}
	c = NCONNECT(node);
	if (c) {
		if (CTARGET(c)) {
			if (NX(CTARGET(c))<NX(node))
				weight -= (EPRIO(CEDGE(c))*layout_nearfactor *
					     (NX(node) - (NX(CTARGET(c))
						+NWIDTH(CTARGET(c))+G_xspace)));
			else	weight += (EPRIO(CEDGE(c))*layout_nearfactor *
					     (NX(CTARGET(c)) - (NX(node)
						+NWIDTH(node)+G_xspace))); 
			nr_edges += (EPRIO(CEDGE(c))*layout_nearfactor);
		}
		if (CTARGET2(c)) {
			if (NX(CTARGET2(c))<NX(node))
				weight -= (EPRIO(CEDGE2(c))*layout_nearfactor *
					    (NX(node) - (NX(CTARGET2(c))
					       +NWIDTH(CTARGET2(c))+G_xspace)));
			else	weight += (EPRIO(CEDGE2(c))*layout_nearfactor *
					    (NX(CTARGET2(c)) - (NX(node)
						+NWIDTH(node)+G_xspace))); 
			nr_edges += (EPRIO(CEDGE2(c))*layout_nearfactor);
		}
	}
	

	if (nr_edges) return(weight/nr_edges);
	else	      return(0);
}


/* Node weights nw on dummy nodes
 * ------------------------------
 *
 *  This is the same as nw, except that we want to avoid bendings
 *  on dummy nodes in certain situations.
 *  If the direction of an edge changes completely, it is not
 *  critical, but if the direction is nearly the same, we must
 *  calculate the position on the node such that just no bending 
 *  occurs. To simplify, we observe only the direct left and right
 *  neighbour of the dummy node.
 *
 *  Example: we want to avoid such situations:
 *
 *  (ax,ay)
 *        \           __(kx,ky)_
 *          \        |
 *         (mx,my)   |            
 *            |      |
 *            |      |   k
 *             \     |
 *               \   |_______
 *                 \
 *                   \
 *                (bx,by)
 *
 *  The corrected situation is:
 *
 *  (ax,ay)
 *        \
 *          \       --(kx,ky)-
 *       (mx,my)   |          
 *            \    |               where obviously holds:
 *             \   |   k          
 *              \  |               (kx-mx)       ky+HEIGHT(k)-my
 *               \ |_______        -------   =   ---------------
 *                \                (bx-mx)       (by-my)
 *                 \
 *                (bx,by)
 *
 *
 *  nwbend(n) < 0  => shift n to the lefti, etc.
 */

static int	nwbend(node, lnode, rnode)
GNODE	node, lnode, rnode;
{
	GNODE pred, succ;
	int ax,ay,mx,my,bx,by,kx,h, dist;
	int act_nw;

	act_nw = nw(node);

	if (NPRED(node)) pred = SOURCE(NPRED(node));
	else return(act_nw);
	if (NSUCC(node)) succ = TARGET(NSUCC(node));
	else return(act_nw);

	ax = NX(pred);
	ay = NY(pred);
	mx = NX(node);
	my = NY(node);
	bx = NX(succ);
	by = NY(succ);

	if ((ax<mx)&&(bx<mx)) return(act_nw);
	if ((ax>mx)&&(bx>mx)) return(act_nw);

	if (ax<bx) {
		if (rnode && act_nw>0) { 
			kx = NX(rnode);
			h  = NY(rnode)+NHEIGHT(rnode)-my;
			if (h-by+my==0) return(act_nw);
			dist = (bx*h - kx*(by-my))/(h-by+my) - mx;
			if (dist<act_nw) act_nw = dist;
		}
		else if (lnode && act_nw<0) { 
			kx = NX(lnode)+NWIDTH(lnode);
			h  = NY(lnode)-my;
			if (h-ay+my==0) return(act_nw);
			dist = (ax*h - kx*(ay-my))/(h-ay+my) - mx;
			if (dist>act_nw) act_nw = dist;
		}
	}
	else if (ax>bx) {
		if (lnode && act_nw<0) { 
			kx = NX(lnode)+NWIDTH(lnode);
			h  = NY(lnode)+NHEIGHT(lnode)-my;
			if (h-by+my==0) return(act_nw);
			dist = (bx*h - kx*(by-my))/(h-by+my) - mx;
			if (dist>act_nw) act_nw = dist;
		}
		else if (rnode && act_nw>0) { 
			kx = NX(rnode);
			h  = NY(rnode)-my;
			if (h-ay+my==0) return(act_nw);
			dist = (ax*h - kx*(ay-my))/(h-ay+my) - mx;
			if (dist<act_nw) act_nw = dist;
		}
	}
	
	return(act_nw);
}


/*--------------------------------------------------------------------*/
/*  Treatement of the save array				      */
/*--------------------------------------------------------------------*/


/*  Put level layer[i] into the slayer_array
 *  ----------------------------------------
 *  This is done to calculate levelshift and levelweight.
 *  Both arrays are initialized, further.
 */

static void	save_plevel(i)
int i;
{
	int j;
	GNLIST hn;

	debugmessage("save_plevel","");
	j  = 0;
	hn = TSUCC(layer[i]);
	while (hn) {
		assert((NPOS(GNNODE(hn))==j));
		slayer_array[j]	 = GNNODE(hn);
		levelshift[j]	 = 0;
		levelweight[j++] = 0;
		hn = GNNEXT(hn);
	}
	assert(j==TANZ(layer[i]));
}


/*--------------------------------------------------------------------*/
/*  For debugging only           			      */
/*--------------------------------------------------------------------*/

#ifdef DRAWDEBUG

static void drawdebug_show_graph(m,i,j,k)
char *m;
int i,j,k;
{
	PRINTF("%s %d (%d %d)\n",m,i,j,k);
	PRINTF("Enter CR into the text window\n");
	shift_left_layout();
	step4_main();
	debug_display_part();
	fgetc(stdin);
}

#endif

