/* SCCS-info %W% %E% */

/*--------------------------------------------------------------------*/
/*                                                                    */
/*              VCG : Visualization of Compiler Graphs                */
/*              --------------------------------------                */
/*                                                                    */
/*   file:         step1.c                                            */
/*   version:      1.00.00                                            */
/*   creation:     14.4.1993                                          */
/*   author:       I. Lemke  (...-Version 0.99.99)                    */
/*                 G. Sander (Version 1.00.00-...)                    */  
/*                 Universitaet des Saarlandes, 66041 Saarbruecken    */
/*                 ESPRIT Project #5399 Compare                       */
/*   description:  Layout phase 1: building a proper hierarchy        */
/*   status:       in work                                            */
/*                                                                    */
/*--------------------------------------------------------------------*/

#ifndef lint
static char *id_string="$Id: step1.c,v 3.9 1994/08/05 14:27:31 sander Exp $";
#endif


/*
 *   Copyright (C) 1993, 1994 by Georg Sander, Iris Lemke, and
 *                               the Compare Consortium 
 *
 *  This program and documentation is free software; you can redistribute 
 *  it under the terms of the  GNU General Public License as published by
 *  the  Free Software Foundation;  either version 2  of the License,  or
 *  (at your option) any later version.
 *
 *  This  program  is  distributed  in  the hope that it will be useful,
 *  but  WITHOUT ANY WARRANTY;  without  even  the  implied  warranty of
 *  MERCHANTABILITY  or  FITNESS  FOR  A  PARTICULAR  PURPOSE.  See  the
 *  GNU General Public License for more details.
 *
 *  You  should  have  received a copy of the GNU General Public License
 *  along  with  this  program;  if  not,  write  to  the  Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  The software is available per anonymous ftp at ftp.cs.uni-sb.de.
 *  Contact  sander@cs.uni-sb.de  for additional information.
 */


/* 
 * $Log: step1.c,v $
 * Revision 3.9  1994/08/05  14:27:31  sander
 * Negative values of G_width, etc. corrected.
 *
 * Revision 3.8  1994/08/05  12:13:25  sander
 * Treelayout added. Attributes "treefactor" and "spreadlevel" added.
 * Scaling as abbreviation of "stretch/shrink" added.
 *
 * Revision 3.7  1994/08/03  13:58:44  sander
 * Horizontal order mechanism changed.
 * Attribute horizontal_order for edges added.
 *
 * Revision 3.6  1994/08/02  15:36:12  sander
 * On layout algorithm minbackward, fine tuning phase corrected.
 * Allow nodes to be pulled just before/after their successors/predecessors,
 * if no backward edges are produced by this.
 *
 * Revision 3.5  1994/05/16  08:56:03  sander
 * shape attribute (boxes, rhombs, ellipses, triangles) added.
 *
 * Revision 3.4  1994/05/05  08:20:30  sander
 * Algorithm late labels added: If labels are inserted
 * after partitioning, this may yield a better layout.
 *
 * Revision 3.3  1994/04/27  16:05:19  sander
 * Some general changes for the PostScript driver.
 * Horizontal order added. Bug fixes of the folding phases:
 * Folding of nested graphs works now.
 *
 * Revision 3.2  1994/03/04  19:11:24  sander
 * Specification of levels per node added.
 * X11 geometry behaviour (option -geometry) changed such
 * that the window is now opened automatically.
 *
 * Revision 3.1  1994/03/01  10:59:55  sander
 * Copyright and Gnu Licence message added.
 * Problem with "nearedges: no" and "selfloops" solved.
 *
 * Revision 2.5  1994/02/10  15:56:32  sander
 * Layoutalgorithm changed: We not only put the start nodes
 * in front of the node list, but additionally add the end
 * nodes (which have no successor) at the end of the node list.
 *
 * Revision 2.4  1994/01/21  19:33:46  sander
 * VCG Version tested on Silicon Graphics IRIX, IBM R6000 AIX and Sun 3/60.
 * Option handling improved. Option -grabinputfocus installed.
 * X11 Font selection scheme implemented. The user can now select a font
 * during installation.
 * Sun K&R C (a nonansi compiler) tested. Some portabitility problems solved.
 *
 * Revision 2.3  1994/01/03  15:29:06  sander
 * First complete X11 version.
 *
 */


/************************************************************************
 * The situation here is the following:
 * -----------------------------------
 * The folding of the graph is done and the adjacency lists are created,
 * but we have recognized that not all visible nodes have their (x,y) 
 * coordinates.
 * Thus, we have to layout the graph. WE SIMPLY IGNORE ALL EXISTING
 * COORDINATES AND LAYOUT THE WHOLE GRAPH.
 *
 * Before we start, the situation is: 
 *  1) We have proper adjacency lists.
 *  2) All visible nodes are in the nodelist or in the labellist.
 *  3) All potentially visible edges are in the lists edgelist or tmpedgelist.
 *     Visible edges can be detected by the EINVISIBLE flag (==0) in these
 *     lists. Note: invisible edges may also be in edgelist or tmpedgelist.
 *  4) An edge is visible iff it is used in the adjacency lists.
 *  5) NCONNECT of all nodes is empty.
 *  6) There is a list of default connections: the near_edge_list.
 *
 * The task of building a proper hierarchy is to distribute vertically 
 * the nodes into layers. The edges starting at one level go to node
 * of the immediate next level. Thus it is necessary to create dummy
 * nodes in one layer if an edge goes accross several layers.
 * There may be edges inside the layer, but only from nodes to directly 
 * neigboured nodes. This situation is summarized with connection nodes,
 * see alloc.h.
 * If we have A-->B and A and B are on the same level, we call B a
 * forward connection of A, and A a backward connection of B. During
 * the layout, the connection edge between A and B is deleted, but can
 * be restored from the NCONNECT-fields. 
 * The distribution of the nodes into layers is done by a depth first 
 * search that conceptually partitioned the edges into `tree edges', 
 * `forward edges', `backward edges' and `cross edges'. 
 * We have implemented two different depth first searches, which can be
 * selected by the flag -d of vcg.
 * Later, we will layout the tree AS TREE and try to add the nontree
 * edges artificially.
 *
 * After that, we have the following situation:
 *    1)  The array layer contains all visible nodes in the TSUCC lists. 
 *        They are distributed at the layer[i] lists and connected by two 
 *        lists TPRED and TSUCC to allow to traverse the nodes of one 
 *	  layer[i] backwards and forwards. However TSUCC and TPRED are
 *  	  different: TSUCC contains all nodes of the layer including those
 *        nodes reacheable by forward connections. TPRED excludes these
 *	  nodes. TANZ(layer[i]) is not yet initialized, i.e. it is 0.
 *	  If it is TREE_LAYOUT, the TPRED lists are not initialized.
 *	  The hierarchy in layer is proper.
 *    2)  labellist and nodelist have not changed.  
 *	  All visible nodes are in nodelist, labellist and dummylist.
 *    3)  All pot. visible edges are in the lists edgelist or tmpedgelist.
 *	  Same as before.
 *    4)  If it is not a TREE_LAYOUT, then: 
 *	  maxindeg and maxoutdeg are the maximal indegree (number of 
 *	  incoming edges) and maximal outdegree (number of outgoing
 *	  edges) occuring at visible nodes. Forward connections are 
 *	  not counted.
 *    5)  maxdepth+1 is the maximal layer !!! NOT maxdepth !!!
 *    6)  NTIEFE(node) is filled for all nodes. 
 *    7)  If it is not a TREE_LAYOUT, then:
 *	  NINDEG and NOUTDEG are filled for all nodes. Forward connections 
 *	  are not counted. Edges of forward connection are bend to the 
 *	  connection reference node, that is the node that has these forward 
 *	  connections but no backward connection.
 *        NCONNECT(node) is filled for nodes that have direct neighbours
 *	  in the layout. The edges in NCONNECT are not anymore in the
 *	  adjacency lists, but still visible. The forward connection nodes 
 * 	  are still in the layers in the TSUCC lists, but not in the
 *        TPRED lists. 
 *    7)  Reverted edges are marked with EART(e)='R' and bidirectional
 *	  edges between adjacent levels are marked with EART(e)='D'.
 *  	  Self loops don't anymore exist.
 *
 * This file provides the following functions:
 * ------------------------------------------
 * step1_main		Main routine to build a proper hierarchy
 * revert_edge	 	reverts an edge in the adjacency lists
 *
 * If DEBUG is switched on, we have further:
 *
 * db_output_graph	  prints all visible nodes and edges of the graph
 * db_output_adjacencies  prints the adjacency lists of the visible nodes
 * db_output_adjacency    prints the adjacency lists of one node
 *
 ************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "globals.h"
#include "alloc.h"
#include "main.h"
#include "folding.h"
#include "steps.h"
#include "timing.h"


/* Prototypes
 * ----------
 */

#ifdef DEBUG
static void	db_output_nodes 	_PP((GNODE node));
static void	db_output_edges  	_PP((GEDGE edge));
#endif

static void 	insert_startnode	_PP((GNODE node));
static void 	insert_endnode		_PP((GNODE node));
static void	prepare_startnodes 	_PP((void));
static void 	prepare_anchoredge	_PP((GEDGE edge));
static void	insert_near_edges	_PP((void));
static int  	check_connect_cycle	_PP((GNODE v,GNODE w,GNODE z));
static void	partition_edges		_PP((void));
static void	depth_first_search 	_PP((GNODE node));
static void	alt_depth_first_search 	_PP((GNODE node));
static void 	start_dfs_backwards	_PP((ADJEDGE edge,GNODE node,int prio));

static void	topological_sort	  _PP((void));
static void 	add_to_zero_indegree_list _PP((GNODE v));
static GNODE 	get_zero_indegree         _PP((void));
static int 	topsort_maxlevel  	  _PP((GNODE node1,GNODE node2));
static void 	topsort_setlevel 	  _PP((GNODE node1,GNODE node2,int l));
static void 	topsort_add_succs 	  _PP((GNODE node1,GNODE node2));
static int 	topsort_indegree          _PP((GNODE node1,GNODE node2));

static void 	add_phase1_labels	_PP((void));

static void 	tune_partitioning	_PP((void));
static int 	tune_node_depth		_PP((GNODE v,int lab));
static void 	create_depth_lists	_PP((void));
static void 	complete_depth_lists	_PP((void));
static void 	calc_connect_adjlists	_PP((GNODE v, GNODE w, GNODE predw));

static void	inspect_edges 		_PP((void));
static void 	check_edge		_PP((GNODE n, ADJEDGE e, int l));
static int	double_edge 		_PP((ADJEDGE edge));
static GNODE	create_dummy 		_PP((int t));
static ADJEDGE	create_edge		_PP((GNODE x, GNODE y,GEDGE e,int t));


/* Global variables
 * ----------------
 */

int	maxindeg;	/* maximal indegree  of a node */
int	maxoutdeg;	/* maximal outdegree of a node */

DEPTH *layer = NULL;	       /* The table of layers     */
int    maxdepth = 0;           /* Max. depth of layout    */
static int size_of_layer = 0;  /* Size of table of layers */ 

/* Macros
 * ------
 */

#define forward_connection1(c)  ((CEDGE(c))&& (EEND(CEDGE(c))==CTARGET(c)))
#define forward_connection2(c)  ((CEDGE2(c))&&(EEND(CEDGE2(c))==CTARGET2(c)))
#define backward_connection1(c) ((CEDGE(c))&& (EEND(CEDGE(c))!=CTARGET(c)))
#define backward_connection2(c) ((CEDGE2(c))&&(EEND(CEDGE2(c))!=CTARGET2(c)))

/*--------------------------------------------------------------------*/
/*  Building a proper hierarchy                                       */
/*--------------------------------------------------------------------*/

void	step1_main()
{
	int i;

	start_time();
	debugmessage("step1_main","");
	assert((dummylist==NULL));

	/* reorder the nodes such that start nodes are found first */

	prepare_startnodes();

	/* insert dummy nodes for edges with anchor connection
	 */

	insert_anchor_edges();

	/* insert near edge connections
	 */

	insert_near_edges();

	/* partition the edges into tree, forward, etc. edges and 
	 * calculate maxdepth.
	 */

	if (layout_flag==3) topological_sort();
	else partition_edges();

	/* Eventually, we have to add the edge labels now.
	 */

	if (edge_label_phase == 1)  add_phase1_labels();

	/* Fine tune this partitioning, such that all edges at a node
 	 * are nearly equidistant.
	 */

	if (fine_tune_layout==1) tune_partitioning();

	/* allocate the array layer */

	if (maxdepth+2 > size_of_layer) {
		if (layer) free(layer);
		layer = (DEPTH *)malloc((maxdepth+2)*sizeof(struct depth_entry));
		if (!layer) Fatal_error("memory exhausted","");
		size_of_layer = maxdepth+2;
#ifdef DEBUG
		PRINTF("Maxdepth of this layout: %d \n",maxdepth);
		PRINTF("Sizeof table `layer': %ld Bytes\n",
			(maxdepth+2)*sizeof(struct depth_entry));
#endif
	}
	for (i=0; i<maxdepth+2; i++) {
		 TANZ(layer[i])      = 0;
		 /* TCROSS is never used for layer */
		 TPRED(layer[i])     = NULL;
		 TSUCC(layer[i])     = NULL;
		 TRESNEEDED(layer[i])= 1;
	}

	/* and fill the array intiefe.
	 * We create only the TSUCC lists of layer[i].
	 */

	create_depth_lists();

	/* check the direction of the edges and create dummy and pseudo nodes,
	 * if edges go beyond several levels, or for self loops. This may 
	 * change the TSUCC lists. After that, the hierarchy is proper.
	 */
 
	inspect_edges();

	/* Now, we derive the TPRED lists from the TSUCC lists
	 * and calculate NINDEG, NOUTDEG, maxindeg and maxoutdeg
	 */

	if (layout_flag == TREE_LAYOUT) {
			stop_time("step1_main");
                        layout_flag = tree_main();
			if (layout_flag != TREE_LAYOUT) { 
				FPRINTF(stderr,"\nThis is not a downward tree. ");
				FPRINTF(stderr,"Continuing with normal layout\n");
			}
        }

	if (layout_flag != TREE_LAYOUT) {
		complete_depth_lists();
		stop_time("step1_main");
	}
}


/*--------------------------------------------------------------------*/
/*  Looking for start nodes                                           */
/*--------------------------------------------------------------------*/

/*  The creation of a proper hierarchy is initially driven by a depth
 *  first search. This depth first search should start at nodes that
 *  have no predecessors (i.e. indegree=0, but note that the field 
 *  NINDEG is not yet initialized properly).
 *  Thus we reorder the nodelist that all nodes with NPRED(node)==NULL
 *  are at the beginning of this list, ordered according increasing
 *  NREFNUM. 
 *  We never start the dfs with a label node, thus we can ignore the
 *  label nodes in labellist for this task.
 *
 *  Further, a second criterium is that nodes without successors
 *  come at last. We reorder the node list that all nodes with 
 *  NSUCC(node)==NULL but NPRED(node)!=NULL are at the end of the list,
 *  ordered according increasing NREFNUM.
 */

/* startnodes is a double linked list to contain temporarily 
 * the start nodes.
 */

static GNODE startnodes;
static GNODE startnodesend;

/*  Insert a node into the startnodes
 *  ---------------------------------
 *  The startnodes are sorted according NREFNUMS.
 */ 

static void 	insert_startnode(node)
GNODE node;
{
	GNODE h,*hp;

	assert((node));
	debugmessage("insert_startnode",(NTITLE(node)?NTITLE(node):"(null)"));

	/* delete the node from the nodelist */
	if (NNEXT(node))   NBEFORE(NNEXT(node)) = NBEFORE(node);
	else 		   nodelistend 	  	= NBEFORE(node);
	if (NBEFORE(node)) NNEXT(NBEFORE(node)) = NNEXT(node);
	else 		   nodelist 	  	= NNEXT(node);

	/* search insertion point */
	h  = startnodes;
	hp = &startnodes;
	while (h) {
		if (NREFNUM(h)>=NREFNUM(node)) break;
		hp = &(NNEXT(h));
		h  = NNEXT(h);
	}

	/* insert the node into the startlist just before h */
	*hp = node;
	if (h) 	NBEFORE(node) = NBEFORE(h);
	else	NBEFORE(node) = startnodesend; 
	NNEXT(node)   = h;
	if (h)  NBEFORE(h)    = node;
	else    startnodesend = node;
}


static GNODE endnodes;
static GNODE endnodesend;

/*  Insert a node into the endnodes
 *  -------------------------------
 *  The endnodes are sorted according NREFNUMS.
 */ 

static void 	insert_endnode(node)
GNODE node;
{
	GNODE h,*hp;

	assert((node));
	debugmessage("insert_endnode",(NTITLE(node)?NTITLE(node):"(null)"));

	/* delete the node from the nodelist */
	if (NNEXT(node))   NBEFORE(NNEXT(node)) = NBEFORE(node);
	else 		   nodelistend 	  	= NBEFORE(node);
	if (NBEFORE(node)) NNEXT(NBEFORE(node)) = NNEXT(node);
	else 		   nodelist 	  	= NNEXT(node);

	/* search insertion point */
	h  = endnodes;
	hp = &endnodes;
	while (h) {
		if (NREFNUM(h)>=NREFNUM(node)) break;
		hp = &(NNEXT(h));
		h  = NNEXT(h);
	}

	/* insert the node into the endlist just before h */
	*hp = node;
	if (h) 	NBEFORE(node) = NBEFORE(h);
	else	NBEFORE(node) = endnodesend; 
	NNEXT(node)   = h;
	if (h)  NBEFORE(h)    = node;
	else    endnodesend = node;
}



/*  Prepare nodelist to have startnodes in front
 *  ============================================
 */

static void	prepare_startnodes()
{
	GNODE	node,nxt_node;

	debugmessage("prepare_startnodes","");
	startnodes = NULL;
	startnodesend = NULL;
	endnodes = NULL;
	endnodesend = NULL;
	
	/* create the list of start nodes and end nodes */
	node 	 = nodelist;
	while (node) {
		nxt_node = NNEXT(node);
		if (!NPRED(node)) 	insert_startnode(node);
		else if (!NSUCC(node)) 	insert_endnode(node);
		node = nxt_node;
	}

	/* and insert the start node list just in front of nodelist */
	if (startnodes) {
		if (nodelist) NBEFORE(nodelist) = startnodesend;
		NNEXT(startnodesend) = nodelist;
		nodelist = startnodes;
		if (!nodelistend) nodelistend = startnodesend;
	}

	/* and insert this end node list just at the end of nodelist */
	if (endnodes) {
		if (nodelistend) NNEXT(nodelistend) = endnodes;
		NBEFORE(endnodes) = nodelistend;
		nodelistend = endnodesend;
		if (!nodelist) nodelist = endnodes;
	}
}	

/*--------------------------------------------------------------------*/
/*  Anchored edges                                                    */
/*--------------------------------------------------------------------*/

/*  Treatment of anchored edges
 *  ===========================
 *  For anchored edges, we create an anchor dummy node, that is 
 *  connected with the source. All edges are replaced by edges to
 *  the anchor node. 
 *  The CONNECT fields of source node and anchor node are filled.
 *  Note: The use of anchored edges restricts the use of nearedges.
 */

void insert_anchor_edges()
{
	GEDGE   edge;

	debugmessage("insert_anchor_edges","");
	assert((dummylist==NULL));
	edge = edgelist;
	while (edge) {
		if (  (EANCHOR(edge)<=64) && (EANCHOR(edge)>0)
		    &&(!EINVISIBLE(edge))) 
			prepare_anchoredge(edge);
		edge = ENEXT(edge);
	}
	edge = tmpedgelist;
	while (edge) {
		if (  (EANCHOR(edge)<=64) && (EANCHOR(edge)>0)
		    &&(!EINVISIBLE(edge))) 
			prepare_anchoredge(edge);
		edge = EINTERN(edge);
	}
}

/*  Prepare one anchor edge
 *  -----------------------
 *  If the anchoredge starts on a node that has already an anchornode,
 *  we redirect the edge to point from this anchornode.
 *  Otherwise we create the anchornode and redirect the edge, too. 
 */

static void prepare_anchoredge(edge)
GEDGE edge;
{
	GEDGE   h;
	GNODE   v;
	CONNECT c1,c2;

	debugmessage("prepare_anchoredge","");
	if ((G_orientation==LEFT_TO_RIGHT)||(G_orientation==RIGHT_TO_LEFT)) {
		G_orientation= TOP_TO_BOTTOM;
		if (!silent) {	
			FPRINTF(stderr,"Orientation ignored, because");
			FPRINTF(stderr," edge attribute `anchor' used !\n");
		}
	}
	c1 = NCONNECT(ESTART(edge));
	if (!c1) { 
		v = create_dummy(-1);
		NHORDER(v) = NHORDER(ESTART(edge));
		NANCHORNODE(v) = 1;
		h = tmpedgealloc(ELSTYLE(edge),
			ETHICKNESS(edge),
			ECLASS(edge),
			200,
			ECOLOR(edge),
               		EARROWSIZE(edge),
			EHORDER(edge));
		EANCHOR(h) = 66;
		ESTART(h) = ESTART(edge);
		EEND(h)   = v;
		ELABEL(h) = NULL;
		c1 = connectalloc(ESTART(h));
		CTARGET(c1) = v;
		CEDGE(c1)   = h;
		c2 = connectalloc(v);
		CTARGET(c2) = ESTART(h);
		CEDGE(c2)   = h;
	}
	v = CTARGET(c1); 		
	assert((v));
	assert((NANCHORNODE(v)));
	h = tmpedgealloc(ELSTYLE(edge),
		ETHICKNESS(edge),
		ECLASS(edge),
		EPRIO(edge),
		ECOLOR(edge),
                EARROWSIZE(edge),
		EHORDER(edge));
	EANCHOR(h) = -EANCHOR(edge);
	ESTART(h) = v;
	EEND(h)   = EEND(edge);
	ELABEL(h) = ELABEL(edge);
	delete_adjedge(edge);
	EINVISIBLE(edge) = 0;
	create_adjedge(h);
}


/*--------------------------------------------------------------------*/
/*  Near edge connections                                             */
/*--------------------------------------------------------------------*/

/*  Treatment of near edge
 *  ======================
 *  If edges are specified as nearedges, they must be layouted at the
 *  same level. We use the connection field of nodes to notify this.
 *  All near_edges are in the near_edge_list. The CONNECT fields
 *  for them are filled here. Later, we will fill additional CONNECT
 *  fields if it happen to be possible to neighbour directly some
 *  nodes that are connected by usual edges.
 */

static void	insert_near_edges()
{
	ADJEDGE edge;
	CONNECT c1,c2;
	int connection_possible;

	debugmessage("insert_near_edges","");
	edge = near_edge_list;
	while (edge) {
		if (!EINVISIBLE(AKANTE(edge))) { 
			c1 = NCONNECT(SOURCE(edge));
			c2 = NCONNECT(TARGET(edge));
			connection_possible = 1;
			if ((c1) && (CTARGET(c1)) && (CTARGET2(c1))) 
				connection_possible = 0;
			if ((c2) && (CTARGET(c2)) && (CTARGET2(c2))) 
				connection_possible = 0;

			if (check_connect_cycle(TARGET(edge),NULL,
						SOURCE(edge))) 
				connection_possible = 0;

			if (connection_possible) {
				if (!c1) {
					c1 = connectalloc(SOURCE(edge));
					CTARGET(c1) = TARGET(edge);
					CEDGE(c1)   = AKANTE(edge);
				}
				else if (!CTARGET2(c1)) {
					CTARGET2(c1) = TARGET(edge);
					CEDGE2(c1)   = AKANTE(edge);
				}
				if (!c2) {
					c2 = connectalloc(TARGET(edge));
					CTARGET(c2) = SOURCE(edge);
					CEDGE(c2)   = AKANTE(edge);
				}
				else if (!CTARGET2(c2)) {
					CTARGET2(c2) = SOURCE(edge);
					CEDGE2(c2)   = AKANTE(edge);
				}
				delete_adjedge(AKANTE(edge));
				EINVISIBLE(AKANTE(edge)) = 0;
			}
			else if (!silent) {
				FPRINTF(stderr,"Nearedge connection (");
				if (NTITLE(SOURCE(edge))[0])
					FPRINTF(stderr,"%s",NTITLE(SOURCE(edge)));
				FPRINTF(stderr," , ");
				if (NTITLE(TARGET(edge))[0])
					FPRINTF(stderr,"%s",NTITLE(TARGET(edge)));
				FPRINTF(stderr,") ignored ! Sorry !\n");
			}
		}
		edge = ANEXT(edge);
	}
}


/* Check the connection chain for cycles 
 * -------------------------------------
 * We want to insert (z,v) as connection. We check
 * whether this yield a connection cycle, i.e. whether
 * z is reacheable from v. w is the node we were just 
 * previously.
 * We return 1, if z is reacheavle.
 */

static int  check_connect_cycle(v,w,z)
GNODE v;
GNODE w;
GNODE z;
{
	CONNECT c;
	int r;

	debugmessage("check_connect_cycle","");

	if (!near_edge_layout) return(1);
	if (!v) return(0);
	if (v==z) return(1);
	r = 0;
	c = NCONNECT(v);
	if (!c) return(0);
	if (CTARGET(c) && (CTARGET(c)!=w)) 
		r |= check_connect_cycle(CTARGET(c),v,z);
	if (CTARGET2(c) && (CTARGET2(c)!=w)) 
		r |= check_connect_cycle(CTARGET2(c),v,z);
	return(r);
}


/*--------------------------------------------------------------------*/
/*  Partition of edges with tree traversal                            */
/*--------------------------------------------------------------------*/

/*  Partition of edges, Driver
 *  ==========================
 *  We calculate by a dfs the spanning tree of the graph to detect:
 *     1) self loops     (v,v)
 *     2) tree edges     (v,w) that are part of the spanning tree
 *     3) forward edges  (v,w) if there is a path on the spanning tree 
 *                             from v to w.
 *     4) backward edges (v,w) if there is a path on the spanning tree
 *			       from w to v
 *     5) cross edges    (v,w) otherwise.
 *
 *  The algorithm is very popular known; e.g., it can be found in 
 *  K.Mehlhorn: Data Structures and Algorithms, Part II, Springer 1984.
 *  The spanning tree is the base of our layout. All nodes of the same
 *  tree depth will have the same vertical coordinates, i.e. they form
 *  a layer. IN FACT, WE ARE ONLY INTERESTED ON THIS DEPTH OF THE
 *  NODES, AND NOT ON THE PARTITIONING ITSELF. 
 *  As side effect, we calculate maxdepth, the maximal depth of the
 *  spanning tree.
 *
 *  If layout_flag 1|2 is specified, we do two partinitioning and take
 *  that one which has a larger|smaller depth. The difference between both is
 *  the order we analyse the edges. 
 */ 

static int	act_dfsnum;  /* actual ref. number of entry of dfs      */
static int	act_cplnum;  /* actual ref. number of completion of dfs */
static int	act_level;   /* actual level (depth) in spanning tree   */

static void partition_edges()
{
	GNODE	node;
	int	depth1;
 
	debugmessage("partition_edges","");

	/* First try */
	/* --------- */

    	act_dfsnum = 1;
    	act_cplnum = 1;
    	maxdepth = 0;		/* maximal depth of the spanning tree */
        
	gs_wait_message('p');
	node = nodelist;
	while (node) {
		if ( !NMARK(node) ) {
			act_level 	= 0; 
			depth_first_search(node);
        	}
		node = NNEXT(node);
   	} 

	/* labels are always reachaeble from other nodes, thus all label
 	 * nodes should already be marked now !!!
	 */

#ifdef CHECK_ASSERTIONS
	node = labellist;
	while (node)	{
		if ( !NMARK(node) ) assert((0));
		node = NNEXT(node);
   	} 
#endif

	/* maximize or minimize ? */
	if (layout_flag==0) return;
	if (layout_flag==TREE_LAYOUT) return;

	/* Second try */
	/* ---------- */

       	depth1 = maxdepth;

	gs_wait_message('p');
	node = nodelist;
	while (node) {
		NMARK(node) = 0; 
		node = NNEXT(node);
   	} 
	node = labellist;
	while (node) {
		NMARK(node) = 0; 
		node = NNEXT(node);
   	} 
	node = dummylist;
	while (node) {
		NMARK(node) = 0; 
		node = NNEXT(node);
   	} 

    	act_dfsnum = 1;
    	act_cplnum = 1;
    	maxdepth = 0;		/* maximal depth of the spanning tree */
        
	node = nodelist;
	while (node) {
		if ( !NMARK(node) ) {
			act_level 	= 0; 
			alt_depth_first_search(node);
        	}
		node = NNEXT(node);
   	} 

	/* labels are always reachable from other nodes, thus all label
 	 * nodes should already be marked now !!!
	 */

#ifdef CHECK_ASSERTIONS
	node = labellist;
	while (node)	{
		if ( !NMARK(node) ) assert((0));
		node = NNEXT(node);
   	} 
#endif

	/* maximize ? */
	if ((layout_flag== 1)&&(depth1 <= maxdepth)) return;
	/* or minimize ? */
	if ((layout_flag== 2)&&(depth1 >= maxdepth)) return;

	/* First try was better: redo it */
	/* ----------------------------- */

	gs_wait_message('p');
	node = nodelist;
	while (node) {
		NMARK(node) = 0; 
		node = NNEXT(node);
   	} 
	node = labellist;
	while (node) {
		NMARK(node) = 0; 
		node = NNEXT(node);
   	} 
	node = dummylist;
	while (node) {
		NMARK(node) = 0; 
		node = NNEXT(node);
   	} 

    	act_dfsnum = 1;
    	act_cplnum = 1;
    	maxdepth = 0;		/* maximal depth of the spanning tree */
        
	node = nodelist;
	while (node) {
		if ( !NMARK(node) ) {
			act_level 	= 0; 
			depth_first_search(node);
        	}
		node = NNEXT(node);
   	} 

	/* labels are always reachaeble from other nodes, thus all label
 	 * nodes should already be marked now !!!
	 */


#ifdef CHECK_ASSERTIONS
	node = labellist;
	while (node)	{
		if ( !NMARK(node) ) assert((0));
		node = NNEXT(node);
   	} 
#endif

}



/*  The depth first search (1. version)
 *  ----------------------------------
 *  traverses the tree starting at node and does the partitioning.  
 */

static void depth_first_search(node)
GNODE	node;
{
    	GNODE  	kn;
    	ADJEDGE	edge;
	int     priority;
	
	assert((node));
	debugmessage("depth_first_sea",(NTITLE(node)?NTITLE(node):"(null)"));
       	if (NMARK(node)) return;
 
	NMARK(node)  = 1;
       	NDFS(node)   = act_dfsnum++;
	if (NLEVEL(node)>=0) act_level = NLEVEL(node);
       	NTIEFE(node) = act_level;
       	maxdepth = (act_level>maxdepth) ? act_level : maxdepth;

	if (NCONNECT(node)) {
		/* Connections should be on the same level !!! 
		 * Note: Connections are never self loops.
		 */
		if (CTARGET(NCONNECT(node))) 
			depth_first_search(CTARGET(NCONNECT(node)));
		if (CTARGET2(NCONNECT(node))) 
			depth_first_search(CTARGET2(NCONNECT(node)));
	}

	priority = 1;
	while (priority>-1) {

		/* Fetch next priority value */
		priority = -1;
    		edge = NSUCC(node);
    		while (edge) {
			assert((SOURCE(edge)==node));
			kn = TARGET(edge);
			if ((!NMARK(kn)) && (EPRIO(AKANTE(edge))>priority)) 
				priority = EPRIO(AKANTE(edge));
			edge = ANEXT(edge);
		}
		if (priority==-1) break;

    		edge = NSUCC(node);
    		while (edge) {
			if (EPRIO(AKANTE(edge))!=priority) {
				edge = ANEXT(edge);
				continue;
			}
			assert((SOURCE(edge)==node));
			kn = TARGET(edge);
        		if ( !NMARK(kn) ) {
			       /*
        		 	* EKIND(edge) = 'T';
			 	*/
        			act_level++;
        			depth_first_search(kn);
				act_level = NTIEFE(node);
        		}
        		else { /* partitioning into forward, backward, cross 
				* and self edges,i.e. kinds 'F', 'B', 'C', 'S'.
				* Note NCOMP(kn)==0 implies that dfs(kn) is not 
				* yet completed, i.e. that after all dfs it will
				* be NCOMP(kn)>NCOMP(node).
				*
            			* if ( NDFS(node)<NDFS(kn) ) EKIND(edge)='F';
            			* else if (NDFS(node)>NDFS(kn) && NCOMP(kn)==0) 
                		* 	EKIND(edge) = 'B';
				* else if (kn == node) EKIND(edge) = 'S';
				* else EKIND(edge) = 'C';	
				*
				* The tags 'T', 'B', 'C' and 'F' are never used!
				* Thus it is nonsense to calculate them.
				* The tag 'S' is important.
				*/
				if (kn == node) EKIND(edge) = 'S';
        		}
            		edge = ANEXT(edge);
		}
	} /* while priority */

       	/* 
	 *  NCOMP(node)= act_cplnum++;
	 */
}
     	

/*  The depth first search (2. version)
 *  ----------------------------------
 *  traverses the tree starting at node and does the partitioning.  
 */

static void alt_depth_first_search(node)
GNODE	node;
{
	GNODE kn;
    	ADJEDGE	edge;
	int     priority;

	assert((node));
	debugmessage("alt_depth_first",(NTITLE(node)?NTITLE(node):"(null)"));
       	if (NMARK(node)) return;
 
	NMARK(node)  = 1;
       	NDFS(node)   = act_dfsnum++;
	if (NLEVEL(node)>=0) act_level = NLEVEL(node);
       	NTIEFE(node) = act_level;
       	maxdepth = (act_level>maxdepth) ? act_level : maxdepth;

	if (NCONNECT(node)) {
		/* Connections should be on the same level !!! 
		 * Note: Connections are never self loops.
		 */
		if (CTARGET(NCONNECT(node))) 
			alt_depth_first_search(CTARGET(NCONNECT(node)));
		if (CTARGET2(NCONNECT(node))) 
			alt_depth_first_search(CTARGET2(NCONNECT(node)));
	}
	priority = 1;
	while (priority>-1) {

		/* Fetch next priority value */
		priority = -1;
    		edge = NSUCC(node);
    		while (edge) {
			assert((SOURCE(edge)==node));
			kn = TARGET(edge);
			if ((!NMARK(kn)) && (EPRIO(AKANTE(edge))>priority)) 
				priority = EPRIO(AKANTE(edge));
			edge = ANEXT(edge);
		}
		if (priority==-1) break;

		start_dfs_backwards(NSUCC(node),node,priority);
	}
}


/*  Traverse of adjacency lists in alt_depth_first_search
 *  -----------------------------------------------------
 *  In alt_depth_first_search, we traverse the adjacency lists
 *  backwards.
 */

static void start_dfs_backwards(edge,node,priority)
ADJEDGE edge;
GNODE   node;
int 	priority;
{
    	GNODE  	kn;

	debugmessage("start_dfs_backwards","");

	if (!edge) return;
	start_dfs_backwards(ANEXT(edge),node,priority);

	if (EPRIO(AKANTE(edge))!=priority) return;

	kn = TARGET(edge);
       	if ( !NMARK(kn) ) {
		/*
       		 * EKIND(edge) = 'T';
		 */
       		act_level++;
       		alt_depth_first_search(kn);
       		act_level = NTIEFE(node);
       	}
       	else { /* partitioning into forward, backward, cross and self
		* edges, i.e. kinds 'F', 'B', 'C', 'S'.
		* Note NCOMP(kn)==0 implies that dfs(kn) is not 
		* yet completed, i.e. that after all dfs it will
		* be NCOMP(kn)>NCOMP(node).
		*
           	* if ( NDFS(node) < NDFS(kn) ) EKIND(edge) = 'F';
           	* else if ( NDFS(node) > NDFS(kn) && NCOMP(kn) == 0) 
               	* 	EKIND(edge) = 'B';
		* else if (kn == node) EKIND(edge) = 'S';
		* else EKIND(edge) = 'C';	
		*
		* The tags 'T', 'B', 'C' and 'F' are never used !
		* Thus it is nonsense to calculate them.
		* The tag 'S' is important.
		*/
		if (kn == node) EKIND(edge) = 'S';
        }
}

/*--------------------------------------------------------------------*/
/*  Partition of edges with topological sorting                       */
/*--------------------------------------------------------------------*/

/* Topological sorting creates a layout with minimum backward edges.
 * If a topological order exists, we have no backward edge. Otherwise,
 * (on cyclic graphs) it results in a graph with minimal backward edges.
 *
 * We keep a list `zero_indegree_list' containing the nodes without
 * unmarked predecessor.
 */

static GNLIST zero_indegree_list; 
static GNLIST zero_free_list; 

/* Add a node to the zero_indegree_list
 * ------------------------------------
 */

static void add_to_zero_indegree_list(v)
GNODE v;
{
	GNLIST h;

	debugmessage("add_to_zero_indegree_list","");

	if (zero_free_list) {
		h = zero_free_list;
		zero_free_list = GNNEXT(zero_free_list);
	}
	else h = tmpnodelist_alloc();
	GNNODE(h) = v;
	GNNEXT(h) = zero_indegree_list;
	zero_indegree_list = h;
}

/* Delete the first entry of the zero_indegree_list
 * ------------------------------------------------
 * and return the node. Returns NULL on failure.
 */

static GNODE get_zero_indegree()
{
	GNLIST h;

	debugmessage("get_zero_indegree","");
	h = zero_indegree_list;
	if (h) {
		zero_indegree_list = GNNEXT(h);
		GNNEXT(h) = zero_free_list;
		zero_free_list = h;
		return(GNNODE(h));
	}
	return(NULL);
}


/* Topological sort try
 * ====================
 */

static void	topological_sort()
{
	GNODE  v;
	int    not_ready, actlevel;

	debugmessage("topological_sort","");

	/* look for the nodes without predecessor and put them into
	 * the zero_indegree_list.
	 */ 

	zero_indegree_list = NULL;
	zero_free_list     = NULL; 
	v = nodelist;
	while (v) {
		if (topsort_indegree(v,v)==0) add_to_zero_indegree_list(v);
		v = NNEXT(v);
	}
	/* Labels don't have zero indegree here */

    	maxdepth = 0;		/* maximal depth of the layout */
        
	gs_wait_message('p');
	not_ready = 1;
	while (not_ready) {
		/* First: the normal topological sorting algorithm
		 */
		while ((v = get_zero_indegree())!=NULL) {
			actlevel = topsort_maxlevel(v,v);
			if (NLEVEL(v)>=0) actlevel = NLEVEL(v);
			if (maxdepth<actlevel) maxdepth = actlevel;
			topsort_setlevel(v,v,actlevel);
			topsort_add_succs(v,v);
		}

		/* Now we have no nodes without predecessor, i.e. all
		 * remaining nodes are part of cycles.
		 * Check whether there are remainig parts.
		 */
		gs_wait_message('p');
		not_ready = 0;	
		v = nodelist;
		while (v && !not_ready) {
			if (!NMARK(v)) {
				add_to_zero_indegree_list(v); not_ready = 1;
			}
			v = NNEXT(v);
		}
		v = labellist;
		while (v && !not_ready) {
			if (!NMARK(v)) {
				add_to_zero_indegree_list(v); not_ready = 1;
			}
			v = NNEXT(v);
		}
		v = dummylist;
		while (v && !not_ready) {
			if (!NMARK(v)) {
				add_to_zero_indegree_list(v); not_ready = 1;
			}
			v = NNEXT(v);
		}
	}
}


/* Calculate the maximal predecessor level
 * ---------------------------------------
 * node1 is the node we want to analyze. node2 is the node we just come from.
 */

static int topsort_maxlevel(node1,node2)
GNODE node1;
GNODE node2;
{
	int result, h;
	ADJEDGE a;
	CONNECT c;

	debugmessage("topsort_maxlevel","");
	result = 0;
	a = NPRED(node1);
	while (a) {
		if (SOURCE(a)==node1) EKIND(a) = 'S';
		else if (NMARK(SOURCE(a))) {
			if (NTIEFE(SOURCE(a))>=result)
				result = NTIEFE(SOURCE(a))+1;
		}
		a = ANEXT(a);
	}
	c = NCONNECT(node1);
	if (c && CTARGET(c) && (CTARGET(c)!=node2)) {  
		h = topsort_maxlevel(CTARGET(c),node1);
		if (h>result) result = h;
	}
	if (c && CTARGET2(c) && (CTARGET2(c)!=node2)) {  
		h = topsort_maxlevel(CTARGET2(c),node1);
		if (h>result) result = h;
	}
	return(result);
}

/* Set NTIEFE of all nearedge node  
 * --------------------------------
 * node1 is the node we want to analyze. node2 is the node we just come from.
 */

static void topsort_setlevel(node1,node2,level)
GNODE node1;
GNODE node2;
int   level;
{
	CONNECT c;

	debugmessage("topsort_setlevel","");
	if ((NLEVEL(node1)>=0)&&(level!=NLEVEL(node1))) {
		if (!silent) {	
			FPRINTF(stderr,"Level specification ignored, ");
			FPRINTF(stderr,"because nearedge was specified !\n");
		}
	}
	NTIEFE(node1) = level;
	NMARK(node1) = 1;
	c = NCONNECT(node1);
	if (c && CTARGET(c) && (CTARGET(c)!=node2))  
		topsort_setlevel(CTARGET(c),node1,level);
	if (c && CTARGET2(c) && (CTARGET2(c)!=node2)) 
		topsort_setlevel(CTARGET2(c),node1,level);
}

/* Add successors to zero_indegree_list
 * ------------------------------------
 * If the successors of this node and all nodes connected to it 
 * by nearedges have nor unmarked predecessors, then we add
 * the successors to the zero_indegree_list.
 * node1 is the node we want to analyze. node2 is the node we just come from.
 */

static void topsort_add_succs(node1,node2)
GNODE node1;
GNODE node2;
{
	ADJEDGE a;
	CONNECT c;

	debugmessage("topsort_add_succs","");

	a = NSUCC(node1);
	while (a) {
		if (  (topsort_indegree(TARGET(a),TARGET(a))==0)
		    &&(!NMARK(TARGET(a))) ) 
			add_to_zero_indegree_list(TARGET(a));
		a = ANEXT(a);
	}
	c = NCONNECT(node1);
	if (c && CTARGET(c) && (CTARGET(c)!=node2))  
		topsort_add_succs(CTARGET(c),node1);
	if (c && CTARGET2(c) && (CTARGET2(c)!=node2)) 
		topsort_add_succs(CTARGET2(c),node1);
}


/* Check how many predecessors of a node are marked
 * ------------------------------------------------
 * returns the number of unmarked indegrees
 * node1 is the node we want to analyze. node2 is the node we just come from.
 */

static int topsort_indegree(node1,node2)
GNODE node1;
GNODE node2;
{
	int result;
	ADJEDGE a;
	CONNECT c;

	debugmessage("topsort_indegree","");
	result = 0;
	a = NPRED(node1);
	while (a) {
		if ( !NMARK(SOURCE(a)) ) result++;
		a = ANEXT(a);
	}
	c = NCONNECT(node1);
	if (c && CTARGET(c) && (CTARGET(c)!=node2))  
		result += topsort_indegree(CTARGET(c),node1);
	if (c && CTARGET2(c) && (CTARGET2(c)!=node2)) 
		result += topsort_indegree(CTARGET2(c),node1);
	return(result);
}


/*--------------------------------------------------------------------*/
/*  Add labels after partitioning                                     */
/*--------------------------------------------------------------------*/

/* If we still don't have the labels, we double the NTIEFE of the nodes
 * and add the labels in between.
 */

static void 	add_phase1_labels()
{
	GNODE v, vl, vt;
	ADJEDGE edge, edgenext;

	debugmessage("add_phase1_labels","");

	maxdepth = 2*maxdepth;
	v = nodelist;
	while (v) {
		NTIEFE(v) = 2* NTIEFE(v); 
		v = NNEXT(v);
	}	

	v = nodelist;
	while (v) {
		edge = NSUCC(v); 
		while (edge) {
			edgenext = ANEXT(edge);
			if (ELABEL(AKANTE(edge))) {
				vl = create_labelnode(AKANTE(edge));
				vt = TARGET(edge);
				NTIEFE(vl) = ( NTIEFE(SOURCE(edge))
					      +NTIEFE(TARGET(edge)))/2;
				(void)create_edge(v,vl,AKANTE(edge),1);
				(void)create_edge(vl,vt,AKANTE(edge),0);
				delete_adjedge(AKANTE(edge));
			}
			edge = edgenext;
		}
		v = NNEXT(v);
	}	
}


/*--------------------------------------------------------------------*/
/*  Relook the partitioning                                           */
/*--------------------------------------------------------------------*/

/*  A second analysis of the partitioning
 *  =====================================
 *  It may happen, that after partitioning of edges, a node is at 
 *  depth i, but all its predecessors are at depth i-1 and all
 *  successors are at depth  i+n. Then it is better to put such
 *  nodes into depth i+n/2, to support that all edges have nearly
 *  the same length.
 *  E.g.:    A --> B ---------------> C  is changed into
 *           A --------> B ---------> C.
 *
 *  A second task here is to push label nodes into an upper depth
 *  if the target of the edge the label belongs to is in an upper
 *  depth.
 *  Note: We do NEVER change the depth of a node that has a connection
 *  field, because this would destroy the nearedge relation.
 *  It gives no sense to do this too often. Thus we stop after 5 times,
 *  even if the result is still not optimal.
 */

static void tune_partitioning()
{
	GNODE v;
	int   changed,count;

	debugmessage("tune_partitioning","");

	count   = 0;
	changed = 1;
	while (changed) {
		changed = 0;
		gs_wait_message('p');

		/* First, check normal nodes */
		v = nodelist;
		while (v) {
			if (!NCONNECT(v)) changed += tune_node_depth(v,0); 
			v = NNEXT(v);
		}	

		/* Then, check labels */
		v = labellist;
		while (v) {
			if (!NCONNECT(v)) changed += tune_node_depth(v,1); 
			v = NNEXT(v);
		}	
		count++;
		if (count>=50) return;
	}
} 


/*  Check tuning of the depth of a node v
 *  -------------------------------------
 *  This node has no nearedge, thus we can change its depth NTIEFE.
 */

static int tune_node_depth(v, lab)
GNODE v;
int lab;
{ 
	int   nodelevel,leveldiff,nr_edges,changed;
	ADJEDGE edge;

	debugmessage("tune_node_depth","");

	/*  If the node has a fixed specified level, we should not
	 *  change it.
	 */
	if (NLEVEL(v)>=0) return(0);
	nr_edges  = 0;
	leveldiff = 0;
	nodelevel = NTIEFE(v);
    	edge = NPRED(v);
    	while (edge) {
		nr_edges += EPRIO(AKANTE(edge));
		if (NSUCC(v)) {
			leveldiff += (EPRIO(AKANTE(edge))*
				(NTIEFE(SOURCE(edge))-nodelevel));
		}
		else {
			leveldiff += (EPRIO(AKANTE(edge))*
				(NTIEFE(SOURCE(edge))-nodelevel+1));
		}
		edge = ANEXT(edge);
	}
    	edge = NSUCC(v);
    	while (edge) {
		nr_edges += EPRIO(AKANTE(edge));
		if (NPRED(v)) {
			leveldiff += (EPRIO(AKANTE(edge))*
				(NTIEFE(TARGET(edge))-nodelevel));
		}
		else {
			leveldiff += (EPRIO(AKANTE(edge))*
				(NTIEFE(TARGET(edge))-nodelevel-1));
		}
		edge = ANEXT(edge);
	}
	if (nr_edges==0) return(0);
	changed = 0;
	if (leveldiff/nr_edges <= -2) {
		changed = 1;
		nodelevel += (leveldiff/nr_edges);
	}
	else if (leveldiff/nr_edges >  0) { 
		changed = 1;
		nodelevel += (leveldiff/nr_edges);
	}

	/* On minbackward, we must ensure that this does not create
	 * additional backward edges.
	 */
	if ((changed) && (layout_flag == 3) && (nodelevel > NTIEFE(v))) {
                edge = NSUCC(v);
                while (edge) {
                        if (nodelevel>=NTIEFE(TARGET(edge)))
				nodelevel = NTIEFE(TARGET(edge)) - 1; 
                        edge = ANEXT(edge);
                }
		if (nodelevel <= NTIEFE(v)) changed = 0;
	}
	if ((changed) && (layout_flag == 3) && (nodelevel < NTIEFE(v))) {
                edge = NPRED(v);
                while (edge) {
                        if (nodelevel>=NTIEFE(SOURCE(edge)))
				nodelevel = NTIEFE(SOURCE(edge)) + 1; 
                        edge = ANEXT(edge);
                }
		if (nodelevel >= NTIEFE(v)) changed = 0;
	}
	if ((changed) && (layout_flag == 3)) {
                edge = NPRED(v);
                while (edge && changed) {
                        if (   (nodelevel<=NTIEFE(SOURCE(edge)))
			    && (NTIEFE(v)> NTIEFE(SOURCE(edge)))) changed = 0;
                        edge = ANEXT(edge);
                }
                edge = NSUCC(v);
                while (edge && changed) {
                        if (   (nodelevel>=NTIEFE(TARGET(edge))) 
			    && (NTIEFE(v)< NTIEFE(TARGET(edge)))) changed = 0;
                        edge = ANEXT(edge);
                }
	}
        if (changed) {
                edge = NPRED(v);
                while (edge && changed) {
                        if (nodelevel==NTIEFE(SOURCE(edge))) changed = 0;
                        edge = ANEXT(edge);
                }
                edge = NSUCC(v);
                while (edge && changed) {
                        if (nodelevel==NTIEFE(TARGET(edge))) changed = 0;
                        edge = ANEXT(edge);
                }
        }
	if (nodelevel<0) return(0);
	if (nodelevel>maxdepth) return(0);
	if (changed) NTIEFE(v) = nodelevel;
	return(changed);
}


/*--------------------------------------------------------------------*/
/*  Fill the array layer                                              */
/*--------------------------------------------------------------------*/

/*  Layer list creation 
 *  ===================
 *  After depth first search and allocation of the array layer,
 *  we traverse all visible nodes and insert them into the array layer.
 *  Note that until now, we create and use only the TSUCC list and
 *  not the TPRED list.
 */
 
static void create_depth_lists()
{
	GNODE h;
	GNLIST hl;
	int t;

	debugmessage("create_depth_lists","");

	h = nodelist;
	while (h) {
		t = NTIEFE(h);
		assert((t<=maxdepth));
		hl = tmpnodelist_alloc();
		GNNEXT(hl) = TSUCC(layer[t]);
		TSUCC(layer[t]) = hl;
        	GNNODE(hl) = h;
		h = NNEXT(h);
	}
	h = labellist;
	while (h) {
		t = NTIEFE(h);
		assert((t<=maxdepth+1));
		hl = tmpnodelist_alloc();
		GNNEXT(hl) = TSUCC(layer[t]);
		TSUCC(layer[t]) = hl;
        	GNNODE(hl) = h;
		h = NNEXT(h);
	}
	h = dummylist;
	while (h) {
		t = NTIEFE(h);
		assert((t<=maxdepth+1));
		hl = tmpnodelist_alloc();
		GNNEXT(hl) = TSUCC(layer[t]);
		TSUCC(layer[t]) = hl;
        	GNNODE(hl) = h;
		h = NNEXT(h);
	}
}


/*  Layer list completion
 *  ---------------------
 *  Because during `inspect_edges' the layers are changed by adding 
 *  dummy nodes, we have only used TSUCC of the layers. Now, at the
 *  end of this module, we derive the TPRED lists from the TSUCC
 *  lists. TPRED(layer[i]) is the reverted list of TSUCC(layer[i]), 
 *  without the connection nodes. E.g. in A->B->C, the TSUCC-list
 *  contains A, B and C but the TPRED-list only contains A.
 *  NINDEG and NOUTDEG are calculated here. 
 *
 *  If a node comes not into the TPRED-list because it has connections,
 *  we mark the node by NMARK.
 */

/* Abreviations to check whether a connection is a forward connection 
 * or backward connection.
 */

static void	complete_depth_lists()
{
	int 	i;
	GNLIST	n,hl;
	GNODE	node;
	ADJEDGE	edge;
	int	backward_connection;
	int	forward_connection;
	CONNECT c;

	debugmessage("complete_depth_lists","");

	maxindeg = maxoutdeg = 0;

	for (i=0; i<=maxdepth+1; i++) {
		n = TSUCC(layer[i]);
		while (n) {
			NMARK(GNNODE(n))=0;
			n = GNNEXT(n);
		}
		n = TSUCC(layer[i]);
		assert((TPRED(layer[i])==NULL));
		while (n) {
			node = GNNODE(n);

			/* If there is a forward connection, save the
			 * adjacency lists. Check for backward connection.
			 */
			backward_connection = 0;
			forward_connection = 0;
			c = NCONNECT(node);
			if (c) {
				if (backward_connection1(c))
					backward_connection = 1;
				if (backward_connection2(c))
					backward_connection = 1;
				if (forward_connection1(c))
					forward_connection = 1;
				if (forward_connection2(c))
					forward_connection = 1;
			}
			if (  (forward_connection)&&(!backward_connection)
			    &&(NMARK(GNNODE(n))==0)) 
				calc_connect_adjlists(node,node,NULL);

			/* fill TPRED-list */
			if (  (!backward_connection)&&(NMARK(GNNODE(n))==0)) {
				hl = tmpnodelist_alloc();
				GNNEXT(hl) = TPRED(layer[i]);
				TPRED(layer[i]) = hl;
				GNNODE(hl) = node;
			}

			/* count the edges. Note: forward connections 
			 * have its successor and all successors of
			 * their direct neigbours. 
			 */
			edge = NSUCC(node);
			while (edge) {
				NOUTDEG(node)++;
				assert((NTIEFE(TARGET(edge))>=i));
				edge = ANEXT(edge);
			}
			edge = NPRED(node);
			while (edge) {
				NINDEG(node)++;
				edge = ANEXT(edge);
			}
			if (NOUTDEG(node) > maxoutdeg)
				maxoutdeg = NOUTDEG(node);
			if (NINDEG(node) > maxindeg)
				maxindeg = NINDEG(node);
			n = GNNEXT(n);
		}
	}
}


/*  Calculate adjacencies at connections
 *  ------------------------------------
 *  Add the adjacencies of a connection node w to the node v.
 *  We start these method by nodes that have only forward connections.
 *  Because connections are not cyclic, we will always find such nodes.
 *  predw indicated where we come, to prevent running in a cycle. 
 *  Further, we mark the node w by NMARK, to prevent to calculate again
 *  the connection of w.
 *  This changes destructively the SOURCE and TARGET-field of nodes.
 *  Note that at connection nodes SOURCE(NSUCC(w))==w and
 *  TARGET(NPRED(w))==w does NOT HOLD anymore after this.
 *
 *  The situation in an example:
 *  Before:   A      B     C             After   A  B  C
 *            |      |     |                      \ | /
 *            V      V     V                       \|/
 *            X----->Y---->Z                        X----->Y---->Z
 *             connections                           connections
 */

static void calc_connect_adjlists(v,w,predw)
GNODE v;
GNODE w;
GNODE predw;
{
	ADJEDGE	edge;
	CONNECT c;

	debugmessage("calc_connect_adjlists","");

	if (v!=w) NMARK(w)=1;
	/* save a copy of the actual adjacency lists of w to v */
	if (v==w) {
		NSVSUCC(w) = NSUCC(w);
		NSVPRED(v) = NPRED(v);
	}
	edge = NSUCC(w);
	if (v==w) NSUCC(v) = NULL;
	while (edge) {
		succedgealloc(v,AKANTE(edge));
		SOURCE(edge) = v;
		edge = ANEXT(edge);
	}
	edge = NPRED(w);
	if (v==w) NPRED(v) = NULL;
	while (edge) {
		prededgealloc(v,AKANTE(edge));
		TARGET(edge) = v;
		edge = ANEXT(edge);
	}

	c = NCONNECT(w);
	if (!c) return;
	if (CTARGET(c) && (CTARGET(c)!=predw)) 
		calc_connect_adjlists(v,CTARGET(c),w);
	if (CTARGET2(c) && (CTARGET2(c)!=predw)) 
		calc_connect_adjlists(v,CTARGET2(c),w);
}



/*--------------------------------------------------------------------*/
/*  Inspect edges                                                     */
/*--------------------------------------------------------------------*/

/*  Analyse whether the hierarchy is proper
 *  =======================================
 *  Note that dummy nodes are inserted at beginning of the layer[i]-lists
 *  i.e. are not automatically checked, if they are inserted at the
 *  actual or an upper layer. Thus we need recursion in check_edge.
 */ 

static void    inspect_edges()
{
	int     i;
	GNLIST	li;
	GNODE   node;
	ADJEDGE edge,nextedge;

	debugmessage("inspect_edges","");
 
	for (i=0; i<=maxdepth; i++) {
		li = TSUCC(layer[i]);
		while (li) {
			node = GNNODE(li);
			edge = NSUCC(node);
			while (edge) {
				/* Because check_edge may delete edge */
				assert((SOURCE(edge)==node));
				nextedge = ANEXT(edge);
				check_edge(node,edge,i);
				edge = nextedge;
			}
			li = GNNEXT(li);
		}
	}
}

/*
 *  Check whether one edge is proper
 *  --------------------------------
 *  Source of the edge is node, which is at this level.
 *  Here is the reason why we need maximal maxdepth+1 layers.
 *  The layout of the nodes at level maxdepth inserts into
 *  the layer maxdepth+1 only dummy nodes, if necessary.
 *  Dummy nodes never cause the creation of an other dummy node. 
 *
 *  Upward edges are reverted.
 *  Edges over more than one level are splitted by dummy nodes.
 *  Edges at the same level are tried to neigbour their nodes directly. 
 *  If there are more than 3 such neighbours, direct neighbouring is 
 *  impossible, thus we use dummy nodes (see (1)). Note that direct 
 *  neigbours stay in the layer-list, but are not enymore in any 
 *  adjacency list. Self loops are layoutes with two dummy nodes (see (2)).
 *  Double edges in adjacent layers are layouted as one edge with
 *  two arrows (see (3)).
 *                                                    N
 *       A<--N-->B   C                  N             ^ 
 *            \     ^                  / ^            |
 *    (1)      \   /          (2)     /   \      (3)  |
 *               D                  D1----D2          v
 *                                                    M
 */

static void check_edge(node,edge,level)
GNODE node;
ADJEDGE edge;
int level;
{
	int edgelen;	/* length of the edge, i.e. difference of levels */
	int i,j;
	GNODE   d1, d2; 	/* for dummy nodes */
	ADJEDGE e1,e2,e3;	/* for dummy edges */
	CONNECT	c1,c2;
	int connection_possible;

	debugmessage("check_edge","");
	assert((node));
	assert((edge));
	assert((AKANTE(edge)));
	assert((NTIEFE(node)==level));
	assert((SOURCE(edge)==node));

	edgelen = NTARTIEFE(edge) - level;
	if (edgelen < 0) { /* Revert edge and continue checking */
		e1 = revert_edge(AKANTE(edge));
		/* it may be an forward edge now */
		check_edge(SOURCE(e1),e1,NTIEFE(SOURCE(e1)));
	}
	else if (edgelen == 0) { /* edge at the same level */
		if (EKIND(edge) == 'S') { /* self loop */
			assert((TARGET(edge)==node));
			assert((level<=maxdepth));
			d1 = create_dummy(level+1);
			d2 = create_dummy(level+1);
			NHORDER(d1) = NHORDER(d2) = EHORDER(AKANTE(edge));
			e1 = create_edge(node,d1,AKANTE(edge),0);
			e2 = create_edge(d1,d2,  AKANTE(edge),0);
			e3 = create_edge(d2,node,AKANTE(edge),1);
			/* e1 is already okay: no recursion necessary  */
			check_edge(SOURCE(e2),e2,NTIEFE(SOURCE(e2)));
			check_edge(SOURCE(e3),e3,NTIEFE(SOURCE(e3)));
			delete_adjedge(AKANTE(edge));
		}
		else { /* no self loop: try to neighbour the nodes */
			c1 = NCONNECT(node);
			c2 = NCONNECT(TARGET(edge));
			connection_possible = 1;
			if ((c1) && (CTARGET(c1)) && (CTARGET2(c1))) 
				connection_possible = 0;
			if ((c2) && (CTARGET(c2)) && (CTARGET2(c2))) 
				connection_possible = 0;
			if (check_connect_cycle(TARGET(edge),NULL,
						SOURCE(edge))) 
				connection_possible = 0;
			if (connection_possible) {
				if (!c1) {
					c1 = connectalloc(node);
					CTARGET(c1) = TARGET(edge);
					CEDGE(c1)   = AKANTE(edge);
				}
				else if (!CTARGET2(c1)) {
					CTARGET2(c1) = TARGET(edge);
					CEDGE2(c1)   = AKANTE(edge);
				}
				if (!c2) {
					c2 = connectalloc(TARGET(edge));
					CTARGET(c2) = node;
					CEDGE(c2)   = AKANTE(edge);
				}
				else if (!CTARGET2(c2)) {
					CTARGET2(c2) = node;
					CEDGE2(c2)   = AKANTE(edge);
				}
				delete_adjedge(AKANTE(edge));
				EINVISIBLE(AKANTE(edge)) = 0;
			}
			else {
				if (level<=maxdepth) 
					d1 = create_dummy(level+1);
				else if (level>0) 
					d1 = create_dummy(level-1);
				
				else { assert((0)); }
				NHORDER(d1) = EHORDER(AKANTE(edge));
				e1 = create_edge(node,d1,AKANTE(edge),0);
				e2 = create_edge(d1,TARGET(edge),AKANTE(edge),1);
				check_edge(SOURCE(e1),e1,NTIEFE(SOURCE(e1)));
				check_edge(SOURCE(e2),e2,NTIEFE(SOURCE(e2)));
				delete_adjedge(AKANTE(edge));
			}
		}
	}
	else if (edgelen == 1) { /* edge at adjacent level */
		if ((EKIND(edge)=='R') && double_edge(edge)) 
			delete_adjedge(AKANTE(edge));
		/* SUCH NODES ARE OKAY */
	}
	else if (edgelen > 1) { /* Forward edges etc. */
		d1 = node;
		if (EKIND(edge)=='R') j = 1; else j = 0;
		for (i=1; i<edgelen; i++) {
			d2 = create_dummy(level+i);
			NHORDER(d2) = EHORDER(AKANTE(edge));
			e1 = create_edge(d1,d2,AKANTE(edge),j);
			/* e1 is already okay: no recursion necessary  */
			d1 = d2;
			j = 0;
		}
		if (EKIND(edge)!='R') j = 1;
		e1 = create_edge(d1,TARGET(edge),AKANTE(edge),j);
		/* e1 is already okay: no recursion necessary  */
		delete_adjedge(AKANTE(edge));
	}
}


/*  Check doublicated edges
 *  -----------------------
 *  check whether edge is double in the adjacency list.
 *  Important are only the doublicated edges from one level
 *  to the next level, to recognize short biconnected edges.
 *  The argument is an edge from level i to level i+1 that
 *  is an reverted edge, i.e. the original edge goes from
 *  level i+1 to i.
 *  We are looking for the corresponding unreverted edge from level i
 *  to level i+1. This edge is marked as 'D' or is dealt to be
 *  not a double edge. 
 *  Returns 1 in this case.
 */

static int	double_edge(edge)
ADJEDGE	edge;
{
	ADJEDGE	l;
	GNODE   d1; 	/* for dummy nodes */
	ADJEDGE e1;	/* for dummy edges */

	debugmessage("double_edge","");
	assert((EKIND(edge)=='R'));
	l = NSUCC(SOURCE(edge));
	while (l) {
		if ((EKIND(l)!='R') && (TARGET(l)==TARGET(edge)) ) {
			if (EARROWSIZE(AKANTE(l))==0) {
				/* it may be an label. Note: edge is reverted*/
				d1 = create_dummy(NTIEFE(TARGET(l)));
				NHORDER(d1) = EHORDER(AKANTE(edge));
				e1 = create_edge(d1,SOURCE(edge),AKANTE(edge),0);
				EARROWSIZE(AKANTE(e1))
					= EARROWSIZE(AKANTE(edge));
				check_edge(SOURCE(e1),e1,NTIEFE(SOURCE(e1)));
				e1 = create_edge(TARGET(edge),d1,AKANTE(edge),0);
				check_edge(SOURCE(e1),e1,NTIEFE(SOURCE(e1)));
				/* The double edge is not anymore a 
				 * double edge. Thus the original edge
				 * can be removed.
				 */
				return(1);
			}
			EKIND(l) = 'D';	   /* double_edge recognized */
			return(1);
		}
		l = ANEXT(l);
	}
	return(0);
}


/*  Revert an edge
 *  --------------
 *  The edge is reused again and stays visible. But source and target 
 *  are exchanged, and the edge is marked by 'R'. The adjacency entry
 *  of the new source node is returned.
 */
 
ADJEDGE revert_edge(edge)
GEDGE edge;
{
	GNODE h;

	debugmessage("revert_edge","");
	delete_adjedge(edge);
	h = ESTART(edge); 
	ESTART(edge) = EEND(edge);
	EEND(edge) = h;
	EART(edge) = 'R';
	create_adjedge(edge);

	/* Warning: We know from alloc.c that this is the edge we just
 	 * created. But it is not quite obvious.
	 */

	return(NSUCC(ESTART(edge)));
}


/*  Create an additional edge
 *  -------------------------
 *  between the nodes `start' and `end'. The attributes of this
 *  new edge are derive from 'edge'.
 *  The attribute ARROWSIZE is only inherited if `arrow' is 1.
 */

static ADJEDGE	create_edge(start, end, edge,arrow)
GNODE	start;
GNODE  	end;
GEDGE   edge;
int	arrow;
{
	GEDGE	h;
	
	h = tmpedgealloc(ELSTYLE(edge),
			 ETHICKNESS(edge),
			 ECLASS(edge),
			 EPRIO(edge),
			 ECOLOR(edge),
			 EARROWSIZE(edge),
			 EHORDER(edge));
	if (!arrow) EARROWSIZE(h) = 0;
	EANCHOR(h) = EANCHOR(edge);
	ESTART(h) = start;
	EEND(h)	  = end;
	if (!arrow) ELABEL(h) = NULL;
	else	    ELABEL(h) = ELABEL(edge);
	if (EART(edge)=='S') EART(h) = 'U';
	else		     EART(h) = EART(edge);
	create_adjedge(h);

	/* Warning: We know from alloc.c that this is the edge we just
 	 * created. But it is not quite obvious.
	 */

	return(NSUCC(ESTART(h)));
}


/*  Create a dummy node
 *  -------------------
 *  returns a new dummy node that is inserted at layer[t].
 *  A dummy node is always a temporary node.
 */

static GNODE create_dummy(t)
int t;
{
	GNLIST hl;
	GNODE  v;

	debugmessage("create_dummy","");
	assert((t<=maxdepth+1));

	v = tmpnodealloc(CENTER,-1,-1,0,-1,G_color,G_color,G_color,1,1,-1);
	NTITLE(v)   = "";
	NLABEL(v)   = "";
	NTIEFE(v)   = t;
	NHORDER(v)  = -1;
	NBEFORE(v)      = NULL;
	NNEXT(v)        = dummylist;
	if (dummylist) NBEFORE(dummylist) = v;
	dummylist       = v;
	dummyanz++;

	/* if t appropriate, insert dummy node into the layer[t] */
	if (t<0) return(v);
	hl = tmpnodelist_alloc();
	GNNEXT(hl) = TSUCC(layer[t]);
	TSUCC(layer[t]) = hl;
       	GNNODE(hl) = v;
	return(v);
}



/*--------------------------------------------------------------------*/
/*  Debugging routines                                                */
/*--------------------------------------------------------------------*/

/*  The following routines are normally not used, but are provided
 *  to simplify debugging later on.
 */

/*  Output the graph textually
 *  ==========================
 */

#ifdef DEBUG

void db_output_graph()
{
	PRINTF("Nodes:\n");
	db_output_nodes(nodelist);
	PRINTF("Labels:\n");
	db_output_nodes(labellist);
	PRINTF("Dummynodes:\n");
	db_output_nodes(dummylist);
	PRINTF("\n\n");
	PRINTF("Edges:\n");
	db_output_edges(edgelist); 
	PRINTF("Temporary edges:\n");
	db_output_edges(tmpedgelist); 
}
 
#endif


/*  Output the nodes 
 *  ----------------
 */

#ifdef DEBUG

static void db_output_nodes(n)
GNODE   n;
{
	if (!n) return; 
	db_output_nodes(NNEXT(n));
	PRINTF("%s, ", NTITLE(n));
}

#endif


/*  Output the edges 
 *  ----------------
 */

#ifdef DEBUG

static void db_output_edges(e)
GEDGE   e;
{
 	if (!e) return; 
	db_output_edges(ENEXT(e));
	if (EINVISIBLE(e)==0)
		PRINTF("(%s -> %s)\n", ESTART(e)->title, EEND(e)->title);
}

#endif


/*  Output the graph in adjacency representation
 *  ============================================
 */
         
#ifdef DEBUG

void db_output_adjacencies()
{
	GNODE	node;
	ADJEDGE	edge;

	PRINTF("\n\nAdjacency lists: ");
	node = nodelist;
	while (node) {
		PRINTF("\n%s(%d)\n", NTITLE(node), NTIEFE(node));
		PRINTF("\n(in:%d,out:%d)\n", NINDEG(node), NOUTDEG(node));
		PRINTF("Succs:");
		edge = NSUCC(node);
		while (edge) {
			PRINTF("|%s ", NTITLE(TARGET(edge)));
			edge = ANEXT(edge);
		}
		PRINTF("\nPreds:");
		edge = NPRED(node);
		while (edge) {
			PRINTF("|%s ", NTITLE(SOURCE(edge))); 
            		edge = ANEXT(edge); 
        	}
		node = NNEXT(node);
	}
	node = labellist;
	while (node) {
		PRINTF("\n%s(%d)\n", NTITLE(node), NTIEFE(node));
		PRINTF("\n(in:%d,out:%d)\n", NINDEG(node), NOUTDEG(node));
		PRINTF("Succs:");
		edge = NSUCC(node);
		while (edge) {
			PRINTF("|%s ", NTITLE(TARGET(edge)));
			edge = ANEXT(edge);
		}
		PRINTF("\nPreds:");
		edge = NPRED(node);
		while (edge) {
			PRINTF("|%s ", NTITLE(SOURCE(edge))); 
            		edge = ANEXT(edge); 
        	}
		node = NNEXT(node);
	}
	node = dummylist;
	while (node) {
		PRINTF("\n%s(%d)\n", NTITLE(node), NTIEFE(node));
		PRINTF("\n(in:%d,out:%d)\n", NINDEG(node), NOUTDEG(node));
		PRINTF("Succs:");
		edge = NSUCC(node);
		while (edge) {
			PRINTF("|%s ", NTITLE(TARGET(edge)));
			edge = ANEXT(edge);
		}
		PRINTF("\nPreds:");
		edge = NPRED(node);
		while (edge) {
			PRINTF("|%s ", NTITLE(SOURCE(edge))); 
            		edge = ANEXT(edge); 
        	}
		node = NNEXT(node);
	}
	PRINTF("\n");
}

#endif

/*  Output one node in adjacency representation
 *  ===========================================
 *  f = 0 => only succs, f = 1 => only preds
 */
         
#ifdef DEBUG

#define gtitle(v)  (NTITLE(v)?NTITLE(v):"??")

void db_output_adjacency(node,f)
GNODE node;
int f;
{
	ADJEDGE	edge;

	PRINTF("\n\nAdjacency lists: ");
	PRINTF("\n%ld %s(%d)\n", node, gtitle(node), NTIEFE(node));
	if (f!=1) {
	PRINTF("Succs:");
	edge = NSUCC(node);
	while (edge) {
		PRINTF("%ld %c:",edge,EKIND(edge));
		PRINTF("(%ld)%s-", SOURCE(edge),gtitle(SOURCE(edge)));
		PRINTF("(%ld)%s ", TARGET(edge),gtitle(TARGET(edge)));
		edge = ANEXT(edge);
	}
	}
	if (f!=0) {
	PRINTF("\nPreds:");
	edge = NPRED(node);
	while (edge) {
		PRINTF("%ld %c:",edge,EKIND(edge));
		PRINTF("(%ld)%s-", SOURCE(edge),gtitle(SOURCE(edge))); 
		PRINTF("(%ld)%s ", TARGET(edge),gtitle(TARGET(edge)));
            	edge = ANEXT(edge); 
        }
	node = NNEXT(node);
	PRINTF("End\n");
	}
}
#endif

/*--------------------------------------------------------------------*/


