/* SCCS-info %W% %E% */

/*--------------------------------------------------------------------*/
/*                                                                    */
/*              VCG : Visualization of Compiler Graphs                */
/*              --------------------------------------                */
/*                                                                    */
/*   file:         step0.c                                            */
/*   version:      1.00.00                                            */
/*   creation:     10.4.1993                                          */
/*   author:       I. Lemke  (...-Version 0.99.99)                    */
/*                 G. Sander (Version 1.00.00-...)                    */  
/*                 Universitaet des Saarlandes, 66041 Saarbruecken    */
/*                 ESPRIT Project #5399 Compare                       */
/*   description:  analysis phase of syntax trees                     */
/*   status:       in work                                            */
/*                                                                    */
/*--------------------------------------------------------------------*/


#ifndef lint
static char *id_string="$Id: step0.c,v 3.15 1994/08/08 16:01:47 sander Exp $";
#endif


/*
 *   Copyright (C) 1993, 1994 by Georg Sander, Iris Lemke, and
 *                               the Compare Consortium 
 *
 *  This program and documentation is free software; you can redistribute 
 *  it under the terms of the  GNU General Public License as published by
 *  the  Free Software Foundation;  either version 2  of the License,  or
 *  (at your option) any later version.
 *
 *  This  program  is  distributed  in  the hope that it will be useful,
 *  but  WITHOUT ANY WARRANTY;  without  even  the  implied  warranty of
 *  MERCHANTABILITY  or  FITNESS  FOR  A  PARTICULAR  PURPOSE.  See  the
 *  GNU General Public License for more details.
 *
 *  You  should  have  received a copy of the GNU General Public License
 *  along  with  this  program;  if  not,  write  to  the  Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  The software is available per anonymous ftp at ftp.cs.uni-sb.de.
 *  Contact  sander@cs.uni-sb.de  for additional information.
 */


/*
 * $Log: step0.c,v $
 * Revision 3.15  1994/08/08  16:01:47  sander
 * Attributes xraster, xlraster, yraster added.
 *
 * Revision 3.14  1994/08/05  14:27:31  sander
 * Negative values of G_width, etc. corrected.
 *
 * Revision 3.13  1994/08/05  12:13:25  sander
 * Treelayout added. Attributes "treefactor" and "spreadlevel" added.
 * Scaling as abbreviation of "stretch/shrink" added.
 *
 * Revision 3.12  1994/08/04  08:58:20  sander
 * Attributes crossing_weight and crossing_optimization added.
 *
 * Revision 3.11  1994/08/03  13:58:44  sander
 * Horizontal order mechanism changed.
 * Attribute horizontal_order for edges added.
 *
 * Revision 3.10  1994/08/02  15:36:12  sander
 * CHECKNODE option added to allow tracing of properties
 * of one single node. For debugging only.
 *
 * Revision 3.9  1994/06/07  14:09:59  sander
 * Splines implemented.
 * HP-UX, Linux, AIX, Sun-Os, IRIX compatibility tested.
 * The tool is now ready to be distributed.
 *
 * Revision 3.8  1994/05/17  16:37:33  sander
 * attribute node_align added to allow nodes to be centered in the levels.
 *
 * Revision 3.7  1994/05/16  08:56:03  sander
 * shape attribute (boxes, rhombs, ellipses, triangles) added.
 *
 * Revision 3.6  1994/05/05  08:20:30  sander
 * Potential infinite number of edge classes are allowed.
 * Hashtable curser management added. This is needed by
 * the Center title dialog box.
 *
 * Revision 3.5  1994/04/27  16:05:19  sander
 * Some general changes for the PostScript driver.
 * Horizontal order added. Bug fixes of the folding phases:
 * Folding of nested graphs works now.
 *
 * Revision 3.4  1994/03/04  19:11:24  sander
 * Specification of levels per node added.
 * X11 geometry behaviour (option -geometry) changed such
 * that the window is now opened automatically.
 *
 * Revision 3.3  1994/03/03  15:35:39  sander
 * Edge line style `invisible' added.
 *
 * Revision 3.2  1994/03/02  11:48:54  sander
 * Layoutalgoritms mindepthslow, maxdepthslow, minindegree, ... mandegree
 * added.
 * Anchors and nearedges are not anymore allowed to be intermixed.
 * Escapes in strings are now allowed.
 *
 * Revision 3.1  1994/03/01  10:59:55  sander
 * Copyright and Gnu Licence message added.
 * Problem with "nearedges: no" and "selfloops" solved.
 *
 * Revision 2.5  1994/02/13  17:26:05  sander
 * attribute 'finetuning' added.
 *
 * Revision 2.4  1994/01/21  19:33:46  sander
 * VCG Version tested on Silicon Graphics IRIX, IBM R6000 AIX and Sun 3/60.
 * Option handling improved. Option -grabinputfocus installed.
 * X11 Font selection scheme implemented. The user can now select a font
 * during installation.
 * Sun K&R C (a nonansi compiler) tested. Some portabitility problems solved.
 *
 * Revision 2.3  1994/01/10  09:22:52  sander
 * Distribution version 1.
 *
 */

/************************************************************************
 * The situation here is the following:
 * -----------------------------------
 * We have already parsed the specification and constructed a syntax tree.
 * The task of this module is to fill nodelist with the exisiting nodes, 
 * graphlist with the existing graph summary nodes and edgelist with the 
 * existing edges.
 * We assume that nodelist, edgelist and graphlist are empty at this point.
 * For the connection structure between graph summary nodes and normal nodes,
 * see alloc.h.
 * We need NOT to create the adjacency lists of the nodes and edges in
 * this module, because this is done later, see folding.c.
 * However we have to prepare folding by recognizing the folding and 
 * invisible attributes. This is done by the following functions/structs: 
 *    - add_foldstart:   remark node as start node of a fold region operation 
 *    - add_foldstop:    remark node as stop  node of a fold region operation
 *    - add_sgfoldstart: remark node as summary node of folded subgraph
 *    - hide_class:      array. Component i is 1, if edge class i+1 is hidden.
 *    - clear_hide_class: initialization of this arry.
 * See folding.c for more information about these functions. We call these
 * functions and structs that are used to recognize which must be folded
 * the `folding keepers'. The real folding according to our settings of the
 * folding keepers is done there.
 * Note that the nodes and edges allocated here are stable, i.e. are not
 * reallocated if the layout of the graph changes. To have fast access to
 * these stable nodes, we use a hash table.
 *
 * The task is done by step0_main. After that, the syntax tree is not
 * anymore necessary, but note that the strings in the syntax tree may
 * be reused here.
 *
 * This file provides the following functions:
 * ------------------------------------------
 * step0_main		main routine of step0: walk through the syntax 
 *			tree, collect attributes and creates the internal
 *                      representation of the graph to be visualized.
 * lookup_hashnode      returns a node of a given title. The node must
 *                      be derived directly from the specification, i.e. 
 *                      no dummy node. The node need not to be visible.
 * search_visible_node  returns a node of a given title, but only if it
 *			is visible. Similar to lookup_hashnode.
 ***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "globals.h"
#include "grammar.h"
#include "alloc.h"
#include "main.h"
#include "folding.h"
#include "steps.h"
#include "timing.h"


/*  Defines
 *  -------
 *  Faster access to the syntax tree.
 */

/* Decode a identifier or string node */

#define SDecode(x)  Decode(get_lnum(x))

/* Standard access to nums */
 
#define FETCHNUM()   get_num(son1(node2))
#define FETCHFLOAT() get_lrealnum(son1(node2))


/* Prototypes
 * ----------
 */

static void	node_analyse	_PP((yysyntaxtree no, GNODE ro, char *gt, char *la, int te, int wi, int he, int bo, int xp, int yp, int fo, int  co, int tc, int bd, int sh, int st, char *i1, char *i2, char *i3, int lev, int shap, int hord));
static void	edge_analyse	_PP((yysyntaxtree no, char *el, int ls, int th, int cl, int ep, int ho, int ec, int ar));

static void   node_attributes	_PP((yysyntaxtree n, GNODE v,int r,int s));
static void   edge_attributes	_PP((yysyntaxtree n, GEDGE e));
static void   graph_attributes 	_PP((yysyntaxtree n, GNODE v, char *d,int r,int s));
static int    get_color		_PP((yysyntaxtree node));
static void   calc_nr_classes	_PP((yysyntaxtree x));
static GNODE  search_node	_PP((yysyntaxtree x,char *s));
static void   check_node	_PP((yysyntaxtree x,GNODE m));
static void   init_hashtable	_PP((void));
static int    hashval		_PP((char *s));
static void   insert_hashnode	_PP((GNODE x));


#ifdef CHECKNODE
static void debug_init_checknode _PP((void));
#else
#define debug_init_checknode()  /**/
#endif

/*--------------------------------------------------------------------*/
/* Analysis of syntax trees                                           */
/*--------------------------------------------------------------------*/

/* Local flags 
 * -----------
 */

static int subg_bit;	/* Nesting level of subgraphs: 0 means: top level */

/*  Driver of tree analysis
 *  =======================
 */

void	step0_main()
{
	start_time();
	debugmessage("step0_main","");
	
	assert((nodelist==NULL));
	assert((graphlist==NULL));
	assert((edgelist==NULL));

	max_nr_classes = 1;
	calc_nr_classes(Syntax_Tree);

	if (max_nr_classes>128) {
		FPRINTF(stderr,"\nWarning: you use %d edge classes !\n", 
			max_nr_classes);
		FPRINTF(stderr,"This may produce a segmentation fault on some systems.\n"); 
		FPRINTF(stderr,"Please do not use more than 128 classes.\n"); 
	}

	if (class_names) free(class_names);
	class_names = (char **)malloc(max_nr_classes * sizeof(char *));
	if (hide_class) free(hide_class);
	hide_class = (int *)malloc(max_nr_classes * sizeof(int));


	subg_bit    = 0;	/* We are outside the top graph       */

	/* Initialize folding attributes of nodes */

        fold_label = NULL;      /* folded node label            */ 
        fold_textm  = -1;       /* folded node textmode         */ 
        fold_width  = -1;       /* folded node width            */ 
        fold_height = -1;       /* folded node height           */
        fold_borderw = 4;       /* folded node border width     */
        fold_xpos    = 0;       /* folded node x position       */ 
        fold_ypos    = 0;       /* folded node y position       */ 
        fold_color   = -1;      /* folded node background color */ 
        fold_textc   = -1;      /* folded node text color       */
        fold_borderc = -1;      /* folded node border color     */ 
        fold_shrink  = -1;      /* folded node shrink factor    */
        fold_stretch = -1;      /* folded node stretch factor   */ 
        fold_info1  = NULL;     /* folded node info field 1     */ 
        fold_info2  = NULL;     /* folded node info field 2     */ 
        fold_info3  = NULL;     /* folded node info field 3     */ 
	fold_level  = -1;       /* folded node level            */
	fold_shape  = -1;       /* folded node shape            */
	fold_horder = -1;       /* folded node horizontal order */

	/* Initialize folding attributes of edges */

        fold_elabel = "...";      /* folded edge label          */
        fold_lstyle =-1;          /* folded edge line style     */
        fold_thick  = 4;          /* folded edge thickness      */
        fold_ecolor =-1;          /* folded edge color          */
        fold_arrows =-1;          /* folded edge arrowsize      */
 
	/* Initialize the folding keepers */

	clear_hide_class();
	clear_folding_keepers();

	/* Initialize the hash table of stable nodes */

	init_hashtable();

	/* Analyze the syntax tree: the values given here are the
	 * default values.
         */

        node_analyse(
		  Syntax_Tree,	/* syntax tree node       */
		  NULL,		/* root node              */
		  G_title,	/* graph title		  */ 
		  NULL,		/* node label		  */
		  CENTER,	/* node textmode          */
		  -1,		/* node width             */
		  -1,		/* node height            */
		  2,		/* node border width      */
		  0,		/* node x pos             */
		  0,		/* node y pos             */
		  -1,		/* node folding class     */
		  G_color,	/* node background color  */
		  BLACK,	/* node text color        */
		  BLACK,	/* node border color      */
		  1,		/* node shrink factor     */
		  1,		/* node stretch factor    */
		  "",		/* node info field 1      */
		  "",		/* node info field 2      */
		  "",		/* node info field 3      */
		  -1,		/* node level		  */
		  0,		/* node shape		  */
		  -1);		/* node horizontal order  */

        edge_analyse(
		  Syntax_Tree,	/* syntax tree node       */
		  NULL,		/* edge label             */
		  SOLID,	/* edge line style        */
		  2,		/* edge thickness         */
		  1,		/* edge class             */
		  1,		/* edge priority          */
		  -1,		/* edge horizontal order  */
		  BLACK,	/* edge color             */
		  10);		/* edge arrowsize         */

	debug_init_checknode();

	stop_time("step0_main without folding");

}




/*--------------------------------------------------------------------*/

/*  Node analysis per subgraph
 *  ==========================
 */

/*  The function node_analyse traverses the syntax tree starting at `node', 
 *  collects the information and allocates the appropriate data structures.
 *  node is assumed to be a T_Co_graph_entry representing a graph.
 *  The situation is always the following:
 *      all simple  nodes are collected into the node list
 *      all summary nodes are collected into the graph list
 *
 *  Inside the function, we build NSGRAPH(root).
 *      `root'        points to the actual summary node of the subgraph
 *      NSGRAPH(root) points to the actual list of nodes of this subgraph
 *      rootend       points to the end of this list
 *
 *  The arguments gtitle ... level are the default values that are 
 *  actually valid for the graph, node or edge.
 */


/*  Abbreviation to add v into the nodelist NSGRAPH(root) */

#define add_to_nodelist_of_root(v)  \
	{ if ( subg_bit > 0 ) {                     \
		NROOT(v)	= root;             \
		l = nodelist_alloc(v);              \
		if (rootend) 	GNNEXT(rootend) = l;\
		else 		NSGRAPH(root) = l;  \
		rootend = l;                        \
	}}



static void	node_analyse(node,root,gtitle,label,textm,
	          width,height,borderw,xpos,ypos,fold,color,textc,borderc,
	          shrink,stretch,info1,info2,info3,level,shape,horder)
	yysyntaxtree node;	/* syntax tree node                   */
	GNODE	root;		/* graph summary node                 */
	char    *gtitle;	/* graph title                        */
	char	*label; 	/* node label			      */
	int     textm;		/* node textmode: CENTER, LEFT, RIGHT */
	int	width;		/* node width                         */
	int 	height;		/* node height                        */
	int	borderw;	/* node border width		      */
	int	xpos;		/* node x position		      */
	int	ypos;		/* node y position		      */
	int	fold;		/* node folding class                 */
	int 	color;		/* node background color              */
	int	textc;		/* node text color                    */
	int	borderc;	/* node border color                  */
	int	shrink;		/* node shrink factor                 */
	int     stretch;	/* node stretch factor                */
	char 	*info1;		/* node info field 1 		      */
	char 	*info2;		/* node info field 2 		      */
	char 	*info3;		/* node info field 3 		      */
	int	level;		/* node level			      */
	int 	shape;		/* node shape			      */
	int 	horder;		/* folded node horizontal order       */
{
	register yysyntaxtree	node1, node2;	/* auxiliary variables */
	GNODE	v;
	GNLIST  l;
	int 	invis;
	GNLIST  rootend;	/* end of the actual node list */
	int     rootshrink;	/* shrink factor of the root   */
	int 	rootstretch;	/* stretch factor of the root  */
	int     h;

	debugmessage("node_analyse","");

	gs_wait_message('a');

	/* Initialize shrink and stretch factors of the root node */

	if (root) {
		rootshrink  = (NSHRINK(root)>0  ? NSHRINK(root) : 1);
		rootstretch = (NSTRETCH(root)>0 ? NSTRETCH(root): 1);
	}
	else    rootshrink = rootstretch = 1;

	/* traverse iteratively the syntax tree: 
 	 * on graph entries, we create a graph summary node, fetch the
	 * 	attributes valid for the summary node and we go into recursion.
	 * on node entries, we create a new node and fetch its attributes.
	 * on graph attributes, we fetch the attributes valid as default 
	 *	values of the whole graph and set the graph attributes 
	 *	G_x, G_y, etc.
 	 * on node default attributes, we change the default variables
	 *	label, ..., stretch.
	 *
	 * Nodes come into the hashtable after fetching all its attributes.
	 * See the corresponding fetch routines.
	 */

	rootend = NULL;
	assert((node && (tag(node) == T_Co_graph_entry)));
	while ( node && (tag(node) == T_Co_graph_entry) ) {
		node1 = son1(node);
		assert(node1);
	        switch (tag(node1)) {
		case T_Co_graph:
			v = graphalloc(label,textm,width,height,borderw,xpos,
				  ypos,-1,color,textc,borderc,shrink,stretch,
				  info1,info2,info3,level,shape,horder);

			add_to_nodelist_of_root(v);

			graph_attributes(son1(node1),v,gtitle,
					 rootshrink,rootstretch);
			subg_bit++;
			node_analyse(son1(node1),v,NTITLE(v),label,textm,
	          		width,height,borderw,xpos,ypos,fold,color,
				textc,borderc,shrink,stretch,info1,info2,
				info3,level,shape,horder);
			subg_bit--;
			break;

		case T_Co_graph_attribute:

			/* This is only used for the top level graph */

			node2 = son1(node1);
			assert((node2));
			switch(tag(node2)) {
                	case T_Co_title:
				if (!subg_bit) G_title = SDecode(son1(node2));
				gtitle = SDecode(son1(node2));
                        	break;
                	case T_Co_label:
                	case T_Co_info1:
                	case T_Co_info2:
                	case T_Co_info3:
                	case T_Co_level:
                	case T_Co_shape:
                	case T_Co_horizontal_order:
                	case T_Co_loc  :
                	case T_Co_textmode:
                	case T_Co_borderwidth:
                	case T_Co_textcolor:
                	case T_Co_bordercolor:
                        	break;
                	case T_Co_xdef:
                        	if (!subg_bit)	G_x = FETCHNUM();
                        	break;
                 	case T_Co_ydef:
                        	if (!subg_bit)  G_y = FETCHNUM();	
                        	break;
                	case T_Co_width:
                        	if (!subg_bit)	{ 	
					G_width  = FETCHNUM();
					if (G_width <= 0) G_width = 100;
					G_width_set = 1;
				}
                        	break;
                	case T_Co_height:
                        	if (!subg_bit) {
					G_height = FETCHNUM();
					if (G_height <= 0) G_height = 100;
					G_height_set = 1;
				}
                        	break;
			case T_Co_xmax:
				if (!subg_bit) {
					if (!G_xymax_final)
						G_xmax = FETCHNUM();
					if (G_xmax <= 0) G_xmax = 100;
					if (G_xmax > (2*ScreenWidth))
						G_xmax = 2*ScreenWidth;
					if (G_width>G_xmax) {
						G_width = G_xmax;
						G_width_set = 1;
					}
				}
			 	break;
			case T_Co_ymax:
				if (!subg_bit) {
					if (!G_xymax_final)
						G_ymax = FETCHNUM();
					if (G_ymax <= 0) G_ymax = 100;
					if (G_ymax > (2*ScreenHeight))
						G_ymax = 2*ScreenHeight;
					if (G_height>G_ymax) {
						G_height = G_ymax;
						G_height_set = 1;
					}
				}
				break;
			case T_Co_infoname:
				if (!subg_bit) {
					h = FETCHNUM();
					if ((h>0)&&(h<=3)) {
						info_name_available = 1;
						info_names[h-1] = 
							SDecode(son2(node2));
					}
				}
                        	break;
			case T_Co_classname:
				if (!subg_bit) {
					h = FETCHNUM();
					if ((h>0)&&(h<=max_nr_classes)) {
						class_name_available = 1;
						class_names[h-1] = 
							SDecode(son2(node2));
					}
				}
                        	break;
                	case T_Co_xbase:
                        	if (!subg_bit)	G_xbase = FETCHNUM();
                        	break;
                	case T_Co_ybase:
                        	if (!subg_bit)	G_ybase = FETCHNUM();
                        	break;
                	case T_Co_xspace:
                        	if (!subg_bit)	G_xspace= FETCHNUM();
                        	break;
                	case T_Co_xlspace:
                        	if (!subg_bit)	G_dspace= FETCHNUM();
                        	break;
                	case T_Co_yspace:
                        	if (!subg_bit)	G_yspace= FETCHNUM();
                        	break;
                	case T_Co_xraster:
                        	if (!subg_bit)	G_xraster= FETCHNUM();
                        	break;
                	case T_Co_xlraster:
                        	if (!subg_bit)	G_dxraster= FETCHNUM();
                        	break;
                	case T_Co_yraster:
                        	if (!subg_bit)	G_yraster= FETCHNUM();
                        	break;
			case T_Co_splinefactor:
                        	if (!subg_bit)	G_flat_factor= FETCHNUM();
                        	break;
			case T_Co_downfactor:
                        	if (!subg_bit)	layout_downfactor= FETCHNUM();
                        	break;
			case T_Co_upfactor:
                        	if (!subg_bit)	layout_upfactor= FETCHNUM();
                        	break;
			case T_Co_nearfactor:
                        	if (!subg_bit)	layout_nearfactor= FETCHNUM();
                        	break;
                	case T_Co_node_alignment:
				if (!subg_bit) {
                        		switch(tag(son1(node2))) {
                               	    	case T_Co_top:
						G_yalign = AL_TOP;
						break;
                               	    	case T_Co_bottom:
						G_yalign = AL_BOTTOM;
						break;
                               	    	case T_Co_center:
						G_yalign = AL_CENTER;
						break;
                               	    	default:
						assert((0));
					}
                        	}
                        	break;
                	case T_Co_orientation:
				if (!subg_bit) {
                        		switch(tag(son1(node2))) {
                               	    	case T_Co_top_to_bottom:
                                       		G_orientation = TOP_TO_BOTTOM;
                                       		break;
                               	    	case T_Co_left_to_right:
                                       		G_orientation = LEFT_TO_RIGHT;
                                       		break;
                               	    	case T_Co_right_to_left:
                                       		G_orientation = RIGHT_TO_LEFT;
                                       		break;
                               	    	case T_Co_bottom_to_top:
                                       		G_orientation = BOTTOM_TO_TOP;
                                       		break;
                               	    	default:
						assert((0));
					}
                        	}
                        	break;
			case T_Co_color:
				if (!subg_bit) 	
					G_color = get_color(son1(node2));
				break;
			case T_Co_folding:
				if (!subg_bit){
					SYERR(node2,
				"attribute `folding' at top level graph");
				}
				break;
			case T_Co_status:
				if (!subg_bit){
					SYERR(node2,
				"attribute `status' at top level graph");
				}
				break;
			case T_Co_hidden:
				invis = FETCHNUM()-1;
				if (!subg_bit) {
					if ((0<=invis) && (invis<max_nr_classes))
                                        	hide_class[invis] = 1;
					else SYERR(node2,
					 "attribute `hidden' out of range (1-max_nr_classes)");
				}
				break;
                	case T_Co_late_edge_label:
				if (!subg_bit) {
                        		switch(tag(son1(node2))) {
                               	    	case T_Co_yes:
                                       		edge_label_phase = 1; 
						break;
				    	case T_Co_no:
                                       		edge_label_phase = 0; 
                                       		break;
                               	    	default:
						assert((0));
                        		}
				}
                        	break;
                	case T_Co_display_edge_label:
				if (!subg_bit) {
                        		switch(tag(son1(node2))) {
                               	    	case T_Co_yes:
                                       		G_displayel = YES; 
						break;
				    	case T_Co_no:
                                       		G_displayel = NO;
                                       		break;
                               	    	default:
						assert((0));
                        		}
				}
                        	break;
                	case T_Co_dirty_edge_label:
				if (!subg_bit) {
                        		switch(tag(son1(node2))) {
                               	    	case T_Co_yes:
                                       		G_dirtyel = YES; 
						break;
				    	case T_Co_no:
                                       		G_dirtyel = NO;
                                       		break;
                               	    	default:
						assert((0));
                        		}
					if (G_dirtyel==YES) G_displayel = YES;
				}
                        	break;
                	case T_Co_finetuning:
				if (!subg_bit) {
                        		switch(tag(son1(node2))) {
                               	    	case T_Co_yes:
                                       		fine_tune_layout = 1;
						break;
				    	case T_Co_no:
                                       		fine_tune_layout = 0;
                                       		break;
                               	    	default:
						assert((0));
                        		}
				}
                        	break;
                	case T_Co_crossing_opt:
				if (!subg_bit) {
                        		switch(tag(son1(node2))) {
                               	    	case T_Co_yes:
                                       		local_unwind = 1;
						break;
				    	case T_Co_no:
                                       		local_unwind = 0;
                                       		break;
                               	    	default:
						assert((0));
                        		}
				}
                        	break;
                	case T_Co_crossing_weight:
				if (!subg_bit) {
                        		switch(tag(son1(node2))) {
                               	    	case T_Co_bary:
                                       		crossing_heuristics = 0;
						break;
				    	case T_Co_median:
                                       		crossing_heuristics = 1;
                                       		break;
                               	    	default:
						assert((0));
                        		}
				}
                        	break;
                	case T_Co_splines:
				if (!subg_bit) {
                        		switch(tag(son1(node2))) {
                               	    	case T_Co_yes:
                                       		G_spline = 1;
						break;
				    	case T_Co_no:
                                       		G_spline = 0;
                                       		break;
                               	    	default:
						assert((0));
                        		}
				}
                        	break;
			case T_Co_dummy  :
				break;
			case T_Co_nonearedges  :
				if (!subg_bit) {
					near_edge_layout = 0;
				}
				break;
                	case T_Co_scaling:
				if (!subg_bit) {
					double hhx;

                        		hhx = FETCHFLOAT();
					if (hhx<0) hhx=-hhx;
					if (hhx>1.0) {
						G_stretch = 100;
						G_shrink = (int)(100.0/hhx);
					}
					else {
						G_stretch = (int)(hhx*100.0);
						G_shrink = 100;
					}
					G_shrink = rootshrink*G_shrink;
					G_stretch= rootstretch*G_stretch;
					if (G_shrink == 0) G_shrink = 1;
					if (G_stretch == 0) G_stretch = 1;
				}
                        	break;
                	case T_Co_shrink:
				if (!subg_bit) {
                        		G_shrink = rootshrink * FETCHNUM();
					if (G_shrink <= 0) G_shrink = 1;
				}
                        	break;
                	case T_Co_stretch:
				if (!subg_bit) {
					G_stretch = rootstretch * FETCHNUM();
					if (G_stretch <= 0) G_stretch = 1;
				}
                        	break;
                	case T_Co_spreadlevel:
				if (!subg_bit) {
					spread_level = FETCHNUM();
				}
                        	break;
                	case T_Co_treefactor:
				if (!subg_bit) {
					tree_factor = FETCHFLOAT();
				}
                        	break;
			case T_Co_layoutalgorithm:
				if (!subg_bit) {
					assert((son1(node2)));
					switch (tag(son1(node2))) {
					case T_Co_maxdepth :
						layout_flag = 1;
						break;
					case T_Co_mindepth :
						layout_flag = 2;
						break;
					case T_Co_minbackwards :
						layout_flag = 3;
						break;
					case T_Co_maxdepthslow :
						layout_flag = 4;
						break;
					case T_Co_mindepthslow :
						layout_flag = 5;
						break;
					case T_Co_maxdegree :
						layout_flag = 11;
						break;
					case T_Co_mindegree :
						layout_flag = 10;
						break;
					case T_Co_maxindegree :
						layout_flag = 7;
						break;
					case T_Co_minindegree :
						layout_flag = 6;
						break;
					case T_Co_maxoutdegree :
						layout_flag = 9;
						break;
					case T_Co_minoutdegree :
						layout_flag = 8;
						break;
					case T_Co_tree :
						layout_flag = TREE_LAYOUT;
						break;
					}	
				}
                        	break;

			default:
				if (silent) break;
				FPRINTF(stderr,"Line %d: attribute %s ",
					xfirst_line(node2),ConstructorName(tag(node2)));
				FPRINTF(stderr,"currently not implemented !\n");
			}
			break;	

		/*---------------- end of T_Co_graph_attribute ----------------*/

		case T_Co_foldnode_defaults:
			if (subg_bit) { 
				SYERR(node2,"folding defaults  allowed only on top level");
				break;
			}
			node2 = son1(node1);
			assert((node2));
			switch(tag(node2)) {
			case T_Co_title:
				SYERR(node2,"foldnode.title not allowed");
				break;
			case T_Co_label:
				fold_label= SDecode(son1(node2));
				break;
			case T_Co_info1:
				fold_info1= SDecode(son1(node2));
				break;
			case T_Co_info2:
				fold_info2= SDecode(son1(node2));
				break;
			case T_Co_info3:
				fold_info3= SDecode(son1(node2));
				break;
			case T_Co_level:
				fold_level= FETCHNUM();
				break;
			case T_Co_shape:
				switch(tag(son1(node2))) {
                               	case T_Co_box:
					fold_shape = BOX; 
					break;
                               	case T_Co_rhomb:
					fold_shape = RHOMB; 
					break;
                               	case T_Co_ellipse:
					fold_shape = ELLIPSE; 
					break;
                               	case T_Co_triangle:
					fold_shape = TRIANGLE; 
					break;
				}
				break;
                	case T_Co_horizontal_order:
				fold_horder= FETCHNUM();
				break;
			case T_Co_textmode:
				switch(tag(son1(node2))) {
                               	case T_Co_center:
                                       	fold_textm = CENTER;
                                       	break;
                               	case T_Co_left_justify:
                                       	fold_textm = LEFT;
                                       	break;
                               	case T_Co_right_justify:
                                       	fold_textm = RIGHT;
                                       	break;
                               	default:
					assert((0));
                       		}
                       		break;
			case T_Co_width:
				fold_width = FETCHNUM();
				break;
			case T_Co_height:
				fold_height = FETCHNUM();
				break;
			case T_Co_borderwidth:
				fold_borderw = FETCHNUM();
				break;
			case T_Co_folding:
				SYERR(node2,"foldnode.fold not allowed");
				break;
			case T_Co_loc:
				fold_xpos = get_num(son1(node2));
				fold_ypos = get_num(son2(node2));
				break;
			case T_Co_color:
				fold_color = get_color(son1(node2));
				break;
			case T_Co_textcolor:
				fold_textc = get_color(son1(node2));
				break;
			case T_Co_bordercolor:
				fold_borderc = get_color(son1(node2));
				break;
                	case T_Co_scaling:
				{
					double hhx;

                        		hhx = FETCHFLOAT();
					if (hhx<0) hhx=-hhx;
					if (hhx>1.0) {
						fold_stretch = 100;
						fold_shrink = (int)(100.0/hhx);
					}
					else {
						fold_stretch = (int)(hhx*100.0);
						fold_shrink = 100;
					}
					fold_shrink = rootshrink*fold_shrink;
					fold_stretch= rootstretch*fold_stretch;
					if (fold_shrink==0) fold_shrink = 1;
					if (fold_stretch==0) fold_stretch = 1;
				}
			case T_Co_shrink:
				fold_shrink = rootshrink * FETCHNUM();
				if (fold_shrink == 0) fold_shrink = 1;
				break;
			case T_Co_stretch:
				fold_stretch = rootstretch * FETCHNUM();
				if (fold_stretch == 0) fold_stretch = 1;
				break;
			default:
				if (silent) break;
				FPRINTF(stderr,"Line %d: attribute %s ",
					xfirst_line(node2),ConstructorName(tag(node2)));
				FPRINTF(stderr,"currently not implemented !\n");
			}
			break; 	

		/*---------------- end of T_Co_foldnode_defaults -----------------*/

		case T_Co_node_defaults:
			node2 = son1(node1);
			assert((node2));
			switch(tag(node2)) {
			case T_Co_title:
				SYERR(node2,"node.title not allowed");
				break;
			case T_Co_label:
				label= SDecode(son1(node2));
				break;
			case T_Co_info1:
				info1= SDecode(son1(node2));
				break;
			case T_Co_info2:
				info2= SDecode(son1(node2));
				break;
			case T_Co_info3:
				info3= SDecode(son1(node2));
				break;
			case T_Co_level:
				level= FETCHNUM();
				break;
			case T_Co_shape:
				switch(tag(son1(node2))) {
                               	case T_Co_box:
					shape = BOX; 
					break;
                               	case T_Co_rhomb:
					shape = RHOMB; 
					break;
                               	case T_Co_ellipse:
					shape = ELLIPSE; 
					break;
                               	case T_Co_triangle:
					shape = TRIANGLE; 
					break;
				}
				break;
                	case T_Co_horizontal_order:
				horder= FETCHNUM();
				break;
			case T_Co_textmode:
				switch(tag(son1(node2))) {
                               	case T_Co_center:
                                       	textm = CENTER;
                                       	break;
                               	case T_Co_left_justify:
                                       	textm = LEFT;
                                       	break;
                               	case T_Co_right_justify:
                                       	textm = RIGHT;
                                       	break;
                               	default:
					assert((0));
                       		}
                       		break;
			case T_Co_width:
				width = FETCHNUM();
				break;
			case T_Co_height:
				height = FETCHNUM();
				break;
			case T_Co_borderwidth:
				borderw = FETCHNUM();
				break;
			case T_Co_folding:
				fold = FETCHNUM();
				break;
			case T_Co_loc:
				xpos = get_num(son1(node2));
				ypos = get_num(son2(node2));
				break;
			case T_Co_color:
				color = get_color(son1(node2));
				break;
			case T_Co_textcolor:
				textc = get_color(son1(node2));
				break;
			case T_Co_bordercolor:
				borderc = get_color(son1(node2));
				break;
                	case T_Co_scaling:
				{
					double hhx;

                        		hhx = FETCHFLOAT();
					if (hhx<0) hhx=-hhx;
					if (hhx>1.0) {
						stretch = 100;
						shrink = (int)(100.0/hhx);
					}
					else {
						stretch = (int)(hhx*100.0);
						shrink = 100;
					}
					shrink = rootshrink*shrink;
					stretch= rootstretch*stretch;
					if (shrink==0)  shrink = 1;
					if (stretch==0) stretch = 1;
				}
			case T_Co_shrink:
				shrink = rootshrink * FETCHNUM();
				if (shrink == 0) shrink = 1;
				break;
			case T_Co_stretch:
				stretch = rootstretch * FETCHNUM();
				if (stretch == 0) stretch = 1;
				break;
			default:
				if (silent) break;
				FPRINTF(stderr,"Line %d: attribute %s ",
					xfirst_line(node2),ConstructorName(tag(node2)));
				FPRINTF(stderr,"currently not implemented !\n");
			}
			break; 	

		/*---------------- end of T_Co_node_defaults ------------------*/

		case T_Co_node:
			v = nodealloc(label,textm,width,height,borderw,xpos,
				 ypos,fold,color,textc,borderc,shrink,stretch,
				 info1,info2,info3,level,shape,horder);

			add_to_nodelist_of_root(v);

			node_attributes(son1(node1),v,rootshrink,rootstretch);
			break;

		/*---------------- end of T_Co_node ---------------------------*/

		}
		node = son2(node);
	} /* while */	
}



/*--------------------------------------------------------------------*/

/*  Edge analysis per subgraph
 *  ==========================
 */

/*  The function edge_analyse traverses the syntax tree starting at `node', 
 *  collects the information and allocates the appropriate data structures.
 *  node is assumed to be a T_Co_graph_entry representing a graph.
 *  The situation is always the following:
 *      all edges are collected into the edge list
 *
 *  The arguments elabel ... arrows are the default values that are 
 *  actually valid for the graph, node or edge.
 */



static void	edge_analyse(node,
		  elabel,lstyle,thick,class,prio,horder,ecolor,arrows)
	yysyntaxtree node;	/* syntax tree node                   */
	char	*elabel;	/* edge label      		      */
	int	lstyle;		/* edge line style                    */
	int	thick;		/* edge thickness                     */
	int	class;		/* edge class                         */
	int	prio; 		/* edge priority                      */
	int	horder;		/* edge Horizontal order              */
	int	ecolor;		/* edge color                         */
	int	arrows;		/* edge arrowsize                     */

{
	register yysyntaxtree	node1, node2;	/* auxiliary variables */
	GEDGE	e;

	debugmessage("edge_analyse","");

	gs_wait_message('a');

	/* traverse iteratively the syntax tree: 
 	 * on graph entries, we create a graph summary node, fetch the
	 * 	attributes valid for the summary node and we go into recursion.
	 * on node entries, we create a new node and fetch its attributes.
	 * on edge entries, we create a new edge and fetch its attributes.
	 * on graph attributes, we fetch the attributes valid as default 
	 *	values of the whole graph and set the graph attributes 
	 *	G_x, G_y, etc.
 	 * on node default attributes, we change the default variables
	 *	label, ..., stretch.
	 * on edge default attributes, we change the default variables
	 *	elabel, ..., arrows. 
	 *
	 * Nodes come into the hashtable after fetching all its attributes.
	 * See the corresponding fetch routines.
	 */

	assert((node && (tag(node) == T_Co_graph_entry)));
	while ( node && (tag(node) == T_Co_graph_entry) ) {
		node1 = son1(node);
		assert(node1);
	        switch (tag(node1)) {
		case T_Co_graph:
			subg_bit++;
			edge_analyse(son1(node1),
				elabel,lstyle,thick,class,prio,
				horder, ecolor, arrows);
			subg_bit--;
			break;

		/*---------------- end of T_Co_node_defaults ------------------*/

		case T_Co_foldedge_defaults:
			if (subg_bit) { 
				SYERR(node2,"folding defaults  allowed only on top level");
				break;
			}
			node2 = son1(node1);
			assert((node2));
			switch(tag(node2)) {
			case T_Co_sourcename:
			case T_Co_targetname:
				SYERR(node2,"foldedge.sourcename and foldedge.targetname not allowed");
				break;
			case T_Co_label:
				fold_elabel=SDecode(son1(node2));
				break;
			case T_Co_linestyle:
				switch(tag(son1(node2))) {
				case T_Co_continuous:
					fold_lstyle = SOLID;
					break;
				case T_Co_dotted:
					fold_lstyle = DOTTED;
					break;
				case T_Co_dashed:
					fold_lstyle = DASHED;
					break;
				case T_Co_invisible:
					fold_lstyle = UNVISIBLE;
					break;
				default:
					assert((0));
				}
				break;
			case T_Co_thickness:
				fold_thick = FETCHNUM();
				break;
			case T_Co_class:
				SYERR(node2,"foldedge.class not allowed");
				break;
			case T_Co_priority:
				SYERR(node2,"foldedge.priority not allowed");
				break;
                	case T_Co_horizontal_order:
				SYERR(node2,"foldedge.horizontal_order not allowed");
				break;
			case T_Co_color:
				fold_ecolor = get_color(son1(node2));
				break;
			case T_Co_arrowsize:
				fold_arrows = FETCHNUM();
				break;
			case T_Co_anchor:
				SYERR(node2,"foldedge.anchor not allowed");
				break;
			default:
				if (silent) break;
				FPRINTF(stderr,"Line %d: attribute %s ",
					xfirst_line(node2),ConstructorName(tag(node2)));
				FPRINTF(stderr,"currently not implemented !\n");
			}
			break;	/* T_Co_foldedge_defaults */

		/*---------------- end of T_Co_foldedge_defaults --------------*/

		case T_Co_edge_defaults:
			node2 = son1(node1);
			assert((node2));
			switch(tag(node2)) {
			case T_Co_sourcename:
			case T_Co_targetname:
				SYERR(node2,"edge.sourcename and edge.targetname not allowed");
				break;
			case T_Co_label:
				elabel=SDecode(son1(node2));
				break;
			case T_Co_linestyle:
				switch(tag(son1(node2))) {
				case T_Co_continuous:
					lstyle = SOLID;
					break;
				case T_Co_dotted:
					lstyle = DOTTED;
					break;
				case T_Co_dashed:
					lstyle = DASHED;
					break;
				case T_Co_invisible:
					lstyle = UNVISIBLE;
					break;
				default:
					assert((0));
				}
				break;
			case T_Co_thickness:
				thick = FETCHNUM();
				break;
			case T_Co_class:
				class = FETCHNUM();
				if (class<=0) class = 1;
				break;
			case T_Co_priority:
				prio = FETCHNUM();
				if (prio<=0) prio = 1;
				break;
                	case T_Co_horizontal_order:
				horder = FETCHNUM();
				break;
			case T_Co_color:
				ecolor = get_color(son1(node2));
				break;
			case T_Co_arrowsize:
				arrows = FETCHNUM();
				break;
			case T_Co_anchor:
				SYERR(node2,"edge.anchor not allowed");
				break;
			default:
				if (silent) break;
				FPRINTF(stderr,"Line %d: attribute %s ",
					xfirst_line(node2),ConstructorName(tag(node2)));
				FPRINTF(stderr,"currently not implemented !\n");
			}
			break;	/* T_Co_edge_defaults */

		/*---------------- end of T_Co_edge_defaults ------------------*/

		case T_Co_edge:
			e = edgealloc(elabel,lstyle,thick,class,prio,ecolor,arrows,horder);
			edge_attributes(son1(node1),e);
			break;

		/*---------------- end of T_Co_edge ---------------------------*/
		case T_Co_near_edge:
			e = edgealloc(elabel,lstyle,thick,class,prio,ecolor,arrows,horder);
			edge_attributes(son1(node1),e);
			if (ESTART(e)==EEND(e)) {
				if (silent) break;
				FPRINTF(stderr,"Line %d: self loop near edge neighbouring ignored !\n",
					xfirst_line(node2));
			}
			else if (EANCHOR(e)) {
				if (silent) break;
				FPRINTF(stderr,"Line %d: near edge with anchorpoint neighbouring ignored !\n",
					xfirst_line(node2));
			}
			else if (ELABEL(e)) {
				if (silent) break;
				FPRINTF(stderr,"Line %d: near edge with label ignored !\n",
					xfirst_line(node2));
			}
			else near_edge_insert(e);
			break;

		/*---------------- end of T_Co_near_edge ----------------------*/

		}
		node = son2(node);
	} /* while */	
}



/*--------------------------------------------------------------------*/

/*  Analysis of the graph attributes
 *  --------------------------------
 *  node is a syntax tree node of sort graph_entry.
 *  v is a summary node whose attributes are now filled.
 *  default title is its title, if no other title is specified.
 *  We look in the graph_entry list for graph_attributes, and
 *  analyze these. Not all graph attributes are relevant for
 *  summary nodes, e.g. these that can only defined for
 *  the outermost graph are irrellevent.
 *  At the end of this function, we insert the summary node into the hash
 *  table by the function `check_node' and recognize its folding 
 *  attribute for the folding keepers.
 */          	


static void	graph_attributes(node,v,defaulttitle,rootshrink,rootstretch)
yysyntaxtree	node;
GNODE	v;
char    *defaulttitle;  /* default title of the graph  */ 
int     rootshrink;	/* shrink factor of the root   */
int 	rootstretch;	/* stretch factor of the root  */
{
	register yysyntaxtree	node1, node2;

	debugmessage("graph_attributes","");
	assert((!node)||(tag(node)==T_Co_graph_entry));
	node1 = node;
	while ( node1 && (tag(node1) == T_Co_graph_entry) ) {
		node2 = son1(node1);
		assert(node2);
		switch(tag(node2)) {
		case T_Co_graph_attribute:
			node2 = son1(node2);
			assert(node2);
			switch (tag(node2)) {
			case T_Co_title:
				NTITLE(v) = SDecode(son1(node2));
		    		break;
			case T_Co_label:
				NLABEL(v) = SDecode(son1(node2));
				break;
			case T_Co_info1:
				NINFO1(v) = SDecode(son1(node2));
				break;
			case T_Co_info2:
				NINFO2(v) = SDecode(son1(node2));
				break;
			case T_Co_info3:
				NINFO3(v) = SDecode(son1(node2));
				break;
			case T_Co_level:
				NLEVEL(v) = FETCHNUM();
		    		break;
			case T_Co_shape:
				switch(tag(son1(node2))) {
                               	case T_Co_box:
					NSHAPE(v) = BOX; 
					break;
                               	case T_Co_rhomb:
					NSHAPE(v) = RHOMB; 
					break;
                               	case T_Co_ellipse:
					NSHAPE(v) = ELLIPSE; 
					break;
                               	case T_Co_triangle:
					NSHAPE(v) = TRIANGLE; 
					break;
				}
				break;
                	case T_Co_horizontal_order:
				NHORDER(v) = FETCHNUM();
		    		break;
			case T_Co_textmode:
				switch(tag(son1(node2))) {
				case T_Co_center:
					NTEXTMODE(v) = CENTER;
					break;
				case T_Co_left_justify:
					NTEXTMODE(v) = LEFT;
					break;
				case T_Co_right_justify:
					NTEXTMODE(v) = RIGHT;
					break;
		    		}
				break;
			case T_Co_borderwidth:
				NBORDERW(v) = FETCHNUM();          
				break;  
			case T_Co_xdef:
				NSX(v) = FETCHNUM();
		    		break;
			case T_Co_ydef:
				NSY(v) = FETCHNUM();          
		    		break;
			case T_Co_loc:
				NSX(v) = get_num(son1(node2));
				NSY(v) = get_num(son2(node2));
				break;
			case T_Co_width:
				NWIDTH(v)  = FETCHNUM();          
				break;
			case T_Co_height:
				NHEIGHT(v) = FETCHNUM();          
				break;
			case T_Co_folding:
				NFOLDING(v) = FETCHNUM();          
				break;
			case T_Co_status:
				switch(tag(son1(node2))) {
				case T_Co_black:
					NSTATE(v) = 2;
					break;
				case T_Co_grey:
					NSTATE(v) = 1;
					break;
				case T_Co_white:
					NSTATE(v) = 0;
					break;
				default:
					assert((0));
				}
				break;
			case T_Co_color:
				NCOLOR(v)=get_color(son1(node2));
				break;
			case T_Co_textcolor:
				NTCOLOR(v) = get_color(son1(node2));
				break;
			case T_Co_bordercolor:
				NBCOLOR(v) = get_color(son1(node2));
				break;
                	case T_Co_scaling:
				{
					double hhx;

                        		hhx = FETCHFLOAT();
					if (hhx<0) hhx=-hhx;
					if (hhx>1.0) {
						NSTRETCH(v) = 100;
						NSHRINK(v) = (int)(100.0/hhx);
					}
					else {
						NSTRETCH(v) = (int)(hhx*100.0);
						NSHRINK(v) = 100;
					}
					NSHRINK(v) = rootshrink*NSHRINK(v);
					NSTRETCH(v)= rootstretch*NSTRETCH(v);
					if (NSHRINK(v)==0)  NSHRINK(v) = 1;
					if (NSTRETCH(v)==0) NSTRETCH(v) = 1;
				}
			case T_Co_shrink:
				NSHRINK(v) = rootshrink * FETCHNUM();          
				if (NSHRINK(v) == 0) NSHRINK(v) = 1;
				break;
			case T_Co_stretch:
				NSTRETCH(v) = rootstretch * FETCHNUM();       
				if (NSTRETCH(v) == 0) NSTRETCH(v) = 1;
				break;
			case T_Co_infoname:
			case T_Co_classname:
			case T_Co_xmax:
			case T_Co_ymax:
			case T_Co_xbase:
			case T_Co_ybase:
			case T_Co_xspace:
			case T_Co_xlspace:
			case T_Co_yspace:
			case T_Co_xraster:
			case T_Co_xlraster:
			case T_Co_yraster:
			case T_Co_orientation:
			case T_Co_node_alignment:
			case T_Co_hidden:
			case T_Co_late_edge_label:
			case T_Co_display_edge_label:
			case T_Co_dirty_edge_label:
                	case T_Co_splines:
                	case T_Co_finetuning:
                	case T_Co_crossing_opt:
                	case T_Co_crossing_weight:
			case T_Co_nonearedges:
			case T_Co_dummy:
			case T_Co_layoutalgorithm:
			case T_Co_splinefactor:
			case T_Co_downfactor:
			case T_Co_upfactor:
			case T_Co_nearfactor:
                	case T_Co_treefactor:
                	case T_Co_spreadlevel:
				if (subg_bit!=0) {
					SYERR(node2,"attribute allowed only on top level");
				}
				break;
			default:
				if (silent) break;
				FPRINTF(stderr,"Line %d: attribute %s ",
					xfirst_line(node2),ConstructorName(tag(node2)));
				FPRINTF(stderr,"currently not implemented !\n");
			}
			break;
		case T_Co_constraint:	
			if (silent) break;
			FPRINTF(stderr,"Line %d: constraint ",
				xfirst_line(node2));
			FPRINTF(stderr,"currently not implemented !\n");
		}
		node1 = son2(node1);
    	}

	/* The default title inherit mechanism */

	if (NTITLE(v)==NULL) NTITLE(v) = defaulttitle;
	if (NLABEL(v)==NULL) NLABEL(v) = NTITLE(v);

	/* For the folding keepers */

	if (NFOLDING(v)==0) {
		/* for the initial folding: we must recognize the
		 * foldstoppers.
		 */
		NFOLDING(v) = -1;
		add_foldstop(v);
	}
	if (NSTATE(v)>0) NFOLDING(v)=1;
	if (NFOLDING(v)>0) {
		/* and the subgraph fold starters */
		add_sgfoldstart(v);
	}
	/* subgraph summary nodes are never region foldstarters */

	check_node(node,v);  /* check node and init into to hashtable */
}

/*--------------------------------------------------------------------*/

/*  Analysis of the node attributes
 *  -------------------------------
 *  node is a syntax tree node of sort node_attribute_list.
 *  v is a node whose attributes are now filled.
 *  At the end of this function, we insert the node into the hash
 *  table by the function `check_node' and recognize its folding 
 *  attribute for the folding keepers.
 */          	


static void	node_attributes(node,v,rootshrink,rootstretch)
yysyntaxtree	node;
GNODE	v;
int     rootshrink;	/* shrink factor of the root   */
int 	rootstretch;	/* stretch factor of the root  */
{
	register yysyntaxtree	node1, node2;

	debugmessage("node_attributes","");
	assert((!node)||(tag(node)==T_Co_node_attribute));
	node1 = node;
	while ( node1 && (tag(node1) == T_Co_node_attribute) ) {
		node2 = son1(node1);
		assert(node2);
		switch(tag(node2)) {
		case T_Co_title:
			NTITLE(v) = SDecode(son1(node2));
		    	break;
		case T_Co_label:
			NLABEL(v) = SDecode(son1(node2));
			break;
		case T_Co_info1:
			NINFO1(v) = SDecode(son1(node2));
			break;
		case T_Co_info2:
			NINFO2(v) = SDecode(son1(node2));
			break;
		case T_Co_info3:
			NINFO3(v) = SDecode(son1(node2));
			break;
		case T_Co_level:
			NLEVEL(v) = FETCHNUM();
			break;
		case T_Co_shape:
			switch(tag(son1(node2))) {
                        case T_Co_box:
				NSHAPE(v) = BOX; 
				break;
                       	case T_Co_rhomb:
				NSHAPE(v) = RHOMB; 
				break;
                       	case T_Co_ellipse:
				NSHAPE(v) = ELLIPSE; 
				break;
                       	case T_Co_triangle:
				NSHAPE(v) = TRIANGLE; 
				break;
			}
			break;
                case T_Co_horizontal_order:
			NHORDER(v) = FETCHNUM();
			break;
		case T_Co_textmode:
			switch(tag(son1(node2))) {
			case T_Co_center:
				NTEXTMODE(v) = CENTER;
				break;
			case T_Co_left_justify:
				NTEXTMODE(v) = LEFT;
				break;
			case T_Co_right_justify:
				NTEXTMODE(v) = RIGHT;
				break;
		    	}
			break;
		case T_Co_width:
			NWIDTH(v) = FETCHNUM();          
			break;
		case T_Co_height:
			NHEIGHT(v) = FETCHNUM();          
			break;
		case T_Co_borderwidth:
			NBORDERW(v) = FETCHNUM();          
			break;  
		case T_Co_loc:
			NSX(v) = get_num(son1(node2));
			NSY(v) = get_num(son2(node2));
			break;
		case T_Co_folding:
			NFOLDING(v) = FETCHNUM();          
			break;
		case T_Co_color:
			NCOLOR(v) = get_color(son1(node2));
			break;
		case T_Co_textcolor:
			NTCOLOR(v) = get_color(son1(node2));
			break;
		case T_Co_bordercolor:
			NBCOLOR(v) = get_color(son1(node2));
			break;
                case T_Co_scaling:
			{
				double hhx;

                       		hhx = FETCHFLOAT();
				if (hhx<0) hhx=-hhx;
				if (hhx>1.0) {
					NSTRETCH(v) = 100;
					NSHRINK(v) = (int)(100.0/hhx);
				}
				else {
					NSTRETCH(v) = (int)(hhx*100.0);
					NSHRINK(v) = 100;
				}
				NSHRINK(v) = rootshrink*NSHRINK(v);
				NSTRETCH(v)= rootstretch*NSTRETCH(v);
				if (NSHRINK(v)==0)  NSHRINK(v) = 1;
				if (NSTRETCH(v)==0) NSTRETCH(v) = 1;
			}
		case T_Co_shrink:
			NSHRINK(v) = rootshrink * FETCHNUM();          
			if (NSHRINK(v) == 0) NSHRINK(v) = 1;
			break;
		case T_Co_stretch:
			NSTRETCH(v) = rootstretch * FETCHNUM();          
			if (NSTRETCH(v) == 0) NSTRETCH(v) = 1;
			break;
		default:
			if (silent) break;
			FPRINTF(stderr,"Line %d: attribute %s ",
				xfirst_line(node2),ConstructorName(tag(node2)));
			FPRINTF(stderr,"currently not implemented !\n");
		}
		node1 = son2(node1);
    	}

	/* The default title inherit mechanism */

	if (NTITLE(v)==NULL) SYERR(node,"Missing title of a node");
	if (NLABEL(v)==NULL) NLABEL(v) = NTITLE(v);

	/* For the folding keepers */

	if (NFOLDING(v)==0) {
		/* for the initial folding: we must recognize the
		 * foldstoppers.
		 */
		NFOLDING(v) = -1;
		add_foldstop(v);
	}
	else if (NFOLDING(v)>0) {
		/* and the fold starters */
		add_foldstart(v);
	}
	check_node(node,v);  /* check node and init into to hashtable */
}

/*--------------------------------------------------------------------*/

/*  Analysis of the edge attributes
 *  -------------------------------
 *  node is a syntax tree node of sort edge_attribute_list.
 *  e is an edge whose attributes are now filled.
 *  Additionally, it is checked whether the edge has valid start and 
 *  end nodes.
 */          	


static void	edge_attributes(node,e)
yysyntaxtree	node;
GEDGE	e;
{
	register yysyntaxtree	node1, node2;

	debugmessage("edge_attributes","");
	assert((!node)||(tag(node)==T_Co_edge_attribute));
	node1 = node;
	while ( node1 && (tag(node1)==T_Co_edge_attribute) ) {
	    	node2 = son1(node1);
		assert(node2);
	    	switch(tag(node2)) {
		case T_Co_sourcename:
			ESTART(e) = search_node(node2,SDecode(son1(node2)));
			break;
		case T_Co_targetname:
			EEND(e)   = search_node(node2,SDecode(son1(node2)));
			break;
		case T_Co_linestyle:
			switch(tag(son1(node2))) {
                        case T_Co_continuous:
                        	ELSTYLE(e) = SOLID;  break;
                        case T_Co_dotted:  
                                ELSTYLE(e) = DOTTED; break; 
                        case T_Co_dashed: 
                                ELSTYLE(e) = DASHED; break; 
			case T_Co_invisible:
				ELSTYLE(e) = UNVISIBLE;
				break;
                        }
			break;
		case T_Co_label:
			ELABEL(e) = SDecode(son1(node2));
			break;
		case T_Co_thickness:
			ETHICKNESS(e) = FETCHNUM();          
			break;
		case T_Co_class:
			ECLASS(e)     = FETCHNUM();          
			if (ECLASS(e)<=0) ECLASS(e) = 1;
			break;
		case T_Co_priority:
			EPRIO(e)     = FETCHNUM();          
			if (EPRIO(e)<=0) EPRIO(e) = 1;
			break;
		case T_Co_color:
			ECOLOR(e) = get_color(son1(node2));
			break;
                case T_Co_horizontal_order:
			EHORDER(e) = FETCHNUM();
			break;
		case T_Co_arrowsize:
			EARROWSIZE(e) = FETCHNUM();          
			break;
		case T_Co_anchor:
			EANCHOR(e) = FETCHNUM();          
			if ((EANCHOR(e)<=0)||(EANCHOR(e)>=63)) 
				SYERR(node2,"Illegal anchorpoint");
			break;
		default:
			FPRINTF(stderr,"Line %d: attribute %s ",
				xfirst_line(node2),ConstructorName(tag(node2)));
			FPRINTF(stderr,"currently not implemented !\n");
		}
	    	node1 = son2(node1);
	}
	if ((ESTART(e)==NULL)||(EEND(e)==NULL)) 
		SYERR(node,"Missing source or target of an edge");
}


/*--------------------------------------------------------------------*/

/*  Analysis of the Color
 *  ---------------------
 *  node is a syntax tree node of sort enum_color.
 *  The return value is the analyzed color.
 */          	


static int get_color(node)
yysyntaxtree node;
{
	debugmessage("get_color","");
	switch(tag(node)) {
	case T_Co_black:
		return(BLACK);
	case T_Co_blue:
		return(BLUE);
	case T_Co_red:
		return(RED);
	case T_Co_green:
		return(GREEN);
	case T_Co_yellow:
		return(YELLOW);
	case T_Co_magenta:
		return(MAGENTA);
	case T_Co_cyan:
		return(CYAN);
	case T_Co_white:
		return(WHITE);
	case T_Co_darkgrey:
		return(DARKGREY);
	case T_Co_darkblue:
		return(DARKBLUE);
	case T_Co_darkred:
		return(DARKRED);
	case T_Co_darkgreen:
		return(DARKGREEN);
	case T_Co_darkyellow:
		return(DARKYELLOW);
	case T_Co_darkmagenta:
		return(DARKMAGENTA);
	case T_Co_darkcyan:
		return(DARKCYAN);
	case T_Co_gold:
		return(GOLD);
	case T_Co_lightgrey:
		return(LIGHTGREY);
	case T_Co_lightblue:
		return(LIGHTBLUE);
	case T_Co_lightred:
		return(LIGHTRED);
	case T_Co_lightgreen:
		return(LIGHTGREEN);
	case T_Co_lightyellow:
		return(LIGHTYELLOW);
	case T_Co_lightmagenta:
		return(LIGHTMAGENTA);
	case T_Co_lightcyan:
		return(LIGHTCYAN);
	case T_Co_lilac:
		return(LILAC);
	case T_Co_turquoise:
		return(TURQUOISE);
	case T_Co_aquamarine:
		return(AQUAMARINE);
	case T_Co_khaki:
		return(KHAKI);
	case T_Co_purple:
		return(PURPLE);
	case T_Co_yellowgreen:
		return(YELLOWGREEN);
	case T_Co_pink:
		return(PINK);
	case T_Co_orange:
		return(ORANGE);
	case T_Co_orchid:
		return(ORCHID);
	}
	assert((0));  /* we should never come to this point */
	return(BLACK);
}

/*--------------------------------------------------------------------*/

/*   Calculate the maximal number of classes
 *   ---------------------------------------
 *   We traverse recursively the syntax tree to look for class declarations.
 *   The maximal number that occurs is stored into max_nr_classes. 
 */


static void calc_nr_classes(node)
yysyntaxtree node;
{
	yysyntaxtree node1, node2;
	int h;

	debugmessage("calc_nr_classes","");

	assert((node && (tag(node) == T_Co_graph_entry)));
	while ( node && (tag(node) == T_Co_graph_entry) ) {
		node1 = son1(node);
		assert(node1);
	        switch (tag(node1)) {
		case T_Co_graph:
			calc_nr_classes(son1(node1));
			break;

		case T_Co_graph_attribute:

			node2 = son1(node1);
			assert((node2));
			switch(tag(node2)) {
			case T_Co_classname:
			case T_Co_hidden:
				h = FETCHNUM();
				if (h>max_nr_classes) max_nr_classes = h;
                        	break;
			}
			break;

		case T_Co_foldedge_defaults:
		case T_Co_edge_defaults:

			node2 = son1(node1);
			assert((node2));
			switch(tag(node2)) {
			case T_Co_class:
				h = FETCHNUM();
				if (h>max_nr_classes) max_nr_classes = h;
				break;
			}
			break;

		case T_Co_edge:
		case T_Co_near_edge:

			node1 = son1(node1);
			while ( node1 && (tag(node1)==T_Co_edge_attribute) ) {
	   		 	node2 = son1(node1);
				assert(node2);
	    			switch(tag(node2)) {
				case T_Co_class:
					h = FETCHNUM();
					if (h>max_nr_classes) 
						max_nr_classes = h;
					break;
				}
				node1 = son2(node1);
			}
		}
		node = son2(node);
	} /* while */	
}

/*--------------------------------------------------------------------*/

/*  Node Search functions 
 *  =====================
 */          	

static char buffer[1024];      /* Buffer to create error messages */


/*  Node Search 
 *  ----------- 
 *  Search a node in the nodelist and graphlist.
 *  Creates an error message, if the node is not avalable.
 */          	

static GNODE search_node(x,title)
yysyntaxtree x;
char *title;
{
	GNODE n;
	debugmessage("search_node",(title?title:"(null)"));
	n = lookup_hashnode(title);
	if (n==NULL) {
		SPRINTF(buffer,"Undefined node %s",title);
		SYERR(x,buffer);
	}
        return(n);
}


/*  Visible Node Search
 *  -------------------
 *  Search a node in the nodelist and graphlist.
 *  Return NULL, if the node is not avalable or invisible.
 */

GNODE   search_visible_node(title)
char    *title;
{
	GNODE   n;
	if (!title) return(NULL);
	debugmessage("search_visible_node",title);
	n = lookup_hashnode(title);

	/* note: at that time point are NINLIST and NINVISIBLE inverse */

	return(n);
}



/*  Node Check  
 *  - - - - -  
 *  Check whether a node already exists in the node list and graphlist.
 *  Creates an error message, if the node is avalable.
 *  Furthermore: insert the node into the hash table.
 */          	
 
static void check_node(x,m)
yysyntaxtree x;
GNODE m;
{
	char *title;
	GNODE n;

	assert((m));
	title = NTITLE(m);
	debugmessage("check_node",(title?title:"(null)"));
	if (title && (!fastflag)) {
		n = lookup_hashnode(title);
		if (n!=NULL) {
			SPRINTF(buffer,"Double defined node %s",title);
			SYERR(x,buffer);
		}
	} 
	insert_hashnode(m);
}
 

/*--------------------------------------------------------------------*/

/*  Node Lookup Hashtable
 *  =====================
 *  To find a node very fast, we use a hash table.
 *  Because in the hash table are no temporary nodes, we can
 *  misuse the NINTERN field of nodes.
 *  The hash table is an array whose entries are lists of GNODE
 *  objects, linked by the NINTERN field. 
 *  The are hashed with respect to the title.
 */

/* The size of the hashtable MAXHASHTABLE should be a prime number. 
 * Good candidates are: 101, 211, 307, 503, 1013, 1511, 2003, 3001, 5003, ...  
 */

#define MAXHASHTABLE 5003 

/*  The hash table 
 */

static GNODE hashtable[MAXHASHTABLE];


/*  Initialization of the Hashtable
 *  -------------------------------
 */ 

static void init_hashtable()
{
	int i;
	for (i=0; i<MAXHASHTABLE; i++) hashtable[i]=NULL;
}


/*  Hashvalue calculation.
 *  ---------------------
 *  The get an useful dispersion, we add for each character 101 and
 *  the ASCII value of the character.  
 *  The hashvalue is between 0 and MAXHASHTABLE-1.
 */

static int hashval(s)
char *s;
{
	register int res;

	assert((s));
	res = 0;
	while (*s) { res = (res + *s + 101); s++; }
	if (res<0) res = -res;
	return(res % MAXHASHTABLE);
}

/*  Insert a node into the hash table.
 *  ----------------------------------
 *  x is the node. The insert is done blind, i.e. it is not checked
 *  whether x is already in the table.
 *  Usually, graph summary nodes and real nodes are inserted after
 *  fetching all attributes.
 */


static void insert_hashnode(x)
GNODE x;
{
	char *title;
	int val;

	assert((x));
	title = NTITLE(x);
	debugmessage("insert_hashnode",(title?title:"(null)"));
	if (!title) {
		FPRINTF(stderr,"Missing title of a node");
		return;
	}
	val = hashval(title);
	assert((NINTERN(x)==NULL));
	NINTERN(x) = hashtable[val];		
	hashtable[val] = x;
}

/*  Lookup in the hash table
 *  ------------------------
 *  returns the first node that has the title, or NULL.
 */

GNODE lookup_hashnode(title)
char *title;
{
	GNODE h;

	debugmessage("lookup_hashnode",(title?title:"(null)"));
	h = hashtable[hashval(title)];
	while (h) {
                if (title==NTITLE(h)) return(h);
                if (strcmp(title,NTITLE(h))==0) return(h);
		h = NINTERN(h);
	}
	return(NULL);	
}


/*--------------------------------------------------------------------*/

/*  Hashtable Cursor
 *  ================
 *
 *  The hash table cursor points into the hashtable and can be
 *  moved backwards and forewards through the hashtable.
 *  It is used to allow easy selection of nodes.
 */

static GNODE act_hash_cursor;
static int   act_hash_pos;
int 	     act_hash_size;


/*  Initialize the cursor
 *  ---------------------
 */

void init_hash_cursor()
{
	GNODE h;
	int i;

	debugmessage("init_hash_cursor","");

	act_hash_cursor = NULL;
	act_hash_pos    = 0;
	act_hash_size   = 0;
	for (i=0; i<MAXHASHTABLE; i++)
               	if (hashtable[i]!=NULL) {
			if (!act_hash_cursor) {
				act_hash_cursor = hashtable[act_hash_pos];
				act_hash_pos = i;
			}
			h = hashtable[i];
			while (h) { act_hash_size++; h=NINTERN(h); }
		} 
}


/*  Position cursor 
 *  ---------------
 */

void position_hash_cursor(j)
int j;
{
	GNODE h;
	int i;

	debugmessage("increment_hash_cursor","");

	assert((j>=0));
	if (!act_hash_cursor) return;
	if (act_hash_size == 0) return;

	for (i=0; i<MAXHASHTABLE; i++)
               	if (hashtable[i]!=NULL) {
			h = hashtable[i];
			while (h) { 
				if (j==0) {
		                	act_hash_cursor = h;
                                	act_hash_pos    = i;
					return;
				}
				j--;
				h=NINTERN(h); 
			}
		} 
}



/*  Get i.th node after the cursor
 *  ------------------------------
 */

GNODE get_hash_cursor_succ(i)
int i;
{
	GNODE h;
	int j;

	debugmessage("get_hash_cursor_succ","");

	if (!act_hash_cursor) return(NULL);
	if (i<0) i=0;

	j = act_hash_pos;
	h = act_hash_cursor;

	while (i>0) {
		i--;
		if (NINTERN(h)) h = NINTERN(h);
		else {
			j++;
			h = NULL;
			for (; j<MAXHASHTABLE; j++)
       		         	if (hashtable[j]!=NULL) {
					h = hashtable[j];
					break;
				} 
			if (!h) return(NULL); 
		}
	}
	return(h);
}


/*--------------------------------------------------------------------*/

/*  For debugging only:
 *  ------------------
 *  Routine that looks for a checknode. This is helpful if we need
 *  to inspect a certain node.
 */

#ifdef CHECKNODE

GNODE debug_checknode = NULL;


static void debug_init_checknode()
{
	GNODE h;

	h = nodelist;
	while (h) {
		if (NREFNUM(h)==1) {
			debug_checknode = h;
			return;
		}
		h = NNEXT(h);
	}
}

#endif /* CHECKNODE */


