/* SCCS-info %W% %E% */

/*--------------------------------------------------------------------*/
/*                                                                    */
/*              VCG : Visualization of Compiler Graphs                */
/*              --------------------------------------                */
/*                                                                    */
/*   file:         alloc.c                                            */
/*   version:      1.00.00                                            */
/*   creation:     14.4.1993                                          */
/*   author:       I. Lemke  (...-Version 0.99.99)                    */
/*                 G. Sander (Version 1.00.00-...)                    */  
/*                 Universitaet des Saarlandes, 66041 Saarbruecken    */
/*                 ESPRIT Project #5399 Compare                       */
/*   description:  Memory Management                                  */
/*   status:       in work                                            */
/*                                                                    */
/*--------------------------------------------------------------------*/

#ifndef lint
static char *id_string="$Id: alloc.c,v 3.7 1994/08/03 13:58:44 sander Exp $";
#endif

/*
 *   Copyright (C) 1993, 1994 by Georg Sander, Iris Lemke, and
 *                               the Compare Consortium 
 *
 *  This program and documentation is free software; you can redistribute 
 *  it under the terms of the  GNU General Public License as published by
 *  the  Free Software Foundation;  either version 2  of the License,  or
 *  (at your option) any later version.
 *
 *  This  program  is  distributed  in  the hope that it will be useful,
 *  but  WITHOUT ANY WARRANTY;  without  even  the  implied  warranty of
 *  MERCHANTABILITY  or  FITNESS  FOR  A  PARTICULAR  PURPOSE.  See  the
 *  GNU General Public License for more details.
 *
 *  You  should  have  received a copy of the GNU General Public License
 *  along  with  this  program;  if  not,  write  to  the  Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  The software is available per anonymous ftp at ftp.cs.uni-sb.de.
 *  Contact  sander@cs.uni-sb.de  for additional information.
 */


/* 
 * $Log: alloc.c,v $
 * Revision 3.7  1994/08/03  13:58:44  sander
 * Horizontal order mechanism changed.
 * Attribute horizontal_order for edges added.
 *
 * Revision 3.6  1994/05/17  16:37:18  sander
 * attribute node_align added to allow nodes to be centered in the levels.
 *
 * Revision 3.5  1994/05/16  08:56:03  sander
 * shape attribute (boxes, rhombs, ellipses, triangles) added.
 *
 * Revision 3.4  1994/05/05  08:20:30  sander
 * dllist_free_all added for the local optimization of crossings.
 *
 * Revision 3.3  1994/04/27  16:05:19  sander
 * Some general changes for the PostScript driver.
 * Horizontal order added. Bug fixes of the folding phases:
 * Folding of nested graphs works now.
 *
 * Revision 3.2  1994/03/04  19:11:24  sander
 * Specification of levels per node added.
 * X11 geometry behaviour (option -geometry) changed such
 * that the window is now opened automatically.
 *
 * Revision 3.1  1994/03/01  10:59:55  sander
 * Copyright and Gnu Licence message added.
 * Problem with "nearedges: no" and "selfloops" solved.
 *
 * Revision 2.2  1994/01/21  19:33:46  sander
 * VCG Version tested on Silicon Graphics IRIX, IBM R6000 AIX and Sun 3/60.
 * Option handling improved. Option -grabinputfocus installed.
 * X11 Font selection scheme implemented. The user can now select a font
 * during installation.
 * Sun K&R C (a nonansi compiler) tested. Some portabitility problems solved.
 *
 * Revision 2.1  1993/12/08  21:20:09  sander
 * Reasonable fast and stable version
 *
 */


/****************************************************************************
 * This file is a collection of auxiliary functions that implement the
 * memory management. It provides the following functions:
 *
 * myalloc		allocates memory from the internal memory management.
 * free_memory		gives the complete memory free.
 *
 * nodealloc		allocates a GNODE object (node), adds it to nodelist
 * graphalloc		allocates a GNODE object (graph), adds it to graphlist
 * tmpnodealloc		allocates a temporary GNODE object
 * free_node		deallocate a nontemporary GNODE object
 *
 * nodelist_alloc	allocates a stable node list element (i.e. a cons 
 *			cell whose head is a GNODE object)
 * tmpnodelist_alloc	allocates a temporary node list element (i.e. a cons 
 *			cell whose head is a GNODE object)
 * free_regionnodelist	deallocate a list of node list elements. 
 *		 	These were used as storage of regions.
 *
 * edgealloc		allocates a GEDGE object (edge), adds it to edgelist
 * tmpedgealloc		allocates a temporary GEDGE object
 *
 * near_edge_insert     insert a near edge into near_edge_list. A near edge
 *			is a special edge that must always be placed near an
 *			other edge. See the nearedge-specification-feature.
 *			We use a special adjacency list to notify all 
 *			near edges.
 * prededgealloc	allocates an edge list element (i.e. a cons cell whose
 *                      head is a GEDGE object), used as adjacency list of
 *                      predecessors of a node.
 * succedgealloc	allocates an edge list element (i.e. a cons cell whose
 *                      head is a GEDGE object), used as adjacency list of
 *                      succecessors of a node.
 *
 * connectalloc		allocates a CONNECT element of a node
 *
 * dllist_alloc		allocates a DLLIST-cons cell. These are double linked
 *			lists of nodes.
 * dllist_free	 	gives one DLLIST-cons cell free.
 *
 * free_all_lists	gives all temporary memory free.
 * reinit_all_lists     reinitialize all memory lists. This is done if the
 *			memory is given free: all lists are set to NULL.
 ***************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include "globals.h"
#include "grammar.h"
#include "main.h"
#include "alloc.h"
#include "folding.h"

#undef DEBUG
#undef debugmessage
#ifdef DEBUG
#define debugmessage(a,b) {FPRINTF(stderr,"Debug: %s %s\n",a,b);}
#else
#define debugmessage(a,b) /**/
#endif


/* Prototypes
 * ----------
 */

static GNODE internal_nodealloc	_PP((void));
static void free_nodelists	_PP((void));
static GEDGE internal_edgealloc	_PP((void));
static void free_tmpedges	_PP((void));
static ADJEDGE  edgelist_alloc	_PP((void));
static void free_edgelists	_PP((void));
static void free_connect	_PP((void));

/*--------------------------------------------------------------------*/
/* Memory allocation                                                  */
/*--------------------------------------------------------------------*/

/*  Core Memory Management 
 *  ======================
 */

#ifdef DEBUG
static long act_alloc_size  = 0;
static long act_alloc_edges = 0;
static long act_alloc_nodes = 0;
#endif

static int node_refnum = 0;	/* reference counter for REFNUM	of nodes */


/*   Core Memory allocation
 *   ----------------------  
 *   allocate x bytes. We use the memory mechanism from the generated 
 *   parser. 
 */

char *myalloc(x)
int 	x;
{
	debugmessage("myalloc. Nr. of Bytes:",my_itoa(x));
#ifdef DEBUG
	act_alloc_size += x;
	PRINTF("Alloc Summary: %ld Bytes allocated\n",act_alloc_size);
#endif
	return(ParseMalloc(x));
}


/*   Core Memory deallocation
 *   ------------------------  
 *   deallocate the complete memory.
 */

void 	free_memory()
{
	debugmessage("free_memory","");
#ifdef DEBUG
	act_alloc_size  = 0;
	act_alloc_nodes = 0;
	act_alloc_edges = 0;
#endif
	FreeHash();
	ParseFree();
	node_refnum = 0;	
	reinit_all_lists();
}

/*--------------------------------------------------------------------*/

/*  Memory Management for Nodes 
 *  ===========================
 */

/*  We distinguish between
 *     1) GNODE objects as nodes from the specification
 *     2) GNODE objects as representation of subgraphs
 *     3) temporary GNODE objects, used to calculate a layout.
 *        They can be deleted on each change of layout.
 */

/*  Nodes from the specification come into the nodelist.
 *  Subgraphs come into the subgraphlist.
 *  Temporary nodes come into the tmpnodelist.
 *  Free GNODE objects are collected into the node_freelist. 
 */

int nodeanz   = 0;    	       /* Number of nodes in nodelist         */
int dummyanz  = 0;             /* Number of dummy nodes (not labels)  */ 
 
 
GNODE nodelist     = NULL;     /* List of all real nodes as specified */
GNODE nodelistend  = NULL;     /* End of this list                    */
GNODE graphlist    = NULL;     /* List of all subgraphs as specified  */
GNODE graphlistend = NULL;     /* End of this list                    */

GNODE invis_nodes  = NULL;     /* List of all invisible nodes         */
GNODE labellist    = NULL;     /* List of dummy nodes that contain    */
                               /* the text of an edge label           */
GNODE labellistend = NULL;     /* End of this list                    */
GNODE dummylist    = NULL;     /* List of other dummy nodes           */

GNODE tmpnodelist   = NULL;     /* list of allocated temoprary nodes */
static GNODE node_freelist = NULL;     /* list of free GNODE objects */



/*  Allocate a GNODE object
 *  -----------------------
 *  First, we look in the free list, if we have a free node. Otherwise,
 *  we allocate a node from the core memory.
 *  We also set some default values.
 */

static GNODE internal_nodealloc()
{
	GNODE   h;

	debugmessage("internal_nodealloc","");
	if (node_freelist) {
		h = node_freelist;
		node_freelist = NINTERN(node_freelist);
	}
	else {
	 	h = (GNODE) myalloc(sizeof(struct gnode));
#ifdef DEBUG
		act_alloc_nodes++;
		PRINTF("Alloc Summary: %ld GNODEs allocated\n",act_alloc_nodes);
#endif
	}

	NREFNUM(h)	= node_refnum++;
	NTITLE(h) 	= NULL;
	NLABEL(h) 	= NULL;
	NINFO1(h)	= NULL;
	NINFO2(h)	= NULL;
	NINFO3(h)	= NULL;
	NLEVEL(h) 	= -1;
	NSHAPE(h) 	= 0;
	NHORDER(h) 	= -1;
	NSX(h)        	= 0;	
	NSY(h)        	= 0;
	NX(h)        	= 0;	
	NY(h)        	= 0;
	NSGRAPH(h)	= NULL;
	NROOT(h)	= NULL;
	NREGREPL(h)	= NULL;
	NREGION(h)	= NULL;
	NREGROOT(h)	= NULL;
	NINLIST(h)	= 1;
	NINVISIBLE(h)	= 0;
	NTIEFE(h)    	= -1;
	NPOS(h)      	= -1;   
	NWEIGHTS(h)     = 0;
	NWEIGHTP(h)	= 0;
	NMARK(h)     	= 0; 
	NREVERT(h)     	= 0; 
	NANCHORNODE(h) 	= 0; 
	h->bary 	= -1;
	NDFS(h)      	= -1; 
	NINDEG(h)    	= 0;
	NOUTDEG(h)	= 0;
	NVPTR(h)	= NULL;
	NPRED(h)      	= NULL;
	NSUCC(h)     	= NULL; 
	NSVPRED(h)     	= NULL;
	NSVSUCC(h)     	= NULL; 
	NPREDL(h)       = NULL;
        NPREDR(h)       = NULL; 
        NSUCCL(h)       = NULL; 
        NSUCCR(h)       = NULL;
	NCONNECT(h)  	= NULL;
	NNEXT(h) 	= NULL;
	NINTERN(h)	= NULL;
	return(h);
}



/*  Allocate a real node 
 *  --------------------
 *  and update the nodelist. Real nodes are such nodes that are specified
 *  as node: { .... }.
 */

GNODE nodealloc(label,textm,width,height,borderw,xpos,ypos,fold,color,
		textc,borderc,shrink,stretch,info1,info2,info3,level,shape,horder)
char 	*label;
int	textm,width,height,borderw,xpos,ypos,fold,color,textc,borderc;
int	shrink,stretch;
char 	*info1,*info2,*info3;
int 	level;
int 	shape;
int     horder;
{
	GNODE   h;

	debugmessage("nodealloc","");
	h = internal_nodealloc();

	NLABEL(h)	= label;
	NLEVEL(h)	= level;
	NSHAPE(h)	= (char)shape;
	NHORDER(h)	= horder;
	NINFO1(h)	= info1;
	NINFO2(h)	= info2;
	NINFO3(h)	= info3;
	NSX(h)		= xpos;
	NSY(h)		= ypos;
	NTEXTMODE(h)	= textm;
	NSTATE(h)    	= 0;   
	NWIDTH(h)    	= width;   
	NHEIGHT(h)   	= height;
	NBORDERW(h)	= borderw;
	NFOLDING(h)	= fold;
	NCOLOR(h)	= color;
	NTCOLOR(h)	= textc;
	NBCOLOR(h)	= borderc;
	NSHRINK(h)	= shrink;
	NSTRETCH(h)	= stretch;
	NBEFORE(h)	= nodelistend;

	if (nodelistend) NNEXT(nodelistend) = h;
	nodelistend	= h;
	nodeanz++;
	if (nodelist == NULL) nodelist = h;
	return(h);
}



/*  Allocate a graph object 
 *  -----------------------
 *  and update the graphlist. Such objects are summary nodes whose graph
 *  is specified as graph: { .... }. 
 */

GNODE graphalloc(label,textm,width,height,borderw,xpos,ypos,fold,color,
		 textc,borderc,shrink,stretch,info1,info2,info3,level,shape,horder)
char 	*label;
int	textm,width,height,borderw,xpos,ypos,fold,color,textc,borderc;
int	shrink,stretch;
char 	*info1,*info2,*info3;
int	level;
int 	shape;
int 	horder;
{
	GNODE   h;

	debugmessage("graphalloc","");
	h = internal_nodealloc();

	NLABEL(h)	= label;
	NLEVEL(h)	= level;
	NSHAPE(h)	= (char)shape;
	NHORDER(h)	= horder;
	NINFO1(h)	= info1;
	NINFO2(h)	= info2;
	NINFO3(h)	= info3;
	NSX(h)		= xpos;
	NSY(h)		= ypos;
	NTEXTMODE(h)	= textm;
	NSTATE(h)    	= 0;   
	NWIDTH(h)	= width;
	NHEIGHT(h)	= height;
	NBORDERW(h)	= borderw;
	NFOLDING(h)	= fold; 
	NCOLOR(h)	= color;
	NTCOLOR(h)	= textc;
	NBCOLOR(h)	= borderc;
	NSHRINK(h)	= shrink;
	NSTRETCH(h)	= stretch;	
	NINLIST(h)	= 0;
	NINVISIBLE(h) 	= 1;
	NDFS(h)		= 0;
	NBEFORE(h)	= graphlistend;

	if (graphlistend) NNEXT(graphlistend) = h;
	graphlistend	= h;
	if (graphlist == NULL) graphlist = h;
	return(h);
}


/*  Allocate a temporary GNODE object 
 *  ---------------------------------
 *  and update the tmpnodelist. These are node for dummy's, label's etc.
 *  These nodes are only needed for the layouting.
 */

GNODE	tmpnodealloc(textm,width,height,borderw,fold,color,textc,borderc,
                 	    shrink,stretch,horder)
int     textm,width,height,borderw,fold,color,textc,borderc;
int     shrink,stretch,horder;
{
	GNODE	h;

	debugmessage("tmpnodealloc","");
	h = internal_nodealloc();

	NHORDER(h)	= horder;
        NTEXTMODE(h)    = textm;
	NSTATE(h)    	= 0;   
        NWIDTH(h)       = width;
        NHEIGHT(h)      = height;
        NBORDERW(h)     = borderw;
        NFOLDING(h)     = fold;
        NCOLOR(h)       = color;
        NTCOLOR(h)      = textc;
        NBCOLOR(h)      = borderc;
        NSHRINK(h)      = shrink;
        NSTRETCH(h)     = stretch;
        NINLIST(h)      = 0;
        NINVISIBLE(h)   = 1;
        NDFS(h)         = 0;
        NBEFORE(h)      = NULL;
	
	NINTERN(h)   = tmpnodelist;
	tmpnodelist  = h;
	return(h);
}


/*  Deallocate all temporary GNODE objects
 *  --------------------------------------
 */

void free_tmpnodes()
{
	GNODE	h;

	debugmessage("free_tmpnodes","");
	h = tmpnodelist;
	if (h) {
		while (NINTERN(h)) h = NINTERN(h);
		NINTERN(h) = node_freelist;
		node_freelist = tmpnodelist;
		tmpnodelist = NULL;
	}
	labellist    = NULL;  
	labellistend = NULL;  
	dummylist = NULL;
	/* Labels and dummys are temporary nodes thus they 
	 * are given free, too 
 	 */
}


/*  Deallocate one GNODE objects
 *  ----------------------------
 *  This object should not be temporary !!!
 *  In fact, it is a region summary substitution node. See folding.c
 *  If the object would be allocated by tmpnodealloc,
 *  it could be given free twice, which is wrong.
 */

void free_node(h)
GNODE h;
{
	debugmessage("free_node","");
	NINTERN(h) = node_freelist;
	node_freelist = h;
}


/*  Give a position, search a node in the node list
 *  -----------------------------------------------
 *  This is used in the menues after selecting a node. 
 *  At this time point, all visible nodes are in the node list.
 */

GNODE	search_xy_node(x,y)
int	x,y;
{
	GNODE	v;
	int	xpos, ypos, width, height;

	v = nodelist;
	while (v) {
		xpos = (NX(v)*G_stretch)/G_shrink - V_xmin;
		ypos = (NY(v)*G_stretch)/G_shrink - V_ymin;
		width = (NWIDTH(v)*G_stretch)/G_shrink;
		height = (NHEIGHT(v)*G_stretch)/G_shrink;
		if ( (xpos <= x) && (x <= xpos+width) && 
	     	     (ypos <= y) && (y <= ypos+height) )
		    	return(v);      /* node found */
		v = NNEXT(v);
	}
	return(NULL);		/* no node found */
} 


/*  Check the graph consistency
 *  ---------------------------
 *  A serious problem is that subgraphs may not have any nodes.
 *  This leads to confusion if such a subgraph is folded.
 *  Thus, for such subgraphs, we add auxiliary nodes.
 */

void check_graph_consistency()
{
	GNODE v,w;

	v = graphlist;

	while (v) {
		if (NSGRAPH(v)==NULL) {
			if (!silent) {
				FPRINTF(stderr,"\nWarning: Graph %s", 
					(NTITLE(v)?NTITLE(v):""));	
				FPRINTF(stderr," has no nodes.");
				FPRINTF(stderr," I add a node ! \n");
			}
			
			w = nodealloc(  NLABEL(v),
                        		NTEXTMODE(v),
                        		NWIDTH(v),
                        		NHEIGHT(v),
                        		NBORDERW(v),
                        		NSX(v),
                        		NSY(v),
                        		NFOLDING(v),
                        		NCOLOR(v),
                        		NTCOLOR(v),
                        		NBCOLOR(v),
                        		NSHRINK(v),
                        		NSTRETCH(v),
                        		NINFO1(v),
                        		NINFO2(v),
                        		NINFO3(v),
                        		NLEVEL(v),
                        		NSHAPE(v),
					NHORDER(v));
			NTITLE(w) = "artificial node";
			NROOT(w) = v;
			NSGRAPH(v) = nodelist_alloc(w);
		}
		v = NNEXT(v);
	}
}

/*--------------------------------------------------------------------*/


/*  Memory Management lists of GNODE objects 
 *  ========================================
 */

/*  Lists of GNODE objects, if they are not connected via internal
 *  GNODE pointers, use special cons-cells, i.e. GNLIST objects, whose 
 *  heads are GNODE objects. Because some cons-cells are temporary, 
 *  we use a similar memory management as for temporary GNODE objects.
 */


static GNLIST tmpnconslist   = NULL;  /* list of allocated cons cells */
static GNLIST foldnconslist   = NULL; /* list of all. fold cons cells */
static GNLIST ncons_freelist = NULL;  /* list of free cons cells      */


/*  Allocate a GNLIST object
 *  ------------------------
 *  First, we look in the free list, if we have a free node. Otherwise,
 *  we allocate a node from the core memory.
 *  We also set some default values.
 *  These node lists are part of the stable graph representation, i.e.
 *  need not to be freed unless a reload of the graph. Thus we don't
 *  store them in the tmpnconslist.
 */

GNLIST  nodelist_alloc(v)
GNODE v;
{
	GNLIST	h;

	debugmessage("nodelist_alloc","");
	h = (GNLIST)myalloc(sizeof(struct gnlist));
	GNINTERN(h) = NULL;
	GNNODE(h)   = v;
	GNNEXT(h)   = NULL;
	return(h);
}

/*  Allocate a temporary GNLIST object
 *  ----------------------------------
 *  First, we look in the free list, if we have a free node. Otherwise,
 *  we allocate a node from the core memory.
 *  We also set some default values.
 *  These node lists are temporary, thus we store them in the
 *  tmpnconslist, to give them free later.
 */

GNLIST  tmpnodelist_alloc()
{
	GNLIST	h;

	debugmessage("tmpnodelist_alloc","");
	if (ncons_freelist) {
		h = ncons_freelist;
		ncons_freelist = GNINTERN(ncons_freelist);
	}
	else    h = (GNLIST)myalloc(sizeof(struct gnlist));
	GNINTERN(h) = tmpnconslist;
	GNNODE(h)   = NULL;
	GNNEXT(h)   = NULL;
	tmpnconslist = h;
	return(h);
}


/*  Allocate a foldlist GNLIST object
 *  ---------------------------------
 *  First, we look in the free list, if we have a free node. Otherwise,
 *  we allocate a node from the core memory.
 *  We also set some default values.
 *  These node lists are used for the folding action keepers in
 *  folding.c. They live longer than temporary nodes, but are
 *  also temporary, because they are deallocated after folding.
 */

GNLIST  foldnodelist_alloc()
{
	GNLIST	h;

	debugmessage("foldnodelist_alloc","");
	if (ncons_freelist) {
		h = ncons_freelist;
		ncons_freelist = GNINTERN(ncons_freelist);
	}
	else    h = (GNLIST)myalloc(sizeof(struct gnlist));
	GNINTERN(h) = foldnconslist;
	GNNODE(h)   = NULL;
	GNNEXT(h)   = NULL;
	foldnconslist = h;
	return(h);
}



/*  Deallocate all temporary GNLIST objects
 *  --------------------------------------
 */

static void free_nodelists()
{
	GNLIST	h;

	debugmessage("free_nodelists","");
	h = tmpnconslist;
	if (h) {
		while(GNINTERN(h)) h = GNINTERN(h);
		GNINTERN(h) = ncons_freelist;
		ncons_freelist = tmpnconslist;
		tmpnconslist = NULL;
	}
}


/*  Deallocate all fold GNLIST objects
 *  ----------------------------------
 */

void free_foldnodelists()
{
	GNLIST	h;

	debugmessage("free_foldnodelists","");
	h = foldnconslist;
	if (h) {
		while(GNINTERN(h)) h = GNINTERN(h);
		GNINTERN(h) = ncons_freelist;
		ncons_freelist = foldnconslist;
		foldnconslist = NULL;
	}
}


/*  Deallocate GNLIST objects of regions
 *  ------------------------------------
 *  These GNLIST objects should be allocated by nodelist_alloc,
 *  i.e. should not be temporary.
 */

void free_regionnodelist(r)
GNLIST	r;
{
	GNLIST	h;

	debugmessage("free_regionnodelists","");
	h = r;
	if (h) {
		while(GNINTERN(h)) h = GNINTERN(h);
		GNINTERN(h) = ncons_freelist;
		ncons_freelist = r;
	}
}


/*--------------------------------------------------------------------*/


/*  Memory Management for Edges 
 *  ===========================
 */

/*  We distinguish between
 *     1) GEDGE objects as edges from the specification
 *     2) GEDGE objects from the specification that are not visualized
 *        directly. Neverthelesss, we need the attributes of these
 *        edges, thus we create a auxiliary GEDGE object for them. 
 *     3) temporary GEDGE objects, used to calculate a layout.
 *        They can be deleted on each change of layout.
 */

/*  Edge from the specification come into the edgelist.
 *  Temporary edges come into the tmpedgelist.
 *  Free GEDGE objects are collected into the node_freelist. 
 */

int edgeanz = 0;      	       /* Number of edges in edgelist         */

GEDGE edgelist     = NULL;     /* List of all real edges as specified */
GEDGE edgelistend  = NULL;     /* End of this list                    */

GEDGE tmpedgelist   = NULL;     /* list of allocated temporary edges */
static GEDGE edge_freelist = NULL;     /* list of free GEDGE objects        */


/*  Allocate a GEDGE object
 *  -----------------------
 *  First, we look in the free list, if we have a free edge. Otherwise,
 *  we allocate an edge from the core memory.
 *  We also set some default values.
 */

static GEDGE internal_edgealloc()
{
	GEDGE   h;

	debugmessage("internal_edgealloc","");
	if (edge_freelist) {
		h = edge_freelist;
		edge_freelist = EINTERN(edge_freelist);
	}
	else {
	 	h = (GEDGE) myalloc(sizeof(struct gedge));
#ifdef DEBUG
		act_alloc_edges++;
		PRINTF("Alloc Summary: %ld GEDGEs allocated\n",act_alloc_edges);
#endif
	}

	ESTART(h)     	= NULL;
	EEND(h)       	= NULL;
	ESTARTX(h)    	= 0;
	ESTARTY(h)    	= 0;
	ETBENDX(h)    	= 0;
	ETBENDY(h)    	= 0;
	EBBENDX(h)    	= 0;
	EBBENDY(h)    	= 0;
	EENDX(h)      	= 0;
	EENDY(h)      	= 0;
	EORI(h)		= NO_ORI;
	EORI2(h)	= NO_ORI;
	ELABEL(h)  	= NULL;
	EART(h)       	= 'U';
	EANCHOR(h)	= 0;
	EINVISIBLE(h)	= 0;
	EWEIGHTS(h)   	= 0;
	EWEIGHTP(h)   	= 0;
	ENEXT(h)	= NULL;
	EINTERN(h)	= NULL;
	return(h);
}


/*  Allocate a real edge 
 *  --------------------
 *  and update the edgelist. These edges are specified, e.g.
 *  as edge: { ... }
 */

GEDGE edgealloc(label,lstyle,thick,class,prio,ecolor,arrows,horder)
char 	*label;
int	lstyle,thick,class,prio,ecolor,arrows,horder;
{
	GEDGE   h;
 
	debugmessage("edgealloc","");
	h = internal_edgealloc();

	ELABEL(h)	= label;
	ELSTYLE(h)    	= lstyle;
	ETHICKNESS(h)	= thick;
	ECLASS(h)	= class;
	EPRIO(h)	= prio;
	EHORDER(h)	= horder;
	ECOLOR(h)	= ecolor;
	EARROWSIZE(h)	= arrows;
	EBEFORE(h)	= edgelistend;

	if (edgelistend) ENEXT(edgelistend) = h;
	edgelistend     = h;
	edgeanz++;
	if (edgelist == NULL) edgelist = h;
	return(h);
}




/*  Allocate a temporary GEDGE object 
 *  ---------------------------------
 *  and update the tmpedgelist. These are edges to dummy nodes or
 *  to labels.
 */


GEDGE	tmpedgealloc(lstyle,thick,class,prio,ecolor,arrows,horder)
int	lstyle,thick,class,prio,ecolor,arrows,horder;
{
	GEDGE	h;

	debugmessage("tmpedgealloc","");
	h = internal_edgealloc();

	ELSTYLE(h)    	= lstyle;
	ETHICKNESS(h)	= thick;
	ECLASS(h)	= class;
	EPRIO(h)	= prio;
	EHORDER(h)	= horder;
	ECOLOR(h)	= ecolor;
	EARROWSIZE(h)	= arrows;
	EBEFORE(h)	= NULL;

	EINTERN(h) = tmpedgelist;
	tmpedgelist = h;
	return(h);
}


/*  Deallocate all temporary GEDGE objects
 *  --------------------------------------
 */

static void free_tmpedges()
{
	GEDGE	h;

	debugmessage("free_tmpedges","");
	h = tmpedgelist;
	if (h) {
		while (EINTERN(h)) h = EINTERN(h);
		EINTERN(h) = edge_freelist;
		edge_freelist = tmpedgelist;
		tmpedgelist = NULL;
	}
}


/*--------------------------------------------------------------------*/

/*  Memory Management lists of GEDGE objects 
 *  ========================================
 */

/*  Lists of GEDGE objects are used in adjacency lists.
 *  We use special cons-cells, i.e. ADJEDGE objects, whose 
 *  heads are GEDGE objects. Because these cons-cells are temporary, 
 *  we use a similar memory management as for temporary GNODE objects.
 *
 *  Further, we have one nontemporary list of edges that contains the
 *  default connections as specified by `near_edge'. This is the
 *  near_edge_list.
 */


/* for stable default connections: */

ADJEDGE near_edge_list = NULL;	/* list of default connections */

/* for temporaries: */

static ADJEDGE tmpeconslist   = NULL;  /* list of allocated cons cells */
static ADJEDGE econs_freelist = NULL;  /* list of free cons cells      */


/*  Insert a near edge into near_edge_list
 *  --------------------------------------
 *  First, we look in the free list, if we have a free cell. Otherwise,
 *  we allocate a cell from the core memory.
 */

void near_edge_insert(e)
GEDGE e;
{
	ADJEDGE	h;

	debugmessage("near_edge_insert","");
	if (econs_freelist) {
		h = econs_freelist;
		econs_freelist = AINTERN(econs_freelist);
	}
	else    h = (ADJEDGE)myalloc(sizeof(struct adjedge));
	AKANTE(h) = e;
	ANEXT(h) = AINTERN(h) = near_edge_list;
	near_edge_list = h;
}


/*  Allocate a ADJEDGE object
 *  -------------------------
 *  First, we look in the free list, if we have a free cell. Otherwise,
 *  we allocate a cell from the core memory.
 */

static ADJEDGE  edgelist_alloc()
{
	ADJEDGE	h;

	debugmessage("edgelist_alloc","");
	if (econs_freelist) {
		h = econs_freelist;
		econs_freelist = AINTERN(econs_freelist);
	}
	else    h = (ADJEDGE)myalloc(sizeof(struct adjedge));
	AINTERN(h) = tmpeconslist;
	tmpeconslist = h;
	return(h);
}


/*  Add a new edge to the predecessors of a node
 *  --------------------------------------------
 */

ADJEDGE prededgealloc(node,edge)
GNODE	node;	
GEDGE   edge;
{
	ADJEDGE e;
        
	/* assert((EEND(edge)==node)); */
	e = edgelist_alloc();
	AKANTE(e)	= edge;
	ANEXT(e)  	= NPRED(node);
	NPRED(node) 	= e;
	return(e);
}


/*  Add a new cons cell to the successors of a node
 *  -----------------------------------------------
 */

ADJEDGE succedgealloc(node,edge)
GNODE   node;        
GEDGE   edge;
{
	ADJEDGE e;   
        
	/* assert((ESTART(edge)==node)); */
	e = edgelist_alloc();
	AKANTE(e)	= edge;
	ANEXT(e) 	= NSUCC(node);
	NSUCC(node)	= e;
	return(e); 
}


/*  Deallocate all temporary ADJEDGE objects
 *  ----------------------------------------
 */

static void free_edgelists()
{
	ADJEDGE	h;

	debugmessage("free_edgelists","");
	h = tmpeconslist;
	if (h) {
		while(AINTERN(h)) h = AINTERN(h);
		AINTERN(h) = econs_freelist;
		econs_freelist = tmpeconslist;
		tmpeconslist = NULL;
	}
}



/*--------------------------------------------------------------------*/

/*  Memory Management for CONNECT objects 
 *  =====================================
 */

/*  CONNECT objects are annotations of GNODE objects.
 *  They indicate that two nodes must be directly neigboured during
 *  the layout. This occurs if nodes are at the same level connected.
 *  E.g.     
 *          /    |    \      this situation is layouted as if we had only
 *         A<----B---->C     one node B. The connections of B are A and C.
 */

static CONNECT 	connectlist      = NULL;   /* list of alloc. connect cells */
static CONNECT 	connect_freelist = NULL;   /* list of free   connect cells */


/*  Allocate a CONNECT object
 *  -------------------------
 *  First, we look in the free list, if we have a free cell. Otherwise,
 *  we allocate a cell from the core memory.
 *  The new connect node is inserted into the connection field of the 
 *  GNODE node.
 */

CONNECT	connectalloc(node)
GNODE	node;
{
	CONNECT	h;

	debugmessage("connectalloc","");
	if (connect_freelist) {
		h = connect_freelist;
		connect_freelist = CINTERN(connect_freelist);
	}
	else 	h = (CONNECT)myalloc(sizeof(struct connect));
	CTARGET(h) 	= NULL;
	CEDGE(h)	= NULL;
	CTARGET2(h)	= NULL;
	CEDGE2(h)	= NULL;
	CINTERN(h) 	= connectlist;
	connectlist	= h;
	NCONNECT(node) 	= h;
	return(h);
}


/*  Deallocate all temporary CONNECT objects
 *  ----------------------------------------
 */

static void free_connect()
{
	CONNECT	h;
		
	debugmessage("free_connect","");
	h = connectlist;
	if (h) {
		while(CINTERN(h)) h = CINTERN(h);
		CINTERN(h) = connect_freelist;
		connect_freelist = connectlist;
		connectlist = NULL;
	}
}


/*--------------------------------------------------------------------*/

/*  Memory Management for DLLIST objects 
 *  ====================================
 */

/*  To have a good layout, we calculate the number of crossings of edges
 *  and try to minimize them. For the calculation of crossings, we need
 *  double linked lists of nodes (see step2.c) representing the upper
 *  list of touched nodes and the lower list of touched nodes. Because
 *  nodes may have multible occurences in these lists, we use the special
 *  data structure DLLIST. 
 *  We reuse the DSUCC field to keep the list of free dllist nodes.
 *  But THIS dllist_freelist is NOT double linked.
 */

static DLLIST	dllist_freelist  = NULL;     /* list of free dllist nodes */


/*  Allocate a DLLIST object
 *  ------------------------
 *  First, we look in the free list, if we have a free cell. Otherwise,
 *  we allocate a cell from the core memory.
 *  The successor is always NULL, because we append at the end of the
 *  list. pred is the predecessor.
 */


DLLIST 	dllist_alloc(node,pred)
GNODE  node;
DLLIST pred;
{
	DLLIST	h;

	debugmessage("dllist_alloc","");
	if (dllist_freelist) {
		h = dllist_freelist;
		dllist_freelist = DSUCC(dllist_freelist);
	}
	else    h = (DLLIST)myalloc(sizeof(struct dllist));
	DNODE(h) = node;
	DPRED(h) = pred;
	DSUCC(h) = NULL;
	return(h);
}


/*  Deallocate one DLLIST objects
 *  -----------------------------
 *  The crossing algorithm is stable enough such that after calculation
 *  of crossings all DLLIST elements are given free by this function.
 *  Thus we don't need any additional memory management.
 */

void	dllist_free(x)
DLLIST x;
{
	debugmessage("dllist_free","");
       	DSUCC(x) = dllist_freelist;
       	dllist_freelist = x;
}


/*  Deallocate a list of DLLIST objects
 *  -----------------------------------
 */

void	dllist_free_all(x)
DLLIST x;
{
	DLLIST h;

	debugmessage("dllist_free","");
	if (x) {
		h = x;
		while (DSUCC(h)) h = DSUCC(h);
        	DSUCC(h) = dllist_freelist;
        	dllist_freelist = x;
	}
}



/*--------------------------------------------------------------------*/


/*  Deallocation of all temporary lists 
 *  ===================================
 */


void	free_all_lists()
{
	free_tmpnodes();
	free_tmpedges();
        free_nodelists();
	free_edgelists();
	free_connect();
}


/*  Reinitialization of all struct keeping lists 
 *  --------------------------------------------
 */

void    reinit_all_lists()
{
	ufoldstart  = NULL;
        foldstart   = NULL;
        foldstops   = NULL;
        f_subgraphs = NULL;
        uf_subgraphs= NULL;
	invis_nodes	 = NULL;
	labellist	 = NULL;
	labellistend 	 = NULL;
	dummylist	 = NULL;

	nodeanz 	 = 0; 
	dummyanz  	 = 0;
	nodelist	 = NULL;
	nodelistend	 = NULL;
	graphlist        = NULL;
	graphlistend     = NULL;
	tmpnodelist	 = NULL;
	node_freelist    = NULL;

	tmpnconslist     = NULL;
	ncons_freelist   = NULL;

	edgeanz 	 = 0;
	edgelist     	 = NULL;
	edgelistend  	 = NULL;
	tmpedgelist   	 = NULL;
	edge_freelist    = NULL;

	near_edge_list   = NULL;
	tmpeconslist     = NULL;
	econs_freelist   = NULL;

	connectlist	 = NULL;
	connect_freelist = NULL;

	dllist_freelist	 = NULL;
}


