/* SCCS-info %W% %E% */

/*--------------------------------------------------------------------*/
/*								      */
/*		VCG : Visualization of Compiler Graphs		      */
/*		--------------------------------------		      */
/*								      */
/*   file:	   X11devla.c					      */
/*   version:	   1.00.00					      */
/*   creation:	   9.4.93					      */
/*   author:	   I. Lemke  (...-Version 0.99.99)		      */
/*		   G. Sander (Version 1.00.00-...)		      */
/*		   Universitaet des Saarlandes, 66041 Saarbruecken    */
/*		   ESPRIT Project #5399 Compare 		      */
/*   description:  Device driver for X11, Layout Dialog Box	      */
/*   status:	   in work					      */
/*								      */
/*--------------------------------------------------------------------*/

#ifndef lint
static char *id_string="$Id: X11devla.c,v 1.6 1994/08/05 12:13:25 sander Exp $";
#endif

/*
 *   Copyright (C) 1993, 1994 by Georg Sander, Iris Lemke, and
 *                               the Compare Consortium 
 *
 *  This program and documentation is free software; you can redistribute 
 *  it under the terms of the  GNU General Public License as published by
 *  the  Free Software Foundation;  either version 2  of the License,  or
 *  (at your option) any later version.
 *
 *  This  program  is  distributed  in  the hope that it will be useful,
 *  but  WITHOUT ANY WARRANTY;  without  even  the  implied  warranty of
 *  MERCHANTABILITY  or  FITNESS  FOR  A  PARTICULAR  PURPOSE.  See  the
 *  GNU General Public License for more details.
 *
 *  You  should  have  received a copy of the GNU General Public License
 *  along  with  this  program;  if  not,  write  to  the  Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  The software is available per anonymous ftp at ftp.cs.uni-sb.de.
 *  Contact  sander@cs.uni-sb.de  for additional information.
 */


/* 
 * $Log: X11devla.c,v $
 * Revision 1.6  1994/08/05  12:13:25  sander
 * Treelayout added. Attributes "treefactor" and "spreadlevel" added.
 * Scaling as abbreviation of "stretch/shrink" added.
 *
 * Revision 1.5  1994/08/03  13:58:44  sander
 * Horizontal order mechanism changed.
 * Attribute horizontal_order for edges added.
 *
 * Revision 1.4  1994/06/07  14:09:59  sander
 * Splines implemented.
 * HP-UX, Linux, AIX, Sun-Os, IRIX compatibility tested.
 * The tool is now ready to be distributed.
 *
 * Revision 1.3  1994/05/17  16:39:10  sander
 * attribute node_align added to allow nodes to be centered in the levels.
 *
 * Revision 1.2  1994/05/16  08:56:03  sander
 * shape attribute (boxes, rhombs, ellipses, triangles) added.
 *
 * Revision 1.1  1994/05/05  08:20:30  sander
 * Initial revision
 *
 */

/************************************************************************
 * Device driver for X11: Layout Dialog Box    
 * ----------------------------------------
 *
 * This module contains the management of the dialog box that appears
 * if we want to change the layout.
 * 
 * This file provides the following functions:
 * ------------------------------------------
 *
 *    x11_init_layout_dialog(rd,rs)      initialize the dialogbox.
 *    x11_destroy_layout_dialog(rd,rs)   destroys   the dialogbox.
 *
 *    x11_print_layout_dialog()
 *
 *    This function opens the dialog box that allows to change the
 *    layout parameters. On successful exit (Okay-button), the
 *    layout parameters are updated, such that they become valid
 *    at the next relayout.  On every exit, the dialog box is
 *    closed. The function returns 1, if the Okay-button was pressed.
 *
 ************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "globals.h"

#ifdef X11 

#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/Xproto.h>
#include <ctype.h>
#include <math.h>
#include "main.h"
#include "grammar.h"
#include "alloc.h"
#include "steps.h"
#include "X11devla.h"


/*--------------------------------------------------------------------*/

#undef EVENT_DEBUG


/* Types 
 * =====
 */

/*   Selection points: On/Off-Buttons, Radio/Buttons, Touch Buttons 
 *   Radio-Buttons are On-buttons, i.e. they don't go off, if we retouch
 *   them. Further, all radio buttons of one section have the same glob.
 */

typedef struct gs_selection_point {
	Window w;			/* Window of the point             */
	char   *t;			/* Text of the point               */
	int    x;			/* X-Position                      */
	int    y;			/* Y-Position                      */
	int    is;			/* actual value                    */
	int    sel;			/* value, if the point is selected */
	int    *glob;			/* corresponding global value      */
	char   key;			/* key action                      */
} GS_SEL_POINT;


/*  Horizontal integer scroll bars
 */


typedef struct gs_scroll_bar {
	Window ws;			/* Window of the bar itself        */
	Window wt;			/* Window of the scroll thumb      */
	char   *t;			/* Text of the scroll bar          */
	char   *f;			/* Format of the value             */
	int    xt;			/* X-Position of the text          */
	int    xv;			/* X-Position of the value         */
	int    xb;			/* X-Position of the bar           */
	int    y;			/* Y-Position                      */
	int    len;			/* Length of scrollbar             */
	int    is;			/* actual value                    */
	int    *glob;			/* corresponding global value      */
#ifdef ANSI_C
	void   (*rf)(int);		/* read function		   */
	void   (*sf)(void);		/* set function		   	   */
#else 
	void   (*rf)();
	void   (*sf)();
#endif
} GS_SCROLL_BAR;




/* Prototypes
 * ==========
 */

void gs_exit            _PP((int x));

static void   allocateDialogBox         _PP((void));
static Window allocate_selection_point  _PP((int x,int y));
static Window allocate_button		_PP((int x,int y));
static Window allocate_general_button   _PP((int x,int y,int w,int h));
static Window allocate_scroll_bar	_PP((int x,int y,int len));

static void draw_dialog_box     _PP((void));
static void draw_text           _PP(( int x, int y, char *m));
static void draw_underl_text    _PP(( int x, int y, char *m));
static void draw_dialog_point   _PP((Window w, int color));
static void draw_sizes		_PP((void));
static void normalize_point     _PP((Window w));


#ifdef EVENT_DEBUG
static void prologue    _PP((XEvent *eventp, char *event_name));
#else
#define prologue(a,b)
#endif

static void     dialog_main_loop        _PP((void));
static void     do_KeyPress             _PP((XEvent *eventp));
static void     do_ButtonPress          _PP((XEvent *eventp));
static void     do_ButtonRelease        _PP((XEvent *eventp));
static void     do_EnterNotify          _PP((XEvent *eventp));
static void     do_LeaveNotify          _PP((XEvent *eventp));
static void     do_Expose               _PP((XEvent *eventp));
static void     do_ConfigureNotify      _PP((XEvent *eventp));
static void     do_MotionNotify         _PP((XEvent *eventp));

static void 	read_40sc_scrollbar	_PP((int val,int i));
static void 	set_40sc_scrollbar	_PP((int i));
static void 	read_5sc_scrollbar	_PP((int val,int i));
static void 	read_5in_scrollbar	_PP((int val,int i));
static void 	set_5sc_scrollbar	_PP((int i));

static void 	read_downfacscrollbar	_PP((int val));
static void 	set_downfacscrollbar	_PP((void));
static void 	read_upfacscrollbar	_PP((int val));
static void 	set_upfacscrollbar	_PP((void));
static void 	read_nearfacscrollbar	_PP((int val));
static void 	set_nearfacscrollbar	_PP((void));
static void 	read_xbasescrollbar	_PP((int val));
static void 	set_xbasescrollbar	_PP((void));
static void 	read_ybasescrollbar	_PP((int val));
static void 	set_ybasescrollbar	_PP((void));
static void 	read_xspacescrollbar	_PP((int val));
static void 	set_xspacescrollbar	_PP((void));
static void 	read_xlspacescrollbar	_PP((int val));
static void 	set_xlspacescrollbar	_PP((void));
static void 	read_yspacescrollbar	_PP((int val));
static void 	set_yspacescrollbar	_PP((void));
static void 	read_baryscrollbar	_PP((int val));
static void 	set_baryscrollbar	_PP((void));
static void 	read_centerscrollbar	_PP((int val));
static void 	set_centerscrollbar	_PP((void));
static void 	read_mediumscrollbar	_PP((int val));
static void 	set_mediumscrollbar	_PP((void));
static void 	read_bendscrollbar	_PP((int val));
static void 	set_bendscrollbar	_PP((void));
static void 	read_splinescrollbar	_PP((int val));
static void 	set_splinescrollbar	_PP((void));


/* Positions and Sizes
 * ===================
 */

/* Size of the dialog box */

#define DIAL_W   570
#define DIAL_H   460

/* Size of a normal point */

#define POINT_W  10 

/* Size of a normal button */

#define BUTTON_W   130

/* Sizes of a scrollbar and its thumb */

#define SCROLL_W  80  
#define SCROLLBAR_LEN 15

/* Position of the edge label phase */

#define ELPH_POS 165

/* Position of Layoutfactors: */

#define LFAC_POS ELPH_POS+20 

/* Position of Layoutspeeds */

#define LSPE_POS LFAC_POS+167

/* Position of Partition algorithms */

#define LPAR_POS LSPE_POS+23



/* Variables
 * =========
 */

/* Dialog handling: On success (i.e. Okay-Button), success becomes 1 */

static int dialog_notready = 1;
static int dialog_success  = 0;


/*  On-Off-points are buttons that switch their state if we select them */

#define MAX_ONOFF_POINTS 7 
static GS_SEL_POINT on_off_point[MAX_ONOFF_POINTS] = {

/* window,  text          , xpos, ypos, isval, selval , globvar                , key */ 
{  0     ,  "edge  labels",    5,   36,     0,      1 , &G_displayel           , 'l' },   
{  0     ,  "dirty labels",    5,   54,     0,      1 , &G_dirtyel             , 'd' },   
{  0     ,  "near edges"  ,    5,   72,     0,      1 , &near_edge_layout      , ' ' },   
{  0     ,  "double edges",    5,   90,     0,      0 , &summarize_double_edges, ' ' },   
{  0     ,  "finetuning"  ,    5,  108,     0,      1 , &fine_tune_layout      , ' ' },
{  0     ,  "splines"     ,    5,  126,     0,      1 , &G_spline              , ' ' },
{  0     ,  "optimize"    ,  380,   72,     0,      1 , &local_unwind          , 'o' }
};



/*  Touch-points are buttons that don't their change state if we select them. 
 *  Instead, some other actions are performed. They don't correspond to 
 *  global variables.
 */

static int dummy;

#define MAX_TOUCH_POINTS 4 
static GS_SEL_POINT touch_point[MAX_TOUCH_POINTS] = {

/* window,  text       , xpos, ypos,     isval, selval, globvar , key */ 
{  0     ,  "slow&nice",  140, LSPE_POS,   0,      1  , &dummy  , 's' },   
{  0     ,  "normal"   ,  260, LSPE_POS,   0,      1  , &dummy  , 'n' },   
{  0     ,  "medium"   ,  360, LSPE_POS,   0,      1  , &dummy  , 'm' },   
{  0     ,  "fast&ugly",  460, LSPE_POS,   0,      1  , &dummy  , 'f' }
};

#define SLOWSPEED 0
#define NORMSPEED 1
#define MEDISPEED 2
#define FASTSPEED 3


/*  Radio buttons are partitioned into those that have the same global 
 *  variable. If a radio-button is selected, all other radio buttons 
 *  of the same partition became unselected. 
 */


#define MAX_RADIO_POINTS 24 
static GS_SEL_POINT radio_point[MAX_RADIO_POINTS] = {

/* window,  text           , xpos,     ypos, isval,  selval     , globvar             , key */ 
{  0     ,  "barycenter"   ,  380,       36,    0,      0       , &crossing_heuristics, ' ' },   
{  0     ,  "mediancenter" ,  380,       54,    0,      1       , &crossing_heuristics, ' ' },   
{  0     ,  "top to bottom",  200,       36,    0, TOP_TO_BOTTOM, &G_orientation,       '1' },   
{  0     ,  "bottom to top",  200,       54,    0, BOTTOM_TO_TOP, &G_orientation,       '2' },   
{  0     ,  "left to right",  200,       72,    0, LEFT_TO_RIGHT, &G_orientation,       '3' },   
{  0     ,  "right to left",  200,       90,    0, RIGHT_TO_LEFT, &G_orientation,       '4' },   

{  0     ,  "top"          ,  380,      108   , 0, AL_TOP       , &G_yalign,            '7' },
{  0     ,  "center"        , 380,      126   , 0, AL_CENTER    , &G_yalign,            '8' },
{  0     ,  "bottom"        , 380,      144   , 0, AL_BOTTOM    , &G_yalign,            '9' },
{  0     ,  "after folding",  160, ELPH_POS   , 0,      0       , &edge_label_phase,    ' ' },
{  0  ,"after partitioning",  340, ELPH_POS   , 0,      1       , &edge_label_phase,    ' ' },
{  0     ,  "normal"       ,    5, LPAR_POS+18, 0,      0       , &layout_flag,         ' ' },   
{  0     ,  "maxdepth"     ,    5, LPAR_POS+36, 0,      1       , &layout_flag,         ' ' },   
{  0     ,  "mindepth"     ,    5, LPAR_POS+54, 0,      2       , &layout_flag,         ' ' },   
{  0     ,  "minbackward"  ,  120, LPAR_POS+18, 0,      3       , &layout_flag,         ' ' },   
{  0     ,  "maxdegree"    ,  120, LPAR_POS+36, 0,     11       , &layout_flag,         ' ' },   
{  0     ,  "mindegree"    ,  120, LPAR_POS+54, 0,     10       , &layout_flag,         ' ' },   
{  0     ,  "maxindegree"  ,  270, LPAR_POS+18, 0,      7       , &layout_flag,         ' ' },   
{  0     ,  "maxoutdegree" ,  270, LPAR_POS+36, 0,      9       , &layout_flag,         ' ' },   
{  0     ,  "maxdepthslow" ,  270, LPAR_POS+54, 0,      4       , &layout_flag,         ' ' },   
{  0     ,  "minindegree"  ,  420, LPAR_POS+18, 0,      6       , &layout_flag,         ' ' },   
{  0     ,  "minoutdegree" ,  420, LPAR_POS+36, 0,      8       , &layout_flag,         ' ' },   
{  0     ,  "mindepthslow" ,  420, LPAR_POS+54, 0,      5       , &layout_flag,         ' ' },
{  0     ,  "tree layout"  ,    5, LPAR_POS+72, 0, TREE_LAYOUT  , &layout_flag,         ' ' }   

};


#define AFTER_FOLD   9
#define AFTER_PART   10
#define TREE_POINT   23


/*  Horizontal integer scrollbars are used to change a value by the read and 
 *  set functions. Here, these values must be integer values.
 */


#define MAX_HSCROLLBARS 13 

#ifdef ANSI_C
static GS_SCROLL_BAR hscrollbar[MAX_HSCROLLBARS] = {

    {  0, 0, "Down-factor:" ,"%d%", 5  , 120, 160, LFAC_POS+18 , SCROLL_W,
		0, &layout_downfactor  , &read_downfacscrollbar, &set_downfacscrollbar },
    {  0, 0, "Up  -factor:" ,"%d%", 5  , 120, 160, LFAC_POS+36 , SCROLL_W,
		0, &layout_upfactor    , &read_upfacscrollbar  , &set_upfacscrollbar   },
    {  0, 0, "Near-factor:" ,"%d%", 5  , 120, 160, LFAC_POS+54 , SCROLL_W,
		0, &layout_nearfactor  , &read_nearfacscrollbar, &set_nearfacscrollbar },
    {  0, 0, "X-Base:"      ,"%d%", 5  , 120, 160, LFAC_POS+72 , SCROLL_W,
		0, &G_xbase            , &read_xbasescrollbar  , &set_xbasescrollbar   },
    {  0, 0, "Y-Base:"      ,"%d%", 5  , 120, 160, LFAC_POS+90 , SCROLL_W,
		0, &G_ybase            , &read_ybasescrollbar  , &set_ybasescrollbar   },
    {  0, 0, "X-Space:"     ,"%d%", 5  , 120, 160, LFAC_POS+108, SCROLL_W,
		0, &G_xspace           , &read_xspacescrollbar , &set_xspacescrollbar  },
    {  0, 0, "Y-Space:"     ,"%d%", 5  , 120, 160, LFAC_POS+126, SCROLL_W,
		0, &G_yspace           , &read_yspacescrollbar , &set_yspacescrollbar  },
    {  0, 0, "XL-Space:"    ,"%d%", 5  , 120, 160, LFAC_POS+144, SCROLL_W,
		0, &G_dspace           , &read_xlspacescrollbar , &set_xlspacescrollbar  },
    {  0, 0, "Crossingred.:","%d%", 280, 410, 450, LFAC_POS+72 , SCROLL_W,
		0, &max_baryiterations , &read_baryscrollbar   , &set_baryscrollbar    },
    {  0, 0, "Mediumshifts:","%d%", 280, 410, 450, LFAC_POS+90 , SCROLL_W,
		0, &max_mediumshifts   , &read_mediumscrollbar , &set_mediumscrollbar  },
    {  0, 0, "Centershifts:","%d%", 280, 410, 450, LFAC_POS+108, SCROLL_W,
		0, &max_centershifts   , &read_centerscrollbar , &set_centerscrollbar  },
    {  0, 0, "Bendingsred.:","%d%", 280, 410, 450, LFAC_POS+126, SCROLL_W,
		0, &max_edgebendings   , &read_bendscrollbar   , &set_bendscrollbar    },
    {  0, 0, "Splinefactor:","%d%", 280, 410, 450, LFAC_POS+144, SCROLL_W,
		0, &G_flat_factor      , &read_splinescrollbar , &set_splinescrollbar  }
};

#else

static GS_SCROLL_BAR hscrollbar[MAX_HSCROLLBARS] = {

    {  0, 0, "Down-factor:" ,"%d%", 5  , 120, 160, LFAC_POS+18 , SCROLL_W,
		0, &layout_downfactor  , read_downfacscrollbar, set_downfacscrollbar },
    {  0, 0, "Up  -factor:" ,"%d%", 5  , 120, 160, LFAC_POS+36 , SCROLL_W,
		0, &layout_upfactor    , read_upfacscrollbar  , set_upfacscrollbar   },
    {  0, 0, "Near-factor:" ,"%d%", 5  , 120, 160, LFAC_POS+54 , SCROLL_W,
		0, &layout_nearfactor  , read_nearfacscrollbar, set_nearfacscrollbar },
    {  0, 0, "X-Base:"      ,"%d%", 5  , 120, 160, LFAC_POS+72 , SCROLL_W,
		0, &G_xbase            , read_xbasescrollbar  , set_xbasescrollbar   },
    {  0, 0, "Y-Base:"      ,"%d%", 5  , 120, 160, LFAC_POS+90 , SCROLL_W,
		0, &G_ybase            , read_ybasescrollbar  , set_ybasescrollbar   },
    {  0, 0, "X-Space:"     ,"%d%", 5  , 120, 160, LFAC_POS+108, SCROLL_W,
		0, &G_xspace           , read_xspacescrollbar , set_xspacescrollbar  },
    {  0, 0, "Y-Space:"     ,"%d%", 5  , 120, 160, LFAC_POS+126, SCROLL_W,
		0, &G_yspace           , read_yspacescrollbar , set_yspacescrollbar  },
    {  0, 0, "XL-Space:"    ,"%d%", 5  , 120, 160, LFAC_POS+144, SCROLL_W,
		0, &G_dspace           , read_xlspacescrollbar ,set_xlspacescrollbar  },
    {  0, 0, "Crossingred.:","%d%", 280, 410, 450, LFAC_POS+72 , SCROLL_W,
		0, &max_baryiterations , read_baryscrollbar   , set_baryscrollbar    },
    {  0, 0, "Mediumshifts:","%d%", 280, 410, 450, LFAC_POS+90 , SCROLL_W,
		0, &max_mediumshifts   , read_mediumscrollbar , set_mediumscrollbar  },
    {  0, 0, "Centershifts:","%d%", 280, 410, 450, LFAC_POS+108, SCROLL_W,
		0, &max_centershifts   , read_centerscrollbar , set_centerscrollbar  },
    {  0, 0, "Bendingsred.:","%d%", 280, 410, 450, LFAC_POS+126, SCROLL_W,
		0, &max_edgebendings   , read_bendscrollbar   , set_bendscrollbar    },
    {  0, 0, "Splinefactor:","%d%", 280, 410, 450, LFAC_POS+144, SCROLL_W,
		0, &G_flat_factor      , read_splinescrollbar , set_splinescrollbar  }
};
#endif

#define DOWNFAC    0
#define UPFAC      1
#define NEARFAC    2
#define XBASESCR   3
#define YBASESCR   4
#define XSPACESCR  5
#define YSPACESCR  6
#define XLSPACESCR 7
#define BARYSCR    8
#define MEDIUMSCR  9
#define CENTERSCR 10
#define BENDSCR   11
#define SPLINESCR 12




/* Root displays */

static Display *root_display;	/* Root display          */
static int      root_screen;	/* and its screen number */


/* Windows */

static Window	dialog_window;

static Window   okay_button;
static Window   cancel_button;



/* Cursors */

static Cursor scrollCursor; 

/* Mouse Buttons */

#define G_MOUSE_LEFT  1
#define G_MOUSE_RIGHT 3


/* the panel font */

extern char Xfontname[512];
static  XFontStruct *dialog_font = NULL;

/* Flag wheter the dialog_box is initialized */

static int dialog_box_there = 0;

/* Flag whether the actual layout is tree layout */

static int is_tree_layout = 0;
static int mytreefactor;

/*--------------------------------------------------------------------*/
/*  Dialog box initialization and entry points                        */
/*--------------------------------------------------------------------*/

/*  Initialize the dialog box 
 *  -------------------------
 *  This is done once at the beginning.
 */

void 	x11_init_layout_dialog(rd, rs)
Display *rd;
int rs;
{
	if (dialog_box_there) return;
	root_display = rd;
	root_screen  = rs;
	dialog_font = XLoadQueryFont(root_display,Xfontname);
	if (!dialog_font) {
		Fatal_error("Font not found. Please use option -font","");
	}

	allocateDialogBox();

	dialog_box_there = 1;
}



/*   Destroy the dialog box 
 *   ----------------------
 *   This is done once on ExitDevice.
 */

void	x11_destroy_layout_dialog(rd, rs)
Display *rd;
int rs;
{
	if (!dialog_box_there) return;
	XDestroyWindow(root_display,dialog_window);
	dialog_box_there = 0;
}



/*   Allocate the Dialog Box 
 *   -----------------------
 *   Scrollbar-Thumbs are returned in last_tmb;
 */

static Window last_tmb;

static void allocateDialogBox()
{
	Window rret1, rret2;
	XSetWindowAttributes attr;
	unsigned long mask = 0L;
	int i;
	unsigned int j;
	int xpos, ypos;
	XSizeHints hints;

	/* select for all events */
	attr.event_mask = KeyPressMask           | 
			  ButtonPressMask        | ButtonReleaseMask  |
			  EnterWindowMask        | 
                          ExposureMask           | StructureNotifyMask | 
                          SubstructureNotifyMask | 
                          FocusChangeMask        | OwnerGrabButtonMask;

	attr.background_pixel = XWhitePixel(root_display,root_screen);
	attr.border_pixel = XBlackPixel(root_display,root_screen);

	mask |= (CWBackPixel | CWBorderPixel | CWEventMask);

	(void)XQueryPointer(root_display,RootWindow(root_display, root_screen),
		&rret1, &rret2,&xpos, &ypos, &i, &i, &j); 

        dialog_window = XCreateWindow(root_display, 
				      RootWindow(root_display, root_screen), 
				      xpos, ypos,
                           	      DIAL_W, DIAL_H, 
				      Xmyborderwidth, (int)CopyFromParent,
                                      InputOutput, 
				      (Visual *)CopyFromParent,
                           	      mask, &attr);

	hints.flags = PPosition;
	hints.x = xpos;
	hints.y = ypos;

	XStoreName(root_display, dialog_window, "VCG Layout");
	XSetNormalHints(root_display, dialog_window, &hints);

	XDefineCursor(root_display,dialog_window, 
			XCreateFontCursor(root_display,XC_left_ptr));

	scrollCursor = XCreateFontCursor(root_display,XC_sb_h_double_arrow);

	for (i=0; i<MAX_ONOFF_POINTS; i++)
		on_off_point[i].w = allocate_selection_point(
					on_off_point[i].x, 
					on_off_point[i].y-10);
	for (i=0; i<MAX_TOUCH_POINTS; i++)
		touch_point[i].w = allocate_selection_point(
					touch_point[i].x,
				    	touch_point[i].y-10);
	for (i=0; i<MAX_RADIO_POINTS; i++)
		radio_point[i].w = allocate_selection_point(
					radio_point[i].x,
					radio_point[i].y-10);

	for (i=0; i<MAX_HSCROLLBARS; i++) {
		hscrollbar[i].ws = allocate_scroll_bar(hscrollbar[i].xb,
						       hscrollbar[i].y-10, 
						       hscrollbar[i].len);
		hscrollbar[i].wt = last_tmb;
	}
	

	cancel_button = allocate_button(280,LFAC_POS-10);
	okay_button   = allocate_button(420,LFAC_POS-10);
}



/*  Allocate a selection point (button or radio button)
 *  ---------------------------------------------------
 */

static Window allocate_selection_point(x,y)
int x, y;
{
	return(allocate_general_button(x,y,POINT_W,POINT_W));
}



/*  Allocate an normal button
 *  -------------------------
 */

static Window allocate_button(x,y)
int x, y;
{
	return(allocate_general_button(x,y,BUTTON_W,25));
}



/*  Allocate a general button or point
 *  ----------------------------------
 */

static Window allocate_general_button(x,y,w,h)
int x, y, w, h;
{
	Window ww;
	XSetWindowAttributes attr;
	unsigned long mask = 0L;

	/* select for all events */
	attr.event_mask = KeyPressMask           | 
			  ButtonPressMask        | ButtonReleaseMask   |
			  EnterWindowMask        | LeaveWindowMask     |
                          ExposureMask           | StructureNotifyMask | 
                          SubstructureNotifyMask | 
                          FocusChangeMask        | OwnerGrabButtonMask;

	attr.background_pixel = XWhitePixel(root_display,root_screen);
	attr.border_pixel = XBlackPixel(root_display,root_screen);

	mask |= (CWBackPixel | CWBorderPixel | CWEventMask);

	ww = XCreateWindow(root_display,
			dialog_window,
			x,y,
			w,h,
			1, (int)CopyFromParent,
			InputOutput,
			(Visual *)CopyFromParent,
			mask, &attr);

	return(ww);
}


/*  Allocate a scroll bar
 *  ---------------------
 */

static Window allocate_scroll_bar(x,y,len)
int x, y, len;
{
	Window w, wt;
	XSetWindowAttributes attr;
	unsigned long mask = 0L;

        /* select for all events */
        attr.event_mask = KeyPressMask           | ButtonMotionMask   |
                          ButtonPressMask        | ButtonReleaseMask  |
                          EnterWindowMask        | LeaveWindowMask    |
                          ExposureMask           | ResizeRedirectMask |
                          SubstructureNotifyMask |
                          FocusChangeMask        | OwnerGrabButtonMask;

        attr.background_pixel = XWhitePixel(root_display,root_screen);
        attr.border_pixel = XBlackPixel(root_display,root_screen);

        mask |= (CWBackPixel | CWBorderPixel | CWEventMask);

        w = XCreateWindow (root_display, dialog_window, x,y,
                           len, 8,
                           1, (int)CopyFromParent,
                           InputOutput,
                           (Visual *)CopyFromParent,
                           mask, &attr);
        wt = XCreateSimpleWindow (root_display, w, 0,0,
                                    SCROLLBAR_LEN, 6,
                                    1,
                                    XWhitePixel(root_display,root_screen),
                                    XBlackPixel(root_display,root_screen));

	XDefineCursor(root_display,w,scrollCursor); 
	
	last_tmb = wt;
	return(w);
}


/*--------------------------------------------------------------------*/
/*   Entry point for the dialog                                       */
/*--------------------------------------------------------------------*/

static int box_exposed;

int x11_print_layout_dialog()
{
	XEvent event;
	int i;

	mytreefactor = (int)(tree_factor * 100);

	box_exposed = 0;

	for (i=0; i<MAX_ONOFF_POINTS; i++)
		on_off_point[i].is = *(on_off_point[i].glob);

	for (i=0; i<MAX_RADIO_POINTS; i++)
		radio_point[i].is = *(radio_point[i].glob);

	for (i=0; i<MAX_HSCROLLBARS; i++) 
		hscrollbar[i].is = *(hscrollbar[i].glob);

	is_tree_layout =  
		(radio_point[TREE_POINT].is == radio_point[TREE_POINT].sel);

	if (is_tree_layout) {
		hscrollbar[MEDIUMSCR].t = "Spreadlevel: ";
		hscrollbar[MEDIUMSCR].glob = &spread_level;
		hscrollbar[MEDIUMSCR].is   = spread_level;
		hscrollbar[CENTERSCR].t = "Treefactor:  ";
		hscrollbar[CENTERSCR].glob = &mytreefactor;
		hscrollbar[CENTERSCR].is = mytreefactor;
	}
	else {
		hscrollbar[MEDIUMSCR].t = "Mediumshifts:";
		hscrollbar[MEDIUMSCR].glob = &max_mediumshifts;
		hscrollbar[MEDIUMSCR].is   = max_mediumshifts;
		hscrollbar[CENTERSCR].t = "Centershifts:";
		hscrollbar[CENTERSCR].glob = &max_centershifts;
		hscrollbar[CENTERSCR].is = max_centershifts;
	}

        XMapRaised(root_display, dialog_window);

	for (i=0; i<MAX_ONOFF_POINTS; i++) {
		XMapRaised(root_display, on_off_point[i].w);
		normalize_point(on_off_point[i].w);
	}
	for (i=0; i<MAX_TOUCH_POINTS; i++) {
		XMapRaised(root_display, touch_point[i].w);
		normalize_point(touch_point[i].w);
	}
	for (i=0; i<MAX_RADIO_POINTS; i++) {
		XMapRaised(root_display, radio_point[i].w);
		normalize_point(radio_point[i].w);
	}
	for (i=0; i<MAX_HSCROLLBARS; i++) {
        	XMapRaised(root_display, hscrollbar[i].ws);
        	XMapRaised(root_display, hscrollbar[i].wt);
	}

        XMapRaised(root_display, okay_button);
        XMapRaised(root_display, cancel_button);
	normalize_point(okay_button);
	normalize_point(cancel_button);

	for (i=0; i<MAX_HSCROLLBARS; i++) (*(hscrollbar[i].sf))();

        XFlush(root_display);

        /*
         * Wait for Exposure event.
         */
#ifndef ULTRIX
#ifndef AIX
        do { XNextEvent(root_display, &event);
        } while (event.type != Expose && event.type != MapNotify);
#endif
#endif

	if (event.type == Expose) do_Expose(&event);
	dialog_main_loop();

        XUnmapWindow(root_display, dialog_window);

	if (dialog_success) {

		for (i=0; i<MAX_ONOFF_POINTS; i++)
			*(on_off_point[i].glob) = on_off_point[i].is;

		for (i=0; i<MAX_RADIO_POINTS; i++) 
			if (radio_point[i].is == radio_point[i].sel) 
				*(radio_point[i].glob) = radio_point[i].is;

		for (i=0; i<MAX_HSCROLLBARS; i++) 
			*(hscrollbar[i].glob) = hscrollbar[i].is;

		tree_factor = (double)mytreefactor/(double)100;
	}

	return(dialog_success);
}



/*--------------------------------------------------------------------*/
/*   Redraw routine                                                   */
/*--------------------------------------------------------------------*/

/*  Draw dialog box
 *  ---------------
 */

static char mxbuffer[256];

static void draw_dialog_box()
{
	int i;

	XSetForeground(root_display,
		XDefaultGC(root_display, root_screen),
		XWhitePixel(root_display, root_screen));
	XFillRectangle(root_display, 
		(Drawable)dialog_window,
		XDefaultGC(root_display, root_screen),
		0,0, DIAL_W, DIAL_H);

	XSetForeground(root_display,
		XDefaultGC(root_display, root_screen),
		XBlackPixel(root_display, root_screen));

	i = XTextWidth(dialog_font,"Cancel (Esc)",strlen("Cancel (Esc)"));
	i = (BUTTON_W-i)/2;
	XDrawString(root_display,
		(Drawable)cancel_button,
		XDefaultGC(root_display, root_screen),
		i,18,"Cancel (Esc)",strlen("Cancel (Esc)"));

	i = XTextWidth(dialog_font,"Okay",strlen("Okay"));
	i = (BUTTON_W-i)/2;
	XDrawString(root_display,
		(Drawable)okay_button,
		XDefaultGC(root_display, root_screen),
		i,18,"Okay",strlen("Okay"));

	draw_underl_text(5,18,"Layout Attributes");
	draw_underl_text(200,18,"Orientation");
	draw_underl_text(380,18,"Crossing-Heuristics");
	draw_underl_text(380,90,"Node Alignment");
	draw_underl_text(5,ELPH_POS,"Adding Labels");
	draw_underl_text(5,LFAC_POS,"Layout Factors");
	draw_underl_text(5,LSPE_POS,"Layout Speed");
	draw_underl_text(5,LPAR_POS,"Partitioning Algorithm");

	for (i=0; i<MAX_ONOFF_POINTS; i++)
		draw_text(on_off_point[i].x+20,on_off_point[i].y,
			  on_off_point[i].t);

	for (i=0; i<MAX_TOUCH_POINTS; i++)
		draw_text(touch_point[i].x+20,touch_point[i].y,
			  touch_point[i].t);

	for (i=0; i<MAX_RADIO_POINTS; i++)
		draw_text(radio_point[i].x+20,radio_point[i].y,
			  radio_point[i].t);

	for (i=0; i<MAX_HSCROLLBARS; i++) 
		draw_text(hscrollbar[i].xt, hscrollbar[i].y, hscrollbar[i].t);

	for (i=0; i<MAX_ONOFF_POINTS; i++)
		if (on_off_point[i].is == on_off_point[i].sel)
			draw_dialog_point(on_off_point[i].w,1);
		else    draw_dialog_point(on_off_point[i].w,0);

	for (i=0; i<MAX_TOUCH_POINTS; i++)
		draw_dialog_point(touch_point[i].w,0);

	for (i=0; i<MAX_RADIO_POINTS; i++)
		if (radio_point[i].is == radio_point[i].sel)
			draw_dialog_point(radio_point[i].w,1);
		else    draw_dialog_point(radio_point[i].w,0);

	draw_sizes();

	for (i=0; i<MAX_HSCROLLBARS; i++) (*(hscrollbar[i].sf))();

	XSync(root_display, 0);
}



static void draw_sizes()
{
	int i;

	XSetForeground(root_display,
		XDefaultGC(root_display, root_screen),
		XWhitePixel(root_display, root_screen));
	XFillRectangle(root_display, 
		(Drawable)dialog_window,
		XDefaultGC(root_display, root_screen),
		120,LFAC_POS+4,100,8*18);
	XFillRectangle(root_display, 
		(Drawable)dialog_window,
		XDefaultGC(root_display, root_screen),
		410,LFAC_POS+56,100,5*18);

	for (i=0; i<MAX_HSCROLLBARS; i++) {
		if (i==CENTERSCR) continue;
		if (hscrollbar[i].is == MAXINT) SPRINTF(mxbuffer, "inf."); 
		else SPRINTF(mxbuffer, hscrollbar[i].f, hscrollbar[i].is);
		draw_text(hscrollbar[i].xv,hscrollbar[i].y,mxbuffer);
	}

	i = CENTERSCR;
	if (is_tree_layout) {
		if (hscrollbar[i].is == MAXINT) hscrollbar[i].is = 330; 
		SPRINTF(mxbuffer, "%4.2f", (float)hscrollbar[i].is/100.0);
		draw_text(hscrollbar[i].xv,hscrollbar[i].y,mxbuffer);
	}
	else {
		if (hscrollbar[i].is == MAXINT) SPRINTF(mxbuffer, "inf."); 
		else SPRINTF(mxbuffer, hscrollbar[i].f, hscrollbar[i].is);
		draw_text(hscrollbar[i].xv,hscrollbar[i].y,mxbuffer);
	}

	draw_underl_text(280,LFAC_POS+54,"Maximal Iteration Factors");

	XSync(root_display, 0);
}



/*  Draw text m into the dialog box at position x, y
 *  ------------------------------------------------
 */

static void draw_text(x,y, m)
int x, y;
char *m;
{
	XSetForeground(root_display,
		XDefaultGC(root_display, root_screen),
		XBlackPixel(root_display, root_screen));
	XDrawString(root_display,
		(Drawable)dialog_window,
		XDefaultGC(root_display, root_screen),
		x,y,m,strlen(m));
}


/*  Draw underlined text m into the dialog box at position x, y
 *  -----------------------------------------------------------
 */

static void draw_underl_text(x,y, m)
int x, y;
char *m;
{
	int len;

	XSetForeground(root_display,
		XDefaultGC(root_display, root_screen),
		XBlackPixel(root_display, root_screen));
	XDrawString(root_display,
		(Drawable)dialog_window,
		XDefaultGC(root_display, root_screen),
		x,y,m,strlen(m));
	len = XTextWidth(dialog_font,m,strlen(m));
	XDrawLine(root_display,(Drawable)dialog_window,
                XDefaultGC(root_display, root_screen),
                x,y+2,x+len,y+2);
}


/*  Draw a dialog point in a color
 *  ------------------------------
 */

static void draw_dialog_point(w, color)
Window w;
int color;
{
	if (color==0) {
		XSetForeground(root_display,
			XDefaultGC(root_display, root_screen),
			XWhitePixel(root_display, root_screen));
	}
	else {
		XSetForeground(root_display,
			XDefaultGC(root_display, root_screen),
			XBlackPixel(root_display, root_screen));
	}
	XFillRectangle(root_display,(Drawable)w, 
			XDefaultGC(root_display, root_screen),
			1,1,
			POINT_W-2,POINT_W-2);
	XSync(root_display, 0);
        XFlush(root_display);
}


/* Normalize a button or point
 * ---------------------------
 */

static void normalize_point(w)
Window w;
{
        XWindowAttributes retWinAttr;

        if (!XGetWindowAttributes(root_display,w,&retWinAttr)) {
                FPRINTF(stderr,"Can't get window attributes.");
                gs_exit(1);
        }
        if (retWinAttr.border_width==3) {
                XMoveWindow(root_display,w,
                        retWinAttr.x+2, retWinAttr.y+2);
        }
        XSetWindowBorderWidth(root_display, w, 1L);
}



/*--------------------------------------------------------------------*/
/*  Event handling                                                    */
/*--------------------------------------------------------------------*/

/* The main loop for interaction
 * =============================
 */

static void dialog_main_loop()
{
	XEvent event;

	dialog_notready = 1;
	dialog_success  = 0;
	while (dialog_notready) {

		XNextEvent (root_display, &event);

		if (event.type != Expose) 
			box_exposed = 0; 

		switch (event.type) {
	  	case KeyPress:
	   		prologue (&event, "KeyPress");
	    		do_KeyPress (&event);
	    		break;
	  	case ButtonPress:
	    		prologue (&event, "ButtonPress");
	    		do_ButtonPress (&event);
	    		break;
	  	case ButtonRelease:
	    		prologue (&event, "ButtonRelease");
	    		do_ButtonRelease (&event);
	    		break;
	  	case EnterNotify:
	    		prologue (&event, "EnterNotify");
			do_EnterNotify(&event);
	    		break;
	  	case LeaveNotify:
	    		prologue (&event, "LeaveNotify");
			do_LeaveNotify(&event);
	    		break;
	  	case FocusIn:
	    		prologue (&event, "FocusIn");
	    		break;
	  	case FocusOut:
	    		prologue (&event, "FocusOut");
	    		break;
	  	case Expose:
	    		prologue (&event, "Expose");
	    		do_Expose (&event);
	    		break;
	  	case ConfigureNotify:
	    		prologue (&event, "ConfigureNotify");
	    		do_ConfigureNotify (&event);
	    		break;
	  	case MotionNotify:
	    		prologue (&event, "MotionNotify");
			do_MotionNotify(&event);
	    		break;
	  	case GravityNotify:
	  	case CirculateRequest:
	  	case PropertyNotify:
	  	case SelectionClear:
	  	case SelectionRequest:
	  	case SelectionNotify:
	  	case ColormapNotify:
	  	case ClientMessage:
	  	case MappingNotify:
	  	case CreateNotify:
	  	case DestroyNotify:
	  	case CirculateNotify:
	  	case VisibilityNotify:
	 	case KeyRelease:
	  	case KeymapNotify:
	  	case NoExpose:
	  	case GraphicsExpose:
	    		prologue (&event, "Unhandled");
	    		break;
		}
    	}
}


#ifdef EVENT_DEBUG

/* For debugging only: Print event message
 * ---------------------------------------
 */

static void prologue (eventp, event_name)
XEvent *eventp;
char *event_name;
{
	XAnyEvent *e = (XAnyEvent *) eventp;

	PRINTF ("\n%s event, serial %ld, synthetic %s, window 0x%lx,\n",
	    event_name, e->serial, e->send_event ? "Yes" : "No", e->window);
}

#endif /* EVENT_DEBUG */



/* Keypress Events
 * ---------------
 */

static void do_KeyPress (eventp)
XEvent *eventp;
{
    	XKeyEvent *e = (XKeyEvent *) eventp;
    	KeySym ks;
    	int nbytes, i,j;
    	char str[256+1];
	char lk, uk; 

    	nbytes = XLookupString (e, str, 256, &ks, NULL);
    	if (nbytes < 0) nbytes = 0;
    	if (nbytes > 256) nbytes = 256;
    	str[nbytes] = '\0';

	if (nbytes==1) {
		/* No operation on ' ' */
		if (str[0]==' ') return;

		for (i=0; i<MAX_ONOFF_POINTS; i++) {
			lk = uk = on_off_point[i].key;
			lk = tolower(lk);
			uk = toupper(uk);
			if ((lk==str[0])||(uk==str[0])) {
				if (on_off_point[i].is == on_off_point[i].sel) 
					on_off_point[i].is = on_off_point[i].sel-1;
				else    on_off_point[i].is = on_off_point[i].sel;
				break;
			}
		}
	
		/* Dirtylabels => Labels */

		if ((i==1) && (on_off_point[1].is == on_off_point[1].sel)) 
			on_off_point[0].is = on_off_point[0].sel;
		if ((i==0) && (on_off_point[0].is != on_off_point[0].sel)) 
			on_off_point[1].is = on_off_point[1].sel-1;


		for (i=0; i<MAX_TOUCH_POINTS; i++) {
			lk = uk = touch_point[i].key;
			lk = tolower(lk);
			uk = toupper(uk);
			if ((lk==str[0])||(uk==str[0])) break;
		}

		/* Management of the touch buttons */

		switch (i) {
		case SLOWSPEED: 
			hscrollbar[BARYSCR].is    = MAXINT;
			hscrollbar[MEDIUMSCR].is  = MAXINT;
			hscrollbar[CENTERSCR].is  = MAXINT;
			hscrollbar[BENDSCR].is    = MAXINT;
			break;
		case NORMSPEED:
			hscrollbar[BARYSCR].is    = MAXINT;
			hscrollbar[MEDIUMSCR].is  = 100;
			hscrollbar[CENTERSCR].is  = 100;
			hscrollbar[BENDSCR].is    = 100;
			break;
		case MEDISPEED:
			hscrollbar[BARYSCR].is    = 10;
			hscrollbar[MEDIUMSCR].is  = 20;
			hscrollbar[CENTERSCR].is  = 30;
			hscrollbar[BENDSCR].is    = 50;
			break;
		case FASTSPEED:
			hscrollbar[BARYSCR].is    = 2;
			hscrollbar[MEDIUMSCR].is  = 2;
			hscrollbar[CENTERSCR].is  = 2; 
			hscrollbar[BENDSCR].is    = 2;
			break;
		}


		for (i=0; i<MAX_RADIO_POINTS; i++) {
			lk = uk = radio_point[i].key;
			lk = tolower(lk);
			uk = toupper(uk);
			if ((lk==str[0])||(uk==str[0])) {
				radio_point[i].is = radio_point[i].sel;
				break;
			}
		}

		if (i<MAX_RADIO_POINTS) {
			for (j=0; j<MAX_RADIO_POINTS; j++) {
				if ((i!=j) && (radio_point[i].glob==radio_point[j].glob)) 
					radio_point[j].is = radio_point[j].sel-1;
			}
		}

		i = is_tree_layout;
		is_tree_layout =  
		   (radio_point[TREE_POINT].is == radio_point[TREE_POINT].sel);

		if (i!=is_tree_layout) {
			if (is_tree_layout) {
				hscrollbar[MEDIUMSCR].t = "Spreadlevel: ";
				hscrollbar[MEDIUMSCR].glob = &spread_level;
				hscrollbar[MEDIUMSCR].is   = spread_level;
				hscrollbar[CENTERSCR].t = "Treefactor:  ";
				hscrollbar[CENTERSCR].glob = &mytreefactor;
				hscrollbar[CENTERSCR].is = mytreefactor;
			}
			else {
				hscrollbar[MEDIUMSCR].t = "Mediumshifts:";
				hscrollbar[MEDIUMSCR].glob = &max_mediumshifts;
				hscrollbar[MEDIUMSCR].is   = max_mediumshifts;
				hscrollbar[CENTERSCR].t = "Centershifts:";
				hscrollbar[CENTERSCR].glob = &max_centershifts;
				hscrollbar[CENTERSCR].is = max_centershifts;
			}
			draw_dialog_box();
		}


		/* No labels => label phase = 0 */

		if (on_off_point[0].is != on_off_point[0].sel) {
			radio_point[AFTER_FOLD].is = radio_point[AFTER_FOLD].sel;
			radio_point[AFTER_PART].is = radio_point[AFTER_PART].sel-1;
		}

		/* Dirty labels => label phase = 0 */

		if (on_off_point[1].is == on_off_point[1].sel) {
			radio_point[AFTER_FOLD].is = radio_point[AFTER_FOLD].sel;
			radio_point[AFTER_PART].is = radio_point[AFTER_PART].sel-1;
		}

		switch (str[0]) {
		case '\r': /* Carriage Return */
		case '\n': /* Carriage Return */
			dialog_notready = 0;
			dialog_success  = 1;
			return;
		case 27: /* Cancel */
			dialog_notready = 0;
			dialog_success  = 0;
			return;
		case 'q': /* Quit */
		case 'Q': /* Quit */
			dialog_notready = 0;
			dialog_success  = 0;
			return;
		}
		draw_dialog_box();
	}
}




/* Button Press 
 * ------------
 */

static Window act_scroll_window = 0;
 
static void do_ButtonPress (eventp)
XEvent *eventp;
{
	XButtonEvent *e = (XButtonEvent *) eventp;
	Window w;
	int i,j;

	w = e->window;

	act_scroll_window = w;	

	for (i=0; i<MAX_ONOFF_POINTS; i++) {
		if (w == on_off_point[i].w) {
			if (on_off_point[i].is == on_off_point[i].sel) {
				on_off_point[i].is = on_off_point[i].sel-1;
				draw_dialog_point(on_off_point[i].w,0);
			}
			else {  
				on_off_point[i].is = on_off_point[i].sel;
				draw_dialog_point(on_off_point[i].w,1);
			}
			break;
		}
	}

	/* Dirtylabels => Labels */

	if ((i==1) && (on_off_point[1].is == on_off_point[1].sel)) {
		on_off_point[0].is = on_off_point[0].sel;
		draw_dialog_point(on_off_point[0].w,1);
	}
	if ((i==0) && (on_off_point[0].is != on_off_point[0].sel)) {
		on_off_point[1].is = on_off_point[1].sel-1;
		draw_dialog_point(on_off_point[1].w,0);
	}

	for (i=0; i<MAX_TOUCH_POINTS; i++) {
		if (w == touch_point[i].w) break;
	}

	switch (i) {
	case SLOWSPEED: 
		hscrollbar[BARYSCR].is    = MAXINT;
		hscrollbar[MEDIUMSCR].is  = MAXINT;
		hscrollbar[CENTERSCR].is  = MAXINT;
		hscrollbar[BENDSCR].is    = MAXINT;
		draw_dialog_box();
		break;
	case NORMSPEED:
		hscrollbar[BARYSCR].is    = MAXINT;
		hscrollbar[MEDIUMSCR].is  = 100;
		hscrollbar[CENTERSCR].is  = 100;
		hscrollbar[BENDSCR].is    = 100;
		draw_dialog_box();
		break;
	case MEDISPEED:
		hscrollbar[BARYSCR].is    = 10;
		hscrollbar[MEDIUMSCR].is  = 20;
		hscrollbar[CENTERSCR].is  = 30;
		hscrollbar[BENDSCR].is    = 50;
		draw_dialog_box();
		break;
	case FASTSPEED:
		hscrollbar[BARYSCR].is    = 2;
		hscrollbar[MEDIUMSCR].is  = 2;
		hscrollbar[CENTERSCR].is  = 2; 
		hscrollbar[BENDSCR].is    = 2;
		draw_dialog_box();
		break;
	}

	for (i=0; i<MAX_RADIO_POINTS; i++) {
		if (w == radio_point[i].w) {
			radio_point[i].is = radio_point[i].sel;
			break;
		}
	}

	if (i<MAX_RADIO_POINTS) {
		for (j=0; j<MAX_RADIO_POINTS; j++) {
			if ((i!=j) && (radio_point[i].glob==radio_point[j].glob)) 
				radio_point[j].is = radio_point[j].sel-1;
		}
	}

	i = is_tree_layout;
	is_tree_layout =  
		(radio_point[TREE_POINT].is == radio_point[TREE_POINT].sel);
	if (i!=is_tree_layout) {
		if (is_tree_layout) {
			hscrollbar[MEDIUMSCR].t = "Spreadlevel: ";
			hscrollbar[MEDIUMSCR].glob = &spread_level;
			hscrollbar[MEDIUMSCR].is   = spread_level;
			hscrollbar[CENTERSCR].t = "Treefactor:  ";
			hscrollbar[CENTERSCR].glob = &mytreefactor;
			hscrollbar[CENTERSCR].is = mytreefactor;
		}
		else {
			hscrollbar[MEDIUMSCR].t = "Mediumshifts:";
			hscrollbar[MEDIUMSCR].glob = &max_mediumshifts;
			hscrollbar[MEDIUMSCR].is   = max_mediumshifts;
			hscrollbar[CENTERSCR].t = "Centershifts:";
			hscrollbar[CENTERSCR].glob = &max_centershifts;
			hscrollbar[CENTERSCR].is = max_centershifts;
		}
		draw_dialog_box();
	}

	/* No labels => label phase = 0 */

	if (on_off_point[0].is != on_off_point[0].sel) {
		radio_point[AFTER_FOLD].is = radio_point[AFTER_FOLD].sel;
		radio_point[AFTER_PART].is = radio_point[AFTER_PART].sel-1;
	}

	/* Dirty labels => label phase = 0 */

	if (on_off_point[1].is == on_off_point[1].sel) {
		radio_point[AFTER_FOLD].is = radio_point[AFTER_FOLD].sel;
		radio_point[AFTER_PART].is = radio_point[AFTER_PART].sel-1;
	}

	for (i=0; i<MAX_RADIO_POINTS; i++)
		if (radio_point[i].is == radio_point[i].sel)
			draw_dialog_point(radio_point[i].w,1);
		else    draw_dialog_point(radio_point[i].w,0);


	for (i=0; i<MAX_HSCROLLBARS; i++) {
		if (act_scroll_window==hscrollbar[i].ws) {
		 	(*(hscrollbar[i].rf))(e->x);
			draw_sizes();
		}
	}

	if (w==okay_button) {
		dialog_notready = 0;
		dialog_success  = 1;
	}
	else if (w==cancel_button) {
		dialog_notready = 0;
		dialog_success  = 0;
	}
}



/* Button Release
 * --------------
 */

static void do_ButtonRelease (eventp)
XEvent *eventp;
{
	XButtonEvent *e = (XButtonEvent *) eventp;
	int i;

	if (act_scroll_window && (e->window!=act_scroll_window)) {
		for (i=0; i<MAX_HSCROLLBARS; i++) (*(hscrollbar[i].sf))();
		draw_dialog_box();
		act_scroll_window = 0; 
		return;
	}

	for (i=0; i<MAX_HSCROLLBARS; i++) {
		if (act_scroll_window==hscrollbar[i].ws) {
		 	(*(hscrollbar[i].rf))(e->x);
			break;
		}
	}
	if (i<MAX_HSCROLLBARS) draw_dialog_box();
	act_scroll_window = 0; 
}



/* Mark a selection point as selectable
 * ------------------------------------
 */

static void do_EnterNotify (eventp)
XEvent *eventp;
{
    	XCrossingEvent *e = (XCrossingEvent *) eventp;
	Window w;
	XWindowAttributes retWinAttr;
	int i, is_point;

	is_point = 0;
	w = e->window;

	for (i=0; i<MAX_ONOFF_POINTS; i++)
		if (w==on_off_point[i].w) is_point = 1;
	for (i=0; i<MAX_TOUCH_POINTS; i++)
		if (w==touch_point[i].w) is_point = 1;
	for (i=0; i<MAX_RADIO_POINTS; i++)
		if (w==radio_point[i].w) is_point = 1;

	if ((w==okay_button)||(w==cancel_button)) is_point = 1;

	if (is_point) {
	        if (!XGetWindowAttributes(root_display,w,&retWinAttr)) {
               		FPRINTF(stderr,"Can't get window attributes.");
                	gs_exit(1);
        	}
		if (retWinAttr.border_width==1) {
			XMoveWindow(root_display,w,
				retWinAttr.x-2, retWinAttr.y-2);
		}
		XSetWindowBorderWidth(root_display, w, 3L);
	}
}


/* Unmark a selection point
 * ------------------------
 * The menupoint is not anymore selected.
 */ 

static void do_LeaveNotify (eventp)
XEvent *eventp;
{
    	XCrossingEvent *e = (XCrossingEvent *) eventp;
	Window w;
	int i, is_point;

	is_point = 0;
	w = e->window;

	for (i=0; i<MAX_ONOFF_POINTS; i++)
		if (w==on_off_point[i].w) is_point = 1;
	for (i=0; i<MAX_TOUCH_POINTS; i++)
		if (w==touch_point[i].w) is_point = 1;
	for (i=0; i<MAX_RADIO_POINTS; i++)
		if (w==radio_point[i].w) is_point = 1;

	if ((w==okay_button)||(w==cancel_button)) is_point = 1;

	if (is_point) normalize_point(w);
}


/* Expose a window
 * ---------------
 */

static void do_Expose (eventp)
XEvent *eventp;
{
	XExposeEvent *e = (XExposeEvent *) eventp;

	if (e->window==dialog_window) {
		if ((e->width>12)&&(e->height>12)&&(!box_exposed)) {
			draw_dialog_box(); 
			box_exposed = 1;
		}
	}
	else box_exposed = 0;
}


/* Resize or move a window
 * -----------------------
 */

static void do_ConfigureNotify (eventp)
XEvent *eventp;
{
	XConfigureEvent *e = (XConfigureEvent *) eventp;

	if (e->window==dialog_window) {
		XResizeWindow(root_display,dialog_window,DIAL_W,DIAL_H);
		/* if (!box_exposed) draw_dialog_box(); */
	}
}


/* Motion with pressed button 
 * --------------------------
 */

static void do_MotionNotify (eventp)
XEvent *eventp;
{
	int i;

	XMotionEvent *e = (XMotionEvent *) eventp;

	for (i=0; i<MAX_HSCROLLBARS; i++) { 
		if (act_scroll_window==hscrollbar[i].ws) {
		 	(*(hscrollbar[i].rf))(e->x);
			break;
		}
	}
	if (i<MAX_HSCROLLBARS) draw_sizes();
}


/*--------------------------------------------------------------------*/
/*  Scrollbar handling 						      */
/*--------------------------------------------------------------------*/


/*  General scrollbar actions with scaling 1
 *  ----------------------------------------
 *  The general scrollbar actions for bars that have a scaling of 1, i.e.
 *  produce values 1, 2 ..., 40.
 */


static void read_40sc_scrollbar(val,i)
int val;
int i;
{
	if (val<0) val = 0;
	if (val+SCROLLBAR_LEN >= hscrollbar[i].len) 
		val = hscrollbar[i].len - SCROLLBAR_LEN;
	XMoveResizeWindow(root_display,hscrollbar[i].wt,val,0,SCROLLBAR_LEN,6);
	hscrollbar[i].is = 1 + 39*val/(hscrollbar[i].len-SCROLLBAR_LEN);
}


static void set_40sc_scrollbar(i)
int i;
{
	int val;

	val =  (hscrollbar[i].is-1)*(hscrollbar[i].len-SCROLLBAR_LEN)/39;
	if (val<0) val = 0;
	if (val+SCROLLBAR_LEN >= hscrollbar[i].len) 
		val = hscrollbar[i].len - SCROLLBAR_LEN;
	XMoveResizeWindow(root_display, hscrollbar[i].wt,val,0,SCROLLBAR_LEN,6);
}




/*  General scrollbar actions with scaling 5
 *  ----------------------------------------
 *  The general scrollbar actions for bars that have a scaling of 5, i.e.
 *  produce values 0, 5, 10, ..., 325.
 */

static void read_5sc_scrollbar(val,i)
int val;
int i;
{
	if (val<0) val = 0;
	if (val+SCROLLBAR_LEN >= hscrollbar[i].len) 
		val = hscrollbar[i].len - SCROLLBAR_LEN;
	XMoveResizeWindow(root_display,hscrollbar[i].wt,val,0,SCROLLBAR_LEN,6);
	hscrollbar[i].is = 5*val;
}

static void set_5sc_scrollbar(i)
int i;
{
	int val;

	val =  hscrollbar[i].is/5;
	if (val<0) val = 0;
	if (val+SCROLLBAR_LEN >= hscrollbar[i].len) 
		val = hscrollbar[i].len - SCROLLBAR_LEN;
	XMoveResizeWindow(root_display, hscrollbar[i].wt,val,0,SCROLLBAR_LEN,6);
}



/*  General scrollbar actions with scaling 5 and infinite
 *  -----------------------------------------------------
 *  The general scrollbar actions for bars that have a scaling of 5, i.e.
 *  produce values 0, 5, 10, ..., 320, infinite.
 */

static void read_5in_scrollbar(val,i)
int val;
int i;
{
	if (val<0) val = 0;
	if (val+SCROLLBAR_LEN >= hscrollbar[i].len) 
		val = hscrollbar[i].len - SCROLLBAR_LEN;
	XMoveResizeWindow(root_display,hscrollbar[i].wt,val,0,SCROLLBAR_LEN,6);
	hscrollbar[i].is = 5*val;
	if (val == hscrollbar[i].len - SCROLLBAR_LEN) hscrollbar[i].is = MAXINT;
}



/*  The instances of the scrollbars actions
 *  ---------------------------------------
 */

static void read_downfacscrollbar(val)
int val;
{
	read_40sc_scrollbar(val,DOWNFAC);
}

static void set_downfacscrollbar()
{
	set_40sc_scrollbar(DOWNFAC);
}

static void read_upfacscrollbar(val)
int val;
{
	read_40sc_scrollbar(val,UPFAC);
}

static void set_upfacscrollbar()
{
	set_40sc_scrollbar(UPFAC);
}

static void read_nearfacscrollbar(val)
int val;
{
	read_40sc_scrollbar(val,NEARFAC);
}

static void set_nearfacscrollbar()
{
	set_40sc_scrollbar(NEARFAC);
}

static void read_xbasescrollbar(val)
int val;
{
	read_5sc_scrollbar(val,XBASESCR);
}

static void set_xbasescrollbar()
{
	set_5sc_scrollbar(XBASESCR);
}

static void read_ybasescrollbar(val)
int val;
{
	read_5sc_scrollbar(val,YBASESCR);
}

static void set_ybasescrollbar()
{
	set_5sc_scrollbar(YBASESCR);
}

static void read_xspacescrollbar(val)
int val;
{
	read_5sc_scrollbar(val,XSPACESCR);
}

static void set_xspacescrollbar()
{
	set_5sc_scrollbar(XSPACESCR);
}

static void read_xlspacescrollbar(val)
int val;
{
	read_5sc_scrollbar(val,XLSPACESCR);
}

static void set_xlspacescrollbar()
{
	set_5sc_scrollbar(XLSPACESCR);
}

static void read_yspacescrollbar(val)
int val;
{
	read_5sc_scrollbar(val,YSPACESCR);
}

static void set_yspacescrollbar()
{
	set_5sc_scrollbar(YSPACESCR);
}

static void read_baryscrollbar(val)
int val;
{
	read_5in_scrollbar(val,BARYSCR);
}

static void set_baryscrollbar()
{
	set_5sc_scrollbar(BARYSCR);
}

static void read_mediumscrollbar(val)
int val;
{
	read_5in_scrollbar(val,MEDIUMSCR);
}

static void set_mediumscrollbar()
{
	set_5sc_scrollbar(MEDIUMSCR);
}

static void read_centerscrollbar(val)
int val;
{
	read_5in_scrollbar(val,CENTERSCR);
}

static void set_centerscrollbar()
{
	set_5sc_scrollbar(CENTERSCR);
}

static void read_bendscrollbar(val)
int val;
{
	read_5in_scrollbar(val,BENDSCR);
}

static void set_bendscrollbar()
{
	set_5sc_scrollbar(BENDSCR);
}

static void read_splinescrollbar(val)
int val;
{
	if (val<0) val = 0;
	if (val+SCROLLBAR_LEN >= hscrollbar[SPLINESCR].len) 
		val = hscrollbar[SPLINESCR].len - SCROLLBAR_LEN;
	XMoveResizeWindow(root_display,hscrollbar[SPLINESCR].wt,
			val,0,SCROLLBAR_LEN,6);
	hscrollbar[SPLINESCR].is = 
		1 + 99*val/(hscrollbar[SPLINESCR].len-SCROLLBAR_LEN);
}

static void set_splinescrollbar()
{
	int val;

	val =  (hscrollbar[SPLINESCR].is-1)*
			(hscrollbar[SPLINESCR].len-SCROLLBAR_LEN)/99;
	if (val<0) val = 0;
	if (val+SCROLLBAR_LEN >= hscrollbar[SPLINESCR].len) 
		val = hscrollbar[SPLINESCR].len - SCROLLBAR_LEN;
	XMoveResizeWindow(root_display, hscrollbar[SPLINESCR].wt,
			val,0,SCROLLBAR_LEN,6);
}


/*--------------------------------------------------------------------*/

#endif /* X11 */

