





/*
 * xv24to8.c  -  contains the 24-to-8-bit Conv24to8() procedure
 *               and the 8-to-24-bit Conv8to24() procedure
 *
 * The Conv24to8 procedure takes a pointer to a 24-bit image (loaded
 * previously).  The image will be a w * h * 3 byte array of
 * bytes.  The image will be arranged with 3 bytes per pixel (in order
 * R, G, and B), pixel 0 at the top left corner.  (As normal.)
 * The procedure also takes a maximum number of colors to use (numcols)
 * and pointers to three 256-long arrays of bytes (to hold the returned
 * colormap)
 *
 * Note that Conv24to8() does NOT free the pic24 image under any circumstances
 *
 * The Conv24to8 procedure will set up the following:  it will allocate, make
 * & return 'pic8', a 'w' by 'h' (passed in values) 8-bit picture.
 * it will load up the rmap, gmap and bmap colormap arrays.  it will NOT 
 * calculate numcols, since the cmap sort procedure has to be called anyway
 *
 * Conv24to8 returns 'pic8' if successful, 'NULL' on failure (presumably on a 
 * malloc())
 *
 * The 'slow' code is based on Heckbert's Median Cut algorithm.
 *
 * contains:
 *   Cont24to8()
 *   Init24to8()
 */

/* Copyright Notice
 * ================
 * Copyright 1989, 1990, 1991, 1992, 1993 by John Bradley
 * 
 * Permission to use, copy, and distribute XV in its entirety, for 
 * non-commercial purposes, is hereby granted without fee, provided that
 * this license information and copyright notice appear in all copies.
 * 
 * Note that distributing XV 'bundled' in with ANY product is considered
 * to be a 'commercial purpose'.
 *
 * Also note that any copies of XV that are distributed MUST be built
 * and/or configured to be in their 'unregistered copy' mode, so that it
 * is made obvious to the user that XV is shareware, and that they should
 * consider donating, or at least reading this License Info.
 * 
 * The software may be modified for your own purposes, but modified
 * versions may NOT be distributed without prior consent of the author.
 * 
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the author be held liable for any damages
 * arising from the use of this software.
 * 
 * If you would like to do something with XV that this copyright
 * prohibits (such as distributing it with a commercial product, 
 * using portions of the source in some other program, etc.), please
 * contact the author (preferably via email).  Arrangements can
 * probably be worked out.
 *
 * XV is shareware for PERSONAL USE only.  You may use XV for your own
 * amusement, and if you find it nifty, useful, generally cool, or of
 * some value to you, your non-deductable donation would be greatly
 * appreciated.  $25 is the suggested donation, though, of course,
 * larger donations are quite welcome.  Folks who donate $25 or more
 * can receive a Real Nice bound copy of the XV manual for no extra
 * charge.
 * 
 * Commercial, government, and institutional users MUST register their
 * copies of XV, for the exceedingly REASONABLE price of just $25 per
 * workstation/X terminal.  Site licenses are available for those who
 * wish to run XV on a large number of machines.  Contact the author
 * for more details.
 *
 * The author may be contacted via:
 *    US Mail:  John Bradley
 *              1053 Floyd Terrace
 *              Bryn Mawr, PA  19010
 *
 *    Phone:    (215) 898-8813
 *    EMail:    bradley@cis.upenn.edu
 */

/*
 * Portions Copyright (C) 1989, 1991 by Jef Poskanzer.  See copyright notice
 * at beginning of relevant code.
 */

#include "xv.h"

#define	MAX_CMAP_SIZE	256
#define	COLOR_DEPTH	8
#define	MAX_COLOR	256
#define	B_DEPTH		5	/* # bits/pixel to use */
#define	B_LEN		(1<<B_DEPTH)
#define	C_DEPTH		3
#define	C_LEN		(1<<C_DEPTH)	/* # cells/color to use */

#define R2FACT 20		/* .300 * .300 * 256 = 23 */
#define G2FACT 39		/* .586 * .586 * 256 = 88 */
#define B2FACT 8		/* .114 * .114 * 256 =  3 */


typedef struct colorbox
  {
    struct colorbox *next, *prev;
    int rmin, rmax, gmin, gmax, bmin, bmax;
    int total;
  }
CBOX;

typedef struct
  {
    int num_ents;
    int entries[MAX_CMAP_SIZE][2];
  }
CCELL;

static byte *pic8;
static byte *pic24;
static byte *rmap, *gmap, *bmap;	/* ptrs to computed colormap */
static int num_colors, WIDE, HIGH;
static int histogram[B_LEN][B_LEN][B_LEN];

CBOX *freeboxes, *usedboxes;
CCELL **ColorCells;

#ifdef __STDC__
static void get_histogram (CBOX *);
static CBOX *largest_box (void);
static void splitbox (CBOX *);
static void shrinkbox (CBOX *);
static void assign_color (CBOX *, byte *, byte *, byte *);
static CCELL *create_colorcell (int, int, int);
static void map_colortable (void);
static int quant_fsdither (void);
#else
static void get_histogram ();
static CBOX *largest_box ();
static void splitbox ();
static void shrinkbox ();
static void assign_color ();
static CCELL *create_colorcell ();
static void map_colortable ();
static int quant_fsdither ();
static int Quick24to8 ();
static int QuickCheck ();
static int ppmquant ();
#endif

static int tbl1[512],		/* tables used in F-S Dithering */
  tbl3[512],			/* contain i/16, 3i/16, 5i/16, 7i/16, */
  tbl5[512],			/* (i=-256..255) respectively */
  tbl7[512];


/****************************/
void 
Init24to8 ()
/****************************/
{
  /* initialize Floyd-Steinberg division tables */
  /* make sure rounding is done correctly for negative values! */

  int i;

  for (i = -256; i < 0; i++)
    {
      tbl1[i + 256] = -((8 - i) / 16);
      tbl3[i + 256] = -((8 - 3 * i) / 16);
      tbl5[i + 256] = -((8 - 5 * i) / 16);
      tbl7[i + 256] = -((8 - 7 * i) / 16);
    }

  for (i = 0; i < 256; i++)
    {
      tbl1[i + 256] = (i + 8) / 16;
      tbl3[i + 256] = (3 * i + 8) / 16;
      tbl5[i + 256] = (5 * i + 8) / 16;
      tbl7[i + 256] = (7 * i + 8) / 16;
    }

#ifdef FOO
  for (i = 0; i < 512; i++)
    {
      printf ("%3d:  tbl1=%3d, tbl3=%3d, tbl5=%3d, tbl7=%3d\n",
	      i - 256, tbl1[i], tbl3[i], tbl5[i], tbl7[i]);
    }
#endif
}



/****************************/
byte *
Conv24to8 (byte * p, int w, int h, int nc, byte * rm, byte * gm, byte * bm)
/****************************/
{
  int i, j;
  CBOX *box_list, *ptr;

  /* returns pointer to new 8-bit-per-pixel image (w*h) if successful, or
     NULL if unsuccessful */

  if (nc <= 0)
    nc = 255;			/* 'nc == 0' breaks code */

  /* copy arguments to local-global variables */

  pic24 = p;
  WIDE = w;
  HIGH = h;
  num_colors = nc;
  rmap = rm;
  gmap = gm;
  bmap = bm;

  if (!pic24)
    {
      fprintf (stderr, "Keine Konvertierung pic24=NULL\n");
      return NULL;
    }

  /* allocate pic immediately, so that if we can't allocate it, we don't
     waste time running this algorithm */

  pic8 = (byte *) malloc (WIDE * HIGH);
  if (!pic8)
    {
      fprintf (stderr, "Conv24to8() - failed to allocate 'pic8'\n");
      return pic8;
    }

  /* fprintf(stderr,"Conv24to8() - malloc 'pic8'\n"); */

/* HEFA
   if (!noqcheck && QuickCheck(pic24,w,h,nc)) { 
   // see if it's a <256 color RGB pic 
   SetISTR(ISTR_INFO,"No color compression was necessary.\n");
   return pic8;   
   }

   else if (conv24 == CONV24_FAST) {
   SetISTR(ISTR_INFO,"Doing 'quick' 24-bit to 8-bit conversion.");
   i = Quick24to8(pic24,w,h);
   if (i) { free(pic8);  pic8 = NULL; }
   return pic8;
   }

   else if (conv24 == CONV24_BEST) {
   SetISTR(ISTR_INFO,"Doing 'best' 24-bit to 8-bit conversion.");
   i = ppmquant(pic24,w,h,nc);
   if (i) { free(pic8);  pic8 = NULL; }
   return pic8;
   }

   else 
   SetISTR(ISTR_INFO,"Doing 'slow' 24-bit to 8-bit conversion.");
 */


/**** STEP 1:  create empty boxes ****/

  usedboxes = NULL;
  box_list = freeboxes = (CBOX *) malloc (num_colors * sizeof (CBOX));

  if (box_list == NULL)
    {
      /* fprintf(stderr,"Conv24to8() - failed to allocate 'freeboxes'\n"); */
      free (pic8);
      pic8 = NULL;
      return pic8;
    }

  for (i = 0; i < num_colors; i++)
    {
      freeboxes[i].next = &freeboxes[i + 1];
      freeboxes[i].prev = &freeboxes[i - 1];
    }
  freeboxes[0].prev = NULL;
  freeboxes[num_colors - 1].next = NULL;


  /* fprintf(stderr,"Conv24to8() - create empty boxes\n"); */


/**** STEP 2: get histogram, initialize first box ****/

  ptr = freeboxes;
  freeboxes = ptr->next;
  if (freeboxes)
    freeboxes->prev = NULL;

  ptr->next = usedboxes;
  usedboxes = ptr;
  if (ptr->next)
    ptr->next->prev = ptr;

  /* fprintf(stderr,"Conv24to8() - vor get histogram, initialize first box\n"); */
  get_histogram (ptr);

  /* fprintf(stderr,"Conv24to8() - get histogram, initialize first box\n"); */



/**** STEP 3: continually subdivide boxes until no more free boxes remain */

  while (freeboxes)
    {
      ptr = largest_box ();
      if (ptr)
	splitbox (ptr);
      else
	break;
    }
  /* fprintf(stderr,"Conv24to8() - continually subdivide boxes\n"); */



/**** STEP 4: assign colors to all boxes ****/

  for (i = 0, ptr = usedboxes; i < num_colors && ptr; i++, ptr = ptr->next)
    {
      assign_color (ptr, &rmap[i], &gmap[i], &bmap[i]);
    }

  /* We're done with the boxes now */
  num_colors = i;
  free (box_list);
  box_list = freeboxes = usedboxes = NULL;

  /* fprintf(stderr,"Conv24to8() - assign colors to all boxes\n"); */



/**** STEP 5: scan histogram and map all values to closest color */

  /* 5a: create cell list as described in Heckbert[2] */

  ColorCells = (CCELL **) calloc (C_LEN * C_LEN * C_LEN, sizeof (CCELL *));
  /* fprintf(stderr,"Conv24to8() - scan histogram\n"); */


  /* 5b: create mapping from truncated pixel space to color table entries */

  map_colortable ();
  /*  fprintf(stderr,"Conv24to8() - create mapping\n"); */


/**** STEP 6: scan image, match input values to table entries */

  i = quant_fsdither ();

  /* free everything that can be freed */
  for (j = 0; j < C_LEN * C_LEN * C_LEN; j++)
    {
      if (ColorCells[j] != NULL)
	free (ColorCells[j]);
    }

  /* fprintf(stderr,"Conv24to8() - scan image\n"); */

  free (ColorCells);

  if (i)
    {
      free (pic8);
      pic8 = NULL;
    }				/* free 'pic' on failure */
  return pic8;
}


/****************************/
static void 
get_histogram (CBOX * box)
/****************************/
{
  int i, j, r, g, b, *ptr;
  byte *p;

  box->rmin = box->gmin = box->bmin = 999;
  box->rmax = box->gmax = box->bmax = -1;
  box->total = WIDE * HIGH;

  /* zero out histogram */
  ptr = &histogram[0][0][0];
  for (i = B_LEN * B_LEN * B_LEN; i > 0; i--)
    *ptr++ = 0;

  /* calculate histogram */
  p = pic24;

  for (i = 0; i < HIGH; i++)
    for (j = 0; j < WIDE; j++)
      {
	r = (*p++) >> (COLOR_DEPTH - B_DEPTH);
	g = (*p++) >> (COLOR_DEPTH - B_DEPTH);
	b = (*p++) >> (COLOR_DEPTH - B_DEPTH);

	if (r < box->rmin)
	  box->rmin = r;
	if (r > box->rmax)
	  box->rmax = r;

	if (g < box->gmin)
	  box->gmin = g;
	if (g > box->gmax)
	  box->gmax = g;

	if (b < box->bmin)
	  box->bmin = b;
	if (b > box->bmax)
	  box->bmax = b;
	histogram[r][g][b]++;
      }
}



/******************************/
static CBOX *
largest_box ()
/******************************/
{
  CBOX *tmp, *ptr;
  int size = -1;

  tmp = usedboxes;
  ptr = NULL;

  while (tmp)
    {
      if ((tmp->rmax > tmp->rmin ||
	   tmp->gmax > tmp->gmin ||
	   tmp->bmax > tmp->bmin) && tmp->total > size)
	{
	  ptr = tmp;
	  size = tmp->total;
	}
      tmp = tmp->next;
    }
  return (ptr);
}



/******************************/
static void 
splitbox (CBOX * ptr)
/******************************/
{
  int hist2[B_LEN], first, last, i, rdel, gdel, bdel;
  CBOX *newb;
  int *iptr, *histp, ir, ig, ib;
  int rmin, rmax, gmin, gmax, bmin, bmax;
  enum
    {
      RED, GREEN, BLUE
    }
  which;

  /*
   * see which axis is the largest, do a histogram along that
   * axis.  Split at median point.  Contract both new boxes to
   * fit points and return
   */

  first = last = 0;		/* shut RT hcc compiler up */

  rmin = ptr->rmin;
  rmax = ptr->rmax;
  gmin = ptr->gmin;
  gmax = ptr->gmax;
  bmin = ptr->bmin;
  bmax = ptr->bmax;

  rdel = rmax - rmin;
  gdel = gmax - gmin;
  bdel = bmax - bmin;

  if (rdel >= gdel && rdel >= bdel)
    which = RED;
  else if (gdel >= bdel)
    which = GREEN;
  else
    which = BLUE;

  /* get histogram along longest axis */
  switch (which)
    {

    case RED:
      histp = &hist2[rmin];
      for (ir = rmin; ir <= rmax; ir++)
	{
	  *histp = 0;
	  for (ig = gmin; ig <= gmax; ig++)
	    {
	      iptr = &histogram[ir][ig][bmin];
	      for (ib = bmin; ib <= bmax; ib++)
		{
		  *histp += *iptr;
		  ++iptr;
		}
	    }
	  ++histp;
	}
      first = rmin;
      last = rmax;
      break;

    case GREEN:
      histp = &hist2[gmin];
      for (ig = gmin; ig <= gmax; ig++)
	{
	  *histp = 0;
	  for (ir = rmin; ir <= rmax; ir++)
	    {
	      iptr = &histogram[ir][ig][bmin];
	      for (ib = bmin; ib <= bmax; ib++)
		{
		  *histp += *iptr;
		  ++iptr;
		}
	    }
	  ++histp;
	}
      first = gmin;
      last = gmax;
      break;

    case BLUE:
      histp = &hist2[bmin];
      for (ib = bmin; ib <= bmax; ib++)
	{
	  *histp = 0;
	  for (ir = rmin; ir <= rmax; ir++)
	    {
	      iptr = &histogram[ir][gmin][ib];
	      for (ig = gmin; ig <= gmax; ig++)
		{
		  *histp += *iptr;
		  iptr += B_LEN;
		}
	    }
	  ++histp;
	}
      first = bmin;
      last = bmax;
      break;
    }


  /* find median point */
  {
    int sum, sum2;

    histp = &hist2[first];

    sum2 = ptr->total / 2;
    histp = &hist2[first];
    sum = 0;

    for (i = first; i <= last && (sum += *histp++) < sum2; i++);
    if (i == first)
      i++;
  }


  /* Create new box, re-allocate points */

  newb = freeboxes;
  freeboxes = newb->next;
  if (freeboxes)
    freeboxes->prev = NULL;

  if (usedboxes)
    usedboxes->prev = newb;
  newb->next = usedboxes;
  usedboxes = newb;

  {
    int sum1, sum2, j;

    histp = &hist2[first];
    sum1 = 0;
    for (j = first; j < i; ++j)
      sum1 += *histp++;
    sum2 = 0;
    for (j = i; j <= last; ++j)
      sum2 += *histp++;
    newb->total = sum1;
    ptr->total = sum2;
  }


  newb->rmin = rmin;
  newb->rmax = rmax;
  newb->gmin = gmin;
  newb->gmax = gmax;
  newb->bmin = bmin;
  newb->bmax = bmax;

  switch (which)
    {
    case RED:
      newb->rmax = i - 1;
      ptr->rmin = i;
      break;
    case GREEN:
      newb->gmax = i - 1;
      ptr->gmin = i;
      break;
    case BLUE:
      newb->bmax = i - 1;
      ptr->bmin = i;
      break;
    }

  shrinkbox (newb);
  shrinkbox (ptr);
}


/****************************/
static void 
shrinkbox (CBOX * box)
/****************************/
{
  int *histp, ir, ig, ib;
  int rmin, rmax, gmin, gmax, bmin, bmax;

  rmin = box->rmin;
  rmax = box->rmax;
  gmin = box->gmin;
  gmax = box->gmax;
  bmin = box->bmin;
  bmax = box->bmax;

  if (rmax > rmin)
    {
      for (ir = rmin; ir <= rmax; ir++)
	for (ig = gmin; ig <= gmax; ig++)
	  {
	    histp = &histogram[ir][ig][bmin];
	    for (ib = bmin; ib <= bmax; ib++)
	      if (*histp++ != 0)
		{
		  box->rmin = rmin = ir;
		  goto have_rmin;
		}
	  }

    have_rmin:
      if (rmax > rmin)
	for (ir = rmax; ir >= rmin; --ir)
	  for (ig = gmin; ig <= gmax; ig++)
	    {
	      histp = &histogram[ir][ig][bmin];
	      for (ib = bmin; ib <= bmax; ib++)
		if (*histp++ != 0)
		  {
		    box->rmax = rmax = ir;
		    goto have_rmax;
		  }
	    }
    }


have_rmax:

  if (gmax > gmin)
    {
      for (ig = gmin; ig <= gmax; ig++)
	for (ir = rmin; ir <= rmax; ir++)
	  {
	    histp = &histogram[ir][ig][bmin];
	    for (ib = bmin; ib <= bmax; ib++)
	      if (*histp++ != 0)
		{
		  box->gmin = gmin = ig;
		  goto have_gmin;
		}
	  }
    have_gmin:
      if (gmax > gmin)
	for (ig = gmax; ig >= gmin; --ig)
	  for (ir = rmin; ir <= rmax; ir++)
	    {
	      histp = &histogram[ir][ig][bmin];
	      for (ib = bmin; ib <= bmax; ib++)
		if (*histp++ != 0)
		  {
		    box->gmax = gmax = ig;
		    goto have_gmax;
		  }
	    }
    }


have_gmax:

  if (bmax > bmin)
    {
      for (ib = bmin; ib <= bmax; ib++)
	for (ir = rmin; ir <= rmax; ir++)
	  {
	    histp = &histogram[ir][gmin][ib];
	    for (ig = gmin; ig <= gmax; ig++)
	      {
		if (*histp != 0)
		  {
		    box->bmin = bmin = ib;
		    goto have_bmin;
		  }
		histp += B_LEN;
	      }
	  }
    have_bmin:
      if (bmax > bmin)
	for (ib = bmax; ib >= bmin; --ib)
	  for (ir = rmin; ir <= rmax; ir++)
	    {
	      histp = &histogram[ir][gmin][ib];
	      for (ig = gmin; ig <= gmax; ig++)
		{
		  if (*histp != 0)
		    {
		      bmax = ib;
		      goto have_bmax;
		    }
		  histp += B_LEN;
		}
	    }
    }

have_bmax:return;
}



/*******************************/
static void 
assign_color (CBOX * ptr, byte * rp, byte * gp, byte * bp)
/*******************************/
{

  int r, g, b;

  r = ((ptr->rmin + ptr->rmax) << (COLOR_DEPTH - B_DEPTH)) / 2;
  g = ((ptr->gmin + ptr->gmax) << (COLOR_DEPTH - B_DEPTH)) / 2;
  b = ((ptr->bmin + ptr->bmax) << (COLOR_DEPTH - B_DEPTH)) / 2;

  *rp = (byte) r;
  *gp = (byte) g;
  *bp = (byte) b;
}



/*******************************/
static CCELL *
create_colorcell (int r1, int g1, int b1)
/*******************************/
{
  register int i;
  register CCELL *ptr;
  register byte *rp, *gp, *bp;
  int ir, ig, ib;
  long dist, mindist, tmp;

  ir = r1 >> (COLOR_DEPTH - C_DEPTH);
  ig = g1 >> (COLOR_DEPTH - C_DEPTH);
  ib = b1 >> (COLOR_DEPTH - C_DEPTH);

  r1 &= ~1 << (COLOR_DEPTH - C_DEPTH);
  g1 &= ~1 << (COLOR_DEPTH - C_DEPTH);
  b1 &= ~1 << (COLOR_DEPTH - C_DEPTH);

  ptr = (CCELL *) malloc (sizeof (CCELL));
  *(ColorCells + ir * C_LEN * C_LEN + ig * C_LEN + ib) = ptr;
  ptr->num_ents = 0;

  /* step 1: find all colors inside this cell, while we're at
     it, find distance of centermost point to furthest
     corner */

  mindist = 2000000000;

  rp = rmap;
  gp = gmap;
  bp = bmap;
  for (i = 0; i < num_colors; i++, rp++, gp++, bp++)
    if (*rp >> (COLOR_DEPTH - C_DEPTH) == ir &&
	*gp >> (COLOR_DEPTH - C_DEPTH) == ig &&
	*bp >> (COLOR_DEPTH - C_DEPTH) == ib)
      {

	ptr->entries[ptr->num_ents][0] = i;
	ptr->entries[ptr->num_ents][1] = 0;
	++ptr->num_ents;

	tmp = *rp - r1;
	if (tmp < (MAX_COLOR / C_LEN / 2))
	  tmp = MAX_COLOR / C_LEN - 1 - tmp;
	dist = (tmp * tmp * R2FACT);

	tmp = *gp - g1;
	if (tmp < (MAX_COLOR / C_LEN / 2))
	  tmp = MAX_COLOR / C_LEN - 1 - tmp;
	dist += (tmp * tmp * G2FACT);

	tmp = *bp - b1;
	if (tmp < (MAX_COLOR / C_LEN / 2))
	  tmp = MAX_COLOR / C_LEN - 1 - tmp;
	dist += (tmp * tmp * B2FACT);

	if (dist < mindist)
	  mindist = dist;
      }


  /* step 3: find all points within that distance to box */

  rp = rmap;
  gp = gmap;
  bp = bmap;
  for (i = 0; i < num_colors; i++, rp++, gp++, bp++)
    if (*rp >> (COLOR_DEPTH - C_DEPTH) != ir ||
	*gp >> (COLOR_DEPTH - C_DEPTH) != ig ||
	*bp >> (COLOR_DEPTH - C_DEPTH) != ib)
      {

	dist = 0;

	if ((tmp = r1 - *rp) > 0 || (tmp = *rp - (r1 + MAX_COLOR / C_LEN - 1)) > 0)
	  dist += (tmp * tmp * R2FACT);

	if ((tmp = g1 - *gp) > 0 || (tmp = *gp - (g1 + MAX_COLOR / C_LEN - 1)) > 0)
	  dist += (tmp * tmp * G2FACT);

	if ((tmp = b1 - *bp) > 0 || (tmp = *bp - (b1 + MAX_COLOR / C_LEN - 1)) > 0)
	  dist += (tmp * tmp * B2FACT);

	if (dist < mindist)
	  {
	    ptr->entries[ptr->num_ents][0] = i;
	    ptr->entries[ptr->num_ents][1] = dist;
	    ++ptr->num_ents;
	  }
      }


  /* sort color cells by distance, use cheap exchange sort */
  {
    int n, next_n;

    n = ptr->num_ents - 1;
    while (n > 0)
      {
	next_n = 0;
	for (i = 0; i < n; ++i)
	  {
	    if (ptr->entries[i][1] > ptr->entries[i + 1][1])
	      {
		tmp = ptr->entries[i][0];
		ptr->entries[i][0] = ptr->entries[i + 1][0];
		ptr->entries[i + 1][0] = tmp;
		tmp = ptr->entries[i][1];
		ptr->entries[i][1] = ptr->entries[i + 1][1];
		ptr->entries[i + 1][1] = tmp;
		next_n = i;
	      }
	  }
	n = next_n;
      }
  }
  return (ptr);
}




/***************************/
static void 
map_colortable ()
/***************************/
{
  int ir, ig, ib, *histp;
  CCELL *cell;

  histp = &histogram[0][0][0];
  for (ir = 0; ir < B_LEN; ir++)
    for (ig = 0; ig < B_LEN; ig++)
      for (ib = 0; ib < B_LEN; ib++)
	{
	  if (*histp == 0)
	    *histp = -1;
	  else
	    {
	      int i, j;
	      long dist, d2, tmp;

	      cell = *(ColorCells +
		       (((ir >> (B_DEPTH - C_DEPTH)) << C_DEPTH * 2)
			+ ((ig >> (B_DEPTH - C_DEPTH)) << C_DEPTH)
			+ (ib >> (B_DEPTH - C_DEPTH))));

	      if (cell == NULL)
		cell = create_colorcell (ir << (COLOR_DEPTH - B_DEPTH),
					 ig << (COLOR_DEPTH - B_DEPTH),
					 ib << (COLOR_DEPTH - B_DEPTH));

	      dist = 2000000000;
	      for (i = 0; i < cell->num_ents && dist > cell->entries[i][1]; i++)
		{
		  j = cell->entries[i][0];
		  d2 = rmap[j] - (ir << (COLOR_DEPTH - B_DEPTH));
		  d2 = (d2 * d2 * R2FACT);
		  tmp = gmap[j] - (ig << (COLOR_DEPTH - B_DEPTH));
		  d2 += (tmp * tmp * G2FACT);
		  tmp = bmap[j] - (ib << (COLOR_DEPTH - B_DEPTH));
		  d2 += (tmp * tmp * B2FACT);
		  if (d2 < dist)
		    {
		      dist = d2;
		      *histp = j;
		    }
		}
	    }
	  histp++;
	}
}



/*****************************/
static int 
quant_fsdither ()
/*****************************/
{
  register int *thisptr, *nextptr;
  int *thisline, *nextline, *tmpptr;
  int r1, g1, b1, r2, g2, b2;
  int i, j, imax, jmax, oval;
  byte *inptr, *outptr;
  int lastline, lastpixel;

  imax = HIGH - 1;
  jmax = WIDE - 1;

  thisline = (int *) malloc (WIDE * 3 * sizeof (int));
  nextline = (int *) malloc (WIDE * 3 * sizeof (int));

  if (thisline == NULL || nextline == NULL)
    {
      fprintf (stderr, "unable to allocate stuff for the 'dither' routine\n");
      return 1;
    }


  inptr = (byte *) pic24;
  outptr = (byte *) pic8;

  /* get first line of picture */
  for (j = WIDE * 3, tmpptr = nextline; j; j--)
    *tmpptr++ = (int) *inptr++;

  for (i = 0; i < HIGH; i++)
    {
      /* swap thisline and nextline */
      tmpptr = thisline;
      thisline = nextline;
      nextline = tmpptr;
      lastline = (i == imax);

/* HEFA
   if ((i&0x1f) == 0) WaitCursor();
 */

      /* read in next line */
      if (!lastline)
	for (j = WIDE * 3, tmpptr = nextline; j; j--)
	  *tmpptr++ = (int) *inptr++;

      /* dither this line and put it into the output picture */
      thisptr = thisline;
      nextptr = nextline;

      for (j = 0; j < WIDE; j++)
	{
	  lastpixel = (j == jmax);

	  r2 = *thisptr++;
	  g2 = *thisptr++;
	  b2 = *thisptr++;

	  RANGE (r2, 0, MAX_COLOR - 1);
	  RANGE (g2, 0, MAX_COLOR - 1);
	  RANGE (b2, 0, MAX_COLOR - 1);

	  r1 = r2;
	  g1 = g2;
	  b1 = b2;

	  r2 >>= (COLOR_DEPTH - B_DEPTH);
	  g2 >>= (COLOR_DEPTH - B_DEPTH);
	  b2 >>= (COLOR_DEPTH - B_DEPTH);

	  if ((oval = histogram[r2][g2][b2]) == -1)
	    {
	      int ci, cj;
	      long dist, d2, tmp;
	      CCELL *cell;

	      cell = *(ColorCells +
		       (((r2 >> (B_DEPTH - C_DEPTH)) << C_DEPTH * 2)
			+ ((g2 >> (B_DEPTH - C_DEPTH)) << C_DEPTH)
			+ (b2 >> (B_DEPTH - C_DEPTH))));

	      if (cell == NULL)
		cell = create_colorcell (r1, g1, b1);

	      dist = 2000000000;
	      for (ci = 0; ci < cell->num_ents && dist > cell->entries[ci][1]; ci++)
		{
		  cj = cell->entries[ci][0];
		  d2 = (rmap[cj] >> (COLOR_DEPTH - B_DEPTH)) - r2;
		  d2 = (d2 * d2 * R2FACT);
		  tmp = (gmap[cj] >> (COLOR_DEPTH - B_DEPTH)) - g2;
		  d2 += (tmp * tmp * G2FACT);
		  tmp = (bmap[cj] >> (COLOR_DEPTH - B_DEPTH)) - b2;
		  d2 += (tmp * tmp * B2FACT);
		  if (d2 < dist)
		    {
		      dist = d2;
		      oval = cj;
		    }
		}
	      histogram[r2][g2][b2] = oval;
	    }

	  *outptr++ = oval;

	  r1 -= rmap[oval];
	  g1 -= gmap[oval];
	  b1 -= bmap[oval];

	  /* don't use tables, because r1,g1,b1 could go negative */
	  if (!lastpixel)
	    {
	      thisptr[0] += (r1 < 0) ? (r1 * 7 - 8) / 16 : (r1 * 7 + 8) / 16;
	      thisptr[1] += (g1 < 0) ? (g1 * 7 - 8) / 16 : (g1 * 7 + 8) / 16;
	      thisptr[2] += (b1 < 0) ? (b1 * 7 - 8) / 16 : (b1 * 7 + 8) / 16;
	    }

	  if (!lastline)
	    {
	      if (j)
		{
		  nextptr[-3] += (r1 < 0) ? (r1 * 3 - 8) / 16 : (r1 * 3 + 8) / 16;
		  nextptr[-2] += (g1 < 0) ? (g1 * 3 - 8) / 16 : (g1 * 3 + 8) / 16;
		  nextptr[-1] += (b1 < 0) ? (b1 * 3 - 8) / 16 : (b1 * 3 + 8) / 16;
		}

	      nextptr[0] += (r1 < 0) ? (r1 * 5 - 8) / 16 : (r1 * 5 + 8) / 16;
	      nextptr[1] += (g1 < 0) ? (g1 * 5 - 8) / 16 : (g1 * 5 + 8) / 16;
	      nextptr[2] += (b1 < 0) ? (b1 * 5 - 8) / 16 : (b1 * 5 + 8) / 16;

	      if (!lastpixel)
		{
		  nextptr[3] += (r1 < 0) ? (r1 - 8) / 16 : (r1 + 8) / 16;
		  nextptr[4] += (g1 < 0) ? (g1 - 8) / 16 : (g1 + 8) / 16;
		  nextptr[5] += (b1 < 0) ? (b1 - 8) / 16 : (b1 + 8) / 16;
		}
	      nextptr += 3;
	    }
	}
    }

  free (thisline);
  free (nextline);
  return 0;
}

/***************************************************************/
byte *
Conv8to24 (byte * pic8, int w, int h, byte * rmap, byte * gmap, byte * bmap)
{
  /* converts an w*h 8-bit image (with colormap rmap,gmap,bmap) into a
   * 24-bit image.  Note, 'pic8' could be NULL
   *
   * returns pointer to new 24-bits-per-pixel image (w*h) if successful,
   * or NULL if unsuccessful
   */

  int i;
  byte *pic24, *sp, *dp;

  pic24 = (byte *) malloc (w * h * 3);
  if (!pic24)
    return pic24;

  for (i = w * h, sp = pic8, dp = pic24; i; i--, sp++)
    {

/* HEFA 
   if ((i&0x1ffff)==0) WaitCursor();
 */

      *dp++ = rmap[*sp];
      *dp++ = gmap[*sp];
      *dp++ = bmap[*sp];
    }

  return pic24;
}
