/*
   Hilfskonstanten und Header fr Vectorrechnung

   This program ist free software; you can redistribute ist and/or
   modify it under the terms of the GNU General Public License as
   publisched by the Free Software Foundation; either version 2 of
   the License, or (at your opption) any later version.

   This program is distributed in the hope that it well be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Copyright (C) 1996, 1997 Helmut Fahrion
 */

#ifndef _VECT_IST_INCLUDE_
#define _VECT_IST_INCLUDE_

#include <math.h>
#include <iostream.h>

#ifndef _G_HAVE_BOOL
#define bool    uns char
#define true    1
#define false   0
#endif 


/*  Fr Kreise Ellipsen und hnliches */
#define ln05    -0.6931471805599
#define pi2      6.28318548202514648437
#define pi       M_PI
#define pi05     M_PI/2.0
#define pi180    0.01745329238474369049
#define pim180 565.4866943359375
#define eps1     0.1
#define eps2     0.01
#define eps3     0.001
#define eps4     0.0001
#define eps5     0.00001
#define eps6     0.000001
#define eps7     0.0000001
#define eps8     0.00000001
#define eps9     0.000000001
#define eps10    0.0000000001
#define eps11    0.00000000001
#define eps12    0.000000000001

// Hchstwert fr Farbanteile
#define MAXRED   1.0
#define MAXGREEN 1.0
#define MAXBLUE  1.0

// Hchstwert fr den Grad der Gleichung
#define MAX_ORDER  15

// Fr vect.C fr Berechnung von Polynomen

// #define EPSILON        1.0e-10
#define EPSILON        1.0e-20
#define COEFF_LIMIT    1.0e-20
#define MAX_ITERATIONS 150           // war 50
#define TWO_PI_3       2.0943951023931954923084
#define TWO_PI_43      4.1887902047863909846168
#define FUDGE_FACTOR1  1.0e12
#define FUDGE_FACTOR2 -1.0e-5
#define FUDGE_FACTOR3  1.0e-7

/* These values determine the minumum and maximum distances
   that qualify as ray-object intersections */
#define Small_Tolerance 0.001
#define Max_Distance    1.0e7

#define VNULL           vector(0.0, 0.0, 0.0)
#define CNULL           color(0.0, 0.0, 0.0)

#define uns unsigned

typedef uns char  byte;	// 0..255          8 Bit
typedef uns short word;	// 0..65535       16 Bit
typedef uns int   dword;	// 0..4294967295  32 Bit 
typedef uns long  longint;	// 0..1.84467e+19 64 Bit

/*
   auf 386/486/586 Prozessoren

   float       32-bit floating point
   double      64-bit floating point
   long double 96-bit floating point
*/

// 4D Vektor fr nurbs
typedef struct vect4d_struct
  {
    float x, y, z, g;
  }
vect4d;


//  Datentypen zur Vectorrechnung
// 3D Vektor
class vector
{
public:

  // Construktoren
  vector ()
  {
  }
  vector (float x1, float y1, float z1)
  {
    x = x1;
    y = y1;
    z = z1;
  }

  // Destructor
  //~vector(){}  

  // Operatoren zur Vektorrechnung  
  vector operator && (vector b)	// Kreuzprodukt
  {
    return vector (y * b.z - z * b.y,
		   -x * b.z + z * b.x,
		   x * b.y - y * b.x);
  }

  float operator *(vector a)	// Skalarprodukt von Vektoren
  {
    return (x * a.x + y * a.y + z * a.z);
  }

  vector operator - (vector b)	// Vector Subtraktion x = a - b
  {
    return vector (x - b.x, y - b.y, z - b.z);
  }

  void operator -= (vector b)	// Vector Subtraktion a = a-b
  {
    x -= b.x;
    y -= b.y;
    z -= b.z;
  }

  vector operator + (vector b)	// Vector Subtraktion r= a+b
  {
    return vector (x + b.x, y + b.y, z + b.z);
  }

  void operator += (vector b)	// Vector Addition r= a+b
  {
    x += b.x;
    y += b.y;
    z += b.z;
  }

  vector operator *(float t)	// Multipliziert mit konstanter x= a*t
  {
    return vector (x * t, y * t, z * t);
  }

  void operator /= (float t)	// Dividiert mit konstanter x= a / t
  {
    x /= t;
    y /= t;
    z /= t;
  }

  void operator /= (vector a)	// Dividiert mit Vector
  {
    x /= a.x;
    y /= a.y;
    z /= a.z;
  }

  vector operator / (vector a)	// Dividiert Vector
  {
    return vector (x / a.x, y / a.y, z /= a.z);
  }

  vector operator / (float t)	// Dividiert mit konstanter t
  {
    return vector (x / t, y / t, z / t);
  }

  void operator *= (float t)	// Multipliziert mit konstanter a= a*t
  {
    x *= t;
    y *= t;
    z *= t;
  }

  vector operator = (vector b)	// Vektor zuweisen x = y = z;
  {
    x = b.x;
    y = b.y;
    z = b.z;
    return (b);
  }

  vector operator ! (void)	// Negation eines Vektors
  {
    return vector (-x, -y, -z);
  }

  void operator |= (vector b)	// normiert einen Vector
  {
    float s = sqrt (b * b);
    if (fabs (s) <= eps8)
      {
	x = 0.0; y = 0.0; z = 0.0;
      }
    else
      {
	s = 1.0 / s; x = s * b.x; y = s * b.y; z = s * b.z;
      }
  }

  float operator || (vector b)	// Abstand zweier Vektoren
  {
    float X, Y, Z;
    X = x - b.x; Y = y - b.y; Z = z - b.z;
    X = X * X + Y * Y + Z * Z;
    return ((fabs (X) > eps5) ? sqrt (X) : 0.0);
  }

  float operator < (vector b)	// Winkel zwischen 2 Vektoren, Richtungscosinus
  {
    float a = sqrt (b.x * b.x + b.y * b.y + b.z * b.z) * sqrt (x * x + y * y + z * z);
    if (fabs (a) > eps5)
      {
	a = (x * b.x + y * b.y + z * b.z) / a;
	if (fabs (a) <= 1.0) return acos (a);
      }
    return 0.0;
  }

  unsigned char operator == (vector b)	// Test auf gesetzen Wert von b
  {
    return ((x == b.x && y == b.y && z == b.z) ? 1 : 0);
  }

  friend ostream & operator << (ostream & os, vector v)
  {
    return os << v.x << " " << v.y << " " << v.z;
  }

  float x, y, z;		// Position
};


// Ojekttyp Farbe
class color
{
public:

  // Construktoren
  color ()
  {
  }

  color (float R, float G, float B)
  {
    r = R; g = G; b = B;
  }

  // Destruktor
  //  ~color() { }

  color operator = (color c)	// Farbe zuweisen x = y = z;
  {
    r = (c.r < MAXRED) ? c.r : MAXRED;
    g = (c.g < MAXGREEN) ? c.g : MAXGREEN;
    b = (c.b < MAXBLUE) ? c.b : MAXBLUE;
    return (c);
  }

  color operator * (float k)	// Dimmerfunktion zur Helligkeitsregelung
  {
    return color (k * r, k * g, k * b);
  }

  color operator / (float k)	// Division zur Mittelwertberechnung
  {
    return color (r / k, g / k, b / k);
  }

  void operator *= (float k)	// Dimmerfunktion zur Helligkeitsregelung
  {
    r *= k; g *= k; b *= k;
  }

  void operator /= (float k)	// Division zur Mittelwertberechnung
  {
    r /= k; g /= k; b /= k;
  }

  color operator * (color c)	// Farbmischung
  {
    return color (r * c.r, g * c.g, b * c.b);
  }

  void operator *= (color c)	// Farbmischung
  {
    r *= c.r; g *= c.g; b *= c.b;
  }

  color operator + (color c)
  {
    return color (r + c.r, g + c.g, b + c.b);
  }

  color operator - (color c)
  {
    return color (r - c.r, g - c.g, b - c.b);
  }

  void operator += (color c)
  {
    r += c.r; g += c.g; b += c.b;
  }

  void operator -= (color c)
  {
    r -= c.r; g -= c.g; b -= c.b;
  }

  void operator |= (color cb)	// normiert einen Vector
  {
    float s = sqrt (cb.r * cb.r + cb.g * cb.g + cb.b * cb.b);
    if (fabs (s) <= eps8)
      {
	r = 0.0; g = 0.0; b = 0.0;
      }
    else
      {
	s = 1.0 / s; r = s * cb.r; g = s * cb.g; b = s * cb.b;
      }
  }

  void operator <= (color c)	// auf Maximum begrenzen
  {
    r = (c.r > MAXRED) ? MAXRED : c.r;
    g = (c.g > MAXGREEN) ? MAXGREEN : c.g;
    b = (c.b > MAXBLUE) ? MAXBLUE : c.b;
  }

  friend ostream & operator << (ostream & os, color c)
  {
    return os << c.r << " " << c.g << " " << c.b;
  }

  void max(color c)
  {
    if (c.r > r) r = c.r;
    if (c.g > g) g = c.g;
    if (c.b > b) b = c.b;
  }

  float r, g, b;
};


// Globale Funktionen zur Vektorrechnung
// Diese Funktionen sind inline Deklariert damit gegenueber einer Implementation
// in C, mit Makros, keine Performanceeinbuse eintreten
// (Es wird hier kein Code generiert, sondern Anstelle eines Aufrufes der Funktion
// wird an dieser Stelle der Code eingebunden (Makro))

// Berechnet den Abstand eines Punktes von einer Ebene
float abs_punkt_ebene (vector p, vector e0, vector e1, vector e2);

// Berechnet Vektor von einem Punkt zu einer Geraden
vector absgeradepunkt (vector pg, vector r, vector p);

// rundet eine Gleitkommazahl in ein Word
unsigned int f2w (float a);

// Betrag eines int (in stdlib definiert)
int abs_int (int a);

// Berechnet den Betrag eines Vektors
float abs (vector a);

// Berechnet den Betrag einer Farbe
float abs (color a);

// Vorzeichen einer Gleitkommazahl
float fsign (float a);

// Vergleicht 2 Gleitkommazahlen mit einer Schranke e
unsigned char feq (float a, float b, float e);

// Dmpfung einer Funktion
float damping (float von, float bis, float x);

// rotiert einen Vektor r nach einen Vektor i
void rotationVN (vector * r, vector i);

// rotiert einen Vektor in Richtung i zur Z Achse
void rotationVZ (vector * r, vector i);

// Rotiert einen Vektor um einen Bestimmten Punkt
void rotation (vector * e, vector p, vector w);

// Berechnet die Lsung einer Qadratischen Gleichung
int solve_quad (float a, float b, float c, float *t1, float *t2);

// Funktionen die zur Arbeit mit Vektoren bentigt werden
// aber beliebig auf alle Vektoren angewendet werden
bool test_gleich (vector v1, vector v2);

// Testet ob 4 Vektoren eine Ebene bilden
unsigned char testebene (vector r0, vector r1, vector r2, vector r3);

// Berechnet Kubische Gleichung
int solve_cubic (double *x, double *y);

// Solve a quartic using the method of Francois Vieta (Circa 1735)
int solve_quartic (double *x, double *results);

// Root solver based on the Sturm sequences for a polynomial
int polysolve (int order, double *Coeffs, double *roots);


#endif


