#ifndef _PLEPH_IST_INCLUDE_
#define _PLEPH_IST_INCLUDE_

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <iostream.h>

#define WAIT getc(stdin)

#define TRUE_ (1)
#define FALSE_ (0)

/*========================================================================+
|                                 DVECTOR                                 |
+=========================================================================+
| AUTOR: K. ARFA-KABOODVAND (Aero-Space Engineer)                         |
+-------------------------------------------------------------------------+
| DESCRIPTION:                                                            |
| Allocate a double vector with subscript range v[nl..nh].                |
+-------------------------------------------------------------------------+
| INPUTS : nl = Start dimension (subscript range) of the vector           |
|          nh = End dimension (subscript range) of the vector             |
+-------------------------------------------------------------------------+
| OUTPUTS: dvector = Allocated vector                                     |
+-------------------------------------------------------------------------+
| REFERENCE: NUMERICAL RECEPIES IN C                                      |
+========================================================================*/
double *dvector(int nl,int nh);

/*------------------------------------------------------------------------+
| INPUTS : nrl = Starting row number (subscript range) of the matrix      |
|          nrh = Ending row number (subscript range) of the matrix        |
|          ncl = Starting column number (subscript range) of the matrix   |
|          nch = Ending column number (subscript range) of the matrix     |
+-------------------------------------------------------------------------+
| OUTPUTS: dmatrix = Allocated matrix                                     |
+-------------------------------------------------------------------------+
| REFERENCE: %                                                            |
+========================================================================*/

char **Cmatrix(int nrl, int nrh, int ncl, int nch);

/*========================================================================+
|                               INITIAL                                   |
+=========================================================================+
| AUTOR: K. ARFA-KABOODVAND (Aero-Space Engineer)                         |
+-------------------------------------------------------------------------+
| DESCRIPTION:                                                            |
| INITIALIZE THE CONSTANTS FROM THE JPL BINARY FILE.                      |
+-------------------------------------------------------------------------+
| INPUTS : %                                                              |
+-------------------------------------------------------------------------+
| OUTPUTS: FP_BIN = POINTER TO THE JPL BINARY FILE.                       |
+-------------------------------------------------------------------------+
| REFERENCE: %                                                            |
+========================================================================*/
FILE *INITIAL(char *Name);
/*========================================================================+
|                          FREE_DVECTOR(DOUBLE)                           |
+=========================================================================+
| AUTOR: K. ARFA-KABOODVAND (Aero-Space Engineer)                         |
+-------------------------------------------------------------------------+
| DESCRIPTION:                                                            |
| Free a double vector allocated with dvector().                          |
+-------------------------------------------------------------------------+
| INPUTS : v  = the vector, which should be free.                         |
|          nl = The starting number of row.                               |
+-------------------------------------------------------------------------+
| OUTPUTS: %                                                              |
+-------------------------------------------------------------------------+
| REFERENCE: NUMERICAL RECEPIES IN C                                      |
+========================================================================*/
void free_dvector(double *v,int nl);

// gibt alles wieder frei
void freeall(void);

/*========================================================================+
|                             FREE_MATRIX(CHAR)                           |
+=========================================================================+
| AUTOR: K. ARFA-KABOODVAND (Aero-Space Engineer)                         |
+-------------------------------------------------------------------------+
| DESCRIPTION:                                                            |
| Free a double matrix allocated with dmatrix().                          |
+-------------------------------------------------------------------------+
| INPUTS : m  = the matrix, which should be free.                         |
|          nrl= The starting number of row.                               |
|          ncl= The starting number of column.                            |
+-------------------------------------------------------------------------+
| OUTPUTS: %                                                              |
+-------------------------------------------------------------------------+
| REFERENCE: %                                                            |
+========================================================================*/
void free_Cmatrix(char **m, int nrl, int ncl);

/*========================================================================+
|                               STATE                                     |
+=========================================================================+
| AUTOR: K. ARFA-KABOODVAND (Aero-Space Engineer)                         |
+-------------------------------------------------------------------------+
| DESCRIPTION:                                                            |
| THIS SUBROUTINE READS AND INTERPOLATES THE JPL PLANETARY EPHEMERIS FILE.|
+-------------------------------------------------------------------------+
| INPUTS :                                                                |
|         JED   DP 2-WORD JULIAN EPHEMERIS EPOCH AT WHICH INTERPOLATION   |
|               IS WANTED.  ANY COMBINATION OF JED(1)+JED(2) WHICH FALLS  |
|               WITHIN THE TIME SPAN ON THE FILE IS A PERMISSIBLE EPOCH.  |
|                A. FOR EASE IN PROGRAMMING, THE USER MAY PUT THE         |
|                   ENTIRE EPOCH IN JED(1) AND SET JED(2)=0.              |
|                B. FOR MAXIMUM INTERPOLATION ACCURACY, SET JED(1) =      |
|                   THE MOST RECENT MIDNIGHT AT OR BEFORE INTERPOLATION   |
|                   EPOCH AND SET JED(2) = FRACTIONAL PART OF A DAY       |
|                   ELAPSED BETWEEN JED(1) AND EPOCH.                     |
|                C. AS AN ALTERNATIVE, IT MAY PROVE CONVENIENT TO SET     |
|                   JED(1) = SOME FIXED EPOCH,SUCH AS START OF INTEGRATION|
|                   AND JED(2) = ELAPSED INTERVAL BETWEEN THEN AND EPOCH. |
|        LIST   12-WORD INTEGER ARRAY SPECIFYING WHAT INTERPOLATION       |
|               IS WANTED FOR EACH OF THE BODIES ON THE FILE.             |
|                         LIST(I)=0, NO INTERPOLATION FOR BODY I          |
|                                =1, POSITION ONLY                        |
|                                =2, POSITION AND VELOCITY                |
|               THE DESIGNATION OF THE ASTRONOMICAL BODIES BY I IS:       |
|                         I = 1: MERCURY                                  |
|                           = 2: VENUS                                    |
|                           = 3: EARTH-MOON BARYCENTER                    |
|                           = 4: MARS                                     |
|                           = 5: JUPITER                                  |
|                           = 6: SATURN                                   |
|                           = 7: URANUS                                   |
|                           = 8: NEPTUNE                                  |
|                           = 9: PLUTO                                    |
|                           =10: GEOCENTRIC MOON                          |
|                           =11: NUTATIONS IN LONGITUDE AND OBLIQUITY     |
|                           =12: LUNAR LIBRATIONS (IF ON FILE)            |
|                                                                         |
+-------------------------------------------------------------------------+
| OUTPUTS:                                                                |
|          PV   DP 6 X 11 ARRAY THAT WILL CONTAIN REQUESTED INTERPOLATED  |
|               QUANTITIES.  THE BODY SPECIFIED BY LIST(I) WILL HAVE ITS  |
|               STATE IN THE ARRAY STARTING AT PV(1,I).  (ON ANY GIVEN    |
|               CALL, ONLY THOSE WORDS IN 'PV' WHICH ARE AFFECTED BY THE  |
|              FIRST 10 'LIST' ENTRIES (AND BY LIST(12) IF LIBRATIONS ARE |
|               ON THE FILE) ARE SET.  THE REST OF THE 'PV' ARRAY         |
|               IS UNTOUCHED.)  THE ORDER OF COMPONENTS STARTING IN       |
|               PV(1,I) IS: X,Y,Z,DX,DY,DZ.                               |
|               ALL OUTPUT VECTORS ARE REFERENCED TO THE EARTH MEAN       |
|               EQUATOR AND EQUINOX OF EPOCH. THE MOON STATE IS ALWAYS    |
|               GEOCENTRIC; THE OTHER NINE STATES ARE EITHER HELIOCENTRIC |
|               OR SOLAR-SYSTEM BARYCENTRIC, DEPENDING ON THE SETTING OF  |
|               COMMON FLAGS (SEE BELOW).                                 |
|               LUNAR LIBRATIONS, IF ON 12, ARE PUT INTO PV(K,11) IF      |
|               LIST(12) IS 1 OR 2.                                       |
|         NUT   DP 4-WORD ARRAY THAT WILL CONTAIN NUTATIONS AND RATES,    |
|               DEPENDING ON THE SETTING OF LIST(11).  THE ORDER OF       |
|               QUANTITIES IN NUT IS:                                     |
|                        D PSI  (NUTATION IN LONGITUDE)                   |
|                        D EPSILON (NUTATION IN OBLIQUITY)                |
|                        D PSI DOT                                        |
|                        D EPSILON DOT                                    |
+-------------------------------------------------------------------------+
| REFERENCE: JPL (FORTRAN CODE)                                           |
+========================================================================*/

int STATE(FILE *fp_Bin,double *jed,int *list,double *pv,double *nut);

/*========================================================================+
|                                PLEPH0                                   |
+=========================================================================+
| AUTOR: K. ARFA-KABOODVAND (Aero-Space Engineer)                         |
+-------------------------------------------------------------------------+
| DESCRIPTION:                                                            |
| THIS SUBROUTINE READS THE JPL PLANETARY EPHEMERIS & GIVES THE POSITION  |
| AND VELOCITY OF THE POINT 'TARG'WITH RESPECT TO 'CENT'.                 |
+-------------------------------------------------------------------------+
| INPUTS :                                                                |
|       JD = D.P. JULIAN EPHEMERIS DATE AT WHICH INTERPOLATION IS WANTED. |
|       ** NOTE THE ENTRY DPLEPH FOR A DOUBLY-DIMENSIONED TIME **         |
|          THE REASON FOR THIS OPTION IS DISCUSSED IN THE SUB. STATE      |
|      TARG = int NUMBER OF 'TARGET' POINT.                               |
|      CENT = int NUMBER OF CENTER POINT.                                 |
|             THE NUMBERING CONVENTION FOR 'TARG' AND 'CENT' IS:          |
|                 1 = MERCURY         8 = NEPTUNE                         |
|                 2 = VENUS           9 = PLUTO                           |
|                 3 = EARTH          10 = MOON                            |
|                 4 = MARS           11 = SUN                             |
|                 5 = JUPITER        12 = SOLAR-SYSTEM BARYCENTER         |
|                 6 = SATURN         13 = EARTH-MOON BARYCENTER           |
|                 7 = URANUS         14 = NUTATIONS (intITUDE AND OBLIQ)  |
|                15 = LIBRATIONS, IF ON EPH FILE                          |
|               (IF NUTATIONS ARE WANTED, SET TARG = 14. FOR LIBRATIONS,  |
|               SET TARG = 15. 'CENT' WILL BE IGNORED ON EITHER CALL.)    |
+-------------------------------------------------------------------------+
| OUTPUTS:                                                                |
|       RRD = OUTPUT 6-WORD D.P. ARRAY CONTAINING POSITION AND VELOCITY   |
|             OF POINT 'TARG' RELATIVE TO 'CENT'. THE UNITS ARE AU AND    |
|             AU/DAY. FOR LIBRATIONS THE UNITS ARE RADIANS AND RADIANS    |
|             PER DAY. IN THE CASE OF NUTATIONS THE FIRST FOUR WORDS OF   |
|             RRD WILL BE SET TO NUTATIONS AND RATES, HAVING UNITS OF     |
|             RADIANS AND RADIANS/DAY.                                    |
|             NOTE: IN MANY CASES THE USER WILL NEED ONLY POSITION        |
|                   VALUES FOR EPHEMERIDES OR NUTATIONS. FOR              |
|                   POSITION-ONLY OUTPUT, THE int VARIABLE 'IPV'          |
|                   IN THE COMMON AREA /PLECOM/ SHOULD BE SET = 1         |
|                   BEFORE THE NEXT CALL TO PLEPH. (ITS DEFAULT           |
|                   VALUE IS 2, WHICH RETURNS BOTH POSITIONS AND          |
|                   RATES.)                                               |
|                                                                         |
|      INSIDE is .TRUE. if the input Julian Ephemeris Date (JD) is within |
|             the ephemeris time span.  If not, INSIDE is set to .FALSE.  |
+-------------------------------------------------------------------------+
| REFERENCE: JPL (Fortran code)                                           |
+========================================================================*/
int PLEPH0(FILE *fp_Bin,int n,double jd,int targ,int cent,
	   double *rrd, int *inside, double *jd2);

int PLEPH(FILE *fp_Bin,double jd,int targ,int cent,double *rrd,int *inside);

int DPLEPH(FILE *fp_Bin,double *jd2,int targ,int cent,double *rrd);



#endif



