#ifndef _VECT_IST_INCLUDE_
#define _VECT_IST_INCLUDE_

/*  
   Hilfskonstanten und Header fr Vectorrechnung 

   This program ist free software; you can redistribute ist and/or
   modify it under the terms of the GNU General Public License as
   publisched by the Free Software Foundation; either version 2 of
   the License, or (at your opption) any later version.

   This program is distributed in the hope that it well be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Copyright (C) 1996 Helmut Fahrion
*/

#include <math.h>

/* Hohe Genauigkeit */
#define float double

/*  Fr Kreise Ellipsen und hnliches */
#define ln05    -0.6931471805599
#define pi2      6.28318548202514648437
#define pi       3.14159265358979323846
#define pi05     1.57079632679489661923
#define pi180    0.01745329238474369049
#define pim180 565.4866943359375
#define eps1     0.1
#define eps2     0.01
#define eps3     0.001
#define eps4     0.0001
#define eps5     0.00001
#define eps8     0.00000001
#define eps12    0.000000000001

// Hchstwert fr Farbanteile
#define MAXRED   1.0
#define MAXGREEN 1.0
#define MAXBLUE  1.0

// Hchstwert fr den Grad der Gleichung
#define MAX_ORDER  15

#define TRUE  1
#define FALSE 0

//  Datentypen zur Vectorrechnung
// 3D Vektor
class vector 
{  
 public: 
 
  // Construktoren
  vector() { };
  vector(float x1, float y1, float z1) 
    { x=x1;  y=y1; z=z1; }

  // Destructor
  //~vector(){}  

  // Operatoren zur Vektorrechnung  
  vector operator &&(vector b)     // Kreuzprodukt
  {
    return vector(y * b.z - z * b.y,
                 -x * b.z + z * b.x,
                  x * b.y - y * b.x );
  }

  float operator *(vector a)        // Skalarprodukt von Vektoren
  {
    return(x * a.x + y * a.y + z * a.z);
  }
  
  vector operator -(vector b)     // Vector Subtraktion x = a - b
  {
    return vector(x - b.x, y - b.y, z - b.z);
  }

  void operator -=(vector b)     // Vector Subtraktion a = a-b
  {
    x -= b.x; y -= b.y; z -= b.z;
  }

  vector operator +(vector b)     // Vector Subtraktion r= a+b
  {
    return vector(x + b.x, y + b.y, z + b.z);
  }

  void operator +=(vector b)     // Vector Addition r= a+b
  {
    x += b.x; y += b.y; z += b.z;
  }

  vector operator *(float t)      // Multipliziert mit konstanter x= a*t
  {
    return vector(x * t, y * t, z * t);
  }

  void operator *=(float t)      // Multipliziert mit konstanter a= a*t
  {
    x *= t; y *= t; z *= t;
  }
 
  vector operator =(vector b)     // Vektor zuweisen x = y = z;
  {
    x = b.x; y = b.y; z = b.z;
    return(b);
  }
  
  void operator |=(vector b)     // normiert einen Vector
  {
    float s = sqrt(b*b);
    if (s == 0.0) 
    {
      x = 0.0; y = 0.0; z = 0.0;
    }
    else 
    {
      s = 1.0 / s; 
      x = s * b.x; y = s * b.y; z = s * b.z;
    }  
  }
  
  float  operator ||(vector b)    // Abstand zweier Vektoren
  { 
    float X, Y, Z;
    X = x - b.x;
    Y = y - b.y;
    Z = z - b.z;
    return(sqrt(X*X+Y*Y+Z*Z));
  }

  float  x, y, z;          // Position

};


// Ojekttyp Farbe
class color
{  
 public: 

  // Construktoren
  color() { }

  color(float R, float G, float B)
  {
    r = R; g = G; b = B;
  }
  
  // Destruktor
  //  ~color() { }

  color operator =(color c)    // Farbe zuweisen x = y = z;
  {
    r = (c.r < MAXRED)   ? c.r : MAXRED;
    g = (c.g < MAXGREEN) ? c.g : MAXGREEN;
    b = (c.b < MAXBLUE)  ? c.b : MAXBLUE;
    return(c);
  }

  color operator *(float k)    // Dimmerfunktion zur Helligkeitsregelung
  { 
    return color(k * r, k * g, k * b);  
  }

  void operator *=(float k)    // Dimmerfunktion zur Helligkeitsregelung
  {
    r *= k; g *= k; b *= k;
  }

  void operator /=(float k)    // Division zur Mittelwertberechnung
  {
    r /= k; g /= k; b /= k;
  }

  color operator *(color c)    // Farbmischung
  { 
    return color(r * c.r, g * c.g, b * c.b);  
  }

  void operator *=(color c)    // Farbmischung
  {
    r *= c.r; g *= c.g; b *= c.b;
  }

  color operator +(color c)
  {
    return color(r + c.r, g + c.g, b + c.b);
  }

  void operator +=(color c)
  {
    r += c.r; g += c.g; b += c.b;
  }
  
  void MAX(void)
  {
    if (r > MAXRED)   r = MAXRED;
    if (g > MAXGREEN) g = MAXGREEN;
    if (b > MAXBLUE)  b = MAXBLUE;    
  }
  
 float r, g, b;
  
};

// Globale Funktionen zur Vektorrechnung

// Rotiert einen Vektor um einen Bestimmten Punkt
void rotation(vector *e, vector p, vector w);

// rundet eine Gleitkommazahl in ein Integer
unsigned int f2w(float a);

// Betrag eines int (in stdlib definiert)
//int abs(int a);

// Berechnet den Betrag eines Vektors
float abs(vector a);

// Vorzeichen einer Gleitkommazahl
float signf(float a);

// Berechnet die Lsung einer Qadratischen Gleichung
int squareequation(float a, float b, float c, float *t1, float *t2);

// Berechnet Kubische Gleichung
int solve_cubic(float *x, float *y);

// Solve a quartic using the method of Francois Vieta (Circa 1735)
int solve_quartic(float *x, float *results);

// Root solver based on the Sturm sequences for a polynomial
int polysolve(int order, float *Coeffs, float *roots);

#endif














