
/*
   Quellfile fr die Vektorrechnung und Lsung von Gleichungen

   Copyright (C) 1996 Helmut Fahrion

   This program ist free software; you can redistribute ist and/or
   modify it under the terms of the GNU General Public License as
   publisched by the Free Software Foundation; either version 2 of
   the License, or (at your opption) any later version.

   This program is distributed in the hope that it well be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "vect.H"
  

#define EPSILON        1.0e-10
#define COEFF_LIMIT    1.0e-20
#define MAX_ITERATIONS 50
#define TWO_PI_3       2.0943951023931954923084
#define TWO_PI_43      4.1887902047863909846168
#define FUDGE_FACTOR1  1.0e12
#define FUDGE_FACTOR2 -1.0e-5
#define FUDGE_FACTOR3  1.0e-7

/* These values determine the minumum and maximum distances
   that qualify as ray-object intersections */
#define Small_Tolerance 0.001
#define Max_Distance    1.0e7



// Funktionen die zur Arbeit mit Vektoren bentigt werden
// aber beliebig auf alle Vektoren angewendet werden

float abs(vector a)
{
 return(sqrt(a.x*a.x + a.y*a.y + a.z*a.z));
}

// Vorzeichen einer Gleitkommazahl
float signf(float a)
{
  return(a >= 0. ? 1. : -1.);
}

// rundet eine Zahl
unsigned int f2w(float a)
{
  return((unsigned int) (a + .5));
}

void rotation(vector *e, vector p, vector w)
{
  float  sw, cw;
  vector pv;
  
  if (w.x != 0.)
  {
    sw = sin(w.x); cw = cos(w.x);
    // Ergebnis holen & Koord. verschieben
    pv = *e - p;
    // Drehung um x Achse
    *e = vector(pv.x, pv.y*cw-pv.z*sw, pv.y*sw+pv.z*cw) + p;
  }
  
  if (w.y != 0.)
  {
    sw = sin(w.y); cw = cos(w.y);
    // Ergebnis holen & Koord. verschieben
    pv = *e - p;
    // Drehung um y Achse
    *e = vector(pv.x*cw+pv.z*sw, pv.y, pv.x*-sw+pv.z* cw) + p;
  }

  if (w.z != 0.)
  {
    sw = sin(w.z); cw = cos(w.z);    
    // Ergebnis holen & Koord. verschieben
    pv = *e - p;
    // Drehung um z Achse
    *e = vector(pv.x*cw-pv.y*sw, pv.x*sw+pv.y*cw, pv.z) + p;
  }
}
 
// Lsen einer quadratischen Gleichung
// a*t^2 + b*t + c = 0
int squareequation(float a, float b, float c, float *t1, float *t2)
{
  float d;

  // |a| < eps ?
  if (fabs(a) < eps8)
  {
    if (fabs(b) < eps8) 
      return 0;
    else 
    {
       *t1 = -c / b;
       return 1;
    }
  } else 
  {
   // a nicht null
    d = b*b - 4*a*c;
    if (fabs(d) < eps8) 
    {
      *t1 = -b/2./a;
      return 1;
    } 
    else 
    {
      if (d > 0.) 
      {
	*t1  = (-b + sqrt(d))/2./a;
	*t2  = (-b - sqrt(d))/2./a;
	// 2 Reelle Lsungen, t1 ist kleinerer Betrag!
	return 2;
      } // if d>0
      else 
	return 0;
    } // else
  } // else
  return 0; // wird nie erreicht
}

/*
  Funktionen zur Lsung von Gleichungen n-ten Grades

  Kodierung: Alexander Enzmann.
  Anpassung: Helmut Fahrion.
*/

typedef struct p {
  int ord;
  float coef[MAX_ORDER+1];
} 
polynomial;

float polyeval(float x, int n, float *Coeffs)
{
  int i;
  float val;
  val = Coeffs[n];
  for (i=n-1; i>=0 ;i--) 
    val = val * x + Coeffs[i];
  return val;
}

int regula_falsa(int order, float *coef, float a, float b, float *val)
{
  int its;
  float fa, fb, x, fx, lfx;
  
  fa = polyeval(a, order, coef);
  fb = polyeval(b, order, coef);
  
  if (fa * fb > 0.)
    return 0;
  
  if (fabs(fa) < COEFF_LIMIT) 
  {
    *val = a;
    return 1;
  }
  
  if (fabs(fb) < COEFF_LIMIT) 
  {
    *val = b;
    return 1;
  }
  
  lfx = fa;
  
  // COOPERATE
    for (its = 0; its < MAX_ITERATIONS; its++) 
    {
      x = (fb * a - fa * b) / (fb - fa);
      fx = polyeval(x, order, coef);
      
      if (fabs(x) > EPSILON) 
      {
	if (fabs(fx / x) < EPSILON) 
        {
	  *val = x;
	  return 1;
        }
      }
      else if (fabs(fx) < EPSILON) 
      {
	*val = x;
	return 1;
      }
      
      if (fa < 0)
	if (fx < 0) 
	{
	  a = x;
	  fa = fx;
	  if ((lfx * fx) > 0)
	    fb /= 2;
	}
	else 
	{
	  b = x;
	  fb = fx;
	  if ((lfx * fx) > 0)
	    fa /= 2;
	}
      else if (fx < 0) 
      {
	b = x;
	fb = fx;
	if ((lfx * fx) > 0)
	  fa /= 2;
      }
      else 
      {
	a = x;
	fa = fx;
	if ((lfx * fx) > 0)
	  fb /= 2;
      }
      if (fabs(b-a) < EPSILON) 
      {
	// Check for underflow in the domain
	*val = x;
	return 1;
      }
      lfx = fx;
    }
  return 0;
}

/*
   Solve the quadratic equation:
      		x[0] * x^2 + x[1] * x + x[2] = 0.

   The value returned by this function is the number of real roots.
   The roots themselves are returned in y[0], y[1].
*/
int solve_quadratic(float *x, float *y)
{
  float d, t, a, b, c;
  a = x[0];
  b = -x[1];
  c = x[2];
  if (a == 0.) 
  {
    if (b == 0.)
      return 0;
    y[0] = c / b;
    return 1;
  }
  d = b * b - 4. * a * c;
  if (d < 0.)
    return 0;
  else if (fabs(d) < COEFF_LIMIT) 
  {
    y[0] = .5 * b / a;
    return 1;
  }
  d = sqrt(d);
  t = 2. * a;
  y[0] = (b + d) / t;
  y[1] = (b - d) / t;
  return 2;
}


/*
   Solve the cubic equation:

      x[0] * x^3 + x[1] * x^2 + x[2] * x + x[3] = 0.

   The result of this function is an integer that tells how many real
   roots exist.  Determination of how many are distinct is up to the
   process that calls this routine.  The roots that exist are stored
   in (y[0], y[1], y[2]).

   Note: this function relies very heavily on trigonometric functions and
   the square root function.  If an alternative solution is found that does
   not rely on transcendentals this code will be replaced.
*/
int solve_cubic(float *x, float *y)
{
  float Q, R, Q3, R2, sQ, d, an, theta, A2, a0, a1, a2, a3;
  a0 = x[0];
  if (a0 == 0.) 
    return solve_quadratic(&x[1], y);
  else 
    if (a0 != 1.0) 
  {
    a1 = x[1] / a0;
    a2 = x[2] / a0;
    a3 = x[3] / a0;
  }
  else 
  {
    a1 = x[1];
    a2 = x[2];
    a3 = x[3];
  }
  A2 = a1 * a1;
  Q = (A2 - 3.0 * a2) / 9.0;
  R = (2.0 * A2 * a1 - 9.0 * a1 * a2 + 27.0 * a3) / 54.0;
  Q3 = Q * Q * Q;
  R2 = R * R;
  d = Q3 - R2;
  an = a1 / 3.0;
  if (d >= 0.0) 
  {
    /* Three real roots. */
    d = R / sqrt(Q3);
    theta = acos(d) / 3.0;
    sQ = -2.0 * sqrt(Q);
    y[0] = sQ * cos(theta) - an;
    y[1] = sQ * cos(theta + TWO_PI_3) - an;
    y[2] = sQ * cos(theta + TWO_PI_43) - an;
    return 3;
  }
  else 
  {
    sQ = pow(sqrt(R2 - Q3) + fabs(R), 1.0 / 3.0);
    if (R < 0)
      y[0] = (sQ + Q / sQ) - an;
    else
      y[0] = -(sQ + Q / sQ) - an;
    return 1;
  }
}

/* Test to see if any coeffs are more than 6 orders of magnitude
   larger than the smallest */
static int difficult_coeffs(int n, float *x)
{
  int i, flag = 0;
  float t, biggest;

  biggest = fabs(x[0]);
  for (i=1;i<=n;i++) 
  {
    t = fabs(x[i]);
    if (t > biggest)
      biggest = t;
  }
  
  /* Everything is zero no sense in doing any more */
  if (biggest == 0.0) return 0;
  
  for (i=0;i<=n;i++)
    if (x[i] != 0.0)
      if (fabs(biggest / x[i]) > FUDGE_FACTOR1) 
      {
	x[i] = 0.0;
	flag = 1;
      }
  
  return flag;
}

/*
 * modp
 *
 *   calculates the modulus of u(x) / v(x) leaving it in r, it
 *  returns 0 if r(x) is a constant.
 *  note: this function assumes the leading coefficient of v 
 *   is 1 or -1
 */
static int modp(polynomial *u, polynomial *v, polynomial *r)
{
  int    i, k, j;
  for (i=0;i<u->ord;i++)
    r[i] = u[i];

  if (v->coef[v->ord] < 0.0) 
  {
    for (k = u->ord - v->ord - 1; k >= 0; k -= 2)
      r->coef[k] = -r->coef[k];
    for (k = u->ord - v->ord; k >= 0; k--)
      for (j = v->ord + k - 1; j >= k; j--)
	r->coef[j] = -r->coef[j] - r->coef[v->ord + k] * v->coef[j - k];
  }
  else 
  {
    for (k = u->ord - v->ord; k >= 0; k--)
      for (j = v->ord + k - 1; j >= k; j--)
	r->coef[j] -= r->coef[v->ord + k] * v->coef[j - k];
  }
  
  k = v->ord - 1;
  while (k >= 0 && fabs(r->coef[k]) < COEFF_LIMIT) 
  {
    r->coef[k] = 0.0;
    k--;
  }
  r->ord = (k < 0) ? 0 : k;
  return(r->ord);
}

/* Build the sturmian sequence for a polynomial */
int buildsturm(int ord, polynomial *sseq)
{
  int i;
  float f, *fp, *fc;
  polynomial *sp;

  sseq[0].ord = ord;
  sseq[1].ord = ord - 1;

  /* calculate the derivative and normalize the leading coefficient. */
  f = fabs(sseq[0].coef[ord] * ord);
  fp = sseq[1].coef;
  fc = sseq[0].coef + 1;
  for (i = 1; i <= ord; i++)
    *fp++ = *fc++ * i / f;
  
  /* construct the rest of the Sturm sequence */
  for (sp = sseq + 2; modp(sp - 2, sp - 1, sp); sp++) 
  {
    /* reverse the sign and normalize */
    f = -fabs(sp->coef[sp->ord]);
    for (fp = &sp->coef[sp->ord]; fp >= sp->coef; fp--)
      *fp /= f;
  }
  sp->coef[0] = -sp->coef[0];   /* reverse the sign */
  return(sp - sseq);
}

/* Find out how many visible intersections there are */
int visible_roots(int np, polynomial *sseq, int *atzer, int *atpos)
{
  int atposinf, atzero;
  polynomial *s;
  float f, lf;

  atposinf = atzero = 0;
  /* changes at positve infinity */
  lf = sseq[0].coef[sseq[0].ord];
  for (s = sseq + 1; s <= sseq + np; s++) 
  {
    f = s->coef[s->ord];
    if (lf == 0.0 || lf * f < 0)
      atposinf++;
    lf = f;
  }
  
  /* Changes at zero */
  lf = sseq[0].coef[0];
  for (s = sseq+1; s <= sseq + np; s++) 
  {
    f = s->coef[0];
    if (lf == 0.0 || lf * f < 0)
      atzero++;
    lf = f;
  }
  
  *atzer = atzero;
  *atpos = atposinf;
  return(atzero - atposinf);
}

/*
 * numchanges
 *
 *   return the number of sign changes in the Sturm sequence in
 * sseq at the value a.
 */
int numchanges(int np, polynomial *sseq, float a)
{
  int   changes;
  float f, lf;
  polynomial   *s;
  changes = 0;
  //COOPERATE
  lf = polyeval(a, sseq[0].ord, sseq[0].coef);
  for (s = sseq + 1; s <= sseq + np; s++) 
  {
    f = polyeval(a, s->ord, s->coef);
    if (lf == 0.0 || lf * f < 0)
      changes++;
    lf = f;
  }
  return(changes);
}


/*
 * sbisect
 *
 *   uses a bisection based on the sturm sequence for the polynomial
 * described in sseq to isolate intervals in which roots occur,
 * the roots are returned in the roots array in order of magnitude.

Note: This routine has one severe bug: When the interval containing the
      root [min, max] has a root at one of its endpoints, as well as one
      within the interval, the root at the endpoint will be returned rather
      than the one inside.

 */
static int
sbisect(int np, polynomial *sseq, float min_value, float max_value, 
	int atmin, int atmax, float *roots)
{
  float  mid;
  int  n1, n2, its, atmid;

  if ((atmin - atmax) == 1) 
  {
    /* first try using regula-falsa to find the root.  */
    if (regula_falsa(sseq->ord, sseq->coef, min_value, max_value, roots))
      return 1;
    else 
    {
      /* That failed, so now find it by bisection */
      for (its = 0; its < MAX_ITERATIONS; its++) 
      {
        mid = (min_value + max_value) / 2;
        atmid = numchanges(np, sseq, mid);
        if (fabs(mid) > EPSILON) 
	{
          if (fabs((max_value - min_value) / mid) < EPSILON) 
	  {
            roots[0] = mid;
            return 1;
	  }
	}
        else if (fabs(max_value - min_value) < EPSILON) 
	{
          roots[0] = mid;
          return 1;
	}
        else if ((atmin - atmid) == 0)
          min_value = mid;
        else
          max_value = mid;
      }
      /* Bisection took too long - just return what we got */
      roots[0] = mid;
      return 1;
    }
  }
  
  /* There is more than one root in the interval.
     Bisect to find new intervals */
  for (its = 0; its < MAX_ITERATIONS; its++) 
  {
    mid = (min_value + max_value) / 2;
    atmid = numchanges(np, sseq, mid);
    n1 = atmin - atmid;
    n2 = atmid - atmax;
    if (n1 != 0 && n2 != 0) 
    {
      n1 = sbisect(np, sseq, min_value, mid, atmin, atmid, roots);
      n2 = sbisect(np, sseq, mid, max_value, atmid, atmax, &roots[n1]);
      return n1 + n2;
    }
    if (n1 == 0)
      min_value = mid;
    else
      max_value = mid;
  }
  
  /* Took too long to bisect.  Just return what we got. */
  roots[0] = mid;
  return 1;
}

/* Solve a quartic using the method of Francois Vieta (Circa 1735) */
int solve_quartic(float *x, float *results)
{
  float cubic[4], roots[3],
        c12, z, p, q, q1, q2, r, d1, d2,
        c0, c1, c2, c3, c4;
  int i;
  
  /* Figure out the size difference between coefficients */
  if (difficult_coeffs(4, x)) 
  {
    if (fabs(x[0]) < COEFF_LIMIT)
      if (fabs(x[1]) < COEFF_LIMIT)
	return solve_quadratic(&x[2], results);
      else
	return solve_cubic(&x[1], results);
    else
      return polysolve(4, x, results);
  }
  
  /* See if the high order term has vanished */
  c0 = x[0];
  if (fabs(c0) < COEFF_LIMIT)
    return solve_cubic(&x[1], results);
  
  /* See if the constant term has vanished */
  if (fabs(x[4]) < COEFF_LIMIT)
    return solve_cubic(x, results);
  
  /* Make sure the quartic has a leading coefficient of 1.0 */
  if (c0 != 1.0) 
  {
    c1 = x[1] / c0;
    c2 = x[2] / c0;
    c3 = x[3] / c0;
    c4 = x[4] / c0;
  }
  else 
  {
    c1 = x[1];
    c2 = x[2];
    c3 = x[3];
    c4 = x[4];
  }
  
  /* Compute the cubic resolvant */
  c12 = c1 * c1;
  p = -0.375 * c12 + c2;
  q = 0.125 * c12 * c1 - 0.5 * c1 * c2 + c3;
  r = -0.01171875 * c12 * c12 + 0.0625 * c12 * c2 - 0.25 * c1 * c3 + c4;
  
  cubic[0] = 1.0;
  cubic[1] = -0.5 * p;
  cubic[2] = -r;
  cubic[3] = 0.5 * r * p - 0.125 * q * q;
  i = solve_cubic(cubic, roots);
  if (i > 0)
    z = roots[0];
  else
    return 0;
  
  d1 = 2.0 * z - p;
  
  if (d1 < 0.0) 
  {
    if (d1 > -EPSILON)
      d1 = 0.0;
    else
      return 0;
  }
  if (d1 < EPSILON) 
  {
    d2 = z * z - r;
    if (d2 < 0.0)
      return 0;
    d2 = sqrt(d2);
  }
  else 
  {
    d1 = sqrt(d1);
    d2 = 0.5 * q / d1;
  }
  
  /* Set up useful values for the quadratic factors */
  q1 = d1 * d1;
  q2 = -0.25 * c1;
  i = 0;
  
  /* Solve the first quadratic */
  p = q1 - 4.0 * (z - d2);
  if (p == 0)
    results[i++] = -0.5 * d1 - q2;
  else if (p > 0) 
  {
    p = sqrt(p);
    results[i++] = -0.5 * (d1 + p) + q2;
    results[i++] = -0.5 * (d1 - p) + q2;
  }
  /* Solve the second quadratic */
  p = q1 - 4.0 * (z + d2);
  if (p == 0)
    results[i++] = 0.5 * d1 - q2;
  else if (p > 0) 
  {
    p = sqrt(p);
    results[i++] = 0.5 * (d1 + p) + q2;
    results[i++] = 0.5 * (d1 - p) + q2;
  }
  return i;
}

/* Root solver based on the Sturm sequences for a polynomial. */
int polysolve(int order, float *Coeffs, float *roots)
{
  polynomial sseq[MAX_ORDER+1];
  float min_value, max_value;
  int i, nroots, np, atmin, atmax;
  
  /* Put the coefficients into the top of the stack. */
  for (i=0;i<=order;i++)
    sseq[0].coef[order-i] = Coeffs[i];
  
  /* Build the Sturm sequence */
  np = buildsturm(order, &sseq[0]);
  
  /* Get the total number of visible roots */
  if ((nroots = visible_roots(np, sseq, &atmin, &atmax)) == 0)
    return 0;
  
  /* Bracket the roots */
  min_value = 0.0;
  max_value = Max_Distance;
  
  atmin = numchanges(np, sseq, min_value);
  atmax = numchanges(np, sseq, max_value);
  nroots = atmin - atmax;
  if (nroots == 0) return 0;
  
  /* perform the bisection. */
  return sbisect(np, sseq, min_value, max_value, atmin, atmax, roots);
}

