/* smooth.c:  read matlab file, smooth data, write results  */
/* input must be in Matlab binary floating-point format */
/* John P. Beale 1995 <beale@jump.stanford.edu>  */

#include <stdio.h>
#include <stdlib.h>             /* ansi malloc */
#include <math.h>
#include <string.h>
#include "hcon.h"

#define Eh(vv, xq, yq)  vv[((yq) * (xsize+2)) + xq]  /* elmt of hf array */
#define Ehn(vv, xq, yq) vv[((yq+1)*(xsize+2))+(xq+1)] /* hf indexed like ua */
#define El(vv, xq, yq)  vv[((yq) * xsize) + xq]  /* elmt of array */

int write_mfile(char *f, char *p);     /* write out matrix as matlab file */
int read_mfile(char *f);               /* read in matlab file to matrix */
int read_mfile2(char *f);              /* read in matlab file to flag mx. */
int flag_read(char *f);                /* read in flag file */
void smooth();                         /* write smoothed pr into pr2 */

static Fmatrix x;                       /* matlab-format file header */
static PTYPE *pr, *pr2;                 /* pointer to real data */
static BYTE *flag;                      /* flag array */
static int xsize, ysize;                /* size of input data array */
static int x4[4] =                      /* 4-dir index offset arr */
   { -1,0,1,0 };
static int y4[4] =
   { 0,-1,0,1 };


int main()
{
char fname[80] = "input.dat";
int i,j,ii,mn,passes;

 printf("smooth.c: read in .mat binary file, smooth it, write it out.\n");
 printf("  If flag.pgm file exists, it is used to indicate areas to smooth\n");
 printf("  (areas where flag.pgm pixels are less than 128, are smoothed).\n");
 printf("  If flag.pgm does not exist, all regions are smoothed.\n");
 printf("  You may specify any number of sucessive smoothing passes.\n\n");
 printf("\nEnter input file name:");
  scanf("%s",fname);
  printf("\nEnter number of passes:");
  scanf("%d",&passes);

  read_mfile(fname);                  /* read in the data */
  flag_read("flag.pgm");

  for (j=1;j<ysize+1;j++) {     /* fill in L+R edges */
   Eh(pr,0,j) = Eh(pr,1,j);
   Eh(pr,xsize+1,j) = Eh(pr,xsize,j);
  }
  for (i=0;i<xsize+2;i++) {     /* fill in T+B edges */
   Eh(pr,i,0) = Eh(pr,i,1);
   Eh(pr,i,ysize+1) = Eh(pr,i,ysize);
  }

  printf("Data array read:\n");
  if (xsize*ysize < 100) {
  for (j=0;j<ysize+2;j++) {
   for (i=0;i<xsize+2;i++) {
    printf("%3.1f ",Eh(pr,i,j));
   }
   printf("\n");
  } /* end for j */
  } /* end if */

  mn = (xsize+2) * (ysize+2);   /* allocate new matrix */
  if ((pr2 = (PTYPE *) malloc(mn*sizeof(PTYPE)) ) == NULL) {
     perror("Could not allocate memory for input array.\n");
     exit(1);
   }

  printf("Pass #:");
  for (ii=0;ii<passes;ii++) {
    printf("%d.",ii+1);
    fflush(stdout);
    smooth();
    for (j=0;j<ysize;j++) {     /* copy middle of pr2 into pr */
     for (i=0;i<xsize;i++) {
      Ehn(pr,i,j) = Ehn(pr2,i,j);
     } /* end for i */
    } /* end for j */
  } /* end for ii */
  printf("\n");


  printf("Data array avgd.:\n");
  if (xsize*ysize < 100) {
   for (j=0;j<ysize+2;j++) {
    for (i=0;i<xsize+2;i++) {
     printf("%3.1f ",Eh(pr,i,j));
    }
    printf("\n");
   } /* end for j */
  }

  printf("\nEnter output file name:");
  scanf("%s",fname);

  write_mfile(fname,"data1");                 /* write data out */

 return(0);
} /* end */

/* -------------------------------------------------------------- */
/* write_mfile -- write out data matrix in Matlab Level 1.0 format */
/*               given a header structure "x" and data array "pr" */
/* -------------------------------------------------------------- */

int write_mfile(char *fname, char *pname)
{
FILE *fpout;
int rv,mn,i,j,c;

   fpout = fopen(fname,"wb");
   if (fpout == NULL)
    { perror("Error opening file.\n");
      exit(1);
    }

   x.type = NUM_FMT*1000 + PREC*10;
   rv = fwrite(&x, sizeof(Fmatrix),(size_t)1, fpout);          /* write header */
   if (rv != 1) {
     perror("Error writing file.\n");
     exit(1);
   }
   rv = fwrite(pname, sizeof(char), (size_t)x.namelen, fpout);   /* write mx name */
   if (rv != x.namelen) {
     perror("Error writing file.\n");
     exit(1);
   }
   mn = x.mrows * x.ncols;

   for (i=0;i<xsize;i++) {
      for (j=0;j<ysize;j++) {   /* format is columns saved first */

       c = fwrite(&Ehn(pr,i,j), sizeof(PTYPE), 1, fpout);
       if (c != 1) {
	 perror("Trouble writing data to file.\n");
	 exit(1);
       }
      } /* end for j */
   } /* end for i */

  if (fclose( fpout ) == EOF) {
	perror("Error closing file.\n");
	exit(1);
  }

  printf("All done with file operations.\n");
  return(0);
}

/* -------------------------------------------------------------- */
/* read_mfile --  read in data matrix in Matlab Level 1.0 format   */
/*               and put data in header "x" and data array "pr"   */
/* -------------------------------------------------------------- */

int read_mfile(char *fname)
{
  char pname[80]="dat1\x00";           /* data matrix name, null-terminated*/
  int mn;

  int a,b,c,i,j;
  int etype;
  int mtype,prec,rem,numtype;
  double tdbl;
  float tflt;
  FILE *fpin;

   /* --------- read in Matlab-format (Level 1.0) file ------- */

   etype = (NUM_FMT*1000) + (PREC*10);    /* expected data type flag */

   if ((fpin = fopen(fname,"rb")) == NULL) {
    perror("could not open file.\n");
    exit(1);
   }

   printf("Opening matlab file %s...\n",fname);

   a = fread(&x, sizeof(Fmatrix),(size_t)1, fpin);          /* read header */
    if (a != 1) {
      perror("Problem reading matlab file header.\n");
      exit(1);
    }
   numtype = (x.type % 10000)/1000;
   prec = (x.type % 100)/10;
   mtype = (x.type % 10);
   rem = (x.type - (1000*numtype + 10*prec + mtype));
   printf("Data type flag: M%d P%d T%d R%d\n",
       numtype,prec,mtype,rem);
   if ( (numtype!=NUM_FMT) || ((prec!=0)&&(prec!=1)) || (mtype!=0) )  {
     perror("Unsupported file format type\n");
     exit(1);
   }
   b = fread(pname, sizeof(char), (size_t)x.namelen, fpin); /* read mx name */
   xsize = x.mrows;
   ysize = x.ncols;
   printf("Array name: %s [%dx%d]\n",pname,xsize,ysize);
   mn = (x.mrows+2) * (x.ncols+2); /* size of array +1 each edge */
   if ((pr = (PTYPE *) malloc(mn*sizeof(PTYPE)) ) == NULL) {
     perror("Could not allocate memory for input array.\n");
     exit(1);
   }
   if (prec==1) {               /* read in floats 1 by 1 */
     for (i=0;i<xsize;i++) {
      for (j=0;j<ysize;j++) {   /* format is columns saved first */
       c = fread(&tflt, sizeof(float), 1, fpin);
       if (c != 1) {
	 perror("Trouble reading data from file.\n");
	 exit(1);
       }
       Ehn(pr,i,j) = (PTYPE) tflt;
      } /* end for j */
     } /* end for i */
   } else if (prec==0) {                     /* read doubles 1 by 1 and convert */

     for (i=0;i<xsize;i++) {
      for (j=0;j<ysize;j++) {   /* format is columns saved first */
       c = fread(&tdbl, sizeof(double), 1, fpin);
       if (c != 1) {
	 perror("Trouble reading data from file.\n");
	 exit(1);
       }
       Ehn(pr,i,j) = (PTYPE) tdbl;
      } /* end for j */
     } /* end for i */

   } else {
	perror("Unsupported floating-point type\n");
	exit(1);
   } /* end else */

   fclose( fpin );

  return(0);

} /* end read_mfile */

/* ------------------------------------------------------------------- */
/* smooth() -- Fill matrix pr2 with 4-neighbor average from matrix pr. */
/*             Copy edge elements of pr to pr2.                        */
/* ------------------------------------------------------------------- */

void smooth()
{
 int ix,iy,i,j;
 PTYPE sum;

  for (iy = 0; iy<ysize; iy++) {
    for (ix = 0; ix<xsize; ix++) {
      if (El(flag,ix,iy) < 128) { /* if flag array value < 128, average */
	sum = 0;
	for(i=0;i<4;i++) {
	    sum += Ehn(pr,ix+x4[i],iy+y4[i]);
	}  /* end for i=1..4 */
	Ehn(pr2,ix,iy) = sum / 4;   /* replace element with avg. of neighbors */
      } else {
	Ehn(pr2,ix,iy) = Ehn(pr,ix,iy);  /* if flag >= 128, don't smooth */
      }
    }  /* end for x=1..xsize-1 */
  } /* end for y=1..ysize-1 */

  for (j=1;j<ysize+1;j++) {     /* fill in L+R edges */
   Eh(pr2,0,j) = Eh(pr,1,j);
   Eh(pr2,xsize+1,j) = Eh(pr,xsize,j);
  }
  for (i=0;i<xsize+2;i++) {     /* fill in T+B edges */
   Eh(pr2,i,0) = Eh(pr,i,1);
   Eh(pr2,i,ysize+1) = Eh(pr,i,ysize);
  }
}


/* ------------------------------------------------------------------*/
/* flag_read -- read in pgm data from file <fname> into flag array,  */
/*              allocating memory to fit a (xsize x ysize) array.    */
/* ------------------------------------------------------------------*/

int flag_read(char *fname)
{
  unsigned long ix, iy;                  /* indexes to hf array */
  size_t hsize;                         /* # bytes memory needed */
  unsigned long npoints;            /* # points that should be in file */
  unsigned long points_read;        /* # points read from hf datafile */
  char sbuf[512],                       /* buffer for file input */
	dummy[2];                       /* used for \n and \0   */
  char *junk;                           /* unconverted part of a number */
  double a, maxval;
  FILE *fpin;                           /* input file pointer */
  BYTE fbin;                            /* PGM binary format flag */
  BYTE oneb;                            /* one byte from file */
  int rval;

  hsize = (size_t) ( xsize * ysize * sizeof(BYTE));
  sprintf(sbuf,"Could not allocate memory for heightfield (%lu bytes)",
    (long)hsize);
  if ((flag = (BYTE *) malloc( hsize ) ) == NULL)   {
   fprintf(stderr,"%s\n",sbuf);
   exit(1);
  }
  memset(flag,(char) 0, hsize);

						/* open input file */
  if ((fpin = fopen(fname,"rb")) == NULL)
   { fprintf(stderr,"Could not open flag file %s\n",fname);
     return(1);
   }

  printf("Opening flag file %s\n",fname);

  if (fscanf(fpin,"%s",sbuf) == EOF)
    {   perror("Could not read file"); exit(1); }
  else {
	printf("Initial file string: %s\n",sbuf);   /* better be P2 */
  }
  fbin = 0;                             /* assume PGM ascii */
  if (strcmp(sbuf,"P5") == 0) fbin = 1; /* PGM binary format */
  if (!fbin && (strcmp(sbuf,"P2") != 0)) { /* not PGM Ascii */
	  fclose(fpin);   /* close file to reopen as binary */
	  rval = read_mfile2(fname);
  } else {

  while ((fscanf(fpin,"%s",sbuf) != EOF) && sbuf[0] == '#') {
	printf("comment:");
	fscanf(fpin,"%[^\n]",sbuf);  /* read comment beyond '#' */
	fscanf(fpin,"%[\n]",dummy);  /* read newline */
	printf("%s\n",sbuf);            /* display comment */
	}

  xsize = (int) strtoul(sbuf, &junk, 10);
  fscanf(fpin,"%s",sbuf);               /* read ysize of array */
  ysize = (int) strtoul(sbuf, &junk, 10);
  fscanf(fpin,"%s",sbuf);               /* read maxval of array */
  maxval = strtod(sbuf, &junk);

  points_read = 0;              /* haven't read any data points yet */
  npoints = (unsigned long int) (xsize * ysize);
  ix = 0;
  iy = 0;                       /* fill array 1..xsize, 1..size */

  printf("reading in...\n");
  if (fbin) {
     while ((fread(&oneb,sizeof(BYTE),1,fpin) == 1) && (points_read < npoints) ) {
	flag[points_read++] = (BYTE) oneb;
     }
  } /* end if */
  else {
      while ((fscanf(fpin,"%s",sbuf) != EOF) && (points_read < npoints) ) {
	a = strtod( sbuf, &junk);
	flag[points_read++] = (BYTE) a;
      }
  } /* end else */


  printf("\n");
  printf("%lu elements read.\n",points_read);
  if (points_read != npoints) {
	printf("Warning: %lu points missing.\n",npoints-points_read);
  }
  rval = fclose( fpin );

  printf("Files closed.\n");
  } /* end else */
  return(rval);
} /* end flag_read */


/* -------------------------------------------------------------- */
/* read_mfile2 --  read in data matrix in Matlab Level 1.0 format   */
/*               and put data in header "x" and data array "flag"   */
/* -------------------------------------------------------------- */

int read_mfile2(char *fname)
{
  char pname[80]="dat1\x00";           /* data matrix name, null-terminated*/
  int mn;

  int a,b,c,i,j;
  int etype;
  int mtype,prec,rem,numtype;
  double tdbl;
  float tflt;
  FILE *fpin;

   /* --------- read in Matlab-format (Level 1.0) file ------- */

   etype = (NUM_FMT*1000) + (PREC*10);    /* expected data type flag */

   if ((fpin = fopen(fname,"rb")) == NULL) {
    perror("could not open file.\n");
    exit(1);
   }

   printf("Opening matlab file %s...\n",fname);

   a = fread(&x, sizeof(Fmatrix),(size_t)1, fpin);          /* read header */
    if (a != 1) {
      perror("Problem reading matlab file header.\n");
      exit(1);
    }
   numtype = (x.type % 10000)/1000;
   prec = (x.type % 100)/10;
   mtype = (x.type % 10);
   rem = (x.type - (1000*numtype + 10*prec + mtype));
   printf("Data type flag: M%d P%d T%d R%d\n",
       numtype,prec,mtype,rem);
   if ( (numtype!=0) || ((prec!=0)&&(prec!=1)) || (mtype!=0) )  {
     perror("Unsupported file format type\n");
     exit(1);
   }
   b = fread(pname, sizeof(char), (size_t)x.namelen, fpin); /* read mx name */
   if ((xsize != x.mrows) || (ysize != x.ncols)) {
     perror("Flag matrix not of correct dimensions\n");
     exit(1);
     }
   printf("Array name: %s [%dx%d]\n",pname,xsize,ysize);
   mn = (x.mrows) * (x.ncols); /* size of array +1 each edge */
   if ((flag = (BYTE *) malloc(mn*sizeof(BYTE)) ) == NULL) {
     perror("Could not allocate memory for input array.\n");
     exit(1);
   }
   if (prec==1) {               /* read in floats 1 by 1 */
     for (i=0;i<xsize;i++) {
      for (j=0;j<ysize;j++) {   /* format is columns saved first */
       c = fread(&tflt, sizeof(float), 1, fpin);
       if (c != 1) {
	 perror("Trouble reading data from file.\n");
	 exit(1);
       }
       El(flag,i,j) = (BYTE) tflt;
      } /* end for j */
     } /* end for i */
   } else if (prec==0) {                     /* read doubles 1 by 1 and convert */

     for (i=0;i<xsize;i++) {
      for (j=0;j<ysize;j++) {   /* format is columns saved first */
       c = fread(&tdbl, sizeof(double), 1, fpin);
       if (c != 1) {
	 perror("Trouble reading data from file.\n");
	 exit(1);
       }
       El(pr,i,j) = (BYTE) tdbl;
      } /* end for j */
     } /* end for i */

   } else {
	perror("Unsupported floating-point type\n");
	exit(1);
   } /* end else */

   fclose( fpin );

  return(0);

} /* end read_mfile2 */
