/* orb.c:  convert a planar heightfield into a spherical heightfield     */
/*         reads in PGM, Matlab, GIF, POT, or TGA                        */
/*         writes out a raw triangle file or a POVray include file with  */
/*         either flat or smooth triangles                               */
/*         by John P. Beale (about 6/?/95)                               */
/*         modified by Carl D. Perkins 7-Oct-95 to do smoothed triangles */
/*         (while I was at it, I fixed this header since the one that    */
/*         here was for an entirely different program)                   */
/*          GIF library routines from gd library                         */

#include <stdio.h>                      /* fopen() sprintf() etc. */
#include <stdlib.h>                     /* strtod() */
#include <math.h>                       /* math stuff */
#include <string.h>                     /* strcpy() */
#include "hcon.h"
#include "hf_io.h"

int gen_sphere(double f_scale, int maxlevel, char *fname_out);

U xsize, ysize;     /* size of hf array (input xsize+2) */
PTYPE *hf;                            /* height field array */
PTYPE fmin = 0, fmax = 1.0;                 /* scaling limits */
Boolean norenorm = FALSE;
Boolean negate = FALSE;
Boolean rescale = FALSE;
Boolean raw_out = FALSE;
Boolean smooth_out = FALSE;

/* ---------------------------------- */

int main(int argc, char **argv)
{ char *usage = 
"orb: An HF -> Spherical HF converter v0.2a (c) John P. Beale 1995\n\
Usage: orb <infile> <outfile> <scale> <levels> [ -smooth ]\n\
  Available input formats: PGM|MAT|GIF|POT|TGA, output RAW or POVray INC\n\
  Determines filetype based on extension, eg:  orb file1.tga file1.inc 1 4\n\
  POT,TGA are 16-bit formats, MAT is Matlab floating point format.\n\n\
  RAW is a dump of the triangles, INC is POV union of triangles or \n\
  union of smooth_triangles \n\
  <scale> is radial extension of hf from unit sphere; eg 2.5 for huge bumps\n\
  <levels> is number of recursive tesselations of the sphere; watch out...\n\
    9th level will generate a 40 megabyte include file!\n\
  -smooth specifies smoothed triangle output\n";
  
  char fname_in[160];
  char fname_out[160];
  char *ext2;
  char *buf;
  int rval;
  int outlen;
  int mlev = 4;
  double feature_scale = 0.1;

  if ((argc < 3) || (argc > 6)) {
	fprintf(stderr,"%s",usage);
	exit(1);
      }
 
  strcpy(fname_in,argv[1]);
  strcpy(fname_out,argv[2]);


  if (argc > 3) {
    feature_scale = (PTYPE) strtod(argv[3], &buf);
    }
  if (argc > 4) {
    mlev = (int) strtod(argv[4], &buf);
    }
  if (argc > 5) {
    smooth_out = TRUE;
  }    

  printf("Height scale = %5.3f  Recursion level = %d\n",feature_scale,mlev);
  outlen = strlen(fname_out);
  
  ext2 = (char *) malloc((size_t)sizeof(char)*5);
  strncpy(ext2,(char *)(&fname_out[outlen-4]),5);

  if (strcmp(ext2,".raw")==0) raw_out = TRUE;

					/* read in heightfield */
   rval = readn_hf(&hf, &xsize, &ysize, fname_in);
   if (rval != 0)
     exit(rval);

   printf("%s: [%d x %d] \n",fname_in,xsize,ysize);
   if (raw_out)   printf("Generating raw triangle file...");
   else {
      if (smooth_out) {
        printf("Generating POV .inc smooth_triangle file...");
      }
      else printf("Generating POV .inc triangle file...");
   }
   fflush(stdout);

   gen_sphere(feature_scale, mlev ,fname_out);
   printf("\n"); 
  
   return(rval);    /* return a value */
} /* end main() */

void hf_print()                /* print out hf array on stdout */
{
  int ix,iy;

   printf("Heightfield array: %dx%d\n",xsize,ysize);
   for (iy = 0; iy<ysize; iy++) {
    printf("%d:",iy);
    for (ix = 0; ix<xsize; ix++) {
	printf(" %1.2f",El(hf,ix,iy));
    }
     printf("\n");
   }
}

