/* mat.c:  i/o for Matlab-format files                  */
/*          GIF library routines from gd library             */

#include <stdio.h>                      /* fopen() sprintf() etc. */
#include <stdlib.h>                     /* strtod() */
#include <math.h>                       /* math stuff */
#include <string.h>                     /* strcpy() */
#include "hcon.h"

int read_mfile(char *f);                 /* read in matlab file to matrix */
int write_mfile(char *f, char *m);      /* write out matlab file */
void hf_print();                        /* print out hf array to stdout */

extern unsigned int xsize, ysize;     /* size of hf array (input xsize+2) */
extern float *hf;                            /* height field array */

/* ---------------------------------- */


/* -------------------------------------------------------------- */
/* read_mfile --  read in data matrix in Matlab Level 1.0 format  */
/*               and put data in header "x" and data array "hf"   */
/* -------------------------------------------------------------- */

int read_mfile(char *fname)
{
  char pname[80]="dat1\x00";           /* data matrix name, null-terminated*/
  int mn;
  Fmatrix hd;                           /* header */

  int a,b,c,i,j;
  int etype;
  int mtype,prec,rem,numtype;
  double tdbl;
  float tflt;
  FILE *fpin;

   /* --------- read in Matlab-format (Level 1.0) file ------- */

   etype = (NUM_FMT*1000) + (PREC*10);    /* expected data type flag */

   if ((fpin = fopen(fname,"rb")) == NULL) {
    perror("could not open file.\n");
    exit(1);
   }

   printf("Opening matlab file %s...\n",fname);

   a = fread(&hd, sizeof(Fmatrix),(size_t)1, fpin);          /* read header */
    if (a != 1) {
      perror("Problem reading matlab file header.\n");
      exit(1);
    }
   numtype = (hd.type % 10000)/1000;
   prec = (hd.type % 100)/10;
   mtype = (hd.type % 10);
   rem = (hd.type - (1000*numtype + 10*prec + mtype));
/*   printf("Data type flag: M%d P%d T%d R%d\n",
       numtype,prec,mtype,rem); */
   if ( (numtype!=0) || ((prec!=0)&&(prec!=1)) || (mtype!=0) )  {
     perror("Unsupported file format type\n");
     exit(1);
   }
   b = fread(pname, sizeof(char), (size_t)hd.namelen, fpin); /* read mx name */
   xsize = hd.ncols;
   ysize = hd.mrows;
   mn = (hd.mrows) * (hd.ncols); /* size of array */
   if ((hf = (PTYPE *) malloc(mn*sizeof(PTYPE)) ) == NULL) {
     perror("Could not allocate memory for input array.\n");
     exit(1);
   }
   if (prec==1) {               /* read in floats 1 by 1 */
     for (i=0;i<xsize;i++) {
      for (j=0;j<ysize;j++) {   /* format is columns saved first */
       c = fread(&tflt, sizeof(float), 1, fpin);
       if (c != 1) {
	 perror("Trouble reading data from file.\n");
	 exit(1);
       }
       El(hf,i,j) = (PTYPE) tflt;
      } /* end for j */
     } /* end for i */
   } else if (prec==0) {                     /* read doubles 1 by 1 and convert */

     for (i=0;i<xsize;i++) {
      for (j=0;j<ysize;j++) {   /* format is columns saved first */
       c = fread(&tdbl, sizeof(double), 1, fpin);
       if (c != 1) {
	 perror("Trouble reading data from file.\n");
	 exit(1);
       }
       El(hf,i,j) = (PTYPE) tdbl;
      } /* end for j */
     } /* end for i */

   } else {
	perror("Unsupported floating-point type\n");
	exit(1);
   } /* end else */

   fclose( fpin );

  return(0);

} /* end read_mfile */


/* --------------------------------------------------------------  */
/* write_mfile -- write out data matrix in Matlab Level 1.0 format */
/*               creates data header fhdr from xsize, ysize and    */
/*               writes data array 'hf'                            */
/* --------------------------------------------------------------  */

int write_mfile(char *fname, char *pname)
{
FILE *fpout;
Fmatrix fhdr;
int rv,mn,i,j,c;

   fpout = fopen(fname,"wb");
   if (fpout == NULL)
    { perror("Error opening file.\n");
      exit(1);
    }
   fhdr.type = NUM_FMT*1000 + PREC*10;
   fhdr.ncols = xsize;
   fhdr.mrows = ysize;
   fhdr.imagf = 0;
   fhdr.namelen = strlen(pname) + 1;

   rv = fwrite(&fhdr, sizeof(Fmatrix),(size_t)1, fpout);          /* write header */
   if (rv != 1) {
     perror("Error writing file.\n");
     exit(1);
   }
   rv = fwrite(pname, sizeof(char), (size_t)fhdr.namelen, fpout);   /* write mx name */
   if (rv != fhdr.namelen) {
     perror("Error writing file.\n");
     exit(1);
   }
   mn = fhdr.mrows * fhdr.ncols;

   for (i=0;i<xsize;i++) {
      for (j=0;j<ysize;j++) {   /* format is columns saved first */

       c = fwrite(&El(hf,i,j), sizeof(PTYPE), 1, fpout);
       if (c != 1) {
	 perror("Trouble writing data to file.\n");
	 exit(1);
       }
      } /* end for j */
   } /* end for i */

  if (fclose( fpout ) == EOF) {
	perror("Error closing file.\n");
	exit(1);
  }

  return(0);
} /* end write_mfile */
