/* joinhf.c:  convert heightfield data to various formats      
 *          read in two arrays of height values,
 *          join them together
 *          write out in specified format
 *           John P. Beale 7/18/95                           
 *          GIF library routines from gd library             */

#include <stdio.h>                      /* fopen() sprintf() etc. */
#include <stdlib.h>                     /* strtod() */
#include <math.h>                       /* math stuff */
#include <string.h>                     /* strcpy() */
#include <ctype.h>
#include "hcon.h"
#include "hf_io.h"

#undef max
#define max(a,b) ((a) > (b) ? (a) : (b))
#undef min
#define min(a,b) ((a) < (b) ? (a) : (b))
#undef abs
#define abs(a) ((a) >= 0 ? (a) : -(a))
#undef odd
#define odd(n) ((n) & 1)

void hf_print();                        /* print out hf array to stdout */
int joinx(PTYPE *hf, PTYPE *hf2, PTYPE **hf3);
int joiny(PTYPE *hf, PTYPE *hf2, PTYPE **hf3);

unsigned int xsize, ysize;     /* size of hf array  */
unsigned int xsize2, ysize2;   /* size of second hf array */
unsigned int xsize3, ysize3;   /* size of output hf array */
PTYPE *hf;                     /* height field array */
PTYPE *hf2;                    /* rescale array */
PTYPE *hf3;                    /* output array */
PTYPE fmin = 0, fmax = 1.0;                 /* scaling limits */
Boolean norenorm = FALSE;
Boolean negate = FALSE;

/* ---------------------------------- */

int main(int argc, char **argv)
{ char *usage = 
"joinhf: A Heightfield Combiner v0.1a (c) <beale@jump.stanford.edu> 1995\n\
Usage: joinhf [ x | y ] <infile1> <infile2> <outfile> \n\
  joins two input files and writes as outfile. Reads/writes many types.\n\
  Determines filetype based on extension, eg:   join x a.pot b.tga c.gif\n\
  x joins horizontally, y joins vertically.\n";
  
  char fname_in[160];
  char fname_in2[160];
  char fname_out[160];
  char c;
  Boolean horizontal = TRUE;
  int rval;

  if ((argc < 5) || (argc > 5)) {
	fprintf(stderr,"%s",usage);
	exit(1);
      }

  c = tolower(argv[1][0]);
  if (c == 'y') horizontal = FALSE;

  strcpy(fname_in,argv[2]);
  strcpy(fname_in2,argv[3]);
  strcpy(fname_out,argv[4]);

					/* read in heightfields */
   rval = readn_hf(&hf, &xsize, &ysize, fname_in);
   if (rval != 0) exit(rval);
   rval = readn_hf(&hf2, &xsize2, &ysize2, fname_in2);
   if (rval != 0) exit(rval);

   printf("Joining %s: [%d x %d] and %s: [%d x %d] %s.\n",
    fname_in,xsize,ysize,fname_in2,xsize2,ysize2,
    (horizontal) ? "horizontally" : "vertically");

   if (horizontal)
     joinx(hf, hf2, &hf3);   /* join heightfields */
   else
     joiny(hf, hf2, &hf3);

   printf("%s: [%d x %d] \n",fname_out,xsize3,ysize3);
   
   if (xsize < 9) hf_print();  /* print small arrays */

   rval = writen_hf(hf3, xsize3, ysize3, fname_out);
   
  return(rval);    /* return a value */

} /* end main() */

void hf_print()                /* print out hf array on stdout */
{
  int ix,iy;

   printf("Heightfield array: %dx%d\n",xsize,ysize);
   for (iy = 0; iy<ysize; iy++) {
    printf("%d:",iy);
    for (ix = 0; ix<xsize; ix++) {
	printf(" %1.2f",El(hf,ix,iy));
    }
     printf("\n");
   }
}

/* joinx() -- Take two input arrays hf and hf2 and join hf2 on the
 *            right-hand edge of hf, yeilding result hf3.
 */

int joinx(PTYPE *hf, PTYPE *hf2, PTYPE **hf3)
{
int ix, iy;
size_t memsize;

  xsize3 = xsize + xsize2;
  ysize3 = max(ysize,ysize2);
  memsize = xsize3 * ysize3 * sizeof(PTYPE);
  hf3[0] = (PTYPE *) malloc(memsize);
  if (hf3[0] == (PTYPE *)NULL) {
   fprintf(stderr,"Could not allocate memory for result array.\n");
   exit(1);
  }

  for (iy=0; iy<ysize3; iy++) {
                                /* copy first hf into hf3 */
    if (iy >= ysize) {         /* ran off bottom of hf */
      for (ix=0; ix<xsize; ix++) {
        El3(hf3[0],ix,iy) = El(hf,0,ysize-1); /* repeat corner pixel */
      }
    } else {
      for (ix=0; ix<xsize; ix++) {
        El3(hf3[0],ix,iy) = El(hf,ix,iy);
      }
    }
				/* copy second hf into hf3 */
    if (iy >= ysize2) {     /* ran off bottom of hf2 */
      for (ix=xsize; ix<xsize+xsize2; ix++) {
        El3(hf3[0],ix,iy) = El2(hf2,0,ysize2-1);
      }
    } else {
      for (ix=xsize; ix<xsize+xsize2; ix++) {
        El3(hf3[0],ix,iy) = El2(hf2,ix-xsize,iy);
      }
    }
   }
  return(0);
} /* end joinx() */

/* joiny() -- Take two input arrays hf and hf2 and join hf2 on the
 *            bottom edge of hf, yeilding result hf3.
 */

int joiny(PTYPE *hf, PTYPE *hf2, PTYPE **hf3)
{
int ix, iy;
size_t memsize;

  xsize3 = max(xsize,xsize2);
  ysize3 = ysize + ysize2;
  memsize = xsize3 * ysize3 * sizeof(PTYPE);
  hf3[0] = (PTYPE *) malloc(memsize);
  if (hf3[0] == (PTYPE *)NULL) {
   fprintf(stderr,"Could not allocate memory for result array.\n");
   exit(1);
  }

  for (ix=0; ix<xsize3; ix++) {
                                /* copy first hf into hf3 */
    if (ix >= xsize) {         /* ran off RH edge of hf ? */
      for (iy=0; iy<ysize; iy++) {
        El3(hf3[0],ix,iy) = El(hf,xsize-1,0); /* repeat corner pixel */
      }
    } else {
      for (iy=0; iy<ysize; iy++) {
        El3(hf3[0],ix,iy) = El(hf,ix,iy);
      }
    }
				/* copy second hf into hf3 */
    if (ix >= xsize2) {     /* ran off RH edge of hf2 ? */
      for (iy=ysize; iy<ysize+ysize2; iy++) {
        El3(hf3[0],ix,iy) = El2(hf2,xsize2-1,0);
      }
    } else {
      for (iy=ysize; iy<ysize+ysize2; iy++) {
        El3(hf3[0],ix,iy) = El2(hf2,ix,iy-ysize);
      }
    }
   }
  return(0);
}  /* end joiny() */
