/* hcon.c:  convert heightfield data to various formats      */
/*          read in array of values in PGM or Matlab format. */
/*          write out in PGM format or GIF format            */
/*           John P. Beale 6/10/95                           */
/*          GIF library routines from gd library             */

#include <stdio.h>                      /* fopen() sprintf() etc. */
#include <stdlib.h>                     /* strtod() */
#include <math.h>                       /* math stuff */
#include <string.h>                     /* strcpy() */
#include "hcon.h"
#include "hf_io.h"

void renorm(int r);                     /* normalize array */

U xsize, ysize;     /* size of hf array (input xsize+2) */
PTYPE *hf;                            /* height field array */
PTYPE fmin = 0, fmax = 1.0;                 /* scaling limits */
Boolean norenorm = FALSE;
Boolean negate = FALSE;
Boolean rescale = FALSE;
Boolean clip = FALSE;

/* ---------------------------------- */

int main(int argc, char **argv)
{ char *usage;
  char *utemp = 
"hcon: A Heightfield Converter v0.1a (c) <beale@jump.stanford.edu> 1995\n\
Usage: hcon <infile> <outfile> [fmin [fmax]]\n\
       hcon <infile> <outfile> -negate\n\
       hcon <infile> <outfile> -rescale\n\
  Input: %s Output: %s\n\
  Determines filetype based on extension, eg:   hcon file1.tga file1.pot\n\
  Rescale option scales data to fill entire range (8|16|32 bits).\n\
  Increase contrast by setting thresholds fmin [0..1], fmax [0..1]. \n\
  If no additional arguments present, no rescaling is done.\n\
  POT,TGA are 16-bit formats, MAT is Matlab floating point format.\n";
  
  char fname_in[160];
  char fname_out[160];
  char *buf;
  int rval;

  usage = (char *)malloc((size_t)(1000 * sizeof(char)));
  sprintf(usage,utemp,Intype_string, Outtype_string);

  if ((argc < 3) || (argc > 6)) {
	fprintf(stderr,"%s",usage);
	exit(1);
      }
 
  strcpy(fname_in,argv[1]);
  strcpy(fname_out,argv[2]);

  if (argc > 3) {
   rescale = TRUE;
   if (strcmp(argv[3],"-negate")==0) {
     negate = TRUE;
     printf("Inverting polarity of image.\n");
   } else if (strcmp(argv[3],"-rescale")==0) {
     printf("Rescaling to fill output range.\n");
   } else {
    clip = TRUE;
    fmin = (PTYPE) strtod(argv[3], &buf);
    if (argc > 4) fmax = (PTYPE) strtod(argv[4], &buf);
   } /* end else */
 } /* end if argc > 3 */

					/* read in heightfield */
   rval = readn_hf(&hf, &xsize, &ysize, fname_in);
   if (rval != 0)
     exit(rval);

   printf("%s: [%d x %d] \n",fname_in,xsize,ysize);
   
   renorm(rescale);   /* rescale array to [0..1] */
   
   if (xsize < 9) hf_print();  /* print small arrays */

   rval = writen_hf(hf, xsize, ysize, fname_out);

   return(rval);    /* return a value */
} /* end main() */

void hf_print()                /* print out hf array on stdout */
{
  int ix,iy;

   printf("Heightfield array: %dx%d\n",xsize,ysize);
   for (iy = 0; iy<ysize; iy++) {
    printf("%d:",iy);
    for (ix = 0; ix<xsize; ix++) {
	printf(" %1.2f",El(hf,ix,iy));
    }
     printf("\n");
   }
}



/* -------------------------------------------------------------------- */
/* renorm() --  scale data in hf[xsize,ysize] to fit in the full        */
/* range 0..1  */
/* -------------------------------------------------------------------- */

void renorm(int rescale)
{
 PTYPE dmax, dmin, tmp, scale;
 int ix,iy;

 dmax = El(hf,0,0);
 dmin = El(hf,0,0);

 /* Optionally negate. Find minimum and maximum values in array. */

   for (iy = 0; iy<ysize; iy++) {
    for (ix = 0; ix<xsize; ix++) {
	if (negate) El(hf,ix,iy) = (1.0 - El(hf,ix,iy));    /* invert */
	tmp = El(hf,ix,iy);
       /* printf("%1.1f ",tmp); */
	if (tmp > dmax) dmax = tmp;
	if (tmp < dmin) dmin = tmp;
    }
   }

   printf("Input file had range %1.5e .. %1.5e\n",dmin,dmax);
   
   if (clip) {
       printf("Selecting range (%1.3e - %1.3e)\n",fmin,fmax);
       dmin = fmin;
       dmax = fmax;
     }
   scale = 1.0 / (dmax - dmin);

/*   if (norenorm) {
     scale = 1.0;
     dmin = 0.0;
   } */
   
   if (rescale) {
    for (iy = 0; iy<ysize; iy++) {
     for (ix = 0; ix<xsize; ix++) {
	 tmp = (El(hf,ix,iy) - dmin) * scale;
	 if (tmp < 0.0) tmp = 0.0;       /* clipping */
	 if (tmp > 1.0) tmp = 1.0;       
	 El(hf,ix,iy) = tmp;
     }
    }
   } /* end if */

}       /* end renorm() */

