/*% cc -g sphere.c -o sphere -lm
 *
 * sphere - generate a triangle mesh approximating a sphere by
 *  recursive subdivision. First approximation is an octahedron;
 *  each level of refinement increases the number of triangles by
 *  a factor of 4.
 * Level 3 (128 triangles) is a good tradeoff if gouraud
 *  shading is used to render the database.
 *
 * Usage: sphere [level] [-p] [-c]
 *      level is an integer >= 1 setting the recursion level (default 1).
 *      -p causes generation of a PPHIGS format ASCII archive
 *          instead of the default generic output format.
 *      -c causes triangles to be generated with vertices in counterclockwise
 *          order as viewed from the outside in a RHS coordinate system.
 *          The default is clockwise order.
 *
 *  The subroutines print_object() and print_triangle() should
 *  be changed to generate whatever the desired database format is.
 *
 * Jon Leech (leech@cs.unc.edu) 3/24/89
 * modified to be called by 'orb.c', to generate 'spherical heightfields'
 *  via tesselation, POV output format.
 *  John Beale (beale@jump.stanford.edu) 8/27/95
 * modified to calculate vertex normals for use in normal interpolation for
 *  smoothing the output via POV smooth_triangle primatives
 *  Carl Perkins (car@gerg.tamu.edu) 10/8/95
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "hcon.h"

typedef struct {
    double  x, y, z;
} point;

typedef struct {
  point pt;    /* coordinates of vertex */
  point n;     /* normal of vertex */
} vertex;

vertex *v, *fv;
int nvert;

typedef struct {
  vertex *v1, *v2, *v3;    /* vertices of triangle */
  point n;                 /* flat triangle's normal */
} triangle;

typedef struct {
    int       npoly;    /* # of triangles in object */
    triangle *poly;     /* list of Triangles */
} object;

/* Forward declarations */
point *normalize(point *p, double fs);
point *midpoint( point *a, point *b );
void print_object(object *obj, int level, char *fname_out);
void print_triangle(triangle *t, FILE *fp);

static double pi2inv = 1.0 / (2.0 * 3.141592653589793238);

extern int xsize, ysize;    /* size of image array to map to sphere */
extern PTYPE *hf;
extern Boolean raw_out;    /* TRUE if we want to generate 'raw' triangles */
extern Boolean smooth_out; /* TRUE if we want to generate smooth triangles */

int gen_sphere(double f_scale, int maxlevel, char *fname_out)

{
    object *old,
           *new,
           oct;
    int     i,
            level;              /* Current subdivision level */
    double rl;                        /* reciprocal of length */

    fv= (vertex *)malloc( sizeof(vertex)* (2 + pow(4,maxlevel)) );
    v= fv;

/* define original octahedron verticies */

    (*v).pt.x = 1;
    (*v).pt.y = 0;
    (*v).pt.z = 0; nvert++; v++;

    (*v).pt.x = 0;
    (*v).pt.y = 0;
    (*v).pt.z = 1; nvert++; v++;

    (*v).pt.x = 0;
    (*v).pt.y = 1;
    (*v).pt.z = 0; nvert++; v++;

    (*v).pt.x = -1;
    (*v).pt.y = 0;
    (*v).pt.z = 0; nvert++; v++;

    (*v).pt.x = 0;
    (*v).pt.y = -1;
    (*v).pt.z = 0; nvert++; v++;

    (*v).pt.x = 0;
    (*v).pt.y = 0;
    (*v).pt.z = -1; nvert++; v++;

    oct.npoly = 8;
    oct.poly = (triangle *)malloc(8 * sizeof(triangle));

    oct.poly[0].v1= &fv[0];
    oct.poly[0].v2= &fv[1];
    oct.poly[0].v3= &fv[2];

    oct.poly[1].v1= &fv[2];
    oct.poly[1].v2= &fv[1];
    oct.poly[1].v3= &fv[3];

    oct.poly[2].v1= &fv[3];
    oct.poly[2].v2= &fv[1];
    oct.poly[2].v3= &fv[4];

    oct.poly[3].v1= &fv[4];
    oct.poly[3].v2= &fv[1];
    oct.poly[3].v3= &fv[0];

    oct.poly[4].v1= &fv[0];
    oct.poly[4].v2= &fv[2];
    oct.poly[4].v3= &fv[5];

    oct.poly[5].v1= &fv[2];
    oct.poly[5].v2= &fv[3];
    oct.poly[5].v3= &fv[5];

    oct.poly[6].v1= &fv[3];
    oct.poly[6].v2= &fv[4];
    oct.poly[6].v3= &fv[5];

    oct.poly[7].v1= &fv[4];
    oct.poly[7].v2= &fv[0];
    oct.poly[7].v3= &fv[5];

    old = &oct;  /* normalize old data to start with: */
    for (i = 0; i < oct.npoly; i++) {
         old->poly[i].v1->pt = *normalize(&old->poly[i].v1->pt,f_scale);
         old->poly[i].v2->pt = *normalize(&old->poly[i].v2->pt,f_scale);
         old->poly[i].v3->pt = *normalize(&old->poly[i].v3->pt,f_scale);
       }

    /* Subdivide each starting triangle (maxlevel - 1) times */
    for (level = 1; level < maxlevel; level++) {
        /* Allocate a new object */
        new = (object *)malloc(sizeof(object));
        if (new == NULL) {
            fprintf(stderr, "Out of memory on subdivision level %d\n",
                level);
            exit(1);
        }
        new->npoly = old->npoly * 4;

        /* Allocate 4* the number of points in the current approximation */
        new->poly  = (triangle *)malloc(new->npoly * sizeof(triangle));
        if (new->poly == NULL) {
            fprintf(stderr, "Out of memory on subdivision level %d\n",
                level);
            exit(1);
        }

        /* Subdivide each triangle in the old approximation and normalize
         *  the new points thus generated to lie on the surface of the unit
         *  sphere.
         * Each input triangle with vertices labelled [0,1,2] as shown
         *  below will be turned into four new triangles:
         *
         *                      Make new points
         *                          a = (0+2)/2
         *                          b = (0+1)/2
         *                          c = (1+2)/2
         *        1
         *       /\             Normalize a, b, c
         *      /  \
         *    b/____\ c         Construct new triangles
         *    /\    /\              [0,b,a]
         *   /  \  /  \             [b,1,c]
         *  /____\/____\            [a,b,c]
         * 0      a     2           [a,c,2]
         */
        for (i = 0; i < old->npoly; i++) {
            triangle
                 *oldt = &old->poly[i],
                 *newt = &new->poly[i*4];
            point a, b, c;
            int ap, bp, cp;
            int j,flag;

            a = *normalize(midpoint(&oldt->v1->pt, &oldt->v3->pt),f_scale);
            flag = -1;
            for (j = 0; j < nvert; j++) {
                if ((fv[j].pt.x == a.x) && 
                    (fv[j].pt.y == a.y) && 
                    (fv[j].pt.z == a.z)) {
                    flag = j; 
                 }
            }
            if (flag == -1) {
                (*v).pt = a; ap = nvert; nvert++; v++;
            } else  {
                ap = flag;
            }

            b = *normalize(midpoint(&oldt->v1->pt, &oldt->v2->pt),f_scale);
            flag = -1;
            for (j = 0; j < nvert; j++) {
                if ((fv[j].pt.x == b.x) && 
                    (fv[j].pt.y == b.y) && 
                    (fv[j].pt.z == b.z)) {
                    flag = j; 
                 }
            }
            if (flag == -1) {
                (*v).pt = b; bp = nvert; nvert++; v++;
            } else  {
                bp = flag;
            }

            c = *normalize(midpoint(&oldt->v2->pt, &oldt->v3->pt),f_scale);
            flag = -1;
            for (j = 0; j < nvert; j++) {
                if ((fv[j].pt.x == c.x) && 
                    (fv[j].pt.y == c.y) && 
                    (fv[j].pt.z == c.z)) {
                    flag = j; 
                 }
            }
            if (flag == -1) {
                (*v).pt = c; cp = nvert; nvert++; v++;
            } else  {
                cp = flag;
            }

            newt->v1 = oldt->v1;
            newt->v2 = &fv[bp];
            newt->v3 = &fv[ap];
            newt++;

            newt->v1 = &fv[bp];
            newt->v2 = oldt->v2;
            newt->v3 = &fv[cp];
            newt++;

            newt->v1 = &fv[ap];
            newt->v2 = &fv[bp];
            newt->v3 = &fv[cp];
            newt++;

            newt->v1 = &fv[ap];
            newt->v2 = &fv[cp];
            newt->v3 = oldt->v3;
        }

        if (level > 1) {
            free(old->poly);
            free(old);
        }

        /* Continue subdividing new triangles */
        old = new;
    }

    if (smooth_out) {
    /* calculate normals for smooth_triangle output*/

        /* calc flat triangle normals */
        for (i = 0; i < new->npoly; i++) {
            point vec1, vec2;

            vec1.x= new->poly[i].v2->pt.x - new->poly[i].v1->pt.x;
            vec1.y= new->poly[i].v2->pt.y - new->poly[i].v1->pt.y;
            vec1.z= new->poly[i].v2->pt.z - new->poly[i].v1->pt.z;
/*            printf( "vec1 = <%g,%g,%g>\n", vec1.x, vec1.y, vec1.z); */
            vec2.x= new->poly[i].v3->pt.x - new->poly[i].v1->pt.x;
            vec2.y= new->poly[i].v3->pt.y - new->poly[i].v1->pt.y;
            vec2.z= new->poly[i].v3->pt.z - new->poly[i].v1->pt.z;
/*            printf( "vec2 = <%g,%g,%g>\n", vec2.x, vec2.y, vec2.z); */

            new->poly[i].n.x= vec1.z*vec2.y - vec1.y*vec2.z;
            new->poly[i].n.y= vec1.x*vec2.z - vec1.z*vec2.x;
            new->poly[i].n.z= vec1.y*vec2.x - vec1.x*vec2.y;

            rl = 1/sqrt( new->poly[i].n.x*new->poly[i].n.x 
                       + new->poly[i].n.y*new->poly[i].n.y 
                       + new->poly[i].n.z*new->poly[i].n.z);

            new->poly[i].n.x = new->poly[i].n.x * rl;
            new->poly[i].n.y = new->poly[i].n.y * rl;
            new->poly[i].n.z = new->poly[i].n.z * rl;

/*            printf( "new->poly[%d].n = <%g,%g,%g>\n", i, 
                    new->poly[i].n.x, new->poly[i].n.y, new->poly[i].n.z);
*/
        }    

        /* calc vertex normals */
        { 
            int tlist[6];
            int tlc, j;

            for( i = 0; i < nvert; i++) {
                tlc= 0;
                fv[i].n.x= 0;
                fv[i].n.y= 0;
                fv[i].n.z= 0;
                for( j = 0; j < new->npoly; j++) {
                    if ((new->poly[j].v1 == &fv[i]) || 
                        (new->poly[j].v2 == &fv[i]) || 
                        (new->poly[j].v3 == &fv[i])) {
                        tlist[tlc]= j;
                        tlc++;
/*                        printf("v[%d] used by t[%d]\n",i,j); */
                    }
                }
                for( j= 0; j < tlc; j++) {
                    fv[i].n.x = fv[i].n.x + new->poly[tlist[j]].n.x ;
                    fv[i].n.y = fv[i].n.y + new->poly[tlist[j]].n.y ;
                    fv[i].n.z = fv[i].n.z + new->poly[tlist[j]].n.z ;
                }
                if ( tlc > 1) { /* renormalize */
                    rl = 1/sqrt( fv[i].n.x*fv[i].n.x 
                              +fv[i].n.y*fv[i].n.y 
                              +fv[i].n.z*fv[i].n.z);
                    fv[i].n.x= fv[i].n.x * rl;
                    fv[i].n.y= fv[i].n.y * rl;
                    fv[i].n.z= fv[i].n.z * rl;
                }
            }
        }
    }

    /* Print out resulting approximation */
    print_object(old, maxlevel, fname_out); 
    return(0);
} /* end gen_sphere() */

/* Normalize a point p */
point *normalize(point *p, double f_scale)
{
    static point r;
    double mag, rad, phi, theta, r2d, scale;
    int xmap, ymap;

    r = *p;
 /*   printf("xs,ys: %d %d",xsize,ysize);

    printf("xyz: %1.3f %1.3f %1.3f ",r.x,r.y,r.z);
 */
    r2d = sqrt(r.x * r.x + r.y * r.y); /* radius in the x-y plane */
    phi = atan2(r.z,r2d);  /* range of atan2 is [-pi..pi] */
    theta = atan2(r.y, r.x);
   
    xmap = (int)((0.5 + pi2inv * theta) * (double)xsize);
    ymap = (int)((0.5 + pi2inv * phi) * (double)ysize);

    rad = 1.0 + f_scale * El(hf,xmap,ymap);

    mag = r.x * r.x + r.y * r.y + r.z * r.z;  /* r^2 of this point */
    if (mag != 0.0) {  /* definitely should not happen */
        scale = rad / sqrt(mag);
        r.x *= scale;
        r.y *= scale;
        r.z *= scale;
    }
/*    printf("xyz: %1.3f %1.3f %1.3f\n",r.x,r.y,r.z); */

    return &r;
}

/* Return the midpoint on the line between two points */
point *midpoint(a, b)
point *a, *b;
{
    static point r;

    r.x = (a->x + b->x) * 0.5;
    r.y = (a->y + b->y) * 0.5;
    r.z = (a->z + b->z) * 0.5;

    return &r;
}

/* Write out all triangles in an object */
void print_object(object *obj, int level, char *fname_out)
{
    int i;
    FILE *fp;

    fp = fopen(fname_out,"w");

    if (raw_out) fprintf(fp,"ORB\n");    /* RAW text format */
    else {
      fprintf(fp,"/* Include file for POV-Ray : orb from heightfield */\n");
      fprintf(fp,"union {\n");
    }
    /* Spit out coordinates for each triangle */
    for (i = 0; i < obj->npoly; i++)
        print_triangle(&obj->poly[i],fp);

    if (!raw_out) fprintf(fp,"} /* end union */\n");
    fclose(fp);
} /* end print_object() */

/* Output a triangle */
void print_triangle(triangle *t, FILE *fp)
{
        /* Modify this to generate your favorite output format
         * Triangle vertices are in t->pt[0..2].{x,y,z}
         * raw and POV-Ray formats are provided.
         */
        if (raw_out) {
            fprintf(fp,"%g %g %g  %g %g %g  %g %g %g ", 
                t->v1->pt.x, t->v1->pt.y, t->v1->pt.z,
                t->v2->pt.x, t->v2->pt.y, t->v2->pt.z,
                t->v3->pt.x, t->v3->pt.y, t->v3->pt.z);
        } else if (smooth_out) {
          fprintf(fp,"smooth_triangle{\n");
          fprintf(fp,"  <%g,%g,%g>,<%g,%g,%g>,\n", 
              t->v1->pt.x, t->v1->pt.y, t->v1->pt.z,
              t->v1->n.x, t->v1->n.y, t->v1->n.z);
          fprintf(fp,"  <%g,%g,%g>,<%g,%g,%g>,\n", 
              t->v2->pt.x, t->v2->pt.y, t->v2->pt.z,
              t->v2->n.x, t->v2->n.y, t->v2->n.z);
          fprintf(fp,"  <%g,%g,%g>,<%g,%g,%g>", 
              t->v3->pt.x, t->v3->pt.y, t->v3->pt.z,
              t->v3->n.x, t->v3->n.y, t->v3->n.z);
          fprintf(fp,"}");
        } else {
          fprintf(fp,"triangle{");
          fprintf(fp,"<%g,%g,%g>,", t->v1->pt.x, t->v1->pt.y, t->v1->pt.z);
          fprintf(fp,"<%g,%g,%g>,", t->v2->pt.x, t->v2->pt.y, t->v2->pt.z);
          fprintf(fp,"<%g,%g,%g>",  t->v3->pt.x, t->v3->pt.y, t->v3->pt.z);
          fprintf(fp,"}");
        }
        fprintf(fp,"\n");

} /* end print_triangle */

