/* copyx.c:  convert heightfield data to various formats      
 *          read in array of height values,
 *          rescale to twice the size,
 *          write out in specified format
 *           John P. Beale 7/18/95                           
 *          GIF library routines from gd library             */

#include <stdio.h>                      /* fopen() sprintf() etc. */
#include <stdlib.h>                     /* strtod() */
#include <math.h>                       /* math stuff */
#include <string.h>                     /* strcpy() */
#include "hcon.h"
#include "hf_io.h"

void hf_print();                        /* print out hf array to stdout */
void doublex(PTYPE **hf, PTYPE **hf2);  /* rescale array size */
void smooth(PTYPE **hf1, PTYPE **hf2);  /* returns hf2 as smooth(hf1) */

unsigned int xsize, ysize;     /* size of hf array (input xsize+2) */
unsigned int xsize2, ysize2;   /* size of rescaled array hf2 */
PTYPE *hf;                            /* height field array */
PTYPE *hf2;                    /* rescale array */
PTYPE fmin = 0, fmax = 1.0;                 /* scaling limits */
Boolean norenorm = FALSE;
Boolean negate = FALSE;

/* ---------------------------------- */

int main(int argc, char **argv)
{ char *usage = 
"copyx: A Heightfield Converter v0.1a (c) <beale@jump.stanford.edu> 1995\n\
Usage: copyx <infile> <outfile> \n\
  doubles size of infile and writes as outfile. Reads many types.\n\
  Determines filetype based on extension, eg:   hcon file1.tga file1.pot\n\
  POT,TGA are 16-bit formats, MAT is Matlab floating point format.\n";
  
  char fname_in[160];
  char fname_out[160];

  int rval;
  int inlen, outlen;
  size_t memspace; 


  if ((argc < 3) || (argc > 3)) {
	fprintf(stderr,"%s",usage);
	exit(1);
      }
 
  strcpy(fname_in,argv[1]);
  strcpy(fname_out,argv[2]);

  inlen = strlen(fname_in);
  outlen = strlen(fname_out);
  
					/* read in heightfield */
   rval = readn_hf(&hf, &xsize, &ysize, fname_in);
   if (rval != 0)
     exit(rval);

   doublex((PTYPE **)&hf, (PTYPE **)&hf2);   /* double in size */
   xsize = xsize2;
   ysize = ysize2;
   free(hf);          /* get rid of old image */

   memspace = (size_t) xsize*ysize*sizeof(PTYPE);

   hf = (PTYPE *)malloc(memspace);        /* space for new image */
   if (hf==NULL) {
     fprintf(stderr,"could not allocate %lu bytes for output array.\n",
	 (unsigned long)memspace);
     exit(1);
   }
   smooth((PTYPE **)&hf2, (PTYPE **)&hf); /* returns smoothed image in hf */

   printf("%s: [%d x %d] \n",fname_out,xsize,ysize);
   
   if (xsize < 9) hf_print();  /* print small arrays */

   rval = writen_hf(hf, xsize, ysize, fname_out);
   
  return(rval);    /* return a value */

} /* end main() */

void hf_print()                /* print out hf array on stdout */
{
  int ix,iy;

   printf("Heightfield array: %dx%d\n",xsize,ysize);
   for (iy = 0; iy<ysize; iy++) {
    printf("%d:",iy);
    for (ix = 0; ix<xsize; ix++) {
	printf(" %1.2f",El(hf,ix,iy));
    }
     printf("\n");
   }
}



/* -------------------------------------------------------------------- */
/* doublex() --  double data in hf[xsize,ysize] to make 2 copies        */
/* -------------------------------------------------------------------- */

void doublex(PTYPE **hf1, PTYPE **hf2)
{
 PTYPE tmp;
 int ix,iy;
 size_t memspace;

 xsize2 = xsize * 2;
 ysize2 = ysize;
 
 memspace = (size_t) xsize2*ysize2*sizeof(PTYPE);

 hf2[0] = (PTYPE *)malloc(memspace);
 if (hf2[0]==NULL) {
   fprintf(stderr,"could not allocate %lu bytes for output array.\n",
	 (unsigned long)memspace);
   exit(1);
 }


   for (iy = 0; iy<ysize; iy++) {
    for (ix = 0; ix<xsize; ix++) {
      tmp = El(hf1[0],ix,iy);
      El2(hf2[0],ix,iy) = tmp;
      El2(hf2[0],ix+xsize,iy) = tmp;
    }
   }

}       /* end doublex() */

/* smooth() - copy smoothed version of argument 1 into argument 2, 
 * both of which are pointers to (xsize x ysize) arrays. 
 */

void smooth(PTYPE **hf1, PTYPE **hf2)  /* copy hf1 into hf2 */
{
int ix, iy;
PTYPE tmp;

   for (iy = 0; iy<ysize; iy++) {
    for (ix = 0; ix<xsize; ix++) {
      tmp = El(hf1[0],ix,iy);
      El2(hf2[0],ix,iy) = tmp;
    }
   }

}
