/* composit.c:  combine two heightfields to make a third     */
/*          read in array of values in PGM or Matlab format. */
/*          write out in PGM format or GIF format            */
/*           John P. Beale 6/10/95                           */
/*          GIF library routines from gd 1.1.1 library       */

#include <stdio.h>                      /* fopen() sprintf() etc. */
#include <stdlib.h>                     /* strtod() */
#include <math.h>                       /* math stuff */
#include <string.h>                     /* strcpy() */
#include "hcon.h"
#include "hf_io.h"

#define ADD 0           /* keytypes for image operation */
#define SUB 1
#define MUL 2
#define EXP 3
#define COM 4

void renorm();                          /* normalize array */
					/* combine two arrays */
float composit(PTYPE **h1, PTYPE **h2, 
   unsigned int x, unsigned int y, int op, PTYPE fscale, PTYPE foff);

unsigned int xsize, ysize;     /* size of hf array (input xsize+2) */
PTYPE *hf;                            /* height field array */
PTYPE fmin = 0, fmax = 1.0;                 /* scaling limits */
Boolean norenorm = FALSE;
Boolean negate = FALSE;

/* ---------------------------------- */

int main(int argc, char **argv)
{ char *usage = 
"composit: A Heightfield Combiner  (c) <beale@jump.stanford.edu> 1995\n\
Usage: composit <opcode> <infile1> <infile2> <outfile> [<f> [<off>]]\n\
  Combines heightfields:\n\
  Input from PGM|MAT|GIF|POT|TGA. Output to PGM|PG8|MAT|OCT|GIF|POT|TGA.\n\
  Input files are required to be of the same dimension, but type may vary.\n\
  <opcode> must be one of the following five:\n\
  add: out = in1 + in2\n\
  sub: out = in1 - in2\n\
  mul: out = in1 * in2\n\
  exp: out = in1 ^ in2\n\
  com: out = each value is greater of (in1, in2)\n\
  f:   factor to scale second file by. Typically much less than 1.0.\n\
  off: factor to offset second file by. Often between -1.0 and 1.0\n";

  PTYPE *hf2;              /* second input data array */
  unsigned int xsize2, ysize2;  /* size of second input data array */
  float frac1;             /* fraction of first file which contributes */
  PTYPE fscale = 1.0;                   /* scale second data file by this */
  PTYPE foffset = 0.0;                  /* offset second data file by this */  

  char fname_in1[160];
  char fname_in2[160];
  char fname_out[160];
  char *buf;
  char opn[40];                  /* operation type string */
  int op;                       /* operation type key */
  int rval;

  if ((argc < 5) || (argc > 7)) {
	fprintf(stderr,"%s",usage);
	exit(1);
      }
 
  strcpy(opn,argv[1]);
  if (strcmp(opn,"add")==0) op = ADD;
  else if (strcmp(opn,"sub")==0) op = SUB;
  else if (strcmp(opn,"mul")==0) op = MUL;
  else if (strcmp(opn,"exp")==0) op = EXP;
  else if (strcmp(opn,"com")==0) op = COM;
  else {
    fprintf(stderr,"Composit: Unknown operation type %s.\n",opn);
    exit(1);
  }
  if (argc>5) {
    fscale = (PTYPE)strtod(argv[5],&buf);
  }
  if (argc>6) {
    foffset = (PTYPE)strtod(argv[6],&buf);
  }

  strcpy(fname_in1,argv[2]);
  strcpy(fname_in2,argv[3]);
  strcpy(fname_out,argv[4]);

				
				/* read in first data array */
   
   rval = readn_hf(&hf, &xsize, &ysize, fname_in1);
   if (rval != 0)
     exit(rval);

   printf("%s: [%d x %d] \n",fname_in1,xsize,ysize);
   
				/* read in second data array */
   rval = readn_hf(&hf2, &xsize2, &ysize2, fname_in2);
   if (rval != 0)
     exit(rval);

   printf("%s: [%d x %d] \n",fname_in2,xsize2,ysize2);

   if ( (xsize != xsize2) || (ysize != ysize2) ) {
     fprintf(stderr,"Oops! input arrays are not of the same size.\n");
     exit(1);
   }

   printf("Compositing data, scale2 = %1.1e offset2 = %1.1e\n",
		fscale,foffset);
   frac1 = composit(&hf,&hf2,xsize,ysize,op,fscale,foffset); 
		   /* composit returns result in first array arg. */
   if (op==COM) 
     printf("Fraction of result taken from %s: %1.3f\n",fname_in1,frac1);

   if (xsize < 9) hf_print();  /* print small arrays */
   printf("Writing output %s...\n",fname_out);
   rval = writen_hf(hf, xsize, ysize, fname_out);
   
  return(rval);    /* return a value */

} /* end main() */

void hf_print()                /* print out hf array on stdout */
{
  int ix,iy;

   printf("Heightfield array: %dx%d\n",xsize,ysize);
   for (iy = 0; iy<ysize; iy++) {
    printf("%d:",iy);
    for (ix = 0; ix<xsize; ix++) {
	printf(" %1.2f",El(hf,ix,iy));
    }
     printf("\n");
   }
}


/* -------------------------------------------------------------------- */
/* renorm() --  scale data in hf[xsize,ysize] to fit in the full        */
/* range 0..1  */
/* -------------------------------------------------------------------- */

void renorm()
{
 PTYPE dmax, dmin, tmp, scale;
 int ix,iy;


 dmax = El(hf,0,0);
 dmin = El(hf,0,0);

   for (iy = 0; iy<ysize; iy++) {
    for (ix = 0; ix<xsize; ix++) {
	if (negate) El(hf,ix,iy) = (1.0 - El(hf,ix,iy));    /* invert */
	tmp = El(hf,ix,iy);
       /* printf("%1.1f ",tmp); */
	if (tmp > dmax) dmax = tmp;
	if (tmp < dmin) dmin = tmp;
    }
   }

   printf("Input file had range %1.5e - %1.5e\n",dmin,dmax);
   if (fmin != 0.0) dmin = fmin;
   if (fmax != 1.0) dmax = fmax;
   scale = 1.0 / (dmax - dmin);
   if (norenorm) {
     scale = 1.0;
     dmin = 0.0;
   }

   for (iy = 0; iy<ysize; iy++) {
    for (ix = 0; ix<xsize; ix++) {
	tmp = (El(hf,ix,iy) - dmin) * scale;
	if (tmp < 0.0) tmp = 0.0;       /* clipping */
	if (tmp > 1.0) tmp = 1.0;       
	El(hf,ix,iy) = tmp;
    }
   }
}       /* end renorm() */


float composit(PTYPE **h1, PTYPE **h2, 
   unsigned int xsize, unsigned int ysize, int op, 
   PTYPE fscale, PTYPE foffset)
{
int ix, iy;
PTYPE ht1, ht2;
float frac = 0;

  for (iy = 0; iy < ysize; iy++) {
   for (ix = 0; ix < xsize; ix++) {
     ht1 = El(h1[0],ix,iy);
     ht2 = El(h2[0],ix,iy) * fscale + foffset;
     if      (op == ADD) El(h1[0],ix,iy) = ht1 + ht2;
     else if (op == SUB) El(h1[0],ix,iy) = ht1 - ht2;
     else if (op == MUL) El(h1[0],ix,iy) = ht1 * ht2;
     else if (op == EXP) {
	if (ht2 < 0) ht2 = 0;
	El(h1[0],ix,iy) = pow(ht1,ht2);
	}
     else if (op == COM) {
       if (ht1 > ht2) {
	 El(h1[0],ix,iy) = ht1;
	 frac++;
       }
	 else El(h1[0],ix,iy) = ht2;
     } /* end if op==COM */

   }  /* end for ix */
  }  /* end for iy */

  frac /= (xsize*ysize);
  return(frac);
} /* end composit() */

