/* hcon.c:  convert heightfield data to various formats      */
/*          read in array of values in PGM or Matlab format. */
/*          write out in PGM format or GIF format            */
/*           John P. Beale 6/10/95                           */
/*          GIF library routines from gd library             */

#include <stdio.h>                      /* fopen() sprintf() etc. */
#include <stdlib.h>                     /* strtod() */
#include <math.h>                       /* math stuff */
#include <string.h>                     /* strcpy() */
#include "hcon.h"
#include "hf_io.h"

void renorm(int r);                     /* normalize array */

U xsize, ysize;     /* size of hf array (input xsize+2) */
PTYPE *hf;                            /* height field array */
PTYPE fmin = 0, fmax = 1.0;                 /* scaling limits */
Boolean norenorm = FALSE;
Boolean negate = FALSE;
Boolean rescale = FALSE;

/* ---------------------------------- */

int main(int argc, char **argv)
{ char *usage = 
"hcon: A Heightfield Converter v0.1a (c) <beale@jump.stanford.edu> 1995\n\
Usage: hcon <infile> <outfile> [fmin [fmax]]\n\
       hcon <infile> <outfile> -negate\n\
       hcon <infile> <outfile> -rescale\n\
  Converts heightfields: PGM|MAT|GIF|POT|TGA to PGM|PG8|MAT|OCT|GIF|POT|TGA\n\
  Determines filetype based on extension, eg:   hcon file1.tga file1.pot\n\
  Rescale option scales data to fill entire range (8|16|32 bits).\n\
  Increase contrast by setting thresholds fmin [0..1], fmax [0..1]. \n\
  If no additional arguments present, no rescaling is done.\n\
  POT,TGA are 16-bit formats, MAT is Matlab floating point format.\n";
  
  char fname_in[160];
  char fname_out[160];
  char *ext1, *ext2;
  char *buf;
  int rval;
  int inlen, outlen, otype, itype;

  if ((argc < 3) || (argc > 6)) {
	fprintf(stderr,"%s",usage);
	exit(1);
      }
 
  strcpy(fname_in,argv[1]);
  strcpy(fname_out,argv[2]);

  if (argc > 3) {
   rescale = TRUE;
   if (strcmp(argv[3],"-negate")==0) {
     negate = TRUE;
     printf("Inverting polarity of image.\n");
   } else if (strcmp(argv[3],"-rescale")==0) {
     printf("Rescaling to fill output range.\n");
   } else {
    fmin = (PTYPE) strtod(argv[3], &buf);
    if (argc > 4) {
      fmax = (PTYPE) strtod(argv[4], &buf);
    }
    if (argc > 5) {
      norenorm = TRUE;
    }
   } /* end else */
  }

  inlen = strlen(fname_in);
  outlen = strlen(fname_out);
  
  ext1 = (char *) malloc((size_t)sizeof(char)*5);
  ext2 = (char *) malloc((size_t)sizeof(char)*5);

  strncpy(ext1,(char *)(&fname_in[inlen-4]),4);
  strncpy(ext2,(char *)(&fname_out[outlen-4]),4);

  if ((ext1==NULL) || (ext2==NULL)) { 
    fprintf(stderr,"filenames must have type extension, eg. topo.gif\n");
    exit(1);
  }

  if       (strcmp(ext1,".gif")==0) itype = GIF;  
   else if (strcmp(ext1,".pot")==0) itype = POT;
   else if (strcmp(ext1,".tga")==0) itype = TGA;
   else if (strcmp(ext1,".pgm")==0) itype = PGM;
   else if (strcmp(ext1,".pg8")==0) itype = PG8;
   else if (strcmp(ext1,".mat")==0) itype = MAT;
   else if (strcmp(ext1,".oct")==0) itype = OCT;
   else if (strcmp(ext1,".png")==0) itype = PNG;
   else { 
     itype = PGM;       /* default type */
     fprintf(stderr,"Warning: %s has a missing or unknown file extension\n",argv[1]);
   }

  if       (strcmp(ext2,".gif")==0) otype = GIF;  
   else if (strcmp(ext2,".pot")==0) otype = POT;
   else if (strcmp(ext2,".tga")==0) otype = TGA;
   else if (strcmp(ext2,".pgm")==0) otype = PGM;
   else if (strcmp(ext2,".pg8")==0) otype = PG8;
   else if (strcmp(ext2,".mat")==0) otype = MAT;
   else if (strcmp(ext2,".oct")==0) otype = OCT;
   else if (strcmp(ext2,".png")==0) otype = PNG;
   else { 
     fprintf(stderr,"Error: %s has a missing or unknown file extension\n",argv[2]);
     exit(1);
   }

   printf("otype = %d\n",otype);
					/* read in heightfield */
   rval = read_hf(&hf, &xsize, &ysize, fname_in, itype);
   if (rval != 0)
     exit(rval);

   printf("%s: [%d x %d] \n",fname_in,xsize,ysize);
   
   renorm(rescale);   /* rescale array to [0..1] */
   
   if (xsize < 9) hf_print();  /* print small arrays */

   rval = write_hf(hf, xsize, ysize, fname_out, otype);

   return(rval);    /* return a value */
} /* end main() */

void hf_print()                /* print out hf array on stdout */
{
  int ix,iy;

   printf("Heightfield array: %dx%d\n",xsize,ysize);
   for (iy = 0; iy<ysize; iy++) {
    printf("%d:",iy);
    for (ix = 0; ix<xsize; ix++) {
	printf(" %1.2f",El(hf,ix,iy));
    }
     printf("\n");
   }
}



/* -------------------------------------------------------------------- */
/* renorm() --  scale data in hf[xsize,ysize] to fit in the full        */
/* range 0..1  */
/* -------------------------------------------------------------------- */

void renorm(int rescale)
{
 PTYPE dmax, dmin, tmp, scale;
 int ix,iy;


 dmax = El(hf,0,0);
 dmin = El(hf,0,0);

   for (iy = 0; iy<ysize; iy++) {
    for (ix = 0; ix<xsize; ix++) {
	if (negate) El(hf,ix,iy) = (1.0 - El(hf,ix,iy));    /* invert */
	tmp = El(hf,ix,iy);
       /* printf("%1.1f ",tmp); */
	if (tmp > dmax) dmax = tmp;
	if (tmp < dmin) dmin = tmp;
    }
   }

   printf("Input file had range %1.5e - %1.5e\n",dmin,dmax);
   if (fmin != 0.0) dmin = fmin;
   if (fmax != 1.0) dmax = fmax;
   scale = 1.0 / (dmax - dmin);
   
   if (norenorm) {
     scale = 1.0;
     dmin = 0.0;
   }
   
   if (rescale) {
    for (iy = 0; iy<ysize; iy++) {
     for (ix = 0; ix<xsize; ix++) {
	 tmp = (El(hf,ix,iy) - dmin) * scale;
	 if (tmp < 0.0) tmp = 0.0;       /* clipping */
	 if (tmp > 1.0) tmp = 1.0;       
	 El(hf,ix,iy) = tmp;
     }
    }
   } /* end if */

}       /* end renorm() */

