/********************************************************************************
 *  Projektname		: AERO
 *  Filename		: koerper.h
 *  Filetyp		: Header
 ********************************************************************************
 *  Modulname		: koerper.c
 *  letzte Aenderung	: 18.05.93
 *  Autor		: Horst Stolz (HUS)
 *  
 *  Beschreibung:
 *    Typdefinitionen die mit einem Koerper zusammenhaengen.
 *
 *  Exportierte Funktionen:
 *    void KoeIniMech(TKoerper *k, TMaterialTabelle *MatTab);
 *    void SetzeKoerperMasselos(TKoerper *k, TBoolean masselos);
 *    void SetzeKoerperGravlos(TKoerper *k, TBoolean nograv);
 *    TBoolean IstKoerperGravlos(TKoerper *k);
 *    TBoolean IstKoerperMasselos(TKoerper *k);
 *    void KoerperKopieren(TKoerper *nach TKoerper *von, TKoerper *neuerober);
 *    TKoerper *KoerperAllozieren();
 *    void ZusGesObjRot(TKoerper *k);
 *
 *  Versionsgeschichte:
 *  24.01.93	Koerperdefinition von folgeschrittberechnung.h abgekoppelt.
 *  27.01.93    Neuer Koerper EBENE, BewTyp um GEFUEHRT erweitert. 
 *  11.02.93    Ergaenzung um RGB-Werte (Hartmut)
 *  20.03.93    Angriffsflaeche zwecks Luftwiderstand
 *  22.03.93    Routinen zwecks Masselos(unbeweglich) und Gravitationslos 
 *  02.04.93    Erweiterung der Koerperstruktur um ZusGesObj spez. Elememte
 *  03.04.93    Routinen um Koerper-Strukt. zu allokieren bzw. zu kopieren
 *  20.04.93    Neues BewTy-Attribut KOLL -> Kollisionstest wenn gesetzt
 *  05.05.93    Id-Nummer pro Koerper
 *  18.05.93    Durchsichtigkeit, Reflexion, Rauhigkeit ergaenzt (Hartmut)
 *  01.06.93    ZusGesObjRot(k) zwecks Drehung eines Zusammengesetzten Obj.
 ********************************************************************************/
#ifndef KOERPER_H
#define KOERPER_H

#ifndef GRUNDTYPEN_H
#include "grundtypen.h"
#endif

#ifndef MATERIALIEN_H
#include "materialien.h"
#endif

#ifndef KRAFT_H
#include "kraft.h"
#endif


/****************************************************************************
 * Geometrische Grundformen bzw. Koerperart:
 * Entweder es handelt sich um eine einfache Koerper wie eine
 * KUGEL, QUADER oder ZYLINDER, oder um einen besonderen Punkt wie 
 * der NAGEL (Masselosen Punkt der fest im Raum steht) oder der
 * masselose PUNKT (frei beweglich im Raum, an ihm soll spaeter eine
 * Kamera oder eine Lichtquelle angebracht werden koennen, hat also 
 * im Gegensatz zum NAGEL eine raeumliche Orientierung)
 * Ein ZUSGESOBJ besteht aus einer Liste von einfachen Grundkoerpern
 * (KUGEL, QUADER, ZYLINDER, PUNKT, MPUNKT). Diese bilden von der
 * raeumlichen Form her eine Einheit.
 * EBENE ist eine Masselose, unendliche masselose Ebene die in der
 * koerperfeste XZ-Ebene liegt. Sie ist nicht beweglich!
 */

typedef	enum EGrundFormen {     /* geometrische Grundformen */
    ZYLINDER, QUADER, KUGEL, MPUNKT, PUNKT, NAGEL, /* raumfester Punkt */
    ZUSGESOBJ, EBENE         
} TGrundFormen;
    

/****************************************************************************
 * Geometrie-Definitionen
 * 3 raeumliche Grundformen, Zusatzstruktuer fuer den PUNKT ist noch nicht
 * angelegt, da Kamera/Licht-Struktur fehlt!
 */


typedef struct TZusGesObj {
    struct TKoerper *KoerperListe;
} TZusGesObj;

typedef struct TZylinder {                   /* Zylinder */
    TReal Hoehe, Radius;		     /* Symmetrie-Achse = z-Achse = [2] */
} TZylinder;

typedef struct TQuader {                   /* Quader */
    TVektor	KantenLaenge;		   /* Kantenlaengen ensprechend den Achsen */
/*  Frueher:
 *    TReal   Breite, Hoehe, Tiefe;
 */
} TQuader;

typedef struct TKugel {                   /* Kugel */
    TReal    Radius;
} TKugel;

typedef union TForm {
    TQuader    Quader;
    TZylinder  Zylinder;
    TKugel     Kugel;
    TZusGesObj ZusGesObj;
} TForm;

/****************************************************************************
 * Private Struktur!!! (Horst)
 *
 * Enhaelt alle Variablen die mit Numerischem Verfahren hochintegriert werden 
 * muesse.
 */
typedef struct TIVariablen {
    TVektor p, v, w;            /* Position, Geschw., Winkelgeschw. */
    TQuaternion	q;		/* Euler-Par */
} TIVariablen;


/****************************************************************************
 * Private Typ-Definition (Horst)
 * Bewegungstyp des Koerpers!
 * Abfrage mit ->BewTyp & ...
 */
typedef enum TBewTyp {
    FREI     = 1,    /* Frei Beweglicher Koerper -> BewGl, Koll. */
    MASSELOS = 2,    /* Es werden keine Bewegungsgleichungen aufgestellt */
    GEFUEHRT = 4,    /* Bewegungsvar p,v,a,q,w,u sind vorgegeben */
    GRAV0    = 8,    /* keine Gravitation */
    KOLL     = 16    /* Kollisionsbehandlung */
} TBewTyp;





/****************************************************************************
 * Koerperdefinition:
 * Ein Koerper besteht zum einen aus seine geometrischen Form(Art). Dazu
 * wird eine geometrische Art im Element "Art" eingetragen. Zusaetzliche
 * Parameter je nach Art sind in "Form" einzutragen!
 * Die physikalischen Eigenschaften: das heisst die Materialart
 * wird in "Material" eingetragen, die Masse(in kg) in "Masse" ergibt sich
 * aus der Materialdichte ((aus der MaterialTabelle) dem Volumen des Koerpers.
 * "Masse" braucht also nicht belegt zu werden sondern wird auch durch
 * KoeIniMech(k) errechnet!
 *
 * Um den Bewegungszustand des Koerpers zu setzen dienen die
 * Vektoren "Position", "Geschwindigkeit" und "Beschleunigung".
 * (Einheiten sind: m, m/s und m/s^2)
 * Ausserdem sind da noch die Eulerparameter "q[0]..q[3]" (siehe dazu "ks.dvi")
 * welche die Orientierung definieren, sowie die "Drehgeschwindigkeit"
 * (in rad/s) und "Drehbeschleunigung" (Einheit rad/s^2).
 *
 */


typedef struct TKoerper {
    struct TKoerper *Naechster;
    TGrundFormen  Art;			  /* Koerperart */
    TForm         Form;			  /* naehre Koerperdefinitionen */
    TMaterial     Material;		  /* Materialnummer */	          
    TReal         Masse;
    TVektor	  Position, Geschwindigkeit; /* r, v */
    TVektor	  Drehgeschwindigkeit;	  /* \omega */
    TQuaternion   q;			  /* Eulerparameter */
    TVektor	  Beschleunigung;	  /* a */
    TVektor       Drehbeschleunigung;	  /* d/dt\omega */
    
/* 
 * Teil von Hartmut: Farben und andere Dinge, die mit der Anzeige zu tun
 * haben. Diese werden wohl noch erweitert, sodass auch Durchsichtigkeits-
 * Informationen fuer den Raytracer enthalten sind.
 */


/* 
 * AStatus ist ein Bitvektor:
 *   Bit 0: Koerper selected (1) oder nicht (0)
 *   Bit 1: Koerper hidden (1) oder nicht (0)
 * Hidden ist gedacht, bei der Eingabe einige Objekte zu verstecken, damit es
 * bei der Darstellung ohne verdeckte Linien nicht zu unuebersichtlich wird.
 * Inwieweit das unterstuetzt werden wird, wird sich noch zeigen.
 */

    unsigned short AStatus;		  /* Bitvektor, siehe oben */
    unsigned short R, G, B;		  /* Farbwerte Rot/Gruen/Blau 0..65535 */
    TReal Durchsichtigkeit;		  /* 0.0..1.0 */
    TReal Reflexion;			  /* 0.0..1.0 */
    TReal Rauhigkeit;			  /* 0.0..1.0 */

/*
 * Privater Teil "Folgeschrittberechnung" (Horst Stolz)
 *
 * folgende Struktur-Elemente werden selbststaendig durch die
 * Folgeschrittberechnung gesetzt. ABER - Die Traegheitstensoren muessen
 * mit der Funktion KoeIniMech(k) vor Aufruf der Folgeschrittberechnung
 * gesetzt werde!!!!!!
 *
 * WICHTIG WICHTIG WICHTIG,
 * die Matrizen Traegheitstensor* sind zwar im privaten Teil werden aber
 * nur ein mal per KoeIniMech initialisiert und muessen somit immer mitkopiert 
 * werden (-> Andi). Dei Traegheitstensoren entsprechen der Massenverteilung
 * des Koerpers und sind somit aehnlich zur Masse oben!
 */

    TMatrix       RotTrans;             /* K->K' Rotations-Matrix aus Euler-par.*/
    TMatrix	  RotTrans1;		/* K'->K Umkehrmatrix */
    TMatrix       Traegheitstensor;	/* sind nur 6 bzw. 3 Werte und nicht 9! */
    TMatrix       Traegheitstensor1;	/* Umkehrmatrix ! */
/*
 * Zwischenwerte fuer Integrationsverfahren werden in den folgenden 
 * Variablen abgelegt. (Privat!!)
 */
    TVektor       a, u; /* Nicht hochzuintegrierende Vars */
    TIVariablen   *ty;   /* ym=... NUR LESEND (kann = y sein!) */
    TIVariablen   *y;    /* Funktionsauswertung - Ausgabe ->ym, k0, k1, ...*/
    
/* Der BewTyp gibt an ob der Koerper FREI Beweglich ist, also Bewegungs-
 * gleichungen beruecksichtigt werden muessen. Beim MASSELOSer Koerper 
 * brauchen keine Bewgl. aufgestellt werden, jegliche Krafteinwirkung
 * verpufft. Hier sollte auch spaeter ein GEFUEHRTER Koerper beruecksichtigt
 * werden.
 *
 * ACHTUNG !!!! BewTyp muss immer korrekt besetzt sein
 */
    TBewTyp       BewTyp;


    
/* OEFFENTLICHER TEIL
 * Erweiterung der Koerperstruktur um Kraefte die auf diesen Koerper angreifen.
 * geht nur bei nicht in einem ZusGesObj eingebetteten Koerper!!
 *
 */
    struct TKraft        *Kraefte;


/* privater Teil, muss aber auch mitkopiert werden!  */
/* Werte im privatem Teil werdes von der FBB-Berechnung erstellt und duerfen nicht
 * vom Benutzer der FBB gesetzt werden!
 */
 
    TReal Angriffsflaeche;       /* fuer Luftwiderstand! (Braeunl -grrrr!?'@#%) */

/* Halbe Kantenlaenge eines Quaders
 * wird auch fuer einen Zylinder benutzt solange keine Zylinder-Kollroutinen
 * vorhanden.
 */
    TVektor HalbeKantenlaenge;       /* in Koerperkoordinaten */

/* Jeder Koerper mit einer geometrischen Ausdehnung (Quader, Kugel, Zylinder) 
 * wird in eine Kugel gehuellt zwecks leichterer Erkennung von Kol. moeglich!
 */
    TReal Kugelhuellenradius;

/* Wenn Koerper zu einem Zusammengesetztem Objekt gehoert steht hier nun die
 * Position und Drehung(Quaternion) bezueglich des Schwerpunktes des 
 * (gesammten) ZusGesObjekts(Root) in den Koerperkoordinaten des Root!!!
 */
    TVektor Position_bzg_ZGO;
    TQuaternion q_bzg_ZGO;

/* Koerper zu dem Teilkoerper gehoert!
 * (Koerper->ZusGes..->KoerperList->...OberKoerper == Koerper)
 * andernfalls ist Koerper==Koerper->OberKoerper!
 */
    struct TKoerper *OberKoerper;

/* Eindeutiger K"orperbezeichner
 */

    long KoerperId;
} TKoerper;






void KoeIniMech(TKoerper *k, TMaterialTabelle *MatTab);
/****************************************************************************
 * Zweck:
 *   Initialisierung der Massen-Traegheitsmatrizen des Koerpers k aus den
 *   gegebenen geometrischen Ma/3en und der Koerpermasse.
 *   KoeInitMech(..) muss vor der Benutzung von FolgeBildBerechnung(..) auf-
 *   gerufen werden! 
 *   => jedes neue Objekt das in die Zustands-Struktur eingefuegt wird muss
 *      per KoeIniMech(k) initialisiert werden!!
 * Bemerkung:
 *   Wird ein Form-Parameter wie Radius oder Zylinder-Hoehe,.. geaendert 
 *   muss KoeIniMech erneut aufgerufen werden!
 */



void SetzeKoerperMasselos(TKoerper *k, TBoolean masselos);
/****************************************************************************
 * Zweck:
 *   Ein Koerper wie Quader, Kugel oder Zylinder kann als Masselos gesetzt
 *   werden was zur Folge hat das er sich nicht mehr bewegen kann.
 *
 * Anwendung:
 *   Nach der Initialisierung per KoeInitMech(k,..) kann durch denn
 *   Aufruf von SetzeKoerperMasselos(k, masselos) der Koerper als Masselos
 *   oder Massebehaftet also frei beweglich markiert werden.
 *
 * Parameter:
 *   k - Koerper der als Masselos bzw. Frei gesetzt werden soll.
 *   masselos - TRUE->Masselos, FALSE->Frei
 *
 * Anwendung:
 *
 *   KoeInitMech(k, mattab);
 *
 *   SetzeKoerperMasselos(k, TRUE);
 *
 */


void SetzeKoerperGravlos(TKoerper *k, TBoolean nograv);
/****************************************************************************
 * Zweck:
 *   Zum An/Ausschalten der Gravitationswirkung auf einen Koerper k.
 *
 * Anwendung:
 *   Wie bei SetKoerperMasselos(..);
 */

void SetzeKoerperKollmoeglich(TKoerper *k, TBoolean koll);
/****************************************************************************
 * Zweck:
 *   Zum An/Ausschalten der Kollisionserkennung fuer den Koerper k
 *
 * Anwendung:
 *   Wie bei SetKoerperMasselos(..);
 */


TBoolean IstKoerperKollmoeglich(TKoerper *k);
TBoolean IstKoerperGravlos(TKoerper *k);
TBoolean IstKoerperMasselos(TKoerper *k);
/****************************************************************************
 * Zweck:
 *   Zum Erfragen ob Koerper Masselos ist bzw. die Gravitation
 *   beruecksichtigt wird oder nicht.
 *
 * Rueckgabewert:
 *   TRUE  -> Masselos bzw. Gravitationslos bzw. Kollisionen m"oglich
 *   FALSE -> Massebehaftet bzw. Gravitationsbehaftet bzw. keine Koll.erkennung
 */

void SetzeKoerperMasse(TKoerper *k, TReal masse);
/****************************************************************************
 * Zweck:
 *   Setzt die Masse eines Koerpers auf den vorgegeben Wert Masse,
 *   Ist Masse==0.0 wird der Koerper auf Masselos gesetzt. (wenn geht)
 *
 * Bemerkung:
 *   Koerperstruktur muss vorher mit KoeIniMech initialisiert worden sein.
 */



void KoerperKopieren(TKoerper *nach, TKoerper *von, TKoerper *neuerober);
/****************************************************************************
 * Zweck:
 *   Kopiert den Inhalt der Koerperstruktur von Koerper "von" zum Koerper
 *   "nach".  
 *   Kraefte-Strukturen oder eingebettete Koerper werden nicht mitkopiert!
 * 
 *   Falls der Koerper ein eingebetteter Koerper ist (ist in der 
 *   Koerperliste eines ZUSGESOBJ)  muss in neuerober der 
 *   Oberkoerper des eingebetteten Koerpers stehen (und zwar die Adr.
 *   der Kopierten Version)! Ansonsten muss nach==neuerober sein!
 *   
 * Anwendung:
 *   
 *   TKoerper k1, k2;
 *
 *   k2.Art = KUGEL;
 *   ...
 *
 *   KoerperKopieren(&k1, &k2, &k2);
 *
 */


TKoerper *KoerperAllozieren();
/****************************************************************************
 * Zweck:
 *   Belegt Speicher fuer eine Koerper-Struktur.
 *   Freigabe kann mit dem free-Aufruf gemacht werden!
 *   Die Allozierung einer Koerperstruktur sollte IMMER mit dieser
 *   Routine erfolgen!
 *   
 * Rueckgabewert:
 *   Zeiger auf die allozierte (aber uninitialisierte) Struktur 
 *   oder NULL wenn kein Speicher angefordert werden konnte.
 *   
 */




void ZusGesObjRot(TKoerper *k);
/****************************************************************************
 * Zweck:
 *   Berechnet die Position und Orientierung der Teilkoerper eines ZusGesOb
 *   wenn die Position bzw. Orientierung des Root-Koerpers veraendert wurde.
 *   
 * Parameter:
 *   Zeiger auf den Root-Koerper des zusges. Objekts.
 *   
 */


#endif
