/********************************************************************************
 *  Projektname:	AERO
 *  Filename:		animat.c
 *  Filetyp:		Modul
 ********************************************************************************
 *  WICHTIG:            Dieser Quelltext benoetigt einen Compiler, der Struktur-
 *                      Zuweisungen beherrscht (z.B. ANSI-C, gcc). Sind also z.B.
 *                      a und b Strukturen gleichen Typs, muss bei b=a; der ge-
 *                      samte Inhalt von a nach b kopiert werden.
 ********************************************************************************
 *  Modulname:		Anzeige
 *  Version:		9
 *  letzte Aenderung:	19.4.94
 *  Autor:  		Hartmut 
 *  Status:		Haupt-Routinen der Anzeige fuer das Animationsfenster
 *										
 *  imp. Bezeichner:    alles aus autils.c
 *  ----------------
 *
 *  exp. Bezeichner:
 *  ----------------
 *  void InitialisiereAnimation(XtAppContext *a, Widget animation):
 *          Initialisieren aller noetigen Variablen. Diese Routine ist einmal
 *          zu Beginn des Programms aufzurufen.
 *
 *  void OeffneAnimation(XtAppContext *a, Widget animation):
 *          Diese Routine muss immer aufgerufen werden wenn das Animationsfenster
 *          geoeffnet wurde (also das Window existiert). Es werden dann die
 *          noetigen Actions eingehaengt und noetige Initialisierungen durch-
 *          gefuehrt.
 *
 *  void SchliesseAnimation(XtAppContext *a, Widget animation):
 *          Diese Routine muss immer aufgerufen werden, bevor (!) das
 *          Animationsfenster geschlossen wird. Die reservierte Pixmap wird
 *          dann wieder freigegeben. Darum duerfen anschliessend auch keine
 *          Aufrufe von AnzeigeAnimation() erfolgen, bis das Fenster wieder
 *          geoeffnet und OeffneAnimation() ausgefuehrt wurde.
 *
 *  void AnzeigeAnimation(Widget animation, TKamera kamera, TZustand *zustand,
 *                        int modus)
 *          Anzeige des angegebenen Zustands auf dem Animationsfenster.
 *          Dabei wird durch modus die Art der Anzeige gesteuert (modus ist
 *          ein Bitvektor, gleiche Funktion wie bei AnzeigeDarstellung()):
 *              Bit 0: Fussboden zeichnen (1) oder nicht (0)
 *              Bit 1: Koordinatenachsen der Fenster zeichnen (1) oder nicht (0)
 *              Bit 2: Koordinatenachsen der einzelnen Koerper zeichnen oder
 *                     nicht (0)
 *                      
 *
 *  Beschreibung:								
 *  -------------								
 *  In diesem Modul sind die Routinen zusammengefasst, die mit dem
 *  Animationsfenster zu tun haben. Es handelt sich dabei um Routinen zur
 *  Auswertung von Events (Groessenaenderung und Redraw) sowie die
 *  eigentlichen Routinen zum Initialisieren und Hinzeichnen des
 *  Animationsfensters. 
 *
 *  Da in diesem Window eine moeglichst flackerfreie Animation ablaufen
 *  soll, wird eine Pixmap im X-Server benutzt, die so gross wie das Window
 *  selbst ist. Anstatt nun in das Window direkt zu malen, wird zuerst auf
 *  die Pixmap gezeichnet und diese dann am Ende auf das Window (und damit
 *  den Bildschirm) kopiert. Dieses Vorgehen hat 2 Vorteile:
 *
 *  1. Der Redraw fuer das Fenster wird sehr einfach: es wird einfach erneut
 *     die Pixmap auf das Window kopiert.
 *  2. Wuerde direkt auf das Window gezeichnet, muesste ja zuerst der alte
 *     Windowinhalt geloescht werden. Dann werden nach und nach die
 *     einzelnen Linien gezeichnet. Dabei kann man keine Aussage darueber
 *     machen, wie lange es tatsaechlich dauert, bis alles auf dem Bild
 *     vorhanden ist (Anzahl Linien im Bild variabel, unterschiedlich
 *     schnelle X-Server, Verzoegerungen durch stark belastetes Netz, ...).
 *     Es ist also sehr wahrscheinlich, dass der Elektronenstrahl das Bild
 *     (zumindest teilweise) auf dem Bildschirm abbildet, solange es noch
 *     nicht vollstaendig ist, zumal sich die Animation wohl nicht mit dem
 *     Bildschirm synchronisieren laesst. Das Ergebnis waere ein rrecht
 *     wildes Geflacker unvollstaendiger Bilder auf dem Schirm.
 *     Durch die Benutzung der Pixmap wird ein zweiter Bildpuffer
 *     realisiert, in dem das Bild (unsichtbar fuer den Benutzer) erzeugt
 *     wird, und das dann einfach ueber das alte Bild geschrieben wird, es
 *     gibt also im ganzen Darstellungsprozess keine Phase, in der ein nicht
 *     vollstaendiges Bild zu sehen ist. Dadurch steht das Bild sehr ruhig,
 *     auch wenn die Bildgenerierung laenger dauert als ein
 *     Elektronenstrahldurchgang ueber den Bildschirm. Diese Technik wird als
 *     Double Buffering bezeichnet.
 *										
 *  Ein Grossteil dieser Routinen aehnelt vom prinzipiellen Aufbau den
 *  Routinen der Darstellung (siehe anzeige.c). Da hier jedoch nicht
 *  explizit eine Darstellungsliste aufgebaut werden muss, sondern direkt in
 *  die Pixmap gezeichnet werden kann, konnten einige Teile vereinfacht
 *  werden. Demgegenueber stehen etwas kompliziertere Algorithmen fuer die
 *  perspektivische Abbildung (z.B. Feder, Daempfer).
 *
 *  Fehler:
 *  -------
 *  diverse kleine Maengel (s.o.: noch zu tun)
 *
 *  Versionsgeschichte:								
 *  -------------------								
 *   1:       Grundlegende Version mit Parallelprojektion, einfach nur,
 *            damit was tut. 
 *   2:       Perspektivische Projektion, erste Verbindungen; Bug entfernt,
 *            der jedesmal, wenn das Animationsfenster geoeffnet wurde, eine
 *            neue Pixmap angefordert hatte, ohne die alte freizugeben.
 *   3:       dreidimensionale Verbindungen Feder und Daempfer
 *   4:       Fenster-Koordinatenachsen, Farbe
 *   5:       Pixmap wird bei SchliesseAnimation() wieder freigegeben
 *   6:       Zusammengesetzte Objekte eingebaut, einige Fehler korrigiert
 *   7:       Umwandlung TReal->AXKoord bei Koordiantenpaaren jetzt ueber
 *            clipping. Ausserdem: Bei Rot-Gruen-Bildern wird schwarz
 *            gezeichnet, wenn beide Pixel (also rot und gruen) gesetzt sind.
 *            Koordinatenachsen werden korrekt gesetzt.
 *   8:       Wenn alle Farben belegt sind, dann Absturz bei Farballozierung
 *            von Read/Write-Cells beseitigt.
 *   9:       Fehler bei 3D-Ausgabe (Rot-Gruen) beseitigt. Kameras wurden falsch
 *            seitlich versetzt fuer die Augen.
 *   
 *   zu tun:
 *   -------
 *   
 ********************************************************************************/

#include <X11/Intrinsic.h>       
#include <X11/StringDefs.h>

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>

#include "animat.h"
#include "autils.h"
#include "Kamera.h"

/*======================================================================*/

/***********************************************************************
 *  Routinen und Variablen fuer das Animationsfenster
 ***********************************************************************/

/* ------- Infos ueber das verwendete Fenster der Animation ------- */

static Window  window_a;		  /* Window-ID fuer Animation */
static Display *display;		  /* Display Zeiger */
static int     screen;			  /* Screen-ID */
static AXKoord winwidth;		  /* Breite des Fensters */
static AXKoord winheight;		  /* Hoehe des Fensters */
static AXKoord windepth;		  /* Bittiefe des Fensters */
					  /* (benoetigt fuer Pixmap) */



/* ---------- Info ueber verwendete Pixmap fuer Double-Buffering ---------- */
static Pixmap pixmap;			  /* Pixmap fuer Fenster-Inhalt */
static TBoolean habepixmap = FALSE;	  /* Flag, ob schon Pixmap erzeugt */




/* ----- Infos ueber die verwendeten Graphics-Contexts der Animation ----- */

static GC anim_GC;			  /* GC fuer normales Zeichnen */
static GC anim_pfeilGC;			  /* GC fuer Koordinatenachsen */
static GC loesch_GC;			  /* GC fuer's Loeschen der Pixmap */
static unsigned int liniendicke;	  /* aktuelle Liniendicke in anim_GC */

XFontStruct *dfont;			  /* Info ueber den Font (noetig */
					  /* fuer die Achsenbeschriftung) */

/* ------ Infos ueber Rot-Gruen-Bilder und deren Farbe -------- */

static int rg_funktion;			  /* Zeichenmodus: GXor oder GXand */
static long farbe[4];			  /* Farben: Schwarz, Rot, Gruen, Weiss */
enum GrundFarben { weiss, rot, gruen, schwarz };


/* ---------- Matrizen zur Transformation von Punkten im Raum ---------- */

static AMatrix M_anim;			  /* Transformation fuer die */
					  /* Kameraposition */
static AMatrix M_t;			  /* Gesamttransformation fuer */
					  /* einen Koerper/eine Verbindung */

/******************************************************************************
 *  Die Matrix M_t enthaelt im einzelnen folgende Transformationen:
 *  1. Objekt in die Welt platzieren (fuer jedes Objekt anders)
 *    1a. Objekt in korrekte Lage drehen (-> Quaternion)
 *    1b. Objekt an seine Position in der Welt verschieben     
 *  2. Kameraposition einbeziehen
 *    2a. Drehung des Koordinatensystems in die Lage der  Kamera
 *    2b. Verschiebung der Kamera nach hinten (in z-Richtung)
 *
 *  Mit dieser Matrix kann die gesamte Transformation (ausser der Projektion
 *  3D->2D) fuer einen Punkt eines Koerpers/einer Verbindung auf einmal
 *  ausgefuehrt werden. 
 *****************************************************************************/



/* -------- Variablen zur Einsparung unnoetiger Funktions-Parameter -------- */

static TReal    d_zoom;			  /* momentaner Zoom-Faktor */
static AXKoord  winmaxx,		  /* maximale x- und y-Koordinate */
                winmaxy;		  /* des momentanen Windows */
static int      modus;			  /* Anzeigemodus fuer das momentane */
					  /* Window (Werte siehe bei */
					  /* AnzeigeDarstellung() ) */



/*======================================================================*/


#define FarbenProMalloc 100

typedef struct farbliste
{ 
     struct farbliste *next;
     AXColornum       farbnr[FarbenProMalloc];
} AFarbliste;


static AFarbliste *farbtabelle = NULL;
static AFarbliste *oldcol;
static farbenanzahl = 0;



/*****************************************************************************
 *  Funktion:       void SetzeFarbe(unsigned short R, unsigned short G,
 *                                                       unsigned short B);
 *  Parameter:      R,G,B: gewuenschte Farbe
 *                            
 *  Rueckgabewert:  
 *
 *  Import. Bezeichner: farbtabelle:  Liste mit den Farbeintraegen
 *                      oldcol:       Zeiger auf aktuellen Farbtabelleneintrag
 *                      farbenanzahl: Anzahl noch freier Farben in diesem Farb-
 *                                    tabelleneintrag
 *
 *  Beschreibung:
 *  -------------
 *  Es wird versucht, die angegebene Farbe in der Farbtabelle zu allozieren.
 *  Falls dies geht, wir die Farbnummer in eine Liste eingetragen, damit sie 
 *  spaeter wieder freigegeben werden kann. Die einzelnen Listeneintraege
 *  koennen mehrere Farbnummern aufnehmen, damit nicht allzuviele malloc()
 *  aufgerufen werden muessen.
 *****************************************************************************/

static void SetzeFarbe(unsigned short R, unsigned short G, unsigned short B)
{
     XColor farbe;
     AFarbliste *zeiger;

     if (modus & Show3dON) return;

     farbe.red   = R;
     farbe.green = G;
     farbe.blue  = B;
     farbe.flags = DoRed | DoGreen | DoBlue;

     AGetColor(display, a_colormap, &farbe, "SetzeFarbe()"); /* Farbe allozieren */
     if (farbenanzahl == 0)		  /* hat in diesem Listeneintrag noch */
     {					  /* eine Farbe platz? */
	  zeiger = (AFarbliste *) malloc(sizeof(AFarbliste)); /* nein, neuen */
	  if (zeiger == NULL)		  /* Listeneintrag anfordern */
	  {
	       AFehler("malloc(): No room for color entries in object color list.\n");
	       return;
	  }
	  if (farbtabelle == NULL)	  /* ggf. Anfangszeiger setzen */
	  {
	       farbtabelle = zeiger;
	  }
	  else
	  {
	       oldcol->next = zeiger;
	  }
	  oldcol         = zeiger;
	  zeiger->next   = NULL;	  /* letztes Element der Liste */
	  farbenanzahl = FarbenProMalloc; /* hier sind alle Plaetze frei */
     }
     farbenanzahl--;		  /* nachste Farbnummer */
     oldcol->farbnr[farbenanzahl] = farbe.pixel;
     XSetForeground(display, anim_GC, farbe.pixel); /* Farbe setzen */
}



/*****************************************************************************
 *  Funktion:       void FarbenFreigeben()
 *  Parameter:      -
 *                            
 *  Rueckgabewert:  -
 *
 *  Import. Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Die Liste, die nach und nach mit SetzeFarbe aufgebaut wurde, wird wieder
 *  freigegeben. Dabei werden auch die Farben wieder dealloziert.
 *****************************************************************************/

static void FarbenFreigeben()
{
     AFarbliste *zeiger, *old;

     if (farbtabelle == NULL) return;	  /* Liste leer? */

     zeiger = farbtabelle;
     while (zeiger->next != NULL)	  /* voller Listeneintrag */
     {					  /* Farben deallozieren */
	  XFreeColors(display, a_colormap, zeiger->farbnr, FarbenProMalloc, 0);
	  old = zeiger;
	  zeiger = zeiger->next;
	  free(old);			  /* Listeneintrag freigeben */

     }
     if (farbenanzahl < FarbenProMalloc)  /* letzter Listeneintrag nur teilweise */
     {					  /* belegt */
	  XFreeColors(display, a_colormap, &(zeiger->farbnr[farbenanzahl]), 
		      FarbenProMalloc-farbenanzahl, 0);	/* Farben deallozieren */
     }
     free(zeiger);			  /* letzten Listeneintrag freigeben */
     farbenanzahl = 0;			  /* keine freien Farben */
     farbtabelle = NULL;		  /* Liste leer */
}


/******************************************************************************
 *  z=GRENZE ist die Gleichung der Ebene, bis zu der Objekte noch als
 *  sichtbar dargestellt werden. Alle Objekte mit z>=GRENZE sind unsichtbar.
 *  (Anmerkung: GRENZE=ANIM_DEFAULTABSTAND geht nicht, da sonst bei der
 *  Projektion durch 0 geteilt wird.)
 ******************************************************************************/

#define GRENZE ANIM_DEFAULTABSTAND-0.1


/******************************************************************************
 *  Das Makro TransformationPunkt entspricht ATransformation, jedoch wird implizit
 *  die Matrix M_t mit der gesamten Koerpertransformation verwendet.
 ******************************************************************************/


#define TransformationPunkt(punkt, x, y, z) ATransformation(punkt, M_t, x, y, z)



/*****************************************************************************
 *  Funktion:       void ProjektionPunkt(TVektor punkt, AXKoord *x, AXKoord *y)
 *  Parameter:      punkt: Koordinaten des Punktes (3D)
 *                            
 *  Rueckgabewert:  x,y:   Koordinaten des Punktes (2D im Window)
 *
 *  Importierte Bezeichner: ANIM_DEFAULTABSTAND, B_ACHSEN_X, B_ACHSEN_Y
 *
 *  Beschreibung:
 *  -------------
 *  Transformation des Punktes punkt auf die 2D-Ebene. Punkt muss dabei
 *  einen z-Wert<0 haben, damit er im Blickfeld der virtuellen Kamera liegt.
 *  Hier (im Gegensatz zur Darstellung in anzeige.c) wird der Punkt wirklich
 *  ueber eine perspektivische Projektion auf das Window abgebildet.
 *****************************************************************************/

static void ProjektionPunkt(TVektor punkt, AXKoord *x, AXKoord *y)
{
#if 1
     /* Perspektivische Projektion */
     TReal f;

     f = d_zoom*ANIM_DEFAULTABSTAND/(ANIM_DEFAULTABSTAND-punkt[2]);

     *x = TReal2AXKoord(B_ACHSEN_X*punkt[0]*f+winmaxx/2 +0.5);
     *y = TReal2AXKoord(B_ACHSEN_Y*punkt[1]*f+winmaxy/2 +0.5);
#else

     /* Parallelprojektion */
     *x = TReal2AXKoord(B_ACHSEN_X*punkt[0]*d_zoom+winmaxx/2 +0.5);
     *y = TReal2AXKoord(B_ACHSEN_Y*punkt[1]*d_zoom+winmaxy/2 +0.5);
#endif
}





/*****************************************************************************
 *  Funktion:       void TRealProjektionPunkt(TVektor punkt, TReal *x, TReal *y)
 *  Parameter:      punkt: Koordinaten des Punktes (3D)
 *                            
 *  Rueckgabewert:  x,y:   Koordinaten des Punktes (2D im Window)
 *
 *  Importierte Bezeichner: ANIM_DEFAULTABSTAND, B_ACHSEN_X, B_ACHSEN_Y
 *
 *  Beschreibung:
 *  -------------
 *  Transformation des Punktes punkt auf die 2D-Ebene. Punkt muss dabei
 *  einen z-Wert<0 haben, damit er im Blickfeld der virtuellen Kamera liegt.
 *  Hier (im Gegensatz zur Darstellung in anzeige.c) wird der Punkt wirklich
 *  ueber eine perspektivische Projektion auf das Window abgebildet.
 *****************************************************************************/

static void TRealProjektionPunkt(TVektor punkt, TReal *x, TReal *y)
{
#if 1
     /* Perspektivische Projektion */
     TReal f;

     f = d_zoom*ANIM_DEFAULTABSTAND/(ANIM_DEFAULTABSTAND-punkt[2]);

     *x = B_ACHSEN_X*punkt[0]*f+winmaxx/2 +0.5;
     *y = B_ACHSEN_Y*punkt[1]*f+winmaxy/2 +0.5;
#else

     /* Parallelprojektion */
     *x = B_ACHSEN_X*punkt[0]*d_zoom+winmaxx/2 +0.5;
     *y = B_ACHSEN_Y*punkt[1]*d_zoom+winmaxy/2 +0.5;
#endif
}




#if 0
/*****************************************************************************
 *  Funktion:       XPoint ProjektionAchsenPunkt(TReal x, TReal y, TReal z)
 *  Parameter:      x,y,z: Koordinaten des Punktes vor der Abbildung
 *                            
 *  Rueckgabewert:         2D-Koordinaten des abgebildeten Punktes
 *
 *  Importierte Bezeichner: M_anim: Matrix fuer Transformation
 *                          winmaxx:   Breite des aktuellen Fensters
 *                          winmaxy:   Hoehe des aktuellen Fensters
 *
 *  Beschreibung:
 *  -------------
 *  Projektion eines Punktes auf das aktuelle Fenster. Dies wird nur mit der 
 *  Matrix M_anim durchgefuehrt und ist damit fuer Punkte geeignet, die nur vom
 *  Fenster und nicht von irgendwelchen Koerpern oder Verbindungen abhaengen
 *  (z.B. Koordinatenachsen). 
 *****************************************************************************/

static XPoint ProjektionAchsenPunkt(TReal x, TReal y, TReal z)
{
     XPoint punkt;
     TVektor p;

     ATransformation(p, M_anim, x, y, z);
     ProjektionPunkt(p, &punkt.x, &punkt.y);

     return punkt;
}
#endif


/*****************************************************************************
 *  Funktion:       void ClipLinie2d(AXKoord x1, AXkoord y1, AXKoord x2, AXKoord y2)
 *  Parameter:      x1, y1 : Anfangspunkt der Linie (2D-Koordinaten)
 *                  x2, y2 : Endpunkt der Linie (2D-Koordinaten)
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   winmaxx: Breite des Windows, es wird auf 0..winmaxx geclipped
 *                  winmaxy: Hoehe des Windows, es wird auf 0..winmaxy geclipped
 *                  
 *
 *  Beschreibung:
 *  -------------
 *  Zeichnen der 2D-Linie (x1,y1)-(x2,y2) in die Pixmap. Das Clipping wird
 *  X-Windows ueberlassen.
 *****************************************************************************/

#if 1   /* da nur eine Linie gezeichnet wird, reicht auch ein Makro */
#define ClipLinie2d(x1, y1, x2, y2) XDrawLine(display, pixmap, anim_GC,\
					      x1, y1, x2, y2);
#endif

#if 0   /* urspruengliche Funktionsdeklaration */
static void ClipLinie2d(AXKoord x1, AXKoord y1, AXKoord x2, AXKoord y2)
{
     XDrawLine(display, pixmap, anim_GC, x1, y1, x2, y2);
}
#endif




/*****************************************************************************
 *  Funktion:       void ClipTReal2AXKoord(TReal *x1, TReal *y1, 
 *                                                 TReal x2, TReal y2)
 *  Parameter:      x1, y1:   ein Linienende (Achtung: Pointer!)
 *                  x2, y2:   das andere Linienende (keine Pointer!)
 *                            
 *  Rueckgabewert:  x1, y1:  AXmin<x1,y1<AXmax, also evtl. veraendert
 *
 *  Importierte Bezeichner: AXmin, AXmax
 *
 *  Beschreibung:
 *  -------------
 *  Die Koordinaten x1 und y1 werden unter Beruecksichtigung von x2 und y2
 *  so korrigiert, dass sie auf jeden Fall im Wertebereich von AXKoord
 *  liegen, sodass sie ohne weiteres gecasted werden koennen. Achtung! Es
 *  wird nur x1 und y1 veraendert. Will man also eine ganze Linie clippen,
 *  muss man diese Routine 2x aufrufen, einmal mit vertauschten Koordinaten.
 *****************************************************************************/

void ClipTReal2AXKoord(TReal *x1, TReal *y1, TReal x2, TReal y2)
{
     if (*x1>AXmax)
     {
	  *y1 = (*y1-y2)*(AXmax-x2)/(*x1-x2)+y2;
	  *x1 = AXmax;
     }
     else if (*x1<AXmin)
     {
	  *y1 = (*y1-y2)*(AXmin-x2)/(*x1-x2)+y2;
	  *x1 = AXmin;
     }

     if (*y1>AXmax)
     {
	  *x1 = (*x1-x2)*(AXmax-y2)/(*y1-y2)+x2;
	  *y1 = AXmax;
     }
     else if (*y1<AXmin)
     {
	  *x1 = (*x1-x2)*(AXmin-y2)/(*y1-y2)+x2;
	  *y1 = AXmin;
     }

}



/*****************************************************************************
 *  Funktion:       void TRealClipLinie2d(TReal x1, TReal y1, 
 *                                                 TReal x2, TReal y2)
 *  Parameter:      x1, y1,
 *                  x2, y2:   Endkoordinaten der Linie
 *                            
 *  Rueckgabewert:  
 *
 *  Importierte Bezeichner: ClipLinie2d()
 *
 *  Beschreibung:
 *  -------------
 *  Die angegebenen Linienkoordinaten werden auf den Wertebereich von AXKoord
 *  ueberprueft. Ueberschreitet mindestens einer der Werte die Grenzen,
 *  wird die Linie so geclipped, dass alle Werte im Wertebereich von AXKoord
 *  liegen. Danach wird die Linie ueber ClipLinie2d() gezeichnet. Es handelt
 *  sich also um eine ClipLinie2d()-Version mit TReal- anstatt AXKoord-
 *  Koordinaten.
 *****************************************************************************/

void TRealClipLinie2d(TReal x1, TReal y1, TReal x2, TReal y2)
{
     ClipTReal2AXKoord(&x1, &y1, x2, y2);
     ClipTReal2AXKoord(&x2, &y2, x1, y1);
     ClipLinie2d((AXKoord) x1, (AXKoord) y1, (AXKoord) x2, (AXKoord) y2);
}





/*****************************************************************************
 *  Funktion:       void ClipLinie3d(TVektor p1, TVektor p2)
 *  Parameter:      p1, p2:  Endpunkte der Linie (im Raum)
 *                            
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   winmaxx: Breite des Windows, es wird auf 0..winmaxx geclipped
 *                  winmaxy: Hoehe des Windows, es wird auf 0..winmaxy geclipped
 *
 *  Beschreibung:
 *  -------------
 *  Die Linie mit den 3D-Endpunkten p1 und p2 wird (mit Parallelprojektion) auf
 *  eine Ebene (2D) projiziert. ACHTUNG: zur Vereinfachung des Algorithmus kann es
 *  sein, dass Anfangs- und Endpunkt logisch vertauscht werden.
 *  Anschliessend wird die 2D-Linie durch Aufruf von ClipLinie2d() auf das
 *  Window gezeichnet.
 *****************************************************************************/


static void ClipLinie3d(TVektor p1, TVektor p2)
{
     TReal x1, y1, x2, y2;
     TReal *hilfs;
     TReal h;

     if (p1[2] >= GRENZE)			 
     {
	  if (p2[2] >= GRENZE) return; /* Linie nicht sichtbar (hinter Kamera) */

	  hilfs = p1;			  /* wenn, dann nur p2 hinter Kamera */
	  p1 = p2;			  /* also: p1 und p2 vertauschen */
	  p2 = hilfs;

     }

     TRealProjektionPunkt(p1, &x1, &y1);  /* Anfangspunkt garantiert sichtbar */

     if (p2[2] < GRENZE)	  /* Linie normal sichtbar */
     {
	  TRealProjektionPunkt(p2, &x2, &y2);
     }
     else				  /* Linie mit Bildebene schneiden */
     {
	  TVektor neu;

	  h = (GRENZE-p1[2])/(p2[2]-p1[2]);
	  neu[0] = p1[0]+h*(p2[0]-p1[0]);
	  neu[1] = p1[1]+h*(p2[1]-p1[1]);
	  neu[2] = GRENZE;

	  TRealProjektionPunkt(neu, &x2, &y2);
     }

     TRealClipLinie2d(x1, y1, x2, y2);
}




/*****************************************************************************
 *  Funktion:       void ClipLinieTReal3d(TVektor cp, TVektor rp)
 *  Parameter:      cp: Clip-Punkt (dieses Koordinatenpaar wird ggf. veraendert)
 *                  rp: Referenz-Punkt (bleibt garantiert unveraendert)
 *                            
 *  Rueckgabewert:  cp:  cp[2]<GRENZE
 *
 *  Importierte Bezeichner: GRENZE
 *
 *  Beschreibung:
 *  -------------
 *  Die Linie mit den Endpunkten cp und rp wird geclipped, wenn der Punkt cp
 *  vor der Projektionsebene zu liegen kaeme. Dies geschieht dann, indem die
 *  Linie soweit verkuerzt wird, dass cp direkt auf der Bildebene (bzw. der
 *  GRENZE-Ebene) liegt.
 *****************************************************************************/

static void ClipLinieTReal3d(TVektor cp, TVektor rp)
{
     TReal h;

     if (cp[2] < GRENZE) return;	  /* Anfangspunkt sichtbar? Alles ok */
     if (rp[2] >= GRENZE) return;         /* ganze Linie unsichtbar? Auch fertig */

     /* hier gilt also: Anfangspunkt cp nicht sichtbar, aber Endpunkt rp. 
	Also clippen bei cp*/
     h = (GRENZE-rp[2])/(cp[2]-rp[2]);
     cp[0] = rp[0]+h*(cp[0]-rp[0]);
     cp[1] = rp[1]+h*(cp[1]-rp[1]);
     cp[2] = GRENZE;
}



/*****************************************************************************
 *  Funktion:       void ClipPfeil3d(TVektor start, TVektor ziel, char *info)
 *  Parameter:      start:    Startpunkt des Pfeils (ohne Pfeilspitze)
 *                  ziel:     Endpunkte der Linie (mit Pfeilspitze)
 *                  info:     Text, der an die Pfeilspitze geschrieben wird
 *                            (sollte genau 1 Zeichen sein, aber als String)
 *                            
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   TRealProjektionPunkt(), ClipLinieTReal3d(), 
 *                  ClipTReal2AXKoord(), ClipLinie2d(), QUADRAT, GRENZE
 *                  
 *
 *  Beschreibung:
 *  -------------
 *  Die Linie mit den 3D-Endpunkten start und ziel wird (mit
 *  perspektivischer Projektion) in das Animationsfenster projiziert und
 *  gezeichnet. Danach wird an die 2D-Linie eine Pfeilspitze gesetzt.
 *  ACHTUNG:
 *  Das Dreieck der Pfeilspitze selbst wird in dieser Version *nicht*
 *  geclipped, d.h. die Pfeilspitze ist immer ganz oder gar nicht zu sehen.
 *****************************************************************************/

static void ClipPfeil3d(TVektor start, TVektor ziel, char *info)
{
     TVektor hilfs;
     TBoolean spitzemalen;
     TReal x1, y1, x2, y2;
     TReal hx2, hy2;

#if 0
     len3d = sqrt(QUADRAT(start[0]-ziel[0])+QUADRAT(start[1]-ziel[1])+
		  QUADRAT(start[2]-ziel[2]));
#endif

     ClipLinieTReal3d(start, ziel);

     hilfs[0] = ziel[0];
     hilfs[1] = ziel[1];
     hilfs[2] = ziel[2];
     ClipLinieTReal3d(ziel, start);

     /* Wenn Linie gar nicht sichtbar, dann zurueck */
     if ((ziel[2]>GRENZE) || (start[2]>GRENZE)) return;

     /* Falls beim Zielpunkt geclipped, dann keine Spitze malen */
     spitzemalen = ((hilfs[0]==ziel[0]) && (hilfs[1]==ziel[1]) && 
		    (hilfs[2]==ziel[2]));

     TRealProjektionPunkt(start, &x1, &y1); /* Anfangspunkt auf 2D umrechnen */
     TRealProjektionPunkt(ziel, &x2, &y2); /* ebenso Endpunkt */

     hx2 = x2;
     hy2 = y2;
     ClipTReal2AXKoord(&x1, &y1, x2, y2); /* 2D-Clipping auf AXKoord Anfang */
     ClipTReal2AXKoord(&x2, &y2, x1, y1); /* ebenso bei der Pfeilspitze */

     /* Linie des Pfeils zeichnen */
     ClipLinie2d((AXKoord) x1, (AXKoord) y1, (AXKoord) x2, (AXKoord) y2);

     spitzemalen = (spitzemalen && (hx2==x2) && (hy2==y2));

     /* Jetzt Spitze malen, wenn weder bei 3D- noch bei 2D-Clipping am */
     /* Linienende (also bei der Pfeilspitze) geclipped wurde. */

     /* ACHTUNG! In dieser Version wird die Pfeilspitze als Gesamtes */
     /* weggelassen,  auch wenn sie eigentlich noch teilweise zu sehen */
     /* waere. */

     if (spitzemalen)
     {
	  XPoint p[3];
	  TReal vx, vy;
	  TReal vlen;
	  int fwidth, fheight, fsize;

	  p[0].x = (AXKoord) x2;
	  p[0].y = (AXKoord) y2;

	  /* 2d-Vektor von ziel nach start */
	  vx = (TReal)((AXKoord) (x1) - p[0].x);
	  vy = (TReal)((AXKoord) (y1) - p[0].y);

	  /* Wenn Vektor Laenge 0 hat (nur 1 Pixel), dann hier fertig */
	  if ((vx==0) && (vy==0)) return;

	  /* diesen Vektor normieren */
	  vlen = sqrt(vx*vx+vy*vy);
	  /* Wenn Pfeilspitze nicht hinpasst, dann auch fertig */ 
	  if (vlen<ANIM_PFEILHOEHE) return;
	  vx /= vlen;
	  vy /= vlen;

	  p[1].x = p[0].x + (AXKoord)(vx*ANIM_PFEILHOEHE-vy*ANIM_PFEILBREITE);
	  p[1].y = p[0].y + (AXKoord)(vy*ANIM_PFEILHOEHE+vx*ANIM_PFEILBREITE);
	  p[2].x = p[0].x + (AXKoord)(vx*ANIM_PFEILHOEHE+vy*ANIM_PFEILBREITE);
	  p[2].y = p[0].y + (AXKoord)(vy*ANIM_PFEILHOEHE-vx*ANIM_PFEILBREITE);

  	  XFillPolygon(display, pixmap, anim_pfeilGC, p, 3, Convex, CoordModeOrigin);
	  fwidth = XTextWidth(dfont, info, 1);
	  fheight = dfont->max_bounds.ascent;
	  fsize = ((fwidth>fheight)?fwidth:fheight)/3;

	  /* Text an Pfeilspitze schreiben */
	  XDrawString(display, pixmap, anim_pfeilGC, 
		      p[0].x+(AXKoord)(vx*ANIM_PFEILHOEHE/2+vy*ANIM_PFEILBREITE*2.5
				       -fsize),
		      p[0].y+(AXKoord)(vy*ANIM_PFEILHOEHE/2-vx*ANIM_PFEILBREITE*2.5
				       +fsize),
		      info, 1);
     }

}



/*****************************************************************************
 *  Funktion:       void ProjektionQuader(TVektor ausdehnung)
 *  Parameter:      ausdehnung: Groesse des Quaders in die 3 Achsenrichtungen
 *                            
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   winmaxx: Breite des Fensters (fuer Clipping auf 0..winmaxx)
 *                  winmaxy: Hoehe des Windows (fuer Clipping auf 0..winmaxy).
 *                  modus:   Anzeigemodus (Bitvektor, siehe AnzeigeAnimation())
 *                                relevant hier: ##### noch nichts #####
 *
 *  Beschreibung:
 *  -------------
 *  Den Quader mit der Groesse ausdehnung auf die Pixmap mit der Groesse
 *  0..winmaxx, 0..winmaxy projizieren. Dazu alle 8 Eckpunkte abbilden
 *  und dann alle 12 Linien berechnen und zeichnen.
 *****************************************************************************/

static void ProjektionQuader(TVektor ausdehnung)
{
     TReal vorne, hinten, oben, unten, rechts, links;
     TVektor luv, lov, rov, ruv, luh, loh, roh, ruh;
     
#if 0
     if (modus & ObjectCoordON)
     {
	  TVektor m, xe, ye, ze;	  /* Koordinatenachsen des Koerpers */

	  TransformationPunkt(m, 0, 0, 0);
	  TransformationPunkt(xe, 0.2, 0, 0); /* Koerper-Achsen projizieren */
	  TransformationPunkt(ye, 0, 0.2, 0);
	  TransformationPunkt(ze, 0, 0, -0.2);
	  ClipLinie3d(m, xe);		  /* Koerper-Achsen clippen und */
	  ClipLinie3d(m, ye);		  /* hinzeichnen */
	  ClipLinie3d(m, ze);
     }
#endif

     rechts= ausdehnung[0]/2;
     links = -rechts;
     oben  = ausdehnung[1]/2;
     unten = -oben;
     vorne = ausdehnung[2]/2;
     hinten= -vorne;

     TransformationPunkt(luv, links, unten, vorne);
     TransformationPunkt(lov, links, oben, vorne);
     TransformationPunkt(rov, rechts, oben, vorne);
     TransformationPunkt(ruv, rechts, unten, vorne);
     TransformationPunkt(luh, links, unten, hinten);
     TransformationPunkt(loh, links, oben, hinten);
     TransformationPunkt(roh, rechts, oben, hinten);
     TransformationPunkt(ruh, rechts, unten, hinten);

#if 0
#ifdef ADEBUG
     printf("ProjektionQuader: Breite: %f, Hoehe: %f, Tiefe: %f\n", ausdehnung[0], ausdehnung[1], ausdehnung[2]);
#endif
#endif
     
     /* vorderes Rechteck */
     ClipLinie3d(luv, lov);
     ClipLinie3d(lov, rov);
     ClipLinie3d(rov, ruv);
     ClipLinie3d(ruv, luv);

     /* hinteres Rechteck */
     ClipLinie3d(luh, loh);
     ClipLinie3d(loh, roh);
     ClipLinie3d(roh, ruh);
     ClipLinie3d(ruh, luh);

     /* Verbindungslinien von vorne nach hinten */
     ClipLinie3d(luv, luh);
     ClipLinie3d(lov, loh);
     ClipLinie3d(rov, roh);
     ClipLinie3d(ruv, ruh);

}		    




/*****************************************************************************
 *  Funktion:       void ProjektionZylinder(TReal radius, TReal hoehe)
 *  Parameter:      radius:  Radius des Zylinders
 *                  hoehe:   Hoehe des Zylinders
 *                            
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   winmaxx: Breite des Fensters (fuer Clipping auf 0..winmaxx)
 *                  winmaxy: Hoehe des Windows (fuer Clipping auf 0..winmaxy).
 *                  modus:   Anzeigemodus (Bitvektor, siehe AnzeigeAnimation())
 *                                relevant hier: ##### noch nichts #####
 *
 *  Beschreibung:
 *  -------------
 *  Den Zylinder mit Radius radius und Hoehe hoehe auf die Pixmap mit der Groesse
 *  0..winmaxx, 0..winmaxy projizieren. Dazu werden ANIM_ZYLPOINTS Punkte auf dem
 *  Rand der beiden Kreis-Grundflaechen berechnet. Dann werden die entsprechenden
 *  Verbindungslinien zwischen diesen Punkten berechnet und gezeichnet. 
 *  Um die Randlinie der Hoehe zu bekommen, wird der maximale x-y-Abstand der
 *  projizierten Randpunkte zum projizierten Mittelpunkt berechnet. Der
 *  groesste Abstand ist der am weitesten aussen liegende Punkt und damit
 *  kann dort die Silhouettenlinie angesetzt werden. Da der 2. Punkt
 *  diametral gegenueberliegt, braucht er nicht gesucht zu werden, sondern
 *  laesst sich durch Addition von ANIM_ZYLPOINTS/2 zum entsprechenden
 *  Index direkt bestimmen.
 *****************************************************************************/

#define ANIM_ZYLPOINTS 12

static void ProjektionZylinder(TReal radius, TReal hoehe)
{
     TReal winkel1, winkel2, h2, d0, d1;
     int i, j;
     TVektor p[ANIM_ZYLPOINTS];
     TVektor q[ANIM_ZYLPOINTS];
     TVektor mu;
     
#if 0
     if (modus & ObjectCoordON)
     {
	  TVektor m, xe, ye, ze;	  /* Koordinatenachsen des Koerpers */

	  TransformationPunkt(m, 0, 0, 0);
	  TransformationPunkt(xe, 0.2, 0, 0); /* Koerper-Achsen projizieren */
	  TransformationPunkt(ye, 0, 0.2, 0);
	  TransformationPunkt(ze, 0, 0, -0.2);
	  ClipLinie3d(m, xe);		  /* Koerper-Achsen clippen und */
	  ClipLinie3d(m, ye);		  /* hinzeichnen */
	  ClipLinie3d(m, ze);
     }
#endif

     winkel1 = radius * 0.8660254;	  /* sin 60 deg = cos 30 deg */
     winkel2 = radius * 0.5;		  /* sin 30 deg = cos 60 deg */
     h2 = hoehe/2;

     TransformationPunkt(mu, 0, 0, h2);

     TransformationPunkt(p[0], radius, 0, h2);
     TransformationPunkt(p[1], winkel1, winkel2, h2);
     TransformationPunkt(p[2], winkel2, winkel1, h2);
     TransformationPunkt(p[3], 0, radius, h2);
     TransformationPunkt(p[4], -winkel2, winkel1, h2);
     TransformationPunkt(p[5], -winkel1, winkel2, h2);
     TransformationPunkt(p[6], -radius, 0, h2);
     TransformationPunkt(p[7], -winkel1, -winkel2, h2);
     TransformationPunkt(p[8], -winkel2, -winkel1, h2);
     TransformationPunkt(p[9], 0, -radius, h2);
     TransformationPunkt(p[10], winkel2, -winkel1, h2);
     TransformationPunkt(p[11], winkel1, -winkel2, h2);

     h2=-h2;

     TransformationPunkt(q[0], radius, 0, h2);
     TransformationPunkt(q[1], winkel1, winkel2, h2);
     TransformationPunkt(q[2], winkel2, winkel1, h2);
     TransformationPunkt(q[3], 0, radius, h2);
     TransformationPunkt(q[4], -winkel2, winkel1, h2);
     TransformationPunkt(q[5], -winkel1, winkel2, h2);
     TransformationPunkt(q[6], -radius, 0, h2);
     TransformationPunkt(q[7], -winkel1, -winkel2, h2);
     TransformationPunkt(q[8], -winkel2, -winkel1, h2);
     TransformationPunkt(q[9], 0, -radius, h2);
     TransformationPunkt(q[10], winkel2, -winkel1, h2);
     TransformationPunkt(q[11], winkel1, -winkel2, h2);

     /* oberer Kreis (eigentlich Ellipse) */
     ClipLinie3d(p[0], p[1]);
     ClipLinie3d(p[1], p[2]);
     ClipLinie3d(p[2], p[3]);
     ClipLinie3d(p[3], p[4]);
     ClipLinie3d(p[4], p[5]);
     ClipLinie3d(p[5], p[6]);
     ClipLinie3d(p[6], p[7]);
     ClipLinie3d(p[7], p[8]);
     ClipLinie3d(p[8], p[9]);
     ClipLinie3d(p[9], p[10]);
     ClipLinie3d(p[10], p[11]);
     ClipLinie3d(p[11], p[0]);
     

     /* unterer Kreis (eigentlich Ellipse) */
     ClipLinie3d(q[0], q[1]);
     ClipLinie3d(q[1], q[2]);
     ClipLinie3d(q[2], q[3]);
     ClipLinie3d(q[3], q[4]);
     ClipLinie3d(q[4], q[5]);
     ClipLinie3d(q[5], q[6]);
     ClipLinie3d(q[6], q[7]);
     ClipLinie3d(q[7], q[8]);
     ClipLinie3d(q[8], q[9]);
     ClipLinie3d(q[9], q[10]);
     ClipLinie3d(q[10], q[11]);
     ClipLinie3d(q[11], q[0]);

     /* Ansatzpunkte fuer Silhouettenlinie suchen */
     i = 0;
     d0 = QUADRAT(p[0][0]-mu[0])+QUADRAT(p[0][1]-mu[1]);

     for (j=1; j<ANIM_ZYLPOINTS/2; j++)
     {
	  d1 = QUADRAT(p[j][0]-mu[0])+QUADRAT(p[j][1]-mu[1]);
	  if (d1>d0)
	  { 
	       d0 = d1;
	       i = j;
	  }
/*	  else break; */
     }

#if 0
#ifdef ADEBUG
     printf("ProjektionZylinder: i=%d\n",i);
#endif
#endif

     /* Silhouettenlinien entlang des Zylinders */
     ClipLinie3d(p[i], q[i]);
     ClipLinie3d(p[i+ANIM_ZYLPOINTS/2], q[i+ANIM_ZYLPOINTS/2]);

}		    




/*****************************************************************************
 *  Funktion:       void ProjektionKugel(TReal radius)
 *
 *  Parameter:      radius:  Radius der Kugel
 *                            
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   winmaxx: Breite des Fensters (fuer Clipping auf 0..winmaxx)
 *                  winmaxy: Hoehe des Windows (fuer Clipping auf 0..winmaxy).
 *                  modus:   Anzeigemodus (Bitvektor, siehe AnzeigeAnimation())
 *                                relevant hier: ##### noch nichts #####
 *
 *  Beschreibung:
 *  -------------
 *  Die Kugel mit Radius radius auf die Pixmap mit Groesse 0..winmaxx, 0..winmaxy
 *  projizieren. Alles Clipping wird X ueberlassen.
 *****************************************************************************/

static void ProjektionKugel(TReal radius)
{
     TVektor m;				  /* Mittelpunkt der transf. Kugel */
     TReal   r;				  /* Radius */
     AXKoord   x,y;			  /* Window-Koordinaten des Mittelpunktes */
     TReal f;				  /* fuer perspektivische Groesse */

     TransformationPunkt(m, 0, 0, 0);

#if 0
     if (modus & ObjectCoordON)
     {
	  TVektor m, xe, ye, ze;	  /* Koordinatenachsen des Koerpers */

	  TransformationPunkt(m, 0, 0, 0);
	  TransformationPunkt(xe, 0.2, 0, 0); /* Koerper-Achsen projizieren */
	  TransformationPunkt(ye, 0, 0.2, 0);
	  TransformationPunkt(ze, 0, 0, -0.2);
	  ClipLinie3d(m, xe);		  /* Koerper-Achsen clippen und  */
	  ClipLinie3d(m, ye);		  /* hinzeichnen */
	  ClipLinie3d(m, ze);
     }
#endif

     

     r = radius;

     /* entscheiden, ob Kugel sichtbar. */

     if ((m[2]-r) > ANIM_DEFAULTABSTAND) return; /* Kugel hinter Kamera unsichtbar */

     ProjektionPunkt(m, &x, &y);

     f = ANIM_DEFAULTABSTAND;		  /* Radius entsprechend Perspektive */
     f = f/(-m[2]+f);			  /* bestimmen */
     r*= f*d_zoom;

     if ((x-r+0.5) < (TReal) AXmin) return; /* Pruefen, ob im Wertebereich */
     if ((x+r+0.5) > (TReal) AXmax) return; /* fuer X-Windows */
     if ((y-r+0.5) < (TReal) AXmin) return; 
     if ((y+r+0.5) > (TReal) AXmax) return; 

     /* Kugelkreis zeichnen */
     XDrawArc(display, pixmap, anim_GC, (AXKoord)(x-r+0.5), (AXKoord)(y-r+0.5),
	      2*r, 2*r, 0, 360*64);
}



/*****************************************************************************
 *  Funktion:       void ProjektionNagel()
 *  Parameter:      -
 *                            
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   winmaxx: Breite des Fensters (fuer Clipping auf 0..winmaxx)
 *                  winmaxy: Hoehe des Windows (fuer Clipping auf 0..winmaxy).
 *                  modus:   Anzeigemodus (Bitvektor, siehe AnzeigeAnimation())
 *                                relevant hier: ##### noch nichts #####
 *
 *  Beschreibung:
 *  -------------
 *  Der Nagel wird auf die Pixmap mit der angegebenen Groesse projiziert und dort 
 *  als kleines Kreuz gezeichnet (Kreuz-Groesse abhaengig von ANIM_NAGELGROESSE).
 *  Die Linien des Kreuzes werden durch Aufruf von ClipLinie2d() auf die Pixmap
 *  gezeichnet.
 *****************************************************************************/

static void ProjektionNagel()
{
     TVektor p;				  /* Position des transformierten Nagels */
     AXKoord   x,y;			  /* Window-Koordinaten des Nagels */
     short groesse;			  /* Ausdehnung der Linien */
     TReal f;				  /* fuer perspektivische Groesse */

     TransformationPunkt(p, 0, 0, 0);	  /* Nagelpunkt transformieren */
     if (p[2]>0) return;		  /* Punkt hinter Kamera nicht sichtbar */

     ProjektionPunkt(p, &x, &y);
     
     f = ANIM_DEFAULTABSTAND;		  /* Groesse des Kreuzes entsprechend der */
     f = f/(-p[2]+f);			  /* Perspektive bestimmen */
     groesse = ANIM_NAGELGROESSE*f*d_zoom;

     /* Linien des Kreuzes auf Windowgroesse clippen */
     ClipLinie2d(x-groesse, y-groesse,
		 x+groesse, y+groesse);
     ClipLinie2d(x+groesse, y-groesse,
		 x-groesse, y+groesse);
}



/*****************************************************************************
 *  Funktion:       void ProjektionEbene(TVektor kampos)
 *
 *  Parameter:      numpos: Darstellungsposition (siehe AnzeigeDarstellung())
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   M_t:      Matrix mit der aktuellen Koerpertransformation (zur
 *                            Berechnung des Normalenvektors der Ebene)
 *                  winmaxx:  Breite des Fensters (fuer Clipping auf 0..winmaxx)
 *                  winmaxy:  Hoehe des Windows (fuer Clipping auf 0..winmaxy).
 *                  modus:    Anzeigemodus (Bitvektor, siehe AnzeigeDarstellung())
 *                                 relevant hier: ##### noch nichts #####
 *
 *  Beschreibung:
 *  -------------
 *  Die Ebene wird als gerasterte Flaeche gezeichnet. Dazu wird der Punkt Q
 *  der Ebene bestimmt, der der Kameraposition am naechsten ist. Ausserdem
 *  wird die Ebene in ein Raster eingeteilt. Nach Transformation der
 *  Rastervektoren wird nun der Rasterschnittpunkt (auf der transformierten
 *  Ebene) gesucht, der Q am naechsten ist und von dort ausgehend entlang
 *  der beiden Rasterlinienrichtungen das Raster gezeichnet.
 *****************************************************************************/

#define E_FAKTOR 2
#define E_ANZAHL 5

static void ProjektionEbene(TVektor kampos)
{
     TVektor n;				  /* Normalenvektor der Ebene */

     TVektor p, q, dp, v;
     TVektor b1, b2;			  /* Transformierte Basisvektoren */
     TVektor p1, p2;			  /* Linienendpunkte */
     TReal lambda;			  /* fuer Punkt q gebraucht */
     TReal ss, tt;			  /* Rasterfaktoren */
     TReal s, t;			  /* deren Integers */
     short i, j;			  /* Indizes, Laufvariablen */

     TransformationPunkt(p, 0, 0, 0);	  /* Aufsetzpunkt der Ebene */

     /* Berechne Normalenvektor durch M_t[0..2][0..2]*(0,1,0), d.h. die */
     /* Positionsverschiebung M_t[0..2][3] wird nicht einberechnet. */
     /* Das Ergebnis ist einfach der Spaltenvektor M_t[0..2][1], dieser Vektor */
     /* muesste eigentlich schon normiert sein. */

     n[0] = M_t[0][1];			  /* hierbei kann es den Normalenvektor */
     n[1] = M_t[1][1];			  /* (0,0,0) automatisch nie geben (das */
     n[2] = M_t[2][1];			  /* ist wichtig fuer lambda (s.u.) */

     ANormiereVektor(n);		  /* mathematisch unnoetig, Vektor */
					  /* muesste schon vorher Laenge 1 */
					  /* haben. Nur zur Sicherheit. */

#if 0
     /* Alternative Berechnungsmethode */
     TransformationPunkt(n, 0, 1, 0);	  /* Normalenvektor der Ebene */
     n[0]-=p[0];			  /* berechnen */
     n[1]-=p[1];
     n[2]-=p[2];
     ANormiereVektor(n);
#endif

     ATransformation(dp, M_anim, kampos[0], kampos[1], kampos[2]);


#if 0
     printf("ProjektionEbene: dp=(%f,%f,%f)\n",dp[0],dp[1],dp[2]);
#endif

     lambda = ((dp[0]-p[0])*n[0]+(dp[1]-p[1])*n[1]+(dp[2]-p[2])*n[2]);

     q[0] = dp[0]-lambda*n[0];	  /* Punkt auf der Ebene, der dp am */
     q[1] = dp[1]-lambda*n[1];	  /* naechsten ist */
     q[2] = dp[2]-lambda*n[2];
     
     /* Berechnen von 2 Basisvektoren in der rotierten Ebene:
           b1 = M_t[0..2][0..2] * (1,0,0) = M_t[0..2][0]
	   b2 = M_t[0..2][0..2] * (0,0,1) = M_t[0..2][2]
	d.h. die Berechnung erfolgt aenlich der von n (s.o.) */

     b1[0] = M_t[0][0];
     b1[1] = M_t[1][0];
     b1[2] = M_t[2][0];

     b2[0] = M_t[0][2];
     b2[1] = M_t[1][2];
     b2[2] = M_t[2][2];

     ANormiereVektor(b1);		  /* eigentlich unnoetig, s.o. bei n */
     ANormiereVektor(b2);		  /* zur Sicherheit trotzdem */

#if 0
     /* alternative Berechnung von b1 und b2: */
     TransformationPunkt(b1, 1.0, 0.0, 0.0);
     TransformationPunkt(b2, 0.0, 0.0, 1.0);

     b1[0]-=p[0];
     b1[1]-=p[1];
     b1[2]-=p[2];

     b2[0]-=p[0];
     b2[1]-=p[1];
     b2[2]-=p[2];

     ANormiereVektor(b1);
     ANormiereVektor(b2);
#endif


     if (b1[2]>0)			  /* ###### falls spaeter symmetrisch um 0, */
					  /* kann dieses hier entfallen #####*/
     {
	  b1[0]=-b1[0];			  /* b1 soll nach "vorne" zeigen */
	  b1[1]=-b1[1];
	  b1[2]=-b1[2];
     }
     if (b2[2]>0)
     {
	  b2[0]=-b2[0];			  /* auch b2 soll nach "vorne" zeigen */
	  b2[1]=-b2[1];
	  b2[2]=-b2[2];
     }
	  
     b1[0] *= E_FAKTOR;
     b1[1] *= E_FAKTOR;
     b1[2] *= E_FAKTOR;

     b2[0] *= E_FAKTOR;
     b2[1] *= E_FAKTOR;
     b2[2] *= E_FAKTOR;

     v[0] = q[0]-p[0];
     v[1] = q[1]-p[1];
     v[2] = q[2]-p[2];

     /* aus v = s*b1 + t*b2  die beiden Werte s und t berechnen --> lineares */
     /* Gleichungssystem*/

     if (b1[0] != 0) i=0;
     else if (b1[1] != 0) i=1;
     else i=2;

     if ((b2[2] != 0) && (i != 2)) j=2;
     else if ((b2[1] != 0) && (i != 1)) j=1;
     else j=0;

     tt = (v[j] - (v[i]*b1[j]/b1[i]))/(b2[j]-(b2[i]*b1[j]/b1[i]));
     ss = (v[i] - tt*b2[i])/b1[i];

     s = (short)(ss+0.5);
     t = (short)(tt+0.5);
     
     for (i=s-E_ANZAHL; i<=s+E_ANZAHL; i++) /* Raster in Richtung b1 */
     {
	  p1[0] = p[0] + i*b1[0] + (t-E_ANZAHL)*b2[0];
	  p1[1] = p[1] + i*b1[1] + (t-E_ANZAHL)*b2[1];
	  p1[2] = p[2] + i*b1[2] + (t-E_ANZAHL)*b2[2];

	  p2[0] = p1[0] + E_ANZAHL*2*b2[0];
	  p2[1] = p1[1] + E_ANZAHL*2*b2[1];
	  p2[2] = p1[2] + E_ANZAHL*2*b2[2];

	  ClipLinie3d(p1, p2);
     }

     for (j=t-E_ANZAHL; j<=t+E_ANZAHL; j++) /* Raster in Richtung b2 */
     {				
	  p1[0] = p[0] + (s-E_ANZAHL)*b1[0] + j*b2[0];
	  p1[1] = p[1] + (s-E_ANZAHL)*b1[1] + j*b2[1];
	  p1[2] = p[2] + (s-E_ANZAHL)*b1[2] + j*b2[2];

	  p2[0] = p1[0] + E_ANZAHL*2*b1[0];
	  p2[1] = p1[1] + E_ANZAHL*2*b1[1];
	  p2[2] = p1[2] + E_ANZAHL*2*b1[2];

	  ClipLinie3d(p1, p2);
     }
}



/*****************************************************************************
 *  Funktion:       void ProjektionKoerper(TKoerper *koerper, TVektor kampos)
 *
 *  Parameter:      koerper: Zeiger auf die 3D-Koerper-Daten im aktuellen Zustand
 *                  kampos:  Position der Kamera (noetig fuer Objekt EBENE)
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   winmaxx: Breite des Fensters (fuer Clipping auf 0..winmaxx)
 *                  winmaxy: Hoehe des Windows (fuer Clipping auf 0..winmaxy).
 *                  modus:   Anzeigemodus (Bitvektor, siehe AnzeigeAnimation())
 *                                relevant hier: ##### noch nichts #####
 *
 *  Beschreibung:
 *  -------------
 *  Die Liste mit verzeigerten Koerpern (i.a. aus dem an AnzeigeAnimation()
 *  uebergebenen Zustand) wird mit Hilfe von TransformationPunkt() auf 2D
 *  projiziert und auf die Pixmap fuer das Animationsfenster gezeichnet.
 *****************************************************************************/

static void ProjektionKoerper(TKoerper *koerper, TVektor kampos)
{
     AMatrix M_koerper;			  /* Matrix fuer Koerperdrehung */

     do
     {

	  MatrixFromQuaternion(koerper->q, M_koerper);

	  M_koerper[0][3] = koerper->Position[0];
	  M_koerper[1][3] = koerper->Position[1];
	  M_koerper[2][3] = koerper->Position[2];

	  AMatrixMult(M_anim, M_koerper, M_t);

#if 0
#ifdef ADEBUG
	  DumpMatrix(M_t, "M_t");
#endif
#endif

	  /* Farbe fuer Koerper setzen und Farbnummer fuer spaeter merken */
	  if ((koerper->Art != ZUSGESOBJ) && (!(modus & Show3dON)))
	  {
	       SetzeFarbe(koerper->R, koerper->G, koerper->B);
	  }

	  if (!(koerper->AStatus & HiddenON))
	  {
	       if (koerper->Art == EBENE)
	       {    /* Ebenen mit gestrichelten Linien zeichnen */
		    XSetLineAttributes(display, anim_GC, liniendicke, 
				       LineDoubleDash, CapButt, JoinRound);
	       }

	       /* Falls Koerper-Koordinatenachsen gezeichnet werden sollen, 
		  diese jetzt hier berechnen */
	       if (modus & ObjectCoordON) 
	       {
		    switch (koerper->Art)
		    {
		      case KUGEL:	  /* nur bei KUGEL, QUADER, ZYLINDER  */
		      case QUADER:	  /* und Ebene sind Koordinatenachsen */
		      case EBENE:	  /* sinnvoll */
		      case ZYLINDER:
			 {
			      TVektor m, xe, ye, ze;	  
			      
			      /* Koerperschwerpunkt */
			      TransformationPunkt(m, 0, 0, 0);

			      /* Nacheinander die 3 Achsenrichungen */
			      TransformationPunkt(xe, 0.2, 0, 0); 
			      TransformationPunkt(ye, 0, 0.2, 0);
			      TransformationPunkt(ze, 0, 0, -0.2);

			      /* Koerper-Achsen clippen und hinzeichnen */
			      ClipLinie3d(m, xe);
			      ClipLinie3d(m, ye);
			      ClipLinie3d(m, ze);
			 }
			 break;
		      default:
			 break;
		    }  /* switch (koerper->Art) */
	       }  /* if (modus & ObjectCoordON) */


	       switch (koerper->Art)
	       {
		 case KUGEL:
		    ProjektionKugel(koerper->Form.Kugel.Radius);
		    break;
		    
		 case QUADER:
		    ProjektionQuader(koerper->Form.Quader.KantenLaenge);
		    break;
		    
		 case ZYLINDER:
		    ProjektionZylinder(koerper->Form.Zylinder.Radius, 
				       koerper->Form.Zylinder.Hoehe);
		    break;
		
		 case EBENE:
		    ProjektionEbene(kampos);
		    break;
		    
    
		 case MPUNKT:		  /* wie Nagel */
		 case PUNKT:		  /* wie Nagel */
		 case NAGEL:
		    ProjektionNagel();
		    break;
		    
		 case ZUSGESOBJ:
		    ProjektionKoerper(koerper->Form.ZusGesObj.KoerperListe, kampos);
		    break;		  /* rekursiver Aufruf fuer Teilkoerper */

		 default: 
		    break;
	       } /* switch (koerper->Art) */

	       if (koerper->Art == EBENE)
	       {    /* wieder normale, durchgezogene Linien */
		    XSetLineAttributes(display, anim_GC, liniendicke, LineSolid, 
				       CapButt, JoinRound);
	       }
	       

	  } /* if (!(koerper->AStatus & HiddenON)) */

	  koerper = koerper->Naechster;

     } while (koerper!=NULL);

}



/*****************************************************************************
 *  Funktion:       void ProjektionStange(TVektor p1, TVektor p2)
 *
 *  Parameter:      p1, p2:  3D-Koordinaten der beiden Stangenendpunkte
 *                            
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   
 *
 *  Beschreibung:
 *  -------------
 *  Die Verbindung der beiden Punkte wird als Linie gezeichnet.
 *****************************************************************************/

static void ProjektionStange(TVektor p1, TVektor p2)
{
     ClipLinie3d(p1, p2);		  /* Verbindungslinie zeichnen */
}




/*****************************************************************************
 *  Funktion:       void ProjektionFeder(x1,y1,x2,y2, TReal laenge, 
 *                                                    TReal federkonst)
 *
 *  Parameter:      x1, y1,
 *                  x2, y2:  2D-Koordinaten der beiden Stangenendpunkte
 *                  laenge:  Ruhelaenge der Feder
 *                  federkonst: Federkonstante der Feder
 *                     
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   -
 *
 *  Beschreibung:
 *  -------------
 *  Es wird eine Spiralfeder dreidimensional hingezeichnet. Die Anzahl der
 *  Windungen ist abhaengig von der Ruhelaenge der Feder.
 *  Die Feder wird in einer Defaultposition definiert und dann durch eine
 *  Matrix an die entsprechende Verbindungsposition im Raum abgebildet.
 *****************************************************************************/

#define WindungenProM 40
#define PunkteProWindung 12
#define FR 0.03

/* Punkte fuer eine Windung bei 12 Punkten pro Windung: */
#define COS00 1
#define COS30 0.8660254
#define COS60 0.5
#define COS90 0
#define C00  1
#define C30  0.8660254
#define C60  0.5
#define C90  0
#define S00  C90
#define S30  C60
#define S60  C30
#define S90  C00

static TReal windungx[PunkteProWindung] = 
                {C00, C30, C60, C90, -C60, -C30, -C00, -C30, -C60, C90, C60, C30};
static TReal windungy[PunkteProWindung] = 
                {S00, S30, S60, S90, S60, S30, S00, -S30, -S60, -S90, -S60, -S30};

static void ProjektionFeder(TVektor p1, TVektor p2, TReal flaenge, TReal federkonst)
{
     AMatrix M_feder;			  /* Transformation fuer die Feder */
     TVektor palt, pneu;		  /* Linienenpunkte */
     short i,j;				  /* Laufvariablen */
     short windungen;			  /* Anzahl Windungen der Feder */

     /* Matrix fuer Transformation der Feder in den Raum berechnen */
     if (MatrixVon2Punkten(p1, p2, M_feder)) return; /* Laenge=0? */

     windungen = (short) (flaenge*WindungenProM);

     ATransformation(pneu, M_feder, 0.0, 0.0, 0.1); /* Anfangsstueck der Feder */
     ClipLinie3d(p1, pneu);
     
     for (i=0; i<windungen; i++)	  /* nacheinander alle Windungen */
     {
	  for (j=0; j<PunkteProWindung; j++)
	  {
	       palt[0] = pneu[0];
	       palt[1] = pneu[1];
	       palt[2] = pneu[2];

	       ATransformation(pneu, M_feder, windungx[j]*FR, windungy[j]*FR, 
			   0.1+i*0.8/windungen+j*0.8/(windungen*PunkteProWindung));
	       ClipLinie3d(palt, pneu);
	  }
     }
     ATransformation(palt, M_feder, C00*FR, S00*FR, 0.9); /* letzte Windung fertig */
     ClipLinie3d(pneu, palt);		  
     ATransformation(pneu, M_feder, 0.0, 0.0, 0.9); /* Zum Mittelpunkt */
     ClipLinie3d(palt, pneu);
     ClipLinie3d(pneu, p2);		  /* Endstueck der Feder */
}




/*****************************************************************************
 *  Funktion:       void ProjektionGelenk(TVektor p1, TVektor p2)
 *
 *  Parameter:      p1, p2:  3D-Koordinaten der Verbindungspunkte (sollten 
 *                           eigentlich identisch sein und somit nur 1 Punkt
 *                           darstellen)
 *                     
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   
 *
 *  Beschreibung:
 *  -------------
 *  Der Verbindungspunkt (hoffentlich sind beide Punkte auf den Koerpern
 *  identisch) wird als etwas dickerer Punkt hingezeichnet.
 *****************************************************************************/

static void ProjektionGelenk(TVektor p1, TVektor p2)
{
     AXKoord x1,y1,x2,y2;		  /* 2D-Koordinaten der Gelenkpunkte */
     short r1,r2;			  /* Radien der Gelenkpunkte */
     TReal f;

     r1 = 1;	
     f = ANIM_DEFAULTABSTAND;		  /* Groesse des Punktes entsprechend der */
     f = f/(-p1[2]+f);			  /* Perspektive bestimmen */
     r1 *= f; 

     r2 = 1;	
     f = ANIM_DEFAULTABSTAND;		  /* Groesse des Punktes entsprechend der */
     f = f/(-p2[2]+f);			  /* Perspektive bestimmen */
     r2 *= f; 

     ProjektionPunkt(p1, &x1, &y1);
     ProjektionPunkt(p2, &x2, &y2);
     XFillArc(display, window_a, anim_GC, x1-r1, y1-r1, 2*r1+1, 2*r1+1, 0, 360*64);
     XFillArc(display, window_a, anim_GC, x2-r2, y2-r2, 2*r2+1, 2*r2+1, 0, 360*64);
     ClipLinie3d(p1, p2);
}




/*****************************************************************************
 *  Funktion:       void ProjektionDaempfer(x1,y1,x2,y2)
 *
 *  Parameter:      x1, y1,
 *                  x2, y2:  2D-Koordinaten der beiden Daempferendpunkte
 *                     
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   
 *
 *  Beschreibung:
 *  -------------
 *  Es wird ein Daempfer in Form eines Zylinders dreidimensional
 *  hingezeichnet.
 *  Der Zylinder wird in einer Defaultposition definiert und dann durch eine
 *  Matrix an die entsprechende Verbindungsposition im Raum abgebildet.
 *****************************************************************************/

#define FD 0.03				  /* Radius des Daempfer-Zylinders */

static void ProjektionDaempfer(TVektor p1, TVektor p2)
{
     AMatrix M_daempfer;		  /* Transformation fuer den Daempfer */
     TVektor pa[PunkteProWindung],
             pe[PunkteProWindung];
     TVektor p;				  
     short j;				  /* Laufvariable */

     /* Matrix fuer Transformation des Daempfers in den Raum berechnen */
     if (MatrixVon2Punkten(p1, p2, M_daempfer)) return; /* Laenge=0? */

     ATransformation(p, M_daempfer, 0.0, 0.0, 0.2); /* Anfangsstueck */
     ClipLinie3d(p1, p);
     
     for (j=0; j<PunkteProWindung; j++)
     {
	  ATransformation(pa[j], M_daempfer, windungx[j]*FD, windungy[j]*FD, 0.2);
	  ATransformation(pe[j], M_daempfer, windungx[j]*FD, windungy[j]*FD, 0.8);
	  ClipLinie3d(pa[j], pe[j]);
     }
     for (j=0; j<PunkteProWindung; j++)
     {
	  ClipLinie3d(pa[j], pa[(j+1)%PunkteProWindung]);
	  ClipLinie3d(pe[j], pe[(j+1)%PunkteProWindung]);
     }

     ATransformation(p, M_daempfer, 0.0, 0.0, 0.8); /* Zum Mittelpunkt */
     ClipLinie3d(p, p2);		  /* Endstueck */
}



/*****************************************************************************
 *  Funktion:       void ProjektionVerbindungen(TVerbindung *verbindung)
 *
 *  Parameter:      verbindung: Zeiger auf die 3D-Verbindungs-Daten im aktuellen
 *                              Zustand
 *
 *  Import. Bez.:   aktuell: Aktuelles Element in der Animationsliste. Anstatt
 *                           Linien wird das umschreibende Quadrat in die Liste
 *                           eingetragen.
 *                  winmaxx: Breite des Fensters (fuer Clipping auf 0..winmaxx)
 *                  winmaxy: Hoehe des Windows (fuer Clipping auf 0..winmaxy).
 *                  modus:   Anzeigemodus (Bitvektor, siehe AnzeigeAnimation())
 *                                relevant hier: ##### noch nichts #####
 *
 *                            
 *  Rueckgabewert:  -
 *
 *  Beschreibung:
 *  -------------
 *  Die Liste mit verzeigerten Verbindungen (i.a. aus dem an AnzeigeAnimation()
 *  uebergebenen Zustand) wird mit Hilfe von TransformationPunkt() auf 2D
 *  projiziert und in die Pixmap der Animation gezeichnet.
 *****************************************************************************/

static void ProjektionVerbindungen(TVerbindung *verbindung)
{
     AMatrix M_verb;			  /* Matrix fuer Verbindungsdrehung */
     TVektor p1, p2;			  /* 3D-Koordinaten der Verbindungspunkte */

     /* Farbe fuer Verbindungen setzen und Farbnummer fuer spaeter merken */
     if (!(modus & Show3dON))
	  SetzeFarbe(VERBINDUNG_R, VERBINDUNG_G, VERBINDUNG_B);

     do
     {
	  MatrixFromQuaternion(verbindung->Koerper1->q, M_verb);
	  
	  M_verb[0][3] = verbindung->Koerper1->Position[0];
	  M_verb[1][3] = verbindung->Koerper1->Position[1];
	  M_verb[2][3] = verbindung->Koerper1->Position[2];
	  
	  AMatrixMult(M_anim, M_verb, M_t);
	  
	  TransformationPunkt(p1, verbindung->VPunkt1[0], verbindung->VPunkt1[1],
			      verbindung->VPunkt1[2]);
	  
	  MatrixFromQuaternion(verbindung->Koerper2->q, M_verb);
	  
	  M_verb[0][3] = verbindung->Koerper2->Position[0];
	  M_verb[1][3] = verbindung->Koerper2->Position[1];
	  M_verb[2][3] = verbindung->Koerper2->Position[2];
	  
	  AMatrixMult(M_anim, M_verb, M_t);
	  
	  TransformationPunkt(p2, verbindung->VPunkt2[0], verbindung->VPunkt2[1],
			      verbindung->VPunkt2[2]);


	  if (1)     /* (!(verbindung->AStatus & HiddenON)) #### ebenso noch */
	       /* keinen  AStatus fuer Selected und Hidden */
	  {
	       switch (verbindung->Art)
	       {
		 case FEDER:
		    ProjektionFeder(p1, p2,
				    verbindung->VerParameter.Feder.Ruhelaenge,
				    verbindung->VerParameter.Feder.Federkonstante);
		    break;
		    
		 case STANGE:
		    ProjektionStange(p1, p2);
		    break;
		    
		 case DAEMPFER:
		    ProjektionDaempfer(p1, p2);
		    break;
		    
		 case GELENK:
		    ProjektionGelenk(p1, p2);
		    break;
	       } /* switch (verbindung->Art) */
	  } /* if (1), s.o. */

	  verbindung = verbindung->Naechste;
    } while (verbindung!=NULL);

}




/*****************************************************************************
 *  Funktion:       void Projektion(TZustand *zustand, TVektor kampos)
 *  Parameter:      zustand: Abzubildender Zustand
 *                  kampos:  Kameraposition (noetig fuer Koerper EBENE)
 *                            
 *  Rueckgabewert:  -
 *
 *  Beschreibung:
 *  -------------
 *  Alle Objekte im aktuellen Zustand werden (unter Verwendung der in M_anim
 *  abgelegten Matrix) auf ein Window abgebildet, wobei die Linien auf die 
 *  Bildgroesse geclipped werden. Dabei wird in darst eine Animationsliste 
 *  aufgebaut, die genau die sichtbaren Linien in diesem Window enthaelt, d.h.
 *  alle nicht sichtbaren Linien werden auch nicht abgelegt. Diese Liste kann
 *  von der dem Window zugehoerigen Redraw-Routine direkt abgearbeitet und auf
 *  das Window gezeichnet werden, dort sind dann also keine komplizierten 
 *  Projektionen mehr noetig. Die aufgebauten Animationslisten erlauben auch
 *  eine relativ einfache Selektion von Objekten.
 *****************************************************************************/

static void Projektion(TZustand *zustand, TVektor kampos)
{
     TKoerper *koerper;
     
     if (zustand == NULL) return;	  /* Wenn Zustand leer, ist nichts zu tun */
     
     koerper = zustand->Koerper;
     if (koerper != NULL)		  /* Wenn keine Koerper existieren, dann */
					  /* existieren auch keine Verbindungen */
     {
	  /* alle Koerper auf das Window abbilden */
	  ProjektionKoerper(koerper, kampos);
	  if (zustand->Verbindungen != NULL)
	  {
	       /* alle Verbindungen auf das Window abbilden */
	       ProjektionVerbindungen(zustand->Verbindungen);
	  }
     }
     
     /* #### Fussboden projizieren #### */
}


/*****************************************************************************
 *  Funktion:       void RefreshAnimation()
 *
 *  Parameter:      -
 *
 *  Rueckgabewert:  -
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Die Pixmap, die das momentane Bild fuer das Animationswindow enthaelt,
 *  wird auf das Window kopiert und damit angezeigt. Diese Routine dient
 *  sowhl zur erstmaligen Anzeige nach AnzeigeAnimation(), als auch zum
 *  Redraw nach Expose-Events.
 *****************************************************************************/

void RefreshAnimation()
{
     XSetFunction(display, anim_GC, GXcopy);
     XCopyArea(display, pixmap, window_a, anim_GC, 0,0, winwidth, winheight, 0,0);
}


/*****************************************************************************
 *  Funktion:       void ProjektionBlickrichtung(TVektor kampos, TQuaternion kamq, 
 *                                               TZustand *zustand, int flag)
 *
 *  Parameter:      kampos:     Kameraposition
 *                  kamq:       Drehlage der Kamera
 *                  zustand:    Zeiger auf den darzustellenden Zustand
 *                  flag:       0: normale Anzeige
 *                             -1: linkes Auge  (nur gruene Linien)
 *                              1: rechtes Auge (nur rote Linien)
 *
 *  Rueckgabewert:  -
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Mit dieser Routine wird der angegebene Zustand zustand perspektivisch
 *  in die dem Animations-Window zugeordnete Pixmap projiziert. Dabei wird
 *  je nach flag eine farbige Darstellung gewaehlt oder (bei 3D-Darstellung)
 *  zwischen einem rein gruenen Bild fuer das linke Auge und einem rein
 *  roten Bild fuer das rechte Auge unterschieden. Bei der 3D-Darstellung
 *  muss diese Routine also 2x (je einmal pro Auge) aufgerufen werden.
 *****************************************************************************/

void ProjektionBlickrichtung(TVektor kampos, TQuaternion kamq, TZustand *zustand,
			     int flag)
{
     MatrixFrom_I_Quaternion(kamq, M_anim);

     M_anim[0][3] = -M_anim[0][0]*kampos[0]-M_anim[0][1]*kampos[1]
	            -M_anim[0][2]*kampos[2];
     M_anim[1][3] = -M_anim[1][0]*kampos[0]-M_anim[1][1]*kampos[1]
	            -M_anim[1][2]*kampos[2];
     M_anim[2][3] = -M_anim[2][0]*kampos[0]-M_anim[2][1]*kampos[1]
	            -M_anim[2][2]*kampos[2];
     
     if (modus & Show3dON) 
     {
	  XSetFunction(display, anim_GC, rg_funktion);
	  if (flag<0)
	       XSetForeground(display, anim_GC, farbe[gruen]);
/*	       SetzeFarbe(GRUEN3D_R, GRUEN3D_G, GRUEN3D_B); */
	  else
	       XSetForeground(display, anim_GC, farbe[rot]);
/*	       SetzeFarbe(ROT3D_R, ROT3D_G, ROT3D_B); */
     }
     else
	  XSetFunction(display, anim_GC, GXcopy);

     if (modus & WindowCoordON)
     {		
	  /* Koordinatenachsen transformieren */

	  TReal f;
	  TVektor p, pp;

#if 1
	  /* benoetigt wird die z-Koordinate des Koordinatenursprungs im Bild,
             also relativ zur Kamera */
	  /* aus M_anim * (0,0,0,1) ergibt sich fuer die z-Koordinate: */
	  f = M_anim[2][3];

#else     /* alternative Berechnungsmethode */
	  {
	       TVektor ursprung;		  /* Koordinatenursprung */

	       ATransformation(ursprung, M_anim, 0, 0, 0);
	       f = ursprung[2];
	  }
#endif

	  if (!(modus & Show3dON))
	  {
	       SetzeFarbe(VERBINDUNG_R, VERBINDUNG_G, VERBINDUNG_B);
	  }

	  f = (ANIM_DEFAULTABSTAND-f)/(ANIM_DEFAULTABSTAND*d_zoom)*.95;
	  
#if 1     /* neue Methode, mit Clipping */

	  /* x-Achse berechnen und zeichnen */
	  ATransformation(p, M_anim, f*winwidth/2, 0, 0); /* rechts */
	  ATransformation(pp, M_anim, -f*winwidth/2, 0, 0); /* links */
	  ClipPfeil3d(pp, p, "x");

	  /* y-Achse berechnen und zeichnen */
	  ATransformation(p, M_anim, 0, f*winheight/2, 0); /* oben */
	  ATransformation(pp, M_anim, 0, -f*winheight/2, 0); /* unten */
	  ClipPfeil3d(pp, p, "y");

	  /* z-Achse berechnen und zeichnen */
	  ATransformation(pp, M_anim, 0, 0, f*winwidth/2); /* vorne */
	  ATransformation(p, M_anim, 0, 0, -f*winwidth/2); /* hinten */
	  ClipPfeil3d(pp, p, "z");


#else     /* alte Methode, ohne Clipping */

	  p[0] = ProjektionAchsenPunkt(f*winwidth/2, 0, 0); /* rechts */
	  pp   = ProjektionAchsenPunkt(f*-winwidth/2, 0, 0); /* links */
	  /* Pfeilspitze */
	  p[1] = ProjektionAchsenPunkt(f*(winwidth/2-2*ANIM_PFEILHOEHE),
				       f*(1.5*ANIM_PFEILBREITE), 0); 
	  p[2] = ProjektionAchsenPunkt(f*(winwidth/2-2*ANIM_PFEILHOEHE),
				       f*(-1.5*ANIM_PFEILBREITE), 0);
	  XDrawLine(display, pixmap, anim_pfeilGC, p[0].x, p[0].y, pp.x, pp.y);
	  XFillPolygon(display, pixmap, anim_pfeilGC, p, 3, Convex, CoordModeOrigin);
	  XDrawString(display, pixmap, anim_pfeilGC, 
		      p[0].x-XTextWidth(dfont, "x",1),
		      p[0].y+dfont->max_bounds.ascent, "x", 1);



	  p[0] = ProjektionAchsenPunkt(0, f*winheight/2, 0);      /* y-Achse */
	  pp   = ProjektionAchsenPunkt(0, f*-winheight/2, 0);
	  p[1] = ProjektionAchsenPunkt(f*(1.5*ANIM_PFEILBREITE),
				       f*(winheight/2-2*ANIM_PFEILHOEHE), 0);
	  p[2] = ProjektionAchsenPunkt(f*(-1.5*ANIM_PFEILBREITE),
				       f*(winheight/2-2*ANIM_PFEILHOEHE), 0);
	  XDrawLine(display, pixmap, anim_pfeilGC, p[0].x, p[0].y, pp.x, pp.y);
	  XFillPolygon(display, pixmap, anim_pfeilGC, p, 3, Convex, CoordModeOrigin);
	  XDrawString(display, pixmap, anim_pfeilGC, 
		      p[0].x+5, p[0].y+dfont->max_bounds.ascent, "y", 1);



	  pp   = ProjektionAchsenPunkt(0, 0, f*winwidth/2);        /* z-Achse */
	  p[0] = ProjektionAchsenPunkt(0, 0, f*-winwidth/2);
	  p[1] = ProjektionAchsenPunkt(f*(1.5*ANIM_PFEILBREITE), 0,
				       f*(-winwidth/2+2*ANIM_PFEILHOEHE));
	  p[2] = ProjektionAchsenPunkt(f*(-1.5*ANIM_PFEILBREITE), 0,
				       f*(-winwidth/2+2*ANIM_PFEILHOEHE));
	  XDrawLine(display, pixmap, anim_pfeilGC, p[0].x, p[0].y, pp.x, pp.y);
	  XFillPolygon(display, pixmap, anim_pfeilGC, p, 3, Convex, CoordModeOrigin);
	  XDrawString(display, pixmap, anim_pfeilGC,
		      p[0].x, p[0].y+dfont->max_bounds.ascent, "z", 1);
#endif

     } /* if (modus & WindowCoordON) */

     Projektion(zustand, kampos);

}

/*****************************************************************************
 *  Funktion:       void AnzeigeAnimation(Widget anim_w, TKamera kamera, 
 *                                        TZustand *zustand, int modus)
 *
 *  Parameter:      anim_w:     Widget fuer das Animationsfenster
 *                  kamera:     Position, Lage und zoom der Kamera
 *                  zustand:    Zeiger auf den darzustellenden Zustand
 *                  modus:      Art der Ausgabe (Bitvektor)
 *                              Bit 0: Fussboden zeichnen (1) oder nicht (0)
 *                              Bit 1: Koordinatenachsen der Fenster zeichnen (1)
 *                                     oder nicht (0)
 *                              Bit 2: Koordinatenachsen der einzelnen Koerper
 *                                     zeichnen (1) oder nicht (0)
 *
 *  Rueckgabewert:  -
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Mit dieser Routine wird der angegebene Zustand zustand perspektivisch
 *  auf das Animations-Window projiziert. Dabei wird alles sofort in eine
 *  Pixmap gemalt, die so gross ist wie das Window. Solange wird noch der
 *  alte Windowinhalt angezeigt. Ist die Projektion fertig, wird durch
 *  Aufruf von RefreshAnimation() die Pixmap in das Window kopiert und damit
 *  der Inhalt angezeigt. Dieses Vorgehen hat 2 Vorteile:
 *   1.) Durch die Pixmap wird der Refresh wesenlich erleichtert, da einfach
 *       nur die Pixmap erneut auf das Window kopiert werden muss (siehe
 *       auch RedrawAnzeige()).
 *   2.) Durch die Pixmap entsteht eine Art doppelter Bildpuffer. Wuerde man
 *       direkt in das Window zeichnen, muesste man es ja vorher loeschen.
 *       Dann wurden nach und nach die Linien gezeichnet. Dies wuerde die
 *       meiste Zeit als halbvolles Window dargestellt, da sich diese Dar-
 *       stellung nicht ohne weiteres mit dem Elektronenstrahl des
 *       Bildschirms synchronisieren laesst. Durch die zusaetzliche Pixmap
 *       ist die Zeit, in der das Bild unvollstaendig auf dem Bildschirm
 *       ist, minimiert worden -> klareres, nicht mehr flackerndes Bild.
 *****************************************************************************/

void AnzeigeAnimation(Widget anim_w, TKamera kamera, TZustand *zustand, 
		      int animmodus)
{ 
     d_zoom = kamera.Zoom*ANIM_ZOOM;	  /* Zoom fuer TransformationPunkt() */
					  /* setzen */
     modus = animmodus;			  /* Modus setzen (fuer Projektionen) */

     winmaxx = winwidth-1;		  /* Groesse des Windows setzen */
     winmaxy = winheight-1;

     /* alte Pixmap loeschen */
#if 0
     XSetFunction(display, anim_GC, GXcopy);
     XSetForeground(display, anim_GC, farbe[weiss]);
#endif
     XFillRectangle(display, pixmap, loesch_GC, 0,0, winwidth, winheight);


     FarbenFreigeben();			  /* alte Farben freigeben */


     if (modus & Show3dON)
     {
	  TQuaternion augenrichtung;	  /* Blickrichtung links/rechts */
	  TVektor augenposition;	  /* Augenposition links/rechts */
	  AMatrix M_auge;		  /* Matrix fuer Augenabstand */
	  TVektor rechts;		  /* Basis-Vektor fuer "rechts von Kamera" */
	  TQuaternion parallaxe_l = { 0.99999846876, 0, 0.00174999196, 0};
	  TQuaternion parallaxe_r = { 0.99999846876, 0, -0.00174999196, 0};

	  MatrixFromQuaternion(kamera.Richtung, M_auge);
	  ATransformation(rechts, M_auge, AUGENABSTAND/2, 0.0, 0.0);

	  /* Bild fuer das linke Auge berechnen */
	  augenposition[0] = kamera.Position[0]-rechts[0];
	  augenposition[1] = kamera.Position[1]-rechts[1];
	  augenposition[2] = kamera.Position[2]-rechts[2];
	  AQuaternionMult(kamera.Richtung, parallaxe_l, augenrichtung);

	  ProjektionBlickrichtung(augenposition, augenrichtung, zustand, -1);

	  /* Bild fuer das rechte Auge berechnen */
	  augenposition[0] = kamera.Position[0]+rechts[0];
	  augenposition[1] = kamera.Position[1]+rechts[1];
	  augenposition[2] = kamera.Position[2]+rechts[2];
	  AQuaternionMult(kamera.Richtung, parallaxe_r, augenrichtung);

	  ProjektionBlickrichtung(augenposition, augenrichtung, zustand, 1);
     }
     else
     {
	  ProjektionBlickrichtung(kamera.Position, kamera.Richtung, zustand, 0);
     }

     RefreshAnimation();		  /* Pixmap mit Bild auf Window kopieren */


}





/***************************************************************************
 *  Funktionen:     void RedrawAnimation(Widget w, XEvent *event, ...)
 *
 *  Parameter:      w:        Widget, fuer das der Xexpose-Event auftrat
 *                  event:    entsprechende expose-Event-Struktur
 *                  
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   -
 *
 *  Beschreibung:
 *  -------------
 *  Neuzeichnen der 4 Animationsfenster. Initiiert werden diese
 *  Routinen jeweils durch einen XExpose-event fuer das entsprechende
 *  Widget/Window. Daraufhin wird die entsprechende Animationsliste
 *  fuer das Window in das Window gemalt. Soll also etwas zu sehen
 *  sein, muessen die Animationslisten zuvor belegt worden sein
 *  (siehe AnzeigeAnimation()). Beim allerersten Redraw (nach dem
 *  Mapping der Windows) wird demnach noch nichts zu sehen sein, da
 *  dann die Listen noch leer sind.
 *  Ausser den jeweiligen Animationslisten werden auch noch die
 *  entsprechenden Koordinatenachsen hingezeichnet.
 ***************************************************************************/


static void RedrawAnimation(Widget w, XEvent *event, String *params, 
				Cardinal *num_params)
{
#if 1
#ifdef ADEBUG
     printf("RedrawAnimation()\n");
#endif
#endif

     while (XCheckWindowEvent(display, window_a, ExposureMask, event))
	  /* Tue nichts */ ;

     RefreshAnimation();

}




/***************************************************************************
 *  Funktionen:     void ConfigureAnimation(Widget w, XEvent *event, ...)
 *
 *  Parameter:      w:        Widget, fuer das der configure-Event auftrat
 *                  event:    entsprechende configure-Event-Struktur
 *                  
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   -
 *
 *  Beschreibung:
 *  -------------
 *  Wenn das Animations-Window auf dem Schirm in Groesse oder Position
 *  veraendert wird (i.a. durch den Benutzer), wird ein Configure-Event
 *  ausgeloest. Hier wird nun auf diese Groessenaenderung reagiert.
 ***************************************************************************/

static void ConfigureAnimation(Widget w, XEvent *event, String *params, 
			       Cardinal *num_params)
{
     XFreePixmap(display, pixmap);
     winwidth  = event->xconfigure.width;
     winheight = event->xconfigure.height;
     pixmap = XCreatePixmap(display, window_a, winwidth, winheight, windepth);
     XFillRectangle(display, pixmap, loesch_GC, 0,0, winwidth, winheight);
}





/***************************************************************************
 *  Funktion:       void InitialisiereAnimation(XtAppContext *app_context,
 *                                              Widget anim_w)
 *
 *  Parameter:      app_context: Toolkit-Kontext
 *                  anim_w:      Widget des Animationsfensters
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   
 *
 *  Beschreibung:
 *  -------------
 *  Setzen einiger Variablen und vor allem der noetigen lokalen
 *  Toolkit-Actions fuer das Animations-Widget. Diese Routine muss einmal zu
 *  Beginn des Programms aufgerufen werden.
 ***************************************************************************/
   
void InitialisiereAnimation(XtAppContext *app_context, Widget anim_w)
{
     static XtActionsRec actions_animation[] =
     { 
	  {"RedrawAnimation",    RedrawAnimation},
	  {"ConfigureAnimation", ConfigureAnimation}
     };
     
     String trans_animation =
	  "#override\n\
           <Expose>:	              RedrawAnimation()  \n\
           <ConfigureNotify>:         ConfigureAnimation()";


     XGCValues values;
     XColor fa[4];			  /* Farben fuer Rot-Gruen-Bild */
     Visual *visual;			  /* Default-Visual */
     int visualclass;			  /* Visual-Class des Default-Visuals */

     XtAppAddActions(*app_context, actions_animation, XtNumber(actions_animation));
     XtOverrideTranslations(anim_w, XtParseTranslationTable(trans_animation));

     display = XtDisplay(anim_w);
     screen  = DefaultScreen(display);

     visual = DefaultVisual(display, screen); 

     /* Die folgenden Befehle sind nur dazu da, auf korrekte Art und Weise */
     /* die Visual-Class des Default-Visuals zu bekommen. Rein theoretisch */
     /* wuerde: visualclass = visual->class; genuegen, jedoch darf offiziell */
     /* nicht auf die Komponenten der Visual-Struktur zugegriffen werden. */
     {
	  XVisualInfo vschablone, *vp;
	  int vanz, i;

	  /* Alle Visuals holen */
	  vp = XGetVisualInfo(display, VisualNoMask, &vschablone, &vanz);

	  /* In dieser Liste das Default-Visual suchen (muss dabei sein) */
	  for (i=0; i<vanz; i++)
	       if (vp[i].visual == visual) break;

	  visualclass = vp[i].class;
	  XFree(vp);
     }

     fa[schwarz].red = fa[gruen].red = 0;
     fa[rot].red = fa[weiss].red = 65535;
     fa[schwarz].green = fa[rot].green = 0;
     fa[gruen].green = fa[weiss].green = 65535;
     fa[schwarz].blue = fa[rot].blue = 0;
     fa[gruen].blue = fa[weiss].blue = 65535;
     fa[weiss].flags = fa[rot].flags = fa[gruen].flags = fa[schwarz].flags = 
	  DoRed | DoGreen | DoBlue;


     if ((visualclass == StaticGray) || 
	 (visualclass == TrueColor) || 
	 (visualclass == StaticColor))
     {
	  if (!(XAllocColor(display, a_colormap, &fa[weiss]) &&
		XAllocColor(display, a_colormap, &fa[rot]) &&
		XAllocColor(display, a_colormap, &fa[gruen]) &&
		XAllocColor(display, a_colormap, &fa[schwarz])))
	  {
	       fprintf(stderr, "No free read only colorcells for red-green 3D image\n");
	  }
	  farbe[weiss]   = fa[weiss].pixel;
	  farbe[rot]     = fa[rot].pixel;
	  farbe[gruen]   = fa[gruen].pixel;
	  farbe[schwarz] = fa[schwarz].pixel;
	  rg_funktion=GXand;

     }
     else
     {
	  long masks[2];

	  rg_funktion=GXor;

	  if (!(XAllocColorCells(display, a_colormap, True, masks, 2, farbe, 1)))
	  {
	       fprintf(stderr, "No free read/write colorcells for red-green
3D image\n");
	       if (!(XAllocColor(display, a_colormap, &fa[weiss]) &&
		     XAllocColor(display, a_colormap, &fa[rot]) &&
		     XAllocColor(display, a_colormap, &fa[gruen]) &&
		     XAllocColor(display, a_colormap, &fa[schwarz])))
	       {
		    fprintf(stderr, "No free read only colorcells for red-green 3D image\n");
	       }
	       farbe[weiss]   = fa[weiss].pixel;
	       farbe[rot]     = fa[rot].pixel;
	       farbe[gruen]   = fa[gruen].pixel;
	       farbe[schwarz] = fa[schwarz].pixel;
	  }

	  else
	  {
	       farbe[rot]     = farbe[weiss] | masks[0];
	       farbe[gruen]   = farbe[weiss] | masks[1];
	       farbe[schwarz] = farbe[weiss] | masks[0] | masks[1];
	       
	       fa[weiss].pixel = farbe[weiss];
	       XStoreColor(display, a_colormap, &fa[weiss]);
	       fa[rot].pixel = farbe[rot];
	       XStoreColor(display, a_colormap, &fa[rot]);
	       fa[gruen].pixel = farbe[gruen];
	       XStoreColor(display, a_colormap, &fa[gruen]);
	       fa[schwarz].pixel = farbe[schwarz];
	       XStoreColor(display, a_colormap, &fa[schwarz]);
	  }  /* if (!(XAllocColorCells(display, a_colormap, True, ...))) */
     }

     values.foreground = farbe[schwarz]; /*BlackPixel(display, screen);*/
     values.background = farbe[weiss];   /*WhitePixel(display, screen);*/
     values.fill_style = FillSolid;
     anim_GC = XtGetGC(anim_w, GCForeground | GCBackground | GCFillStyle, &values);

     anim_pfeilGC = XtGetGC(anim_w, GCForeground | GCBackground | GCFillStyle, 
			    &values);

     liniendicke = 0;

     values.foreground = farbe[weiss]; /* WhitePixel(display, screen); */
     values.background = farbe[weiss]; /* BlackPixel(display, screen); */
     values.fill_style = FillSolid;
     values.function = GXcopy;
     loesch_GC = XtGetGC(anim_w, 
			 GCForeground | GCBackground | GCFillStyle | GCFunction,
			 &values);

 


     dfont = XQueryFont(display, XGContextFromGC(anim_pfeilGC));
}




/***************************************************************************
 *  Funktion:       void OeffneAnimation(XtAppContext *app_context,
 *                                              Widget anim_w)
 *
 *  Parameter:      app_context: Toolkit-Kontext
 *                  anim_w:      Widget des Animationsfensters
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   
 *
 *  Beschreibung:
 *  -------------
 *  Jedesmal, wenn das Animationsfenster geoeffnet wird, muss diese
 *  Routine aufgerufen werden. Hier werden dann windowspezifische
 *  Dinge erfragt und die Pixmap fuer den Redraw (bzw. das
 *  Double-Buffering der Animation) eingerichtet.
 *  Da hier also Dinge ueber das Window des Widgets erfragt werden,
 *  muss das Window schon gemapped sein. 
 ***************************************************************************/

void OeffneAnimation(XtAppContext *app_context, Widget anim_w)
{ 
     XWindowAttributes getattributes;
     XSetWindowAttributes setattributes;

     window_a  = XtWindow(anim_w);
     display = XtDisplay(anim_w);
     screen  = DefaultScreen(display);

     XGetWindowAttributes(display, window_a, &getattributes);

     winwidth = getattributes.width;
     winheight = getattributes.height;
     windepth = getattributes.depth;

     setattributes.backing_store = NotUseful;
     setattributes.bit_gravity = CenterGravity; 
     XChangeWindowAttributes(display, window_a, CWBackingStore | CWBitGravity,
			     &setattributes);

     if (!habepixmap)
     {
	  habepixmap = TRUE;
	  pixmap = XCreatePixmap(display, window_a, winwidth, winheight, windepth);
     }
     XFillRectangle(display, pixmap, loesch_GC, 0,0, winwidth, winheight);
}




/***************************************************************************
 *  Funktion:       void SchliesseAnimation(XtAppContext *app_context,
 *                                              Widget anim_w)
 *
 *  Parameter:      app_context: Toolkit-Kontext
 *                  anim_w:      Widget des Animationsfensters
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   
 *
 *  Beschreibung:
 *  -------------
 *  Jedesmal, wenn das Animationsfenster geschlossen wird, sollte
 *  diese Routine aufgerufen werden. Eigentlich sollten hier die
 *  lokalen Actions aus der Translation-Table entfernt werden und bei
 *  OeffneAnimayion() jeweils wieder eingehaengt werden, aber leider
 *  ist das Loeschen einzelner Aktionen nicht so einfach, wie das
 *  Hinzufuegen. Da es auch so keine groesseren Probleme macht, ist
 *  dieser Teil der Routine hinfaellig. Ausserdem kann hier die Pixmap
 *  voruebergehend freigegeben werden, bis das Fenster das naechstemal
 *  geoeffnet wird.
 ***************************************************************************/

void SchliesseAnimation(XtAppContext *app_context, Widget anim_w)
{
     if (habepixmap) 
     {
	  XFreePixmap(display, pixmap);
	  habepixmap = FALSE;
     }
}




