/* rgba_image.c : RGBA image handling routines
**
** Written and Copyright (C) 1994, 1996 by Michael J. Gourlay
**
** PROVIDED AS IS.  NO WARRANTEES, EXPRESS OR IMPLIED.
*/

#include "my_malloc.h"

#include "tga.h"

#include "rgba_image.h"

#define MAX(x,y) ((x)>(y) ? (x) : (y))




/* --------------------------------------------------------------- */
/* initImageRGBA: initialize a new image members
*/
void
initImageRGBA(RgbaImageT *imgP)
{
  imgP->nrows = 0;
  imgP->ncols = 0;
  imgP->compressed = 0;
  imgP->pixel_size = 0;
  imgP->color_mapped = 0;
  imgP->type = 0;
  imgP->ri = NULL;
  imgP->gi = NULL;
  imgP->bi = NULL;
  imgP->ai = NULL;
}




/* --------------------------------------------------------------- */

/* freeImageRGBA: free memory of the img
*/
void
freeImageRGBA(RgbaImageT *imgP)
{
  if(imgP->ri != NULL) {
    my_free(imgP->ri, "freeImageRGBA");
    imgP->ri = NULL;
    imgP->gi = NULL;
    imgP->bi = NULL;
    imgP->ai = NULL;
  }
}




/* --------------------------------------------------------------- */

/* allocImageRGBA: allocate memory for the img
**   proc: name of the calling procedure (for debugging)
**
**   return -1 if failed, 0 otherwize
*/
int
allocImageRGBA(RgbaImageT *imgP, char *proc)
{
  /* Use only one allocation to ensure that the image data is
  ** contiguous.  This makes it easier to use other image format schemes
  ** which have parameters such as "pitch" which is the address
  ** difference between two vertically adjacent pixels, and "offset[3]"
  ** which has the offsets from the address of a pixel to the addresses
  ** of the bytes containing red, green, and blue components.  I.e. some
  ** formats can use either XY interleaving or Z stacking, just by
  ** altering these parameters.
  ** -- 02aug96 MJG
  */
  if((imgP->ri=MY_CALLOC(imgP->ncols * imgP->nrows * 4,
                                   UCHAR, "allocImageRGBA")) == NULL)
  {
    fprintf(stderr, "%s: Bad Alloc\n", proc);
    return(-1);
  }

  imgP->gi = & ( imgP->ri[imgP->ncols * imgP->nrows * 1] );
  imgP->bi = & ( imgP->ri[imgP->ncols * imgP->nrows * 2] );
  imgP->ai = & ( imgP->ri[imgP->ncols * imgP->nrows * 3] );

  return(0);
}




/* --------------------------------------------------------------- */

/* loadImageRGBA: load image into memory.
** -- Frees old image space.
** -- Allocates new image space.
*/
int
loadImageRGBA(char *fn, RgbaImageT *imgP)
{
  int           tga_return;
  tga_hdr_t     tga_hdr;
  FILE         *infP=NULL;

  /* Open the input file for binary reading */
  if(fn!=NULL && (infP=fopen(fn, "rb"))==NULL) {
    fprintf(stderr, "loadImageRGBA: could not open '%s' for input\n", fn);
    return(-1);
  }

  /* Load the image header */
    /* Targa */
    if(tga_return = load_tga_header(&tga_hdr, imgP, infP)) {
      fprintf(stderr, "load_tga_header returned %i\n", tga_return);
      return(tga_return);
    }

  /* Free the memory for the previous image planes */
  freeImageRGBA(imgP);

  /* Allocate new memory for the new image planes */
  if(allocImageRGBA(imgP, "loadImageRGBA")) return(-1);

  /* Load the new image */
    /* Targa */
    load_tga_image(&tga_hdr, imgP, infP);

  /* Close the input file */
  fclose(infP);

  return(0);
}




/* --------------------------------------------------------------- */

/* saveImageRGBA: dissolve 2 images and save dissolved image to file
** dimensions of the output image are the same as the srcP.
*/
int
saveImageRGBA(char *fn, RgbaImageT *siP, RgbaImageT *diP, double t)
{
  register int   nx, ny;
  register int   xi, yi;
  register int   rsi, gsi, bsi, asi;
  register int   rdi, gdi, bdi, adi;
  int            tga_return;
  RgbaImageT   img;
  tga_hdr_t      tga_hdr;
  FILE          *outfP=NULL;

  /* Initialize the info for the saved image */
  ny = img.nrows = siP->nrows;
  nx = img.ncols = siP->ncols;

  img.compressed = img.color_mapped = 0;
  img.pixel_size = 32;

  if(siP->compressed || diP->compressed) img.compressed = 1;
  img.pixel_size = MAX(siP->pixel_size, diP->pixel_size);
  if(siP->color_mapped && diP->color_mapped) img.color_mapped = 1;

  /* Allocate space for image data */
  if(allocImageRGBA(&img, "saveImageRGBA")) return(-1);

  /* Dissolve the two images according to the dissolve parameter */
  for(yi=0; yi<ny; yi++) {
    for(xi=0; xi<nx; xi++) {
      rsi = (1.0-t) * siP->ri[yi * nx + xi];
      gsi = (1.0-t) * siP->gi[yi * nx + xi];
      bsi = (1.0-t) * siP->bi[yi * nx + xi];
      asi = (1.0-t) * siP->ai[yi * nx + xi];
      if((diP!=NULL) && (xi<diP->ncols) && (yi < diP->nrows)) {
        rdi = t * diP->ri[yi * diP->ncols + xi];
        gdi = t * diP->gi[yi * diP->ncols + xi];
        bdi = t * diP->bi[yi * diP->ncols + xi];
        adi = t * diP->ai[yi * diP->ncols + xi];
      } else {
        rdi = 0;
        gdi = 0;
        bdi = 0;
        adi = 0;
      }
      img.ri[yi*nx+xi] = (int)(rsi + rdi + 0.5);
      img.gi[yi*nx+xi] = (int)(gsi + gdi + 0.5);
      img.bi[yi*nx+xi] = (int)(bsi + bdi + 0.5);
      img.ai[yi*nx+xi] = (int)(asi + adi + 0.5);
    }
  }

  /* Open the output image file for binary writing */
  if(fn!=NULL && (outfP=fopen(fn, "wb"))==NULL) {
    fprintf(stderr, "saveImageRGBA: could not open '%s' for output\n", fn);
    return(-1);
  }

  /* Set the image header */
    /* Targa */
    tga_hdr.id_len = 0;

    /* cmap_type depends on the img_type */
    tga_hdr.cmap_type = 0;

    /* img_type comes from the user */
    tga_hdr.img_type = TGA_RGB;

    if(img.compressed) tga_hdr.img_type += TGA_RLE;

    tga_hdr.cmap_index = 0;

    /* cmap_len depends on the img_type and pixel_size */
    tga_hdr.cmap_len = 0;

    /* cmap_size depends on the img_type and pixel_size */
    tga_hdr.cmap_size = 0;

    tga_hdr.x_off = 0;
    tga_hdr.y_off = 0;

    /* pixel_size depends on the img_type */
    tga_hdr.pixel_size = img.pixel_size;

    tga_hdr.att_bits = 0;
    tga_hdr.reserved = 0;
    tga_hdr.origin_bit = 0;
    tga_hdr.interleave = TGA_IL_None;

  /* Save the image header */
    /* Targa */
    if(tga_return = save_tga_header(&tga_hdr, &img, outfP)) {
      fprintf(stderr, "save_tga_header returned %i\n", tga_return);
      return(tga_return);
    }

  /* Save the dissolved image */
    /* Targa */
    save_tga_image(&tga_hdr, &img, outfP);

  /* Free the dissolved image */
  freeImageRGBA(&img);

  /* Close the output image file */
  fclose(outfP);

  return(0);
}




/* --------------------------------------------------------------- */

/* make_test_image: generate a test image
**
** Uses the incoming values of ncols and nrows to determine image size.
** If ncols or nrows are zero, default values are used instead.
**
**   type: which test image to use.
**
** Memory for the images is allocated and imgP is set.
*/
int
make_test_image(RgbaImageT *imgP, int type)
{
  register int   xi, yi;
  register UCHAR p;

  if(imgP->ncols <= 0) imgP->ncols = 320;
  if(imgP->nrows <= 0) imgP->nrows = 240;
  imgP->compressed = 1;
  imgP->color_mapped = 0;
  imgP->pixel_size = 24;
  imgP->type = TARGA_MAGIC;

  if(allocImageRGBA(imgP, "make_test_image")) return(1);

  for(yi=0; yi<imgP->nrows; yi++) {
    for(xi=0; xi<imgP->ncols; xi++) {
      p = 15 + 240*((float)xi/imgP->ncols)*((float)yi/imgP->nrows);
      if((xi%40>20 && yi%40<20) || (xi%40<20 && yi%40>20))
        p=0;

      if(type & 1) {
        imgP->ri[yi*(imgP->ncols) + xi] = p;
      } else {
        imgP->ri[yi*(imgP->ncols) + xi] = RGBA_IMAGE_MAXVAL - p;
      }

      if(type & 2) {
        imgP->gi[yi*(imgP->ncols) + xi] = p;
      } else {
        imgP->gi[yi*(imgP->ncols) + xi] = RGBA_IMAGE_MAXVAL - p;
      }

      if(type & 4) {
        imgP->bi[yi*(imgP->ncols) + xi] = p;
      } else {
        imgP->bi[yi*(imgP->ncols) + xi] = RGBA_IMAGE_MAXVAL - p;
      }

      imgP->ai[yi*(imgP->ncols) + xi] = RGBA_IMAGE_OPAQUE;
    }
  }
  return(0);
}
