#include "surface.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>


/* Multiplies two 4x4 matrices */
void matrixmultiply(matrix a, matrix b, matrix ans)
{
   int i, j, k;

   for (i = 0; i < 4; i++) {
      for (j = 0; j < 4; j++) {
         ans[i][j] = 0.0;
         for (k = 0; k < 4; k++)
            ans[i][j] += a[i][k] * b[k][j];
      }
   }
}

/* Return magnitude of vector */
FLOATTYPE magnitude(vector v)
{
   return (sqrt(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]));
}

/* Compute vector product - scalarprod is defined as macro in Surface.h */
void vecprod(vector a, vector b, vector ans)
{
   ans[0] = a[1] * b[2] - a[2] * b[1];
   ans[1] = a[2] * b[0] - a[0] * b[2];
   ans[2] = a[0] * b[1] - a[1] * b[0];
}


/* Normalise vector */
void normalise(vector v)
{
   FLOATTYPE mag;

   mag = sqrt(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);
   v[0] /= mag;
   v[1] /= mag;
   v[2] /= mag;
}


/*
 * finds nearest point to (x,y) in current patch returns TRUE if there is one
 * within sqrt(SELECTDIST)
 */
bool
nearestpoint(Position x, Position y, int *col, int *row)
{
   extern objectdata *object;

   FLOATTYPE closest = BIG, distance;
   static int i, j;
   patch *current = object->currentpatch;

   for (i = 0; i < 4; i++)
      for (j = 0; j < 4; j++) {
         distance = distsq(current->data[i][j]->screen.x, 
                           current->data[i][j]->screen.y, x, y);
         if (distance < closest) {
            (*col) = i;
            (*row) = j;
            closest = distance;
         }
      }
   if (closest < SELECTDIST)
      return TRUE;
   else
      return FALSE;
}


/*
 * Shifts the world coordinates of the point pointed at by vertexptr through
 * the vector move
 */
void movepoint(vertex * vertexptr, vector move)
{
   int i;

   for (i = 0; i < 3; i++)
      vertexptr->world[i] += move[i];
}

/* Goes through each patch, resetting the visited flag to false */
void unvisit(objectdata * object)
{
   patch *patchptr;

   patchptr = object->firstpatch;
   do {
      patchptr->visited = FALSE;
      patchptr = patchptr->next;
   } while (patchptr != NULL);
}

/* Generate error message and quit */
void error(char *message)
{
   fprintf(stderr, "%s\n", message);
   exit(1);
}

/* Transpose matrix */
void transpose(matrix operand, matrix trans)
{
   int i, j;

   for (i = 0; i < 4; i++)
      for (j = 0; j < 4; j++)
         trans[i][j] = operand[j][i];
}

/*
 * Renumber vertices in object - after deletion, numbers may not be
 * continuous
 */
void renumber(objectdata * object)
{
   vertex *vertexptr;
   patch *patchptr;
   int number = 1;

   vertexptr = object->firstvertex;
   do {
      vertexptr->number = number++;
      vertexptr = vertexptr->next;
   } while (vertexptr != NULL);

   number = 1;
   patchptr = object->firstpatch;
   do {
      patchptr->number = number++;
      patchptr = patchptr->next;
   } while (patchptr != NULL);
}

/* Beep, print message in Status window */
void printstatus(char *message)
{
   extern Display *statusdisplay;

   XBell(statusdisplay, 100);
   updatestatus(3, message);
}

/* Disconnect vertex from vertex list and delete it */
void killvertex(objectdata * object, vertex * vertexptr)
{
   /* Decrement numvertices and check for problems */
   if (--object->numvertices == 0)
      error("KillVertex: Operation results in empty vertexlist");
   /* Disconnect from vertex list */
   if (vertexptr == object->firstvertex) {
      object->firstvertex = vertexptr->next;
      vertexptr->next->previous = NULL;
   } else
      vertexptr->previous->next = vertexptr->next;
   if (vertexptr == object->lastvertex) {
      object->lastvertex = vertexptr->previous;
      vertexptr->previous->next = NULL;
   } else
      vertexptr->next->previous = vertexptr->previous;
   /* Finally... */
   free(vertexptr);
}

/* Disconnect patch from patchlist and delete it */
void deletepatch(objectdata * object, patch * kill)
{
   int i, j;

   if (object->numpatches == 1) {
      printstatus("DeletePatch: Cannot delete only patch in object.");
      return;
   }
   /* Disconnect patch */
   if (kill != object->firstpatch)
      kill->previous->next = kill->next;
   else {
      object->firstpatch = kill->next;
      kill->next->previous = NULL;
   }
   if (kill != object->lastpatch)
      kill->next->previous = kill->previous;
   else {
      object->lastpatch = kill->previous;
      kill->previous->next = NULL;
   }
   if (kill->n != NULL)
      kill->n->s = NULL;
   if (kill->s != NULL)
      kill->s->n = NULL;
   if (kill->e != NULL)
      kill->e->w = NULL;
   if (kill->w != NULL)
      kill->w->e = NULL;

   /* Decrement occurrence count for vertices, and delete if zero */
   for (i = 0; i < 4; i++)
      for (j = 0; j < 4; j++)
         if (--(kill->data[i][j]->occurrences) == 0)
            killvertex(object, kill->data[i][j]);

   /* Decrement numpatches */
   object->numpatches -= 1;

   /* If patch is the currentpatch, then invent a new one */
   if (kill == object->currentpatch)
      object->currentpatch = object->firstpatch;

   /* Free patch node */
   free(kill);
}

/* Malloc new vertex, and link it into vertexlist */
vertex *
       createvertex(objectdata * object)
{
   vertex *vertexptr;

   /* Malloc vertex */
   vertexptr = (vertex *) malloc(sizeof(vertex));
   if (vertexptr == NULL)
      error("Malloc failed in CreateVertex\n");

   /* Set variables */
   vertexptr->world[0] = 0.0;
   vertexptr->world[1] = 0.0;
   vertexptr->world[2] = 0.0;
   vertexptr->world[3] = 1.0;
   vertexptr->occurrences = 1;
   vertexptr->number = object->lastvertex->number + 1;

   /* Link into vertexlist */
   vertexptr->previous = object->lastvertex;
   object->lastvertex->next = vertexptr;
   object->lastvertex = vertexptr;
   vertexptr->next = NULL;
   object->numvertices += 1;

   return vertexptr;
}

/* Make a copy of the object - not including connectivity data */
void copyobject(objectdata * original, objectdata * copy)
{
   int i, j;
   vertex **vertices, *origvptr, *copyvptr;
   patch *origpptr, *copypptr;

   #ifdef DEBUG
   /* Do quick check */
   check(original);
   if (original->numpatches != copy->numpatches)
      error("CopyObject: Object structures have different numpatches.\n");
   if (original->numvertices != copy->numvertices)
      error("CopyObject: Object structures have different numvertices.\n");
   #endif

   /* Renumber original object - may be jumps in id numbers of vertices */
   renumber(original);

   /* Malloc array in which to store address of the vertices */
   vertices = (vertex **) malloc(original->numvertices * sizeof(vertex *));

   /* Copy vertices */
   origvptr = original->firstvertex;
   copyvptr = copy->firstvertex;
   do {
      for (i = 0; i < 3; i++)
         copyvptr->world[i] = origvptr->world[i];
      copyvptr->world[3] = 1;
      copyvptr->occurrences = origvptr->occurrences;
      copyvptr->number = origvptr->number;
      vertices[copyvptr->number - 1] = copyvptr;
      copyvptr = copyvptr->next;
   } while ((origvptr = origvptr->next) != NULL);

   /* Copy patches - don't worry about connectivity - its gotta be fast! */
   origpptr = original->firstpatch;
   copypptr = copy->firstpatch;
   do {
      for (i = 0; i < 4; i++)
         for (j = 0; j < 4; j++)
            copypptr->data[i][j] = vertices[origpptr->data[i][j]->number - 1];
      copypptr->connectpos = origpptr->connectpos;
      copypptr->number = origpptr->number;
      copypptr = copypptr->next;
   } while ((origpptr = origpptr->next) != NULL);

   /* Copy bounds and connectivity data */
   copy->connect = original->connect;
   for (i = 0; i < 3; i++) {
      copy->bounds.min[i] = original->bounds.min[i];
      copy->bounds.max[i] = original->bounds.max[i];
   }

   /* Free vertices array */
   free(vertices);
}

/*
 * Recursively traces through an object, marking patches as visited. It does
 * not trace past the current patch; this is used to detect disconnected
 * objects that may result from deletion of the current patch
 */
void tracevisit(patch * patchptr, patch * avoid)
{
   patchptr->visited = TRUE;
   if (patchptr != avoid) {
      if (patchptr->n != NULL)
         if (!patchptr->n->visited)
            tracevisit(patchptr->n, avoid);
      if (patchptr->s != NULL)
         if (!patchptr->s->visited)
            tracevisit(patchptr->s, avoid);
      if (patchptr->e != NULL)
         if (!patchptr->e->visited)
            tracevisit(patchptr->e, avoid);
      if (patchptr->w != NULL)
         if (!patchptr->w->visited)
            tracevisit(patchptr->w, avoid);
   }
}


/*
 * Recursively traces through an object, generating the coordinates used for
 * the connectivity diagram
 */
void traceconnect(patch * patchptr)
{
   patchptr->visited = TRUE;

   if (patchptr->n != NULL)
      if (!patchptr->n->visited) {
         patchptr->n->connectpos.x = patchptr->connectpos.x;
         patchptr->n->connectpos.y = patchptr->connectpos.y + 1;
         traceconnect(patchptr->n);
      }
   if (patchptr->s != NULL)
      if (!patchptr->s->visited) {
         patchptr->s->connectpos.x = patchptr->connectpos.x;
         patchptr->s->connectpos.y = patchptr->connectpos.y - 1;
         traceconnect(patchptr->s);
      }
   if (patchptr->e != NULL)
      if (!patchptr->e->visited) {
         patchptr->e->connectpos.x = patchptr->connectpos.x + 1;
         patchptr->e->connectpos.y = patchptr->connectpos.y;
         traceconnect(patchptr->e);
      }
   if (patchptr->w != NULL)
      if (!patchptr->w->visited) {
         patchptr->w->connectpos.x = patchptr->connectpos.x - 1;
         patchptr->w->connectpos.y = patchptr->connectpos.y;
         traceconnect(patchptr->w);
      }
}

/* Malloc space for a new object, creating linked vertexlist and patch */
objectdata *
           createobject(int numpatches, int numvertices)
{
   objectdata *newobjectptr;
   patch *patchptr;
   vertex *vertexptr;
   int i;

   newobjectptr = (objectdata *) malloc(sizeof(objectdata));
   if (newobjectptr == NULL)
      error("Malloc failed in CreateObject\n");

   newobjectptr->numvertices = numvertices;
   newobjectptr->numpatches = numpatches;

   /* Malloc space for the patches */
   patchptr = (patch *) malloc(sizeof(patch));
   if (patchptr == NULL)
      error("Malloc failed in CreateObject.");
   newobjectptr->firstpatch = patchptr;
   patchptr->previous = NULL;
   for (i = 1; i < numpatches; i++) {
      patchptr->n = patchptr->s = patchptr->e = patchptr->w = NULL;
      if ((patchptr->next = (patch *) malloc(sizeof(patch))) == NULL)
         error("Malloc failed in CreateObject.");
      patchptr->next->previous = patchptr;
      patchptr = patchptr->next;
   }
   newobjectptr->lastpatch = patchptr;
   patchptr->n = patchptr->s = patchptr->e = patchptr->w = NULL;
   patchptr->next = NULL;

   /* Malloc space for the vertices */
   vertexptr = (vertex *) malloc(sizeof(vertex));
   if (vertexptr == NULL)
      error("Malloc failed in CreateObject.");
   newobjectptr->firstvertex = vertexptr;
   vertexptr->previous = NULL;
   for (i = 1; i < numvertices; i++) {
      if ((vertexptr->next = (vertex *) malloc(sizeof(vertex))) == NULL)
         error("Malloc failed in CreateObject.");
      vertexptr->next->previous = vertexptr;
      vertexptr = vertexptr->next;
   }
   newobjectptr->lastvertex = vertexptr;
   vertexptr->next = NULL;

   return newobjectptr;
}

/* Finds the bounding box of the object */
void findworldbounds(objectdata * object)
{
   int i;
   vertex *vertexptr;
   vector max =
   {
      -BIG, -BIG, -BIG}, min =
   {
      BIG, BIG, BIG};

   vertexptr = object->firstvertex;
   do {
      for (i = 0; i < 3; i++) {
         min[i] = min(vertexptr->world[i], min[i]);
         max[i] = max(vertexptr->world[i], max[i]);
      }
   } while ((vertexptr = vertexptr->next) != NULL);
   for (i = 0; i < 3; i++) {
      object->bounds.min[i] = min[i];
      object->bounds.max[i] = max[i];
   }
}

/* Generates the coordinates used for the connectivity diagram */
void getconnectcoords(objectdata * object)
{
   /* Reset visited flags */
   unvisit(object);

   /* Generate connectivity diagram coords and bounds */
   object->firstpatch->connectpos.x = 0;
   object->firstpatch->connectpos.y = 0;
   traceconnect(object->firstpatch);
}

/* Generates the bounds used to scale the connectivity diagram */
void getconnectivitybounds(objectdata * object)
{
   patch *patchptr;
   int xmin = BIGINT, ymin = BIGINT, xmax = -BIGINT, ymax = -BIGINT;

   patchptr = object->firstpatch;
   do {
      xmin = min(xmin, patchptr->connectpos.x);
      xmax = max(xmax, patchptr->connectpos.x);
      ymin = min(ymin, patchptr->connectpos.y);
      ymax = max(ymax, patchptr->connectpos.y);

      patchptr = patchptr->next;
   } while (patchptr != NULL);

   object->connect.xmin = xmin;
   object->connect.xmax = xmax;
   object->connect.ymin = ymin;
   object->connect.ymax = ymax;
}
