// PVVIEWER
// Panard Vision Demo
// (C) 1996-98 Olivier Brunet (bruneto@efrei.fr)

// Here you will find source for the pvview program. This is an example
// of usage of Panard Vision (think of this like a tutorial).

// For a mirror example search for PV_CreateMirror

#include <dlfcn.h>				// this is for the handling of dynamic drivers
#include <stdio.h>
#include "pvision.h"
#include "pmotion.h"
#include "3dspm.h"
#include "3dsread.h"
#include "qkread.h"
#include "getopt.h"
#include "pvgl.h"
#include "pv3dfx.h"
#include "pvut.h"

static void fatal(char *s)
{
    printf("\n\nFATAL! : %s\n", s);
    exit(1);
}

int GetMaskSize(unsigned x)
{ 
	int i=0;
	while((!(x&1))&&(x!=0))
	{
		x>>=1;		
	}

	while(x&1)
	{
		x>>=1;
		i++;
	}
	return i;
}

int GetMaskPos(unsigned x)
{ 
	int i=0;
	while((!(x&1))&&(x!=0))
	{
		x>>=1;		
		i++;
	}

	return i;
}


int Resx = 320,Resy = 200;
int depth;

#include "xcode.inc"     // This is the X aware display code, nothing to do with PV

PVWorld *Wor;
PMotion *Anim;
PVCam *Cam;

unsigned Gl=0;
unsigned RenderMode = PVM_PALETIZED8;
unsigned TV = FALSE;
unsigned ANIMATION = FALSE;
float ANIMSPEED=1;
float STEP=30;
unsigned CACHESIZE=8*1024*1024;
char MatFile[255]="";

/***********************************************************************/

static void help()
{
    printf("\n\tPVVIEWER [file1.3DS] [file.bsp] [filex.3ds] [options]\n");
    printf("\tex: PVVIEWER -a zob.3ds -x640 zumbala.3ds -y480 -d\n");
    printf("Options:\n");
    printf("\t  -xNNN   where NNN is the horizontal resolution\n");
    printf("\t  -yNNN   where NNN is the vertical resolution\n");
    printf("\t  -r      use fake 16bits render mode.\n");
    printf("\t  -z      use RGB render mode.\n");
    printf("\t  -t      TV mode.\n");
    printf("\t  -a      Use animation info from .3ds. (MUST be first on the command line)\n");
    printf("\t  -ffile  Use file for material definitions.\n");
    printf("\t  -g     	Use hardware accelaration\n");    
    printf("\t  		-g1     	OpenGL driver\n");        
    printf("\t  		-g2     	3dfx driver\n");        
}

char optionStr[] = "hx:y:rtzaf:g:";
void parseArguments(int argc, char *argv[])
/****************************************************************************
*
* Function:     parseArguments
* Parameters:   argc    - Number of command line arguments
*               argv    - Array of command line arguments
*
* Description:  Parses the command line and forces detection of specific
*               SuperVGA's if specified.
*
****************************************************************************/
{
    int i,
     option;
    char *argument;
    /* Parse command line options */

    i = i;
    do
    {
        option = getopt2(argc, argv, optionStr, &argument);
        switch (option)
        {

        case 'x':
            Resx = atoi(argument);
            break;
        case 'y':
            Resy = atoi(argument);
            break;
        case 'r':
            RenderMode = PVM_RGB16;
            break;
        case 'z':
            RenderMode = PVM_RGB;            
            break;
        case 't':
            TV = TRUE;
            break;
        case 'a':
            ANIMATION = TRUE;
            break;
        case 'g':
            Gl=atoi(argument);
            break;
        case ALLDONE:
            break;

        case PARAMETER:
            printf("Loading %s...\n", argv[nextargv]);
            if(strstr(argv[nextargv],".bsp"))
            {
            	if(LoadMeshesFromQuake(argv[nextargv],Wor)!=COOL) fatal("Error Loading Quake map");
            }
            else
            {
            	if (LoadMeshFrom3DS(argv[nextargv], Wor) != COOL)
                	fatal("Unable to load mesh file.");
            	if (ANIMATION == TRUE)
            	{
                	printf("Loading animation info...\n");
                	if (LoadAnimFrom3DS(argv[nextargv], Anim, Wor) != COOL)
                    	fatal("Error reading animation file.");
            	}
            }
            argv++;
            break;
        case 'f':strcpy(MatFile,argument);break;            
        case 'h':
        case INVALID:
        default:
            help();
            exit(1);
        }
    }
    while (option != ALLDONE);
}
////////////////////////////////

int main(int argc, char **argv)
{
    PVMesh *o;
    char Done;
    unsigned ch;
    XEvent event;
    int pending;
    char bind[255];
    float x,y,z;
    PVPoint OldCamT;
    PVMesh *DummyCam;
    int collision=0;
    pvuMatInfo MatInfo;
    int hInst;
    
    printf("Panard-Vision Viewer X Demo Program V1.64e (C) 1997-98, SMKaribou/GMF (bruneto@efrei.fr)\n");
    printf("-h for help.\n\n");
    printf("Keys : N=Normal Mode, ,=MipMapping, J=Front2Back, F&G=Change field of view.\n");
    printf("       B=Bilinear filtering V=ZBufering X,C=Move 1 light H=Collision ON/OFF\n\n");
    printf(" Compiled on %s at %s.\n",__DATE__,__TIME__);
    printf(" PMLib (Version %s) compiled on %s at %s.\n",PMOTION_VERSION,PMOTION_DATE,PMOTION_TIME);

    InitPVision();
       
    if ((Wor = PV_CreateWorld()) == NULL) fatal("Not enough memory for PVWorld.");
    if ((Anim = PM_CreateTree()) == NULL) fatal("Not enough memory to create root node for animation.\n");
    if ((Cam = PV_CreateCam("CAMERA"))==NULL) fatal("Not enough memory for camera.\n");    

    // Need to create and assign a cam before calling LoadAnimFrom3DS()    
    Wor->Camera=Cam;

    // Analyze command line
    if (argc<2)
    {
        help();
        exit(1);
    }
    parseArguments(argc,argv);

    if(TV==FALSE)
    {
        PV_SetCamFieldOfView(Cam,Resx/Resy);
		Cam->Height=Resy;
        Cam->Width=Resx;
        Cam->CenterX=Resx/2;
        Cam->CenterY=Resy/2;
    }
    else
    {
        PV_SetCamFieldOfView(Cam,2*Resx/Resy);
		Cam->Height=Resy/2;
        Cam->Width=Resx;
        Cam->CenterX=Resx/2;
        Cam->CenterY=Resy/4;
    }
    Cam->BackDist=300000;

    // VIDEO
    if (Open_Graphics(Resx,Resy,&depth,"Panard Viewer"))
    {
        Close_Graphics();
        exit(1);
    }
    if(!Gl)
    { 
    	if((!(RenderMode&(PVM_RGB|PVM_RGB16)))&&(depth!=8)) fatal("You must specify either -r or -z in DirectColor mode.");    
    	if((RenderMode&PVM_RGB16)&&(depth!=16)) fatal("-r must only be used on 16 bits displays.");
    	if((RenderMode&PVM_RGB)&&(depth<=8)) fatal("-r must only be used on 16,24,32 bits displays.");
    }
    
    printf("Rendering in %dx%dx%dc\n",Resx,Resy,depth==32?(1<<24):1<<depth);
    
    Wor->ReservedColors = 0;
    
    // Materials defs
  	MatInfo.CACHESIZE=CACHESIZE;
    MatInfo.STEP=STEP;
    MatInfo.SPEED=ANIMSPEED;
    if(pvuLoadMaterialsDefinitions(Wor,MatFile,&MatInfo)!=COOL) fatal("Error during material loading");
    ANIMSPEED=MatInfo.SPEED;
    STEP=MatInfo.STEP;
    CACHESIZE=MatInfo.CACHESIZE;
    
    printf("Allocating %uMb texture cache...\n",CACHESIZE/(1024*1024));
    PV_InitTextureCache(CACHESIZE); // 8Mo texture cache
    
    printf("Thinking....\n");

    if(PV_SetMode(RenderMode)) fatal("Unable to set render mode.");
    if(Gl)
    {
	    PVMaterial *m;
	    
    	switch(Gl)
    	{
    		case 1:hInst=dlopen("libpvgl.so",RTLD_NOW|RTLD_GLOBAL);break;
    		case 2:hInst=dlopen("libpv3dfx.so",RTLD_NOW|RTLD_GLOBAL);break;    		
    		default:
    			fatal("Invalid driver requested");
    	}
    	if(!hInst) 
    	{
    		fputs (dlerror(), stderr);
    		fatal("Unable to load requested driver");
    	}

    	if(PV_SetHardwareDriver(dlsym(hInst, "PVDriver"))!=COOL) fatal("No hardware support.");

    	if(PV_InitAccelSupport(Win)!=COOL) fatal("Error during hardware init.");
    	if(PV_SetMode(PV_Mode|PVM_USEHARDWARE|PVM_ZBUFFER)) fatal("Unable to set hardware render mode.");
    	printf("Succesfully initialized %s.\n",PV_GetHardwareDriver()->Desc);
    	
    	// Hardware don't like states change
		PV_SetSortOrder(Wor,PVW_MATERIAL);

		// Iterates through all meshes, to make them sorted, the Quake drivers for instance
		// Sets the NO_SORT flag by default, which would anihilates the materials sorting
		o=Wor->Objs;
		while(o!=NULL)
		{
			REMOVEFLAG(o->Flags,MESH_NOSORT);
			o=o->Next;
		}

		// To prevent sorting glitches, (Quake drivers uses BSPs) enable ZBuffer		
		m=Wor->Materials;
		while(m!=NULL)
		{
			m->Type|=ZBUFFER;
			m=m->Next;
		}
    }
    RenderMode=PV_Mode;

    // Order of the next operations does matter
    // Is it the right way to setup RGB?
    {
    Visual *vsl=DefaultVisual(display,0);
    unsigned RedMask,GreenMask,BlueMask,AlphaMask;
    
    if(depth==16) 
    {
    	RedMask=vsl->red_mask;
    	GreenMask=vsl->green_mask;    	
    	BlueMask=vsl->blue_mask;
    	AlphaMask=0;
    }
    if(depth==24) 
    {
	    RedMask=vsl->red_mask;
    	GreenMask=vsl->green_mask;    	
    	BlueMask=vsl->blue_mask;
    	AlphaMask=0;    
    }
    if(depth==32) 
    {
	RedMask=vsl->red_mask;
    	GreenMask=vsl->green_mask;    	
    	BlueMask=vsl->blue_mask;
    	AlphaMask=0xFF000000;
    }
    if(depth!=8) if(PV_SetRGBIndexingMode(GetMaskSize(RedMask),GetMaskSize(GreenMask),GetMaskSize(BlueMask),GetMaskPos(RedMask),GetMaskPos(GreenMask),GetMaskPos(BlueMask),GetMaskSize(AlphaMask))!=COOL) fatal("RGB Setup : Internal Error\n");
    }
    
    if (PV_CompileMeshes(Wor) != COOL) fatal("Error during meshes compilation.");
    if (PV_CompileMaterials(Wor, NULL, NULL) != COOL) fatal("Error during materials compilation.");
    if((depth==8)&&(Wor->Global256Palette!=NULL)) Store_Colors(Wor->Global256Palette);

    // Collision Info
    printf("Computing collision infos...\n");
    o=Wor->Objs;x=y=z=0;
    while(o!=NULL)
    {
        if(o->NbrFaces!=0) if(PV_MeshComputeCollisionTree(o)!=COOL) fatal("Not enough memory for collision infos");
		printf("\t %s (%lu vertexes, %lu faces, %lu boxes)\n",o->Name,o->NbrVertexes,o->NbrFaces,o->NbrBoxes);
		x+=o->Pivot.xf; y+=o->Pivot.yf; z+=o->Pivot.zf;
        o=o->Next;
    }
	printf("World : %lu vertexes, %lu faces\n",Wor->NbrVertexes,Wor->NbrFaces);
    printf("%lu objects in the world.\n",Wor->NbrObjs);
    DummyCam=PV_CreateDummyMesh(STEP/2+5,STEP/2+5,STEP/2+5);
    if(DummyCam==NULL) fatal("Not enough memory for dummy cam.\n");
    PV_AddMesh(Wor,DummyCam);
	x/=Wor->NbrObjs; y/=Wor->NbrObjs; z/=Wor->NbrObjs;
	PV_SetCamPos(Cam,x,y,z);
    PV_SetCamAngles(Cam,0,0,0);
     
    // Setting up Clipping infos
    if (TV == FALSE)
       { if (PV_SetClipLimit(0, Resx - 1, 0, Resy - 1, Resx*(depth/8)) != COOL) fatal("Not enough memory for S-Buffer.\n");}
    else 
    	if (PV_SetClipLimit(0, Resx - 1, 0, Resy / 2, Resx*(depth/8) * 2) != COOL) fatal("Not enough memory for S-Buffer.\n");
    	
   	// Integrated Mirror tutorial :)
	// Uncomment the follwoing code to create 2 mirrors,
    // these are setup to appear with cowsol1.3ds
    /*PV_AddMirror(Wor,PV_CreateMirror());
    Wor->Mirrors->NbrVertexes=4;
    Wor->Mirrors->Vertex[0].xf=-170;
    Wor->Mirrors->Vertex[0].yf=-30;
    Wor->Mirrors->Vertex[0].zf=0;
    Wor->Mirrors->Vertex[1].xf=30;
    Wor->Mirrors->Vertex[1].yf=-30;
    Wor->Mirrors->Vertex[1].zf=0;
    Wor->Mirrors->Vertex[2].xf=30;
    Wor->Mirrors->Vertex[2].yf=120;
    Wor->Mirrors->Vertex[2].zf=0;
    Wor->Mirrors->Vertex[3].xf=-170;
    Wor->Mirrors->Vertex[3].yf=120;
    Wor->Mirrors->Vertex[3].zf=0;
    PV_SetupMirror(Wor->Mirrors);

    PV_AddMirror(Wor,PV_CreateMirror());
    Wor->Mirrors->NbrVertexes=4;
    Wor->Mirrors->Vertex[0].xf=50;
    Wor->Mirrors->Vertex[0].yf=-30;
    Wor->Mirrors->Vertex[0].zf=-30;
    Wor->Mirrors->Vertex[1].xf=50;
    Wor->Mirrors->Vertex[1].yf=-30;
    Wor->Mirrors->Vertex[1].zf=30;
    Wor->Mirrors->Vertex[2].xf=50;
    Wor->Mirrors->Vertex[2].yf=30;
    Wor->Mirrors->Vertex[2].zf=30;
    Wor->Mirrors->Vertex[3].xf=50;
    Wor->Mirrors->Vertex[3].yf=30;
    Wor->Mirrors->Vertex[3].zf=-30;
    PV_SetupMirror(Wor->Mirrors);*/
	        
    Done = 0;
    ch=1;
    do
    {
#undef CurrentTime                // ????????????????? What's That !!!
        if (ANIMATION == TRUE)
        {
            PM_SetCurrentTime(Anim, Anim->CurrentTime + 0.5);
            PM_ComputeCurrentTime(Anim);
        }

        if((ch!=0)||(ANIMATION==TRUE)){
        	
        	PV_BeginFrame();
        	PV_RenderWorld (Wor, Bitmap[0]);
        	PV_EndFrame();
        	
        	if(Gl)
        	{
     	       	PV_FlipSurface();
	        	PV_FillSurface(0,0,0,0,0);
        	}
        	else
        	{
             	Refresh ();
        		memset (Bitmap[0], 0, Resx * Resy*(depth/8));
        	}
        }
    
        OldCamT=Cam->pos;        
        pending = XPending(display);
        ch=0;
        while (pending > 0)
        {
            XNextEvent(display, &event);
            if (event.type == KeyPress)
            {
    			XLookupString(&event.xkey,bind,255,NULL, NULL);
            
			ch=bind[0];
                switch(ch)
                {
                    // +
                case '+':
                    PV_ScaleWorld(Wor, 1.1, 1.1, 1.1);
                    break;

                    // -
                case '-':
                    PV_ScaleWorld(Wor, 0.9, 0.9, 0.9);
                    break;

                    // 4
                case '4':
                    PV_SetCamPos(Cam, Cam->pos.xf - 1, Cam->pos.yf, Cam->pos.zf);
                    break;

                    // 6
                case '6':
                    PV_SetCamPos(Cam, Cam->pos.xf + 1, Cam->pos.yf, Cam->pos.zf);
                    break;

                    // 2
                case '2':
                    PV_SetCamPos(Cam, Cam->pos.xf, Cam->pos.yf + 1, Cam->pos.zf);
                    break;

                    // 8
                case '8':
                    PV_SetCamPos(Cam, Cam->pos.xf, Cam->pos.yf - 1, Cam->pos.zf);
                    break;

                    // 7
                case '7':
                    PV_CamAhead(Cam, -STEP);
                    break;

                    // 1
                case '1':
                    PV_CamAhead(Cam, STEP);
                    break;

                    // A
                case 'a':
                    PV_SetCamAngles(Cam, Cam->yaw, Cam->pitch - 0.1, Cam->roll);
                    break;

                    // Q
                case 'q':
                    PV_SetCamAngles(Cam, Cam->yaw, Cam->pitch + 0.1, Cam->roll);
                    break;

                    // O
                case 'o':
                    PV_SetCamAngles(Cam, Cam->yaw - 0.1, Cam->pitch, Cam->roll);
                    break;

                    // P
                case 'p':
                    PV_SetCamAngles(Cam, Cam->yaw + 0.1, Cam->pitch, Cam->roll);
                    break;

                    // Z
                case 'z':
                    PV_SetCamAngles(Cam, Cam->yaw, Cam->pitch, Cam->roll - 0.1);
                    break;

                    // S
                case 's':
                    PV_SetCamAngles(Cam, Cam->yaw, Cam->pitch, Cam->roll + 0.1);
                    break;

                    // N
                case 'n':
                    PV_SetMode(RenderMode);
                    PV_SetSortOrder(Wor,PVW_BACK_2_FRONT);                    
                    break;

                    // M
                case 'm':
                    PV_SetMode(PV_Mode | PVM_MIPMAPPING);
                    break;

                    // X
                case 'x':
                    PV_SetLightDirection(Wor->Lights, Wor->Lights->Direction.xf + 0.1, Wor->Lights->Direction.yf, Wor->Lights->Direction.zf);
                    break;

                    // C
                case 'c':
                    PV_SetLightDirection(Wor->Lights, Wor->Lights->Direction.xf - 0.1, Wor->Lights->Direction.yf, Wor->Lights->Direction.zf);
                    break;

                    // V
                case 'v':
				    PV_SetMode(PV_Mode|PVM_ZBUFFER);
                    break;

                    // B
                case 'b':
               	    PV_SetMode(PV_Mode|PVM_BILINEAR);
                    break; 

                    // J
                case 'j':
                    PV_SetMode(PV_Mode | PVM_SBUFFER);
                    break;
                    // F
                    
                case 'f':
                    PV_SetCamFieldOfView(Cam, Cam->fieldofview + 0.1);
                    break;
                    // G
                case 'g':
                    PV_SetCamFieldOfView(Cam, Cam->fieldofview - 0.1);
                    break;
					// H
                case 'h':
                	collision=1-collision;
                	printf("Collision state: %u\n",collision);
                	break;
                    // D
                case 'd':
                    Wor->AmbientLight.b += 0.05;
                    break;
                    // E
                case 'e':
                    Wor->AmbientLight.b -= 0.05;
                    break;
                    // ESC
                case 27:
                    Done = 1;
                default:ch=0;
                }
            }
            pending--;

            // Perform basic collision detection
    		if(collision)
    		{
        		DummyCam->Position=Cam->pos;
        		o=Wor->Objs;
        		while(o!=NULL)
        		{
            		if (PV_CollisionTest(o,DummyCam,COLLISION_NO_REPORT)==COLLIDE_YES)
            		{
                		 Cam->pos=OldCamT;
                		 break;
            		}
            		o=o->Next;
        		}
    		}

        }
    }
    while (Done == 0);

    Close_Graphics();
    
	PV_GarbageCollector(Wor);
    PV_KillWorld(Wor);
    PM_KillTree(Anim);
    PV_KillCam(Cam);                // The cam is the only object we need to destroy, others
                                    // are automagically cleaned by KillWorld (if they were
                                    // attached to the world of course.

    PV_EndAccelSupport();           // Should be the last thing done, since this destroy
                                    // all downloaded materials.
                                    // hence, PV_KillWorld() should be called before.    

    printf("\n\n(C) 1997-98, SMKaribou\n\n");
    
    return 0;
}
