// Panard Vision
// 3D Realtime Engine
// Utilities Library
// (C) 1998 Olivier Brunet
// Before using this library consult the LICENSE file

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "pvut.h"

static PVWorld *Wor;

//********************************************************************************************
//************************************** JPEG Loading
//********************************************************************************************

// JPEG
#ifdef __cplusplus
extern "C" {
#endif
#include "jpeglib.h"
#ifdef __cplusplus
}
#endif
#include <setjmp.h>

struct my_error_mgr {
  struct jpeg_error_mgr pub;	/* "public" fields */

  jmp_buf setjmp_buffer;	/* for return to caller */
};

typedef struct my_error_mgr * my_error_ptr;

static void my_error_exit (j_common_ptr cinfo)
{
  my_error_ptr myerr = (my_error_ptr) cinfo->err;

  (*cinfo->err->output_message) (cinfo);

  longjmp(myerr->setjmp_buffer, 1);
}

int PVAPI pvuLoadJpeg(char * filename,PVMaterial *m)
{
  struct jpeg_decompress_struct cinfo;
  struct my_error_mgr jerr;
  FILE * infile;		/* source file */
  char **buffer;
  char *buffer2;
  int row_stride;		/* physical row width in output buffer */
  unsigned i;

#ifdef __PVUT_VERBOSE__
  printf("Loading %s for %s...\n", filename, m->Name);
#endif

  if ((infile = fopen(filename, "rb")) == NULL)
	{
#ifdef __PVUT_VERBOSE__
		printf("Error opening %s, aborting.\n",filename);
#endif
		return FILE_IOERROR;
	}

  cinfo.err = jpeg_std_error(&jerr.pub);
  jerr.pub.error_exit = my_error_exit;
  if (setjmp(jerr.setjmp_buffer)) {
    jpeg_destroy_decompress(&cinfo);
    fclose(infile);
    return BIZAR_ERROR;
  }
  jpeg_create_decompress(&cinfo);

  jpeg_stdio_src(&cinfo, infile);

  (void) jpeg_read_header(&cinfo, 1);

  cinfo.out_color_space=JCS_RGB;

  (void) jpeg_start_decompress(&cinfo);

  row_stride = cinfo.output_width * cinfo.output_components;

  buffer=(char**)malloc(cinfo.output_height*sizeof(void*));
  if(buffer==NULL) return NO_MEMORY;

  buffer2=(char*)malloc(row_stride*cinfo.output_height);
  if(buffer2==NULL) return NO_MEMORY;

  for(i=0;i<cinfo.output_height;i++) buffer[i]=&buffer2[i*row_stride];

#ifdef __PVUT_VERBOSE__
  printf("\t %dx%d %d channels image.\n",cinfo.output_width,cinfo.output_height,cinfo.output_components);
#endif
  if(cinfo.output_components!=3) return BAD_FILE_FORMAT;

  while (cinfo.output_scanline < cinfo.output_height) {
    jpeg_read_scanlines(&cinfo, (unsigned char**)&buffer[cinfo.output_scanline],1);
  }

  (void) jpeg_finish_decompress(&cinfo);
  jpeg_destroy_decompress(&cinfo);

  fclose(infile);

  free(buffer);
  if(PV_SetMaterialTexture(m,cinfo.output_width,cinfo.output_height,(unsigned char *)buffer2,0)!=COOL) return BAD_FILE_FORMAT;
  REMOVEFLAG(m->TextureFlags,TEXTURE_PALETIZED8);
  ADDFLAG(m->TextureFlags,TEXTURE_RGB);

  return COOL;
}

//********************************************************************************************
//************************************** RAW Loading
//********************************************************************************************

static int filesize(FILE *fp)
{
    int sp,sf;

    sp=ftell(fp);
    fseek(fp,0L,SEEK_END);
    sf=ftell(fp);
    fseek(fp,sp,SEEK_SET);
    return sf;
}

int PVAPI pvuLoadRaw(char *na,PVMaterial *mat)
{
    FILE *f;
    unsigned long n;
    UPVD8 *t=NULL;
    PVRGB *pal=NULL;

    if ((f=fopen(na,"rb"))==NULL)
	{
#ifdef __PVUT_VERBOSE__
		printf("Error opening %s, aborting.\n",na);
#endif
		return FILE_IOERROR;
	}

    n=filesize(f);
#ifdef __PVUT_VERBOSE__
	printf("Loading %s(%d) for %s...\n",na,n,mat->Name);
#endif
    if ((t=(UPVD8*)malloc(n))==NULL)
    {
       fclose(f);
	   return NO_MEMORY;
    }
    fread(t,1,n,f);
    fclose(f);

    pal=(PVRGB*)malloc(sizeof(PVRGB)*256);
    memcpy(pal,&t[256*256],256*sizeof(PVRGB));
	if(PV_SetMaterialTexture(mat,256,256,t,pal)!=COOL) return BAD_FILE_FORMAT;

	ADDFLAG(mat->TextureFlags,TEXTURE_PALETIZED8);
    REMOVEFLAG(mat->TextureFlags,TEXTURE_RGB);
	return COOL;
}

int PVAPI pvuLoadBump(char *na,PVMaterial *mat)
{
    UPVD8 *t;
    FILE *f;
    unsigned n;

    if ((f=fopen(na,"rb"))==NULL)
	{
#ifdef __PVUT_VERBOSE__
		printf("Error opening %s, aborting.\n",na);
#endif
		return FILE_IOERROR;
	}
#ifdef __PVUT_VERBOSE__
	printf("Loading %s for %s's bump map...\n",na,mat->Name);
#endif
    n=filesize(f);
    if ((t=(UPVD8*)malloc(n))==NULL) return NO_MEMORY;
    fread(t,n,1,f);
    fclose(f);
    if(PV_BuildBumpMap(mat,t,256,256,1)!=COOL) return NO_MEMORY;
    free(t);
	return COOL;
}

//********************************************************************************************
//************************************** Material File
//********************************************************************************************

static char *ReadLine(FILE *f)
{
    static char b[2048];
    unsigned i=0;

    do
    {
        fscanf(f,"%s",b);
    }
    while(b[0]=='#');

    while(b[i]!=0) {if(b[i]=='$') b[i]=' ';i++;}
#ifdef __UNIX__
    i=0;
    while(b[i]!=0) {if(b[i]=='\\') b[i]='/';i++;}
#endif

    return b;
}

static int DecodeFlags(char *a2)
{
    int flags=0,i;
    
    if(a2!=NULL)
            {
                // convert to upper case
                i=0;
                while(a2[i]!=0) {if((a2[i]>=97)&&(a2[i]<=122)) a2[i]-=32;i++;}
                        
                if(strcmp("NOTHING",a2)==0) flags|=NOTHING;
				if(strcmp("FLAT",a2)==0) flags|=FLAT;
                if(strcmp("GOURAUD",a2)==0) flags|=GOURAUD;
                if(strcmp("PHONG",a2)==0) flags|=PHONG;
                if(strcmp("BUMP",a2)==0) flags|=BUMP;
                if(strcmp("MAPPING",a2)==0) flags|=MAPPING;
                if(strcmp("AMBIENT_MAPPING",a2)==0) flags|=AMBIENT_MAPPING;
                if(strcmp("AUTOMATIC_PERSPECTIVE",a2)==0) flags|=AUTOMATIC_PERSPECTIVE;
                if(strcmp("AUTOMATIC_BILINEAR",a2)==0) flags|=AUTOMATIC_BILINEAR;
                if(strcmp("PERSPECTIVE",a2)==0) flags|=PERSPECTIVE;
                if(strcmp("ZBUFFER",a2)==0) flags|=ZBUFFER;
                if(strcmp("TEXTURE_NONE",a2)==0) flags|=TEXTURE_NONE;
                if(strcmp("TEXTURE_PALETIZED8",a2)==0) flags|=TEXTURE_PALETIZED8;
                if(strcmp("TEXTURE_RGB",a2)==0) flags|=TEXTURE_RGB;
                if(strcmp("TEXTURE_MIPMAP",a2)==0) flags|=TEXTURE_MIPMAP;
                if(strcmp("TEXTURE_BILINEAR",a2)==0) flags|=TEXTURE_BILINEAR;
				if(strcmp("TEXTURE_TRILINEAR",a2)==0) flags|=TEXTURE_TRILINEAR;
				if(strcmp("DOUBLE_SIDED",a2)==0) flags|=DOUBLE_SIDED;
            }

    return flags;
}

int PVAPI pvuLoadMaterialsDefinitions(PVWorld *zeworld,char *s,pvuMatInfo *matinfo)
{
    FILE *f;
    char *l,*a1,*a2;
    PVRGBF col,col2,col3;
    unsigned i,flags1,flags2;
    PVMaterial *mat;
	int h;

	if(zeworld==NULL) return ARG_INVALID;
	if((s==NULL)||(strcmp(s,"")==0)) return COOL;

	Wor=zeworld;

	if ((f=fopen(s,"rt"))==NULL) return FILE_IOERROR;
#ifdef __PVUT_VERBOSE__
	printf("Reading %s\n",s);
#endif
    while(!feof(f))
    {
        l=ReadLine(f);

        if(strcmp(l,"END")==0) break;

		if(strcmp(l,"SPEED")==0)
		{
			l=ReadLine(f);
			if(matinfo!=NULL) sscanf(l,"%f",&matinfo->SPEED);
		}

		if(strcmp(l,"CACHESIZE")==0)
		{
			l=ReadLine(f);
			if(matinfo!=NULL) sscanf(l,"%f",&matinfo->CACHESIZE);
		}

		if(strcmp(l,"STEP")==0)
		{
			l=ReadLine(f);
			if(matinfo!=NULL) sscanf(l,"%f",&matinfo->STEP);
		}

        if(strcmp(l,"AMBIENT")==0)
        {
            l=ReadLine(f);
            sscanf(l,"%f",&col.r);
            l=ReadLine(f);
            sscanf(l,"%f",&col.g);
            l=ReadLine(f);
            sscanf(l,"%f",&col.b);
#ifdef __PVUT_VERBOSE__
            printf("Setting ambiant light to (%f,%f,%f)\n",col.r,col.g,col.b);
#endif
            PV_WorldSetAmbientLight(Wor,col);
        }

        if(strcmp(l,"MATERIAL")==0)
        {
            l=ReadLine(f);
            a1=strdup(l);

            l=ReadLine(f);
            a2=strtok(l,"|");
            flags1=DecodeFlags(a2);
            while((a2=strtok(NULL,"|"))!=NULL)
            {
                flags1|=DecodeFlags(a2);
            }

            l=ReadLine(f);
            sscanf(l,"%f",&col.r);
            l=ReadLine(f);
            sscanf(l,"%f",&col.g);
            l=ReadLine(f);
            sscanf(l,"%f",&col.b);

            l=ReadLine(f);
            sscanf(l,"%f",&col2.r);
            l=ReadLine(f);
            sscanf(l,"%f",&col2.g);
            l=ReadLine(f);
            sscanf(l,"%f",&col2.b);

            l=ReadLine(f);
            sscanf(l,"%f",&col3.r);
            l=ReadLine(f);
            sscanf(l,"%f",&col3.g);
            l=ReadLine(f);
            sscanf(l,"%f",&col3.b);

            l=ReadLine(f);
            sscanf(l,"%u",&i);

            l=ReadLine(f);
            a2=strtok(l,"|");
            flags2=DecodeFlags(a2);
            while((a2=strtok(NULL,"|"))!=NULL)
            {
                flags2|=DecodeFlags(a2);
            }
            l=ReadLine(f);

            mat=PV_CreateMaterial(a1,flags1,flags2,0);
            if(mat==NULL) return NO_MEMORY;
            PV_SetMaterialLightInfo(mat,col,col2,col3,i);

			if(flags2&(TEXTURE_RGB|TEXTURE_RGBA|TEXTURE_PALETIZED8))
			{
				if(strstr(l,".jpg")!=NULL)
				{
					h=pvuLoadJpeg(l,mat);
					if(h==COOL) PV_AddMaterial(Wor,mat);
					else return h;
				}
				else
				{
					h=pvuLoadRaw(l,mat);
					if(h==COOL) PV_AddMaterial(Wor,mat);
					else return h;
				}
			}
			else PV_AddMaterial(Wor,mat);

            l=ReadLine(f);
            if(flags1&(BUMP|U_BUMP))
			{
				h=pvuLoadBump(l,mat);
				if(h!=COOL) return h;
			}
            free(a1);
        }
    }
    fclose(f);

	return COOL;
}

////////////////////////////////////////////////////////////////////////////////////////////
//////////////////// MATERIALS

// WARNING: this routine saves a compiled material to reload it later and gain the compile time.
//			the load and save routines are highly correlated, no version checking is done.
//			Hence a material saved by version X of PV may not be readable by a version Y !
//			Error checking is very simple.
int PVAPI pvuSaveCompiledMaterial(PVWorld *Wor,char *matname,char *filename)
{
    FILE *fp;
    int i;
    PVMaterial *mat1;
    char tmp[255];

	if(Wor==NULL) return ARG_INVALID;
	if(matname==NULL) return ARG_INVALID;
	
	mat1=PV_GetMaterialPtrByName(matname,Wor);
    if(filename==NULL)
	{
		strcpy(tmp,matname);
		strcat(tmp,".mat");
	}
	else
	{
		strcpy(tmp,filename);
	}
    fp=fopen(tmp, "wb");
	if(fp==NULL) return FILE_IOERROR;

	// save the whole material structure including some unuseful infos and
	// bad pointers, but it's just one line of c :)
    fwrite(mat1,sizeof(PVMaterial),1,fp);

    if(mat1->Pal!=NULL) fwrite(mat1->Pal,768,1,fp);   // palette

    // Here is things :)
    // save number of textures in this material
    if(mat1->Tex!=NULL)
    {
        fwrite(&mat1->Tex[0],sizeof(PVTexture),1,fp);
        fwrite(mat1->Tex[0].Texture,mat1->Tex[0].Width*mat1->Tex[0].Height,1,fp);
    }

    for(i=1;i<mat1->NbrMipMaps;i++)
    {
        fwrite(&mat1->Tex[i],sizeof(PVTexture),1,fp);
        fwrite(mat1->Tex[i].Texture,mat1->Tex[i].Width*mat1->Tex[i].Height,1,fp);
    }

	// Then Save the color maps
	if(mat1->PaletteTranscodeTable!=NULL)
		fwrite(mat1->PaletteTranscodeTable,256*256,1,fp); // this one is ALWAYS
                                                          //256*256, and is used only in paletized mode (PVM_PALETIZED8)
	if(mat1->RGB16TranscodeTable!=NULL)
		fwrite(mat1->RGB16TranscodeTable,256*256*2,1,fp); // this one is ALWAYS
                                                          //256*256, but 16 bit and is only used in fake 16 bit modes (PVM_RGB16)
	fclose(fp);
	
	return COOL;
}

int PVAPI pvuLoadCompiledMaterial(PVWorld *w,char *matname,char *filename)
{
    FILE *fp;
    int i;
    PVMaterial *mat1;
    char tmp[255];

	if(w==NULL) return ARG_INVALID;
	if(matname==NULL) return ARG_INVALID;	

    mat1=PV_CreateMaterial("",0,0,0);
    if(mat1==NULL) return NO_MEMORY;

    if(filename==NULL)
	{
		strcpy(tmp,matname);
		strcat(tmp,".mat");
	}
	else
	{
		strcpy(tmp,filename);
	}
    fp=fopen(tmp, "rb");


    fread(mat1,sizeof(PVMaterial),1,fp);

    if(mat1->Pal!=NULL)
    {
        mat1->Pal=(PVRGB*)malloc(256*sizeof(PVRGB));
        fread(mat1->Pal,768,1,fp);   // palette
    }

    if(mat1->Tex!=NULL)
    {
        if(mat1->NbrMipMaps==0)
            mat1->Tex=(PVTexture*)malloc(sizeof(PVTexture));
        else
            mat1->Tex=(PVTexture*)malloc(mat1->NbrMipMaps*sizeof(PVTexture));
        fread(&mat1->Tex[0],sizeof(PVTexture),1,fp);
    }

    for(i=1;i<mat1->NbrMipMaps;i++)
    {
        fread(&mat1->Tex[i],sizeof(PVTexture),1,fp);
        fread(mat1->Tex[i].Texture,mat1->Tex[i].Width*mat1->Tex[i].Height,1,fp);
    }

	// Then Save the color maps
	if(mat1->PaletteTranscodeTable!=NULL)
    {
        mat1->PaletteTranscodeTable=(UPVD8*)malloc(256*256);
        fread(mat1->PaletteTranscodeTable,256*256,1,fp); // this one is ALWAYS
    }
                                                          //256*256, and is used only in paletized mode (PVM_PALETIZED8)
	if(mat1->RGB16TranscodeTable!=NULL)
    {
        mat1->RGB16TranscodeTable=(UPVD16*)malloc(256*256*2);
        fread(mat1->RGB16TranscodeTable,256*256*2,1,fp); // this one is ALWAYS
                                                          //256*256, but 16 bit and is only used in fake 16 bit modes (PVM_RGB16)
    }
	fclose(fp);

    mat1->RefCnt=0;
    mat1->Next=NULL;
    mat1->Name=strdup(matname);

    PV_AddMaterial(w,mat1);
	return COOL;
}



////////////////////////////////////////////////////////////////////////////////////////////
//////////////////// MISC

char *PVAPI pvuTranslateError(int e)
{
	switch(e)
	{
	case COOL:return "No error (Yes !)";
	case FILE_IOERROR:return "IO error during a file access";
	case ALREADY_ASSIGNED:return "Texture has already been set";
	case NO_MEMORY: return "Not enough memory to carry operation";
	case BIZAR_ERROR:return "Undetermined error (but there is one)";
	case ARG_INVALID:return "Invalid arguments used";
	case ITEM_NOT_FOUND:return "item not found";
	case MATERIAL_NOT_REGISTERED:return "Material not registered";
	case INVALID_CAM:return "Invalid camera";
	case INVALID_TEXTURE:return "Invalid texture";
	case NO_MATERIALS:return "No materials";
	case ALREADY_IN_HIERARCHY:return "This mesh has already been added in a hierarchy";
	case MESH_HAS_CHILD:return "This mesh cannot be added because he has child";
	case NO_ACCELSUPPORT: return "No hardware support";
	case ACCEL_NO_MEMORY: return "No more memory on hardware";
	case ACCEL_FUNCTION_UNSUPPORTED: return "Acceleration function unsupported by hardware";
	case NO_HARDWARE_SET: return "No hardware driver set by PV_SetHardwareDriver()";
	case INVALID_DRIVER: return "Wrong version of driver";
	case NO_COLLISIONINFO: return "No collision info";
	case BAD_FILE_FORMAT: return "Bad file format";
	case NOT_AVAILABLE:return "Data not available";
	default:return "Unknown error code";
	}
}
