// Panard-Vision
// 3D real time engine
// (C) 1997-98, Olivier Brunet
//
// Before using this library consult the LICENSE file


#ifndef _PVISION_H_
#define _PVISION_H_

#define __PV_VERSION__ 98         
#define PV_MN			4				  // Magic number

#include <stddef.h>						  // size_t
#include "config.h"

#if !defined(__UNIX__) && !defined(DJGPP)
#ifdef __WATCOMC__
#pragma pack(push,8)
#else
#pragma pack(push,PACK_STRUCTURE)
#endif
#endif

#ifdef __cplusplus
extern "C" {
#endif


//-------------------------------------------------------------------------------------
// ###############################################################	General settings
//-------------------------------------------------------------------------------------

// Assembly code & HLine routines calling convention is __cdecl
#ifdef  __GNUC__
#define __cdecl     // GCC doesn't know about __cdecl modifier
#endif

#ifndef inline
#define inline
#endif

#ifndef PI
#define PI 3.141592654
#endif

#if defined(_MSC_VER) || defined(__UNIX__)
#ifndef max
#define max(a,b) (a>b?a:b)
#endif

#ifndef min
#define min(a,b) (a<b?a:b)
#endif
#endif

// Float MiniMax
#ifndef MAXFLOAT
#include <float.h>
#define MINFLOAT -FLT_MAX
#define MAXFLOAT FLT_MAX
#endif

// Nothing interesting here
PVEXPORT char *PVISION_DATE;
PVEXPORT char *PVISION_TIME;
PVEXPORT char *PVISION_VERSION;

// Version management
// The old macro was not correctly expanded by some preprocessors
//#define InitPVision() 
//	InitPVisionMN(PV_MN);
#ifndef __INITPVISIONMN__
#define __INITPVISIONMN__

#ifdef __BORLANDC__
#include <float.h> 
#endif

PVEXPORT void PVAPI InitPVisionMN(int mn);
static void InitPVision(void)
{
#ifdef __BORLANDC__
	_control87(MCW_EM, MCW_EM);
#endif
	InitPVisionMN(PV_MN);
}
#endif

// Memory management
#ifdef PVMEM_OVERLOAD
#define malloc	pvmalloc
#define free	pvfree
#define calloc	pvcalloc
#endif


//-------------------------------------------------------------------------------------
// ###############################################################	Basic types
//-------------------------------------------------------------------------------------

typedef struct _PVQuat										// A Quaternion for animation
{
    float w,x,y,z;
} PVQuat;

typedef char PVD8;											// 8 bit data
typedef unsigned char UPVD8;

typedef short PVD16;										// 16 bit data
typedef unsigned short UPVD16;

typedef int PVD32;											// 32 bit data
typedef unsigned int UPVD32;

typedef UPVD32 PVFLAGS;										// Flags word

typedef float PVMat3x3[3][3];								// A 3x3 Matrix

typedef struct _PVPoint										// A PVPoint in 3d space
{
	float xf,yf,zf;
} PVPoint;

typedef struct _PVRGB										// PVRGB color (0..255)
{
   UPVD8 r,g,b;
} PVRGB;

typedef struct _PVRGBF										// RGBA color in float (0..1.0)
{
   float r,g,b,a;
} PVRGBF;

typedef struct _PVPlane										// A plane in 3d space
{
    double d;
    PVPoint Normal;
} PVPlane;

typedef struct _PVUserPlane									// Another plane in 3D space (not for user)
{
    PVPoint Normal,Pos;
    PVPlane Plane;
} PVUserPlane;

typedef struct _PVPoly										// A polygon
{
    UPVD8 NbrVertices;
    union {
	unsigned Vertice[MAX_VERTICES_PER_POLY*2];
	unsigned Vertices[MAX_VERTICES_PER_POLY*2];
	};
} PVPoly;

typedef struct _PVSpline1									// Spline Curve
{
	unsigned Order,NbrComponent;
	float *Points;
	float u1,u2;
} PVSpline1;

typedef struct _PVSpline2									// Spline Surf
{
	unsigned UOrder,VOrder,NbrComponent;
	float *Points;
	float u1,u2;
	float v1,v2;
	UPVD8 CalcNormal,AutoNormalize;
} PVSpline2;

//-------------------------------------------------------------------------------------
// ###############################################################	Collision Detection
//-------------------------------------------------------------------------------------
// PV_GetCollisionTable()

struct _PVFace;
typedef struct _PVCollidingFaces
{
    struct _PVFace *Face1;
    struct _PVFace *Face2;
} PVCollidingFaces;


//-------------------------------------------------------------------------------------
// ###############################################################	Panard Vision Working modes
//-------------------------------------------------------------------------------------
// PV_Mode, PV_SetMode()

// The following flags are used with the PV_SetMode() function to set the Panard Vision rednering
// mode. The current state is availlable via the PV_Mode variable.

#define PVM_PALETIZED8      1						// The following 3 flags are mutually exclusive
#define PVM_RGB16           2
#define PVM_RGB             4

#define PVM_SBUFFER         8						// Ignored when using hardware
#define PVM_MIPMAPPING      16
#define PVM_BILINEAR        32
#define PVM_ZBUFFER         64
#define PVM_TRILINEAR       128						// Only supported in hardware
#define PVM_ALPHABLENDING   256						// Only supported in hardware
#define PVM_ALPHATESTING	1024					// Only supported in hardware

#define PVM_USEHARDWARE		512						// Using this flag may set other PVM flags (PVM_RGB)
                                                    // and change RGB pixel indexing mode (exint PV_SetMode())

PVEXPORT PVFLAGS PV_Mode;


//-------------------------------------------------------------------------------------
// ###############################################################	Panard Vision Pipeline Control
//-------------------------------------------------------------------------------------
// PV_PipelineControl, PV_SetPipelineControl()

// The following flags are used with the PV_SetPipelineControl() function to set the Panard Vision
// pipeline mode. The current state is availlable via the PV_PipeLineControl variable.

#define PVP_DISABLELIGHTING		1					// Disable lighting stage
#define PVP_DISABLESORTING		2					// Disable sorting stage
#define PVP_DISABLETRANSFORM	4					// Disable transformation stage
													// Rotated isn't valid anymore
#define PVP_NO_ZBUFFERCLEAR     8                   // Don't clear the ZBuffer before rendering a frame
#define PVP_NO_TRIANGULATE		16					// Don't triangulate polygons before sending them to hardware driver
													// (hardware only)

PVEXPORT PVFLAGS PV_PipelineControl;


//-------------------------------------------------------------------------------------
// ###############################################################	Camera structure
//-------------------------------------------------------------------------------------

#define CAMERA_AUTOLOCATE		1				// Set this flag to allow PV
												// to find in the graph world in
												// which cell your camera is lying
												// (useful for portal rendering)

typedef struct _PVCamera
{
	char *Name;
    float fieldofview;
    unsigned Height,Width;
    float roll,pitch,yaw;
    float xscreenscale,yscreenscale,fscale;
    float CenterX,CenterY;
    PVPoint pos;
    PVPoint target;
    PVMat3x3 Matrix;                // [0][x]=vright [1][x]=vup [2][x]=vpn

    float FrontDist,BackDist;     // Position of the front and back clipping planes
    char UserPlanesEnabled[MAX_USER_CLIPPLANES];
    PVUserPlane UserPlanes[MAX_USER_CLIPPLANES];

	struct _PVMesh *StartingLocation;	 // Useful for Portal rendering
	PVFLAGS Flags;
} PVCam;


//-------------------------------------------------------------------------------------
// ###############################################################  Light Structure
//-------------------------------------------------------------------------------------

typedef enum _PVLightType {PVL_DIRECTIONAL,PVL_PARALLEL,PVL_INFINITEPOINT,PVL_POINT,PVL_SPOT,PVL_USER_LIGHT} PVLightType;

// Flags used with PVLight.Flags
#define LIGHT_NORMAL 0
#define LIGHT_FORGET 1                  // Switch off the light

typedef struct _PVLight
{
    unsigned RefCnt;
	char *Name;
    PVLightType Type;
    PVFLAGS Flags;
    PVRGBF Color;
    float Intensity;
    float Range;
    float Attenuation0,Attenuation1,Attenuation2;
    float FallOff,Theta,Phi;
    float cosTh,cosPh;
    PVPoint Position;
    PVPoint Direction;
    PVPoint Target;
    struct _PVLight *Next;
    void *UserData;
} PVLight;


//-------------------------------------------------------------------------------------
// ###############################################################  Material Structure
//-------------------------------------------------------------------------------------

// Shading Flags, Rendering Flags and Material properties are combine in PVMaterial.Type

// Shading Flags
#define NOTHING         0
#define FLAT            1
#define GOURAUD         2
#define PHONG           4
#define BUMP            8
#define U_PHONG         16
#define U_BUMP          32
#define LIGHTMAP        64                                  // Each face with this material
                                                            // must have a lightmap

#define SHADE_MASK      (1+2+4+8+16+32+64)


// Rendering Flags
#define MAPPING                     128
#define AMBIENT_MAPPING             256
#define PERSPECTIVE                 512

#define RENDER_MASK              (128+256+512)


// Material properties
#define ZBUFFER                     1024					// Used to activate ZBuffering on a material
#define AUTOMATIC_BILINEAR          2048					// Used with TEXTURE_BILINEAR
#define AUTOMATIC_PERSPECTIVE       4096					// Used with PERSPECTIVE|MAPPING
#define USER_FX                     8192					// Used to create special effects handled by user routines
#define DOUBLE_SIDED				8192*2					// Will not perform backface culling

// Material misc flags
#define COLORTABLE_DONT_FREE		(1<<31)					// The user is responsible for managing
															// colortable memory


// Texture Flags are set in PVMaterial.TextureFlags

// Texture Flags
#define TEXTURE_NONE            0
#define TEXTURE_PALETIZED8      1
#define TEXTURE_RGB             2
#define TEXTURE_RGBA			4							// This flag is only here to permit user
															// to be able to use Alpha component in their
															// own routine and in hardware. Don't use this flag 
															// in software with the common
															// PanardVision rendering routines, as it will have
															// unpredicable results (crash :).
#define TEXTURE_MIPMAP          8
#define TEXTURE_BILINEAR        16							// Only in PVM_RGB mode
#define TEXTURE_TRILINEAR       (32|TEXTURE_MIPMAP)			// Only if hardware supports it
#define TEXTURE_ALPHABLENDING   64							// Only if hardware supports it
#define TEXTURE_MANUALMIPMAPS   128                         // MipMaps will not be computed by PV but by user

// added after materials compilation
#define TEXTURE_RGBDIRECT       1024
#define TEXTURE_QUANTIZED       2048
#define TEXTURE_PROCESSED       4096

// Texture management flags
#define TEXTURE_ATEX_DONT_FREE		(1<<29)					// Don't frees Auxilliray texture (see below)
#define TEXTURE_TEX_DONT_FREE		(1<<30)					// Texture and Pal arrays will not be freed when discarded
															// useful for not malloced textures

// Texture wrapping modes (hardware only, software only supports TEXTURE_WRAP
typedef enum _PVTextureRepeat {TEXTURE_WRAP,TEXTURE_CLAMP} PVTextureRepeat;

// ZBuffer test type, (hardware only, software only supports DEPTH_LESS)
typedef enum _PVCmpTest {CMP_LESS,CMP_NEVER,CMP_EQUAL,CMP_LEQUAL,CMP_GREATER,CMP_NOTEQUAL,CMP_GEQUAL,CMP_ALWAYS} PVCmpTest;

// Alpha Blending modes, hardware only
// BLEND_ALPHA_SATURATE is only valid for source factor
typedef enum _PVBlendFactor {BLEND_ZERO,BLEND_ONE,BLEND_DST_COLOR,BLEND_ONE_MINUS_DST_COLOR,BLEND_SRC_ALPHA,BLEND_ONE_MINUS_SRC_ALPHA,BLEND_DST_ALPHA,BLEND_ONE_MINUS_DST_ALPHA,BLEND_ALPHA_SATURATE} PVBlendFactor;

// A LOD of a mipmap
typedef struct _PVTexture
{
    unsigned Width,Height;
    unsigned ShiftHeight,ShiftWidth;
    UPVD8 *Texture;
} PVTexture;

// Bump textures
typedef struct _PVBumpInfo
{
    PVD8 nx,ny;
}PVBumpInfo;

typedef struct _PVBumpMap
{
    unsigned Width,Height;
    PVBumpInfo *Map;
} PVBumpMap;

// A Texture basis for on the fly texture coordinates generation
typedef struct _PVTexBasis
{
    unsigned RefCnt;
    float SBasis[4],TBasis[4];
} PVTexBasis;

// Hint definitions
// Flags for quality hint
#define PHQ_NORMAL	0
#define PHQ_HIGH	1
#define PHQ_LOW		2

// Flags for location hint
#define PHL_NORMAL	0
#define PHL_STATIC	1
#define PHL_DYNAMIC	2

typedef struct _PVMaterialHint
{
	PVFLAGS Quality;
	PVFLAGS Location;
	PVFLAGS Misc;
} PVMaterialHint;

typedef struct _PVMaterial
{
    unsigned RefCnt;
	char *Name;
    PVFLAGS Type;

    unsigned char NbrMipMaps;
    PVFLAGS TextureFlags;
    PVRGB *Pal;
    PVTexture *Tex;
	PVTextureRepeat RepeatU,RepeatV;

	PVCmpTest DepthTest,AlphaTest;
	float AlphaReference;							// Alpha Test, hardware only

	PVBlendFactor BlendRgbSrcFactor,BlendRgbDstFactor,BlendAlphaSrcFactor,BlendAlphaDstFactor;
	float AlphaConstant;

    PVTexture AuxiliaryTexture;

    PVBumpMap BumpMap;

    UPVD8  StartingColorIndex,EndingColorIndex;           // for pure colors

    PVRGBF Diffuse,Specular,Emissive;
    unsigned SpecularPower;

    UPVD8 *PaletteTranscodeTable;
    UPVD16 *RGB16TranscodeTable;

    PVD32 UserData;														// Used to handle special FX by users routines

    struct _PVMaterial *Next;

	PVMaterialHint Hint;
} PVMaterial;

#define MAPPED_MATERIAL			 (MAPPING|PHONG|U_PHONG|BUMP|U_BUMP)	// != 0 if this material uses a texture


//-------------------------------------------------------------------------------------
// ###############################################################  Mesh
//-------------------------------------------------------------------------------------

// Flags used with Mesh.Flags
#define MESH_FORGET          1					// Hide mesh (but not his children)
#define MESH_NOSORT          2					// Copy without sorting mesh->Visible to world->Visible
#define MESH_NOLIGHTING		 4					// Used to speed up, does not perform lighting operations
												// for this mesh, DO NOT USE IT on a mesh using a shaded
												// material !!
#define MESH_VISIBLE         256				// A part of the mesh was visible last frame (internal usage)
#define MESH_FORGET_CHILD	 8					// Do not draw children of this mesh
#define MESH_INHERIT		 16					// Use Father mesh basis as a origin of the local basis
#define MESH_PORTAL			 32					// This mesh act as a portal, the first face defines
												// the portal polygon (should not be used by clients)
#define MESH_CULL_CHILD		 64					// Culls children if father mesh is culled
#define MESH_INHERIT_CULL	 128				// Inherite visibility state from the father
#define MESH_SWITCHNODE		 512				// This mesh is not a real mesh, but a switch node
												// (should not be set by clients)

#define MESH_FORGET_ALL		 (MESH_FORGET|MESH_FORGET_CHILD)	// Guess :)


// Flags used for return values of user pipelines
#define MESH_NOT_MOVED          32			// Mesh not moved on screen (internal usage)
#define MESH_RECOMPUTE_LIGHT    64			// Light should be recomputed
#define MESH_FRUSTUM_CLIPPED    128			// a part of the mesh is not totally inside frustum


// The following flags are used with each PVFace.Flags
#define TEXTURE_BASIS   (1<<29)					// This face will use basis oriented u,v generation
#define U_WRAP          (1<<30)					// Texture wraps along the S axis
#define V_WRAP          (1<<31)					// Texture wraps along the T axis

// LightMaps
// Light maps flags
#define LIGHTMAP_RGB    1                          // the lightmap is RGB 8.8.8 format
#define LIGHTMAP_MONO   2                          // the lightmap is indexed (monochromatic)

#define LIGHTMAP_PROCESSED (1<<31)                 // Sets by hardware driver
                                                   // Should not be set by users

typedef struct _PVLightMap
{
    unsigned RefCnt;
	PVFLAGS Flags;
    char Sampling;                                  // number of bytes specified in power
                                                    // of 2 (ie 4 = 16 bytes)
    unsigned NbrLightMaps,CurrentLightMap;
    unsigned Width,Height;                          // Width&Height of lightmap for mipmap0
                                                    // Filled by PV at material compilation time
	float MinU,MinV;								// Filled by PV at material compilation time
	float MaxU,MaxV;								// Filled by PV at material compilation time

	float ScaleU,ScaleV,su[MAX_LIGHTMAPS],sv[MAX_LIGHTMAPS]; // Filled by PV

    UPVD8* Maps[MAX_LIGHTMAPS];
    unsigned SurfaceHandle[MAX_MIPMAP_NUM];
} PVLightMap;

// A Vertex of a mesh
typedef struct _PVVertex3D
{
	float xf,yf,zf;
    PVPoint Normal;
    PVFLAGS Flags;
    PVFLAGS FrustumFlags;
} PVVertex;

// Mapping Coordinates
typedef struct _PVMapInfo
{
    float u,v;
    float AmbientU,AmbientV;
} PVMapInfo;

// A porjected vertex
typedef struct _PVScreenPoint
{
    float xf,yf;
    float InvertZ;
} PVScreenPoint;

// Shading/Color info for a vertex
typedef struct _PVShadeInfo
{
    UPVD8 Shade;
    PVRGBF Color;
	PVRGBF Specular;
} PVShadeInfo;

// A bounding box
typedef struct _PVBox
{
    PVFLAGS Flags;
	unsigned int NbrFaces;
    PVPoint p0,t0;
    unsigned *FaceIndex;
    struct _PVBox *Child[8];
	float Volume;
} PVBox;

// A *CONVEX* face
typedef struct _PVFace
{
    char *Material;									// Material Name of the face, before compilation
    PVMaterial *MaterialInfo;						// Material pointer afetr compilation
    void (PVAPI * Filler)(struct _PVFace *f);		// Filler for this face
    PVFLAGS Flags;									// Rendering flags
    union {
	UPVD8 NbrVertexes;								// Number of vertices in this faces
	UPVD8 NbrVertices;								// Number of vertices in this faces
	};
    unsigned V[MAX_VERTICES_PER_POLY];				// These are index to Father->Vertex
    int ZAvg;										// Average Z for this face
    float DistanceToCam;							// Z distance from the camera
    struct _PVMesh *Father;							// PVMesh where this face lives
    PVPoly *Poly;									// For clipping
    PVPoint Normal;									// Normal to this face
    PVTexBasis *TexBasis;							// Pointer to a texture basis for this face
	PVLightMap *LightMap;							// Pointer to a lighmap for this face
} PVFace;

struct _PVWorld;

typedef struct _PVMesh
{
		struct _PVWorld *Owner;
		char *Name;

        PVFLAGS Flags;
        
		unsigned NbrFaces;
		union {
		unsigned NbrVertexes;
		unsigned NbrVertices;
		};
		unsigned NbrVisibles;
		union {
		unsigned NbrSwitchItems;
		unsigned NbrBoxes;
		};
        unsigned MaxNbrClippedFaces;
		union {
		unsigned MaxNbrClippedVertexes;
		unsigned MaxNbrClippedVertices;
		};
        PVPoint Pivot;

        char *VertexVisible;
        unsigned nbrclipv;

        PVPoint OldPos,LastWorldPos,Position;
        PVMat3x3 OldMatrix,LastWorldMatrix,Matrix;
        float ScaleX,ScaleY,ScaleZ;

		PVVertex *Vertex;
        PVFace *Face;
		
        PVMapInfo		*Mapping;

        PVPoint			*Rotated;
        PVScreenPoint	*Projected;

        PVFace          **Visible;

        PVShadeInfo		*Shading;

        PVBox			*Box;
		
		union
		{
			unsigned		NbrVertexBox;
			struct _PVMesh* (PVAPI *SwitchCallBack)(struct _PVMesh*);
		};

        PVPoly			*Polys;								// [MaxNbrClipFaces]
        union
		{
			unsigned		NbrPolys;
			struct _PVMesh *LastSwitchPath;
		};

		void			*CollideInfo;

        int (PVAPI * UserPipeline)(struct _PVMesh *m);

        void *UserData;

        struct _PVMesh *Next,*Child,*Father;
		
		unsigned LastFrame;		
		PVMat3x3 WorldMatrix;
		PVPoint WorldPos;	
		PVPoint HierPivot;

		// since 0.98b
		PVPoint WorldHierPos;
		void*	HardwarePrivate;
} PVMesh;


//-------------------------------------------------------------------------------------
// ###############################################################  Mirror structure
//-------------------------------------------------------------------------------------

// Structure for a convex mirror
// Up to 8 vertex per mirror (mini 3 :)

// The following flags are to be used with PVMirror.Flags
#define PV_MIRROR_FORGET            1
#define PV_MIRROR_TRANSPARENT       2
#define PV_MIRROR_DARKER            4

typedef struct _PVMirror
{
    unsigned RefCnt;
	PVFLAGS Flags;
    union {
	unsigned NbrVertexes;
	unsigned NbrVertices;
	};
    PVPoint Vertex[MAX_VERTICES_PER_POLY*2];        // Warning a part of this array should be left empty for clipping

    float ZMin;

    PVPoint Normal;
    double d;

    PVMesh *Father;
    PVMat3x3  Matrix;
    PVPoint Pos,Pivot;

    PVMat3x3  SymMatrix;

    // Special Caps
    float CutDist;

    float Transparency;
    float Reflectance;

    struct _PVMirror *Next;
} PVMirror;

//-------------------------------------------------------------------------------------
// ###############################################################	Fog Infos
//-------------------------------------------------------------------------------------
typedef enum _PVFogType {PVF_NONE=0,PVF_LINEAR,PVF_EXP,PVF_EXP2} PVFogType;

typedef struct _PVFogInfo
{
	PVRGB Color;
	PVFogType Type;
	float Density;		
	float Start,End;	// up to the far clipping plane
} PVFogInfo;

//-------------------------------------------------------------------------------------
// ###############################################################  World Structure
//-------------------------------------------------------------------------------------

// The follwing flags are used with the PV_SetSortOrder function to modify the sort order
// of a world

#define PVW_BACK_2_FRONT    2					// Sorts polygons back to front
#define PVW_FRONT_2_BACK    4					// Sorts polygons front to back
#define PVW_MATERIAL	    8					// Sorts polygons by materials

typedef struct _PVWorld
{
    unsigned     NbrFaces;
	union {
	unsigned NbrVertexes;
	unsigned NbrVertices;
	};
	unsigned NbrVisibles;
    unsigned     NbrObjs;
    PVFLAGS      Flags;

    PVRGBF AmbientLight;

    PVMaterial *Materials;

    PVFace    **Visible;

    PVMesh *Objs;
    PVLight *Lights;

    UPVD8 ReservedColors;
    PVRGB *Global256Palette;

    PVCam *Camera;

    PVMirror *Mirrors;

	unsigned LastFrame;

	// 0.98b
	PVFogInfo Fog;
	void *HardwarePrivate;

} PVWorld;


//-------------------------------------------------------------------------------------
// ###############################################################  Panard Vision API
//-------------------------------------------------------------------------------------

// General Constants
#define PV_ENABLE   1
#define PV_DISABLE  2

// These one are for decoding the flags returned by PV_GetFrustumFlags()
#define FRUSTUM_TOP			8
#define FRUSTUM_BOTTOM		4
#define FRUSTUM_LEFT		1
#define FRUSTUM_RIGHT		2
#define FRUSTUM_BACK		32
#define FRUSTUM_FRONT		16
// Then follows the SpecialClipPlanes, and the User defines camera clipplanes


// (not) Useful macros
#define ISFLAGSET(f,ftt) (f&ftt)
#define ADDFLAG(f,ftt) (f|=ftt)
#define REMOVEFLAG(f,ftt) (f&=~ftt)
#define FORCE_MESH_RECALC(x)    {(x)->OldPos.xf=MAXFLOAT;}

// Panard Primitives
// Drawable primitives
typedef enum _pvPRIMIT {PV_TRIANGLES,PV_TRIANGLES_STRIP,PV_TRIANGLES_FAN,PV_POLYS,PV_QUADS} pvPRIMIT;
typedef enum _pvCULL {PV_CULL_NONE,PV_CULL_CW,PV_CULL_CCW} pvCULL;
typedef enum _pvMODE {PV_2D,PV_3D} pvMODE;
typedef enum _pvLIGHT {PV_LIGHTING,PV_NOLIGHTING} pvLIGHT;

// For indexed drawing
#define PPI_MAPCOORD		1
#define PPI_AMBMAPCOORD		2
#define PPI_COLOR			4
#define PPI_SPECULAR		8
#define PPI_NORMAL			16
#define PPI_COLORINDEX		32

PVEXPORT void PVAPI pvReset(void);
PVEXPORT void PVAPI pvSetMode(pvMODE mode);
PVEXPORT int PVAPI pvGetErrorCode(void);

PVEXPORT void PVAPI pvSetCamera(PVCam *cam);
PVEXPORT void PVAPI pvSetModelMatrix(PVMat3x3 mat);
PVEXPORT void PVAPI pvSetModelPos(PVPoint p);
PVEXPORT void PVAPI pvSetModelPivot(PVPoint p);
PVEXPORT void PVAPI pvSetDepthField(double front,double back);
PVEXPORT void PVAPI pvSetCull(pvCULL cull);

PVEXPORT void PVAPI pvSetLightingMode(pvLIGHT mode);
PVEXPORT void PVAPI pvAddLight(PVLight *l,unsigned *handle);
PVEXPORT void PVAPI pvRemoveLight(unsigned handle);
PVEXPORT void PVAPI pvSetAmbientLight(float r,float g,float b,float a);

PVEXPORT void PVAPI pvSetFogType(PVFogType type);
PVEXPORT void PVAPI pvSetFogStart(float start);
PVEXPORT void PVAPI pvSetFogEnd(float end);
PVEXPORT void PVAPI pvSetFogDensity(float density);
PVEXPORT void PVAPI pvSetFogColor(float r,float g,float b);

PVEXPORT void PVAPI pvEnableIndex(PVFLAGS flags);
PVEXPORT void PVAPI pvSetVertexIndex(float *xyz0,unsigned stride);
PVEXPORT void PVAPI pvSetMapIndex(float *uv0,unsigned stride);
PVEXPORT void PVAPI pvSetAmbMapIndex(float *auav0,unsigned stride);
PVEXPORT void PVAPI pvSetColorIndex(float *rgba0,unsigned stride);
PVEXPORT void PVAPI pvSetSpecularIndex(float *rgba0,unsigned stride);
PVEXPORT void PVAPI pvSetNormalIndex(float *xyz0,unsigned stride);
PVEXPORT void PVAPI pvSetColorIndexIndex(UPVD8 *c0,unsigned stride);

PVEXPORT void PVAPI pvBegin(pvPRIMIT prim,UPVD8 *Scr);

PVEXPORT void PVAPI pvSetMaterial(PVMaterial *m);
PVEXPORT void PVAPI pvMapCoord(float u,float v);
PVEXPORT void PVAPI pvAmbMapCoord(float u,float v);
PVEXPORT void PVAPI pvNormal(float x,float y,float z);
PVEXPORT void PVAPI pvColor(float r,float g,float b,float a);
PVEXPORT void PVAPI pvSpecular(float r,float g,float b,float a);
PVEXPORT void PVAPI pvColorIndex(UPVD8 index);
PVEXPORT void PVAPI pvSetWrapFlag(PVFLAGS flags);
PVEXPORT void PVAPI pvVertexf(float x,float y,float z);
PVEXPORT void PVAPI pvVertex(double x,double y,double z);

// The same but a little more friendly with memory bandwidth
PVEXPORT void PVAPI pvMapCoordv(float *uv);
PVEXPORT void PVAPI pvAmbMapCoordv(float *uv);
PVEXPORT void PVAPI pvNormalv(float *xyz);
PVEXPORT void PVAPI pvColorv(float *rgba);
PVEXPORT void PVAPI pvSpecularv(float *rgba);
PVEXPORT void PVAPI pvVertexfv(float *xyz);
PVEXPORT void PVAPI pvVertexv(double *xyz);

PVEXPORT void PVAPI pvVertexIndexedf(unsigned index);

PVEXPORT void PVAPI pvEnd(void);


// Math stuffs
PVEXPORT void PVAPI SetupMatrix3x3(PVMat3x3 Matrix,float AngleX,float AngleY,float AngleZ);
PVEXPORT void PVAPI RotatePoint3x3(PVPoint *v,PVMat3x3 m,PVPoint *p);
PVEXPORT void PVAPI RotateInvertPoint(PVPoint *v,PVMat3x3 m,PVPoint *p);
PVEXPORT void PVAPI OrthonormalizeMatrix(PVMat3x3 m);
PVEXPORT void PVAPI MatrixMul(PVMat3x3 in1, PVMat3x3 in2, PVMat3x3 out);

PVEXPORT void PVAPI QuatExp( PVQuat *q, PVQuat *dest );
PVEXPORT void PVAPI QuatInv( PVQuat *q, PVQuat *dest );
PVEXPORT void PVAPI QuatLog( PVQuat *q, PVQuat *dest );
PVEXPORT void PVAPI QuatLnDif( PVQuat *p, PVQuat *q, PVQuat *dest );
PVEXPORT float PVAPI QuatMod(PVQuat *q );
PVEXPORT float PVAPI QuatDot( PVQuat *q1, PVQuat *q2 );
PVEXPORT float PVAPI QuatDotUnit( PVQuat *q1, PVQuat *q2 );
PVEXPORT void PVAPI QuatNegate( PVQuat *q );
PVEXPORT void PVAPI QuatMul(PVQuat *q1,PVQuat *q2,PVQuat *q3);
PVEXPORT void PVAPI QuatToMatrix(PVQuat *q,PVMat3x3 Matrix);
PVEXPORT void PVAPI MatrixToQuat(PVMat3x3 Matrix,PVQuat *q);
PVEXPORT void PVAPI QuatSlerp(PVQuat *a,PVQuat *b, PVQuat *dest, float time,float spin );
PVEXPORT void PVAPI QuatSlerpLong( PVQuat *a,PVQuat *b, PVQuat *dest, float time,float spin );
PVEXPORT void PVAPI QuatScale( PVQuat *q,float s,PVQuat *dest );
PVEXPORT void PVAPI QuatUnit( PVQuat *q,PVQuat *dest );
PVEXPORT void PVAPI OrientationToQuat(float x,float y,float z,float a,PVQuat *q);
PVEXPORT void PVAPI QuatToOrientation( PVQuat *q, float *x,float *y,float *z,float *angle );


// Misc
PVEXPORT int PVAPI PV_SetMode(PVFLAGS mode);
PVEXPORT int PVAPI PV_SetPipelineControl(PVFLAGS mode);
PVEXPORT int PVAPI PV_SetClipLimit(unsigned int minx,unsigned int maxx, unsigned int miny, unsigned int maxy, unsigned int ScanLine);
PVEXPORT void PVAPI PV_TriangulateFace(PVFace *f,void (PVAPI * Func)(PVFace *f));
PVEXPORT void PVAPI PV_GarbageCollector(PVWorld *w);
PVEXPORT void PVAPI PV_Fatal(char *s,unsigned int t);      // Exit!
PVEXPORT void PVAPI PV_PrepareFace(PVFace *f);
PVEXPORT void *PVAPI PV_GetFiller(PVFLAGS flags);
PVEXPORT void __cdecl PV_Log(char *logfile,char *fmt, ...);
PVEXPORT PVTexture *PVAPI PV_MipResample(UPVD8 *mip, unsigned width,unsigned height,char grow,PVFLAGS texflags,PVRGB *Pal,unsigned reserved);
PVEXPORT void *PVAPI pvmalloc( size_t size );
PVEXPORT void *PVAPI pvcalloc( size_t num, size_t size );
PVEXPORT void PVAPI pvfree( void *memblock );


// Frustum
PVEXPORT int PVAPI PV_GetFrustumFlags(PVPoint *v);
PVEXPORT int PVAPI PV_ClipFaceToFrustum(PVFace *f);
PVEXPORT int PVAPI PV_GetNbrFreeSpecialClipPlanes(void);
PVEXPORT PVPlane *PVAPI PV_AllocClipPlanes(unsigned n);
PVEXPORT void PVAPI PV_FreeLastClipPlane(void);
PVEXPORT void PVAPI PV_FreeAllClipPlanes(void);
PVEXPORT void PVAPI PV_SetPlane(PVPlane *plane,double position,PVPoint *normal);
PVEXPORT unsigned PVAPI PV_GetNbrClipPlanes(void);
PVEXPORT void PVAPI PV_SetNbrClipPlanes(unsigned nbr);


// LightMaps
PVEXPORT void PVAPI PV_ResetTextureCache(void);
PVEXPORT int PVAPI PV_InitTextureCache(unsigned size);
PVEXPORT void PVAPI PV_ReleaseTextureCache(void);
PVEXPORT UPVD8 *PVAPI PV_GetCachedTexture(PVFace *f,unsigned MipMapNum,unsigned *Width,unsigned *Height);
PVEXPORT void PVAPI PV_DumpCache(void);
PVEXPORT PVLightMap *PVAPI PV_CreateLightMapInfo(void);
PVEXPORT void PVAPI PV_KillLightMapInfo(PVLightMap *l);
PVEXPORT void PVAPI PV_AttachLightMap(PVFace *f,PVLightMap *b);
PVEXPORT void PVAPI PV_SetLightMapExtent(PVFace *f,unsigned MipMapNum,PVLightMap *l);


// PVWorld
PVEXPORT PVWorld * PVAPI PV_CreateWorld(void);
PVEXPORT void PVAPI PV_KillWorld(PVWorld *w);
PVEXPORT int PVAPI PV_SetSortOrder(PVWorld *w,PVFLAGS f);
PVEXPORT int PVAPI PV_SortWorld(PVWorld *w);
PVEXPORT int PVAPI PV_RenderWorld(PVWorld *w,UPVD8 *Scr);
PVEXPORT void PVAPI PV_AddMaterial(PVWorld *w,PVMaterial *m);
PVEXPORT PVMaterial * PVAPI PV_GetMaterialPtrByName(char *name,PVWorld *z);
PVEXPORT int PVAPI PV_WorldSetAmbientLight(PVWorld *w,PVRGBF a);
PVEXPORT void PVAPI PV_AddLight(PVWorld *w,PVLight *l);
PVEXPORT int PVAPI PV_RemoveLightByName(PVWorld *w,char *s);
PVEXPORT int PVAPI PV_RemoveLightByPtr(PVWorld *w,PVLight *o);
PVEXPORT PVLight * PVAPI PV_GetLightPtr(PVWorld *w,char *s);
PVEXPORT int PVAPI PV_CompileMeshes(PVWorld *z);
PVEXPORT int PVAPI PV_CompileMesh(PVMesh *m);
PVEXPORT void PVAPI PV_ScaleWorld(PVWorld *w,float ax,float ay,float az);
PVEXPORT int PVAPI PV_CompileMaterials(PVWorld *w,UPVD8 (*CalcFunc)(PVRGB *pal,PVMaterial *m,unsigned col,unsigned lightlevel,PVRGBF ambient,unsigned reserved),UPVD16 (*CalcFunc16)(PVMaterial *m,unsigned col,unsigned lightlevel,PVRGBF ambient));
PVEXPORT void PVAPI PV_AddMirror(PVWorld *w,PVMirror *l);
PVEXPORT int PVAPI PV_RemoveMirrorByPtr(PVWorld *w,PVMirror *o);
PVEXPORT int PVAPI PV_AddMesh(PVWorld *w,PVMesh *no);
PVEXPORT int PVAPI PV_RemoveMeshByPtr(PVMesh *);
PVEXPORT int PVAPI PV_RemoveMeshByName(PVWorld *,char *);
PVEXPORT PVMesh * PVAPI PV_GetMeshPtr(PVWorld *w,char *s);
PVEXPORT int PVAPI PV_AddChildMesh(PVMesh *father,PVMesh *child);
PVEXPORT PVMesh *PVAPI PV_UnlinkMesh(PVMesh *o);


// Camera
PVEXPORT PVCam * PVAPI PV_CreateCam(char *n);
PVEXPORT void PVAPI PV_KillCam(PVCam *t);
PVEXPORT void PVAPI PV_RAZCam(PVCam *c);
PVEXPORT void PVAPI PV_ComputeCam(PVCam *c);
PVEXPORT void PVAPI PV_CamAhead(PVCam *c,float a);
PVEXPORT int PVAPI PV_SetCamName(PVCam *c,char *n);
PVEXPORT void PVAPI PV_SetCamFieldOfView(PVCam *c,float f);
PVEXPORT void PVAPI PV_SetCamAngles(PVCam *c,float yaw,float pitch,float roll);
PVEXPORT void PVAPI PV_SetCamPos(PVCam *c,float x,float y, float z);
PVEXPORT void PVAPI PV_SetCamTarget(PVCam *ca,float x,float y, float z);
PVEXPORT void PVAPI PV_SetCamRoll(PVCam *c,float roll);
PVEXPORT int PVAPI PV_EnableClipPlane(PVCam *c,unsigned i,unsigned val);
PVEXPORT int PVAPI PV_SetClipPlane(PVCam *c,unsigned i,PVPoint Pos,PVPoint Normal);
PVEXPORT PVMesh *PVAPI PV_FindCurrentCameraCell(PVMesh *m,PVCam *c);


// Light
PVEXPORT PVLight * PVAPI PV_CreateLight(PVLightType f,char *name);
PVEXPORT int PVAPI PV_SetLightName(PVLight *l,char *n);
PVEXPORT void PVAPI PV_KillLight(PVLight *l);
PVEXPORT void PVAPI PV_SetLightDirection(PVLight *l,float x,float y,float z);
PVEXPORT void PVAPI PV_SetLightPosition(PVLight *l,float x,float y,float z);
PVEXPORT void PVAPI PV_SetLightTarget(PVLight *l,float x,float y,float z);
PVEXPORT int PVAPI PV_SetLightIntensity(PVLight *l,float x);
PVEXPORT void PVAPI PV_SetLightFlags(PVLight *l,PVFLAGS x);
PVEXPORT void PVAPI PV_SetLightType(PVLight *l,PVLightType x);
PVEXPORT void PVAPI PV_SetLightRange(PVLight *l,float x);
PVEXPORT void PVAPI PV_SetLightAttenuation(PVLight *l,float a0,float a1,float a2);
PVEXPORT void PVAPI PV_SetLightSpot(PVLight *l,float Theta,float Phi,float Falloff);
PVEXPORT void PVAPI PV_RecomputeLight(void);
PVEXPORT void PVAPI PV_NoRecomputeLight(void);
PVEXPORT int PVAPI PV_LightRecomputeNeeded(void);


// Mesh
PVEXPORT PVMesh * PVAPI PV_CreateMesh(unsigned maxclipface,unsigned maxclipvertex);  // Deprecated, use PV_SimpleCreateMesh instead, here for compatibility only
PVEXPORT PVMesh * PVAPI PV_SimpleCreateMesh(unsigned nbrfaces,unsigned nbrvertices,unsigned maxclipface,unsigned maxclipvertex);
PVEXPORT PVMesh *PVAPI PV_CreateMeshPortal(void);
PVEXPORT void PVAPI PV_SetupPortal(PVMesh *m);
PVEXPORT void PVAPI PV_KillMesh(PVMesh *no);
PVEXPORT void PVAPI PV_MeshCompileVertexes(PVMesh *o);
PVEXPORT void PVAPI PV_MeshTranslateVertexes(PVMesh *o,float x,float y,float z);
PVEXPORT void PVAPI PV_MeshRotateVertexes(PVMesh *o,PVMat3x3 m);
PVEXPORT void PVAPI PV_MeshScaleVertexes(PVMesh *o,float a,float b, float c);
PVEXPORT int PVAPI PV_MeshNormCalc(PVMesh *o);
PVEXPORT void PVAPI PV_MeshBuildBoxes(PVMesh *o,unsigned nbrptsperbox);
PVEXPORT void PVAPI PV_MeshSetupMatrix(PVMesh *o,float AngleX,float AngleY,float AngleZ);
PVEXPORT void PVAPI PV_MeshSetupMatrixDirect(PVMesh *o,PVMat3x3 mat);
PVEXPORT void PVAPI PV_MeshSetupInvertMatrixDirect(PVMesh *no,PVMat3x3 t2);
PVEXPORT void PVAPI PV_MeshSetupPos(PVMesh *o,float X,float Y,float Z);
PVEXPORT void PVAPI PV_ComputeMeshToWorldMatrix(PVMesh *o);
PVEXPORT void PVAPI PV_MeshSetFlags(PVMesh *o,PVFLAGS f);
PVEXPORT int PVAPI PV_MeshRAZShade(PVMesh *o);
PVEXPORT void PVAPI PV_MeshLight(PVMesh *o,PVLight *l2);
PVEXPORT void PVAPI PV_MeshLightRGB(PVMesh *o,PVLight *l2);
PVEXPORT unsigned PVAPI PV_KillMeshList(PVMesh *m,unsigned *nbrfaces,unsigned *nbrvtx);
PVEXPORT PVMesh *PVAPI PV_SearchMeshList(PVMesh *m,char *name);
PVEXPORT int PVAPI PV_IterateMeshList(PVMesh *m,int (* UserFunc)(PVMesh *,int userarg),int userarg);
PVEXPORT PVMesh *PVAPI PV_CloneMesh(PVMesh *o);
PVEXPORT int PVAPI PV_GetMeshAAExtent(PVMesh *m,PVPoint *org,PVPoint *extent);
PVEXPORT void PVAPI PV_TransformPointLikeMesh(PVPoint *src,PVMesh *m,PVCam *c,PVPoint *rotated,PVScreenPoint *dest);
// More grammatically correct function names
#define PV_MeshCompileVertices PV_MeshCompileVertexes
#define PV_MeshTranslateVertices PV_MeshTranslateVertexes
#define PV_MeshRotateVertices PV_MeshRotateVertexes
#define PV_MeshScaleVertices PV_MeshScaleVertexes



// Switchable nodes
PVEXPORT PVMesh *PVAPI PV_CreateSwitchNode(PVMesh * (PVAPI *callback)(PVMesh*));
PVEXPORT int PVAPI PV_AddSwitch(PVMesh *switchnode,PVMesh *switchitem);
PVEXPORT PVMesh *PVAPI PV_RemoveSwitch(PVMesh *switchnode,int itemnbr);
PVEXPORT PVMesh *PVAPI PV_GetSwitchItem(PVMesh *switchnode,int itemnbr);


// Material
PVEXPORT PVMaterial * PVAPI PV_CreateMaterial(char *n,PVFLAGS type,PVFLAGS flags,unsigned char nbrmipmaps);
PVEXPORT void PVAPI PV_KillMaterial(PVMaterial *m);
PVEXPORT void PVAPI PV_SwapMaterials(PVMaterial *m1,PVMaterial *m2);
PVEXPORT void PVAPI PV_SetMaterialTextureFlags(PVMaterial *m,PVFLAGS f);
PVEXPORT void PVAPI PV_SetMaterialType(PVMaterial *m,PVFLAGS f);
PVEXPORT int PVAPI PV_SetMaterialTexture(PVMaterial *m,unsigned width,unsigned height,UPVD8 *texture,PVRGB *pal);
PVEXPORT int PVAPI PV_SetMaterialAuxiliaryTexture(PVMaterial *m,unsigned width,unsigned height,UPVD8 *texture);
PVEXPORT void PVAPI PV_SetMaterialPureColorsIndex(PVMaterial *m,UPVD8 start,UPVD8 end);
PVEXPORT int PVAPI PV_SetMaterialLightInfo(PVMaterial *m,PVRGBF Emissive,PVRGBF Diffuse,PVRGBF Specular,unsigned specularpower);
PVEXPORT int PVAPI PV_SetMaterialPhongPalette(PVMaterial *t,PVRGBF ambient);
PVEXPORT UPVD8 PVAPI PV_LookClosestColor(PVRGB *pal,PVRGB c,unsigned reserved);
PVEXPORT int PVAPI PV_CreateMipMapTextures(PVMaterial *m,PVRGB *WorldPal,unsigned reserved);
PVEXPORT int PVAPI PV_BuildBumpMap(PVMaterial *m,UPVD8 *Texture,unsigned width,unsigned height,float scale);
PVEXPORT PVD32 PVAPI PV_MakeNormalizedColorRGB(PVRGB c);
PVEXPORT PVRGB PVAPI PV_DecomposeNormalizedColor(UPVD32 c);
PVEXPORT int PVAPI PV_SetRGBIndexingMode(UPVD8 RMaskSize,UPVD8 GMaskSize,UPVD8 BMaskSize,
                            UPVD8 RFieldP,UPVD8 GFieldP, UPVD8 BFieldP,UPVD8 AMaskS);
PVEXPORT int PVAPI PV_CreateMaterialLightTranscodeTable16(PVMaterial *mat,unsigned TexNumber,UPVD16 (*CalcFunc)(PVMaterial *m,unsigned col,unsigned lightlevel,PVRGBF ambient),PVRGBF ambient);
PVEXPORT int PVAPI PV_CreateMaterialLightTranscodeTable(PVMaterial *mat,unsigned TexNumber,PVRGB *pal,UPVD8 (*CalcFunc)(PVRGB *pal,PVMaterial *m,unsigned col,unsigned lightlevel,PVRGBF ambient,unsigned reserved),PVRGBF ambient,unsigned reserved);
PVEXPORT int PVAPI PV_CreateGlobal256Palette(PVMaterial *z,UPVD8 ReservedColors,PVRGB **finalpal);
PVEXPORT int PVAPI PV_ConvertMaterial(PVMaterial *m);
PVEXPORT int PVAPI PV_MaterialGammaCorrect(PVMaterial *m,float gamma);
PVEXPORT int PVAPI PV_GammaCorrectPal(PVRGB Pal[256],float gamma);
PVEXPORT int PVAPI PV_SetMaterialMipMap(PVMaterial *m,unsigned mipnbr,UPVD8 *texture); // availlable mipmaps are from 1 to NbrMipMaps-1
PVEXPORT int PVAPI PV_CompileMaterial(PVMaterial *m,UPVD8 (*CalcFunc)(PVRGB *pal,PVMaterial *m,unsigned col,unsigned lightlevel,PVRGBF ambient,unsigned reserved),UPVD16 (*CalcFunc16)(PVMaterial *m,unsigned col,unsigned lightlevel,PVRGBF ambient),PVRGBF ambient,unsigned reservedcolors);
PVEXPORT PVTexBasis *PVAPI PV_CreateTextureBasis(float s[4],float t[4]);
PVEXPORT void PVAPI PV_KillTextureBasis(PVTexBasis *b);
PVEXPORT void PVAPI PV_AttachTextureBasis(PVFace *f,PVTexBasis *b);


// Splines
PVEXPORT PVSpline1 *PVAPI PV_CreateSpline1(unsigned order,unsigned nbrcomponent);
PVEXPORT PVSpline2 *PVAPI PV_CreateSpline2(unsigned uorder,unsigned vorder,unsigned nbrcomponent);
PVEXPORT void PVAPI PV_KillSpline1(PVSpline1 *s);
PVEXPORT void PVAPI PV_KillSpline2(PVSpline2 *s);
PVEXPORT void PVAPI PV_EvalSpline1(PVSpline1 *pvs,float u,float result[]);
PVEXPORT void PVAPI PV_EvalSpline2(PVSpline2 *pvs,float u, float v,float result[],PVPoint *Normal);


// Software ZBuffer
PVEXPORT void PVAPI PV_SetZBufferClearRoutine(void (PVAPI * ZBufferRoutine)(void));
PVEXPORT void PVAPI PV_ClearZBuffer(void);
PVEXPORT void PVAPI PV_ClearZBufferNormal(void);
PVEXPORT float * PVAPI PV_GetZBuffer(void);


// Mirrors
PVEXPORT PVMirror *PVAPI PV_CreateMirror(void);
PVEXPORT void PVAPI PV_KillMirror(PVMirror *m);
PVEXPORT void PVAPI PV_SetupMirror(PVMirror *m);
PVEXPORT void PVAPI PV_PrepareMirror(PVPoly *m,PVCam *sc,PVPoint vtx[]);


// Collision detection
// Combinable Flags for PV_CollisionTest()
#define COLLISION_NORMAL	0
#define	COLLISION_NO_REPORT	1									// Fast collision detection, but no information on which faces are colliding
#define COLLISION_FAST		(2|COLLISION_NO_REPORT)				// Uses a little faster detection scheme, but far less precise (maybe wrong in some case), depends on environnement

// API
PVEXPORT int PVAPI PV_MeshComputeCollisionTree(PVMesh *m);
PVEXPORT void PVAPI PV_MeshKillCollisionTree(PVMesh *m);
PVEXPORT int PVAPI PV_CollisionTest(PVMesh *m1,PVMesh *m2,PVFLAGS flags);
PVEXPORT PVCollidingFaces *PVAPI PV_GetCollisionTable(void);
PVEXPORT unsigned PVAPI PV_GetNbrCollision(void);

PVEXPORT PVMesh *PVAPI PV_CreateDummyMesh(float radiusx,float radiusy,float radiusz);	// Creates an invisible mesh
																						// Use it to simulate the camera for collision
																						// NEVER makes this mesh visible
																						// or you will die.

//-------------------------------------------------------------------------------------
// ###############################################################  Callbacks functions
//-------------------------------------------------------------------------------------

// User Defined Fonctions, default is provided
PVEXPORT void (PVAPI *PV_UserCleanUp)(unsigned t);               // Called when a fatal error occur. T is the error code
PVEXPORT void (PVAPI *  PV_UserLight)(PVMesh *,PVLight *);       // Called to render an user light
PVEXPORT void (PVAPI * PV_UserMeshCleanUp)(PVMesh *no);          // Called when a mesh is killed
PVEXPORT int (PVAPI * PV_UserCompileMaterials)(PVMesh *m,PVMaterial *mat,unsigned FaceNumber); // Used to add special FX to PV
																		// return 0 if no errors
																		// return non zero if fatal error occured
PVEXPORT int (PVAPI *PV_UserRenderWorld)(PVWorld *,UPVD8 *);		// Called before rendering a polygon list
																		// user should return COOL to continue the process
																		// or something else to stop it


//-------------------------------------------------------------------------------------
// ###############################################################  Misc global variables
//-------------------------------------------------------------------------------------

// RGB Stuffs (use PV_SetRGBIndexingMode to set this up)
PVEXPORT UPVD8 RMaskSize,GMaskSize,BMaskSize;
PVEXPORT UPVD8 PixelSize;
PVEXPORT UPVD32 RedMask,GreenMask,BlueMask;
PVEXPORT unsigned RedFact,GreenFact,BlueFact;
PVEXPORT UPVD8 RFieldPos,GFieldPos,BFieldPos;


//-------------------------------------------------------------------------------------
// ###############################################################  Error Codes
//-------------------------------------------------------------------------------------

#define COOL                    0
#define FILE_IOERROR            1
#define ALREADY_ASSIGNED        2
#define NO_MEMORY               3
#define BIZAR_ERROR             5
#define ARG_INVALID             6
#define ITEM_NOT_FOUND          7
#define MATERIAL_NOT_REGISTERED 8
#define INVALID_CAM             9
#define INVALID_TEXTURE         10
#define NO_MATERIALS            11
#define	ALREADY_IN_HIERARCHY	12
#define MESH_HAS_CHILD			13
#define NOT_AVAILABLE			14

#define NO_ACCELSUPPORT			100
#define ACCEL_NO_MEMORY			101
#define ACCEL_FUNCTION_UNSUPPORTED	102
#define NO_HARDWARE_SET         103
#define INVALID_DRIVER			104

#define COLLIDE_YES             200     //returned by PV_IsMeshesColliding
#define COLLIDE_NO              201     //returned by PV_IsMeshesColliding
#define NO_COLLISIONINFO        202


//-------------------------------------------------------------------------------------
// ###############################################################  Generic Filler API
//-------------------------------------------------------------------------------------

#define MAX_INCREMENT 10    // minus 1 ie 9

// Flags for software rasterizer (GenFill_Flags)
#define PVR_OPTIMPOLYS	1	// Try to optimize polygon rendering

////////////////////////////////////////////////////////////////////////////

typedef struct _PVFillerUserFunct {
    int(__cdecl *InitFunc1)(void);
    void(__cdecl *InitFunc2)(void);
} FillerUserFunct;

////////////////////////////////////////////////////////////////////////////

PVEXPORT void (__cdecl *HLineRoutine)(unsigned, unsigned);  // Scanline filler

PVEXPORT unsigned GenFill_NbrInc,GenFill_NbrIncF;
PVEXPORT unsigned GenFill_p1,GenFill_p2,GenFill_p3;
PVEXPORT double GenFill_GradientX[MAX_INCREMENT];
PVEXPORT double GenFill_GradientY[MAX_INCREMENT];
PVEXPORT int GenFill_iGradientX[MAX_INCREMENT];
PVEXPORT float GenFill_InitialValues[3][MAX_INCREMENT];
PVEXPORT PVFace *GenFill_CurrentFace;

PVEXPORT unsigned GenFill_CurrentLeftVal[MAX_INCREMENT];
PVEXPORT double GenFill_CurrentLeftValF[MAX_INCREMENT];

PVEXPORT unsigned GenFill_Height;
PVEXPORT unsigned GenFill_CurrentY;

PVEXPORT PVFLAGS GenFill_Flags;

PVEXPORT PVRGBF *RGBAmbientLight;

// Screen Layout, used by fillers
// Use PV_SetClipLimit
PVEXPORT int ClipMinX;
PVEXPORT int ClipMinY;
PVEXPORT int ClipMaxX;
PVEXPORT int ClipMaxY;

PVEXPORT unsigned int ScanLength;     // Scanline length
PVEXPORT UPVD8 *TriFill_BufOfs;       // Drawing Buffer

////////////////////////////////////////////////////////////////////////////
PVEXPORT void PVAPI GenFiller(PVFace *f,FillerUserFunct *FUT);
PVEXPORT void PVAPI GenFill_Wrap(unsigned i);
PVEXPORT unsigned PVAPI IsWrapped(unsigned i);
PVEXPORT unsigned PVAPI GetMipMap(unsigned nbrmips,float mu,float mv);   // Linear gradients
PVEXPORT unsigned PVAPI GetMipMapIndex(unsigned nbrmips,float d,unsigned incu,unsigned incv);
PVEXPORT void PVAPI PV_ResetGenFillerEdges(void);


//-------------------------------------------------------------------------------------
// ###############################################################  Generic Perspective Span Renderer API
//-------------------------------------------------------------------------------------

#define AFFINE_LENGTH 16

PVEXPORT int GenPHLine_NbrIncF2F;
PVEXPORT double GradientXAff[MAX_INCREMENT];
PVEXPORT void (__cdecl *SLineRoutine)(unsigned, unsigned);
PVEXPORT void (__cdecl *HLPInitFunc)(void);

PVEXPORT void __cdecl GenPHLine(unsigned deb,unsigned fin);
PVEXPORT void PVAPI PV_SetAutomaticPerspectiveRatio(float r);
PVEXPORT float PVAPI PV_GetAutomaticPerspectiveRatio(float r);
PVEXPORT int PVAPI PerspectiveNeeded(float z1,float z2,float z3);
PVEXPORT void PVAPI PV_SetPerspectiveMipBias(float bias);


//-------------------------------------------------------------------------------------
// ###############################################################  Hardware driver API
//-------------------------------------------------------------------------------------

// General Flags
#define PHG_ZBUFFER				1				// Hardware supports ZBUffering
#define PHG_ALPHABUFFER			2				// Hardware supprots alpha buffer
#define PHG_ALPHATEST			4				// Hardware supports alpha testing against a reference alpha value

// Frame Access Flags
#define PHF_FRAMEBUFFER         1               // The hardware supports Direct frame buffer access
#define PHF_DEPTHBUFFER         2               // The hardware supports Direct depth buffer access
#define PHF_ALPHABUFFER         4               // The hardware supports Direct alpha buffer access
#define PHF_BITBLTFB			8				// The driver supports the BitBltFB function
#define PHF_BITBLTDB			16				// The driver supports the BitBltDB function
#define PHF_BITBLTAB			32				// The driver supports the BitBltAB function

// Texturing flags
#define PHT_BILINEAR			1				// Hardware supports Bilinear filtering
#define PHT_MIPMAP				2				// Hardware supports Mipmapping
#define PHT_TRILINEAR			4				// Hardware supports trilinear filtering

// Blending Flags
#define PHB_SRCRGBBLEND			1				// These flags indicates which fields are supported
#define PHB_DSTRGBBLEND			2				// in material blending
#define PHB_SRCALPHABLEND		4
#define PHB_DSTALPHABLEND		8

// Format Caps
#define PHD_DEPTHPROP			1				// Format of the depth buffer is propriatry
#define PHD_DEPTH16				2				// depth values are 16 bits (fixed)
#define PHD_DEPTHFLOAT			4

// Constants needed to define acces to the FrameBuffer
#define PFB_READONLY			1				// Give a lock for reading only
#define PFB_WRITEONLY			2				// Give a lock for writing only

// Defines the current state of the hardware
typedef struct _PVHardwareCaps {
	PVFLAGS GeneralCaps;
	PVFLAGS FrameCaps;
	PVFLAGS TextureCaps;
	PVFLAGS BlendCaps;
	PVFLAGS FormatCaps;

	unsigned BitsPerPixel;										// Number of bits per pixel in the frame buffer
	unsigned NbrBitsRed,NbrBitsGreen,NbrBitsBlue,NbrBitsAlpha;	// Size of each components
	unsigned RedPos,GreenPos,BluePos,AlphaPos;					// Position of each color component in the pixel
	int  RowStride;												// Length of a framebuffer scanline in bytes

	int PhysicalResx,PhysicalResY;								// Physical resolution of the screen
	unsigned NbrSurf;											// Number of surfaces available

	unsigned BitsPerDepth;										// Number of bits in a depth value red from the depth buffer

	unsigned BitsPerAlpha;										// Number of bits in an alpha value red from the alpha buffer
} PVHardwareCaps;

// This tructure defines the entry points for a valid driver.
typedef struct _PVHardwareDriver {
	unsigned Version;
	unsigned MagicNumber;
	unsigned SizeOfCaps;										// Size of the driver's PVHardwareCaps
	PVFLAGS Caps;												// Copied as the new pipeline control word, when driver is first initialized
	char *Desc;													// Name of the driver
	int (PVAPI * Detect)(void);									// Are you there method, !=0 if succesful detect
	int (PVAPI * InitSupport)(long i);							// Init hardware, return COOL if no error
	void (PVAPI * EndSupport)(void);							// unInit Hardware
	int (PVAPI * SetViewPort)(unsigned int cmx,unsigned int cMx,unsigned int cmy,unsigned int cMy); // Set drawing area if supported by hardware
    int (PVAPI * LoadTexture)(PVMaterial *m);					// Loads a material to hardware
    void (PVAPI * DeleteTexture)(PVMaterial *m);				// Frees a previously downloaded texture
    void *(PVAPI *GetFiller)(PVFLAGS flags);					// Gets a pointer to a filler according to the flags
	int (PVAPI * PreRender)(PVWorld *w,unsigned surafecnum,PVFLAGS PVMode,PVFLAGS PVPipe);	// Called before the effective renering starts
	void (PVAPI * PrepareFace)(PVFace *f);						// Called before each face
	void (PVAPI * PostRender)(void);							// Called at the end of the rendering
	void (PVAPI * BeginFrame)(PVFLAGS pvmode,PVFLAGS pvpipe);	// Mark the beginning of a scene
	void (PVAPI * EndFrame)(void);								// Guess
	int (PVAPI * FlipSurface)(void);							// Swap Buffers (for n-buffering hardware)
	int (PVAPI * FillSurface)(unsigned surfacenum,float r,float g,float b,float a);	// Fill the surfacenum buffer with the specified color
	void (PVAPI * RefreshMaterial)(PVMaterial *m);				//  Update the material state (not the texture content !)

	PVHardwareCaps *(PVAPI * GetInfo)(void);					// Retrieve driver/hardware capabilities
	void *(PVAPI * LockFB)(unsigned surfacenum,PVFLAGS mode);	// Lock the frame buffer, returns a pointer to the frame buffer
	void (PVAPI * UnLockFB)(unsigned surfacenum,PVFLAGS mode);	// Unlock a previously locked frame buffer
	void *(PVAPI * LockDB)(PVFLAGS mode);						// Lock the depth buffer, returns a pointer to the depth buffer
	void (PVAPI * UnLockDB)(PVFLAGS mode);						// Unlock a previously locked depth buffer
	void *(PVAPI * LockAB)(PVFLAGS mode);						// Lock the alpha buffer, returns a pointer to the alpha buffer
	void (PVAPI * UnLockAB)(PVFLAGS mode);						// Unlock a previously locked alpha buffer
	int (PVAPI * BitBltFB)(unsigned surfacenum,unsigned xdst,unsigned ydst,void *src,unsigned width, unsigned height,unsigned stride); // BitBlt a rectangular arrys direcly to the frame buffer (source should be at the format reported by getinfo)
	int (PVAPI * BitBltDB)(unsigned xdst,unsigned ydst,void *src,unsigned width, unsigned height,unsigned stride); // BitBlt a rectangular arrys direcly to the depth buffer (source should be at the format reported by getinfo)
	int (PVAPI * BitBltAB)(unsigned xdst,unsigned ydst,void *src,unsigned width, unsigned height,unsigned stride); // BitBlt a rectangular arrys direcly to the alpha buffer (source should be at the format reported by getinfo)

    int (PVAPI * LoadLightMap)(PVLightMap *l);                  // Loads a lightmap to hardware
    void (PVAPI * DeleteLightMap)(PVLightMap *l);               // Remove lightmaps from memory
} PVHardwareDriver;

PVEXPORT int PVAPI PV_SetHardwareDriver(PVHardwareDriver *hd);
PVEXPORT PVHardwareDriver * PVAPI PV_GetHardwareDriver(void);

PVEXPORT int PVAPI PV_InitAccelSupport(long i);
PVEXPORT void PVAPI PV_EndAccelSupport(void);
PVEXPORT int PVAPI PV_FlipSurface(void);
PVEXPORT int PVAPI PV_FillSurface(unsigned surfacenum,float r,float g,float b,float a);
PVEXPORT void PVAPI PV_BeginFrame(void);
PVEXPORT void PVAPI PV_EndFrame(void);
PVEXPORT void PVAPI PV_RefreshMaterial(PVMaterial *m);
PVEXPORT void PVAPI PV_GetHardwareInfo(PVHardwareCaps *caps);	  // fill in a PREALLOCATED PVhardware caps structure. Client is responsible for memory allocation/deallocation
PVEXPORT void * PVAPI PV_LockFrameBuffer(unsigned surfacenum,PVFLAGS mode);
PVEXPORT void PVAPI PV_UnLockFrameBuffer(unsigned surfacenum,PVFLAGS mode);
PVEXPORT void * PVAPI PV_LockDepthBuffer(PVFLAGS mode);
PVEXPORT void PVAPI PV_UnLockDepthBuffer(PVFLAGS mode);
PVEXPORT void * PVAPI PV_LockAlphaBuffer(PVFLAGS mode);
PVEXPORT void PVAPI PV_UnLockAlphaBuffer(PVFLAGS mode);
PVEXPORT int PVAPI PV_BitBltFrameBuffer(unsigned surfacenum,unsigned xdest,unsigned ydest,void *src,unsigned width, unsigned height,unsigned stride);
PVEXPORT int PVAPI PV_BitBltDepthBuffer(unsigned xdest,unsigned ydest,void *src,unsigned width, unsigned height,unsigned stride);
PVEXPORT int PVAPI PV_BitBltAlphaBuffer(unsigned xdest,unsigned ydest,void *src,unsigned width, unsigned height,unsigned stride);

#if !defined(__UNIX__) && !defined(DJGPP)
#pragma pack(pop)
#endif

#ifdef __cplusplus
}
#endif

#endif
